/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic.io;

import java.util.List;

import javax.persistence.AttributeOverride;
import javax.persistence.Column;

import org.eclipse.jsr220orm.core.internal.options.IntOption;
import org.eclipse.jsr220orm.generic.GenericEntityModelManager;
import org.eclipse.jsr220orm.generic.reflect.RAnnotatedElement;
import org.eclipse.jsr220orm.generic.reflect.RClass;
import org.eclipse.jsr220orm.metadata.AttributeMetaData;
import org.eclipse.jsr220orm.metadata.BasicAttribute;
import org.eclipse.jsr220orm.metadata.EmbeddedAttribute;
import org.eclipse.jsr220orm.metadata.EntityMetaData;
import org.eclipse.jsr220orm.metadata.OrmColumn;
import org.eclipse.jsr220orm.metadata.OrmTable;
import org.eclipse.jsr220orm.metadata.SimpleTypeMetaData;
import org.eclipse.jsr220orm.metadata.TypeMetaData;

/**
 * Handles mapping for attributes that override attributes from another class
 * (e.g. embedded mapping or inherited from a embeddable superclass) that are 
 * stored in our table. Currently only BasicAttribute's are implemented.
 */
public class OverrideAttributeIO extends AttributeIO {

	protected AttributeMetaData amd;
	protected EmbeddedAttribute embeddedAttribute;
	
	protected String defColName;
	
	public OverrideAttributeIO(EntityIO entityIO, 
			EmbeddedAttribute embeddedAttribute) {
		super(entityIO);
		this.embeddedAttribute = embeddedAttribute;
	}

	/**
	 * We don't use this update method.
	 */
	public boolean updateModelFromMetaData(RClass cls,
			RAnnotatedElement attribute, boolean metaDataChanged) {
		throw new IllegalStateException("updateModelFromMetaData invoked on " +
				this);
	}

	/**
	 * We dont use this update method.
	 */
	protected void updateMetaDataFromModel(RClass cls, 
			RAnnotatedElement attribute) {
		throw new IllegalStateException("updateMetaDataFromModel invoked on " +
				this);
	}

	public AttributeMetaData getAttributeMetaData() {
		return amd;
	}

	public void getPossibleMappings(List ans) {
	}

	public IntOption getMapping() {
		return null;
	}

	public void setMapping(IntOption mapping) {
	}
	
	/**
	 * Update our model from our meta data and the attribute we are
	 * overriding. 
	 */
	public boolean updateModelFromMetaData(RClass cls,
			AttributeMetaData overrideOf, AttributeOverride override,
			boolean metaDataChanged, int relativePositionOffset) {
		if (!(overrideOf instanceof BasicAttribute)) {
			throw new IllegalArgumentException("Unsupported: " + overrideOf);
		}
		
		EntityMetaData emd = entityIO.getEntityMetaData();
		
		boolean isNewAttribute = amd == null;
		if (isNewAttribute) {
			GenericEntityModelManager mm = entityIO.getModelManager();
			amd = (AttributeMetaData)mm.getFactory().create(overrideOf.eClass());			
			amd.registerAdapterFactory(this);			
			amd.eAdapters().add(entityIO);						
		}
		
		amd.setOverrideOf(overrideOf);
		amd.setName(overrideOf.getName());
		amd.setGetterMethodName(overrideOf.getGetterMethodName());
		amd.setSetterMethodName(overrideOf.getSetterMethodName());
		amd.setJavaType(overrideOf.getJavaType());
		amd.setOptional(overrideOf.isOptional());
		amd.setFetchType(overrideOf.getFetchType());
		
		if (embeddedAttribute == null) {
			if (amd.isNonPersistent()) {
				emd.getAttributeList().add(amd);
			}
			if (overrideOf.isPrimaryKey()) {
				if (!amd.isPrimaryKey()) {
					emd.getPrimaryKeyList().add(amd);
				}
			} else {
				if (amd.isPrimaryKey()) {
					emd.getPrimaryKeyList().remove(amd);
				}
			}
			if (overrideOf.isVersion()) {
				if (!amd.isVersion()) {
					emd.getVersionList().add(amd);
				}
			} else {
				if (amd.isVersion()) {
					emd.getVersionList().remove(amd);
				}
			}
		} else {
			if (amd.getEmbeddedIn() == null) {
				amd.setEmbeddedIn(embeddedAttribute);
			}
		}
		
		if (amd instanceof BasicAttribute) {
			updateModelFromMetaDataBasic(cls, (BasicAttribute)overrideOf, 
					override, metaDataChanged, relativePositionOffset);
		} else {
			throw new IllegalStateException("Not implemented: " + 
					overrideOf);
		}
		
		return true;
	}
	
	protected OrmColumn getDefaultColumn() {
		//return overrideOf.getColumn();
		TypeMetaData tmd = amd.getJavaType();
		if (tmd instanceof SimpleTypeMetaData) {
			return ((SimpleTypeMetaData)tmd).getColumn();
		} else {
			return null;
		}		
	}
	
	protected void updateModelFromMetaDataBasic(RClass cls,
			BasicAttribute overrideOf, AttributeOverride override, 
			boolean metaDataChanged, int relativePositionOffset) {
		GenericEntityModelManager mm = entityIO.getModelManager();
		BasicAttribute amd = (BasicAttribute)this.amd;

		OrmColumn dc = getDefaultColumn();
		OrmColumn c = amd.getColumn();
		OrmColumn oc = overrideOf.getColumn();

		if (dc == null) {
			if (c != null) {
				c.delete();
				amd.setColumn(null);
			}
			return;
		} 
		
		c = entityIO.ensureOrmColumn(c);
		String cd;
		if (override != null) {
			defColName = mm.getDefaultColumnName(amd);
			Column column = override.column();
			fillDefaults((AnnotationEx)column, dc);
			c.setJdbcType(column.jdbcType());
			c.setInsertable(column.insertable());
			if (mm.getVendorDef().isNumericType(c.getJdbcType())) {
				c.setLength(column.precision());		
			} else {
				c.setLength(column.length());			
			}
			c.setScale(column.scale());
			((AnnotationEx)column).setDefault("name", defColName);
			c.setName(column.name());
			c.setNullable(column.nullable());
			c.setUpdatable(column.updatable());
			cd = column.columnDefinition();
			if (cd.length() == 0) {
				c.setColumnDefinitionSpecified(false);
				cd = mm.getColumnDefinition(c);
			} else {
				c.setColumnDefinitionSpecified(true);
			}
		} else {
			if (oc != null) {
				oc.copyTo(c);
			} else {
				dc.copyTo(c);
				c.setName(mm.getDefaultColumnName(amd));
			}
			defColName = c.getName();
			c.setColumnDefinitionSpecified(false);
			cd = mm.getColumnDefinition(c);
		}
		c.setColumnDefinition(cd);
		c.setOriginalColumnDefinition(cd);
		mm.updateDatabaseType(c);
				
		String comment;
		int pos;
		if (embeddedAttribute == null) {
			comment = overrideOf.getEntityMetaData().getSchemaName() + "." + 
				overrideOf.getName();
			pos = oc == null ? 0 : oc.getRelativePositionInTable();
		} else {
			comment = embeddedAttribute.getEntityMetaData().getSchemaName() + 
				"." + embeddedAttribute.getName() + "." + overrideOf.getName();
			pos = relativePositionOffset;
		}
		c.setComment(comment);
		c.setRelativePositionInTable(pos);
		
		if (amd.getColumn() != c) {
			amd.setColumn(c);
		}
		OrmTable table = entityIO.getEntityMetaData().getTable();
		c.setTable(table);
		if (amd.isPrimaryKey()) {
			if (!c.isPrimaryKey()) {
				table.getPrimaryKeyList().add(c);
			}
		} else {
			if (c.isPrimaryKey()) {
				table.getPrimaryKeyList().remove(c);
			}
		}
	}
	
	/**
	 * Set defaults on the column annotation from dc. Excludes 
	 * columnDefinition.
	 */
	protected void fillDefaults(AnnotationEx column, OrmColumn dc) {
		column.setDefault("name", dc.getName());
		column.setDefault("nullable", dc.isNullable());
		column.setDefault("insertable", dc.isInsertable());
		column.setDefault("updatable", dc.isUpdatable());
		column.setDefault("length", dc.getLength());
		column.setDefault("precision", dc.getLength());
		column.setDefault("scale", dc.getScale());
		column.setDefault("jdbcType", dc.getJdbcType());
	}

	/**
	 * Update the annotations and/or XML from our attribute.
	 * If this method returns false then the attribute no longer exists and
	 * should be removed from the model.
	 */
	public boolean updateMetaDataFromModel(AttributeOverride override) {
		AttributeMetaData overrideOf = amd.getOverrideOf();
		if (overrideOf == null) {
			return false;
		}
		if (amd instanceof BasicAttribute) {
			updateMetaDataFromModelBasic(override);
		} else {
			throw new IllegalStateException("Not implemented: " + amd);
		}
		return true;
	}
		
	protected void updateMetaDataFromModelBasic(AttributeOverride override) {
		if (override == null) {
			return;
		}

		GenericEntityModelManager mm = entityIO.getModelManager();
		BasicAttribute amd = (BasicAttribute)this.amd;
		BasicAttribute overrideOf = (BasicAttribute)amd.getOverrideOf();
		
		OrmColumn dc;
		if (((AnnotationEx)override).getValueCount() == 0) {
			dc = overrideOf.getColumn();
		} else {
			dc = getDefaultColumn();
		}
		OrmColumn c = amd.getColumn();
		if (dc == null || c == null) {
			return;
		}
		
		AnnotationEx column = (AnnotationEx)override.column();
		fillDefaults(column, dc);
		column.set("jdbcType", c.getJdbcType());
		column.set("insertable", c.isInsertable());
		if (mm.getVendorDef().isNumericType(c.getJdbcType())) {
			column.set("precision", c.getLength());
			column.set("length", null);
		} else {
			column.set("length", c.getLength());	
			column.set("precision", null);		
		}
		column.setDefault("name", defColName);
		column.set("name", c.getName(), true);
		c.setName(((Column)column).name());
		column.set("nullable", c.isNullable());
		column.set("scale", c.getScale());
		column.set("updatable", c.isUpdatable());

		// if the columnDefinition has changed then apply the change to the
		// annotation - this may remove it if it now matches the default
		boolean colDefChanged = !c.getColumnDefinition().equals(
				c.getOriginalColumnDefinition());
		String stdColDef = mm.getColumnDefinition(c);
		if (colDefChanged) {
			column.setDefault("columnDefinition", stdColDef);			
			column.set("columnDefinition", c.getColumnDefinition(), true);
		}
		
		// if the annotation now has no columnDefinition value then put
		// the default value into the model
		if (!column.hasValue("columnDefinition")) {
			c.setColumnDefinition(stdColDef);
		}
		
		// reset the originalColumnDefinition attribute to detect more changes
		c.setOriginalColumnDefinition(c.getColumnDefinition());		
	}
	
}
