/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic.io.ast;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.FieldDeclaration;
import org.eclipse.jdt.core.dom.ITypeBinding;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.Type;
import org.eclipse.jdt.core.dom.TypeDeclaration;
import org.eclipse.jdt.core.dom.VariableDeclarationFragment;
import org.eclipse.jsr220orm.generic.Utils;
import org.eclipse.jsr220orm.generic.reflect.RClass;
import org.eclipse.jsr220orm.generic.reflect.RField;
import org.eclipse.jsr220orm.generic.reflect.RMethod;

/**
 * Implementation using an Eclipse AST tree to provide the data. 
 */
public class AstRClass extends AstRAnnotatedElement implements RClass {

	protected final TypeDeclaration typeDec;
	protected final ICompilationUnit typeCu;
	protected final CompilationUnit root;
	protected final String fullClassName;
	
	protected AstRField[] fields;
	protected Map<String, RField> fieldMap;
	protected AstRMethod[] methods;
	protected Map<String, RMethod> methodMap;
	
	protected int modifiers = -1;
	
	public AstRClass(AstRClassFactory factory, TypeDeclaration typeDec,
			String fullClassName, ICompilationUnit typeCu, 
			CompilationUnit root, ImportListHelper importListHelper) {
		super(factory, typeDec.modifiers(), importListHelper, 0);
		this.typeDec = typeDec;
		this.fullClassName = fullClassName;
		this.typeCu = typeCu;
		this.root = root;
		initAnnotations();
	}

	public TypeDeclaration getTypeDec() {
		return typeDec;
	}

	public ICompilationUnit getTypeCu() {
		return typeCu;
	}

	public CompilationUnit getRoot() {
		return root;
	}

	public String getName() {
		return fullClassName;
	}

	public String getSimpleName() {
		return typeDec.getName().getFullyQualifiedName();
	}
	
	public String getFullyQualifiedName() {
		return fullClassName;
	}

	public String getTypeName() {
		return getFullyQualifiedName();
	}
	
	public boolean isTypeEnum() {
		return false;
	}

	public boolean isTypeSerializable() {
		return false;
	}	

	public RField[] getDeclaredFields() {
		if (fields == null) {
			FieldDeclaration[] all = typeDec.getFields();
			int n = all.length;
			ArrayList list = new ArrayList(n);
			for (int i = 0; i < n; i++) {
				for (Iterator j = all[i].fragments().iterator(); j.hasNext(); ) {
					VariableDeclarationFragment varDecl = 
						(VariableDeclarationFragment)j.next();
					list.add(new AstRField(this, 
							(FieldDeclaration)varDecl.getParent(), varDecl, i));
				}
			}
			n = list.size();
			fields = new AstRField[n];
			list.toArray(fields);
		}
		return fields;
	}

	public RMethod[] getDeclaredMethods() {
		if (methods == null) {
			MethodDeclaration[] all = typeDec.getMethods();
			int n = all.length;
			methods = new AstRMethod[n];
			for (int i = 0; i < n; i++) {
				methods[i] = new AstRMethod(this, all[i], i);
			}
		}
		return methods;
	}
	
	public String getSuperclassName() {
		Type st = typeDec.getSuperclassType();
		if (st == null) {
			return null;
		} else {
			ITypeBinding resolveBinding = st.resolveBinding();
            if (resolveBinding != null) {
                return resolveBinding.getQualifiedName();
            }else{
                return null;
            }
		}
	}

	public RField getDeclaredField(String name) {
		if (fieldMap == null) {
			fieldMap = new HashMap();
			getDeclaredFields();
			for (int i = fields.length - 1; i >= 0; i--) {
				RField f = fields[i];
				fieldMap.put(f.getName(), f);
			}
		}
		return fieldMap.get(name);
	}

	public RMethod getDeclaredMethod(String name) {
		if (methodMap == null) {
			methodMap = new HashMap();
			getDeclaredMethods();
			for (int i = methods.length - 1; i >= 0; i--) {
				RMethod m = methods[i];
				methodMap.put(m.getName(), m);
			}
		}
		return methodMap.get(name);
	}

	public boolean isField() {
		return false;
	}

	public Map getLocation() {
		return Utils.createMarkerLocation(typeDec.getName());
	}

	public RClass getRClass() {
		return null;
	}

	public String[] getActualTypeArguments() {
		return null;
	}

}
