/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.ui.internal.diagram.er;

import org.eclipse.draw2d.AbstractConnectionAnchor;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;

public class ColumnConnectionAnchor extends AbstractConnectionAnchor {

    /**
     * Constructs a new ColumnConnectionAnchor.
     */
    protected ColumnConnectionAnchor() {
    }

    /**
     * Constructs a ColumnConnectionAnchor with the given <i>owner</i>figure.
     * 
     * @param owner
     *            The owner figure
     * @since 2.0
     */
    public ColumnConnectionAnchor(IFigure owner) {
        super(owner);
    }

    /**
     * Gets a Rectangle from {@link #getBox()} and returns the Point where a
     * line from the center of the Rectangle to the Point <i>reference</i>
     * intersects the Rectangle.
     * 
     * @param reference
     *            The reference point
     * @return The anchor location
     */
    public Point getLocation(Point reference) {
        Rectangle r = Rectangle.SINGLETON;
        r.setBounds(getBox());
        r.translate(-1, -1);
        r.resize(1, 1);

        getOwner().translateToAbsolute(r);
        float centerX = r.x + 0.5f * r.width;
        float centerY = r.y + 0.5f * r.height;

        int x = centerX < reference.x ? r.x + r.width : r.x;
        return new Point(x, Math.round(centerY));
    }

    /**
     * Returns the bounds of this ColumnConnectionAnchor's owner. Subclasses can override
     * this method to adjust the box the anchor can be placed on. For instance,
     * the owner figure may have a drop shadow that should not be included in
     * the box.
     * 
     * @return The bounds of this ColumnConnectionAnchor's owner
     * @since 2.0
     */
    protected Rectangle getBox() {
        return getOwner().getBounds();
    }

    /**
     * Returns the anchor's reference point. In the case of the ColumnConnectionAnchor,
     * this is the center of the anchor's owner.
     * 
     * @return The reference point
     */
    public Point getReferencePoint() {
        Point ref = getBox().getCenter();
        getOwner().translateToAbsolute(ref);
        return ref;
    }

    /**
     * Returns <code>true</code> if the other anchor has the same owner and
     * box.
     * 
     * @param obj
     *            the other anchor
     * @return <code>true</code> if equal
     */
    public boolean equals(Object obj) {
        if (obj instanceof ColumnConnectionAnchor) {
            ColumnConnectionAnchor other = (ColumnConnectionAnchor) obj;
            return other.getOwner() == getOwner()
                    && other.getBox().equals(getBox());
        }
        return false;
    }

    /**
     * The owning figure's hashcode is used since equality is approximately
     * based on the owner.
     * 
     * @return the hash code.
     */
    public int hashCode() {
        if (getOwner() != null)
            return getOwner().hashCode();
        else
            return super.hashCode();
    }
}
