/***********************************************************************
* Name:
*        dapiqry.c
*
* Function:
*        Functions invoked via menu selections
*
* Environment:
*        This is a PLATFORM-INDEPENDENT source file. As such it may
*        contain no dependencies on any specific operating system
*        environment or hardware platform.
*
* Description:
*        The file contains the functions that are invoked from the
*        menu processing code and tables.
***********************************************************************/
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "dsmapitd.h"        /* Standard DSM API header.      */
#include "dsmapifp.h"
#include "dapitype.h"        /* Sample API data types.        */
#include "dsmrc.h"           /* DSM API return codes.         */
#include "dapiproc.h"

#include "dapint64.h"        /* 64 bit integer functions.     */
#include "dapiutil.h"        /* Various utility functions.    */

/*----------------------------------------------------------------------+
| Private typedefs
.----------------------------------------------------------------------*/
typedef struct
{
   char                 *fs;
   char                 *hl;
   char                 *ll;
   dsUint8_t             objType;
} dsmObjNameP;

/*----------------------------------------------------------------------+
| Linked list structure to store restore/retrieve queries
.----------------------------------------------------------------------*/
typedef struct SGetInfo
{
   struct SGetInfo      *next;            /* Next item in list, null if last  */
   dsStruct64_t          item64;          /* Number of item in list.          */
   dsmObjNameP          *objName;         /* Name of object                   */
   ObjID                 objId;           /* Id of object for restore.        */
   dsUint32_t            copyGroup;       /* Needed to delete object          */
   dsUint160_t           restoreOrderExt; /* Sort field for optimum restore.  */
   dsUint8_t             mediaClass;      /* Item offline/online/other?       */
   dsUint16_t            objInfolen;      /* Length of buffer below.          */
   char                 *objInfo;         /* Not a null terminated string!    */
} GetInfo;

/*----------------------------------------------------------------------+
| Linked list structure to store fs queries.  Not in 64bit format!
.----------------------------------------------------------------------*/
typedef struct SfsInfo
{
   struct SfsInfo       *next;            /* Next item in list, null if last  */
   dsUint32_t            item;            /* Number of item in list.          */
   char                 *fsName;          /* Name of fs                       */
} fsInfo;

/*----------------------------------------------------------------------+
| Structure to maintain data on current state of validation check
.----------------------------------------------------------------------*/
typedef struct
{
   char                 *seed;             /* Pointer to the seed string.          */
   dsUint16_t            seed_len;         /* Length of seed string.               */
   char                 *seed_cursor;      /* Position of validate in seed string. */
   dsStruct64_t          total_sz64;       /* Total size thus far of verified obj. */
   dsStruct64_t          orig_sz;          /* Orig size of obj as sent by api.     */
   dsUint32_t            seed_string_size ;/* Size of entire seed string           */
   dsBool_t              first_time;       /* First time checking for partial obj? */
} seedValidate;

/*----------------------------------------------------------------------+
| Private routines
.----------------------------------------------------------------------*/
static void       QryDateToStr         (char *s,dsmDate date);
static dsInt16_t  QryAddGetLlist       (dsUint16_t get_type,
                                        dsStruct64_t qry_item64,
                                        GetInfo **llHead,
                                        GetInfo **llTail,
                                        char *buff);
static dsInt16_t  QryFreeGetLlist      (GetInfo **llHead,
                                        GetInfo **llTail);
static dsInt16_t  QryFreeGetLlistItem  (GetInfo **item);
static dsInt16_t  QryDoGet             (dsUint16_t get_type,
                                        GetInfo *llHead,
                                        GetInfo *llTail,
                                        dsStruct64_t  items64,
                                        dsBool_t  mountWait);
static dsInt16_t  QryDoDelete          (dsUint16_t delete_type,
                                        GetInfo *llHead,
                                        GetInfo *llTail,
                                        dsStruct64_t  items64);
static dsInt16_t  QryDoRestoreObjs     (dsUint16_t get_type,
                                        GetInfo *llStart,
                                        char *rest_ibuff,
                                        dsUint32_t items,
                                        dsBool_t mountWait,
                                        char *part_pbuff);
static dsInt16_t  Qry_Validate_Data    (DataBlk *dataBlk,
                                        seedValidate *seedV,
                                        dsUint32_t item,
                                        char *part_pbuff);
static dsInt16_t  QryBeginTxn          (dsUint32_t  dsmHandle, /* DSM API sess handle  */
                                        dsUint32_t  *txn_objs);/* Objects in this txn. */
static void       QryPrintMediaClass   (dsUint8_t mediaClass);
static void       QryPrintObjType      (dsUint8_t objType);
static void       QryPrintObjInfo      (dsUint16_t objInfolen, char *objInfo);
static void       QryPrintObjState     (dsUint8_t objState);
static void       QryPrintCopySer      (dsUint8_t copySer);
static void       QryPrintCopyMode     (dsUint8_t copyMode);
static dsInt16_t  QryAddFsLlist        (dsUint32_t  qry_item32,
                                        fsInfo **llHead,
                                        fsInfo **llTail,
                                        qryRespFSData *fsResp);
static dsInt16_t  QryFreeFsLlist       (fsInfo **llHead,
                                        fsInfo **llTail);
static dsInt16_t  QryDoFsDelete        (fsInfo *llHead,
                                        fsInfo *llTail,
                                        dsUint32_t  items32);
static dsInt16_t  QryMakeSeed          (seedValidate  *seedV,
                                        char *seed_str );

static void       SortObjects          (GetInfo *, char *, int);
static int        SortRestoreOrder     (const void *, const void  *);
static void       xStringToDate        (char *s, dsmDate *date);
static void       xStringToTime        (char *s, dsmDate *date);
static void       DateToDate           (dsmDate fromDate, dsmDate *toDate);


/*----------------------------------------------------------------------+
| Public routines
+----------------------------------------------------------------------*/
extern dsInt8_t no_restore, no_retrieve;

/*----------------------------------------------------------------------+
| Name:    perform_setup()
|
| Action:  Take the information passed in the dialog to this function,
|          create the call to dsmSetUp function.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_setup(dialog *sel_dialog,dsUint16_t dummy)
   {
   dialog      *dlg;
   dsUint32_t  i;
   dsInt16_t   rc;
   envSetUp    dsmEnvSetUp;

   memset(&dsmEnvSetUp, 0x00, sizeof(envSetUp));
   dsmEnvSetUp.stVersion = envSetUpVersion;
   dsmEnvSetUp.argv = globalArgv;

   /* First pull all values out of the passed dialog for our use.  */
   i = 0;
   dlg = &sel_dialog[i];
   while (dlg->item_type != DSMAPI_END)
   {
      switch (dlg->item_type)
      {
          case DSMAPI_DSMIDIR :
            if (!*dlg->item_buff)
            {
               strcpy(dsmEnvSetUp.dsmiDir,"");
            }
            else 
               strcpy(dsmEnvSetUp.dsmiDir,dlg->item_buff);
            break;
          
          case DSMAPI_DSMICONFIG :
            if (!*dlg->item_buff)
            {
               strcpy(dsmEnvSetUp.dsmiConfig,"");
            }
            else 
               strcpy(dsmEnvSetUp.dsmiConfig,dlg->item_buff);
            break;
         
          case DSMAPI_DSMILOG :
            if (!*dlg->item_buff)
            {
               strcpy(dsmEnvSetUp.dsmiLog,"");
            }
            else 
               strcpy(dsmEnvSetUp.dsmiLog,dlg->item_buff);
            break;
         
          case DSMAPI_LOGNAME :
            if (!*dlg->item_buff)
            {
               strcpy(dsmEnvSetUp.logName,"");
            }
            else
               strcpy(dsmEnvSetUp.logName,dlg->item_buff);
            break;


         default :
            ;
      }
      i++;
      dlg = &sel_dialog[i];
   }
   
   rc = dsmSetUp(DSM_SINGLETHREAD, &dsmEnvSetUp);
   if ( rc |= DSM_RC_OK)
   {
      printf("\n*** dsmSetUp failed: \n");
      rcApiOut(0, rc);
      dsmCleanUp(DSM_SINGLETHREAD);
   }
   else
      printf("Finished dsmSetUp successfully\n");
   return rc;
}

/*----------------------------------------------------------------------+
| Name:    perform_del_access()
|
| Action:  Take the information passed in the dialog to this function,
|          create the call to dsmDeleteaccess function.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_del_access(dialog *sel_dialog,dsUint16_t dummy)
{
   dialog       *dlg;
   dsUint32_t   i;
   dsInt16_t    rc;
   dsStruct64_t ruleNumber;

   if (!dsmHandle)
   {
      printf("You must signon before performing this action.\n");
      return(0);
   }


   /* First pull all values out of the passed dialog for our use.  */
   i = 0;
   dlg = &sel_dialog[i];
   while (dlg->item_type != DSMAPI_END)
   {
      switch (dlg->item_type)
      {
        case DSMAPI_ACCESSRULE :
           ruleNumber = dChtoI64(dlg->item_buff,10);   
           break;

         default :
            ;
      }
      i++;
      dlg = &sel_dialog[i];
   }
   
   rc = dsmDeleteAccess(dsmHandle, ruleNumber.lo);
   if ( rc |= DSM_RC_OK)
   {
      printf("\n*** dsmDeleteAccess failed: \n");
      rcApiOut(dsmHandle, rc);
   }
   else
      printf(" Finished Deleting Access Rule %d successfully\n",ruleNumber.lo);
  return rc;
}


/*----------------------------------------------------------------------+
| Name:    perform_log_event()
|
| Action:  Take the information passed in the dialog to this function,
|          create the call to dsmLogEvent function.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_log_event(dialog *sel_dialog,dsUint16_t log_event1)
{
   dialog         *dlg;
   dsUint32_t      i;
   dsInt16_t       rc;
   dsmLogExIn_t    logExIn;
   dsmLogExOut_t   logExOut;

   memset(&logExIn, 0x00, sizeof(dsmLogExIn_t));
   memset(&logExOut, 0x00, sizeof(dsmLogExOut_t));

   logExIn.message = (char *)malloc(DSM_MAX_RC_MSG_LENGTH);

   if (!dsmHandle)
   {
      printf("You must signon before performing this action.\n");
      return(0);
   }


   /* First pull all values out of the passed dialog for our use.  */
   i = 0;
   dlg = &sel_dialog[i];
   while (dlg->item_type != DSMAPI_END)
   {
      switch (dlg->item_type)
      {
         case DSMAPI_LOGEVENT :
              switch (*dlg->item_buff)
              {
               case 'L':
               case 'l':
                  logExIn.logType = logLocal;
                  break;
               case 'S':
               case 's':
                  logExIn.logType = logServer;
                  break;
               case 'B':
               case 'b':
                  logExIn.logType = logBoth;
                  break;
               default :
                  logExIn.logType = logServer;
               }
            break;

         case DSMAPI_APPMSG :

            if (dlg->item_buff[0] != 0)
               {
                strcpy(logExIn.appMsgID,dlg->item_buff);
               }
            else
               {
                strcpy(logExIn.appMsgID, "SMP0001");
               }
            break;


         case DSMAPI_LOGSEV :
              switch (*dlg->item_buff)
              {
               case 'I':
               case 'i':
                  logExIn.severity = logSevInfo;
                  break;
               case 'W':
               case 'w':
                  logExIn.severity = logSevWarning;
                  break;
               case 'E':
               case 'e':
                  logExIn.severity = logSevError;
                  break;
               case 'S':
               case 's':
                  logExIn.severity = logSevSevere;
                  break;
               default :
                  logExIn.severity = logSevInfo;
               }
            break;

         case DSMAPI_LOGMESSAGE :
           strcpy(logExIn.message,dlg->item_buff);
           break;

         default :
            ;
      }
      i++;
      dlg = &sel_dialog[i];
   }


   rc = dsmLogEventEx(dsmHandle, &logExIn, &logExOut);
   if ( rc |= DSM_RC_OK)
   {
      printf("\n*** dsmLogEvent failed: \n");
      rcApiOut(dsmHandle, rc);
   }
   else
      printf(" Finished sending a log message successfully\n");
   free(logExIn.message);
   return rc;
}

/*----------------------------------------------------------------------+
| Name:    perform_set_access()
|
| Action:  Take the information passed in the dialog to this function,
|          create the call to dsmLogEvent function.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_set_access(dialog *sel_dialog,dsUint16_t set_access1)
{
   dialog        *dlg;
   dsUint32_t     i;
   dsInt16_t      rc;
   char          *owner;
   char          *node;
   dsmAccessType accessType;
   dsmObjName    objName; 

   if (!dsmHandle)
   {
      printf("You must signon before performing this action.\n");
      return(0);
   }


   /* First pull all values out of the passed dialog for our use.  */
   i = 0;
   dlg = &sel_dialog[i];
   while (dlg->item_type != DSMAPI_END)
   {
      switch (dlg->item_type)
      {
         case DSMAPI_FILESPACE :
            if ((strlen(dlg->item_buff) > DSM_MAX_FSNAME_LENGTH)
               || !*dlg->item_buff)
            {
               strcpy(objName.fs,"*");
            }
            else strcpy(objName.fs,dlg->item_buff);
            break;
         case DSMAPI_HIGHLEVEL :
            if ((strlen(dlg->item_buff) > DSM_MAX_HL_LENGTH)
               || !*dlg->item_buff)
            {
              objName.hl[0] = '*';  /* don't use delimiter - optional */
              objName.hl[1] = 0x00;
            }
             else strcpy(objName.hl,dlg->item_buff);
             break;
         case DSMAPI_LOWLEVEL :
            if ((strlen(dlg->item_buff) > DSM_MAX_LL_LENGTH)
               || !*dlg->item_buff)
            {
              objName.ll[0] = dsmSessInfo.hldelim;
              objName.ll[1] = '*';
              objName.ll[2] = 0x00;
            }
            else strcpy(objName.ll,dlg->item_buff);
            break;
          
         case DSMAPI_OBJTYPE :
            switch (*dlg->item_buff)
            {
               case 'B':
               case 'b':
                     accessType = atBackup; break;
               case 'A':
               case 'a':
                     accessType = atArchive; break;
               default :
                     accessType = atBackup; break;
                  
            }

         case DSMAPI_NODENAME :
              node = dlg->item_buff;
              break;
         case DSMAPI_OWNER :
              owner = dlg->item_buff;

         default :
            ;
      }
      i++;
      dlg = &sel_dialog[i];
   }
   
   rc = dsmSetAccess(dsmHandle, accessType, &objName, node, owner);
   if ( rc |= DSM_RC_OK)
   {
      printf("\n*** dsmSetAccess failed: \n");
      rcApiOut(dsmHandle, rc);
   }
   else
      printf(" Finished Set Access successfully\n");

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    perform_update_obj()
|
| Action:  Take the information passed in the dialog to this function,
|          build the proper query, issue it, and display results to the
|          screen.  This handles 3 query types:  Backup, fast Backup,
|          and archive queries.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_update_obj(dialog *sel_dialog,dsUint16_t parm_get_type)
{

   dsUint16_t     i;
   dsBool_t       done = bFalse;
   dialog        *dlg;
   dsmObjName     objName;      /* Big object for some compiler stacks......*/
   dsInt16_t      rc;
   char          *s;
   char           splat[2];
   char           objInfo[DSM_MAX_OBJINFO_LENGTH];
   dsBool_t       updSize = bFalse;
   dsBool_t       mountWait = bFalse;
   dsStruct64_t   filesize64;
   dsUint32_t     updAction =0;
   ObjAttr        objAttr;

   memset(&objName,0x00,sizeof(objName));
   memset(&objAttr,0x00,sizeof(objAttr));


   strcpy(splat, "*");

   if (!dsmHandle)
   {
      printf("You must signon before performing this action.\n");
      return(0);
   }

   if (!(s = (char*)malloc(sizeof(dsmObjName))))
   {
      printf("Memory alloc failed in perform_update_obj.\n");
      return RC_NO_MEMORY;
   }



   /*-----------------------------------------------------------------------
   |  First pull all values out of the passed dialog for our use.
   .-----------------------------------------------------------------------*/
   i = 0;
   dlg = &sel_dialog[i];
   while (dlg->item_type != DSMAPI_END)
   {
      switch (dlg->item_type)
      {
         case DSMAPI_FILESPACE :
            if ((strlen(dlg->item_buff) > DSM_MAX_FSNAME_LENGTH)
               || !*dlg->item_buff)
            {
               strcpy(objName.fs,"*");
            }
            else strcpy(objName.fs,dlg->item_buff);
            break;
         case DSMAPI_HIGHLEVEL :
            if ((strlen(dlg->item_buff) > DSM_MAX_HL_LENGTH)
               || !*dlg->item_buff)
            {
              objName.hl[0] = '*';  /* don't use delimiter - optional */
              objName.hl[1] = 0x00;
            }
             else strcpy(objName.hl,dlg->item_buff);
             break;
         case DSMAPI_LOWLEVEL :
            if ((strlen(dlg->item_buff) > DSM_MAX_LL_LENGTH)
               || !*dlg->item_buff)
            {
              objName.ll[0] = dsmSessInfo.hldelim;
              objName.ll[1] = '*';
              objName.ll[2] = 0x00;
            }
            else strcpy(objName.ll,dlg->item_buff);
            break;
         case DSMAPI_OBJOWNER :
            if ((strlen(dlg->item_buff) > DSM_MAX_OWNER_LENGTH)
               || !*dlg->item_buff)
            {
               strcpy(objAttr.owner,"");
            }
            else strcpy(objAttr.owner,dlg->item_buff);
            break;
         case DSMAPI_OBJTYPE :
            switch (*dlg->item_buff)
            {
               case 'F':
               case 'f':
                  objName.objType = DSM_OBJ_FILE; break;
               case 'D':
               case 'd':
                  objName.objType = DSM_OBJ_DIRECTORY; break;
               case 'Q':
               case 'q':
                    objName.objType = DSM_OBJ_QRY_FILES_AND_DIRS; break;
               case 'N':
               case 'n':
                    objName.objType = DSM_OBJ_NQR_FILES_AND_DIRS; break;    
               case 'W':
               case 'w':
                  objName.objType = DSM_OBJ_WILDCARD; break;
               default :
                  objName.objType = DSM_OBJ_FILE;
            }
            break;
         case DSMAPI_MNTWAIT :
            switch (*dlg->item_buff)
            {
               case 'Y':
               case 'y':
                  mountWait = bTrue;              break;
               case 'N':
               case 'n':
                  mountWait = bFalse;             break;
               default :
                  mountWait = bFalse;             break;
            }
            break;
         case DSMAPI_UPDOWNER :
            switch (*dlg->item_buff)
            {
               case 'Y':
               case 'y':
                     updAction = updAction | DSM_BACKUPD_OWNER;         break;
               case 'N':
               case 'n':
               default :
                      break;
            }
            break;
         case DSMAPI_UPDOBJINFO :
            switch (*dlg->item_buff)
            {
               case 'Y':
               case 'y':
                  updAction = updAction | DSM_BACKUPD_OBJINFO;
                  updSize = bTrue;
                  break;
               case 'N':
               case 'n':
               default :
                    break;
            }
            break;

         case DSMAPI_OBJINFO :
            if ((strlen(dlg->item_buff) > DSM_MAX_OBJINFO_LENGTH)
               || !*dlg->item_buff)
            {
               strcpy(objInfo,"");
            }
            else strcpy(objInfo,dlg->item_buff);
            objAttr.objInfo = objInfo;
            objAttr.objInfoLength = strlen(objAttr.objInfo) + 1;
            break;
         case  DSMAPI_OBSIZEEST :
            filesize64 = dChtoI64(dlg->item_buff,10);
            if (updSize){
               objAttr.sizeEstimate.hi = filesize64.hi;
               objAttr.sizeEstimate.lo = filesize64.lo;
            }
            else
            {
               objAttr.sizeEstimate.hi = 0;
               objAttr.sizeEstimate.lo = 0;
            }

            break;

         default :
            ;
      }
      i++;
      dlg = &sel_dialog[i];
   }

   objAttr.stVersion = ObjAttrVersion;

   if ((rc = dsmUpdateObj(dsmHandle, stBackup, NULL, &objName,
                           &objAttr, updAction))!= DSM_RC_OK)
   {
      printf("*** dsmUpdateObj failed: \n");
      rcApiOut(dsmHandle, rc);
   }
   else
      printf("\n Finished updating Object Info successfully\n");

    free(s);
    return rc;
}


/*----------------------------------------------------------------------+
| Name:    perform_update_objEx()
|
| Action:  Take the information passed in the dialog to this function,
|          build the proper query, issue it, and display results to the
|          screen.  This handles 3 query types:  Backup, fast Backup,
|          and archive queries.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_update_objEx(dialog *sel_dialog,dsUint16_t parm_get_type)
{

   dsUint16_t     i;
   dsBool_t       done = bFalse;
   dialog        *dlg;
   dsmObjName     objName;      /* Big object for some compiler stacks......*/
   dsInt16_t      rc;
   char          *s;
   char           splat[2];
   char           objInfo[DSM_MAX_OBJINFO_LENGTH];
   dsBool_t       updSize = bFalse;
   dsBool_t       mountWait = bFalse;
   dsStruct64_t   filesize64;
   dsUint32_t     updAction =0;
   ObjAttr        objAttr;
   char           *descr;
   dsmSendType    sendType;
   dsStruct64_t  objIdHi64, objIdLo64, objId;
   dsmUpdateObjExIn_t   dsmUpdateObjExIn;
   dsmUpdateObjExOut_t  dsmUpdateObjExOut;
      

   memset(&dsmUpdateObjExIn, 0x00, sizeof(dsmUpdateObjExIn_t));
   memset(&dsmUpdateObjExOut, 0x00, sizeof(dsmUpdateObjExOut_t));
   memset(&objAttr, 0x00, sizeof(ObjAttr));
   


   memset(&objName,0x00,sizeof(objName));
   memset(&objAttr,0x00,sizeof(objAttr));
   objId.hi = 0;
   objId.lo= 0;

   strcpy(splat, "*");

   if (!dsmHandle)
   {
      printf("You must signon before performing this action.\n");
      return(0);
   }

   if (!(s = (char*)malloc(sizeof(dsmObjName))))
   {
      printf("Memory alloc failed in perform_update_obj.\n");
      return RC_NO_MEMORY;
   }



   /*-----------------------------------------------------------------------
   |  First pull all values out of the passed dialog for our use.
   .-----------------------------------------------------------------------*/
   i = 0;
   dlg = &sel_dialog[i];
   while (dlg->item_type != DSMAPI_END)
   {
      switch (dlg->item_type)
      {
         case DSMAPI_FILESPACE :
            if ((strlen(dlg->item_buff) > DSM_MAX_FSNAME_LENGTH)
               || !*dlg->item_buff)
            {
               strcpy(objName.fs,"*");
            }
            else strcpy(objName.fs,dlg->item_buff);
            break;
         case DSMAPI_HIGHLEVEL :
            if ((strlen(dlg->item_buff) > DSM_MAX_HL_LENGTH)
               || !*dlg->item_buff)
            {
              objName.hl[0] = '*';  /* don't use delimiter - optional */
              objName.hl[1] = 0x00;
            }
             else strcpy(objName.hl,dlg->item_buff);
             break;
         case DSMAPI_LOWLEVEL :
            if ((strlen(dlg->item_buff) > DSM_MAX_LL_LENGTH)
               || !*dlg->item_buff)
            {
              objName.ll[0] = dsmSessInfo.hldelim;
              objName.ll[1] = '*';
              objName.ll[2] = 0x00;
            }
            else strcpy(objName.ll,dlg->item_buff);
            break;
         case DSMAPI_OBJOWNER :
            if ((strlen(dlg->item_buff) > DSM_MAX_OWNER_LENGTH)
               || !*dlg->item_buff)
            {
               strcpy(objAttr.owner,"");
            }
            else strcpy(objAttr.owner,dlg->item_buff);
            break;

        case DSMAPI_REPOSITORY :
            switch (*dlg->item_buff)
            {
               case 'B':
               case 'b':
                  sendType = stBackup; break;
               case 'A':
               case 'a':
                  sendType = stArchive; break;
               default :
                  sendType = stBackup;
            }

         case DSMAPI_OBJTYPE :
            switch (*dlg->item_buff)
            {
               case 'F':
               case 'f':
                  objName.objType = DSM_OBJ_FILE; break;
               case 'D':
               case 'd':
                  objName.objType = DSM_OBJ_DIRECTORY; break;
               case 'Q':
               case 'q':
                    objName.objType = DSM_OBJ_QRY_FILES_AND_DIRS; break;
               case 'N':
               case 'n':
                    objName.objType = DSM_OBJ_NQR_FILES_AND_DIRS; break;    
               case 'W':
               case 'w':
                  objName.objType = DSM_OBJ_WILDCARD; break;
               default :
                  objName.objType = DSM_OBJ_FILE;
            }
            break;

         case DSMAPI_OBJIDHI :
            objIdHi64 = dChtoI64(dlg->item_buff,10);
            objId.hi = objIdHi64.lo;
            break;
         case DSMAPI_OBJIDLO :
            objIdLo64 = dChtoI64(dlg->item_buff,10);
            objId.lo = objIdLo64.lo;
            break;

         case DSMAPI_UPDOWNER :
            switch (*dlg->item_buff)
            {
               case 'Y':
               case 'y':
                  if (sendType == stBackup)
                     updAction = updAction | DSM_BACKUPD_OWNER;   
                  else
                     updAction = updAction | DSM_ARCHUPD_OWNER;   
                     break;
               case 'N':
               case 'n':
               default :
                      break;
            }
            break;
         case DSMAPI_UPDOBJINFO :
            switch (*dlg->item_buff)
            {
               case 'Y':
               case 'y':
                  if (sendType == stBackup)
                     updAction = updAction | DSM_BACKUPD_OBJINFO;
                  else
                     updAction = updAction | DSM_ARCHUPD_OBJINFO;
                  updSize = bTrue;
                  break;
               case 'N':
               case 'n':
               default :
                    break;
            }
            break;

        case DSMAPI_UPDDESCR :
         switch (*dlg->item_buff)
         {
            case 'Y':
            case 'y':
               if (sendType == stBackup)
                  updAction = updAction;
               else      
                  updAction = updAction | DSM_ARCHUPD_DESCR;
               updSize = bTrue;
               break;
            case 'N':
            case 'n':
            default :
                 break;
         }
         break;

         case DSMAPI_OBJINFO :
            if ((strlen(dlg->item_buff) > DSM_MAX_OBJINFO_LENGTH)
               || !*dlg->item_buff)
            {
               strcpy(objInfo,"");
            }
            else strcpy(objInfo,dlg->item_buff);
            objAttr.objInfo = objInfo;
            objAttr.objInfoLength = strlen(objAttr.objInfo) + 1;
            break;
         case  DSMAPI_OBSIZEEST :
            filesize64 = dChtoI64(dlg->item_buff,10);
            if (updSize){
               objAttr.sizeEstimate.hi = filesize64.hi;
               objAttr.sizeEstimate.lo = filesize64.lo;
            }
            else
            {
               objAttr.sizeEstimate.hi = 0;
               objAttr.sizeEstimate.lo = 0;
            }

            break;

         case DSMAPI_OBJDESCR :
               if ((strlen(dlg->item_buff) > DSM_MAX_DESCR_LENGTH)
                  || !*dlg->item_buff)
               {
                  descr = splat;
               }
               else 
                  descr = dlg->item_buff;
               break;

         default :
            ;
      }
      i++;
      dlg = &sel_dialog[i];
   }

   dsmUpdateObjExIn.stVersion  = dsmUpdateObjExInVersion;         
   dsmUpdateObjExIn.dsmHandle  = dsmHandle;
   dsmUpdateObjExIn.sendType   = sendType;
   dsmUpdateObjExIn.descrP     = descr;
   dsmUpdateObjExIn.objNameP   = &objName;
   dsmUpdateObjExIn.objAttrPtr = &objAttr;
   dsmUpdateObjExIn.objUpdAct  = updAction;
   dsmUpdateObjExIn.archObjId  = objId;
          
      

   if ((rc = dsmUpdateObjEx(&dsmUpdateObjExIn, &dsmUpdateObjExOut)) != DSM_RC_OK) 
   {
      printf("*** dsmUpdateObjEx failed: \n");
      rcApiOut(dsmHandle, rc);
   }
   else
      printf("\n Finished updating Object Info successfully\n");
   
   free(s);
   return rc;
}


/*----------------------------------------------------------------------+
| Name:    perform_update_arc()
|
| Action:  Take the information passed in the dialog to this function,
|          build the proper query, issue it, and display results to the
|          screen.  This handles 3 query types:  Backup, fast Backup,
|          and archive queries.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_update_arc(dialog *sel_dialog,dsUint16_t parm_get_type)
{

   dsUint16_t   i;
   dsBool_t     done = bFalse;
   dialog      *dlg;
   dsmObjName   objName;      /* Big object for some compiler stacks......*/
   dsInt16_t    rc;
   char        *s;
   char         splat[2];
   char         objInfo[DSM_MAX_OBJINFO_LENGTH];
   dsBool_t     updSize = bFalse;
   dsBool_t     mountWait = bFalse;
   dsStruct64_t filesize64;
   dsUint32_t   updAction =0;
   ObjAttr      objAttr;
   char        *descr;

   memset(&objName,0x00,sizeof(objName));
   memset(&objAttr,0x00,sizeof(objAttr));


   strcpy(splat, "*");

   if (!dsmHandle)
   {
   printf("You must signon before performing this action.\n");
   return(0);
   }

   if (!(s = (char*)malloc(sizeof(dsmObjName))))
   {
   printf("Memory alloc failed in perform_update_obj.\n");
   return RC_NO_MEMORY;
   }
   /*-----------------------------------------------------------------------
   |  First pull all values out of the passed dialog for our use.
   .-----------------------------------------------------------------------*/
   i = 0;
   dlg = &sel_dialog[i];
   while (dlg->item_type != DSMAPI_END)
   {
      switch (dlg->item_type)
      {
         case DSMAPI_FILESPACE :
            if ((strlen(dlg->item_buff) > DSM_MAX_FSNAME_LENGTH)
               || !*dlg->item_buff)
            {
               strcpy(objName.fs,"*");
            }
            else strcpy(objName.fs,dlg->item_buff);
            break;
         case DSMAPI_HIGHLEVEL :
            if ((strlen(dlg->item_buff) > DSM_MAX_HL_LENGTH)
               || !*dlg->item_buff)
            {
              objName.hl[0] = '*';  /* don't use delimiter - optional */
              objName.hl[1] = 0x00;
            }
             else strcpy(objName.hl,dlg->item_buff);
             break;
         case DSMAPI_LOWLEVEL :
            if ((strlen(dlg->item_buff) > DSM_MAX_LL_LENGTH)
               || !*dlg->item_buff)
            {
              objName.ll[0] = dsmSessInfo.hldelim;
              objName.ll[1] = '*';
              objName.ll[2] = 0x00;
            }
            else strcpy(objName.ll,dlg->item_buff);
            break;
         case DSMAPI_OBJOWNER :
            if ((strlen(dlg->item_buff) > DSM_MAX_OWNER_LENGTH)
               || !*dlg->item_buff)
            {
               strcpy(objAttr.owner,"");
            }
            else strcpy(objAttr.owner,dlg->item_buff);
            break;

         case DSMAPI_OBJTYPE :
            switch (*dlg->item_buff)
            {
               case 'F':
               case 'f':
                   objName.objType = DSM_OBJ_FILE; break;
               case 'D':
               case 'd':
                   objName.objType = DSM_OBJ_DIRECTORY; break;
               case 'Q':
               case 'q':
                   objName.objType = DSM_OBJ_QRY_FILES_AND_DIRS; break;
               case 'N':
               case 'n':
                   objName.objType = DSM_OBJ_NQR_FILES_AND_DIRS; break;    
               case 'W':
               case 'w':
                   objName.objType = DSM_OBJ_WILDCARD; break;
               default :
                   objName.objType = DSM_OBJ_FILE;
            }
            break;
         case DSMAPI_MNTWAIT :
            switch (*dlg->item_buff)
            {
               case 'Y':
               case 'y':
                  mountWait = bTrue;              break;
               case 'N':
               case 'n':
                  mountWait = bFalse;             break;
               default :
                  mountWait = bFalse;             break;
            }
            break;
         case DSMAPI_UPDOWNER :
            switch (*dlg->item_buff)
            {
               case 'Y':
               case 'y':
                     updAction = updAction | DSM_ARCHUPD_OWNER;         
                     break;
               case 'N':
               case 'n':
               default :
                      break;
            }
            break;
          case DSMAPI_UPDOBJINFO :
               switch (*dlg->item_buff)
               {
                  case 'Y':
                  case 'y':
                     updAction = updAction | DSM_ARCHUPD_OBJINFO;
                     updSize = bTrue;
                     break;
                  case 'N':
                  case 'n':
                  default :
                       break;
               }
               break;
          case DSMAPI_UPDDESCR :
               switch (*dlg->item_buff)
               {
                  case 'Y':
                  case 'y':
                     updAction = updAction | DSM_ARCHUPD_DESCR;
                     updSize = bTrue;
                     break;
                  case 'N':
                  case 'n':
                  default :
                       break;
               }
               break;

         case DSMAPI_OBJINFO :
               if ((strlen(dlg->item_buff) > DSM_MAX_OBJINFO_LENGTH)
                  || !*dlg->item_buff)
               {
                  strcpy(objInfo,"");
               }
               else strcpy(objInfo,dlg->item_buff);
               objAttr.objInfo = objInfo;
               objAttr.objInfoLength = strlen(objAttr.objInfo) + 1;
               break;

         case  DSMAPI_OBSIZEEST :
               filesize64 = dChtoI64(dlg->item_buff,10);
               if (updSize){
                  objAttr.sizeEstimate.hi = filesize64.hi;
                  objAttr.sizeEstimate.lo = filesize64.lo;
               }
               else
               {
                  objAttr.sizeEstimate.hi = 0;
                  objAttr.sizeEstimate.lo = 0;
               }

               break;
         case DSMAPI_OBJDESCR :
               if ((strlen(dlg->item_buff) > DSM_MAX_DESCR_LENGTH)
                  || !*dlg->item_buff)
               {
                  descr = splat;
               }
               else 
                  descr = dlg->item_buff;
               break;

         default :
            ;
      }
      i++;
      dlg = &sel_dialog[i];
   }

   objAttr.stVersion = ObjAttrVersion;

   if ((rc = dsmUpdateObj(dsmHandle, stArchive, descr, &objName,
                        &objAttr, updAction))!= DSM_RC_OK)
   {
      printf("*** dsmUpdateObj for archive failed: \n");
      rcApiOut(dsmHandle, rc);
   }
   else
      printf("\n Finished updating Archive Object Info successfully\n");

   free(s);
   return rc;
}

/*----------------------------------------------------------------------+
| Name:    perform_update_fs()
|
| Action:  Take the information passed in the dialog to this function,
|          build the proper query, issue it, and display results to the
|          screen.  This handles 3 query types:  Backup, fast Backup,
|          and archive queries.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_update_fs(dialog *sel_dialog,dsUint16_t parm_get_type)
{

dsUint16_t    i;
dsBool_t      done = bFalse;
dialog       *dlg;
dsInt16_t     rc;
char         *s;
char          fsName[DSM_MAX_FSNAME_LENGTH+1];
dsStruct64_t  capacity64, occupancy64;
dsUint32_t    updAction = 0;
dsmFSUpd      updFilespace;
char          sInfo[DSM_MAX_FSINFO_LENGTH + 1];


if (!dsmHandle)
{
   printf("You must signon before performing this action.\n");
   return(0);
}

if (!(s = (char*)malloc(sizeof(dsmObjName))))
{
   printf("Memory alloc failed in perform_update_obj.\n");
   return RC_NO_MEMORY;
}



/*-----------------------------------------------------------------------
|  First pull all values out of the passed dialog for our use.
.-----------------------------------------------------------------------*/
i = 0;
dlg = &sel_dialog[i];
while (dlg->item_type != DSMAPI_END)
{
   switch (dlg->item_type)
   {
      case DSMAPI_FILESPACE :
         if ((strlen(dlg->item_buff) > DSM_MAX_FSNAME_LENGTH)
            || !*dlg->item_buff)
         {
            strcpy(fsName,"*");
         }
         else strcpy(fsName,dlg->item_buff);
         break;
     case DSMAPI_UPDFSINFO :
         switch (*dlg->item_buff)
         {
            case 'Y':
            case 'y':
                  updAction = updAction | DSM_FSUPD_FSINFO;         break;
            case 'N':
            case 'n':
            default :
                   break;
         }
         break;

     case DSMAPI_UPDFSCAPACITY :
         switch (*dlg->item_buff)
         {
            case 'Y':
            case 'y':
                  updAction = updAction | DSM_FSUPD_CAPACITY;         break;
            case 'N':
            case 'n':
            default :
                   break;
         }
         break;

     case DSMAPI_UPDFSOCCUPANCY :
         switch (*dlg->item_buff)
         {
            case 'Y':
            case 'y':
                  updAction = updAction | DSM_FSUPD_OCCUPANCY;      break;
            case 'N':
            case 'n':
            default :
                   break;
         }
         break;
     case  DSMAPI_UPDFSBACKSTARTDATE:
         switch (*dlg->item_buff)
         {
            case 'Y':
            case 'y':
                  updAction = updAction | DSM_FSUPD_BACKSTARTDATE;     break;
            case 'N':
            case 'n':
            default :
                   break;
         }
         break;
     case  DSMAPI_UPDFSBACKCOMPLTDATE:
         switch (*dlg->item_buff)
         {
            case 'Y':
            case 'y':
                  updAction = updAction | DSM_FSUPD_BACKCOMPLETEDATE;     break;
            case 'N':
            case 'n':
            default :
                   break;
         }
         break;
  case DSMAPI_FSINFO :
         if ((strlen(dlg->item_buff) > DSM_MAX_FSINFO_LENGTH)
            || !*dlg->item_buff)
         {
            strcpy(sInfo,"");
         }
         else
            strcpy(sInfo, dlg->item_buff);

         break;
  case  DSMAPI_FSOCCUPANCY :
            occupancy64 = dChtoI64(dlg->item_buff,10);
            updFilespace.occupancy.hi = occupancy64.hi ;
            updFilespace.occupancy.lo = occupancy64.lo;
            break;
  case  DSMAPI_FSCAPACITY :
            capacity64 = dChtoI64(dlg->item_buff,10);
            updFilespace.capacity.hi = capacity64.hi ;
            updFilespace.capacity.lo = capacity64.lo;
         break;

  default :
         ;
   }
   i++;
   dlg = &sel_dialog[i];
}
UpdateFileSpace(&updFilespace, sInfo);


updFilespace.stVersion = dsmFSUpdVersion;
updFilespace.fsType = 0;                       /* no change */

if ((rc = dsmUpdateFS(dsmHandle, fsName, &updFilespace,updAction))
                         != DSM_RC_OK)
{
   printf("*** dsmUpdateFS failed: \n");
   rcApiOut(dsmHandle, rc);
}
else
   printf("\n Finished Updating file system information successfully.\n");

 free(s);
 return rc;
}

/*----------------------------------------------------------------------+
| Name:    perform_query_fs()
|
| Action:  Take the information passed in the dialog to this function,
|          build the backup objects dynamically, and send them to the
|          server.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_query_fs(dialog *sel_dialog, dsUint16_t fs_action)
{
   dialog         *dlg;
   dsUint32_t      i;
   dsInt16_t       rc;
   qryFSData       qFSData;
   DataBlk         qData;
   qryRespFSData   qRespFSData, *fsResp;
   char           *fs;
   dsBool_t        first_fs = bTrue;
   fsInfo         *fsllHead = NULL;
   fsInfo         *fsllTail = NULL;
   dsUint32_t      qry_item = 0;
   
   char            backStartDate[20],     backCompleteDate[20];
   char            lastReplStartDate[20];
   char            lastReplCmpltDate[20];
   char            lastBackOpDateFromServer[20];
   char            lastArchOpDateFromServer[20];
   char            lastSpMgOpDateFromServer[20];
   char            lastBackOpDateFromLocal[20];
   char            lastArchOpDateFromLocal[20];
   char            lastSpMgOpDateFromLocal[20];
   char            fs_head[200];
   char            fs_div[200];
   char            fs_info[DSM_MAX_FSINFO_LENGTH+1]={0}; /* copy array into here to make string*/
   
   strcpy(fs_head, "Filespace Name  Type            Occupancy  Capacity   Start             End              ");
   strcpy(fs_div,  "--------------- --------------- ---------- ---------- ----------------- -----------------");

   if (!dsmHandle)
   {
      printf("You must signon before performing this action.\n");
      return(0);
   }
   
   if (!(fs = (char*)malloc(DSM_MAX_FSNAME_LENGTH + 1))) return (RC_NO_MEMORY);
   memset(fs,0x00,(DSM_MAX_FSNAME_LENGTH + 1));
   
   /* First pull all values out of the passed dialog for our use.  */
   i = 0;
   dlg = &sel_dialog[i];
   while (dlg->item_type != DSMAPI_END)
   {
      switch (dlg->item_type)
      {
         case DSMAPI_FILESPACE :
            if ((strlen(dlg->item_buff) > DSM_MAX_FSNAME_LENGTH)
                || !*dlg->item_buff)
            {
               strcpy(fs,"*");
            }
            else strcpy(fs,dlg->item_buff);
            break;
            
         default :
            ;
      }
      i++;
      dlg = &sel_dialog[i];
   }
   
   qFSData.stVersion = qryFSDataVersion;
   qFSData.fsName    = fs;
   
   qData.stVersion = DataBlkVersion;
   qData.bufferLen = sizeof(qryRespFSData);
   qData.bufferPtr = (char *)&qRespFSData;
   
   qRespFSData.stVersion = qryRespFSDataVersion;
   
   if ((rc = dsmBeginQuery(dsmHandle,qtFilespace,(dsmQueryBuff *)&qFSData)))
   {
      printf("*** dsmBeginQuery failed: \n");
      rcApiOut(dsmHandle, rc);
      rc = (RC_SESSION_FAILED);
   }
   else
   {
      while ((rc = dsmGetNextQObj(dsmHandle,&qData)) == DSM_RC_MORE_DATA)
      {
         qry_item++;
         fsResp = (qryRespFSData *)qData.bufferPtr;
         switch(fs_action)
         {
               /*------------------------------------
                |  Display fs info from the query
                .-------------------------------------*/
            case (fs_query) :
               if (first_fs)
               {
                  printf("%s\n%s\n",fs_head,fs_div);
                  first_fs = bFalse;
               }
               
               FSInfoResp(fs_info, fsResp); /* format area into string */
               QryDateToStr(backStartDate,fsResp->backStartDate);
               QryDateToStr(backCompleteDate,fsResp->backCompleteDate);
               QryDateToStr(backStartDate,     fsResp->backStartDate);
               QryDateToStr(backCompleteDate,  fsResp->backCompleteDate);
               
               QryDateToStr(lastReplStartDate, fsResp->lastReplStartDate);
               QryDateToStr(lastReplCmpltDate, fsResp->lastReplCmpltDate);
               QryDateToStr(lastBackOpDateFromServer, fsResp->lastBackOpDateFromServer);
               QryDateToStr(lastArchOpDateFromServer, fsResp->lastArchOpDateFromServer);
               QryDateToStr(lastSpMgOpDateFromServer, fsResp->lastSpMgOpDateFromServer);
               
               QryDateToStr(lastBackOpDateFromLocal,  fsResp->lastBackOpDateFromLocal);
               QryDateToStr(lastArchOpDateFromLocal,  fsResp->lastArchOpDateFromLocal);
               QryDateToStr(lastSpMgOpDateFromLocal,  fsResp->lastSpMgOpDateFromLocal);
               
               printf("%-15s %-15s %10u %10u %17s %17s\n",
                      fsResp->fsName,
                      fsResp->fsType,
                      fsResp->occupancy.lo,
                      fsResp->capacity.lo,
                      backStartDate, backCompleteDate);
               
               printf("\n   %s %s\n   %s %s\n\n",
                      "  Start of last Replication : ", lastReplStartDate, 
                      "  End of last Replication   : ", lastReplCmpltDate);
               
               printf("   %37s %24s\n","Server", "Local");
               printf("   %s %-25s %-s \n   %s %-25s %-s\n   %s %-25s %-s\n",

                      "  Last backup store date    : " ,lastBackOpDateFromServer, lastBackOpDateFromLocal,
                      "  Last archive store date   : " ,lastArchOpDateFromServer, lastArchOpDateFromLocal,
                      "  Last HSM store date       : " ,lastSpMgOpDateFromServer, lastSpMgOpDateFromLocal);
               
               printf("     FSINFO : %s\n\n", fs_info);
               break;
               /*-----------------------------------------------------
                |  Save info from the query so we can delete the fs(s)
                .-------------------------------------------------------*/
            case (fs_delete) :
               rc = QryAddFsLlist(qry_item,
                                  &fsllHead,
                                  &fsllTail,
                                  fsResp);
               if (rc)
               {
                  QryFreeFsLlist(&fsllHead,
                                 &fsllTail);
                  printf("FS Delete failed during the filespace query.\n");
               }
               break;
               /*-----------------------------------------------------
                |  Default processing.
                .-------------------------------------------------------*/
            default : ;
         }
      }
      if (rc && (rc != DSM_RC_FINISHED))
      {
         printf("\n*** dsmGetNextQObj failed: \n");
         rcApiOut(dsmHandle, rc);
      }
      rc = dsmEndQuery(dsmHandle);
   }
   
   if (fs_delete)
   {
      rc = QryDoFsDelete(fsllHead,fsllTail,qry_item);
      QryFreeFsLlist(&fsllHead,&fsllTail);
   }
   free(fs);
   return rc;
}

/*----------------------------------------------------------------------+
| Name:    perform_query_mc()
|
| Action:  Take the information passed in the dialog to this function,
|          build the backup objects dynamically, and send them to the
|          server.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_query_mc(dialog *sel_dialog,dsUint16_t parm1)
{
dialog    *dlg;
dsUint32_t     i;
dsInt16_t      rc;

qryMCData     qMCData;
DataBlk       qData;
qryRespMCDetailData qRespMCData, *mcResp;
char *mc, *s;
dsBool_t  done = bFalse;
dsUint32_t  qry_item;

char mc_head[50];
char mc_div[50];

strcpy(mc_head, "Mgmt Class Name                Description");
strcpy(mc_div,  "------------------------------ ------------------");

if (!dsmHandle)
  {
  printf("You must signon before performing this action.\n");
  return(0);
  }

if (!(mc = (char*)malloc(DSM_MAX_MC_NAME_LENGTH + 1))) return (RC_NO_MEMORY);
if (!(s = (char*)malloc(1024))) return (RC_NO_MEMORY);  /* Work buffer */

/* First pull all values out of the passed dialog for our use.  */
i = 0;
dlg = &sel_dialog[i];
while (dlg->item_type != DSMAPI_END)
  {
    switch (dlg->item_type)
      {
      case DSMAPI_MC :
        if ((strlen(dlg->item_buff) > DSM_MAX_MC_NAME_LENGTH)
           || !*dlg->item_buff)
          {
          *mc = 0x00;       /* Default to all if null entered. */
          }
        else strcpy(mc,dlg->item_buff);
        break;
      default :
         ;
      }
    i++;
    dlg = &sel_dialog[i];
  }

/*--------------------------------------------------------------------
|  Fill in the qryMCData structure with the query criteria we want.
.---------------------------------------------------------------------*/
qMCData.stVersion = qryMCDataVersion;
qMCData.mcName    = mc;
qMCData.mcDetail  = bTrue;

qData.stVersion = DataBlkVersion;
qData.bufferLen = sizeof(qryRespMCDetailData);
qData.bufferPtr = (char *)&qRespMCData;

qRespMCData.stVersion = qryRespMCDetailDataVersion;

if ((rc = dsmBeginQuery(dsmHandle,qtMC,(dsmQueryBuff *)&qMCData)))
{
   printf("*** dsmBeginQuery failed: \n");
   rcApiOut(dsmHandle, rc);
   rc = (RC_SESSION_FAILED);
}

else
{
   done = bFalse;
   qry_item = 0;
   while (!done)
   {
      rc = dsmGetNextQObj(dsmHandle,&qData);
      if ((   (rc == DSM_RC_OK)
           || (rc == DSM_RC_MORE_DATA)
           || (rc == DSM_RC_FINISHED))
          && qData.numBytes)
      {
         qry_item++;
         mcResp = (qryRespMCDetailData *)qData.bufferPtr;
         printf("Mgmt. Class %u:\n",qry_item);
         printf("              Name: %s\n",mcResp->mcName);
         printf("       Description: %s\n",mcResp->mcDesc);

         printf("    Backup CG Name: %s\n",mcResp->backupDet.cgName);
         printf("            Frequency: %u\n",mcResp->backupDet.frequency);
         printf("     Ver. Data Exists: %u\n",mcResp->backupDet.verDataExst);
         printf("    Ver. Data Deleted: %u\n",mcResp->backupDet.verDataDltd);
         printf("     Retain Extra Ver: %u\n",mcResp->backupDet.retXtraVers);
         printf("      Retain Only Ver: %u\n",mcResp->backupDet.retOnlyVers);
         QryPrintCopySer(mcResp->backupDet.copySer);
         QryPrintCopyMode(mcResp->backupDet.copyMode);
         printf("     Copy Destination: %s\n",mcResp->backupDet.destName);
         printf(" Lan free Destination: %s\n",mcResp->backupDet.bLanFreeDest ? "YES" : "NO");
         printf("     Deduplicate Data: %s\n",mcResp->backupDet.bDeduplicate ? "YES" : "NO");
         printf("   Archive CG Name: %s\n",mcResp->archDet.cgName);
         printf("            Frequency: %u\n",mcResp->archDet.frequency);
         printf("      Retain versions: %u\n",mcResp->archDet.retainVers);
         QryPrintCopySer(mcResp->archDet.copySer);
         QryPrintCopyMode(mcResp->archDet.copyMode);
         printf("     Copy Destination: %s\n",mcResp->archDet.destName);
         printf(" Lan free Destination: %s\n",mcResp->backupDet.bLanFreeDest ? "YES" : "NO");
         printf("     Retain Init     : %s\n",mcResp->archDet.retainInit == ARCH_RETINIT_CREATE ?
                                                 "CREATE" : "EVENT");
         printf("     Retain Minimum  : %d\n",mcResp->archDet.retainMin);
         printf("     Deduplicate Data: %s\n",mcResp->archDet.bDeduplicate ? "YES" : "NO");
      }
      else
      {
         done = bTrue;
         if (rc != DSM_RC_FINISHED)
         {
            printf("*** dsmGetNextQObj failed: \n");
            rcApiOut(dsmHandle, rc);
         }
      }
      if (rc == DSM_RC_FINISHED) done = bTrue;
   }
   rc = dsmEndQuery(dsmHandle);
}

free(mc);
free(s);
return rc;

}

/*----------------------------------------------------------------------+
| Name:    perform_query_prx()
|
| Action:  Take the information passed in the dialog to this function,
|          build the backup objects dynamically, and send them to the
|          server.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_query_prx(dialog *sel_dialog,dsUint16_t parm1)
{
dialog               *dlg;
dsUint32_t           i;
dsInt16_t            rc;
dsmQueryType         queryType;
qryProxyNodeData     qpnd;
qryRespProxyNodeData qrpn;
DataBlk              qDataBlkArea;
char                 *targetNode;

if (!dsmHandle)
{
   printf("You must signon before performing this action.\n");
   return(0);
}
memset(&qrpn, 0x00, sizeof(qryRespProxyNodeData));

switch (parm1) {
    case qtProxyNodeAuth:
        qpnd.targetNodeName = NULL; 
        queryType = qtProxyNodeAuth;
        printf("Target Node Name\n");
        printf("----------------\n");
        break;
    case qtProxyNodePeer:
        /* First pull all values out of the passed dialog for our use.  */
        i = 0;
        dlg = &sel_dialog[i];
        while (dlg->item_type != DSMAPI_END)
        {
            switch (dlg->item_type)
                {
                case DSMAPI_PRXTRG :
                    targetNode = dlg->item_buff;
                    break;
                default :
                    printf("*** Proxy peer query dialog does not match code in perform_query_prx! ***\n");
                }
            i++;
            dlg = &sel_dialog[i];
        }
        qpnd.targetNodeName = targetNode; 
        queryType = qtProxyNodePeer;
        printf("Target Node Name    Peer Node Name      Peer HL Address     Peer LL Address\n");
        printf("------------------- ------------------- ------------------- ---------------\n");
        break;
    default:
        printf("Unknown query type: %d.\n", parm1);
        return(0);
}

qrpn.stVersion = qryRespProxyNodeDataVersion;
qpnd.stVersion = qryProxyNodeDataVersion;

qDataBlkArea.stVersion = DataBlkVersion;
qDataBlkArea.bufferPtr = (char *)&qrpn;
qDataBlkArea.bufferLen = sizeof(qryRespProxyNodeData);

if ((rc=dsmBeginQuery(dsmHandle,queryType, (void *)&qpnd)) 
    != DSM_RC_OK) 
{
   printf("*** dsmBeginQuery failed: \n");
   rcApiOut(dsmHandle, rc);
   return rc;
}

while ((rc = dsmGetNextQObj(dsmHandle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
{
    if (queryType == qtProxyNodeAuth)
        printf("%s\n", qrpn.targetNodeName);
    else if (queryType == qtProxyNodePeer)
        printf("%-20s%-20s%-20s%-15s\n",
            qrpn.targetNodeName,qrpn.peerNodeName, 
            qrpn.hlAddress, qrpn.llAddress);
}

if (rc != DSM_RC_FINISHED && rc != DSM_RC_MORE_DATA) 
{
   printf("*** dsmGetNextQObj failed: \n");
   rcApiOut(dsmHandle, rc);
}

if ((rc = dsmEndQuery(dsmHandle)) != DSM_RC_OK) 
{
   printf("*** dsmEndQuery failed: \n");
   rcApiOut(dsmHandle, rc);
   return rc;
}
return rc;
}

/*----------------------------------------------------------------------+
| Name:    perform_query_get()
|
| Action:  Take the information passed in the dialog to this function,
|          build the proper query, issue it, and display results to the
|          screen.  This handles 3 query types:  Backup, fast Backup,
|          and archive queries.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_query_get(dialog *sel_dialog,dsUint16_t parm_get_type)
{

dsUint16_t     i;
char       *desc;
dsBool_t     done = bFalse;
dialog    *dlg;
dsmObjName objName;      /* Big object for some compiler stacks......*/
dsInt16_t      rc;
dsUint8_t      active_type;
dsBool_t     fast_active;
qryArchiveData     archiveData;
qryRespArchiveData respArchive;
qryBackupData      backupData;
qryRespBackupData  respBackup;
qryABackupData     abackupData;
qryARespBackupData respABackup;
char          owner[DSM_MAX_OWNER_LENGTH];
dsmQueryType  query_type;
dsmQueryBuff  *query_buff;
DataBlk       qData;
dsUint16_t        get_type;
dsStruct64_t        qry_item64;
char          *s;
char          splat[2];
GetInfo       *llHead = NULL, *llTail = NULL, *ll = NULL;
dsBool_t        details = bFalse;
char          t[50], t4[50];
dsBool_t        mountWait = bFalse;
char          go[INPUT_BUFFER_SIZE];
dsUint16_t        year = DATE_MINUS_INFINITE;
dsUint16_t        hiyear = DATE_PLUS_INFINITE;
dsmDate       loinsDate,
              hiinsDate,
              loexpDate,
              hiexpDate,
              pitDate;

loinsDate.year = DATE_MINUS_INFINITE;
hiinsDate.year = DATE_PLUS_INFINITE;
loexpDate.year = DATE_MINUS_INFINITE;
hiexpDate.year = DATE_PLUS_INFINITE;
pitDate.year   = DATE_MINUS_INFINITE;

strcpy(owner, "");
strcpy(splat, "*");

if (!dsmHandle)
{
   printf("You must signon before performing this action.\n");
   return(0);
}

if (!(s = (char*)malloc(sizeof(dsmObjName))))
{
   printf("Memory alloc failed in perform_query_get.\n");
   return RC_NO_MEMORY;
}

get_type = parm_get_type;    /* Put into a value we can modify. */

memset(&objName,0x00,sizeof(dsmObjName));

/*-----------------------------------------------------------------------
|  First pull all values out of the passed dialog for our use.
.-----------------------------------------------------------------------*/
i = 0;
dlg = &sel_dialog[i];

while (dlg->item_type != DSMAPI_END)
{
   switch (dlg->item_type)
   {
      case DSMAPI_FILESPACE :
         if ((strlen(dlg->item_buff) > DSM_MAX_FSNAME_LENGTH)
            || !*dlg->item_buff)
         {
            strcpy(objName.fs,"*");
         }
         else strcpy(objName.fs,dlg->item_buff);
         break;
      case DSMAPI_HIGHLEVEL :
         if ((strlen(dlg->item_buff) > DSM_MAX_HL_LENGTH)
            || !*dlg->item_buff)
         {
           objName.hl[0] = '*';  /* don't use delimiter - optional */
           objName.hl[1] = 0x00;
         }
          else strcpy(objName.hl,dlg->item_buff);
          break;
      case DSMAPI_LOWLEVEL :
         if ((strlen(dlg->item_buff) > DSM_MAX_LL_LENGTH)
            || !*dlg->item_buff)
         {
           objName.ll[0] = dsmSessInfo.hldelim;
           objName.ll[1] = '*';
           objName.ll[2] = 0x00;
         }
         else strcpy(objName.ll,dlg->item_buff);
         break;
      case DSMAPI_OBJOWNER :
         if ((strlen(dlg->item_buff) > DSM_MAX_OWNER_LENGTH)
            || !*dlg->item_buff)
         {
            strcpy(owner,"");
         }
         else strcpy(owner,dlg->item_buff);
         break;

   case DSMAPI_OBJTYPE :
     switch (*dlg->item_buff)
     {
        case 'F':
        case 'f':
           objName.objType = DSM_OBJ_FILE; break;
        case 'D':
        case 'd':
           objName.objType = DSM_OBJ_DIRECTORY; break;
        case 'Q':
        case 'q':
             objName.objType = DSM_OBJ_QRY_FILES_AND_DIRS; break;
       case 'N':
       case 'n':
             objName.objType = DSM_OBJ_NQR_FILES_AND_DIRS; break;    
        case 'W':
        case 'w':
           objName.objType = DSM_OBJ_WILDCARD; break;
        default :
           objName.objType = DSM_OBJ_FILE;
     }
     break;
      
      case DSMAPI_ACTIVETYPE :
         switch (*dlg->item_buff)
         {
            case 'A':
            case 'a':
               active_type = DSM_ACTIVE; break;
            case 'I':
            case 'i':
               active_type = DSM_INACTIVE; break;
            case 'B' :
            case 'b' :
               active_type = DSM_ANY_MATCH; break;
            default :
               active_type = DSM_ANY_MATCH;
         }
         break;
      case DSMAPI_DESCRIPTION :
         if ((strlen(dlg->item_buff) > DSM_MAX_DESCR_LENGTH)
            || !*dlg->item_buff)
         {
            desc = splat;
         }
         else desc = dlg->item_buff;
         break;
      case DSMAPI_FASTACTIVE :
         switch (*dlg->item_buff)
         {
            case 'Y':
            case 'y':
               fast_active = bTrue;
               /* change get_type only for Query Backup */
               if (get_type == Backup_Get)
                   get_type = ABackup_Get;
               break;
            case 'N':
            case 'n':
               fast_active = bFalse; break;
            default :
               fast_active = bFalse;
         }
      case DSMAPI_DETAILS :
         switch (*dlg->item_buff)
         {
            case 'Y':
            case 'y':
               details = bTrue;
               break;
            case 'N':
            case 'n':
               details = bFalse;
               break;
            default :
               details = pref.qry_details;
         }
         break;
      case DSMAPI_MNTWAIT :
         switch (*dlg->item_buff)
         {
            case 'Y':
            case 'y':
               mountWait = bTrue;              break;
            case 'N':
            case 'n':
               mountWait = bFalse;             break;
            default :
               mountWait = bFalse;             break;
         }
         break;
      case DSMAPI_LOINSDATE:
           xStringToDate(dlg->item_buff, &loinsDate);
           break;

      case DSMAPI_LOINSTIME:
           xStringToTime(dlg->item_buff, &loinsDate);
           break;


      case DSMAPI_HIINSDATE:
           xStringToDate(dlg->item_buff, &hiinsDate);
           break;

      case DSMAPI_HIINSTIME:
           xStringToTime(dlg->item_buff, &hiinsDate);
           break;

      case DSMAPI_LOEXPDATE:
           xStringToDate(dlg->item_buff, &loexpDate);
           break;

      case DSMAPI_LOEXPTIME:
           xStringToTime(dlg->item_buff, &loexpDate);
           break;

      case DSMAPI_HIEXPDATE:
           xStringToDate(dlg->item_buff, &hiexpDate);
           break;

      case DSMAPI_HIEXPTIME:
           xStringToTime(dlg->item_buff, &hiexpDate);
           break;

      case DSMAPI_PITDATE :
           xStringToDate(dlg->item_buff, &pitDate);
           break;

     case DSMAPI_PITTIME :
           xStringToTime(dlg->item_buff, &pitDate);
           break;
  default :
         ;
   }
   i++;
   dlg = &sel_dialog[i];

}

/*----------------------------------------------------------------------
|  Fill in the proper structures for the type of query being performed.
.-----------------------------------------------------------------------*/
switch (get_type)
{
   case ABackup_Get :
      abackupData.stVersion = qryABackupDataVersion;
      abackupData.objName   = &objName;

      query_type = qtBackupActive;
      query_buff = (dsmQueryBuff *)&abackupData;

      qData.stVersion = DataBlkVersion;
      qData.bufferLen = sizeof(qryARespBackupData);
      qData.bufferPtr = (char *)&respABackup;
      respABackup.stVersion = qryARespBackupDataVersion;
      break;

   case Backup_Get  :
   case Restore_Get :
   case Backup_Delete :
      backupData.stVersion = qryBackupDataVersion;
      backupData.objName   = &objName;
      backupData.owner     = owner;
      backupData.objState  = active_type;
      DateToDate(pitDate, &backupData.pitDate);
      query_type = qtBackup;
      query_buff = (dsmQueryBuff *)&backupData;

      qData.stVersion = DataBlkVersion;
      qData.bufferLen = sizeof(qryRespBackupData);
      qData.bufferPtr = (char *)&respBackup;
      respBackup.stVersion = qryRespBackupDataVersion;;
      break;

   case Archive_Get  :
   case Retrieve_Get :
   case Archive_Delete :
      archiveData.stVersion = qryArchiveDataVersion;
      archiveData.objName   = &objName;
      archiveData.owner     = owner;

      DateToDate(loinsDate, &archiveData.insDateLowerBound);
      DateToDate(hiinsDate, &archiveData.insDateUpperBound);
      DateToDate(loexpDate, &archiveData.expDateLowerBound);
      DateToDate(hiexpDate, &archiveData.expDateUpperBound);

      archiveData.descr = desc;
      query_type = qtArchive;
      query_buff = (dsmQueryBuff *)&archiveData;

      qData.stVersion = DataBlkVersion;
      qData.bufferLen = sizeof(qryRespArchiveData);
      qData.bufferPtr = (char *)&respArchive;
      respArchive.stVersion = qryRespArchiveDataVersion;
      break;
   default : ;
}
/*----------------------------------------------------------------------
|  Now issue the begin query call, and if no errors issue get next calls
|  until all query information has been processed.  For each query response
|  output the proper info to the screen or save it for later processing.
.-----------------------------------------------------------------------*/
if ((rc = dsmBeginQuery(dsmHandle,query_type,query_buff)))
{
   printf("*** dsmBeginQuery failed: \n");
   rcApiOut(dsmHandle, rc);
   rc = (RC_SESSION_FAILED);
}
else
{
   done = bFalse;
   qry_item64.hi = 0; qry_item64.lo = 0;
   while (!done)
   {
      rc = dsmGetNextQObj(dsmHandle,&qData);
      if ((   (rc == DSM_RC_OK)
           || (rc == DSM_RC_MORE_DATA)
           || (rc == DSM_RC_FINISHED))
          && qData.numBytes)
      {

         switch(get_type)
         {
            /*-----------------------------
            |  Display archive query results
            .------------------------------*/
            case Archive_Get :
               qry_item64 = dIncr64(&qry_item64,1);
               if (details) printf("\n");
               dI64toCh(&qry_item64,t4,10);  format_number(t4,t);
               printf("Item %s: ",t);                     /* Item number */
               printf("%s%s%s\n",respArchive.objName.fs,  /* Name        */
                                 respArchive.objName.hl,
                                 respArchive.objName.ll);

               if (!details) break;

               QryPrintObjType(respArchive.objName.objType);
               printf("   Desc: %s\n",respArchive.descr);
               QryDateToStr(s,respArchive.insDate);
               printf("   Insert date: %s\n",s);
               QryDateToStr(s,respArchive.expDate);
               printf("   Expiration date: %s\n",s);
               printf("   Owner: %s\n",respArchive.owner);
               printf("   Restore order: %u-%u-%u-%u-%u\n",
                               respArchive.restoreOrderExt.top,
                               respArchive.restoreOrderExt.hi_hi,
                               respArchive.restoreOrderExt.hi_lo,
                               respArchive.restoreOrderExt.lo_hi,
                               respArchive.restoreOrderExt.lo_lo);
               printf("   Object id: %u-%u\n",respArchive.objId.hi,
                                              respArchive.objId.lo);
               printf("   Copy group: %u\n",respArchive.copyGroup);
               QryPrintMediaClass(respArchive.mediaClass);
               printf("   Mgmt class: %s\n",respArchive.mcName);
               QryPrintObjInfo(respArchive.objInfolen,
                               respArchive.objInfo);
               printf("   Estimated size : %u  %u \n",respArchive.sizeEstimate.hi,
                                                       respArchive.sizeEstimate.lo);
               printf("   Retention Initiated: %s\n", respArchive.retentionInitiated == DSM_ARCH_RETINIT_STARTED ?
                                               "STARTED" :
                                               respArchive.retentionInitiated == DSM_ARCH_RETINIT_PENDING ?
                                               "PENDING" : "UNKNOWN");
               printf("   Object Held : %s\n", respArchive.objHeld == DSM_ARCH_HELD_TRUE ?
                                               "TRUE" :
                                               respArchive.objHeld == DSM_ARCH_HELD_FALSE ?
                                               "FALSE" : "UNKNOWN");
               printf("   Compression : %s\n   Compression Type : %s\n", respArchive.compressType == DSM_OBJ_COMPRESSED_YES ?
                                               "YES" :
                                               respArchive.compressType == DSM_OBJ_COMPRESSED_NO ?
                                               "NO" : "UNKNOWN",
                                               strlen(respArchive.compressAlg) ? respArchive.compressAlg : "NONE");
               printf("   Encryption : %s\n   Encryption Strength : %s\n", respArchive.encryptionType & DSM_ENCRYPT_CLIENTENCRKEY ?
                                               "CLIENTENCRKEY" :
                                               respArchive.encryptionType & DSM_ENCRYPT_USER ?
                                               "USER" : "NO",
                                               respArchive.encryptionType & DSM_ENCRYPT_AES_256BIT ? "AES_256BIT" :
                                               respArchive.encryptionType & DSM_ENCRYPT_AES_128BIT ? "AES_128BIT" :
                                               respArchive.encryptionType& DSM_ENCRYPT_DES_56BIT ? "DES_56BIT" : 
                                               "NONE");
               printf("   Client Deduplicated : %s\n", respArchive.clientDeduplicated ?
                                               "YES" : "NO");

               printf("Press any key to continue\n");
               ourGetS(go);
               break;
            /*-----------------------------
            |  Display backup query results
            .------------------------------*/
            case Backup_Get  :
               qry_item64 = dIncr64(&qry_item64,1);
               if (details) printf("\n");
               dI64toCh(&qry_item64,t4,10);  format_number(t4,t);
               printf("Item %s: ",t);                     /* Item number */
               printf("%s%s%s\n",respBackup.objName.fs,
                                 respBackup.objName.hl,
                                 respBackup.objName.ll);

               if (!details) break;

               QryPrintObjType(respBackup.objName.objType);
               QryPrintObjState(respBackup.objState);
               QryDateToStr(s,respBackup.insDate);
               printf("   Insert date: %s\n",s);
               QryDateToStr(s,respBackup.expDate);
               printf("   Expiration date: %s\n",s);
               printf("   Owner: %s\n",respBackup.owner);
               printf("   Restore order: %u-%u-%u-%u-%u\n",
                               respBackup.restoreOrderExt.top,
                               respBackup.restoreOrderExt.hi_hi,
                               respBackup.restoreOrderExt.hi_lo,
                               respBackup.restoreOrderExt.lo_hi,
                               respBackup.restoreOrderExt.lo_lo);
               printf("   Object id: %u-%u\n",respBackup.objId.hi,
                                             respBackup.objId.lo);
               printf("   Copy group: %u\n",respBackup.copyGroup);
               QryPrintMediaClass(respBackup.mediaClass);
               printf("   Mgmt class: %s\n",respBackup.mcName);
               QryPrintObjInfo(respBackup.objInfolen,
                               respBackup.objInfo);
               printf("   Estimated size : %u  %u \n",respBackup.sizeEstimate.hi,
                                                        respBackup.sizeEstimate.lo);
               printf("   Compression : %s\n   Compression Type : %s\n", respBackup.compressType == DSM_OBJ_COMPRESSED_YES ?
                                               "YES" :
                                               respBackup.compressType == DSM_OBJ_COMPRESSED_NO ?
                                               "NO" : "UNKNOWN",
                                               strlen(respBackup.compressAlg) ? respBackup.compressAlg : "NONE");
               printf("   Encryption : %s\n   Encryption Strength : %s\n", respBackup.encryptionType & DSM_ENCRYPT_CLIENTENCRKEY ?
                                               "CLIENTENCRKEY" :
                                               respBackup.encryptionType & DSM_ENCRYPT_USER ?
                                               "USER" : "NO",
                                               respBackup.encryptionType & DSM_ENCRYPT_AES_256BIT ? "AES_256BIT" :
                                               respBackup.encryptionType & DSM_ENCRYPT_AES_128BIT ? "AES_128BIT" :
                                               respBackup.encryptionType& DSM_ENCRYPT_DES_56BIT ? "DES_56BIT" : 
                                               "NONE");
               printf("   Client Deduplicated : %s\n", respBackup.clientDeduplicated ?
                                               "YES" : "NO");
               
               printf("Press any key to continue\n");
               ourGetS(go);

               break;
            /*------------------------------------
            |  Display active backup query results
            .------------------------------------*/
            case ABackup_Get :
               qry_item64 = dIncr64(&qry_item64,1);
               if (details) printf("\n");
               dI64toCh(&qry_item64,t4,10);  format_number(t4,t);
               printf("Item %s: ",t);                     /* Item number */
               printf("%s%s%s\n",respABackup.objName.fs,
                                 respABackup.objName.hl,
                                 respABackup.objName.ll);

               if (!details) break;

               QryPrintObjType(respABackup.objName.objType);
               QryDateToStr(s,respABackup.insDate);
               printf("   Insert date: %s\n",s);
               printf("   Owner: %s\n",respABackup.owner);
               printf("   Copy group: %u\n",respABackup.copyGroup);
               QryPrintObjInfo(respABackup.objInfolen,
                               respABackup.objInfo);
               printf("   Mgmt class: %s\n",respABackup.mcName);
               printf("Press any key to continue\n");
               ourGetS(go);
               break;
            /*-----------------------------------------------
            |  Save key data from query for restore or delete
            .------------------------------------------------*/
            case Restore_Get :
            case Backup_Delete :

               /*---------------------------------------
               |  Only can delete active backup files!
               .--------------------------------------*/
               if ((get_type == Backup_Delete)
                   && (respBackup.objState != DSM_ACTIVE))
                  break;
               qry_item64 = dIncr64(&qry_item64,1);
               rc = QryAddGetLlist(get_type,
                                  qry_item64,
                                  &llHead,
                                  &llTail,
                                  (char *)&respBackup);
               if (rc)
               {
                  QryFreeGetLlist(&llHead,
                                 &llTail);
                  switch (get_type)
                  {
                     case Restore_Get :
                        printf("Restore failed during the backup query.\n");
                        break;
                     case Backup_Delete :
                     default :
                        printf("Backup delete failed during the query.\n");
                  }

               }
               break;
            /*------------------------------------------------
            |  Save key data from query for retrieve or delete
            .------------------------------------------------*/
            case Retrieve_Get :
            case Archive_Delete :
               qry_item64 = dIncr64(&qry_item64,1);
               rc = QryAddGetLlist(get_type,
                                  qry_item64,
                                  &llHead,
                                  &llTail,
                                  (char *)&respArchive);
               if (rc)
               {
               QryFreeGetLlist(&llHead,
                              &llTail);
                  switch (get_type)
                  {
                     case Retrieve_Get :
                        printf("Retrieve failed during the backup query.\n");
                        break;
                     case Archive_Delete :
                     default :
                        printf("Archive delete failed during the query.\n");
                  }
               }
               break;
            /*------------------
            |  Default catch all
            .-------------------*/
            default : ;
         }
      }
      else
      {
         done = bTrue;
         if (rc != DSM_RC_FINISHED)
         {
            printf("*** dsmGetNextQObj failed: \n");
            rcApiOut(dsmHandle, rc);
         }
      if (rc == DSM_RC_FINISHED) done = bTrue;
      }

   }

   rc = dsmEndQuery(dsmHandle);
}

if (   (get_type == Restore_Get)
    || (get_type == Retrieve_Get))
{
   rc = QryDoGet(get_type,llHead,llTail,qry_item64,mountWait);
   QryFreeGetLlist(&llHead,&llTail);
}

if (   (get_type == Backup_Delete)
    || (get_type == Archive_Delete))
{
   rc = QryDoDelete(get_type,llHead,llTail,qry_item64);
   QryFreeGetLlist(&llHead,&llTail);
}

free(s);
return rc;
}
/*----------------------------------------------------------------------+
| Name:    perform_del_obj()
|
| Action:  Take the information passed in the dialog to this function,
|          build the proper query, issue it, and display results to the
|          screen.  This handles 3 query types:  Backup, fast Backup,
|          and archive queries.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_del_obj(dialog *sel_dialog,dsUint16_t parm_get_type)
{

dsUint16_t    i;
dsBool_t      done = bFalse;
dialog        *dlg;
dsInt16_t     rc;
char          splat[2];
dsStruct64_t  objIdHi64, objIdLo64, objId;
dsUint8_t     type;
dsUint16_t     reason;
dsmDelInfo    delInfo;

strcpy(splat, "*");

if (!dsmHandle)
{
   printf("You must signon before performing this action.\n");
   return(0);
}
i = 0;
dlg = &sel_dialog[i];
while (dlg->item_type != DSMAPI_END)
{
   switch (dlg->item_type)
   {
     case DSMAPI_OBJIDHI :
         objIdHi64 = dChtoI64(dlg->item_buff,10);
         objId.hi = objIdHi64.lo;
         break;
     case DSMAPI_OBJIDLO :
         objIdLo64 = dChtoI64(dlg->item_buff,10);
         objId.lo = objIdLo64.lo;
         break;
     case DSMAPI_REPOSITORY :
         switch (*dlg->item_buff)
         {
            case 'B':
            case 'b':
               type = DSM_BACKUP_REP; break;
            case 'A':
            case 'a':
               type = DSM_ARCHIVE_REP; break;
            default :
               type = DSM_BACKUP_REP;
         }
         break;
   }
   i++;
   dlg = &sel_dialog[i];
}


   if ((rc = dsmBeginTxn(dsmHandle)) != DSM_RC_OK)
   {
      printf("\n*** dsmDeleteObj failed: \n");
      rcApiOut(dsmHandle, rc);
      return rc;
   }

switch(type)
{
   case DSM_BACKUP_REP:
     delInfo.backIDInfo.stVersion = delBackIDVersion;
     delInfo.backIDInfo.objId = objId;
     rc = dsmDeleteObj(dsmHandle,dtBackupID,delInfo);
     break;
   case DSM_ARCHIVE_REP:
     delInfo.archInfo.stVersion = delArchVersion;
     delInfo.archInfo.objId = objId;
     rc = dsmDeleteObj(dsmHandle,dtArchive,delInfo);
} 

   if ( rc |= DSM_RC_OK)
   {
      printf("\n*** dsmDeleteObj failed: \n");
      rcApiOut(dsmHandle, rc);
      dsmEndTxn(dsmHandle,DSM_VOTE_COMMIT, &reason);
      return rc;
   }

   if ((rc = dsmEndTxn(dsmHandle,DSM_VOTE_COMMIT, &reason)) != DSM_RC_OK)
   {
      printf("\n*** dsmDeleteObj failed: \n");
      rcApiOut(dsmHandle, rc);
      return rc;
   }
   else
      printf(" Finished Delete Obj ID successfully\n");

  return rc;
}
/*----------------------------------------------------------------------+
| Name:    perform_ret_event()
|
| Action:  Take the information passed in the dialog to this function,
|          build the proper query, issue it, and display results to the
|          screen.  This handles 3 query types:  Backup, fast Backup,
|          and archive queries.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_ret_event(dialog *sel_dialog,dsUint16_t parm_get_type)
{

dsUint16_t     i;
dsBool_t       done = bFalse;
dialog         *dlg;
dsInt16_t      rc;
dsStruct64_t   objIdHi64, objIdLo64, objId;
dsmEventType_t type;
dsUint16_t     reason;
dsmObjList_t   objList;
dsmRetentionEventIn_t  dsmRetentionEventIn;
dsmRetentionEventOut_t dsmRetentionEventOut;

if (!dsmHandle)
{
   printf("You must signon before performing this action.\n");
   return(0);
}
i = 0;
dlg = &sel_dialog[i];
while (dlg->item_type != DSMAPI_END)
{
   switch (dlg->item_type)
   {
     case DSMAPI_OBJIDHI :
         objIdHi64 = dChtoI64(dlg->item_buff,10);
         objId.hi = objIdHi64.lo;
         break;
     case DSMAPI_OBJIDLO :
         objIdLo64 = dChtoI64(dlg->item_buff,10);
         objId.lo = objIdLo64.lo;
         break;
     case DSMAPI_EVENTTYPE :
         switch (*dlg->item_buff)
         {
            case 'A':
            case 'a':
               type = eventRetentionActivate; break;
            case 'H':
            case 'h':
               type = eventHoldObj; break;
            case 'R':
            case 'r':
               type = eventReleaseObj; break;
            default :
               type = eventRetentionActivate;
         }
         break;
   }
   i++;
   dlg = &sel_dialog[i];
}


   if ((rc = dsmBeginTxn(dsmHandle)) != DSM_RC_OK)
   {
      printf("\n*** dsmRetentionEvent failed: \n");
      rcApiOut(dsmHandle, rc);
      return rc;
   }

   memset(&dsmRetentionEventIn,  0x00, sizeof(dsmRetentionEventIn_t));
   memset(&dsmRetentionEventOut, 0x00, sizeof(dsmRetentionEventOut_t));

   dsmRetentionEventIn.stVersion  = dsmRetentionEventInVersion;
   dsmRetentionEventIn.dsmHandle  = dsmHandle;
   dsmRetentionEventIn.eventType  = type;
   objList.stVersion = dsmObjlistVersion;
   objList.numObjId = 1;
   objList.objId    = &objId;


   dsmRetentionEventIn.objList    = objList;

   
   dsmRetentionEventOut.stVersion = dsmRetentionEventOutVersion; 

   rc = dsmRetentionEvent(&dsmRetentionEventIn, &dsmRetentionEventOut);


   if ( rc |= DSM_RC_OK)
   {
      printf("\n*** dsmRetentionEvent failed: \n");
      rcApiOut(dsmHandle, rc);
      dsmEndTxn(dsmHandle,DSM_VOTE_COMMIT, &reason);
      return rc;
   }

   if ((rc = dsmEndTxn(dsmHandle,DSM_VOTE_COMMIT, &reason)) != DSM_RC_OK)
   {
      printf("\n*** dsmRetentionEvent failed: \n");
      rcApiOut(dsmHandle, rc);
      rcApiOut(dsmHandle, reason);
      return rc;
   }
   else
      printf(" Finished Retention Event successfully\n");

  return rc;
}
/*----------------------------------------------------------------------+
| Name:    perform_rename_obj()
|
| Action:  Take the information passed in the dialog to this function,
|          build the proper query, issue it, and display results to the
|          screen.  This handles 3 query types:  Backup, fast Backup,
|          and archive queries.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t perform_rename_obj(dialog *sel_dialog,dsUint16_t parm_get_type)
{

dsUint16_t         i;
dialog             *dlg;
dsInt16_t          rc;
dsStruct64_t       objIdHi64, objIdLo64, objId;
dsUint16_t         reason;
dsmObjName         objName;
dsmRenameIn_t      renameIn;
dsmRenameOut_t     renameOut;

   memset(&renameIn, 0x00, sizeof(dsmRenameIn_t));
   memset(&renameOut, 0x00, sizeof(dsmRenameOut_t));


if (!dsmHandle)
{
   printf("You must signon before performing this action.\n");
   return(0);
}

i = 0;
dlg = &sel_dialog[i];
while (dlg->item_type != DSMAPI_END)
{
   switch (dlg->item_type)
{
      case DSMAPI_FILESPACE :
         strcpy(objName.fs,dlg->item_buff);
         break;
      case DSMAPI_HIGHLEVEL :
         strcpy(objName.hl,dlg->item_buff);
         break;
      case DSMAPI_LOWLEVEL :
         strcpy(objName.ll,dlg->item_buff);
         break;
      case DSMAPI_OBJTYPE :
         switch (*dlg->item_buff)
         {
            case 'F':
            case 'f':
               objName.objType = DSM_OBJ_FILE; break;
            case 'D':
            case 'd':
               objName.objType = DSM_OBJ_DIRECTORY; break;
            default :
               objName.objType = DSM_OBJ_FILE;
         }
      case DSMAPI_NEWHIGHLEVEL :
         strcpy(renameIn.newHl,dlg->item_buff);
         break;
      case DSMAPI_NEWLOWLEVEL :
         strcpy(renameIn.newLl,dlg->item_buff);
         break;
      case DSMAPI_MERGE :
         switch (*dlg->item_buff)
         {
            case 'Y':
            case 'y':
               renameIn.merge   = bTrue;     break;
            case 'N':
            case 'n':
               renameIn.merge   = bFalse;    break;
            default :
               renameIn.merge   = bFalse;    break;
         }

     case DSMAPI_OBJIDHI :
         objIdHi64 = dChtoI64(dlg->item_buff,10);
         renameIn.objId.hi = objIdHi64.lo;
         break;
     case DSMAPI_OBJIDLO :
         objIdLo64 = dChtoI64(dlg->item_buff,10);
         renameIn.objId.lo = objIdLo64.lo;
         break;
     case DSMAPI_REPOSITORY :
         switch (*dlg->item_buff)
         {
            case 'B':
            case 'b':
               renameIn.repository = DSM_BACKUP_REP; break;
            case 'A':
            case 'a':
               renameIn.repository = DSM_ARCHIVE_REP; break;
            default :
               renameIn.repository = DSM_BACKUP_REP;
         }
         break;
   }
   i++;
   dlg = &sel_dialog[i];
}

     objId.hi = objIdHi64.lo;
     objId.lo = objIdHi64.lo;

   renameIn.stVersion = dsmRenameInVersion;
   renameIn.dsmHandle = dsmHandle;
   renameIn.objNameP = &objName;
   renameOut.stVersion = dsmRenameOutVersion;
   
   if ((rc = dsmBeginTxn(dsmHandle)) != DSM_RC_OK)
   {
      printf("\n*** dsmDeleteObj failed: \n");
      rcApiOut(dsmHandle, rc);
      return rc;
   }

   rc = dsmRenameObj(&renameIn, &renameOut); 
   if ( rc |= DSM_RC_OK)
   {
      printf("\n*** dsmRenameObj failed: \n");
      rcApiOut(dsmHandle, rc);
      dsmEndTxn(dsmHandle,DSM_VOTE_COMMIT, &reason);
      return rc;
   }

   if ((rc = dsmEndTxn(dsmHandle,DSM_VOTE_COMMIT, &reason)) != DSM_RC_OK)
   {
      printf("\n*** dsmRenameObj failed: \n");
      rcApiOut(dsmHandle, rc);
      return rc;
   }
   else
      printf(" Finished dsmRenameObj successfully\n");

  return rc;
}

/*----------------------------------------------------------------------+
| Name:    QryDateToStr()
|
| Action:  A dumb, non-NLS enabled date to string conversion routine.
|          Just put the passed date into the passed string in the form
|          yyyy/mm/dd hh:mm:ss
|
| Input:   s - Pointer to a buffer big enough to hold the converted date
|          date - Date to convert.
|
| Returns: None
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static void QryDateToStr(char *s,dsmDate date)
{
   sprintf(s,"%i/%i/%i %i:%i:%i",
                      date.year,
               (dsInt16_t)date.month,
               (dsInt16_t)date.day,
               (dsInt16_t)date.hour,
               (dsInt16_t)date.minute,
               (dsInt16_t)date.second);
}

/*----------------------------------------------------------------------+
| static void xStringToDate(char *s, dsmDate *date)
|
| "s" has to be in"mmddyyyy" format
|
| if "s" is "06181997",
|    month = 06
|    day   = 18
|    year  = 1997
+------------------------------------------------------------------------*/
static void xStringToDate(char *s, dsmDate *date)
{
   dsUint32_t        temp;
   dsUint32_t        temp1;

/*   printf("\n Date String: %s", s); */
   /* if user key in some inputs */
   if (s[0] != '\0')
   {
      temp1 = atol(s);
      date->month = temp1 / 1000000;
      temp  = temp1 % 1000000;
      date->day   = temp / 10000;
      date->year  = temp % 10000;
   }
   /* if user just hit enter key */
   /* everythin else except year is initialized. */
   /* we don't know it's hi or lo year. */
   else
   {
      date->month = 12;
      date->day   = 31;
   }
/*   printf("\ndate = %d, month = %d, year = %d", date->day, date->month, date->year);*/

} /* static void xStringToDate(char *s, dsmDate *date) */

/*----------------------------------------------------------------------+
| static void xStringToTime(char *s, dsmDate *date)
|
| "s" has to be in"hhmm" format
|
| if "s" is "0618",
|    hour   = 06
|    minute = 18
+------------------------------------------------------------------------*/
static void xStringToTime(char *s, dsmDate *date)
{
   dsUint32_t temp;


   /* user key in something */
   if(s[0] != '\0')
   {
      temp = atol(s);
      date->hour = temp / 100;
      date->minute  = temp % 100;
   }
   else
   /* user hit enter.  assing defaults */
   {
      date->hour = 23;
      date->minute = 59;
   }

} /* static void xStringToTime(char *s, dsmDate *date) */

/*----------------------------------------------------------------------+
| static void DateToDate(dsmDate fromDate, dsmDate *toDate)
|
| transfer date from "fromDate" to "toDate"
+------------------------------------------------------------------------*/
static void DateToDate(dsmDate fromDate, dsmDate *toDate)
{

/*   printf("\ndate = %d, month = %d, year = %d", fromDate.day, fromDate.month, fromDate.year);
*/
   toDate->year = fromDate.year;
   toDate->month = fromDate.month;
   toDate->day = fromDate.day;
   toDate->hour = fromDate.hour;
   toDate->minute = fromDate.minute;
   toDate->second = 0;

/*   printf("\nDATE = %d, MONTH = %d, YEAR = %d", toDate->day, toDate->month, toDate->year);
   printf("\nHOUR = %d, MINUTE = %d, SECOND = %d", toDate->hour, toDate->minute, toDate->second);
*/

}/* static void DateToDate(dsmDate fromDate, dsmDate *toDate) */





/*----------------------------------------------------------------------+
| Name:    QryAddGetLlist()
|
| Action:  Add the passed structure to the passed linked list.  Handle
|          the standard situation of new list or existing list.  Note
|          list structure is malloc'd and fields within the list are also
|          malloc'd.  These will all be freed by the corresponding free
|          llist routine.  A more efficient method would be to use a
|          memory pool manager package for these small memory allocs, but
|          then this is just a sample app. for the api!
|
| Input:
|
| Returns: None
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static dsInt16_t QryAddGetLlist(dsUint16_t get_type,
                            dsStruct64_t  qry_item64,
                            GetInfo **llHead,
                            GetInfo **llTail,
                            char *buff)
{
   qryRespArchiveData *respA;
   qryRespBackupData  *respB;
   GetInfo           *newP = NULL, *ll = NULL;
   dsmObjNameP        objName;
   ObjID              objId;
   dsUint32_t             copyGroup;
   dsUint160_t        restoreOrder;
   dsUint8_t              mediaClass;
   dsUint16_t             objInfolen;
   char               *objInfo;
   smpObjInfo         *tobjInfo;
   char               t[50], t4[50];



   switch (get_type)
   {
      case (Restore_Get) :
      case (Backup_Delete) :
         respB = (qryRespBackupData *)buff;
         objName.fs = respB->objName.fs;
         objName.hl = respB->objName.hl;
         objName.ll = respB->objName.ll;
         objName.objType = respB->objName.objType;
         objId.hi   = respB->objId.hi;
         objId.lo   = respB->objId.lo;
         copyGroup  = respB->copyGroup;
         restoreOrder.top = respB->restoreOrderExt.top;
         restoreOrder.hi_hi = respB->restoreOrderExt.hi_hi;
         restoreOrder.hi_lo = respB->restoreOrderExt.hi_lo;
         restoreOrder.lo_hi = respB->restoreOrderExt.lo_hi;
         restoreOrder.lo_lo = respB->restoreOrderExt.lo_lo;
         mediaClass      = respB->mediaClass;
         objInfolen      = respB->objInfolen;
         objInfo         = respB->objInfo;
         break;
      case (Retrieve_Get) :
      case (Archive_Delete) :
         respA = (qryRespArchiveData *)buff;
         objName.fs = respA->objName.fs;
         objName.hl = respA->objName.hl;
         objName.ll = respA->objName.ll;
         objName.objType = respA->objName.objType;
         objId.hi   = respA->objId.hi;
         objId.lo   = respA->objId.lo;
         copyGroup  = respA->copyGroup;
         restoreOrder.top = respA->restoreOrderExt.top;
         restoreOrder.hi_hi = respA->restoreOrderExt.hi_hi;
         restoreOrder.hi_lo = respA->restoreOrderExt.hi_lo;
         restoreOrder.lo_hi = respA->restoreOrderExt.lo_hi;
         restoreOrder.lo_lo = respA->restoreOrderExt.lo_lo;
         mediaClass      = respA->mediaClass;
         objInfolen      = respA->objInfolen;
         objInfo         = respA->objInfo;
         break;
      default : ;
   }

   if (!(newP = (GetInfo *)malloc(sizeof(GetInfo))))
     return RC_NO_MEMORY;

   memset(newP,0x00,sizeof(GetInfo));
   /*--------------------------------------------------------------------
   | First get memory areas for other parts of new item.
   .--------------------------------------------------------------------*/
   if (!(newP->objName = (dsmObjNameP*)malloc(sizeof(dsmObjNameP))))
   {
      QryFreeGetLlistItem(&newP); return RC_NO_MEMORY;
   }

   if (!(newP->objName->fs = (char*)malloc(strlen(objName.fs) + 1)))
   {
      QryFreeGetLlistItem(&newP); return RC_NO_MEMORY;
   }

   if (!(newP->objName->hl = (char*)malloc(strlen(objName.hl) + 1)))
   {
      QryFreeGetLlistItem(&newP); return RC_NO_MEMORY;
   }

   if (!(newP->objName->ll = (char*)malloc(strlen(objName.ll) + 1)))
   {
      QryFreeGetLlistItem(&newP); return RC_NO_MEMORY;
   }

   if (!(newP->objInfo = (char*)malloc(objInfolen)))
   {
      QryFreeGetLlistItem(&newP); return RC_NO_MEMORY;
   }

   if (!*llHead)
   {
     *llHead = newP;
     *llTail = newP;
   }
   else
   {
      ll = *llTail;
      ll->next = newP;
      *llTail  = newP;
   }

   newP->next = NULL;
   newP->item64.hi = qry_item64.hi;
   newP->item64.lo = qry_item64.lo;
   strcpy(newP->objName->fs,objName.fs);
   strcpy(newP->objName->hl,objName.hl);
   strcpy(newP->objName->ll,objName.ll);
   newP->objName->objType = objName.objType;

   newP->objId.hi = objId.hi;
   newP->objId.lo = objId.lo;

   newP->restoreOrderExt.top = restoreOrder.top;
   newP->restoreOrderExt.hi_hi = restoreOrder.hi_hi;
   newP->restoreOrderExt.hi_lo = restoreOrder.hi_lo;
   newP->restoreOrderExt.lo_hi = restoreOrder.lo_hi;
   newP->restoreOrderExt.lo_lo = restoreOrder.lo_lo;

   newP->copyGroup  = copyGroup;
   newP->mediaClass = mediaClass;

   newP->objInfolen = objInfolen;
   memcpy(newP->objInfo,objInfo,objInfolen);

   if (pref.verbose)
   {
      tobjInfo = (smpObjInfo *)objInfo;
      dI64toCh(&(tobjInfo->size),t4,10);  format_number(t4,t);
      printf("Object info for %s%s%s :\n",objName.fs,objName.hl,objName.ll);
      printf("   Size of object : %s\n",t);
      printf("   Seed string ==>%s<==\n",tobjInfo->seed);
   }

   return 0;
}

/*----------------------------------------------------------------------+
| Name:    QryDoGet()
|
| Action:  A linked list of items to restore has been created.  Process
|          this list to create the proper list for the GetData calls.
|
| Input:
|
| Returns: None
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static dsInt16_t QryDoGet(dsUint16_t get_type,
                      GetInfo *llHead,
                      GetInfo *llTail,
                      dsStruct64_t  items64,
                      dsBool_t  mountWait)
{
   GetInfo *ll = NULL, *llStart = NULL; /* Pointers to objects in the list */
   char    *rest_ibuff, *ibuff_cursor;  /* Object id buffer and index in buff*/
   char    *part_pbuff, *pbuff_cursor;  /* Partial object pointers */
   dsUint32_t  loop_item;                   /* Loop counter */
   ObjID   *objId;                      /* Unique id for object */
   PartialObjData *partialObjData;      /* Pointer to partial object data */
   dsInt16_t  rc;
   char   buff[INPUT_BUFFER_SIZE];
   char   t[50], t2[50], t4[50];
   dsStruct64_t offset,length;
   dsBool_t   partialObjRequested = bFalse;

   if (!llHead) return 0;  /* No items in list to restore */

   /* Get buffer to put object ids in to be restored */
   if (!(rest_ibuff = (char*)malloc((pref.max_restore_items * sizeof(ObjID))+1)))
        return RC_NO_MEMORY;

   /* Get buffer to put partial object data into */
   if (!(part_pbuff = (char*)malloc((pref.max_restore_items * sizeof(PartialObjData))+1)))
        return RC_NO_MEMORY;
   pbuff_cursor = part_pbuff;

   ll = llHead; llStart = llHead;  /* Init list traversal items */
   ibuff_cursor = rest_ibuff;
   loop_item = 0;

   dI64toCh(&items64,t4,10);               format_number(t4,t);
   sprintf(t4,"%u",pref.sz_recv_buff);  format_number(t4,t2);

   switch(get_type)
   {
      case (Restore_Get) :
         printf("Starting restore of %s objects in buffers %s bytes long.\n",
           t,t2);
         break;
      case (Retrieve_Get) :
         printf("Starting retrieve of %s objects in buffers %s bytes long.\n",
           t,t2);
         break;
      default : ;
   }

/*   printf("Press enter to proceed.\n");
   ourGetS(buff); */

   if (((no_restore == 1) && (get_type == Restore_Get)) ||
       ((no_retrieve == 1) && (get_type == Retrieve_Get)))
      ;
   else
      printf("Enter 0 for offset and length for normal object retrieve\n");

   for(;;)
   {
      /*------------------------------------------------------------------
      | Bump through list creating list of object ids to restore until
      | list is full or no more items to put in it.  32bit test was
      .-----------------------------------------------------------------*/
      if ((loop_item < pref.max_restore_items) && ll)
      {
         /* Set object id list */
         objId = (ObjID *)ibuff_cursor;    /* Cast buffer to proper type */
         objId->hi = ll->objId.hi;         /* Load up hi 4 bytes...      */
         objId->lo = ll->objId.lo;         /* And lo 4 bytes.            */
         /* Set partial object data list */
         partialObjData = (PartialObjData *)pbuff_cursor;

         if (((no_restore == 1) && (get_type == Restore_Get)) ||
             ((no_retrieve == 1) && (get_type == Retrieve_Get)))
            offset = dChtoI64("0",10);
         else {
            printf("Enter partial object offset for object %i: \n",
               loop_item + 1);
            ourGetS(buff);
            offset = dChtoI64(buff,10);
         }

         if (!partialObjRequested && (offset.hi || offset.lo))
            partialObjRequested = bTrue;
         partialObjData->partialObjOffset.hi = offset.hi;
         partialObjData->partialObjOffset.lo = offset.lo;

         if (((no_restore == 1) && (get_type == Restore_Get)) ||
             ((no_retrieve == 1) && (get_type == Retrieve_Get)))
            length = dChtoI64("0",10);
         else {
            printf("Enter partial object length for object %i: \n",
               loop_item + 1);
            ourGetS(buff);
            length = dChtoI64(buff,10);
         }

         if (!partialObjRequested && (length.hi || length.lo))
            partialObjRequested = bTrue;
         partialObjData->partialObjLength.hi = length.hi;
         partialObjData->partialObjLength.lo = length.lo;
         /* Increment cursors and loop counters */
         pbuff_cursor += sizeof(PartialObjData);
         ibuff_cursor += sizeof(ObjID);    /* Point to next location.    */
         loop_item++;                      /* Item in list.              */
         ll = ll->next;                    /* Go to next item if any.    */
      }
      else
{

         /*--------------------------------------------------------------
         | See if there are items to actually restore.  If so do so else
         | done with the restore process.
         .--------------------------------------------------------------*/
         if (loop_item)
         {

            SortObjects(llStart, rest_ibuff, loop_item);

            if (partialObjRequested) /* If partial object request */
               rc = QryDoRestoreObjs(get_type,llStart,rest_ibuff,loop_item,
                                  mountWait, part_pbuff);
            else                     /* else not a partial object request */
               rc = QryDoRestoreObjs(get_type,llStart,rest_ibuff,loop_item,
                                  mountWait, NULL);
         }
         else break;  /* ended on an even boundry! */

         /*--------------------------------------------------------------
         | Now see if there are more items to process, if so do so.
         .---------------------------------------------------------------*/
         if (ll)
         {
            loop_item = 0;               /* Start new cycle */
            ibuff_cursor = rest_ibuff;   /* Point for front of buff */
            pbuff_cursor = part_pbuff;   /* Point for front of buff */
            llStart = ll;                /* New cursor into linked list */
         }
         else break;
      }
   }
   free(rest_ibuff);
   free(part_pbuff);
   return 0;
}

/*----------------------------------------------------------------------+
| Name:    QryDoRestoreObjs
|
| Action:  We now have a list of object id's to restore.  Create the
|          calls to receive the data and validate it as it is received.
|
| Input:   llStart : Pointer to the linked list item that is first to be
|                    restored.  Each item in this list has a 1-1
|                    correspondance with the rest_ibuff list of obj ids.
|
|          rest_ibuff : Restore index buffer with object id's to resore.
|                       This is input to the api getdata calls.
|
|          items : Number of items to process.  Not really needed but
|                  good info to have for human output.
|
| Returns: None
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static dsInt16_t QryDoRestoreObjs(dsUint16_t  get_type,
                              GetInfo *llStart,
                              char    *rest_ibuff,
                              dsUint32_t  items,
                              dsBool_t  mountWait,
                              char    *part_pbuff)
{
   dsInt16_t    rc;
   dsUint32_t   i;
   char         *get_buff;
   ObjID        *objId;
   DataBlk      dataBlk;
   GetInfo      *ll = NULL;
   dsBool_t     done;
   dsBool_t     validate_data = bFalse;
   dsmGetList   getList;
   smpObjInfo   *objInfo;
   seedValidate seedV;
   char         t[30];
   dsStruct64_t ver_rc;

   if (!(get_buff = (char*)malloc(pref.sz_recv_buff)))
   {
      printf("Memory alloc error on get_guff.\n");
      return RC_NO_MEMORY;
   }

   getList.stVersion = dsmGetListPORVersion;
   getList.numObjId  = items;
   getList.objId     = (ObjID *)rest_ibuff;
   getList.partialObjData = (PartialObjData *)part_pbuff;
   /*-----------------------------------------------------------------------
   |  Issue the BeginGetData call with the list of ojbect id's we will be
   |  obtaining from the server.  This is then followed with a get for
   |  each separate object id and its associated data.  As data is
   |  received it will be verified against the actual data sent by looking
   |  at the objInfo field to calculate correctness.
   .-----------------------------------------------------------------------*/
   switch(get_type)
   {
      case (Restore_Get) :
        rc = dsmBeginGetData(dsmHandle,mountWait,gtBackup,&getList);
        break;
      case (Retrieve_Get) :
        rc = dsmBeginGetData(dsmHandle,mountWait,gtArchive,&getList);
        break;
      default : ;
   }

   if (rc)
   {
      printf("*** dsmBeginGetData failed: \n");
      rcApiOut(dsmHandle, rc);
      return rc;
   }

   ll = llStart;
   objId = (ObjID *)rest_ibuff;
   if (!(seedV.seed = (char*)malloc(pref.sz_send_buff)))
   {
      printf("Malloc failed for validation seed buff.\n");
      return RC_NO_MEMORY;
   }
   seedV.seed_string_size = pref.sz_send_buff ;

   /*-----------------------------------------------------------------------
   |  Now go get each object from the list for restore or retrieve purposes.
   |  Each object will be verified with data stored in the info field when
   |  it was sent to insure integrity of the test.
   .-----------------------------------------------------------------------*/
   for (i=0;(i < items) && ll; i++)
   {
      dataBlk.stVersion = DataBlkVersion;
      dataBlk.bufferLen = pref.sz_recv_buff;
      dataBlk.numBytes  = 0;
      dataBlk.bufferPtr = get_buff;

      dI64toCh(&(ll->item64),t,10);

      if (get_type == Restore_Get)
        printf("Restoring object %s: %s%s%s :\n",t,
                ll->objName->fs,ll->objName->hl,ll->objName->ll);
      else  /* Retrieve_Get */
        printf("Retrieving object %s: %s%s%s :\n",t,
                ll->objName->fs,ll->objName->hl,ll->objName->ll);

      /*--------------------------------------------------------------------
      |  See if this was an object the sample api sent to the server.  If
      |  so init structures with object attributes for data validation
      |  checks.  If not our data then bypass data verification checks.
      .--------------------------------------------------------------------*/
      validate_data = bFalse;
      if (ll->objInfolen >= sizeof(smpObjInfo))   /* Even big enough for us?*/
      {
         objInfo = (smpObjInfo *)ll->objInfo;
         if (!(strcmp(api_eyecatcher,objInfo->eyecatcher)))
         {
            validate_data = bTrue;
            seedV.seed_len = 0;
            seedV.orig_sz.hi = objInfo->size.hi;
            seedV.orig_sz.lo = objInfo->size.lo;
            seedV.total_sz64.hi = 0;
            seedV.total_sz64.lo = 0;
            seedV.seed_cursor = seedV.seed;
            seedV.first_time = bTrue;

            QryMakeSeed(&seedV,objInfo->seed);
         }
      }

      rc = dsmGetObj(dsmHandle,objId,&dataBlk);

      /*--------------------------------------------------------------------
      |  Go check the results of the above get, and loop through getdata
      |  calls as needed until all data is obtained from the server.  Note
      |  that each buffer received will be validated to insure data
      |  integrity if the data in the objInfo field was generated by the
      |  sample api program.  Otherwise data validation is skipped.
      .--------------------------------------------------------------------*/
      done = bFalse;
      ver_rc = dMake64(0);
      while(!done)
      {
         if (   (rc == DSM_RC_MORE_DATA)
             || (rc == DSM_RC_FINISHED))
         {

            /* validate data here */
            if (validate_data)
            {
               ver_rc = dIncr64(&ver_rc, Qry_Validate_Data(&dataBlk,&seedV,i,part_pbuff));
            }

            if (rc == DSM_RC_MORE_DATA)
            {
               dataBlk.numBytes = 0;
               rc = dsmGetData(dsmHandle,&dataBlk);
            }
            else
            {
               done = bTrue;
               if (validate_data)
               {
                  if (dCmp64(&(seedV.total_sz64),&(seedV.orig_sz)) != DS_EQUAL)
                  {
                     if (!(ver_rc.hi || ver_rc.lo))
                        printf("Object content verified\n");
                     dI64toCh(&(seedV.total_sz64),t,10);
                     printf(" Object length returned: %s\n", t );
                     dI64toCh(&(seedV.orig_sz),t,10);
                     printf(" Original object length: %s\n", t );
                     }
                  else
                  {
                      if (!(ver_rc.hi || ver_rc.lo))
                        printf("Object content and length verified!\n");
                  }
                     
               }
            else printf(" Done!\n");
            }
         }
         else
         {
            printf("*** dsmGetObj or dsmGetData failed: \n");
            rcApiOut(dsmHandle, rc);
            if ( rc == DSM_RC_ENC_WRONG_KEY )
            {  
                printf("You have been disconnected from the server.\n");
                printf("Please restart the session with the correct encryption password.\n");
                dsmTerminate(dsmHandle);
                free(seedV.seed);  
                free(get_buff);
                return rc;
            }               
            else
               done = bTrue;
         }
      } /* while */
      rc = dsmEndGetObj(dsmHandle);
      if (rc)
      {
         printf("*** dsmEndGetObj failed: \n");
         rcApiOut(dsmHandle, rc);
         /* do not return let sample continue */
      }

      ll = ll->next;
      objId++;  /* Next object to process. */
   } /* for */

   rc = dsmEndGetData(dsmHandle);
   if (rc)
   {
      printf("*** dsmEndGetData failed: \n");
      rcApiOut(dsmHandle, rc);
      /* do not return let sample continue */
   }
   free(seedV.seed); 
   free(get_buff);
   return 0;
}


/*----------------------------------------------------------------------+
| Name:    QryDoDelete()
|
| Action:  A linked list of items to delete has been created.  Process
|          this list within transaction boundaries to delete the objects.
|
| Input:
|
| Returns: None
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static dsInt16_t QryDoDelete(dsUint16_t delete_type,
                         GetInfo *llHead,
                         GetInfo *llTail,
                         dsStruct64_t  items64)
{
   GetInfo *ll = NULL;                  /* Pointers to objects in the list */
   dsInt16_t  rc;
   char   buff[INPUT_BUFFER_SIZE];
   dsmDelInfo delInfo;
   dsmDelType delType;
   dsBool_t  in_txn = bFalse;
   dsBool_t  new_txn_needed = bFalse;
   dsUint16_t  txn_reason;
   dsUint32_t  txn_objs;
   dsmObjName objName;
   char t[50], t2[50], t4[50];

   if (!llHead) return 0;  /* No items in list to delete */

   ll = llHead;                    /* Init list traversal items */

   dI64toCh(&items64,t4,10); format_number(t4,t);
   printf("Starting delete of %s objects.\nPress enter to proceed.\n",t);
   ourGetS(buff);

   /*----------------------------------------------------------------------+
   | Bump through the linked list deleting objects within the bounds of
   | multiple object transactions.
   .----------------------------------------------------------------------*/
   while (ll)
   {
      dI64toCh(&(ll->item64),t4,10);  format_number(t4,t2);
      printf("Deleting object %s of %s: %s%s%s\n",
             t2,t,ll->objName->fs,ll->objName->hl,ll->objName->ll);

      if (!in_txn)
      {
         if ((rc = QryBeginTxn(dsmHandle,     /* DSM API sess handle         */
                              &txn_objs)))    /* Objects in this txn.        */

         {
            printf("*** dsmBeginTxn(1) failed: \n");
            rcApiOut(dsmHandle, rc);
            rc = (RC_SESSION_FAILED);
            break;
         }
         /*-----------------------------------------------------------------.
         | If we get here we are in session and within the bounds of a txn
         | and all of our transaction counters, etc., are initialized.
         .-----------------------------------------------------------------*/
         new_txn_needed = bFalse;
         in_txn = bTrue;
      }

      /*--------------------------------------------------------------------.
      | Are we within the max number of objects for a transaction?
      .--------------------------------------------------------------------*/
      if (txn_objs >= dsmSessInfo.maxObjPerTxn)
      {
         new_txn_needed = bTrue;
         printf("End txn: Current txn objects (%u) >= max txn objects(%u).\n",
               txn_objs,dsmSessInfo.maxObjPerTxn);
      }

      /*--------------------------------------------------------------------.
      | See if a new txn is needed.  If so end current txn and start a new
      | one.
      .--------------------------------------------------------------------*/
      if (new_txn_needed)
      {
         txn_reason = 0;
         rc = dsmEndTxn(dsmHandle,        /* DSM API sess handle         */
                        DSM_VOTE_COMMIT,  /* Commit transaction.         */
                        &txn_reason);     /* Reason if txn aborted.      */
         if (rc || txn_reason)
         {
            printf("*** dsmEndTxn failed: \n");
            rcApiOut(dsmHandle, rc);
            printf("   Reason = ");
            rcApiOut(dsmHandle, txn_reason);
            rc = (RC_SESSION_FAILED);
            in_txn = bFalse;
            break;
         }

         if ((rc = QryBeginTxn(dsmHandle,     /* DSM API sess handle         */
                             &txn_objs)))    /* Objects in this txn.        */


         {
            printf("*** dsmBeginTxn(2) failed: \n");
            rcApiOut(dsmHandle, rc);
            rc = (RC_SESSION_FAILED);
            break;
         }
         new_txn_needed = bFalse;
      }


      /*--------------------------------------------------------------------.
      | Now we can build and send the delete object api call.
      | We are in a transaction and are within the bounds of a
      | single or multiple transaction.
      .--------------------------------------------------------------------*/
      switch(delete_type)
      {
         case Backup_Delete :
            strcpy(objName.fs,ll->objName->fs);
            strcpy(objName.hl,ll->objName->hl);
            strcpy(objName.ll,ll->objName->ll);
            objName.objType = ll->objName->objType;
            delInfo.backInfo.stVersion = delBackVersion;
            delInfo.backInfo.objNameP  = &objName;
            delInfo.backInfo.copyGroup = ll->copyGroup;
            delType = dtBackup;
            break;
         case Archive_Delete :
            delInfo.archInfo.stVersion = delArchVersion;
            delInfo.archInfo.objId.hi  = ll->objId.hi;
            delInfo.archInfo.objId.lo  = ll->objId.lo;
            delType = dtArchive;
            break;
         default : ;
      }

      rc = dsmDeleteObj(dsmHandle,delType,delInfo);

      if (rc)
      {
         printf("*** dsmDeleteObj failed: \n");
         rcApiOut(dsmHandle, rc);
         rc = (RC_SESSION_FAILED);
         break;
      }
      ll = ll->next;
      txn_objs++;
   }

   if (in_txn)
   {
      txn_reason = 0;
      rc = dsmEndTxn(dsmHandle,          /* DSM API sess handle         */
                    DSM_VOTE_COMMIT,     /* Commit transaction.         */
                    &txn_reason);        /* Reason if txn aborted.      */
      if (rc || txn_reason)
      {
         printf("*** dsmEndTxn(2) failed: \n");
         rcApiOut(dsmHandle, rc);
         printf("   Reason = ");
         rcApiOut(dsmHandle, txn_reason);
         rc = (RC_SESSION_FAILED);
      }

   }
   return rc;
}


/*-----------------------------------------------------------------------
|  Free the passed linked list and all associated storage
.-----------------------------------------------------------------------*/
static dsInt16_t QryFreeGetLlist(GetInfo **llHead,
                            GetInfo **llTail)
{
   GetInfo *ll = NULL;

   ll = *llHead;
   while (ll)
   {
      *llHead = ll->next;
      QryFreeGetLlistItem(&ll);
      ll = *llHead;
   }
   *llHead = NULL;
   *llTail = NULL;
   return 0;
}

/*-----------------------------------------------------------------------
|  Free associated storage for the passed item.
.-----------------------------------------------------------------------*/
static dsInt16_t QryFreeGetLlistItem(GetInfo **item)
{
   GetInfo *ll = NULL;

   ll = *item;
   free(ll->objName->fs);
   free(ll->objName->hl);
   free(ll->objName->ll);
   free(ll->objName);
   free(ll->objInfo);
   free(*item);
   *item = NULL;
   return 0;
}

/*----------------------------------------------------------------------+
| Name:    QryValidateData
|
| Action:  Check the data in the seed string against the returned buffer
|          to insure proper match.
|
| Input:
|
| Returns: None
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static dsInt16_t Qry_Validate_Data(DataBlk *dataBlk,seedValidate *seedV, dsUint32_t item, char *part_pbuff)
{
   dsUint32_t   i;
   dsStruct64_t   num_of_buffers, rmdr, temp64;
   PartialObjData  *t_buff;

   /* Find the partial object data for this object */
   if (part_pbuff && seedV->first_time )
      {
      t_buff = (PartialObjData *)
        (part_pbuff + (sizeof(PartialObjData) * item)) ;

      temp64 = dMake64(pref.sz_send_buff);
      num_of_buffers = dDiv64(&(t_buff->partialObjOffset),
                             &temp64,
                             &rmdr);
      seedV->seed_cursor = seedV->seed + rmdr.lo;
      seedV->first_time = bFalse ;
  }
   /* Verify each byte returned */
   for (i=0; i < dataBlk->numBytes; i++)
   {
      if ((seedV->seed_cursor - seedV->seed) == seedV->seed_string_size)
         seedV->seed_cursor = seedV->seed ; /* reset cursor */
      if (dataBlk->bufferPtr[i] == *seedV->seed_cursor)
         {
         seedV->total_sz64 = dIncr64(&(seedV->total_sz64), (dsUint32_t) 1);
         seedV->seed_cursor += 1 ;
         }
      else
         {
         printf("** Error: Validation failed ** \n");
         return 1;
         }

   }
return 0;
}

/*----------------------------------------------------------------------+
| Name:    QryBeginTxn()
|
| Action:  Do a begin txn to the API as well as .............
|
| Input:
|
| Returns:
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static dsInt16_t QryBeginTxn(dsUint32_t  dsmHandle,        /* DSM API sess handle  */
                         dsUint32_t  *txn_objs)        /* Objects in this txn. */

{
dsInt16_t rc;

rc = dsmBeginTxn(dsmHandle);
*txn_objs = 0;
return rc;
}

/*----------------------------------------------------------------------+
| Name:    QryPrintMediaClass
|
| Action:  Output text for the media class passed.
+----------------------------------------------------------------------*/
static void QryPrintMediaClass(dsUint8_t mediaClass)
{
   printf("   Media class: ");
   switch (mediaClass)
   {
      case (MEDIA_FIXED) :
         printf("Fixed\n");       break;
      case (MEDIA_LIBRARY) :
         printf("Library\n");     break;
      case (MEDIA_NETWORK) :
         printf("Networked\n");   break;
      case (MEDIA_SHELF) :
         printf("Shelf\n");       break;
      case (MEDIA_OFFSITE) :
         printf("Offsite\n");     break;
      case (MEDIA_UNAVAILABLE) :
         printf("Unavailable\n"); break;
      default :
          printf("*** Unknown : %iu ***\n",(dsUint16_t) mediaClass);
   }
}

/*----------------------------------------------------------------------+
| Name:    QryPrintObjType
|
| Action:  Output text for the object type passed.
+----------------------------------------------------------------------*/
static void QryPrintObjType(dsUint8_t objType)
{
   printf("   Object type: ");
   switch (objType)
   {
      case (DSM_OBJ_FILE) :
         printf("File\n");        break;
      case (DSM_OBJ_DIRECTORY) :
         printf("Directory\n");   break;
      default :
         printf("*** Unknown : %iu ***\n",(dsUint16_t) objType);
   }
}

/*----------------------------------------------------------------------+
| Name:    QryPrintObjInfo
|
| Action:  Output text for the object info.
+----------------------------------------------------------------------*/
static void  QryPrintObjInfo(dsUint16_t objInfolen, char *objInfo)
{
  int i;
  smpObjInfo  *smpobjInfo;

  smpobjInfo = (smpObjInfo *) objInfo;
  printf("   Object info is        :");

  if( !(strcmp(api_eyecatcher,smpobjInfo->eyecatcher)) )
     printf("%s", api_eyecatcher);

  else
     for (i = 0; i < objInfolen; i++)
     {
           
         /*   if ( isalnum(objInfo[i])) break; 
            if ( objInfo[i] == '\0') break; */ 
            printf("%c", *objInfo++);
     }
  printf("\n   Object info length is :%d\n",objInfolen);

}

/*----------------------------------------------------------------------+
| Name:    QryPrintObjState
|
| Action:  Output text for the object state passed.
+----------------------------------------------------------------------*/
static void QryPrintObjState(dsUint8_t objState)
{
   printf("   Object state: ");
   switch (objState)
   {
      case (DSM_ACTIVE) :
         printf("Active\n");     break;
      case (DSM_INACTIVE) :
         printf("Inactive\n");   break;
      default :
         printf("*** Unknown : %iu ***\n",(dsUint16_t) objState);
   }
}

/*----------------------------------------------------------------------+
| Name:    QryPrintCopySer
|
| Action:  Output text for the copy serialization value passed.
+----------------------------------------------------------------------*/
static void QryPrintCopySer(dsUint8_t copySer)
{
   printf("   Copy Serialization: ");
   switch (copySer)
   {
      case (Copy_Serial_Static) :
         printf("Static\n");          break;
      case (Copy_Serial_Shared_Static) :
         printf("Shared Static\n");   break;
      case (Copy_Serial_Shared_Dynamic) :
         printf("Shared Dynamic\n");  break;
      case (Copy_Serial_Dynamic) :
         printf("Dynamic\n");         break;
      default :
         printf("*** Unknown : %iu ***\n",(dsUint16_t) copySer);
   }

}


/*----------------------------------------------------------------------+
| Name:    QryPrintCopyMode
|
| Action:  Output text for the copy mode value passed.
+----------------------------------------------------------------------*/
static void QryPrintCopyMode(dsUint8_t copyMode)
{
   printf("            Copy Mode: ");
   switch (copyMode)
   {
      case (Copy_Mode_Modified) :
         printf("Modified");        break;
      case (Copy_Mode_Absolute) :
         printf("Absolute");        break;
      default :
         printf("*** Unknown : %iu ***\n",(dsUint16_t) copyMode);
   }
}



/*----------------------------------------------------------------------+
| Name:    QryAddFsLlist()
|
| Action:  Add the passed structure to the passed linked list.  Handle
|          the standard situation of new list or existing list.  Note
|          list structure is malloc'd and fields within the list are also
|          malloc'd.  These will all be freed by the corresponding free
|          llist routine.  A more efficient method would be to use a
|          memory pool manager package for these small memory allocs, but
|          then this is just a sample app. for the api!
|
| Input:
|
| Returns: None
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static dsInt16_t QryAddFsLlist(dsUint32_t  qry_item,
                           fsInfo **llHead,
                           fsInfo **llTail,
                           qryRespFSData *fsResp)

{
   fsInfo  *newP = NULL, *ll;

   if (!(newP = (fsInfo *)malloc(sizeof(fsInfo))))
     return RC_NO_MEMORY;

   memset(newP,0x00,sizeof(fsInfo));
   /*--------------------------------------------------------------------
   | First get memory areas for other parts of new item.
   .--------------------------------------------------------------------*/
   if (!(newP->fsName = (char*)malloc(strlen(fsResp->fsName) + 1)))
   {
      free(newP); return RC_NO_MEMORY;
   }

   if (!*llHead)
   {
     *llHead = newP;
     *llTail = newP;
   }
   else
   {
      ll = *llTail;
      ll->next = newP;
      *llTail  = newP;
   }

   newP->next = NULL;
   newP->item = qry_item;
   strcpy(newP->fsName,fsResp->fsName);

   if (pref.verbose)
   {
      printf("Inserted fs %s into the linked list.\n",newP->fsName);
   }

   return 0;
}

/*-----------------------------------------------------------------------
|  Free the passed linked list and all associated storage
.-----------------------------------------------------------------------*/
static dsInt16_t QryFreeFsLlist(fsInfo **llHead,
                            fsInfo **llTail)
{
   fsInfo *ll;

   ll = *llHead;
   while (ll)
   {
      *llHead = ll->next;
      free(ll->fsName);
      free(ll);
      ll = *llHead;
   }
   *llHead = NULL;
   *llTail = NULL;
   return 0;
}


/*----------------------------------------------------------------------+
| Name:    QryDoFsDelete()
|
| Action:  A linked list of fs's to delete has been created.  Process
|          this list to delete the filespaces.
|
| Input:
|
| Returns: None
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static dsInt16_t QryDoFsDelete(fsInfo *llHead,
                           fsInfo *llTail,
                           dsUint32_t  items)
{
   fsInfo *ll;                         /* Pointers to objects in the list */
   dsInt16_t  rc;
   char   buff[INPUT_BUFFER_SIZE];
   char   t[50], t2[50], t4[50];

   if (!llHead) return 0;          /* No items in list to delete */

   ll = llHead;                    /* Init list traversal items */

   sprintf(t4,"%u",items);  format_number(t4,t);
   printf("Starting delete of %s filespaces.\nPress enter to proceed.\n",t);
   ourGetS(buff);
   if (*buff)
   {
      printf("Filespace delete terminating at user request.\n");
      return 0;
   }

   /*----------------------------------------------------------------------+
   | Bump through the linked list deleting filespaces.
   .----------------------------------------------------------------------*/
   while (ll)
   {
      sprintf(t4,"%u",ll->item); format_number(t4,t2);
      printf("\nDeleting filespace %s of %s: %s",t2,t,ll->fsName);
      rc = dsmDeleteFS(dsmHandle,ll->fsName,DSM_REPOS_ALL);
      if (rc)
      {
         printf("  FAILED!!! ");
         rcApiOut(dsmHandle, rc);
      }
      else printf("  OK!\n");

   ll = ll->next;
   }

   return 0;
}


/*----------------------------------------------------------------------+
| Name:    QryMakeSeed()
|
| Action:  Creates the seed string as it was originally built when
|          stored.
|
| Input:
|
| Returns: None
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static dsInt16_t QryMakeSeed(seedValidate *seedV,
                         char *seed_str)
{
char *cursor, *end_of_buff;
char seed_count[15] = "0";
int i,x,seed_len;

   seed_len = strlen(seed_str);
   cursor = seedV->seed;
   end_of_buff = seedV->seed + pref.sz_send_buff;
   while (end_of_buff > (cursor + seed_len + strlen(seed_count) + 1))
{
      memcpy(cursor,seed_str,seed_len);
      cursor += seed_len;
      x = sprintf(seed_count, "%d", (int)((cursor - seedV->seed) + 1));
      memcpy(cursor,seed_count,strlen(seed_count));
      cursor += strlen(seed_count);
   }
   for (i = end_of_buff - cursor; i ; --i )
   {
      *cursor = 0xFF ;
      cursor += 1 ;
   }

   return 0;
}

/*----------------------------------------------------------------------+
| Name:    SortObjects()
|
| Action:  Sort all the objects to be restored and returned them in the
|          buffer
|
| Input:
|          GetInfo  *llHead;   link list of obj information
|          int      loop_item  items to sort
|
| Returns: rest_ibuff           obj information is sorted
|
| Side
| Effects: None
|
| Notes:   max objects that can be restored in this sample is 3000.  When
|          declaring GetInfo array, this funciton uses only 100 items.
|          If want to increase, change to appropriate number.
|
+----------------------------------------------------------------------*/


static void SortObjects(GetInfo *llHead,
                        char *rest_ibuff, int loop_item)


{
   GetInfo *ll = llHead;
   GetInfo sortorder[100];   /* can only sort 100 items */
   ObjID   *localObjId;

   int  i = 0;


/*************************************************/
/*  transfer the contents of ll to sortorder     */
/*************************************************/

   for(i = 0; i < loop_item; i++)
   {
      sortorder[i].next = ll->next;
      sortorder[i].item64 = ll->item64;
      sortorder[i].objName = ll->objName;
      sortorder[i].objId = ll->objId;
      sortorder[i].copyGroup = ll->copyGroup;
      sortorder[i].restoreOrderExt = ll->restoreOrderExt;
      sortorder[i].mediaClass = ll->mediaClass;
      sortorder[i].objInfolen = ll->objInfolen;
      sortorder[i].objInfo = ll->objInfo;

      ll = ll->next;
   }


   qsort((void *)sortorder, loop_item, sizeof(GetInfo), SortRestoreOrder);


   /**********************************************/
   /* re-arrange the structure in sorted order   */
   /**********************************************/
   ll = llHead;

   for(i = 0; i < loop_item; i++)
   {
      ll->item64 = sortorder[i].item64;
      ll->objName = sortorder[i].objName;
      ll->objId = sortorder[i].objId;
      ll->copyGroup = sortorder[i].copyGroup;
      ll->restoreOrderExt = sortorder[i].restoreOrderExt;
      ll->mediaClass = sortorder[i].mediaClass;
      ll->objInfolen = sortorder[i].objInfolen ;
      ll->objInfo = sortorder[i].objInfo;

      ll = ll->next;
   }

   /**********************************************/
   /* transfer objIDs into rest_ibuff            */
   /**********************************************/

   localObjId = (ObjID *)rest_ibuff;

   for (i = 0; i < loop_item; i++)
   {
      localObjId->hi = sortorder[i].objId.hi;
      localObjId->lo = sortorder[i].objId.lo;
      localObjId++;
   }

}  /* static void SortObjects() */


/*----------------------------------------------------------------------+
| Name:    SortRestoreOrder()
|
| Action:  This function compares the restoreOrderExt field from the two
|          structures passed.
|
|
| Input:
|          GetInfo *a, *b       two structures to compare
|
| Returns: if a->restoreOrderExt > b->restoreOrderExt (return DS_GREATERTHAN)
|          if a->restoreOrderExt < b->restoreOrderExt (return DS_LESSTHAN)
|          if a->restoreOrderExt = b->restoreOrderExt (return DS_EQUAL)
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/


int SortRestoreOrder(const void *ptr1, const void *ptr2)

{
    GetInfo *a = (GetInfo *)ptr1;
    GetInfo *b = (GetInfo *)ptr2;

    if (a->restoreOrderExt.top > b->restoreOrderExt.top)
       return(DS_GREATERTHAN);

    else if (a->restoreOrderExt.top < b->restoreOrderExt.top)
       return(DS_LESSTHAN);

    else if (a->restoreOrderExt.hi_hi > b->restoreOrderExt.hi_hi)
       return(DS_GREATERTHAN);

    else if (a->restoreOrderExt.hi_hi < b->restoreOrderExt.hi_hi)
       return(DS_LESSTHAN);

    else if (a->restoreOrderExt.hi_lo > b->restoreOrderExt.hi_lo)
       return(DS_GREATERTHAN);

    else if (a->restoreOrderExt.hi_lo < b->restoreOrderExt.hi_lo)
       return(DS_LESSTHAN);

    else if (a->restoreOrderExt.lo_hi > b->restoreOrderExt.lo_hi)
       return(DS_GREATERTHAN);

    else if (a->restoreOrderExt.lo_hi < b->restoreOrderExt.lo_hi)
       return(DS_LESSTHAN);

    else if (a->restoreOrderExt.lo_lo > b->restoreOrderExt.lo_lo)
       return(DS_GREATERTHAN);

    else if (a->restoreOrderExt.lo_lo < b->restoreOrderExt.lo_lo)
       return(DS_LESSTHAN);

    else
       return(DS_EQUAL);
}





