//
// Author: 
//   Mikael Hallendal <micke@imendio.com>
//
// (C) 2004 Imendio AB
// 

using GConf;
using Glade;
using Gdk;
using Gtk;
using GtkSharp;
using Gnome;
using Mono.Posix;
using System;
using System.Collections;
using System.IO;
using System.Text;

namespace Imendio.Blam {

    public class Application : Program {
	public static Application TheApp;
	
	[Widget] Gtk.Window         mainWindow;
	[Widget] Gtk.ScrolledWindow channelListSw;
	[Widget] Gtk.ScrolledWindow itemListSw;
	[Widget] Gtk.Paned          channelPaned;
	[Widget] Gtk.Paned          itemPaned;
	[Widget] Gtk.Label          blogNameLabel;
	[Widget] Gtk.Statusbar      statusbar;
        //[Widget] Gtk.HBox           labelSpinnerBox;
	[Widget] Gtk.Label          channelsLabel;
	private string              channelsLabelText;

	[Widget] Gtk.MenuItem       refreshChannelMenuItem;
	[Widget] Gtk.MenuItem       markChannelAsReadMenuItem;
	[Widget] Gtk.MenuItem       removeChannelMenuItem;
	[Widget] Gtk.MenuItem       editChannelMenuItem;
	
	[Widget] Gtk.MenuItem       markEntryAsUnreadMenuItem;
	[Widget] Gtk.MenuItem       nextUnreadMenuItem;
	
        //[Widget] Gtk.MenuItem       printMenuItem;

	private Gtk.FileSelection   exportFileDialog;
	
        //private Spinner spinner;

	private ChannelList channelList;
	private ItemList    itemList;
	private ItemView    itemView;

	private TrayIcon    trayIcon;

	public static string BaseDir;
	
	private MessageConnection messageConn;

	private AddChannelDialog  addChannelDialog;
	private ChannelDialog     channelDialog;
	private PreferencesDialog preferencesDialog;
	private OpmlDialog        opmlDialog;
	
	private ChannelCollection mCollection;

	private uint              mAutoRefreshId;

	enum TargetType {
	    String,
	    UriList
	};

	public static TargetEntry[] DragEntries = new TargetEntry[] {
	    new TargetEntry("STRING", 0, (uint)TargetType.String),
	    new TargetEntry("text/plain", 0, (uint)TargetType.String),
	    new TargetEntry("text/uri-list", 0, (uint)TargetType.UriList)
	};

	public Gtk.Window Window {
	    get {
		return (Gtk.Window) mainWindow;
	    }
	}

	public ChannelCollection CCollection {
	    get {
		return mCollection;
	    }
	}

	public ItemList ItemList {
	    get {
		return itemList;
	    }
	}
	
        public Application (string[] args, params object[] props) : base ("Blam", Defines.VERSION, Modules.UI, args, props)
        {
	    messageConn = new MessageConnection ("Blam");
	    if (!messageConn.IsServer) {
		messageConn.Send ("ShowWindow");
		messageConn.Close ();
		Gdk.Global.NotifyStartupComplete ();
		Environment.Exit (0);
	    }

	    messageConn.SetCallback (new MessageConnection.MessageReceivedHandler (MessageReceivedCb));

	    Proxy.InitProxy ();
            Utils.GeckoInit ();

	    string homeDirectory = Environment.GetEnvironmentVariable("HOME");

	    Mono.Posix.Catalog.Init ("blam", Defines.GNOME_LOCALE_DIR);

            BaseDir = homeDirectory + "/.gnome2/blam";
	    if (!File.Exists(BaseDir)) {
                Directory.CreateDirectory(BaseDir);
	    }

            mCollection = ChannelCollection.LoadFromFile (BaseDir + "/collection.xml");

	    mCollection.ChannelUpdated         += ChannelUpdatedCb;
	    mCollection.ChannelAdded           += ChannelAddedCb;
	    mCollection.ChannelRemoved         += ChannelRemovedCb;
	    mCollection.ChannelRefreshStarted  += ChannelRefreshStartedCb;
	    mCollection.ChannelRefreshFinished += ChannelRefreshFinishedCb;

            PrepareGUI();

	    Conf.AddNotify (new NotifyEventHandler (ConfNotifyHandler));
            GLib.Idle.Add (new GLib.IdleHandler (IdleRefreshAll));
	}
	    
	private void PrepareGUI()
	{
	    Glade.XML gladeXML = Glade.XML.FromAssembly("blam.glade", 
							"mainWindow", null);
	    gladeXML.Autoconnect(this);
					      
	    channelList = new ChannelList(mCollection.Channels);
	    ((Container)channelListSw).Child = channelList;

	    channelList.ChannelSelectedEvent   += ChannelSelected;
	    channelList.EditChannelEvent       += EditChannelActivated;
	    channelList.MarkChannelAsReadEvent += MarkChannelAsReadActivated;
	    channelList.RemoveChannelEvent     += RemoveChannelActivated;
	    channelList.UpdateChannelEvent     += RefreshChannelActivated;
	   
	    itemView = new ItemView (mCollection);

            Frame f = new Frame ();
            f.Shadow = ShadowType.In;
            f.Add (itemView.Widget);
            itemPaned.Add2 (f);
            f.Show ();
            itemView.OnUrl += OnUrl;

	    itemList = new ItemList(itemView);
            ((Container)itemListSw).Child = itemList;

	    itemList.ItemSelected += ItemSelected;

	    trayIcon = new TrayIcon (Catalog.GetString ("Imendio Blam News Reader"));
	    trayIcon.Image.Pixbuf = Gdk.Pixbuf.LoadFromResource ("blam-tray-icon.png");
	    trayIcon.ButtonPressEvent += TrayIconButtonPressCb;
/*
	    spinner = new Spinner();
	    labelSpinnerBox.PackEnd(spinner, false, false, 0);

	    SizeGroup sizeGroup = new SizeGroup(SizeGroupMode.Vertical);
	    sizeGroup.AddWidget(channelsLabel);
	    sizeGroup.AddWidget(labelSpinnerBox);
*/
	    channelsLabelText = channelsLabel.Text;
	    UpdateTotalNumberOfUnread ();
	    
	    //printMenuItem.Sensitive = false;
	    SensitizeChannelMenuItems(false);

	    // Setup drag-n-drop
	    Gtk.Drag.DestSet(mainWindow, DestDefaults.All, 
			     DragEntries, DragAction.Copy | DragAction.Move);
	    mainWindow.DragDataReceived += DragDataReceivedCb;

	    RestoreWindowState();

	    mainWindow.Icon = Gdk.Pixbuf.LoadFromResource("blam.png");
            
            mainWindow.ShowAll ();
	    
	    channelDialog = new ChannelDialog (this);
	    addChannelDialog = new AddChannelDialog (this);
	    preferencesDialog = new PreferencesDialog (this.Window);
	    opmlDialog = new OpmlDialog (this.Window);
	    opmlDialog.ChannelAdded += mCollection.Add;
	    opmlDialog.ImportFinished += OpmlImportFinished;

	}

	private void ChannelSelected(Channel channel)
	{
	    if (channel == null) {
		SensitizeChannelMenuItems(false);
		return;
	    }
	    
	    itemList.CurrentChannel = channel;
	    
	    blogNameLabel.Markup = "<b>" + channel.Name + "</b>";
	    mainWindow.Title = "Blam - " + channel.Name;
	    
	    SensitizeChannelMenuItems(true);
	}

	private void SensitizeChannelMenuItems (bool sensitive)
	{
	    refreshChannelMenuItem.Sensitive = sensitive;
	    removeChannelMenuItem.Sensitive = sensitive;
	    editChannelMenuItem.Sensitive = sensitive;
	    markChannelAsReadMenuItem.Sensitive = sensitive;
	}
	
	private void ItemSelected(Imendio.Blam.Item item)
	{
	    //spinner.Start();
	    itemView.CurrentItem = item;
	    //printMenuItem.Sensitive = true;
	}

	private void MarkEntryAsUnreadActivated (object obj, EventArgs args) 
	{
	    Item item = itemList.GetSelected ();
	    if (item == null) {
		return;
	    }

	    // Toggle unread status
	    item.SetUnread (!item.Unread, false);
	}

	private void MenuChannelActivated (object obj, EventArgs args)
	{
	    Channel channel = channelList.GetSelected ();

	    bool sensitive = true;
	    if (channel == null) {
		sensitive = false;
	    }
	    
	    SensitizeChannelMenuItems (sensitive);
	}

	private void MenuEntryActivated (object obj, EventArgs args)
	{
	    Item item = itemList.GetSelected ();

	    if (item == null) {
		markEntryAsUnreadMenuItem.Sensitive = false;
		return;
	    }

	    markEntryAsUnreadMenuItem.Sensitive = true;

	    string str = "";
	    if (!item.Unread) {
		str = Catalog.GetString ("_Mark as unread");
	    } else {
		str = Catalog.GetString ("_Mark as read");
	    }

	    ((Label) markEntryAsUnreadMenuItem.Child).TextWithMnemonic = str;
	}

	private void NextUnreadActivated(object obj, EventArgs args)
	{
	    if (channelList.NextUnread()) {
		itemList.NextUnread();
	    }
	}
  
	private void MainWindowKeyPressed (object obj, KeyPressEventArgs args)
	{
	    switch (args.Event.Key) {
	    case (Gdk.Key.period):
	    case (Gdk.Key.bracketright):
		// Couldn't figure out how to get this to the menu item itself.
		nextUnreadMenuItem.Activate ();
		break;
	    }
	}

        /*private void PrintActivated(object obj, EventArgs args)
	{ 
	    Blam.PrintDialog dialog = new Blam.PrintDialog(itemView);
	    dialog.Run();
	}*/

	private void ImportOpmlActivated (object obj, EventArgs args)
	{
	    opmlDialog.Show ();
	}

	private bool ShowFileExistsDialog (Gtk.Window parentWindow, string fileName)
	{
	    string str = String.Format (Catalog.GetString ("File {0} already exists"), fileName);
	    
	    string msg = Catalog.GetString ("Do you want to overwrite the file?");

	    Gtk.Dialog dialog = ConfirmationDialog.Create (parentWindow,
							   Catalog.GetString ("_Overwrite"),
							   str, msg);

	    int result = dialog.Run ();
	    dialog.Destroy ();

	    switch (result) {
	    case (int)ResponseType.Ok:
		return true;
	    }
	    
	    return false;
	}

	private void ExportOpmlActivated (object obj, EventArgs args)
	{
	    if (exportFileDialog == null) {
		exportFileDialog = new Gtk.FileSelection (Catalog.GetString ("Export to..."));
		exportFileDialog.Modal = true;
		exportFileDialog.ShowFileops = true;
		exportFileDialog.TransientFor = mainWindow;
	    }

	    bool finished = false;
	    bool write = false;
	    string fileName = "";

	    while (!finished) {
		int result = exportFileDialog.Run ();

		switch (result) {
		case (int)ResponseType.Ok:
		    fileName = exportFileDialog.Filename;
		    
		    if (!File.Exists (fileName)) {
			write = true;
			finished = true;
		    } else {
			write = ShowFileExistsDialog (exportFileDialog, fileName);
			if (write) {
			    finished = true;
			}
		    }
		    break;
		case (int)ResponseType.Cancel:
		    finished = true;
		    break;
		}
	    }
	    
	    exportFileDialog.Hide ();

	    if (write) {
		OpmlWriter.Write (mCollection, fileName);
	    }
	}

	private void OpmlImportFinished (string status)
	{
	    uint contextId = statusbar.GetContextId("status");
	   
	    if (status != null) {
		statusbar.Push(contextId, status);
	    }
	}

	private void QuitActivated(object obj, EventArgs args)
	{
	    SaveWindowState();
	    mainWindow.Hide();

	    mCollection.SaveToFile ();

	    Quit();
	}

	private void CopyActivated (object obj, EventArgs args)
	{
            // FIXME: Check how to do this 
            //itemView.Copy ();
	}

	private void PreferencesActivated (object obj, EventArgs args)
	{
	    preferencesDialog.Show ();
	}

	private void AboutActivated (object obj, EventArgs args)
	{
	    About.Show (mainWindow);
	}
	
	private void AddChannelActivated (object obj, EventArgs args)
	{
	    addChannelDialog.Show ();
	}

	private void EditChannelActivated (Channel channel) 
	{
	    if (channel != null) {
		channelDialog.Show (channel);
	    }
	}
	private void EditChannelActivated(object obj, EventArgs args)
	{
	    EditChannelActivated (channelList.GetSelected ());
	}

	private void MarkChannelAsReadActivated (Channel channel)
	{
	    if (channel.MarkAsRead ()) {
		CCollection.Update (channel);
	    }
	}

	private void MarkChannelAsReadActivated (object obj, EventArgs args)
	{
	    MarkChannelAsReadActivated (channelList.GetSelected ());
	}

	private void RemoveChannelActivated(Channel channel) 
	{
	    if (channel != null) {
		RemoveChannelDialog.Show (mainWindow, mCollection, channel);
	    }
	}

	private void RemoveChannelActivated(object obj, EventArgs args)
	{
	    Channel channel = channelList.GetSelected ();

	    RemoveChannelActivated(channel);
	}

	private void RefreshChannelActivated(Channel channel)
	{
	    if (channel != null) {
		mCollection.Refresh (channel);
	    }
	}
	private void RefreshChannelActivated(object obj, EventArgs args)
	{
	    Channel channel = channelList.GetSelected ();

	    RefreshChannelActivated(channel);
	}
	
	private void RefreshAllActivated(object obj, EventArgs args)
	{
	    mCollection.RefreshAll ();
            
	    StartStopAutoRefresh ();
	}

	private void DragDataReceivedCb(object o, DragDataReceivedArgs args)
	{
	    SelectionData d = args.SelectionData;
	    
	    if (d.Length < 0 && d.Format != 8){
		Gtk.Drag.Finish(args.Context, false, false, args.Time);
		return;
	    }
	   
	    Gtk.Drag.Finish(args.Context, true, true, args.Time);

	    UTF8Encoding encoding = new UTF8Encoding( );
	    string text = encoding.GetString(d.Data);

	    addChannelDialog.Show (text);
	}
	
        private void OnUrl (string url)
        { 
            uint contextId = statusbar.GetContextId("on_url");
            
            statusbar.Pop(contextId);
            if (url != null) {
                statusbar.Push(contextId, url);
            }
        }

	private void ChannelAddedCb (Channel channel)
	{
	    channelList.Add (channel);
	}
	
	private void ChannelUpdatedCb (Channel channel)
	{
	    channelList.Updated (channel);
	    UpdateTotalNumberOfUnread ();
	}

	private void ChannelRemovedCb (Channel channel) 
	{
	    channelList.Remove (channel);
	    UpdateTotalNumberOfUnread ();
	}

	private void ChannelRefreshStartedCb (Channel channel)
	{
	    uint contextId = statusbar.GetContextId("update-status");
	    string statusString;

	    statusString = String.Format (Catalog.GetString ("Refreshing: {0}"), channel.Name);

	    statusbar.Push (contextId, statusString);
	}

	private void ChannelRefreshFinishedCb (Channel channel)
	{
	    uint contextId = statusbar.GetContextId("update-status");

	    statusbar.Pop (contextId);
	    channelList.Updated (channel);
	    UpdateTotalNumberOfUnread ();

	    if (channelList.GetSelected () == channel) {
		itemList.UpdateList ();
	    }
	}

	private bool IdleRefreshAll ()
	{
	    mCollection.RefreshAll ();

	    StartStopAutoRefresh ();
	    return false;
	}

	private bool TimeoutRefreshAll ()
	{
	    mCollection.RefreshAll ();

	    /* Continue until source is removed */
	    return true;
	}

	private void StartStopAutoRefresh ()
	{
	    bool doAutoRefresh;
	    int  refreshRate;

	    if (mAutoRefreshId != 0) {
		GLib.Source.Remove (mAutoRefreshId);
		mAutoRefreshId = 0;
	    }

	    doAutoRefresh = Conf.Get (Preference.AUTO_REFRESH, false);
	    if (!doAutoRefresh) {
		return;
	    }

	    refreshRate = Conf.Get (Preference.AUTO_REFRESH_RATE, 15);
	    if (refreshRate >= 1) {
		mAutoRefreshId = GLib.Timeout.Add ((uint) (refreshRate * 60 * 1000),
						   new GLib.TimeoutHandler (TimeoutRefreshAll));
	    }
	}

	private void ConfNotifyHandler (object sender, NotifyEventArgs args)
	{
	    if (args.Key == Conf.GetFullKey (Preference.AUTO_REFRESH) ||
		args.Key == Conf.GetFullKey (Preference.AUTO_REFRESH_RATE)) {
		StartStopAutoRefresh ();
	    }
	}

	private void TrayIconButtonPressCb (object o, ButtonPressEventArgs args)
	{
	    mainWindow.Present ();
	}

        private void ItemLoadStart (object obj, EventArgs args)
        {
            System.Console.WriteLine ("Foo");
            //spinner.Start ();
        }

	private void ItemLoadDone(object obj, EventArgs args)
	{
            System.Console.WriteLine ("Bar");
            //spinner.Stop();
	}

	private void RestoreWindowState() 
	{
	    int width, height;
	    int position_x, position_y;

	    width = Conf.Get ("ui/main_window_width", 600);
	    height = Conf.Get("ui/main_window_height", 400);

	    mainWindow.Resize(width, height);

	    position_x = Conf.Get("ui/main_window_position_x", -1);
	    position_y = Conf.Get("ui/main_window_position_y", -1);

	    if (position_x >= 0 && position_y >= 0) {
		mainWindow.Move(position_x, position_y);
	    }

	    width = Conf.Get("ui/channel_list_width", 180);
	    channelPaned.Position = width;

	    height = Conf.Get("ui/item_list_height", 100);
	    itemPaned.Position = height;
	}

	private void SaveWindowState()
	{
	    int height, width;
	    int position_x, position_y;

	    mainWindow.GetSize(out width, out height);
	    
	    Conf.Set("ui/main_window_width", width);
	    Conf.Set("ui/main_window_height", height);

	    Conf.Set("ui/channel_list_width", channelPaned.Position);
	    Conf.Set("ui/item_list_height", itemPaned.Position);

	    mainWindow.GetPosition(out position_x, out position_y);

	    Conf.Set("ui/main_window_position_x", position_x);
	    Conf.Set("ui/main_window_position_y", position_y);

	    Conf.Sync ();
	}

	private void UpdateTotalNumberOfUnread ()
	{
	    int nrOfUnread;

	    nrOfUnread = mCollection.NrOfUnreadItems;

	    channelsLabel.Markup = string.Format("<b>" + 
						 channelsLabelText + 
						 "</b>",
						 nrOfUnread);
	    if (nrOfUnread > 0) {
		trayIcon.Show ();
		trayIcon.Tooltip = string.Format (Catalog.GetPluralString ("{0} unread item", "{0} unread items", nrOfUnread), 
						  nrOfUnread);
	    } else {
		trayIcon.Hide ();
	    }
	}
	
	private void MessageReceivedCb (string message, IntPtr userData)
	{
	    if (!message.Equals ("ShowWindow")) {
		return;
	    }

	    mainWindow.Present ();
	    mCollection.RefreshAll ();
	}

        public static void Main(string[] args) 
	{
            TheApp = new Application (args);
            TheApp.Run ();
	    // new Application(args).Run();
	}
    }
}
