/* GNOME DB libary
 * Copyright (C) 1998-2001, The Free Software Foundation
 *
 * AUTHORS:
 *	Michael Lausch <michael@lausch.at>
 *	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gnome-db-error.h"
#include "gnome-db-util.h"
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-stock.h>

struct _GnomeDbErrorPrivate {
	GList *current_list;
	gint current_pos;

	GtkWidget *error_number;
	GtkWidget *error_description;

	GtkWidget *detail_button;
	GtkWidget *detail_container;
	GtkWidget *error_source;
	GtkWidget *error_help_url;
	GtkWidget *error_sqlstate;
	GtkWidget *error_native;
	GtkWidget *error_command;
};

enum {
	GNOME_DB_NEXT,
	GNOME_DB_PREV,
	LAST_SIGNAL
};

static gint gnome_db_error_signals[LAST_SIGNAL] = {0,};

static void gnome_db_error_class_init (GnomeDbErrorClass* klass);
static void gnome_db_error_init       (GnomeDbError* error);
static void gnome_db_error_destroy    (GtkObject *object);

/*
 * Private functions
 */
static void
display_current_error (GnomeDbError *error_widget)
{
	GdaError *error;
	GList *l;
	gchar *tmp;

	g_return_if_fail (GNOME_DB_IS_ERROR (error_widget));

	l = g_list_nth (error_widget->priv->current_list,
			error_widget->priv->current_pos);
	if (!l)
		return;

	error = GDA_ERROR (l->data);
	tmp = g_strdup_printf ("%d", gda_error_get_number (error));
	gtk_entry_set_text (GTK_ENTRY (error_widget->priv->error_number), tmp);
	g_free (tmp);

	tmp = gda_error_get_description (error);
	gtk_editable_delete_text (GTK_EDITABLE (error_widget->priv->error_description), 0, -1);
	gtk_text_insert (GTK_TEXT (error_widget->priv->error_description),
			 NULL, NULL, NULL, tmp, strlen (tmp));

	gtk_entry_set_text (GTK_ENTRY (error_widget->priv->error_source),
			    gda_error_get_source (error));
	gtk_entry_set_text (GTK_ENTRY (error_widget->priv->error_help_url),
			    gda_error_get_help_url (error));
	gtk_entry_set_text (GTK_ENTRY (error_widget->priv->error_sqlstate),
			    gda_error_get_sqlstate (error));
	gtk_entry_set_text (GTK_ENTRY (error_widget->priv->error_native),
			    gda_error_get_native (error));
	gtk_entry_set_text (GTK_ENTRY (error_widget->priv->error_command),
			    gda_error_get_real_command (error));
}

/*
 * Callbacks
 */
static void
detail_button_clicked_cb (GtkButton *button, gpointer user_data)
{
	GtkLabel *label;
	GnomeDbError *error_widget = (GnomeDbError *) user_data;

	g_return_if_fail (GNOME_DB_IS_ERROR (error_widget));

	label = GTK_LABEL (GTK_BIN (error_widget->priv->detail_container)->child);
	if (GTK_WIDGET_VISIBLE (error_widget->priv->detail_container)) {
		gtk_widget_hide (error_widget->priv->detail_container);
		gtk_label_set_text (label, _("Show detail >>"));
	}
	else {
		gtk_widget_show (error_widget->priv->detail_container);
		gtk_label_set_text (label, _("<< Hide detail"));
	}
}

/*
 * GnomeDbError class implementation
 */
static void
gnome_db_error_real_prev (GnomeDbError *error_widget)
{
	g_return_if_fail (GNOME_DB_IS_ERROR (error_widget));

	if (error_widget->priv->current_pos > 0) {
		error_widget->priv->current_pos--;
		display_current_error (error_widget);
	}
}

void
gnome_db_error_real_next (GnomeDbError *error_widget)
{
	g_return_if_fail (GNOME_DB_IS_ERROR (error_widget));

	if (error_widget->priv->current_pos <
	    g_list_length (error_widget->priv->current_list) - 1) {
		error_widget->priv->current_pos++;
		display_current_error (error_widget);
	}
}

guint
gnome_db_error_get_type(void)
{
	static guint type = 0;

	if (!type) {
		GtkTypeInfo info = {
			"GnomeDbError",
			sizeof (GnomeDbError),
			sizeof (GnomeDbErrorClass),
			(GtkClassInitFunc) gnome_db_error_class_init,
			(GtkObjectInitFunc) gnome_db_error_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL,
		};
		type = gtk_type_unique(gtk_vbox_get_type(), &info);
	}
	return type;
}

static void
gnome_db_error_class_init(GnomeDbErrorClass* klass)
{
	GtkObjectClass* object_class;

	object_class = (GtkObjectClass*) klass;

	gnome_db_error_signals[GNOME_DB_PREV] =
		gtk_signal_new("previous",
			       GTK_RUN_LAST,
			       object_class->type,
			       GTK_SIGNAL_OFFSET(GnomeDbErrorClass, prev),
			       gtk_signal_default_marshaller,
			       GTK_TYPE_INT, 0);
	gnome_db_error_signals[GNOME_DB_NEXT] =
		gtk_signal_new("next",
			       GTK_RUN_LAST,
			       object_class->type,
			       GTK_SIGNAL_OFFSET(GnomeDbErrorClass, next),
			       gtk_signal_default_marshaller,
			       GTK_TYPE_INT, 0);
		      
	gtk_object_class_add_signals(object_class, gnome_db_error_signals, LAST_SIGNAL);

	object_class->destroy = gnome_db_error_destroy;
	klass->next = gnome_db_error_real_next;
	klass->prev = gnome_db_error_real_prev;
}

static void
gnome_db_error_init (GnomeDbError* error_widget)
{
	GtkWidget *table;
	GtkWidget *detail_table;
	GtkWidget *label;
	GtkWidget *scroll;

	error_widget->priv = g_new (GnomeDbErrorPrivate, 1);
	error_widget->priv->current_list = NULL;
	error_widget->priv->current_pos = -1;

	/* create main container */
	table = gnome_db_new_table_widget (3, 3, FALSE);
	gtk_box_pack_start (GTK_BOX (error_widget), table, TRUE, TRUE, 0);

	label = gnome_db_new_label_widget (_("Error number"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	error_widget->priv->error_number = gnome_db_new_entry_widget (0, FALSE);
	gtk_table_attach (GTK_TABLE (table), error_widget->priv->error_number,
			  1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	error_widget->priv->detail_button = gnome_db_new_button_widget (_("Show detail >>"));
	gtk_signal_connect (GTK_OBJECT (error_widget->priv->detail_button),
			    "clicked",
			    GTK_SIGNAL_FUNC (detail_button_clicked_cb),
			    error_widget);
	gtk_table_attach (GTK_TABLE (table), error_widget->priv->detail_button,
			  2, 3, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	label = gnome_db_new_label_widget (_("Description"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
	scroll = gnome_db_new_scrolled_window_widget ();
	gtk_table_attach (GTK_TABLE (table), scroll, 1, 2, 1, 2,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK, 3, 3);
	error_widget->priv->error_description = gnome_db_new_text_widget ();
	gtk_container_add (GTK_CONTAINER (scroll), error_widget->priv->error_description);

	/* create the detail container */
	error_widget->priv->detail_container = gnome_db_new_frame_widget (_("Detail"));
	gtk_table_attach (GTK_TABLE (table), error_widget->priv->detail_container,
			  0, 3, 2, 3, GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND, 3, 3);
	detail_table = gnome_db_new_table_widget (2, 5, FALSE);
	gtk_container_add (GTK_CONTAINER (error_widget->priv->detail_container), detail_table);

	label = gnome_db_new_label_widget (_("Source"));
	gtk_table_attach (GTK_TABLE (detail_table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	error_widget->priv->error_source = gnome_db_new_entry_widget (0, FALSE);
	gtk_table_attach (GTK_TABLE (detail_table), error_widget->priv->error_source,
			  1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	label = gnome_db_new_label_widget (_("Help"));
	gtk_table_attach (GTK_TABLE (detail_table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
	error_widget->priv->error_help_url = gnome_db_new_entry_widget (0, FALSE);
	gtk_table_attach (GTK_TABLE (detail_table), error_widget->priv->error_help_url,
			  1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

	label = gnome_db_new_label_widget (_("SQL state"));
	gtk_table_attach (GTK_TABLE (detail_table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
	error_widget->priv->error_sqlstate = gnome_db_new_entry_widget (0, FALSE);
	gtk_table_attach (GTK_TABLE (detail_table), error_widget->priv->error_sqlstate,
			  1, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);

	label = gnome_db_new_label_widget (_("Native message"));
	gtk_table_attach (GTK_TABLE (detail_table), label, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 3, 3);
	error_widget->priv->error_native = gnome_db_new_entry_widget (0, FALSE);
	gtk_table_attach (GTK_TABLE (detail_table), error_widget->priv->error_native,
			  1, 2, 3, 4, GTK_FILL, GTK_FILL, 3, 3);

	label = gnome_db_new_label_widget (_("Command"));
	gtk_table_attach (GTK_TABLE (detail_table), label, 0, 1, 4, 5, GTK_FILL, GTK_FILL, 3, 3);
	error_widget->priv->error_command = gnome_db_new_entry_widget (0, FALSE);
	gtk_table_attach (GTK_TABLE (detail_table), error_widget->priv->error_command,
			  1, 2, 4, 5, GTK_FILL, GTK_FILL, 3, 3);

	gtk_widget_hide (error_widget->priv->detail_container);
}

static void
gnome_db_error_destroy (GtkObject *object)
{
	GtkObjectClass *parent_class;
	GnomeDbError *error_widget = (GnomeDbError *) object;

	g_return_if_fail (GNOME_DB_IS_ERROR (error_widget));

	/* free memory */
	gda_error_list_free (error_widget->priv->current_list);
	g_free (error_widget->priv);
	error_widget->priv = NULL;

	parent_class = gtk_type_class (gtk_vbox_get_type ());
	if (parent_class && parent_class->destroy)
		parent_class->destroy (object);
}

/**
 * gnome_db_error_new
 *
 * Create a new GnomeDbError widget, which is a special widget used to
 * traverse errors occurred for a given connection. It is a simple container
 * widget that you can pack into any widget you want.
 *
 * Returns: a pointer to the new widget, or NULL on error
 */
GtkWidget*
gnome_db_error_new (void)
{
	GnomeDbError* error_widget;
  
	error_widget = GNOME_DB_ERROR (gtk_type_new (gnome_db_error_get_type()));
	return GTK_WIDGET (error_widget);
}

/**
 * gnome_db_error_clear
 */
void
gnome_db_error_clear (GnomeDbError *error_widget)
{
	g_return_if_fail (GNOME_DB_IS_ERROR (error_widget));

	gtk_entry_set_text (GTK_ENTRY (error_widget->priv->error_number), "");
	gtk_editable_delete_text (GTK_EDITABLE (error_widget->priv->error_description), 0, -1);
	gtk_entry_set_text (GTK_ENTRY (error_widget->priv->error_source), "");
	gtk_entry_set_text (GTK_ENTRY (error_widget->priv->error_help_url), "");
	gtk_entry_set_text (GTK_ENTRY (error_widget->priv->error_sqlstate), "");
	gtk_entry_set_text (GTK_ENTRY (error_widget->priv->error_native), "");
	gtk_entry_set_text (GTK_ENTRY (error_widget->priv->error_command), "");

	gda_error_list_free (error_widget->priv->current_list);
	error_widget->priv->current_list = NULL;
}

/**
 * gnome_db_error_show
 * @error_widget: the GnomeDbError widget
 *
 * Update the given #GnomeDbError widget to show the latest errors
 * from the connection associated with it.
 */
void
gnome_db_error_show (GnomeDbError *error_widget, GList *errors)
{
	GList *l;
	GList *new_list = NULL;

	g_return_if_fail (GNOME_DB_IS_ERROR (error_widget));

	gnome_db_error_clear (error_widget);
	for (l = errors; l != NULL; l = g_list_next (l)) {
		GdaError *error = gda_error_new ();
		GdaError *tmp_error = GDA_ERROR (l->data);

		gda_error_set_number (error, gda_error_get_number (tmp_error));
		gda_error_set_description (error, gda_error_get_description (tmp_error));
		gda_error_set_source (error, gda_error_get_source (tmp_error));
		gda_error_set_sqlstate (error, gda_error_get_sqlstate (tmp_error));
		gda_error_set_help_url (error, gda_error_get_help_url (tmp_error));
		gda_error_set_native (error, gda_error_get_native (tmp_error));
		gda_error_set_real_command (error, gda_error_get_real_command (tmp_error));

		new_list = g_list_append (new_list, error);
	}

	error_widget->priv->current_list = new_list;
	error_widget->priv->current_pos = 0;
	display_current_error (error_widget);
}

/**
 * gnome_db_error_prev
 * @error_widget: the GnomeDbError widget
 *
 * Display the previous error in the given GnomeDbError widget
 */
void
gnome_db_error_prev(GnomeDbError* error_widget)
{
	gtk_signal_emit (GTK_OBJECT (error_widget),
			 gnome_db_error_signals[GNOME_DB_PREV]);
}

/**
 * gnome_db_error_next
 * @error_widget: the GnomeDbError widget
 *
 * Display the next error in the given GnomeDbError widget
 */
void
gnome_db_error_next(GnomeDbError* error_widget)
{
	gtk_signal_emit (GTK_OBJECT (error_widget),
			 gnome_db_error_signals[GNOME_DB_NEXT]);
}
