/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <src/Scanner.h>
#include <gee/arraylist.h>
#include <gio/gio.h>
#include <gee/collection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/map.h>
#include <gee/set.h>
#include "src/ChannelListWriter.h"




static char* string_substring (const char* self, glong offset, glong len);
static char* bool_to_string (gboolean self);
static char* string_replace (const char* self, const char* old, const char* replacement);
struct _DVBScannerPrivate {
	DVBDevice* _Device;
	GeeArrayList* new_channels;
	guint check_for_lock_event_id;
	guint wait_for_tables_event_id;
	gboolean nit_arrived;
	gboolean sdt_arrived;
	gboolean pat_arrived;
	gboolean locked;
};

#define DVB_SCANNER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_SCANNER, DVBScannerPrivate))
enum  {
	DVB_SCANNER_DUMMY_PROPERTY,
	DVB_SCANNER_DEVICE,
	DVB_SCANNER_CHANNELS
};
#define DVB_SCANNER_BASE_PIDS "16:17:18"
#define DVB_SCANNER_PIPELINE_TEMPLATE "dvbsrc name=dvbsrc adapter=%u frontend=%u pids=%s stats-reporting-interval=0 ! mpegtsparse ! fakesink silent=true"
static void dvb_scanner_real_prepare (DVBScanner* self);
static DVBScannedItem* dvb_scanner_real_get_scanned_item (DVBScanner* self, const GstStructure* structure);
static DVBChannel* dvb_scanner_real_get_new_channel (DVBScanner* self);
static void dvb_scanner_real_add_values_from_structure_to_channel (DVBScanner* self, const GstStructure* delivery, DVBChannel* channel);
static void _dvb_scanner_bus_watch_func_gst_bus_message (GstBus* _sender, GstMessage* message, gpointer self);
static gboolean _dvb_scanner_check_for_lock_gsource_func (gpointer self);
static gboolean _dvb_scanner_wait_for_tables_gsource_func (gpointer self);
static void dvb_scanner_set_Device (DVBScanner* self, DVBDevice* value);
static guint _dvb_scanned_item_hash_ghash_func (void* key);
static gboolean _dvb_scanned_item_equal_gequal_func (void* a, void* b);
static GObject * dvb_scanner_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer dvb_scanner_parent_class = NULL;
static void dvb_scanner_finalize (GObject* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);


static void g_cclosure_user_marshal_VOID__UINT_UINT (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);
static void g_cclosure_user_marshal_VOID__UINT_UINT_STRING_STRING_STRING_BOOLEAN (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);

static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}


static char* bool_to_string (gboolean self) {
	if (self) {
		return g_strdup ("true");
	} else {
		return g_strdup ("false");
	}
}


static char* string_replace (const char* self, const char* old, const char* replacement) {
	GError * inner_error;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (old != NULL, NULL);
	g_return_val_if_fail (replacement != NULL, NULL);
	inner_error = NULL;
	{
		char* _tmp0;
		GRegex* _tmp1;
		GRegex* regex;
		char* _tmp2;
		char* _tmp3;
		_tmp0 = NULL;
		_tmp1 = NULL;
		regex = (_tmp1 = g_regex_new (_tmp0 = g_regex_escape_string (old, -1), 0, 0, &inner_error), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
		if (inner_error != NULL) {
			if (inner_error->domain == G_REGEX_ERROR) {
				goto __catch22_g_regex_error;
			}
			goto __finally22;
		}
		_tmp2 = g_regex_replace_literal (regex, self, (glong) (-1), 0, replacement, 0, &inner_error);
		if (inner_error != NULL) {
			(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
			if (inner_error->domain == G_REGEX_ERROR) {
				goto __catch22_g_regex_error;
			}
			goto __finally22;
		}
		_tmp3 = NULL;
		return (_tmp3 = _tmp2, (regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL)), _tmp3);
	}
	goto __finally22;
	__catch22_g_regex_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			g_assert_not_reached ();
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally22:
	if (inner_error != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return NULL;
	}
}


static void dvb_scanner_real_prepare (DVBScanner* self) {
	g_return_if_fail (self != NULL);
	g_critical ("Type `%s' does not implement abstract method `dvb_scanner_prepare'", g_type_name (G_TYPE_FROM_INSTANCE (self)));
	return;
}


/**
         * Setup the pipeline correctly
         */
void dvb_scanner_prepare (DVBScanner* self) {
	DVB_SCANNER_GET_CLASS (self)->prepare (self);
}


static DVBScannedItem* dvb_scanner_real_get_scanned_item (DVBScanner* self, const GstStructure* structure) {
	g_return_val_if_fail (self != NULL, NULL);
	g_critical ("Type `%s' does not implement abstract method `dvb_scanner_get_scanned_item'", g_type_name (G_TYPE_FROM_INSTANCE (self)));
	return NULL;
}


/**
         * Use the frequency and possibly other data to
         * mark the tuning paramters as already used
         */
DVBScannedItem* dvb_scanner_get_scanned_item (DVBScanner* self, const GstStructure* structure) {
	return DVB_SCANNER_GET_CLASS (self)->get_scanned_item (self, structure);
}


static DVBChannel* dvb_scanner_real_get_new_channel (DVBScanner* self) {
	g_return_val_if_fail (self != NULL, NULL);
	g_critical ("Type `%s' does not implement abstract method `dvb_scanner_get_new_channel'", g_type_name (G_TYPE_FROM_INSTANCE (self)));
	return NULL;
}


/**
         * Return a new empty channel
         */
DVBChannel* dvb_scanner_get_new_channel (DVBScanner* self) {
	return DVB_SCANNER_GET_CLASS (self)->get_new_channel (self);
}


static void dvb_scanner_real_add_values_from_structure_to_channel (DVBScanner* self, const GstStructure* delivery, DVBChannel* channel) {
	g_return_if_fail (self != NULL);
	g_critical ("Type `%s' does not implement abstract method `dvb_scanner_add_values_from_structure_to_channel'", g_type_name (G_TYPE_FROM_INSTANCE (self)));
	return;
}


/**
         * Retrieve the data from structure and add it to the Channel
         */
void dvb_scanner_add_values_from_structure_to_channel (DVBScanner* self, const GstStructure* delivery, DVBChannel* channel) {
	DVB_SCANNER_GET_CLASS (self)->add_values_from_structure_to_channel (self, delivery, channel);
}


static void _dvb_scanner_bus_watch_func_gst_bus_message (GstBus* _sender, GstMessage* message, gpointer self) {
	dvb_scanner_bus_watch_func (self, _sender, message);
}


/**
         * Start the scanner
         */
void dvb_scanner_Run (DVBScanner* self) {
	GError * inner_error;
	DVBChannelList* _tmp0;
	GstBus* bus;
	g_return_if_fail (self != NULL);
	inner_error = NULL;
	_tmp0 = NULL;
	self->channels = (_tmp0 = dvb_channel_list_new (NULL), (self->channels == NULL) ? NULL : (self->channels = (g_object_unref (self->channels), NULL)), _tmp0);
	{
		char* _tmp1;
		GstElement* _tmp2;
		GstElement* _tmp3;
		GstElement* _tmp4;
		_tmp1 = NULL;
		_tmp2 = NULL;
		_tmp3 = (_tmp2 = gst_parse_launch (_tmp1 = g_strdup_printf (DVB_SCANNER_PIPELINE_TEMPLATE, dvb_device_get_Adapter (self->priv->_Device), dvb_device_get_Frontend (self->priv->_Device), DVB_SCANNER_BASE_PIDS), &inner_error), _tmp1 = (g_free (_tmp1), NULL), _tmp2);
		if (inner_error != NULL) {
			goto __catch23_g_error;
			goto __finally23;
		}
		_tmp4 = NULL;
		self->pipeline = (_tmp4 = _tmp3, (self->pipeline == NULL) ? NULL : (self->pipeline = (gst_object_unref (self->pipeline), NULL)), _tmp4);
	}
	goto __finally23;
	__catch23_g_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			g_error ("Scanner.vala:146: Could not create pipeline: %s", e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			return;
		}
	}
	__finally23:
	if (inner_error != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	bus = gst_element_get_bus (self->pipeline);
	gst_bus_add_signal_watch (bus);
	g_signal_connect_object (bus, "message", (GCallback) _dvb_scanner_bus_watch_func_gst_bus_message, self, 0);
	gst_element_set_state (self->pipeline, GST_STATE_READY);
	dvb_scanner_start_scan (self);
	(bus == NULL) ? NULL : (bus = (gst_object_unref (bus), NULL));
}


/**
         * Abort scanning and cleanup
         */
void dvb_scanner_Destroy (DVBScanner* self) {
	DVBChannelList* _tmp0;
	g_return_if_fail (self != NULL);
	dvb_scanner_remove_check_for_lock_timeout (self);
	dvb_scanner_remove_wait_for_tables_timeout (self);
	dvb_scanner_clear_and_reset_all (self);
	dvb_channel_list_clear (self->channels);
	_tmp0 = NULL;
	self->channels = (_tmp0 = NULL, (self->channels == NULL) ? NULL : (self->channels = (g_object_unref (self->channels), NULL)), _tmp0);
	g_signal_emit_by_name (self, "destroyed");
}


/** 
         * @path: Location where the file will be stored
         *
         * Write all the channels stored in this.Channels to file
         */
gboolean dvb_scanner_WriteChannelsToFile (DVBScanner* self, const char* path) {
	GError * inner_error;
	gboolean ret;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (path != NULL, FALSE);
	inner_error = NULL;
	ret = FALSE;
	{
		GFile* _tmp0;
		DVBChannelListWriter* _tmp1;
		DVBChannelListWriter* writer;
		_tmp0 = NULL;
		_tmp1 = NULL;
		writer = (_tmp1 = dvb_channel_list_writer_new (_tmp0 = g_file_new_for_path (path), &inner_error), (_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)), _tmp1);
		if (inner_error != NULL) {
			if (inner_error->domain == G_IO_ERROR) {
				goto __catch24_g_io_error;
			}
			goto __finally24;
		}
		{
			GeeIterator* _c_it;
			_c_it = gee_iterable_iterator ((GeeIterable*) self->channels);
			while (gee_iterator_next (_c_it)) {
				DVBChannel* c;
				c = (DVBChannel*) gee_iterator_get (_c_it);
				dvb_channel_list_writer_write (writer, c, &inner_error);
				if (inner_error != NULL) {
					(c == NULL) ? NULL : (c = (g_object_unref (c), NULL));
					(_c_it == NULL) ? NULL : (_c_it = (g_object_unref (_c_it), NULL));
					(writer == NULL) ? NULL : (writer = (g_object_unref (writer), NULL));
					if (inner_error->domain == G_IO_ERROR) {
						goto __catch24_g_io_error;
					}
					goto __finally24;
				}
				(c == NULL) ? NULL : (c = (g_object_unref (c), NULL));
			}
			(_c_it == NULL) ? NULL : (_c_it = (g_object_unref (_c_it), NULL));
		}
		dvb_channel_list_writer_close (writer, &inner_error);
		if (inner_error != NULL) {
			(writer == NULL) ? NULL : (writer = (g_object_unref (writer), NULL));
			if (inner_error->domain == G_IO_ERROR) {
				goto __catch24_g_io_error;
			}
			goto __finally24;
		}
		ret = TRUE;
		(writer == NULL) ? NULL : (writer = (g_object_unref (writer), NULL));
	}
	goto __finally24;
	__catch24_g_io_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			g_critical ("Scanner.vala:186: Could not write channels file: %s", e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally24:
	if (inner_error != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return FALSE;
	}
	return ret;
}


void dvb_scanner_clear_and_reset_all (DVBScanner* self) {
	GstStructure* _tmp1;
	g_return_if_fail (self != NULL);
	if (self->pipeline != NULL) {
		GstBus* bus;
		GstElement* _tmp0;
		bus = gst_element_get_bus (self->pipeline);
		gst_bus_remove_signal_watch (bus);
		gst_element_set_state (self->pipeline, GST_STATE_NULL);
		/* Free pipeline*/
		_tmp0 = NULL;
		self->pipeline = (_tmp0 = NULL, (self->pipeline == NULL) ? NULL : (self->pipeline = (gst_object_unref (self->pipeline), NULL)), _tmp0);
		(bus == NULL) ? NULL : (bus = (gst_object_unref (bus), NULL));
	}
	gee_map_clear ((GeeMap*) self->transport_streams);
	gee_collection_clear ((GeeCollection*) self->scanned_frequencies);
	dvb_scanner_clear_frequencies (self);
	_tmp1 = NULL;
	self->current_tuning_params = (_tmp1 = NULL, (self->current_tuning_params == NULL) ? NULL : (self->current_tuning_params = (gst_structure_free (self->current_tuning_params), NULL)), _tmp1);
	gee_collection_clear ((GeeCollection*) self->priv->new_channels);
}


void dvb_scanner_clear_frequencies (DVBScanner* self) {
	g_return_if_fail (self != NULL);
	while (!g_queue_is_empty (self->frequencies)) {
		GstStructure* s;
		GstStructure* _tmp0;
		s = (GstStructure*) g_queue_pop_head (self->frequencies);
		/* Force that gst_structure_free is called*/
		_tmp0 = NULL;
		s = (_tmp0 = NULL, (s == NULL) ? NULL : (s = (gst_structure_free (s), NULL)), _tmp0);
		(s == NULL) ? NULL : (s = (gst_structure_free (s), NULL));
	}
	g_queue_clear (self->frequencies);
}


void dvb_scanner_add_structure_to_scan (DVBScanner* self, GstStructure* structure) {
	DVBScannedItem* item;
	g_return_if_fail (self != NULL);
	g_return_if_fail (structure != NULL);
	if (structure == NULL) {
		(structure == NULL) ? NULL : (structure = (gst_structure_free (structure), NULL));
		return;
	}
	item = dvb_scanner_get_scanned_item (self, structure);
	if (!gee_collection_contains ((GeeCollection*) self->scanned_frequencies, item)) {
		const GstStructure* _tmp0;
		g_debug ("Scanner.vala:223: Queueing new frequency %u", dvb_scanned_item_get_Frequency (item));
		_tmp0 = NULL;
		g_queue_push_tail (self->frequencies, (_tmp0 = structure, (_tmp0 == NULL) ? NULL : gst_structure_copy (_tmp0)));
		gee_collection_add ((GeeCollection*) self->scanned_frequencies, item);
	}
	(item == NULL) ? NULL : (item = (g_object_unref (item), NULL));
	(structure == NULL) ? NULL : (structure = (gst_structure_free (structure), NULL));
}


static gboolean _dvb_scanner_check_for_lock_gsource_func (gpointer self) {
	return dvb_scanner_check_for_lock (self);
}


/**
         * Pick up the next tuning paramters from the queue
         * and start scanning with them
         */
void dvb_scanner_start_scan (DVBScanner* self) {
	gboolean _tmp0;
	gboolean _tmp1;
	gboolean all_tables;
	char* _tmp5;
	char* _tmp4;
	char* _tmp3;
	char* _tmp2;
	GstStructure* _tmp6;
	guint freq;
	GstElement* dvbsrc;
	g_return_if_fail (self != NULL);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (self->priv->sdt_arrived) {
		_tmp1 = self->priv->nit_arrived;
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		_tmp0 = self->priv->pat_arrived;
	} else {
		_tmp0 = FALSE;
	}
	all_tables = _tmp0;
	_tmp5 = NULL;
	_tmp4 = NULL;
	_tmp3 = NULL;
	_tmp2 = NULL;
	g_debug ("Scanner.vala:235: Received all tables: %s (pat: %s, sdt: %s, nit: %s)", _tmp2 = bool_to_string (all_tables), _tmp3 = bool_to_string (self->priv->pat_arrived), _tmp4 = bool_to_string (self->priv->sdt_arrived), _tmp5 = bool_to_string (self->priv->nit_arrived));
	_tmp5 = (g_free (_tmp5), NULL);
	_tmp4 = (g_free (_tmp4), NULL);
	_tmp3 = (g_free (_tmp3), NULL);
	_tmp2 = (g_free (_tmp2), NULL);
	self->priv->nit_arrived = FALSE;
	self->priv->sdt_arrived = FALSE;
	self->priv->pat_arrived = FALSE;
	self->priv->locked = FALSE;
	if (self->current_tuning_params != NULL) {
		guint old_freq;
		old_freq = 0U;
		gst_structure_get_uint (self->current_tuning_params, "frequency", &old_freq);
		g_signal_emit_by_name (self, "frequency-scanned", old_freq, self->frequencies->length);
	}
	if (g_queue_is_empty (self->frequencies)) {
		g_message ("Scanner.vala:251: Finished scanning");
		/* We don't have all the information for those channels
		 remove them*/
		g_debug ("Scanner.vala:254: %u channels still have missing or invalid information", (guint) gee_collection_get_size ((GeeCollection*) self->priv->new_channels));
		{
			GeeIterator* _sid_it;
			_sid_it = gee_iterable_iterator ((GeeIterable*) self->priv->new_channels);
			while (gee_iterator_next (_sid_it)) {
				guint sid;
				sid = GPOINTER_TO_UINT (gee_iterator_get (_sid_it));
				dvb_channel_list_remove (self->channels, sid);
			}
			(_sid_it == NULL) ? NULL : (_sid_it = (g_object_unref (_sid_it), NULL));
		}
		dvb_scanner_clear_and_reset_all (self);
		g_signal_emit_by_name (self, "finished");
		return;
	}
	_tmp6 = NULL;
	self->current_tuning_params = (_tmp6 = (GstStructure*) g_queue_pop_head (self->frequencies), (self->current_tuning_params == NULL) ? NULL : (self->current_tuning_params = (gst_structure_free (self->current_tuning_params), NULL)), _tmp6);
	/* Remember that we already scanned this frequency*/
	freq = 0U;
	gst_structure_get_uint (self->current_tuning_params, "frequency", &freq);
	g_debug ("Scanner.vala:270: Starting scanning frequency %u (%u left)", freq, g_queue_get_length (self->frequencies));
	gst_element_set_state (self->pipeline, GST_STATE_READY);
	dvb_scanner_prepare (self);
	/* Reset PIDs*/
	dvbsrc = gst_bin_get_by_name (GST_BIN (self->pipeline), "dvbsrc");
	g_object_set ((GObject*) dvbsrc, "pids", DVB_SCANNER_BASE_PIDS, NULL);
	gst_element_set_state (self->pipeline, GST_STATE_PLAYING);
	self->priv->check_for_lock_event_id = g_timeout_add_seconds ((guint) 5, _dvb_scanner_check_for_lock_gsource_func, self);
	(dvbsrc == NULL) ? NULL : (dvbsrc = (gst_object_unref (dvbsrc), NULL));
}


/**
         * Check if we received a lock with the currently
         * used tuning parameters
         */
gboolean dvb_scanner_check_for_lock (DVBScanner* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	self->priv->check_for_lock_event_id = (guint) 0;
	if (!self->priv->locked) {
		gst_element_set_state (self->pipeline, GST_STATE_READY);
		dvb_scanner_start_scan (self);
	}
	return FALSE;
}


gboolean dvb_scanner_wait_for_tables (DVBScanner* self) {
	gboolean _tmp0;
	gboolean _tmp1;
	g_return_val_if_fail (self != NULL, FALSE);
	self->priv->wait_for_tables_event_id = (guint) 0;
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (self->priv->sdt_arrived) {
		_tmp1 = self->priv->nit_arrived;
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		_tmp0 = self->priv->pat_arrived;
	} else {
		_tmp0 = FALSE;
	}
	if (!_tmp0) {
		gst_element_set_state (self->pipeline, GST_STATE_READY);
		dvb_scanner_start_scan (self);
	}
	return FALSE;
}


void dvb_scanner_remove_check_for_lock_timeout (DVBScanner* self) {
	g_return_if_fail (self != NULL);
	if (self->priv->check_for_lock_event_id != 0) {
		g_source_remove (self->priv->check_for_lock_event_id);
		self->priv->check_for_lock_event_id = (guint) 0;
	}
}


void dvb_scanner_remove_wait_for_tables_timeout (DVBScanner* self) {
	g_return_if_fail (self != NULL);
	if (self->priv->wait_for_tables_event_id != 0) {
		g_source_remove (self->priv->wait_for_tables_event_id);
		self->priv->wait_for_tables_event_id = (guint) 0;
	}
}


void dvb_scanner_set_uint_property (GstElement* src, const GstStructure* params, const char* key) {
	guint val;
	g_return_if_fail (src != NULL);
	g_return_if_fail (params != NULL);
	g_return_if_fail (key != NULL);
	val = 0U;
	gst_structure_get_uint (params, key, &val);
	g_object_set ((GObject*) src, key, val, NULL);
}


static gboolean _dvb_scanner_wait_for_tables_gsource_func (gpointer self) {
	return dvb_scanner_wait_for_tables (self);
}


void dvb_scanner_on_dvb_frontend_stats_structure (DVBScanner* self, const GstStructure* structure) {
	gboolean has_lock;
	gboolean _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (structure != NULL);
	has_lock = FALSE;
	gst_structure_get_boolean (structure, "lock", &has_lock);
	_tmp0 = FALSE;
	if (has_lock) {
		_tmp0 = !self->priv->locked;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		g_debug ("Scanner.vala:334: Got lock");
		dvb_scanner_remove_check_for_lock_timeout (self);
		self->priv->wait_for_tables_event_id = g_timeout_add_seconds ((guint) 10, _dvb_scanner_wait_for_tables_gsource_func, self);
	}
}


void dvb_scanner_on_dvb_read_failure_structure (DVBScanner* self) {
	g_return_if_fail (self != NULL);
	g_warning ("Scanner.vala:342: Read failure");
}


/*
            this.Destroy ();
            */
void dvb_scanner_on_pat_structure (DVBScanner* self, const GstStructure* structure) {
	GeeSet* pid_set;
	GValue programs;
	guint size;
	GValue val = {0};
	const GstStructure* program;
	GString* new_pids;
	char* pids;
	GstElement* dvbsrc;
	g_return_if_fail (self != NULL);
	g_return_if_fail (structure != NULL);
	g_debug ("Scanner.vala:349: Received PAT");
	pid_set = (GeeSet*) gee_hash_set_new (G_TYPE_UINT, NULL, NULL, g_direct_hash, g_direct_equal);
	/* add BASE_PIDS*/
	gee_collection_add ((GeeCollection*) pid_set, GUINT_TO_POINTER ((guint) 16));
	gee_collection_add ((GeeCollection*) pid_set, GUINT_TO_POINTER ((guint) 17));
	gee_collection_add ((GeeCollection*) pid_set, GUINT_TO_POINTER ((guint) 18));
	programs = *gst_structure_get_value (structure, "programs");
	size = gst_value_list_get_size (&programs);
	program = NULL;
	{
		guint i;
		/* Iterate over programs*/
		i = (guint) 0;
		for (; i < size; i++) {
			guint sid;
			guint pmt;
			val = *gst_value_list_get_value (&programs, i);
			program = gst_value_get_structure (&val);
			sid = 0U;
			gst_structure_get_uint (program, "program-number", &sid);
			pmt = 0U;
			gst_structure_get_uint (program, "pid", &pmt);
			gee_collection_add ((GeeCollection*) pid_set, GUINT_TO_POINTER (pmt));
		}
	}
	new_pids = g_string_new ("");
	{
		GeeIterator* _pid_it;
		_pid_it = gee_iterable_iterator ((GeeIterable*) pid_set);
		while (gee_iterator_next (_pid_it)) {
			guint pid;
			char* _tmp0;
			pid = GPOINTER_TO_UINT (gee_iterator_get (_pid_it));
			_tmp0 = NULL;
			g_string_append (new_pids, _tmp0 = g_strdup_printf ("%u:", pid));
			_tmp0 = (g_free (_tmp0), NULL);
		}
		(_pid_it == NULL) ? NULL : (_pid_it = (g_object_unref (_pid_it), NULL));
	}
	pids = string_substring (new_pids->str, (glong) 0, new_pids->len - 1);
	g_debug ("Scanner.vala:381: Setting %d pids: %s", gee_collection_get_size ((GeeCollection*) pid_set), pids);
	/* We want to parse the pmt as well*/
	dvbsrc = gst_bin_get_by_name (GST_BIN (self->pipeline), "dvbsrc");
	g_object_set ((GObject*) dvbsrc, "pids", pids, NULL);
	self->priv->pat_arrived = TRUE;
	(pid_set == NULL) ? NULL : (pid_set = (g_object_unref (pid_set), NULL));
	(new_pids == NULL) ? NULL : (new_pids = (g_string_free (new_pids, TRUE), NULL));
	pids = (g_free (pids), NULL);
	(dvbsrc == NULL) ? NULL : (dvbsrc = (gst_object_unref (dvbsrc), NULL));
}


void dvb_scanner_on_sdt_structure (DVBScanner* self, const GstStructure* structure) {
	guint tsid;
	GValue services;
	guint size;
	GValue val = {0};
	const GstStructure* service;
	g_return_if_fail (self != NULL);
	g_return_if_fail (structure != NULL);
	g_debug ("Scanner.vala:390: Received SDT");
	tsid = 0U;
	gst_structure_get_uint (structure, "transport-stream-id", &tsid);
	services = *gst_structure_get_value (structure, "services");
	size = gst_value_list_get_size (&services);
	service = NULL;
	{
		guint i;
		/* Iterate over services*/
		i = (guint) 0;
		for (; i < size; i++) {
			const char* _tmp0;
			char* name;
			char* _tmp1;
			gint _tmp2;
			gint sid;
			DVBChannel* channel;
			const char* _tmp6;
			char* provider;
			gboolean _tmp7;
			val = *gst_value_list_get_value (&services, i);
			service = gst_value_get_structure (&val);
			/* Returns "service-%d"*/
			_tmp0 = NULL;
			name = (_tmp0 = gst_structure_get_name (service), (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
			/* Get the number at the end*/
			_tmp1 = NULL;
			sid = (_tmp2 = atoi (_tmp1 = string_substring (name, (glong) 8, strlen (name) - 8)), _tmp1 = (g_free (_tmp1), NULL), _tmp2);
			if (gst_structure_has_field (service, "name")) {
				char* _tmp4;
				const char* _tmp3;
				_tmp4 = NULL;
				_tmp3 = NULL;
				name = (_tmp4 = (_tmp3 = gst_structure_get_string (service, "name"), (_tmp3 == NULL) ? NULL : g_strdup (_tmp3)), name = (g_free (name), NULL), _tmp4);
			}
			if (!dvb_channel_list_contains (self->channels, (guint) sid)) {
				dvb_scanner_add_new_channel (self, (guint) sid);
			}
			channel = dvb_channel_list_get_channel (self->channels, (guint) sid);
			if (gst_structure_has_field (service, "scrambled")) {
				gboolean scrambled;
				scrambled = FALSE;
				gst_structure_get_boolean (service, "scrambled", &scrambled);
				dvb_channel_set_Scrambled (channel, scrambled);
			} else {
				dvb_channel_set_Scrambled (channel, FALSE);
			}
			if (g_utf8_validate (name, -1, NULL)) {
				char* _tmp5;
				_tmp5 = NULL;
				dvb_channel_set_Name (channel, _tmp5 = string_replace (name, "\\s", " "));
				_tmp5 = (g_free (_tmp5), NULL);
			}
			dvb_channel_set_TransportStreamId (channel, tsid);
			_tmp6 = NULL;
			provider = (_tmp6 = gst_structure_get_string (service, "provider-name"), (_tmp6 == NULL) ? NULL : g_strdup (_tmp6));
			_tmp7 = FALSE;
			if (provider != NULL) {
				_tmp7 = g_utf8_validate (provider, -1, NULL);
			} else {
				_tmp7 = FALSE;
			}
			if (_tmp7) {
				dvb_channel_set_Network (channel, provider);
			} else {
				dvb_channel_set_Network (channel, "");
			}
			name = (g_free (name), NULL);
			(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
			provider = (g_free (provider), NULL);
		}
	}
	self->priv->sdt_arrived = TRUE;
}


void dvb_scanner_on_nit_structure (DVBScanner* self, const GstStructure* structure) {
	char* name;
	GValue transports;
	guint size;
	GValue val = {0};
	const GstStructure* transport;
	g_return_if_fail (self != NULL);
	g_return_if_fail (structure != NULL);
	g_debug ("Scanner.vala:444: Received NIT");
	name = NULL;
	if (gst_structure_has_field (structure, "network-name")) {
		char* _tmp1;
		const char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		name = (_tmp1 = (_tmp0 = gst_structure_get_string (structure, "network-name"), (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), name = (g_free (name), NULL), _tmp1);
	} else {
		guint nid;
		char* _tmp2;
		nid = 0U;
		gst_structure_get_uint (structure, "network-id", &nid);
		_tmp2 = NULL;
		name = (_tmp2 = g_strdup_printf ("%u", nid), name = (g_free (name), NULL), _tmp2);
	}
	g_debug ("Scanner.vala:454: Network name '%s'", name);
	transports = *gst_structure_get_value (structure, "transports");
	size = gst_value_list_get_size (&transports);
	transport = NULL;
	{
		guint i;
		/* Iterate over transports*/
		i = (guint) 0;
		for (; i < size; i++) {
			guint tsid;
			val = *gst_value_list_get_value (&transports, i);
			transport = gst_value_get_structure (&val);
			tsid = 0U;
			gst_structure_get_uint (transport, "transport-stream-id", &tsid);
			if (gst_structure_has_field (transport, "delivery")) {
				GValue delivery_val;
				const GstStructure* delivery;
				guint freq;
				const GstStructure* _tmp3;
				delivery_val = *gst_structure_get_value (transport, "delivery");
				delivery = gst_value_get_structure (&delivery_val);
				g_debug ("Scanner.vala:473: Received TS %u", tsid);
				gee_map_set ((GeeMap*) self->transport_streams, GUINT_TO_POINTER (tsid), delivery);
				freq = 0U;
				gst_structure_get_uint (delivery, "frequency", &freq);
				/* Takes care of duplicates*/
				_tmp3 = NULL;
				dvb_scanner_add_structure_to_scan (self, (_tmp3 = delivery, (_tmp3 == NULL) ? NULL : gst_structure_copy (_tmp3)));
			}
			if (gst_structure_has_field (transport, "channels")) {
				GValue channels;
				guint channels_size;
				GValue channel_val = {0};
				const GstStructure* channel_struct;
				channels = *gst_structure_get_value (transport, "channels");
				channels_size = gst_value_list_get_size (&channels);
				channel_struct = NULL;
				{
					gint j;
					/* Iterate over channels*/
					j = 0;
					for (; j < channels_size; j++) {
						guint sid;
						DVBChannel* dvb_channel;
						guint lcnumber;
						channel_val = *gst_value_list_get_value (&channels, (guint) j);
						channel_struct = gst_value_get_structure (&channel_val);
						sid = 0U;
						gst_structure_get_uint (channel_struct, "service-id", &sid);
						if (!dvb_channel_list_contains (self->channels, sid)) {
							dvb_scanner_add_new_channel (self, sid);
						}
						dvb_channel = dvb_channel_list_get_channel (self->channels, sid);
						if (g_utf8_validate (name, -1, NULL)) {
							dvb_channel_set_Network (dvb_channel, name);
						} else {
							dvb_channel_set_Network (dvb_channel, "");
						}
						lcnumber = 0U;
						gst_structure_get_uint (channel_struct, "logical-channel-number", &lcnumber);
						dvb_channel_set_LogicalChannelNumber (dvb_channel, &lcnumber);
						(dvb_channel == NULL) ? NULL : (dvb_channel = (g_object_unref (dvb_channel), NULL));
					}
				}
			}
		}
	}
	self->priv->nit_arrived = TRUE;
	name = (g_free (name), NULL);
}


void dvb_scanner_on_pmt_structure (DVBScanner* self, const GstStructure* structure) {
	guint program_number;
	DVBChannel* dvb_channel;
	GValue streams;
	guint size;
	GValue stream_val = {0};
	const GstStructure* stream;
	g_return_if_fail (self != NULL);
	g_return_if_fail (structure != NULL);
	g_debug ("Scanner.vala:519: Received PMT");
	program_number = 0U;
	gst_structure_get_uint (structure, "program-number", &program_number);
	if (!dvb_channel_list_contains (self->channels, program_number)) {
		dvb_scanner_add_new_channel (self, program_number);
	}
	dvb_channel = dvb_channel_list_get_channel (self->channels, program_number);
	streams = *gst_structure_get_value (structure, "streams");
	size = gst_value_list_get_size (&streams);
	stream = NULL;
	{
		gint i;
		/* Iterate over streams*/
		i = 0;
		for (; i < size; i++) {
			guint pid;
			guint stream_type;
			stream_val = *gst_value_list_get_value (&streams, (guint) i);
			stream = gst_value_get_structure (&stream_val);
			pid = 0U;
			gst_structure_get_uint (stream, "pid", &pid);
			/* See ISO/IEC 13818-1 Table 2-29*/
			stream_type = 0U;
			gst_structure_get_uint (stream, "stream-type", &stream_type);
			switch (stream_type) {
				case 0x01:
				case 0x02:
				case 0x1b:
				{
					g_debug ("Scanner.vala:551: Found video PID %u for channel %u", pid, program_number);
					dvb_channel_set_VideoPID (dvb_channel, pid);
					break;
				}
				case 0x03:
				case 0x04:
				case 0x0f:
				case 0x11:
				{
					g_debug ("Scanner.vala:559: Found audio PID %u for channel %u", pid, program_number);
					gee_collection_add ((GeeCollection*) dvb_channel_get_AudioPIDs (dvb_channel), GUINT_TO_POINTER (pid));
					break;
				}
				default:
				{
					g_debug ("Scanner.vala:564: Other stream type: 0x%02x", stream_type);
					break;
				}
			}
		}
	}
	(dvb_channel == NULL) ? NULL : (dvb_channel = (g_object_unref (dvb_channel), NULL));
}


void dvb_scanner_bus_watch_func (DVBScanner* self, GstBus* bus, GstMessage* message) {
	gboolean _tmp4;
	gboolean _tmp10;
	gboolean _tmp11;
	g_return_if_fail (self != NULL);
	g_return_if_fail (bus != NULL);
	g_return_if_fail (message != NULL);
	switch (message->type) {
		case GST_MESSAGE_ELEMENT:
		{
			{
				if (_vala_strcmp0 (gst_structure_get_name (message->structure), "dvb-frontend-stats") == 0) {
					dvb_scanner_on_dvb_frontend_stats_structure (self, message->structure);
				} else {
					if (_vala_strcmp0 (gst_structure_get_name (message->structure), "dvb-read-failure") == 0) {
						dvb_scanner_on_dvb_read_failure_structure (self);
					} else {
						if (_vala_strcmp0 (gst_structure_get_name (message->structure), "sdt") == 0) {
							dvb_scanner_on_sdt_structure (self, message->structure);
						} else {
							if (_vala_strcmp0 (gst_structure_get_name (message->structure), "nit") == 0) {
								dvb_scanner_on_nit_structure (self, message->structure);
							} else {
								if (_vala_strcmp0 (gst_structure_get_name (message->structure), "pat") == 0) {
									dvb_scanner_on_pat_structure (self, message->structure);
								} else {
									if (_vala_strcmp0 (gst_structure_get_name (message->structure), "pmt") == 0) {
										dvb_scanner_on_pmt_structure (self, message->structure);
									}
								}
							}
						}
					}
				}
				break;
			}
		}
		case GST_MESSAGE_ERROR:
		{
			{
				GError* gerror;
				char* debug;
				char* _tmp3;
				char* _tmp2;
				GError* _tmp1;
				GError* _tmp0;
				gerror = NULL;
				debug = NULL;
				_tmp3 = NULL;
				_tmp2 = NULL;
				_tmp1 = NULL;
				_tmp0 = NULL;
				(gst_message_parse_error (message, &_tmp0, &_tmp2), gerror = (_tmp1 = _tmp0, (gerror == NULL) ? NULL : (gerror = (g_error_free (gerror), NULL)), _tmp1));
				debug = (_tmp3 = _tmp2, debug = (g_free (debug), NULL), _tmp3);
				g_critical ("Scanner.vala:591: %s %s", gerror->message, debug);
				dvb_scanner_Destroy (self);
				(gerror == NULL) ? NULL : (gerror = (g_error_free (gerror), NULL));
				debug = (g_free (debug), NULL);
				break;
			}
		}
	}
	_tmp4 = FALSE;
	if (self->priv->nit_arrived) {
		_tmp4 = self->priv->sdt_arrived;
	} else {
		_tmp4 = FALSE;
	}
	/* NIT gives us the transport stream, SDT links SID and TS ID */
	if (_tmp4) {
		{
			GeeIterator* _sid_it;
			_sid_it = gee_iterable_iterator ((GeeIterable*) self->priv->new_channels);
			while (gee_iterator_next (_sid_it)) {
				guint sid;
				DVBChannel* channel;
				guint tsid;
				sid = GPOINTER_TO_UINT (gee_iterator_get (_sid_it));
				channel = dvb_channel_list_get_channel (self->channels, sid);
				tsid = dvb_channel_get_TransportStreamId (channel);
				/* Check if already came across the transport stream*/
				if (gee_map_contains ((GeeMap*) self->transport_streams, GUINT_TO_POINTER (tsid))) {
					GstStructure* _tmp5;
					/* add values from Gst.Structure to Channel*/
					_tmp5 = NULL;
					dvb_scanner_add_values_from_structure_to_channel (self, _tmp5 = (GstStructure*) gee_map_get ((GeeMap*) self->transport_streams, GUINT_TO_POINTER (tsid)), channel);
					(_tmp5 == NULL) ? NULL : (_tmp5 = (gst_structure_free (_tmp5), NULL));
				} else {
					g_debug ("Scanner.vala:610: TS %u for channel %u does not exist", tsid, sid);
					/* We haven't seen the transport stream, yet
					 Maybe it comes with a later bus message*/
					self->priv->nit_arrived = FALSE;
				}
				(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
			}
			(_sid_it == NULL) ? NULL : (_sid_it = (g_object_unref (_sid_it), NULL));
		}
		/* We received all tables at least once. Add valid channels.*/
		if (self->priv->pat_arrived) {
			GeeArrayList* del_channels;
			del_channels = gee_array_list_new (G_TYPE_UINT, NULL, NULL, g_direct_equal);
			{
				GeeIterator* _sid_it;
				_sid_it = gee_iterable_iterator ((GeeIterable*) self->priv->new_channels);
				while (gee_iterator_next (_sid_it)) {
					guint sid;
					DVBChannel* channel;
					sid = GPOINTER_TO_UINT (gee_iterator_get (_sid_it));
					channel = dvb_channel_list_get_channel (self->channels, sid);
					/* If this fails we may miss video or audio pid,
					 because we didn't came across the sdt, yet   */
					if (dvb_channel_is_valid (channel)) {
						const char* _tmp6;
						const char* _tmp7;
						char* type;
						char* _tmp8;
						_tmp6 = NULL;
						if (dvb_channel_get_VideoPID (channel) == 0) {
							_tmp6 = "Radio";
						} else {
							_tmp6 = "TV";
						}
						_tmp7 = NULL;
						type = (_tmp7 = _tmp6, (_tmp7 == NULL) ? NULL : g_strdup (_tmp7));
						_tmp8 = NULL;
						g_debug ("Scanner.vala:628: Channel added: %s", _tmp8 = dvb_channel_to_string (channel));
						_tmp8 = (g_free (_tmp8), NULL);
						g_signal_emit_by_name (self, "channel-added", dvb_channel_get_Frequency (channel), sid, dvb_channel_get_Name (channel), dvb_channel_get_Network (channel), type, dvb_channel_get_Scrambled (channel));
						/* Mark channel for deletion of this.new_channels*/
						gee_collection_add ((GeeCollection*) del_channels, GUINT_TO_POINTER (sid));
						type = (g_free (type), NULL);
					} else {
						char* _tmp9;
						_tmp9 = NULL;
						g_debug ("Scanner.vala:635: Channel %u is not valid: %s", sid, _tmp9 = dvb_channel_to_string (channel));
						_tmp9 = (g_free (_tmp9), NULL);
					}
					(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
				}
				(_sid_it == NULL) ? NULL : (_sid_it = (g_object_unref (_sid_it), NULL));
			}
			/* Only remove those channels we have all the information for*/
			{
				GeeIterator* _sid_it;
				/* Only remove those channels we have all the information for*/
				_sid_it = gee_iterable_iterator ((GeeIterable*) del_channels);
				/* Only remove those channels we have all the information for*/
				while (gee_iterator_next (_sid_it)) {
					guint sid;
					/* Only remove those channels we have all the information for*/
					sid = GPOINTER_TO_UINT (gee_iterator_get (_sid_it));
					gee_collection_remove ((GeeCollection*) self->priv->new_channels, GUINT_TO_POINTER (sid));
				}
				(_sid_it == NULL) ? NULL : (_sid_it = (g_object_unref (_sid_it), NULL));
			}
			(del_channels == NULL) ? NULL : (del_channels = (g_object_unref (del_channels), NULL));
		}
	}
	_tmp10 = FALSE;
	_tmp11 = FALSE;
	if (self->priv->sdt_arrived) {
		_tmp11 = self->priv->nit_arrived;
	} else {
		_tmp11 = FALSE;
	}
	if (_tmp11) {
		_tmp10 = self->priv->pat_arrived;
	} else {
		_tmp10 = FALSE;
	}
	/* If we collect all information we can continue scanning
	 the next frequency*/
	if (_tmp10) {
		dvb_scanner_remove_wait_for_tables_timeout (self);
		dvb_scanner_start_scan (self);
	}
}


void dvb_scanner_add_new_channel (DVBScanner* self, guint sid) {
	DVBChannel* new_channel;
	g_return_if_fail (self != NULL);
	g_debug ("Scanner.vala:657: Adding new channel with SID %u", sid);
	new_channel = dvb_scanner_get_new_channel (self);
	dvb_channel_set_Sid (new_channel, sid);
	dvb_channel_list_add (self->channels, new_channel);
	gee_collection_add ((GeeCollection*) self->priv->new_channels, GUINT_TO_POINTER (sid));
	(new_channel == NULL) ? NULL : (new_channel = (g_object_unref (new_channel), NULL));
}


DVBDevice* dvb_scanner_get_Device (DVBScanner* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_Device;
}


static void dvb_scanner_set_Device (DVBScanner* self, DVBDevice* value) {
	DVBDevice* _tmp2;
	DVBDevice* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_Device = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_object_ref (_tmp1)), (self->priv->_Device == NULL) ? NULL : (self->priv->_Device = (g_object_unref (self->priv->_Device), NULL)), _tmp2);
	g_object_notify ((GObject *) self, "Device");
}


DVBChannelList* dvb_scanner_get_Channels (DVBScanner* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->channels;
}


static guint _dvb_scanned_item_hash_ghash_func (void* key) {
	return dvb_scanned_item_hash (key);
}


static gboolean _dvb_scanned_item_equal_gequal_func (void* a, void* b) {
	return dvb_scanned_item_equal (a, b);
}


static GObject * dvb_scanner_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBScannerClass * klass;
	GObjectClass * parent_class;
	DVBScanner * self;
	klass = DVB_SCANNER_CLASS (g_type_class_peek (DVB_TYPE_SCANNER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_SCANNER (obj);
	{
		GeeHashSet* _tmp0;
		GeeArrayList* _tmp1;
		GQueue* _tmp2;
		GeeHashMap* _tmp3;
		_tmp0 = NULL;
		self->scanned_frequencies = (_tmp0 = gee_hash_set_new (DVB_TYPE_SCANNED_ITEM, (GBoxedCopyFunc) g_object_ref, g_object_unref, _dvb_scanned_item_hash_ghash_func, _dvb_scanned_item_equal_gequal_func), (self->scanned_frequencies == NULL) ? NULL : (self->scanned_frequencies = (g_object_unref (self->scanned_frequencies), NULL)), _tmp0);
		_tmp1 = NULL;
		self->priv->new_channels = (_tmp1 = gee_array_list_new (G_TYPE_UINT, NULL, NULL, g_direct_equal), (self->priv->new_channels == NULL) ? NULL : (self->priv->new_channels = (g_object_unref (self->priv->new_channels), NULL)), _tmp1);
		_tmp2 = NULL;
		self->frequencies = (_tmp2 = g_queue_new (), (self->frequencies == NULL) ? NULL : (self->frequencies = (g_queue_free (self->frequencies), NULL)), _tmp2);
		_tmp3 = NULL;
		self->transport_streams = (_tmp3 = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, G_TYPE_POINTER, (GBoxedCopyFunc) gst_structure_copy, gst_structure_free, g_direct_hash, g_direct_equal, g_direct_equal), (self->transport_streams == NULL) ? NULL : (self->transport_streams = (g_object_unref (self->transport_streams), NULL)), _tmp3);
	}
	return obj;
}


static void dvb_scanner_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	DVBScanner * self;
	gpointer boxed;
	self = DVB_SCANNER (object);
	switch (property_id) {
		case DVB_SCANNER_DEVICE:
		g_value_set_object (value, dvb_scanner_get_Device (self));
		break;
		case DVB_SCANNER_CHANNELS:
		g_value_set_object (value, dvb_scanner_get_Channels (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_scanner_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	DVBScanner * self;
	self = DVB_SCANNER (object);
	switch (property_id) {
		case DVB_SCANNER_DEVICE:
		dvb_scanner_set_Device (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_scanner_class_init (DVBScannerClass * klass) {
	dvb_scanner_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBScannerPrivate));
	G_OBJECT_CLASS (klass)->get_property = dvb_scanner_get_property;
	G_OBJECT_CLASS (klass)->set_property = dvb_scanner_set_property;
	G_OBJECT_CLASS (klass)->constructor = dvb_scanner_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_scanner_finalize;
	DVB_SCANNER_CLASS (klass)->prepare = dvb_scanner_real_prepare;
	DVB_SCANNER_CLASS (klass)->get_scanned_item = dvb_scanner_real_get_scanned_item;
	DVB_SCANNER_CLASS (klass)->get_new_channel = dvb_scanner_real_get_new_channel;
	DVB_SCANNER_CLASS (klass)->add_values_from_structure_to_channel = dvb_scanner_real_add_values_from_structure_to_channel;
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_SCANNER_DEVICE, g_param_spec_object ("Device", "Device", "Device", DVB_TYPE_DEVICE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_SCANNER_CHANNELS, g_param_spec_object ("Channels", "Channels", "Channels", DVB_TYPE_CHANNEL_LIST, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_signal_new ("destroyed", DVB_TYPE_SCANNER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
	g_signal_new ("frequency_scanned", DVB_TYPE_SCANNER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_user_marshal_VOID__UINT_UINT, G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_UINT);
	g_signal_new ("channel_added", DVB_TYPE_SCANNER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_user_marshal_VOID__UINT_UINT_STRING_STRING_STRING_BOOLEAN, G_TYPE_NONE, 6, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_BOOLEAN);
	g_signal_new ("finished", DVB_TYPE_SCANNER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
}


static void dvb_scanner_instance_init (DVBScanner * self) {
	self->priv = DVB_SCANNER_GET_PRIVATE (self);
}


static void dvb_scanner_finalize (GObject* obj) {
	DVBScanner * self;
	self = DVB_SCANNER (obj);
	(self->priv->_Device == NULL) ? NULL : (self->priv->_Device = (g_object_unref (self->priv->_Device), NULL));
	(self->channels == NULL) ? NULL : (self->channels = (g_object_unref (self->channels), NULL));
	(self->pipeline == NULL) ? NULL : (self->pipeline = (gst_object_unref (self->pipeline), NULL));
	(self->frequencies == NULL) ? NULL : (self->frequencies = (g_queue_free (self->frequencies), NULL));
	(self->current_tuning_params == NULL) ? NULL : (self->current_tuning_params = (gst_structure_free (self->current_tuning_params), NULL));
	(self->scanned_frequencies == NULL) ? NULL : (self->scanned_frequencies = (g_object_unref (self->scanned_frequencies), NULL));
	(self->transport_streams == NULL) ? NULL : (self->transport_streams = (g_object_unref (self->transport_streams), NULL));
	(self->priv->new_channels == NULL) ? NULL : (self->priv->new_channels = (g_object_unref (self->priv->new_channels), NULL));
	G_OBJECT_CLASS (dvb_scanner_parent_class)->finalize (obj);
}


GType dvb_scanner_get_type (void) {
	static GType dvb_scanner_type_id = 0;
	if (dvb_scanner_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBScannerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_scanner_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBScanner), 0, (GInstanceInitFunc) dvb_scanner_instance_init, NULL };
		dvb_scanner_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBScanner", &g_define_type_info, G_TYPE_FLAG_ABSTRACT);
	}
	return dvb_scanner_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}



static void g_cclosure_user_marshal_VOID__UINT_UINT (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef void (*GMarshalFunc_VOID__UINT_UINT) (gpointer data1, guint arg_1, guint arg_2, gpointer data2);
	register GMarshalFunc_VOID__UINT_UINT callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	cc = (GCClosure *) closure;
	g_return_if_fail (n_param_values == 3);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = (GMarshalFunc_VOID__UINT_UINT) (marshal_data ? marshal_data : cc->callback);
	callback (data1, g_value_get_uint (param_values + 1), g_value_get_uint (param_values + 2), data2);
}


static void g_cclosure_user_marshal_VOID__UINT_UINT_STRING_STRING_STRING_BOOLEAN (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef void (*GMarshalFunc_VOID__UINT_UINT_STRING_STRING_STRING_BOOLEAN) (gpointer data1, guint arg_1, guint arg_2, const char* arg_3, const char* arg_4, const char* arg_5, gboolean arg_6, gpointer data2);
	register GMarshalFunc_VOID__UINT_UINT_STRING_STRING_STRING_BOOLEAN callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	cc = (GCClosure *) closure;
	g_return_if_fail (n_param_values == 7);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = (GMarshalFunc_VOID__UINT_UINT_STRING_STRING_STRING_BOOLEAN) (marshal_data ? marshal_data : cc->callback);
	callback (data1, g_value_get_uint (param_values + 1), g_value_get_uint (param_values + 2), g_value_get_string (param_values + 3), g_value_get_string (param_values + 4), g_value_get_string (param_values + 5), g_value_get_boolean (param_values + 6), data2);
}



