/* 
   Copyright (C) 2004 James Bowes <bowes@cs.dal.ca>
   Copyright (C) 2004 GNOME Love Project <gnome-love@gnome.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>
#include <libgnomeui/gnome-ui-init.h>
#include <libgnomeui/gnome-stock-icons.h>
#include <gnome-keyring.h>

#include <string.h>
#include <stdlib.h>

#include "gnome-keyring-manager.h"
#include "gnome-keyring-manager-i18n.h"
#include "gnome-keyring-manager-util.h"
#include "gnome-keyring-manager-attribute-display.h"
#include "gnome-keyring-manager-keyring-manager.h"
#include "gnome-keyring-manager-attribute-editor.h"
#include "gnome-keyring-manager-new-item-dialog.h"
#include "gnome-keyring-manager-keyring-editor.h"

#define GKM_KEYRING_EDITOR_GCONF_DIR GNOME_KEYRING_MANAGER_GCONF_PREFIX "/keyring-editor"

enum
{
  ITEM_COLUMN_NAME = 0,
  ITEM_COLUMN_TYPE,
  ITEM_COLUMN_MTIME,
  ITEM_COLUMN_CTIME,
  ITEM_COLUMN_ID,
  ITEM_COLUMN_SECRET,
  NUM_ITEM_COLUMNS
};

struct _GKMKeyringEditorPrivate
{
  GtkWidget *items_tree_view;
  GtkWidget *attributes_display;

  GtkWidget *secret_label;
  GtkWidget *secret_field;
  GtkWidget *secret_multiline;
  GtkWidget *secret_scroll;
  GtkWidget *secret_event_box;
  GtkWidget *secret_expander;
  GtkWidget *secret_dnd_label;
  GtkWidget *delete_button;
  GtkWidget *add_button;
  GtkWidget *edit_button;

  gboolean is_multiline_secret;

  GtkUIManager *ui_manager;

  GtkListStore *keyring_items;

  char *keyring_name;

  GSList *gconf_cnxn_ids;
};

enum
{
  GKM_DND_TEXT_PLAIN,
  GKM_DND_UTF8_STRING,
  GKM_DND_COMPOUND_TEXT,
  GKM_DND_TEXT,
  GKM_DND_STRING,
  GKM_DND_NTARGETS
};


static GtkTargetEntry drag_types [] =
{
  { "text/plain",   0, GKM_DND_TEXT_PLAIN },
  { "UTF8_STRING",   0, GKM_DND_UTF8_STRING },
  { "COMPOUND_TEXT", 0, GKM_DND_COMPOUND_TEXT },
  { "TEXT",          0, GKM_DND_TEXT },
  { "STRING",        0, GKM_DND_STRING }
};

/* What should really be done with this? */
#define GKM_DND_KEYRING_ITEM 12

static GtkTargetEntry keyring_item_drag_type [] =
{
  { "gkm-keyring-item", 0, GKM_DND_KEYRING_ITEM }
};

static void gkm_keyring_editor_class_init (GKMKeyringEditor *class);
static void gkm_keyring_editor_init       (GKMKeyringEditor *window);
static void gkm_keyring_editor_finalize   (GObject *object);
static void gkm_keyring_editor_destroy    (GtkObject *object);

const char *keyring_item_type_text_from_enum (GnomeKeyringItemType type);
static void gkm_keyring_editor_setup_keyring_items_tree (GKMKeyringEditor *editor);

static void open_keyring_manager_callback (GtkWidget *widget, GKMKeyringEditor *editor);
static void add_keyring_item_callback (GtkWidget *widget, GKMKeyringEditor *editor);
static void remove_keyring_item_callback (GtkWidget *widget, GKMKeyringEditor *editor);
static void edit_keyring_item_callback (GtkWidget *widget, GKMKeyringEditor *editor);
static void drag_keyring_item_secret_data_get_callback (GtkWidget *widget, GdkDragContext *context, GtkSelectionData *selection_data,
							guint info, guint32 time, gpointer callback_data);

static void drag_keyring_item_data_get_callback (GtkWidget *widget, GdkDragContext *context, GtkSelectionData *selection_data,
						 guint info, guint32 time, GKMKeyringEditor *editor);
static void drop_keyring_item_data_recieved_callback (GtkWidget *widget, GdkDragContext *context, gint x, gint y, 
						      GtkSelectionData *selection_data, guint info, guint32 time, GKMKeyringEditor *editor);

static void close_action_callback (GtkAction *action, GKMKeyringEditor *editor);
static void about_action_callback (GtkAction *action, GKMKeyringEditor *editor);
static void toggle_view_column_action_callback (GtkToggleAction *action, GtkWidget *widget);
static gboolean keyring_item_secret_changed_callback (GtkWidget *entry, GdkEventFocus *event, GKMKeyringEditor *editor);

static void gkm_keyring_editor_expander_changed_callback (GtkWidget *expander, GParamSpec *spec, GKMKeyringEditor *editor);

static void keyring_items_tree_context_menu_callback (GtkWidget *tree_view, GKMKeyringEditor *editor);
static gboolean keyring_items_tree_button_press_callback (GtkWidget *tree_view, GdkEventButton *event, GKMKeyringEditor *editor);

static void keyring_item_tree_selection_changed (GtkTreeSelection *selection, GKMKeyringEditor *editor);
static void keyring_editor_update_sensitivity (GKMKeyringEditor *editor, gboolean sensitive);

static void gkm_keyring_editor_update_keyring_items (GKMKeyringEditor *transient_parent);
static void gkm_keyring_editor_update_keyring_item_info (GKMKeyringEditor *editor, guint32 item_id, GtkTreeIter *iter);

static GtkWindowClass *parent_class = NULL;

static GConfClient *gconf_client = NULL;

GType
gkm_keyring_editor_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo info =
      {
        sizeof (GKMKeyringEditorClass),
	NULL,		/* base_init */
	NULL,		/* base_finalize */
        (GClassInitFunc) gkm_keyring_editor_class_init,
	NULL,		/* class_finalize */
	NULL,		/* class_data */ 
	sizeof (GKMKeyringEditor),
	0,		/* n_preallocs */
	(GInstanceInitFunc) gkm_keyring_editor_init,
	0
      };

      type = g_type_register_static (GTK_TYPE_WINDOW, "GKMKeyringEditor", &info, 0);
    }

  return type;
}

static void
gkm_keyring_editor_class_init (GKMKeyringEditor *class)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (class);
  GtkObjectClass *object_class = GTK_OBJECT_CLASS (class);

  parent_class = g_type_class_peek_parent (class);

  gobject_class->finalize = gkm_keyring_editor_finalize;

  object_class->destroy = gkm_keyring_editor_destroy;

  gconf_client = gconf_client_get_default ();
}

static GtkActionEntry entries[] =
{
    { "KeyringsMenu", NULL, N_("_Keyrings"), NULL, NULL, NULL },
    { "EditMenu", NULL, N_("_Edit"), NULL, NULL, NULL },
    { "ViewMenu", NULL, N_("_View"), NULL, NULL, NULL },
    { "HelpMenu", NULL, N_("_Help"), NULL, NULL, NULL },

    { "Close", GTK_STOCK_CLOSE, N_("_Close"), "<control>W", N_("Close the keyring editor window"), 
      G_CALLBACK (close_action_callback) },
    { "OpenKeyringManager", NULL, N_("_Open Keyring Manager"), "<control>O", N_("Open the Gnome Keyring Manager"), 
      G_CALLBACK (open_keyring_manager_callback) },
    { "About", GNOME_STOCK_ABOUT, N_("_About"), NULL, NULL,
      G_CALLBACK (about_action_callback) },

    { "EditItem", NULL, N_("_Edit"), NULL, N_("Edit the selected keyring item"),
      G_CALLBACK (edit_keyring_item_callback) },
    { "DeleteItem", GTK_STOCK_DELETE, N_("_Delete"), NULL, N_("Delete the selected keyring item"),
      G_CALLBACK (remove_keyring_item_callback) }
};

static GtkToggleActionEntry toggle_entries[] =
{
  { "ToggleModificationTime", NULL, N_("Modification Time"), NULL, N_("Toggle display of keyring item modification time"), 
    G_CALLBACK (toggle_view_column_action_callback), FALSE },
  { "ToggleCreationTime", NULL, N_("Creation Time"), NULL, N_("Toggle display of keyring item creation time"), 
    G_CALLBACK (toggle_view_column_action_callback), FALSE }
};

static const char *ui_description =
  "<ui>"
  "  <menubar name='KeyringEditorMenu'>"
  "    <menu action='KeyringsMenu'>"
  "      <menuitem action='OpenKeyringManager'/>"
  "      <separator/>"
  "      <menuitem action='Close'/>"
  "    </menu>"
  "    <menu action='EditMenu'>"
  "    </menu>"
  "    <menu action='ViewMenu'>"
  "      <menuitem action='ToggleModificationTime'/>"
  "      <menuitem action='ToggleCreationTime'/>"
  "    </menu>"
  "    <menu action='HelpMenu'>"
  "      <menuitem action='About'/>"
  "    </menu>"
  "  </menubar>"
  "  <popup name='KeyringItemContextMenu'>"
  "    <menuitem action='EditItem'/>"
  "    <menuitem action='DeleteItem'/>"
  "  </popup>"
  "</ui>";

static void
gkm_keyring_editor_init (GKMKeyringEditor *editor)
{
  GtkWidget *main_vbox;
  GtkWidget *groups_vbox;
  GtkWidget *group_vbox;
  GtkWidget *hbox;
  GtkWidget *vbuttonbox;
  GtkWidget *frame;
  GtkWidget *scrolledwindow;
  GtkWidget *label;
  GtkWidget *image;
  GtkWidget *menubar;
  GtkWidget *alignment;
  GtkSizeGroup *button_sizegroup;
  GtkTooltips *tooltips;
  GtkAccelGroup *accel_group;
  GtkActionGroup *action_group;
  GdkPixbuf *keyring_pixbuf;
  GError *error;
  char *text;
  GtkTreeSelection *item_tree_selection;

  editor->priv = g_new0 (GKMKeyringEditorPrivate, 1);

  gtk_window_set_default_size (GTK_WINDOW (editor), 450, 600);

  tooltips = gtk_tooltips_new ();
  button_sizegroup = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);

  main_vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (editor), main_vbox);
  
  action_group = gtk_action_group_new ("KeyringEditorMenuActions");
  gtk_action_group_add_actions (action_group, entries, G_N_ELEMENTS (entries), editor);
  gtk_action_group_add_toggle_actions (action_group, toggle_entries, G_N_ELEMENTS (toggle_entries), editor);
  
  editor->priv->ui_manager = gtk_ui_manager_new ();
  gtk_ui_manager_insert_action_group (editor->priv->ui_manager, action_group, 0);

  accel_group = gtk_ui_manager_get_accel_group (editor->priv->ui_manager);
  gtk_window_add_accel_group (GTK_WINDOW (editor), accel_group);

  error = NULL;
  if (!gtk_ui_manager_add_ui_from_string (editor->priv->ui_manager, ui_description, -1, &error))
    {
      g_message ("Building menus failed: %s", error->message);
      g_error_free (error);
      exit (1);
    }

  menubar = gtk_ui_manager_get_widget (editor->priv->ui_manager, "/KeyringEditorMenu");
  gtk_box_pack_start (GTK_BOX (main_vbox), menubar, FALSE, FALSE, 0);
  
  groups_vbox = gtk_vbox_new (FALSE, 18);
  gtk_container_set_border_width (GTK_CONTAINER (groups_vbox), 12);
  gtk_box_pack_start (GTK_BOX (main_vbox), groups_vbox, TRUE, TRUE, 0);

  /* The Items group */
  group_vbox = gtk_vbox_new (FALSE, 12);
  gtk_box_pack_start (GTK_BOX (groups_vbox), group_vbox, TRUE, TRUE, 0);

  label = gtk_label_new (NULL);
  text = g_strdup_printf ("<span weight='bold'>%s</span>", _("Items"));
  gtk_label_set_markup (GTK_LABEL (label), text);
  g_free (text);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (group_vbox), label, FALSE, FALSE, 0);
  
  alignment = gtk_alignment_new (0.0, 0.0, 1.0, 1.0);
  gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0, 18, 0);
  gtk_box_pack_start (GTK_BOX (group_vbox), alignment, TRUE, TRUE, 0);

  hbox = gtk_hbox_new (FALSE, 12);
  gtk_container_add (GTK_CONTAINER (alignment), hbox);
  
  scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_SHADOW_ETCHED_IN);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (hbox), scrolledwindow, TRUE, TRUE, 0);
 
  editor->priv->items_tree_view = gtk_tree_view_new ();
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (editor->priv->items_tree_view), TRUE);
  gtk_container_add (GTK_CONTAINER (scrolledwindow), editor->priv->items_tree_view);
  g_object_set_data (G_OBJECT (editor), "keyring-items", editor->priv->items_tree_view);
  
  item_tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (editor->priv->items_tree_view));
  g_signal_connect( item_tree_selection, "changed",
                    G_CALLBACK (keyring_item_tree_selection_changed), editor);

  gkm_keyring_editor_setup_keyring_items_tree (editor);
  g_signal_connect (editor->priv->items_tree_view, "button-press-event",
		    G_CALLBACK (keyring_items_tree_button_press_callback), editor);
  g_signal_connect (editor->priv->items_tree_view, "popup-menu", 
		    G_CALLBACK (keyring_items_tree_context_menu_callback), editor);
  
  gtk_drag_source_set (GTK_WIDGET (editor->priv->items_tree_view),
                       GDK_BUTTON1_MASK | GDK_BUTTON3_MASK,
                       keyring_item_drag_type, 1,
                       GDK_ACTION_MOVE);
  g_signal_connect (G_OBJECT (editor->priv->items_tree_view), "drag_data_get",
		    G_CALLBACK (drag_keyring_item_data_get_callback), editor);
  {
    SetColumnPrefsData toggle [] =
    {
      { ITEM_COLUMN_MTIME, "ToggleModificationTime",	"mtime" },
      { ITEM_COLUMN_CTIME, "ToggleCreationTime",	"ctime" },
    };

    editor->priv->gconf_cnxn_ids =
      set_column_visibility_preferences (toggle,
                                         G_N_ELEMENTS (toggle),
                                         GKM_KEYRING_EDITOR_GCONF_DIR,
                                         GTK_TREE_VIEW (editor->priv->items_tree_view),
                                         action_group,
                                         gconf_client);
  }

  vbuttonbox = gtk_vbutton_box_new ();
  gtk_box_set_spacing (GTK_BOX (vbuttonbox), 12);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (vbuttonbox), GTK_BUTTONBOX_START);
  gtk_box_pack_start (GTK_BOX (hbox), vbuttonbox, FALSE, FALSE, 0);

  editor->priv->delete_button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
  gtk_size_group_add_widget (button_sizegroup, editor->priv->delete_button);
  gtk_box_pack_start (GTK_BOX (vbuttonbox), editor->priv->delete_button, FALSE, FALSE, 0);
  g_signal_connect (editor->priv->delete_button, "clicked", G_CALLBACK (remove_keyring_item_callback), editor);

  editor->priv->add_button = gtk_button_new_from_stock (GTK_STOCK_NEW);
  gtk_size_group_add_widget (button_sizegroup, editor->priv->add_button);
  gtk_box_pack_start (GTK_BOX (vbuttonbox), editor->priv->add_button, FALSE, FALSE, 0);
  g_signal_connect (editor->priv->add_button, "clicked", G_CALLBACK (add_keyring_item_callback), editor);

  /* Item Attributes group */
  group_vbox = gtk_vbox_new (FALSE, 12);
  gtk_box_pack_start (GTK_BOX (groups_vbox), group_vbox, FALSE, FALSE, 0);

  label = gtk_label_new (NULL);
  text = g_strdup_printf("<span weight=\"bold\">%s</span>", _("Item Attributes"));
  gtk_label_set_markup (GTK_LABEL (label), text);
  g_free (text);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (group_vbox), label, FALSE, FALSE, 0);

  alignment = gtk_alignment_new (0.0, 0.0, 1.0, 1.0);
  gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0, 18, 0);
  gtk_box_pack_start (GTK_BOX (group_vbox), alignment, FALSE, FALSE, 0);

  hbox = gtk_hbox_new (FALSE, 12);
  gtk_container_add (GTK_CONTAINER (alignment), hbox);

  editor->priv->attributes_display = gkm_attribute_display_new ();
  gtk_box_pack_start (GTK_BOX (hbox), editor->priv->attributes_display, TRUE, TRUE, 0);
  g_object_set_data (G_OBJECT (editor), "keyring-item-attributes", editor->priv->attributes_display);
  
  vbuttonbox = gtk_vbutton_box_new ();
  gtk_box_set_spacing (GTK_BOX (vbuttonbox), 12);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (vbuttonbox), GTK_BUTTONBOX_START);
  gtk_box_pack_start (GTK_BOX (hbox), vbuttonbox, FALSE, FALSE, 0);

  editor->priv->edit_button = gtk_button_new_with_mnemonic (_("_Advanced"));
  gtk_size_group_add_widget (button_sizegroup, editor->priv->edit_button);
  gtk_box_pack_start (GTK_BOX (vbuttonbox), editor->priv->edit_button, FALSE, FALSE, 0);
  g_signal_connect (editor->priv->edit_button, "clicked", G_CALLBACK (edit_keyring_item_callback), editor);

  gtk_tooltips_set_tip (GTK_TOOLTIPS (tooltips), editor->priv->edit_button, _("Edit the selected item."), NULL);

  editor->priv->secret_expander = gtk_expander_new_with_mnemonic (_("_Show Secret"));
  gtk_expander_set_spacing (GTK_EXPANDER (editor->priv->secret_expander), 6);
  gtk_box_pack_start (GTK_BOX (groups_vbox), editor->priv->secret_expander, FALSE, FALSE, 0);
  g_signal_connect (G_OBJECT (editor->priv->secret_expander), "notify::expanded", 
  		    G_CALLBACK (gkm_keyring_editor_expander_changed_callback), editor);
 
  hbox = gtk_hbox_new (FALSE, 12);
  gtk_container_add (GTK_CONTAINER (editor->priv->secret_expander), hbox);
  
  alignment = gtk_alignment_new (0.0, 0.0, 0.0, 0.0);
  gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 4, 0, 2, 0);
  gtk_box_pack_start (GTK_BOX (hbox), alignment, FALSE, FALSE, 0);
  
  editor->priv->secret_label = gtk_label_new_with_mnemonic (_("S_ecret:"));
  gtk_container_add (GTK_CONTAINER (alignment), editor->priv->secret_label);

  editor->priv->secret_field = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (hbox), editor->priv->secret_field, TRUE, TRUE, 0);
  g_object_set_data (G_OBJECT (editor), "keyring-item-secret", editor->priv->secret_field);
  g_signal_connect (G_OBJECT (editor->priv->secret_field), "focus-out-event", 
		    G_CALLBACK (keyring_item_secret_changed_callback), editor);

   editor->priv->secret_scroll = gtk_scrolled_window_new (NULL, NULL);
   gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (editor->priv->secret_scroll), GTK_SHADOW_ETCHED_IN);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (editor->priv->secret_scroll),
				   GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
   gtk_box_pack_start (GTK_BOX (hbox), editor->priv->secret_scroll, TRUE, TRUE, 0);

   editor->priv->secret_multiline = gtk_text_view_new();
   gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (editor->priv->secret_multiline), GTK_WRAP_WORD_CHAR);
   gtk_container_add (GTK_CONTAINER (editor->priv->secret_scroll), editor->priv->secret_multiline);
   g_signal_connect (G_OBJECT (editor->priv->secret_multiline), "focus-out-event",
		     G_CALLBACK (keyring_item_secret_changed_callback), editor);

  hbox = gtk_hbox_new (FALSE, 12);
  editor->priv->secret_event_box = gtk_event_box_new ();
  gtk_event_box_set_visible_window (GTK_EVENT_BOX (editor->priv->secret_event_box), FALSE);
  frame = gtk_frame_new ("");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_container_add (GTK_CONTAINER (editor->priv->secret_event_box), frame);
  keyring_pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
					     "stock_keyring", 48, 0, &error);
  if (keyring_pixbuf != NULL)
    {
      image = gtk_image_new_from_pixbuf (keyring_pixbuf);
      g_object_unref (keyring_pixbuf);
    }
  else
    {
       image = gtk_image_new_from_stock (GTK_STOCK_DND, GTK_ICON_SIZE_DND);
    }
  gtk_container_add (GTK_CONTAINER (frame), image);
  gtk_box_pack_start (GTK_BOX (hbox), editor->priv->secret_event_box, FALSE, FALSE, 0);
  editor->priv->secret_dnd_label = gtk_label_new (_("You can copy the secret by dragging this image"));
  gtk_box_pack_start (GTK_BOX (hbox), editor->priv->secret_dnd_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (groups_vbox), hbox, FALSE, FALSE, 0);
  gtk_drag_source_set (GTK_WIDGET (editor->priv->secret_event_box),
                       GDK_BUTTON1_MASK | GDK_BUTTON3_MASK,
                       drag_types, G_N_ELEMENTS (drag_types),
                       GDK_ACTION_COPY);
  g_signal_connect (G_OBJECT (editor->priv->secret_event_box), "drag_data_get",
                    G_CALLBACK (drag_keyring_item_secret_data_get_callback), editor);

  gtk_drag_dest_set (GTK_WIDGET (editor),
		     GTK_DEST_DEFAULT_DROP | GTK_DEST_DEFAULT_MOTION,
		     keyring_item_drag_type, 1,
		     GDK_ACTION_MOVE);
  g_signal_connect (G_OBJECT (editor), "drag_data_received",
		    G_CALLBACK (drop_keyring_item_data_recieved_callback), editor);

  keyring_editor_update_sensitivity (editor, FALSE);

  gtk_widget_show_all (main_vbox);
}

static void
gkm_keyring_editor_finalize (GObject *object)
{
  GKMKeyringEditor *editor;

  g_return_if_fail (GKM_IS_KEYRING_EDITOR (object));

  editor = GKM_KEYRING_EDITOR (object);

  g_free (editor->priv->keyring_name);

  close_gconf_connections (editor->priv->gconf_cnxn_ids, GKM_KEYRING_EDITOR_GCONF_DIR, gconf_client);
  
  g_object_unref (G_OBJECT (editor->priv->ui_manager));
 
  g_free (editor->priv);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gkm_keyring_editor_destroy (GtkObject *object)
{
  GKMKeyringEditor *editor;

  g_return_if_fail (GKM_IS_KEYRING_EDITOR (object));

  editor = GKM_KEYRING_EDITOR (object);

  GTK_OBJECT_CLASS (parent_class)->destroy (GTK_OBJECT (editor));
}

GtkWidget *
gkm_keyring_editor_new (const char *keyring_name,
			GtkWindow  *parent)
{
  GKMKeyringEditor *editor;

  editor = g_object_new (GKM_TYPE_KEYRING_EDITOR, NULL);

  if (keyring_name != NULL)
    {
      if (strcmp (keyring_name, "session") == 0)
        {
          gtk_window_set_title (GTK_WINDOW (editor), _("Session Keyring"));
	}
      else
        {
	  char * window_title;
	  
	  window_title = g_strdup_printf (_("Keyring %s"), keyring_name);
	  gtk_window_set_title (GTK_WINDOW (editor), window_title);

	  g_free (window_title);
        }

      editor->priv->keyring_name = g_strdup (keyring_name);
    }
  
  gkm_keyring_editor_update_keyring_items (editor);
  
  gtk_window_set_transient_for (GTK_WINDOW (editor), parent);

  return GTK_WIDGET (editor);
}

static gint
compare_keyring_item_types (GtkTreeModel *model,
			    GtkTreeIter  *a,
			    GtkTreeIter  *b,
			    gpointer      data)
{
  gint column;
  int type1;
  int type2;

  column = GPOINTER_TO_INT (data);

  gtk_tree_model_get (model, a, column, &type1, -1);
  gtk_tree_model_get (model, b, column, &type2, -1);

  return g_utf8_collate (keyring_item_type_text_from_enum (type1), keyring_item_type_text_from_enum (type2));

}

const char *
keyring_item_type_text_from_enum (GnomeKeyringItemType type)
{
  const char *text;
  
  switch (type)
    {
      case GNOME_KEYRING_ITEM_GENERIC_SECRET:
        text = _("Secret");
        break;
      case GNOME_KEYRING_ITEM_NETWORK_PASSWORD:
        text = _("Network Password");
        break;
      case GNOME_KEYRING_ITEM_NOTE:
        text = _("Note");
        break;
      default:
        g_warning (_("Unknown keyring item type: %d"), type);
        text = "";
        break;
    }

  return text;
}

static void 
keyring_item_type_get_pixbuf (GtkTreeViewColumn *column G_GNUC_UNUSED, 
                              GtkCellRenderer *renderer, 
                              GtkTreeModel *model,
                              GtkTreeIter *iter, 
                              gpointer *data G_GNUC_UNUSED)
{
  int type;

  gtk_tree_model_get (model, iter, ITEM_COLUMN_TYPE, &type, -1);
  switch (type)
    {
      case GNOME_KEYRING_ITEM_GENERIC_SECRET:
        g_object_set (G_OBJECT (renderer), "stock-id", GNOME_STOCK_AUTHENTICATION, NULL);
        break;
      case GNOME_KEYRING_ITEM_NETWORK_PASSWORD:
        g_object_set (G_OBJECT (renderer), "stock-id", GTK_STOCK_NETWORK, NULL);
        break;
      case GNOME_KEYRING_ITEM_NOTE:
        g_object_set (G_OBJECT (renderer), "stock-id", GNOME_STOCK_BOOK_OPEN, NULL);
        break;
      default:
        g_object_set (G_OBJECT (renderer), "stock-id", GNOME_STOCK_BLANK, NULL);
    }
}

static void
gkm_keyring_editor_setup_keyring_items_tree (GKMKeyringEditor *editor)
{
  GtkTreeViewColumn   *column;
  GtkCellRenderer     *renderer;
  
  g_return_if_fail (GKM_IS_KEYRING_EDITOR (editor));
  g_assert (editor->priv->keyring_items == NULL);
  
  editor->priv->keyring_items = gtk_list_store_new (NUM_ITEM_COLUMNS, 
                                                    /* ITEM_COLUMN_NAME */   G_TYPE_STRING, 
                                                    /* ITEM_COLUMN_TYPE */   G_TYPE_INT, 
                                                    /* ITEM_COLUMN_MTIME */  G_TYPE_UINT, 
                                                    /* ITEM_COLUMN_CTIME */  G_TYPE_UINT, 
                                                    /* ITEM_COLUMN_ID */     G_TYPE_UINT, 
                                                    /* ITEM_COLUMN_SECRET */ G_TYPE_STRING);
  
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Type"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (editor->priv->items_tree_view), column);
  renderer = gtk_cell_renderer_pixbuf_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_set_cell_data_func (column, renderer, (GtkTreeCellDataFunc) keyring_item_type_get_pixbuf, NULL, NULL);
  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (editor->priv->keyring_items), ITEM_COLUMN_TYPE, 
                                   (GtkTreeIterCompareFunc) compare_keyring_item_types, GINT_TO_POINTER (ITEM_COLUMN_TYPE), NULL);
  gtk_tree_view_column_set_sort_column_id (column, ITEM_COLUMN_TYPE);

  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Name"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (editor->priv->items_tree_view), column);
  gtk_tree_view_column_set_expand (column, TRUE);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "text", ITEM_COLUMN_NAME);
  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (editor->priv->keyring_items),
  				   ITEM_COLUMN_NAME,
				   (GtkTreeIterCompareFunc) tree_model_compare_strings,
				   GINT_TO_POINTER (ITEM_COLUMN_NAME), NULL);
  gtk_tree_view_column_set_sort_column_id (column, ITEM_COLUMN_NAME);
  
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Modification Time"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (editor->priv->items_tree_view), column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_set_visible (column, FALSE);
  gtk_tree_view_column_set_cell_data_func (column,
  				           renderer,
				           (GtkTreeCellDataFunc) format_date_cell_data_func,
				           GINT_TO_POINTER (ITEM_COLUMN_MTIME), NULL);
  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (editor->priv->keyring_items),
  				   ITEM_COLUMN_MTIME,
				   (GtkTreeIterCompareFunc) tree_model_compare_uints,
				   GINT_TO_POINTER (ITEM_COLUMN_MTIME), NULL);
  gtk_tree_view_column_set_sort_column_id (column, ITEM_COLUMN_MTIME);

  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Creation Time"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (editor->priv->items_tree_view), column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_set_visible (column, FALSE);
  gtk_tree_view_column_set_cell_data_func (column,
  				           renderer,
				           (GtkTreeCellDataFunc) format_date_cell_data_func,
				           GINT_TO_POINTER (ITEM_COLUMN_CTIME), NULL);
  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (editor->priv->keyring_items),
  				   ITEM_COLUMN_CTIME,
				   (GtkTreeIterCompareFunc) tree_model_compare_uints,
				   GINT_TO_POINTER (ITEM_COLUMN_CTIME), NULL);
  gtk_tree_view_column_set_sort_column_id (column, ITEM_COLUMN_CTIME);

  gtk_tree_sortable_set_default_sort_func (GTK_TREE_SORTABLE (editor->priv->keyring_items),
				           (GtkTreeIterCompareFunc) tree_model_compare_uints,
				           GINT_TO_POINTER (ITEM_COLUMN_ID), NULL);

  gtk_tree_view_set_model (GTK_TREE_VIEW (editor->priv->items_tree_view), GTK_TREE_MODEL (editor->priv->keyring_items));
  g_object_unref (G_OBJECT (editor->priv->keyring_items));
  gtk_window_set_focus (GTK_WINDOW (editor), NULL);
}

/***********************************************************
 * Show the proper widget when the expander is opened.
 */

static void
gkm_keyring_editor_show_secret_widget (GKMKeyringEditor *editor)
{
  if (editor->priv->is_multiline_secret)
    {
      gtk_widget_show (editor->priv->secret_scroll);
      gtk_label_set_text_with_mnemonic (GTK_LABEL (editor->priv->secret_label),
					_("_Text:"));
      gtk_label_set_mnemonic_widget (GTK_LABEL (editor->priv->secret_label), 
  				     editor->priv->secret_multiline);
      
      gtk_widget_hide (editor->priv->secret_field);
    }
  else
    {
      gtk_widget_show (editor->priv->secret_field);
      gtk_label_set_text_with_mnemonic (GTK_LABEL (editor->priv->secret_label),
					_("S_ecret:"));
      gtk_label_set_mnemonic_widget (GTK_LABEL (editor->priv->secret_label), 
				     editor->priv->secret_field);

      gtk_widget_hide (editor->priv->secret_scroll);
    }
}

static void 
gkm_keyring_editor_expander_changed_callback (GtkWidget *expander, 
					      GParamSpec *spec G_GNUC_UNUSED, 
					      GKMKeyringEditor *editor)
{
  if (gtk_expander_get_expanded (GTK_EXPANDER (expander)))
    {
      gkm_keyring_editor_show_secret_widget (editor);
    }
}

/***********************************************************
 * Get and set the secret widget.
 */

static gchar *
gkm_keyring_editor_get_secret (GKMKeyringEditor *editor)
{
  gchar *secret;

  g_return_val_if_fail (GKM_IS_KEYRING_EDITOR (editor), NULL);

  if (editor->priv->is_multiline_secret)
    {
      GtkTextBuffer *buffer;
      GtkTextIter start;
      GtkTextIter end;

      buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (editor->priv->secret_multiline));

      gtk_text_buffer_get_start_iter (buffer, &start);
      gtk_text_buffer_get_end_iter (buffer, &end);

      secret = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
    }
  else
    {
      secret = g_strdup (gtk_entry_get_text (GTK_ENTRY (editor->priv->secret_field)));
    }

  return secret;
}

static void
gkm_keyring_editor_set_secret (GKMKeyringEditor *editor, gchar *secret)
{
  g_return_if_fail (GKM_IS_KEYRING_EDITOR (editor));
  
  if (editor->priv->is_multiline_secret)
    {
      GtkTextBuffer *buffer;
      
      buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (editor->priv->secret_multiline));
      
      gtk_text_buffer_set_text (buffer, secret, -1);
    }
  else
    {
      gtk_entry_set_text (GTK_ENTRY (editor->priv->secret_field), secret);
    }
}

/***********************************************************
 * Open the keyring manager, or present it if already opened.
 */

static void
open_keyring_manager_callback (GtkWidget        *button G_GNUC_UNUSED,
			       GKMKeyringEditor *editor)
{
  g_return_if_fail (GKM_IS_KEYRING_EDITOR (editor));

  gkm_application_open_keyring_manager ();
}

/***************************************************************************
 * Create a new keyring item.
 */

static void
add_keyring_item_done_callback (GnomeKeyringResult result,
				guint32 val G_GNUC_UNUSED,
				GKMKeyringEditor *editor)
{
  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (GTK_WINDOW (editor), result);

      return;
    } 

  gkm_keyring_editor_update_keyring_items (editor); 
}

static void
add_keyring_item_action_callback (GKMNewItemDialog *dialog,
				  gint              response_id,
				  GKMKeyringEditor *editor)
{
  g_return_if_fail (GKM_IS_NEW_ITEM_DIALOG (dialog));
  g_return_if_fail (GKM_IS_KEYRING_EDITOR (editor));

  switch (response_id)
    {
      case GTK_RESPONSE_CANCEL:
        break;
      case GTK_RESPONSE_ACCEPT:
        {
	  GnomeKeyringItemType       type;
	  const gchar               *item_name;
	  gchar                     *secret;
	  GnomeKeyringAttributeList *list;
	  
          type = gkm_new_item_dialog_get_item_type (dialog);
	  item_name = gkm_new_item_dialog_get_item_name (dialog);
	  secret = gkm_new_item_dialog_get_item_secret (dialog);
	  list = gkm_new_item_dialog_get_item_attribute_list (dialog);
	  
          gnome_keyring_item_create (editor->priv->keyring_name,
				     type,
				     item_name,
				     list,
				     secret,
				     FALSE,
				     (GnomeKeyringOperationGetIntCallback) add_keyring_item_done_callback,
				     editor, NULL);
        
          gnome_keyring_attribute_list_free (list);
	  g_free (secret);

          break;
	}
    }

  gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
add_keyring_item_callback (GtkWidget        *button G_GNUC_UNUSED,
 		           GKMKeyringEditor *editor)
{
  GtkWidget *dialog;

  g_return_if_fail (GKM_IS_KEYRING_EDITOR (editor));

  dialog = gkm_new_item_dialog_new (editor->priv->keyring_name, GTK_WINDOW (editor));

  g_signal_connect (G_OBJECT (dialog), 
  		    "response", 
		    G_CALLBACK (add_keyring_item_action_callback), 
		    editor);

  gtk_window_present (GTK_WINDOW (dialog));
}

typedef struct _SaveItemCallbackData
{
  GKMKeyringEditor    *editor;
  GKMAttributeEditor  *attribute_editor;
  guint32	       item_id;
  GtkTreeRowReference *row;
} SaveItemCallbackData;

static void
save_item_callback_data_free (SaveItemCallbackData *data)
{
  gtk_tree_row_reference_free (data->row);
  g_free (data);
}


static void
attributes_saved_callback (GnomeKeyringResult result,
			   SaveItemCallbackData *data)
{
  GtkTreePath *path;
  GtkTreeIter iter;
  GtkTreeSelection *selection;

  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (NULL, result);
      return;
    }

  if (!gtk_tree_row_reference_valid (data->row))
    {
      g_warning (_("A row disapeared while we were waiting for the data..."));

      return;
    }

  path = gtk_tree_row_reference_get_path (data->row);
  gtk_tree_model_get_iter (GTK_TREE_MODEL (data->editor->priv->keyring_items), &iter, path);
  gtk_tree_path_free (path);

  gkm_keyring_editor_update_keyring_item_info (data->editor, data->item_id, &iter);
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(data->editor->priv->items_tree_view));
  keyring_item_tree_selection_changed (selection, data->editor);
}

static void
acl_saved_callback (GnomeKeyringResult result,
		    SaveItemCallbackData *data)
{
  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (NULL, result);
      return;
    }

  if (!gtk_tree_row_reference_valid (data->row))
    {
      g_warning (_("A row disapeared while we were waiting for the data..."));

      return;
    }
}

void
save_attributes (SaveItemCallbackData *data)
{
  GnomeKeyringAttributeList *attribute_list;

  attribute_list = gkm_attribute_editor_get_attribute_list (data->attribute_editor);

  gnome_keyring_item_set_attributes (data->editor->priv->keyring_name, data->item_id, attribute_list,
				     (GnomeKeyringOperationDoneCallback) attributes_saved_callback,
				     data, NULL);

  gnome_keyring_attribute_list_free (attribute_list);
}

void
save_acl (SaveItemCallbackData *data)
{
  GList *acl;

  acl = gkm_attribute_editor_get_acl (data->attribute_editor);

  gnome_keyring_item_set_acl (data->editor->priv->keyring_name, data->item_id, acl,
			      (GnomeKeyringOperationDoneCallback) acl_saved_callback,
			      data, (GDestroyNotify) save_item_callback_data_free);

  gnome_keyring_acl_free (acl);
}



static gboolean
edit_keyring_item_closed_callback (GtkWidget *widget, GdkEvent *event G_GNUC_UNUSED, SaveItemCallbackData *data)
{
  if (data->attribute_editor->attributes_modified == TRUE)
    {
      save_attributes (data);
    }
  if (data->attribute_editor->acl_modified == TRUE)
    {
      save_acl (data);
    }


  gtk_widget_destroy (widget);

  return FALSE;
}




static void
edit_keyring_item_callback (GtkWidget        *button G_GNUC_UNUSED,
   	                    GKMKeyringEditor *editor G_GNUC_UNUSED)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  g_return_if_fail (GKM_IS_KEYRING_EDITOR (editor));

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (editor->priv->items_tree_view));
  
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      GtkTreePath *tmp_path;
      SaveItemCallbackData *data;
      GtkWidget *attribute_editor;
      guint item_id;
      char *item_name;

      gtk_tree_model_get (model, &iter, ITEM_COLUMN_ID, &item_id, -1);
      gtk_tree_model_get (model, &iter, ITEM_COLUMN_NAME, &item_name, -1);

      attribute_editor = gkm_attribute_editor_new (editor->priv->keyring_name, item_id, item_name);

      data = g_new0 (SaveItemCallbackData, 1);
      data->editor = editor;
      data->attribute_editor = GKM_ATTRIBUTE_EDITOR (attribute_editor);
      data->item_id = item_id;
      tmp_path = gtk_tree_model_get_path (model, &iter);
      data->row = gtk_tree_row_reference_new (model, tmp_path);
      gtk_tree_path_free (tmp_path);

      g_signal_connect (attribute_editor, "delete-event", G_CALLBACK (edit_keyring_item_closed_callback), data);

      gtk_widget_show (GTK_WIDGET (attribute_editor));
      g_free (item_name); 
    }
}

static void
drag_keyring_item_secret_data_get_callback (GtkWidget *widget G_GNUC_UNUSED,
					    GdkDragContext *context G_GNUC_UNUSED,
					    GtkSelectionData *selection_data,
					    guint info,
					    guint32 time G_GNUC_UNUSED,
					    gpointer callback_data)
{
  const gchar *secret;
  GKMKeyringEditor *editor;

  editor = GKM_KEYRING_EDITOR (callback_data);

  g_assert (selection_data != NULL);

  secret = gkm_keyring_editor_get_secret (editor);
  
  switch (info)
    {
      case GKM_DND_TEXT_PLAIN:
      case GKM_DND_UTF8_STRING:
      case GKM_DND_COMPOUND_TEXT:
      case GKM_DND_TEXT:
      case GKM_DND_STRING:
        gtk_selection_data_set (selection_data,
                                selection_data->target,
                                8, (guchar *) secret,
                                strlen (secret));
        break;
      default:
        g_assert_not_reached ();
    }
}

/********************************************************
 * DnD of keyring items between keyrings.
 */

static void
drag_keyring_item_data_get_callback (GtkWidget        *tree_view G_GNUC_UNUSED,
				     GdkDragContext   *context G_GNUC_UNUSED,
				     GtkSelectionData *selection_data,
				     guint             info,
				     guint32           time G_GNUC_UNUSED,
				     GKMKeyringEditor *editor)
{
  g_return_if_fail (GKM_IS_KEYRING_EDITOR (editor));
  g_assert (selection_data != NULL);
  g_assert (info == GKM_DND_KEYRING_ITEM);

  gtk_selection_data_set (selection_data,
			  selection_data->target,
			  32,
			  (gpointer) &editor, 4);
}

typedef struct _GKMDndItemData
{
  GKMKeyringEditor *editor;
  GKMKeyringEditor *source_editor;
  guint32 item_id;
  gchar *keyring_name;
  GnomeKeyringItemInfo *info;
} GKMDndItemData;

static GKMDndItemData *
gkm_dnd_item_data_clone (GKMDndItemData *data)
{
  GKMDndItemData *new_data;

  new_data = g_new0 (GKMDndItemData, 1);

  new_data->editor = data->editor;
  new_data->source_editor = data->source_editor;
  new_data->item_id = data->item_id;

  if (data->info != NULL)
    {
      new_data->info = gnome_keyring_item_info_copy (data->info);
    }

  return new_data;
}

static void
gkm_dnd_item_data_free (GKMDndItemData *data)
{
  gnome_keyring_item_info_free (data->info);

  g_free (data);
}

static void
dnd_item_delete_callback (GnomeKeyringResult result,
			  GKMKeyringEditor *editor)
{
  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (GTK_WINDOW (editor), result);
      return;
    }

  gkm_keyring_editor_update_keyring_items (editor);
}

static void
dnd_item_create_callback (GnomeKeyringResult  result,
			  guint32             item_id G_GNUC_UNUSED,
			  GKMDndItemData     *data)
{
  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (GTK_WINDOW (data->editor), result);
      return;
    }

  gkm_keyring_editor_update_keyring_items (data->editor);

  gnome_keyring_item_delete (data->source_editor->priv->keyring_name, 
			     data->item_id,
			     (GnomeKeyringOperationDoneCallback) dnd_item_delete_callback,
			     data->source_editor, NULL);
}

static void
dnd_item_get_attributes_callback (GnomeKeyringResult         result,
				  GnomeKeyringAttributeList *attributes,
				  GKMDndItemData            *data)
{
  gchar *item_name;
  gchar *secret;
  GnomeKeyringItemType type;

  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (GTK_WINDOW (data->editor), result);
      return;
    } 
 
  type = gnome_keyring_item_info_get_type (data->info);
  item_name = gnome_keyring_item_info_get_display_name (data->info);
  secret = gnome_keyring_item_info_get_secret (data->info);

  gnome_keyring_item_create (data->editor->priv->keyring_name,
			     type,
			     item_name,
			     attributes,
			     secret,
			     FALSE,
			     (GnomeKeyringOperationGetIntCallback) dnd_item_create_callback,
			     gkm_dnd_item_data_clone (data), 
			     (GDestroyNotify) gkm_dnd_item_data_free);
}

static void
dnd_item_get_info_callback (GnomeKeyringResult    result,
			    GnomeKeyringItemInfo *info,
			    GKMDndItemData       *data)
{
  GKMDndItemData *new_data;

  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (GTK_WINDOW (data->editor), result);
      return;
    } 

  new_data = gkm_dnd_item_data_clone (data);

  new_data->info = gnome_keyring_item_info_copy (info);

  gnome_keyring_item_get_attributes (data->source_editor->priv->keyring_name,
				     data->item_id,
				     (GnomeKeyringOperationGetAttributesCallback) dnd_item_get_attributes_callback,
				     new_data,
				     (GDestroyNotify) gkm_dnd_item_data_free);
}

static void
drop_keyring_item_data_recieved_callback (GtkWidget        *widget G_GNUC_UNUSED,
					  GdkDragContext   *context, 
					  gint              x G_GNUC_UNUSED, 
					  gint              y G_GNUC_UNUSED, 
					  GtkSelectionData *selection_data, 
					  guint             info, 
					  guint32           time, 
					  GKMKeyringEditor *editor)
{
  GtkTreeSelection *selection;
  GtkTreeIter       iter;
  GtkTreeModel     *model;
  GKMDndItemData   *data;
  GKMKeyringEditor *source;

  g_return_if_fail (GKM_IS_KEYRING_EDITOR (editor));

  g_return_if_fail (info == GKM_DND_KEYRING_ITEM);

  source = GKM_KEYRING_EDITOR (* (gint *) selection_data->data);

  g_return_if_fail (GKM_IS_KEYRING_EDITOR (source));

  if (strcmp (source->priv->keyring_name, editor->priv->keyring_name) == 0)
    {
      gdk_drag_status (context, 0, time);

      return;
    }

  data = g_new0 (GKMDndItemData, 1);
  data->editor = editor;
  data->source_editor = source;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (source->priv->items_tree_view));

  g_return_if_fail (selection != NULL);

  gtk_tree_selection_get_selected (selection, &model, &iter);
  gtk_tree_model_get (model, &iter, ITEM_COLUMN_ID, &data->item_id, -1);

  gnome_keyring_item_get_info (source->priv->keyring_name, 
			       data->item_id,
			       (GnomeKeyringOperationGetItemInfoCallback) dnd_item_get_info_callback,
			       data,
			       (GDestroyNotify) gkm_dnd_item_data_free);
}

/********************************************************
 * Delete a keyring item
 */

typedef struct _DeleteItemCallbackData
{
  GKMKeyringEditor *editor;
  GtkTreeIter	    iter;
  GtkTreeSelection  *selection;
  guint32	    item_id;
} DeleteItemCallbackData;

static void
delete_item_callback_data_free (DeleteItemCallbackData *data)
{
  g_free (data);
}

static void
delete_item_done_callback (GnomeKeyringResult        result,
			   DeleteItemCallbackData   *data)
{
  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (GTK_WINDOW (data->editor), result);
      return;
    }

  if (gtk_list_store_remove (data->editor->priv->keyring_items, &data->iter))
    {
      gtk_tree_selection_select_iter (data->selection, &data->iter);
    }
  else
    {
      if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (data->editor->priv->keyring_items), &data->iter))
        {
          gtk_tree_selection_select_iter (data->selection, &data->iter);
        }
    }
}

static void
delete_keyring_item_dialog_callback (GtkWidget *dialog,
				     GtkResponseType response,
				     DeleteItemCallbackData *data)
{
  if (response == GTK_RESPONSE_ACCEPT)
    {
      gnome_keyring_item_delete (data->editor->priv->keyring_name,
      			         data->item_id,
		    	         (GnomeKeyringOperationDoneCallback) delete_item_done_callback,
	                         data,
			         (GDestroyNotify) delete_item_callback_data_free);
    }

  gtk_widget_destroy (dialog);
}

/* When we can delete Keyrings, we'll probably want to move this dialog to its own class */
static void
remove_keyring_item_callback (GtkWidget        *button G_GNUC_UNUSED,
 		              GKMKeyringEditor *editor)
{
  DeleteItemCallbackData *data;
  GtkTreeModel *model;

  g_return_if_fail (GKM_IS_KEYRING_EDITOR (editor));

  data = g_new0 (DeleteItemCallbackData, 1);
  data->editor = editor;

  data->selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (editor->priv->items_tree_view));
  
  if (gtk_tree_selection_get_selected (data->selection, &model, &data->iter))
    {
      GtkWidget *dialog;
      GtkWidget *label;
      GtkWidget *image;
      GtkWidget *hbox;
      gchar     *item_name;
      gchar     *text;
      gchar     *markup_text;
      gchar     *label_text;
      
      gtk_tree_model_get (model, &data->iter, ITEM_COLUMN_ID, &data->item_id, ITEM_COLUMN_NAME, &item_name, -1);

      dialog = gtk_dialog_new_with_buttons (NULL,
      					    GTK_WINDOW (editor),
					    GTK_DIALOG_MODAL | 
					    GTK_DIALOG_DESTROY_WITH_PARENT |
					    GTK_DIALOG_NO_SEPARATOR,
					    GTK_STOCK_CANCEL,
					    GTK_RESPONSE_REJECT,
					    _("Delete item"),
					    GTK_RESPONSE_ACCEPT,
					    NULL);
      
      gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_REJECT);

      gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);
      gtk_window_set_default_size (GTK_WINDOW (dialog), 300, -1);
      gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 12);
      gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);

      hbox = gtk_hbox_new (FALSE, 12);
      gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);    
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, FALSE, FALSE, 0);

      image = gtk_image_new_from_stock (GTK_STOCK_DIALOG_WARNING, GTK_ICON_SIZE_DIALOG);
      gtk_misc_set_alignment (GTK_MISC (image), 0.5, 0.0);
      gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);

      label = gtk_label_new (NULL);
      gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
      gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
      gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
      gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
      gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

      text = g_strdup_printf (_("Delete the '%s' keyring item?"), item_name);
      markup_text = g_strdup_printf ("<span weight='bold'>%s</span>\n\n", text);
      label_text = g_strconcat (markup_text, _("Deleting a keyring item cannot be undone."), NULL);

      gtk_label_set_markup (GTK_LABEL (label), label_text);

      g_free (item_name);
      g_free (text);
      g_free (markup_text);
      g_free (label_text);

      g_signal_connect (G_OBJECT (dialog), 
                        "response", 
		        G_CALLBACK (delete_keyring_item_dialog_callback), 
		        data);
     
      gtk_widget_show_all (dialog);
    }
  else
    {
     delete_item_callback_data_free (data); 
    }
}

/********************************************************
 * Toggle column display
 */

static void
toggle_view_column_action_callback (GtkToggleAction *toggle_action,
				    GtkWidget	    *widget G_GNUC_UNUSED)
{
  toggle_view_column_action (toggle_action, gconf_client);
}

/********************************************************
 * Show the About dialog
 */

static void
about_action_callback (GtkAction	*action G_GNUC_UNUSED,
		       GKMKeyringEditor *editor G_GNUC_UNUSED)
{
  gkm_application_open_about_dialog (GTK_WINDOW (editor));
}

/********************************************************
 * The Close menu
 */

static void
close_action_callback (GtkAction         *action G_GNUC_UNUSED,
                       GKMKeyringEditor *editor)
{
  g_return_if_fail (GKM_IS_KEYRING_EDITOR (editor));

  gtk_widget_destroy (GTK_WIDGET (editor));
}
 
/********************************************************
 * Change the keyring secret
 */

typedef struct _ChangeSecretCallbackData
{
  GKMKeyringEditor *editor;
  char *old_secret;
  char *new_secret;
  guint32 item_id;
  GtkTreeIter iter;
} ChangeSecretCallbackData;

static ChangeSecretCallbackData *
change_secret_callback_data_clone (ChangeSecretCallbackData *data)
{
  ChangeSecretCallbackData *data_new;

  data_new = g_new0 (ChangeSecretCallbackData, 1);

  data_new->editor = data->editor;
  data_new->item_id = data->item_id;
  data_new->old_secret = g_strdup (data->old_secret);
  data_new->new_secret = g_strdup (data->new_secret);
  data_new->iter = data->iter;

  return data_new;
}

static void
change_secret_callback_data_free (ChangeSecretCallbackData *data)
{
  g_free (data->old_secret);
  g_free (data->new_secret);
  g_free (data);
}

static void
change_secret_set_info_callback (GnomeKeyringResult        result,
				 ChangeSecretCallbackData *data)
{
  if (result != GNOME_KEYRING_RESULT_OK)
    {
      gtk_label_set_text (GTK_LABEL (data->editor->priv->secret_field), data->old_secret);
      gkm_keyring_editor_set_secret (data->editor, data->old_secret);
      
      complain_about_gnome_keyring_bad_result (GTK_WINDOW (data->editor), result);
      return;
    }

   gtk_list_store_set (data->editor->priv->keyring_items, &data->iter, ITEM_COLUMN_SECRET, data->new_secret, -1);
}

static void
change_secret_get_info_callback (GnomeKeyringResult        result,
			         GnomeKeyringItemInfo     *info,
			         ChangeSecretCallbackData *data)
{
  if (result != GNOME_KEYRING_RESULT_OK)
    {
      gkm_keyring_editor_set_secret (data->editor, data->old_secret);
 
      complain_about_gnome_keyring_bad_result (GTK_WINDOW (data->editor), result);
      return;
    } 

  gnome_keyring_item_info_set_secret (info, data->new_secret);

  gnome_keyring_item_set_info (data->editor->priv->keyring_name,
  			       data->item_id,
  			       info,
			       (GnomeKeyringOperationDoneCallback) change_secret_set_info_callback,
			       change_secret_callback_data_clone (data),
			       (GDestroyNotify) change_secret_callback_data_free);
}

static gboolean
keyring_item_secret_changed_callback (GtkWidget *entry G_GNUC_UNUSED,
				      GdkEventFocus *event G_GNUC_UNUSED,
				      GKMKeyringEditor *editor)
{
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;
  gchar            *new_secret;
  gchar	           *secret;
  guint32	    item_id;

  g_return_val_if_fail (GKM_IS_KEYRING_EDITOR (editor), FALSE);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (editor->priv->items_tree_view));
  
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      gtk_tree_model_get (model, &iter, ITEM_COLUMN_ID, &item_id, ITEM_COLUMN_SECRET, &secret, -1);
      new_secret = gkm_keyring_editor_get_secret (editor);

      if (strcmp (new_secret, secret) != 0)
        {
        ChangeSecretCallbackData *data;

        data = g_new0 (ChangeSecretCallbackData, 1);
	data->editor = editor;
	data->item_id = item_id;
	data->old_secret = g_strdup (secret);
	data->new_secret = new_secret;
	data->iter = iter;
	
	gnome_keyring_item_get_info (editor->priv->keyring_name,
				     item_id,
				     (GnomeKeyringOperationGetItemInfoCallback) change_secret_get_info_callback,
				     data, 
				     (GDestroyNotify) change_secret_callback_data_free); 
	}

      g_free (secret);
    }
  
  return FALSE;
}

/********************************************************
 * Display the keyring items context menu
 */

static void
keyring_items_tree_popup_menu (GtkWidget        *tree_view,
			       GdkEventButton   *event,
			       GKMKeyringEditor *editor)
{
  GtkWidget *context_menu;
  
  g_return_if_fail (GTK_IS_TREE_VIEW (tree_view));
  g_return_if_fail (GKM_IS_KEYRING_EDITOR (editor));

  context_menu = gtk_ui_manager_get_widget (editor->priv->ui_manager, "/KeyringItemContextMenu");
  gtk_menu_popup (GTK_MENU (context_menu), NULL, NULL, NULL, NULL,
  		  (event != NULL) ? event->button : 0,
		  gdk_event_get_time ( (GdkEvent *) event));
}

static void 
keyring_items_tree_context_menu_callback (GtkWidget *tree_view, 
                                          GKMKeyringEditor *editor)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));

  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      keyring_items_tree_popup_menu (tree_view, NULL, editor);
    }
}

static gboolean 
keyring_items_tree_button_press_callback (GtkWidget        *tree_view,
					  GdkEventButton   *event,
					  GKMKeyringEditor *editor)
{
  if (event->type == GDK_BUTTON_PRESS && event->button == 3)
    {
       GtkTreeSelection *selection;
       GtkTreePath *path;

       selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));

       /* Only bring up the menu if there is a row under the mouse. */ 
       if (gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (tree_view),
                                          event->x, event->y,
                                          &path, NULL, NULL, NULL))
         {
           GtkTreeIter iter;

           gtk_tree_selection_unselect_all (selection);
           gtk_tree_selection_select_path (selection, path);
           gtk_tree_model_get_iter (GTK_TREE_MODEL (editor->priv->keyring_items), &iter, path);
           gtk_tree_path_free (path);

	   keyring_items_tree_popup_menu (tree_view, event, editor);
	 }

      return TRUE;
    }
  else if (event->type == GDK_2BUTTON_PRESS && event->button == 1)
    {
      edit_keyring_item_callback (NULL, editor);

      return TRUE;
    }
  else
    {
      return FALSE;
    }
}

/********************************************************
 * Keyring item selection callback
 */

typedef struct _ItemTreeSelectionCallbackData
{
  GKMKeyringEditor *editor;
  GnomeKeyringItemType item_type;
} ItemTreeSelectionCallbackData;

static void
change_displayed_attributes_callback (GnomeKeyringResult             result,
				      GnomeKeyringAttributeList     *attributes,
				      ItemTreeSelectionCallbackData *data)
{
  if (result != GNOME_KEYRING_RESULT_OK)
    {
      complain_about_gnome_keyring_bad_result (GTK_WINDOW (data->editor), result);
      return;
    }

  gkm_attribute_display_set (GKM_ATTRIBUTE_DISPLAY (data->editor->priv->attributes_display), data->item_type, attributes);
}

static void
keyring_item_tree_selection_changed (GtkTreeSelection  *selection,
                                     GKMKeyringEditor  *editor)
{
  GtkTreeModel     *model;
  GtkTreeIter       iter;
  char             *secret;
  guint32	    item_id;
  ItemTreeSelectionCallbackData *data;

  g_return_if_fail (GKM_IS_KEYRING_EDITOR (editor));
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      keyring_editor_update_sensitivity (editor, TRUE);

      data =  g_new0 (ItemTreeSelectionCallbackData, 1);
      data->editor = editor;

      gtk_tree_model_get (model, &iter, ITEM_COLUMN_ID, &item_id, ITEM_COLUMN_TYPE, &data->item_type, ITEM_COLUMN_SECRET, &secret, -1);

      editor->priv->is_multiline_secret =  (data->item_type == GNOME_KEYRING_ITEM_NOTE) ? TRUE : FALSE;
      gkm_keyring_editor_show_secret_widget (editor);
      gkm_keyring_editor_set_secret (editor, secret);

      gnome_keyring_item_get_attributes (editor->priv->keyring_name, 
				     item_id, 
				     (GnomeKeyringOperationGetAttributesCallback) change_displayed_attributes_callback,
				     data, g_free);
    }
  else
    {
      keyring_editor_update_sensitivity (editor, FALSE);
      gkm_keyring_editor_set_secret (editor, "");
    }
}

static void
keyring_editor_update_sensitivity (GKMKeyringEditor *editor, 
                                   gboolean sensitive)
{
  gtk_widget_set_sensitive (editor->priv->delete_button, sensitive);
  gtk_widget_set_sensitive (editor->priv->edit_button, sensitive);
  gtk_widget_set_sensitive (editor->priv->secret_label, sensitive);
  gtk_widget_set_sensitive (editor->priv->secret_event_box, sensitive);
  gtk_widget_set_sensitive (editor->priv->secret_dnd_label, sensitive);
  gtk_widget_set_sensitive (editor->priv->secret_expander, sensitive);
}

/********************************************************
 * Search for a keyring item
 */

typedef struct _SearchKeyringItemCallbackData
{
  guint32     item_id;
  GtkTreeIter iter;
  gboolean    found;
} SearchKeyringItemCallbackData;

static gboolean
search_keyring_item_worker (GtkTreeModel                  *model,
                            GtkTreePath                   *path G_GNUC_UNUSED,
                            GtkTreeIter                   *iter,
                            SearchKeyringItemCallbackData *data)
{
  guint32 item_id;
 
  g_assert (data != NULL);

  gtk_tree_model_get (model, iter, ITEM_COLUMN_ID, &item_id, -1);

  if (item_id == data->item_id) 
    {
      data->iter = *iter;
      data->found = TRUE;
    }

  return data->found;
}

/**
 * search_keyring_item:
 * @editor: a #GKMKeyringEditor.
 * @item_id: id of the keyring item to look for.
 * @iter: a pointer to a #GtkTreeIter or %NULL.
 * @path: a pointer to a pointer to a #GtkTreePath or %NULL.
 *
 * Looks for @item_id in the keyring_items #GtkTreeModel, and if it is found,
 * fills those of @iter, @path, and @row which are not %NULL so that 
 * they point to it.
 *
 * Returns: %TRUE if @item_id was found; %FALSE otherwise.
 **/

static gboolean
search_keyring_item (GKMKeyringEditor	  *editor,
		     guint32               item_id, 
                     GtkTreeIter          *iter, 
                     GtkTreePath         **path,
                     GtkTreeRowReference **row)
{
  SearchKeyringItemCallbackData data;

  data.item_id = item_id;
  data.found = FALSE;

  gtk_tree_model_foreach (GTK_TREE_MODEL (editor->priv->keyring_items), (GtkTreeModelForeachFunc) search_keyring_item_worker, &data);

  if (data.found)
    {
      if (iter != NULL)
        {
          *iter = data.iter;
        }
      if (path != NULL)
        {
          *path = gtk_tree_model_get_path (GTK_TREE_MODEL (editor->priv->keyring_items), &data.iter);
        }
      if (row != NULL)
        {
          if (path != NULL)
            {
              *row = gtk_tree_row_reference_new (GTK_TREE_MODEL (editor->priv->keyring_items), *path);
            }
          else
            {
              GtkTreePath *tmp_path;

              tmp_path = gtk_tree_model_get_path (GTK_TREE_MODEL (editor->priv->keyring_items), &data.iter);

              *row = gtk_tree_row_reference_new (GTK_TREE_MODEL (editor->priv->keyring_items), tmp_path);
              gtk_tree_path_free (tmp_path);
            }
        }
    }

  return data.found;
}

/********************************************************
 * Update info on the keyring items.
 */

static void
update_keyring_items_worker_callback (GnomeKeyringResult  result, 
                                      GList              *list, 
                                      GKMKeyringEditor   *editor)
{
  GList *tmp;

  g_return_if_fail (GKM_IS_KEYRING_EDITOR (editor));

  if (result != GNOME_KEYRING_RESULT_OK)
    {
      /* FIXME: same as update_keyrings_worker_callback */
      complain_about_gnome_keyring_bad_result (NULL, result);

      return;
    }
       
  gtk_list_store_clear (editor->priv->keyring_items);

  for (tmp = list; tmp != NULL; tmp = tmp->next)
    {
      GtkTreeIter iter;

      gtk_list_store_append (editor->priv->keyring_items, &iter);
      gtk_list_store_set (editor->priv->keyring_items, &iter, ITEM_COLUMN_ID, tmp->data, -1);

      gkm_keyring_editor_update_keyring_item_info (editor, (guint32) tmp->data, &iter);
    }
}

static void
gkm_keyring_editor_update_keyring_items (GKMKeyringEditor *transient_parent)
{
  gnome_keyring_list_item_ids (transient_parent->priv->keyring_name,
  			       (GnomeKeyringOperationGetListCallback) update_keyring_items_worker_callback, 
                               transient_parent, NULL);
}
 
/********************************************************
 * Update info on a keyring item.
 */

typedef struct _UpdateKeyringItemCallbackData
{
  GKMKeyringEditor *editor;
  GtkTreeRowReference *row;
} UpdateKeyringItemCallbackData;

static void
update_keyring_item_callback_data_free (UpdateKeyringItemCallbackData *data)
{
  gtk_tree_row_reference_free (data->row);
  g_free (data);
}

static void
update_keyring_item_info_worker_callback (GnomeKeyringResult             result,
                                          GnomeKeyringItemInfo          *info,
                                          UpdateKeyringItemCallbackData *data)
{
  char *value;
  time_t time;
  GtkTreePath *path;
  GtkTreeIter iter;
  GnomeKeyringItemType type;
  
  if (result != GNOME_KEYRING_RESULT_OK) 
    {
      g_warning (_("Failed to get keyring info."));

      return;
    }

  if (!gtk_tree_row_reference_valid (data->row))
    {
      g_warning (_("A row disapeared while we were waiting for the data..."));

      return;
    }

  path = gtk_tree_row_reference_get_path (data->row);
  gtk_tree_model_get_iter (GTK_TREE_MODEL (data->editor->priv->keyring_items), &iter, path);
  gtk_tree_path_free (path);
  
  value = gnome_keyring_item_info_get_display_name (info);
  gtk_list_store_set (data->editor->priv->keyring_items, &iter, ITEM_COLUMN_NAME, value, -1);
  g_free (value);

  type = gnome_keyring_item_info_get_type (info);
  gtk_list_store_set (data->editor->priv->keyring_items, &iter, ITEM_COLUMN_TYPE, type, -1);

  value = gnome_keyring_item_info_get_secret (info);
  gtk_list_store_set (data->editor->priv->keyring_items, &iter, ITEM_COLUMN_SECRET, value, -1);
  g_free (value);

  time = gnome_keyring_item_info_get_ctime (info);
  gtk_list_store_set (data->editor->priv->keyring_items, &iter, ITEM_COLUMN_CTIME, time, -1);

  time = gnome_keyring_item_info_get_mtime (info);
  gtk_list_store_set (data->editor->priv->keyring_items, &iter, ITEM_COLUMN_MTIME, time, -1);
}

/**
 * gkm_keyring_editor_update_keyring_item_info:
 * @editor: a #GKMKeyringEditor.
 * @item_id: the id of the item whose info is to be updated.
 * @iter: a #GtkTreeIter pointing to @keyring in the keyrings @GtkListStore, 
 * or %NULL.
 *
 * Update the information on @keyring in the keyrings liststore.
 **/

static void
gkm_keyring_editor_update_keyring_item_info (GKMKeyringEditor *editor, 
		 			     guint32           item_id,
					     GtkTreeIter      *iter)
{
  UpdateKeyringItemCallbackData *data;

  data = g_new0 (UpdateKeyringItemCallbackData, 1);
  data->editor = editor;

  if (iter == NULL)
    {
      if (!search_keyring_item (editor, item_id, NULL, NULL, &data->row))
        {
          g_warning ("gkm_keyring_editor_update_keyring_item_info: Tried to update the information of a keyring item we don't know about");
        }
    }
  else
    {
      GtkTreePath *path;

      path = gtk_tree_model_get_path (GTK_TREE_MODEL (editor->priv->keyring_items), iter);
      data->row = gtk_tree_row_reference_new (GTK_TREE_MODEL (editor->priv->keyring_items), path);
      gtk_tree_path_free (path);
    }
  
  gnome_keyring_item_get_info (editor->priv->keyring_name,
  			       item_id,
                               (GnomeKeyringOperationGetItemInfoCallback) update_keyring_item_info_worker_callback, 
                               data, (GDestroyNotify) update_keyring_item_callback_data_free);
}
