/* 
   Copyright (C) 2004 James Bowes <bowes@cs.dal.ca>
   Copyright (C) 2004 GNOME Love Project <gnome-love@gnome.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glade/glade.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <gnome-keyring.h>

#include <stdlib.h>
#include <string.h>

#include "gnome-keyring-manager-i18n.h"
#include "gnome-keyring-manager-util.h"
#include "gnome-keyring-manager-attribute-display.h"

#define GLADE_ROOT "attributes-display_table"

struct _GKMAttributeDisplayPrivate
{
  GnomeKeyringItemType item_type;
  GnomeKeyringAttributeList *attributes;

  gboolean secret_visible;

  GtkWidget **attribute_entries;
  GladeXML   *xml;
};

enum
{
  USER_ENTRY = 0,
  DOMAIN_ENTRY,
  SERVER_ENTRY,
  OBJECT_ENTRY,
  METHOD_ENTRY,
  PROTOCOL_ENTRY,
  PORT_ENTRY,
  PASSWORD_ENTRY,
  NUM_ENTRIES
};

enum
{
  BLANK_PAGE = 0,
  SECRET_PAGE,
  NETWORK_PAGE,
  NOTE_PAGE,
  NUM_PAGES
};

static GtkWidget *generic_secret_contruct_page (GtkWidget **labels);
static gboolean   generic_secret_update_page (GKMAttributeDisplay *display);
static GtkWidget *network_password_construct_page (GtkWidget **labels);
static gboolean   network_password_update_page (GKMAttributeDisplay *display);
static GtkWidget *note_construct_page (GtkWidget **labels);
static gboolean   note_update_page (GKMAttributeDisplay *display);

void gkm_attribute_display_clear (GKMAttributeDisplay *display);

static void gkm_attribute_display_show_password_toggle (GtkWidget           *widget,
							GKMAttributeDisplay *self);

static void gkm_attribute_display_class_init (GKMAttributeDisplay *class);
static void gkm_attribute_display_init       (GKMAttributeDisplay *display);
static void gkm_attribute_display_finalize   (GObject *object);
static void gkm_attribute_display_destroy    (GtkObject *object);

static GtkBinClass *parent_class = NULL;

GType
gkm_attribute_display_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo info =
      {
        sizeof (GKMAttributeDisplayClass),
	NULL,		/* base_init */
	NULL,		/* base_finalize */
        (GClassInitFunc) gkm_attribute_display_class_init,
	NULL,		/* class_finalize */
	NULL,		/* class_data */ 
	sizeof (GKMAttributeDisplay),
	0,		/* n_preallocs */
	(GInstanceInitFunc) gkm_attribute_display_init,
	0
      };

      type = g_type_register_static (GTK_TYPE_VBOX, "GKMAttributeDisplay", &info, 0);
    }

  return type;
}

static void
gkm_attribute_display_class_init (GKMAttributeDisplay *class)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (class);
  GtkObjectClass *object_class = GTK_OBJECT_CLASS (class);

  parent_class = g_type_class_peek_parent (class);

  gobject_class->finalize = gkm_attribute_display_finalize;

  object_class->destroy = gkm_attribute_display_destroy;
}

static void
gkm_attribute_display_init (GKMAttributeDisplay *self)
{
  GtkWidget *widget;

  self->priv = g_new0 (GKMAttributeDisplayPrivate, 1);

  self->priv->attribute_entries = g_new0 (GtkWidget *, NUM_ENTRIES);

  self->priv->xml = glade_xml_new (GLADEDIR GLADEFILE,
				   GLADE_ROOT, NULL);

  if(!self->priv->xml) 
    {
      g_warning ("Unable to load the glade file. Your install is broken.");
      exit (1);
    }

  self->priv->attributes = gnome_keyring_attribute_list_new ();

  self->priv->secret_visible = FALSE;

  widget = glade_xml_get_widget (self->priv->xml, GLADE_ROOT);

  gtk_container_add (GTK_CONTAINER (self), widget);

  gtk_widget_show_all (GTK_WIDGET (self));

  /* Get refs for the attribute entries */
  
  widget = glade_xml_get_widget (self->priv->xml, "server_entry");
  self->priv->attribute_entries[SERVER_ENTRY] = widget;

  widget = glade_xml_get_widget (self->priv->xml, "username_entry");
  self->priv->attribute_entries[USER_ENTRY] = widget;

  widget = glade_xml_get_widget (self->priv->xml, "password_entry");
  self->priv->attribute_entries[PASSWORD_ENTRY] = widget;

  widget = glade_xml_get_widget (self->priv->xml, "port_spinbutton");
  self->priv->attribute_entries[PORT_ENTRY] = widget;

  widget = glade_xml_get_widget (self->priv->xml, "protocol_entry");
  self->priv->attribute_entries[PROTOCOL_ENTRY] = widget;

  widget = glade_xml_get_widget (self->priv->xml, "method_entry");
  self->priv->attribute_entries[METHOD_ENTRY] = widget;

  widget = glade_xml_get_widget (self->priv->xml, "domain_entry");
  self->priv->attribute_entries[DOMAIN_ENTRY] = widget;

  widget = glade_xml_get_widget (self->priv->xml, "object_entry");
  self->priv->attribute_entries[OBJECT_ENTRY] = widget;

  widget = glade_xml_get_widget (self->priv->xml, "password_checkbutton");

  g_signal_connect (widget,
		    "toggled",
		    G_CALLBACK (gkm_attribute_display_show_password_toggle),
		    self);
}

static void
gkm_attribute_display_finalize (GObject *object)
{
  GKMAttributeDisplay *display;

  g_return_if_fail (GKM_IS_ATTRIBUTE_DISPLAY (object));

  display = GKM_ATTRIBUTE_DISPLAY (object);

  gnome_keyring_attribute_list_free (display->priv->attributes);
  g_object_unref (display->priv->xml);
  g_free (display->priv);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gkm_attribute_display_destroy (GtkObject *object)
{
  GKMAttributeDisplay *display;

  g_return_if_fail (GKM_IS_ATTRIBUTE_DISPLAY (object));

  display = GKM_ATTRIBUTE_DISPLAY (object);

  GTK_OBJECT_CLASS (parent_class)->destroy (GTK_OBJECT (display));
}

GtkWidget *
gkm_attribute_display_new (void)
{
  GKMAttributeDisplay *display;

  display = g_object_new (GKM_TYPE_ATTRIBUTE_DISPLAY, NULL);

  return GTK_WIDGET (display);
}

/**************************************************
 * Generic secrets
 */

/* FIXME: What are the standard attributes for this? Does it have any? */
/* FIXME: These need to do something useful */

static gboolean
generic_secret_update_page (GKMAttributeDisplay *self)
{
  if (self->priv->attributes->len > 0)
    {
      return TRUE;
    }
  else
    {
      return FALSE;
    }
}

/**************************************************
 * Network passwords
 */

static gboolean
network_password_update_page (GKMAttributeDisplay *self)
{
  guint32 i;
  gboolean non_standard_attributes;
 
  non_standard_attributes = FALSE;
  for (i = 0; i < self->priv->attributes->len; i++)
    {
      GnomeKeyringAttribute attribute;
      
      attribute = gnome_keyring_attribute_list_index(self->priv->attributes, i);

      if (strcmp (attribute.name, "user") == 0 
	  && attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_STRING)
	{
	  gtk_entry_set_text (GTK_ENTRY (self->priv->attribute_entries[USER_ENTRY]), 
			      attribute.value.string);
	}
      else if (strcmp (attribute.name, "domain") == 0 
	       && attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_STRING)
	{
	  gtk_entry_set_text (GTK_ENTRY (self->priv->attribute_entries[DOMAIN_ENTRY]),
			      attribute.value.string);
	}
      else if (strcmp (attribute.name, "server") == 0 
	       && attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_STRING)
	{
	  gtk_entry_set_text (GTK_ENTRY (self->priv->attribute_entries[SERVER_ENTRY]),
			      attribute.value.string);
	}
      else if (strcmp (attribute.name, "object") == 0 
	       && attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_STRING)
	{
	  gtk_entry_set_text (GTK_ENTRY (self->priv->attribute_entries[OBJECT_ENTRY]),
			      attribute.value.string);
	}
      else if (strcmp (attribute.name, "authtype") == 0 
	       && attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_STRING)
	{
	  gtk_entry_set_text (GTK_ENTRY (self->priv->attribute_entries[METHOD_ENTRY]),
			      attribute.value.string);
	}
      else if (strcmp (attribute.name, "protocol") == 0 
	       && attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_STRING)
	{
	  gtk_entry_set_text (GTK_ENTRY (self->priv->attribute_entries[PROTOCOL_ENTRY]), 
			      attribute.value.string);
	}
      else if (strcmp (attribute.name, "port") == 0 
	       && attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_UINT32)
	{
 	  gtk_spin_button_set_value (GTK_SPIN_BUTTON (self->priv->attribute_entries[PORT_ENTRY]), 
				     attribute.value.integer);
	}
      else
        {
          non_standard_attributes = TRUE;
	}
    }

  return non_standard_attributes;
}

/**************************************************
 * Notes
 */

static gboolean
note_update_page (GKMAttributeDisplay *self)
{
  if (self->priv->attributes->len > 0)
    {
      return TRUE;
    }
  else
    {
      return FALSE;
    }
}

static void
update_attribute_display (GKMAttributeDisplay *self)
{
  gboolean non_standard_attributes;
  int page;

  g_return_if_fail (self->priv->attributes != NULL);
  
  gkm_attribute_display_clear (self);
  /* gtk_widget_hide (self->priv->unknown_attributes_warning); */

  switch (self->priv->item_type)
    {
      case GNOME_KEYRING_ITEM_GENERIC_SECRET:
        non_standard_attributes = generic_secret_update_page (self);
        page = SECRET_PAGE;
        break;
      case GNOME_KEYRING_ITEM_NETWORK_PASSWORD:
        non_standard_attributes = network_password_update_page (self);
        page = NETWORK_PAGE;
        break;
      case GNOME_KEYRING_ITEM_NOTE:
        non_standard_attributes = note_update_page (self);
        page = NOTE_PAGE;
	break;
      default:
        non_standard_attributes = TRUE;
	page = BLANK_PAGE;
    }

/*   gtk_notebook_set_current_page (GTK_NOTEBOOK (self->priv->attributes_book), page); */
/*   if (non_standard_attributes) */
/*     { */
/*       gtk_widget_show (self->priv->unknown_attributes_warning); */
/*     } */
}

/**************************************************
 * Get and set the type
 */

void
gkm_attribute_display_clear (GKMAttributeDisplay *self)
{
  int i;

  /* Don't clear out the password */
  for (i = 0; i < NUM_ENTRIES - 1; i++)
    {
      gtk_entry_set_text (GTK_ENTRY (self->priv->attribute_entries[i]), "");
    }
}

void 
gkm_attribute_display_set_secret (GKMAttributeDisplay *self, 
				  const gchar         *secret)
{
  g_return_if_fail (GKM_IS_ATTRIBUTE_DISPLAY (self));
  
  gtk_entry_set_text (GTK_ENTRY (self->priv->attribute_entries[PASSWORD_ENTRY]),
		      secret);
  
  update_attribute_display (self);
}

gchar *
gkm_attribute_display_get_secret (GKMAttributeDisplay *self)
{
  gchar *secret;
  GtkWidget *widget;

  g_return_val_if_fail (GKM_IS_ATTRIBUTE_DISPLAY (self), NULL);

  widget =  self->priv->attribute_entries[PASSWORD_ENTRY];
  
  secret = g_strdup (gtk_entry_get_text (GTK_ENTRY (widget)));

  return secret;
}

void 
gkm_attribute_display_set_item_type (GKMAttributeDisplay  *self, 
				     GnomeKeyringItemType  item_type)
{
  g_return_if_fail (GKM_IS_ATTRIBUTE_DISPLAY (self));

  self->priv->item_type = item_type;
  
  update_attribute_display (self);
}

GnomeKeyringItemType 
gkm_attribute_display_get_item_type (GKMAttributeDisplay *self)
{
  g_return_val_if_fail (GKM_IS_ATTRIBUTE_DISPLAY (self), GNOME_KEYRING_ITEM_NO_TYPE);

  return self->priv->item_type;
}

void
gkm_attribute_display_set_attributes (GKMAttributeDisplay       *self,
				      GnomeKeyringAttributeList *attributes)
{
  g_return_if_fail (GKM_IS_ATTRIBUTE_DISPLAY (self));


  gnome_keyring_attribute_list_free (self->priv->attributes);
  self->priv->attributes = gnome_keyring_attribute_list_copy (attributes);

  update_attribute_display (self);
}

GnomeKeyringAttributeList *
gkm_attribute_display_get_attributes (GKMAttributeDisplay *self)
{
  g_return_val_if_fail (GKM_IS_ATTRIBUTE_DISPLAY (self), NULL);

  return gnome_keyring_attribute_list_copy (self->priv->attributes);
}

static void
gkm_attribute_display_show_password_toggle (GtkWidget           *widget G_GNUC_UNUSED,
					    GKMAttributeDisplay *self)
{
  self->priv->secret_visible = !self->priv->secret_visible;

  gtk_entry_set_visibility (GTK_ENTRY (self->priv->attribute_entries[PASSWORD_ENTRY]),
			   self->priv->secret_visible);


  /* Don't let a user change the password if its not visible (they won't know what
     they're changing). */
  gtk_entry_set_editable (GTK_ENTRY (self->priv->attribute_entries[PASSWORD_ENTRY]),
			  self->priv->secret_visible);
}


/* Glade custom widget init function */
GtkWidget *
gkm_attribute_display_glade_new (gchar *widget_name G_GNUC_UNUSED, 
				 gchar *string1     G_GNUC_UNUSED, 
				 gchar *string2     G_GNUC_UNUSED,
				 gint   int1        G_GNUC_UNUSED, 
				 gint   int2        G_GNUC_UNUSED)
{
  return gkm_attribute_display_new ();
}
