/* Paned Widgets
 *
 * The GtkHPaned and GtkVPaned Widgets divide their content
 * area into two panes with a divider in between that the
 * user can adjust. A separate child is placed into each
 * pane.
 *
 * There are a number of options that can be set for each pane.
 * This test contains both a horizontal (HPaned) and a vertical
 * (VPaned) widget, and allows you to adjust the options for
 * each side of each widget.
 */

#include <gtk/gtk.h>
#include <gtkmm.h>

class Example_Panes : public Gtk::Window
{
public:
  Example_Panes();
  virtual ~Example_Panes();

protected:
  virtual Gtk::Frame* create_pane_options(Gtk::Paned& paned, const Glib::ustring& frame_label,
                                          const Glib::ustring& label1, const Glib::ustring& label2);

  //signal handlers:
  virtual void on_checkbutton_shrink(Gtk::Widget* child);
  virtual void on_checkbutton_resize(Gtk::Widget* child);

  //Member widgets:
  Gtk::Frame m_Frame1, m_Frame2, m_Frame3;
  Gtk::VBox m_VBox;
  Gtk::VPaned m_VPaned;
  Gtk::HPaned m_HPaned;
  Gtk::Button m_Button;
};

//Called by DemoWindow;
Gtk::Window* do_panes()
{
  return new Example_Panes();
}

Example_Panes::Example_Panes()
: m_Button("_Hi there", true)
{
  set_title("Panes");
  set_border_width(0);

  add(m_VBox);

  m_VBox.pack_start(m_VPaned);
  m_VPaned.set_border_width(5);

  m_VPaned.add1(m_HPaned);

  m_Frame1.set_shadow_type(Gtk::SHADOW_IN);
  m_Frame1.set_size_request(60, 60);
  m_HPaned.add1(m_Frame1);
  m_Frame1.add(m_Button);

  m_Frame2.set_shadow_type(Gtk::SHADOW_IN);
  m_Frame2.set_size_request(80, 60);
  m_HPaned.add2(m_Frame2);

  m_Frame3.set_shadow_type(Gtk::SHADOW_IN);
  m_Frame3.set_size_request(60, 80);
  m_VPaned.add2(m_Frame3);


  /* Now create toggle buttons to control sizing */
  m_VBox.pack_start(*(create_pane_options(m_HPaned, "Horizontal", "Left", "Right")), false, false);
	m_VBox.pack_start(*(create_pane_options(m_VPaned, "Vertical", "Top", "Bottom")), false, false);

  show_all();
}

Example_Panes::~Example_Panes()
{
}

Gtk::Frame*
Example_Panes::create_pane_options(Gtk::Paned& paned,
  const Glib::ustring& frame_label,
  const Glib::ustring& label1, const Glib::ustring& label2)
{
  Gtk::Frame* pFrame = Gtk::manage(new Gtk::Frame(frame_label));
  pFrame->set_border_width(4);

  Gtk::Table* pTable = Gtk::manage(new Gtk::Table(3, 2, true));
  pFrame->add(*pTable);

  Gtk::Label* pLabel = Gtk::manage(new Gtk::Label(label1));
  pTable->attach(*pLabel,  0, 1, 0, 1);

  Gtk::CheckButton* pCheckButtonResize = Gtk::manage(new Gtk::CheckButton("_Resize", true));
  pTable->attach(*pCheckButtonResize, 0, 1, 1, 2);
  pCheckButtonResize->signal_toggled().connect(
      SigC::bind(SigC::slot(*this, &Example_Panes::on_checkbutton_resize), paned.get_child1()));

  Gtk::CheckButton* pCheckButtonShrink = Gtk::manage(new Gtk::CheckButton("_Shrink", true));
  pTable->attach(*pCheckButtonShrink, 0, 1, 2, 3);
  pCheckButtonShrink->set_active(true);
  pCheckButtonShrink->signal_toggled().connect(
      SigC::bind(SigC::slot(*this, &Example_Panes::on_checkbutton_shrink), paned.get_child1()));

  Gtk::Label* pLabel2 = Gtk::manage(new Gtk::Label(label2));
  pTable->attach(*pLabel2, 1, 2, 0, 1);

  Gtk::CheckButton* pCheckButtonResize2 = Gtk::manage(new Gtk::CheckButton("_Resize", true));
  pTable->attach(*pCheckButtonResize2, 1, 2, 1, 2);
  pCheckButtonResize2->set_active(true);
  pCheckButtonResize2->signal_toggled().connect(
      SigC::bind(SigC::slot(*this, &Example_Panes::on_checkbutton_resize), paned.get_child2()));

  Gtk::CheckButton* pCheckButtonShrink2 = Gtk::manage(new Gtk::CheckButton("_Shrink", true));
  pTable->attach(*pCheckButtonShrink2, 1, 2, 2, 3);
  pCheckButtonShrink2->set_active(true);
  pCheckButtonShrink2->signal_toggled().connect(
      SigC::bind(SigC::slot(*this, &Example_Panes::on_checkbutton_shrink), paned.get_child2()));

  return pFrame;
}

void Example_Panes::on_checkbutton_resize(Gtk::Widget* pChild)
{
  Gtk::Paned* pPaned = dynamic_cast<Gtk::Paned*>(pChild->get_parent());
  if(pPaned)
  {
    bool is_child1 = (pChild == pPaned->get_child1());

    bool resize = is_child1 ? pPaned->get_child1_resize() : pPaned->get_child2_resize();
    bool shrink = is_child1 ? pPaned->get_child1_shrink() : pPaned->get_child2_shrink();

    Glib::RefPtr<Gtk::Widget> refChild(pChild); //Ensure that it stays alive while it's unparented
    refChild->reference();
    pPaned->remove(*pChild);
    if (is_child1)
      pPaned->pack1(*pChild, !resize, shrink);
    else
      pPaned->pack2(*pChild, !resize, shrink);
  }
}

void Example_Panes::on_checkbutton_shrink(Gtk::Widget* pChild)
{
  Gtk::Paned* pPaned = dynamic_cast<Gtk::Paned*>(pChild->get_parent());
  if(pPaned)
  {
    bool is_child1 = (pChild == pPaned->get_child1());

    bool resize = is_child1 ? pPaned->get_child1_resize() : pPaned->get_child2_resize();
    bool shrink = is_child1 ? pPaned->get_child1_shrink() : pPaned->get_child2_shrink();

    Glib::RefPtr<Gtk::Widget> refChild(pChild); //Ensure that it stays alive while it's unparented.
    refChild->reference();
    pPaned->remove(*pChild);
    if (is_child1)
      pPaned->pack1(*pChild, resize, !shrink);
    else
      pPaned->pack2(*pChild, resize, !shrink);
  }
}


