
#include <gtkmm/treeiter.h>
#include <gtkmm/private/treeiter_p.h>

// -*- c++ -*-
/* $Id: treeiter.ccg,v 1.4 2002/04/08 17:20:02 daniel Exp $ */

/*
 *
 * Copyright 1998-1999 The Gtk-- Development Team
 * Copyright 2001      Free Software Foundation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

namespace Gtk
{

/**** Gtk::TreeIter ********************************************************/

TreeIter::TreeIter()
:
  TreeIterBase(),
  model_      (0),
  is_end_     (false)
{}

TreeIter::TreeIter(GtkTreeModel* model)
:
  TreeIterBase(),
  model_      (model),
  is_end_     (false)
{}

TreeIter::TreeIter(GtkTreeModel* model, const GtkTreeIter* iter)
:
  TreeIterBase(iter),
  model_      (model),
  is_end_     (iter == 0)
{}

TreeIter& TreeIter::operator++()
{
  g_assert(!is_end_);

  GtkTreeIter previous = gobject_;

  if(!gtk_tree_model_iter_next(model_, &gobject_))
  {
    GLIBMM_INITIALIZE_STRUCT(gobject_, GtkTreeIter);
    is_end_ = true;
    gtk_tree_model_iter_parent(model_, &gobject_, &previous);
  }

  return *this;
}

const TreeIter TreeIter::operator++(int)
{
  g_assert(!is_end_);

  TreeIter previous (*this);

  if(!gtk_tree_model_iter_next(model_, &gobject_))
  {
    GLIBMM_INITIALIZE_STRUCT(gobject_, GtkTreeIter);
    is_end_ = true;
    gtk_tree_model_iter_parent(model_, &gobject_, &previous.gobject_);
  }

  return previous;
}

const TreeIterChildren& TreeIter::children() const
{
  g_assert(!is_end_);

  return static_cast<const TreeIterChildren&>(*this);
}

TreeIter TreeIter::parent() const
{
  TreeIter iter (model_);

  if(is_end_)
    iter.gobject_ = gobject_;
  else
    gtk_tree_model_iter_parent(model_, iter.gobj(), const_cast<GtkTreeIter*>(&gobject_));

  return iter;
}

bool TreeIter::equal(const TreeIter& other) const
{
  g_assert(model_ == other.model_);
  g_assert(gobject_.stamp == other.gobject_.stamp || is_end_ || other.is_end_);

  return (is_end_ == other.is_end_) &&
         (gobject_.user_data  == other.gobject_.user_data)  &&
         (gobject_.user_data2 == other.gobject_.user_data2) &&
         (gobject_.user_data3 == other.gobject_.user_data3);
}

TreeIter::operator bool() const
{
  return (!is_end_ && gobject_.stamp != 0);
}

void TreeIter::setup_end_iterator(const TreeIter& last_valid)
{
  g_assert(model_ == last_valid.model_);

  if(last_valid.is_end_)
  {
    gobject_ = last_valid.gobject_;
  }
  else
  {
    GLIBMM_INITIALIZE_STRUCT(gobject_, GtkTreeIter);
    gtk_tree_model_iter_parent(model_, &gobject_, const_cast<GtkTreeIter*>(&last_valid.gobject_));
  }

  is_end_ = true;
}


/**** Gtk::TreeIterChildren ************************************************/

TreeIter TreeIterChildren::begin() const
{
  TreeIter iter (model_);

  // If the iterator is invalid (stamp == 0), assume a 'virtual' toplevel
  // node.  This behaviour is needed to implement Gtk::TreeModel::children().

  if(gobject_.stamp != 0)
  {
    if(!gtk_tree_model_iter_children(model_, iter.gobj(), const_cast<GtkTreeIter*>(&gobject_)))
    {
      // Assign the already known parent, in order to create an end iterator.
      iter.gobject_ = gobject_;
      iter.is_end_ = true;
    }
  }
  else
  {
    if(!gtk_tree_model_get_iter_first(model_, iter.gobj()))
    {
      // No need to copy the GtkTreeIter, since iter.gobject_ is already empty.
      iter.is_end_ = true;
    }
  }

  return iter;
}

TreeIter TreeIterChildren::end() const
{
  // Just copy the parent, and turn it into an end iterator.
  TreeIter iter (*this);
  iter.is_end_ = true;
  return iter;
}

TreeIter TreeIterChildren::operator[](unsigned int index) const
{
  TreeIter iter (model_);

  GtkTreeIter *const parent =
      (gobject_.stamp != 0) ? const_cast<GtkTreeIter*>(&gobject_) : 0;

  if(!gtk_tree_model_iter_nth_child(model_, iter.gobj(), parent, index))
  {
    // Assign the already known parent, in order to create an end iterator.
    iter.gobject_ = gobject_;
    iter.is_end_ = true;
  }

  return iter;
}

unsigned int TreeIterChildren::size() const
{
  GtkTreeIter *const parent =
      (gobject_.stamp != 0) ? const_cast<GtkTreeIter*>(&gobject_) : 0;

  return gtk_tree_model_iter_n_children(model_, parent);
}

bool TreeIterChildren::empty() const
{
  // If the iterator is invalid (stamp == 0), assume a 'virtual' toplevel
  // node.  This behaviour is needed to implement Gtk::TreeModel::children().

  if(gobject_.stamp == 0)
  {
    GtkTreeIter dummy;
    return !gtk_tree_model_get_iter_first(model_, &dummy);
  }

  return !gtk_tree_model_iter_has_child(model_, const_cast<GtkTreeIter*>(&gobject_));
}

} // namespace Gtk


namespace
{
} // anonymous namespace


namespace Glib
{

Gtk::TreeIterBase& wrap(GtkTreeIter* object)
{
  return *reinterpret_cast<Gtk::TreeIterBase*>(object);
}

const Gtk::TreeIterBase& wrap(const GtkTreeIter* object)
{
  return *reinterpret_cast<const Gtk::TreeIterBase*>(object);
}

} // namespace Glib


namespace Gtk
{


// static
GType TreeIterBase::get_type()
{
  return gtk_tree_iter_get_type();
}

TreeIterBase::TreeIterBase()
{
  GLIBMM_INITIALIZE_STRUCT(gobject_, GtkTreeIter);
}

TreeIterBase::TreeIterBase(const GtkTreeIter* gobject)
{
  if(gobject)
    gobject_ = *gobject;
  else
    GLIBMM_INITIALIZE_STRUCT(gobject_, GtkTreeIter);
}


} // namespace Gtk

