// -*- c++ -*-
#ifndef _GLIBMM_REFPTR_H
#define _GLIBMM_REFPTR_H

/* $Id: refptr.h,v 1.27 2002/03/16 22:59:57 daniel Exp $ */

/* refptr.h
 *
 * Copyright 2001 Free Software Foundation
 * Copyright (C) 1998-2001 The Gtk-- Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


namespace Glib
{

/** RefPtr<> is a reference-counting shared smartpointer.
 *  See the "Memory Management" section in the "Programming with gtkmm" book.
 */
template<class T_CppObject>
class RefPtr
{
public:
  RefPtr(); // is_null() will be true. Use of -> will segfault.
  ~RefPtr();

  explicit RefPtr(T_CppObject* pCppObject);

  RefPtr(const RefPtr& src);

  template<class T_CastFrom>
  RefPtr(const RefPtr<T_CastFrom>& src);

  RefPtr& operator=(const RefPtr& src);
  RefPtr& operator=(T_CppObject* pCppObject); //Shouldn't be necessary - use Something::create() to get a RefPtr.

  template<class T_CastFrom>
  RefPtr& operator=(const RefPtr<T_CastFrom>& src);

  bool operator==(const RefPtr& src) const;
  bool operator!=(const RefPtr& src) const;

  T_CppObject* operator->() const;

  operator bool() const; // e.g. if(ref)
  bool is_null() const;  // whether there is no underlying instance

  void clear();

  ///e.g. ref_derived = RefPtr<Derived>::cast_dynamic(ref_base);
  template <class T_CastFrom>
  static inline RefPtr<T_CppObject> cast_dynamic(const RefPtr<T_CastFrom>& src);

  ///e.g. ref_derived = RefPtr<Derived>::cast_static(ref_base);
  template <class T_CastFrom>
  static inline RefPtr<T_CppObject> cast_static(const RefPtr<T_CastFrom>& src);

private:
  T_CppObject* pCppObject_;
};


// RefPtr<>::operator->() comes first here since it's used by other methods.
// If it would come after them it wouldn't be inlined.

template<class T_CppObject> inline
T_CppObject* RefPtr<T_CppObject>::operator->() const
{
  return pCppObject_;
}

template<class T_CppObject> inline
RefPtr<T_CppObject>::RefPtr()
:
  pCppObject_(0)
{}

template<class T_CppObject> inline
RefPtr<T_CppObject>::~RefPtr()
{
  if(pCppObject_)
    pCppObject_->unreference(); // This could cause pCppObject to be deleted.
}

template<class T_CppObject> inline
RefPtr<T_CppObject>::RefPtr(T_CppObject* pCppObject)
:
  pCppObject_(pCppObject)
{
}


template<class T_CppObject> inline
RefPtr<T_CppObject>::RefPtr(const RefPtr<T_CppObject>& src)
:
  pCppObject_(src.pCppObject_)
{
  if(pCppObject_)
    pCppObject_->reference();
}

// The templated ctor allows copy construction from any object that's
// castable.  Thus, it does downcasts:
//   base_ref = derived_ref
template<class T_CppObject>
  template<class T_CastFrom>
inline
RefPtr<T_CppObject>::RefPtr(const RefPtr<T_CastFrom>& src)
:
  // A different RefPtr<> will not allow us access to pCppObject_.  We need
  // to add a get_underlying() for this, but that would encourage incorrect
  // use, so we use the less well-known operator->() accessor:
  pCppObject_(src.operator->())
{
  if(pCppObject_)
    pCppObject_->reference();
}

template<class T_CppObject> inline
RefPtr<T_CppObject>&
RefPtr<T_CppObject>::operator=(const RefPtr<T_CppObject>& src)
{
  // Prevent self-assignment.
  if(pCppObject_ != src.pCppObject_)
  {

    // Unreference and forget about the previous instance.
    clear();

    // Get the other instance.
    pCppObject_ = src.pCppObject_;

    // Reference the new instance.
    if(pCppObject_)
    {
      pCppObject_->reference();
    }
  }

  return *this;
}

template<class T_CppObject> inline
RefPtr<T_CppObject>&
RefPtr<T_CppObject>::operator=(T_CppObject* pCppObject)
{
   // Prevent self-assignment.
  if(pCppObject_ != pCppObject)
  {
    // Unreference and forget about the previous instance.
    clear();
    
    // Get the other instance.
    pCppObject_ = pCppObject;

    // Reference the new instance.
    if(pCppObject_)
      pCppObject_->reference();
  }

  return *this;
}

/*
template<class T_CppObject> inline
RefPtr<T_CppObject>&
RefPtr<T_CppObject>::operator=(const RefPtr<T_CppObject>& src)
{
  RefPtr<T_CppObject> tmp(src);

  // Use swap technique for copy assignment.
  T_CppObject *const tp = pCppObject_;
  pCppObject_ = tmp.pCppObject_;
  tmp.pCppObject_ = tp;

  return *this;
}
*/

template<class T_CppObject>
  template<class T_CastFrom>
inline
RefPtr<T_CppObject>&
RefPtr<T_CppObject>::operator=(const RefPtr<T_CastFrom>& src)
{
  // A different RefPtr<> will not allow us access to pCppObject_.  We need
  // to add a get_underlying() for this, but that would encourage incorrect
  // use, so we use the less well-known operator->() accessor:

  // Prevent self-assignment.
  if(pCppObject_ != src.operator->())
  {

    // Unreference and forget about the previous instance.
    clear();

    // Get the other instance.
    pCppObject_ = src.operator->();

    // Reference the new instance.
    if(pCppObject_)
    {
      pCppObject_->reference();
    }
  }

  return *this;
}

/*
template<class T_CppObject>
  template<class T_RefPtrCastFrom>
inline
RefPtr<T_CppObject>&
RefPtr<T_CppObject>::operator=(const T_RefPtrCastFrom& src)
{
  RefPtr<T_CppObject> tmp(src);

  // Use swap technique for copy assignment.
  T_CppObject *const tp = pCppObject_;
  pCppObject_ = tmp.pCppObject_;
  tmp.pCppObject_ = tp;

  return *this;
}
*/

template<class T_CppObject> inline
bool RefPtr<T_CppObject>::operator==(const RefPtr<T_CppObject>& src) const
{
  return (pCppObject_ == src.pCppObject_);
}

template<class T_CppObject> inline
bool RefPtr<T_CppObject>::operator!=(const RefPtr<T_CppObject>& src) const
{
  return (pCppObject_ != src.pCppObject_);
}

template<class T_CppObject> inline
RefPtr<T_CppObject>::operator bool() const
{
  return (pCppObject_ != 0);
}

template<class T_CppObject> inline
bool RefPtr<T_CppObject>::is_null() const
{
  return (pCppObject_ == 0);
}


template<class T_CppObject>
void RefPtr<T_CppObject>::clear()
{
  if(pCppObject_)
  {
    pCppObject_->unreference(); // might cause deletion of pCppObject_
    pCppObject_ = 0;
  }
}

template <class T_CppObject>
  template <class T_CastFrom>
inline
RefPtr<T_CppObject> RefPtr<T_CppObject>::cast_dynamic(const RefPtr<T_CastFrom>& src)
{
  T_CppObject *const pCppObject = dynamic_cast<T_CppObject*>(src.operator->());

  if(pCppObject)
    pCppObject->reference();

  return RefPtr<T_CppObject>(pCppObject);
}

template <class T_CppObject>
  template <class T_CastFrom>
inline
RefPtr<T_CppObject> RefPtr<T_CppObject>::cast_static(const RefPtr<T_CastFrom>& src)
{
  T_CppObject *const pCppObject = static_cast<T_CppObject*>(src.operator->());

  if(pCppObject)
    pCppObject->reference();

  return RefPtr<T_CppObject>(pCppObject);
}

} // namespace Glib


#endif /* _GLIBMM_REFPTR_H */

