/* $Id: demowindow.cc,v 1.34 2002/04/27 14:13:12 murrayc Exp $ */

/* demowindow.cc
 *
 * Copyright (C) 2001 The gtkmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gtkmm/main.h"
#include "gtkmm/cellrenderertext.h"
#include "gtkmm/treeviewcolumn.h"
#include "demowindow.h"
#include "textwidget.h"
#include "demos.h"
#include <vector>
#include <cctype>
#include <cerrno>


namespace
{

struct DemoColumns : public Gtk::TreeModelColumnRecord
{
  Gtk::TreeModelColumn<Glib::ustring>         title;
  Gtk::TreeModelColumn<Glib::ustring>         filename;
  Gtk::TreeModelColumn_Pointer<type_slotDo*>  slot;
  Gtk::TreeModelColumn<bool>                  italic;

  DemoColumns() { add(title); add(filename); add(slot); add(italic); }
};

// Singleton accessor function.
const DemoColumns& demo_columns()
{
  static DemoColumns column_record;
  return column_record;
}

} // anonymous namespace


DemoWindow::DemoWindow()
: m_TextWidget_Info(false),
  m_TextWidget_Source(true)
{
  m_pWindow_Example = 0;

  set_title("gtkmm Code Demos");

  add(m_HBox);

  //Tree:
  m_refTreeStore = Gtk::TreeStore::create(demo_columns());
  m_TreeView.set_model(m_refTreeStore);

  m_refTreeSelection = m_TreeView.get_selection();
  m_refTreeSelection->set_mode(Gtk::SELECTION_BROWSE);

  m_TreeView.set_size_request(200, -1);

  fill_tree();

  m_HBox.pack_start(m_TreeView, false, false);

  //Notebook:
  m_Notebook.pages().push_back(Gtk::Notebook_Helpers::TabElem(m_TextWidget_Info, "_Info", true)); //true = use mnemonic.
  m_Notebook.pages().push_back(Gtk::Notebook_Helpers::TabElem(m_TextWidget_Source, "_Source", true)); //true = use mnemonic.
  m_HBox.pack_start(m_Notebook);


  set_default_size (600, 400);


  load_file (testgtk_demos[0].filename);
  show_all();
}

void DemoWindow::fill_tree()
{
  const DemoColumns& columns = demo_columns();

  /* this code only supports 1 level of children. If we
   * want more we probably have to use a recursing function.
   */
  for(Demo* d = testgtk_demos; d && d->title; ++d)
  {
    const Gtk::TreeIter row = m_refTreeStore->append();

    m_refTreeStore->set_value(row, columns.title,    d->title);
    m_refTreeStore->set_value(row, columns.filename, d->filename),
    m_refTreeStore->set_value(row, columns.slot,     &d->slot);
    m_refTreeStore->set_value(row, columns.italic,   false);

    for(Demo* child = d->children; child && child->title; ++child)
    {
      const Gtk::TreeIter child_row = m_refTreeStore->append(row.children());

      m_refTreeStore->set_value(child_row, columns.title,    child->title);
      m_refTreeStore->set_value(child_row, columns.filename, child->filename),
      m_refTreeStore->set_value(child_row, columns.slot,     &child->slot);
      m_refTreeStore->set_value(child_row, columns.italic,   false);
    }
  }

  Gtk::CellRendererText* pCell = Gtk::manage(new Gtk::CellRendererText());
  pCell->property_style() = Pango::STYLE_ITALIC;

  Gtk::TreeViewColumn* pColumn = new Gtk::TreeViewColumn("Widget (double click for demo)", *pCell);
  pColumn->add_attribute(pCell->property_text(), columns.title);
  pColumn->add_attribute(pCell->property_style_set(), columns.italic);

  m_TreeView.append_column(*pColumn);

  m_refTreeSelection->signal_changed().connect(SigC::slot(*this, &DemoWindow::on_treeselection_changed));
  m_TreeView.signal_row_activated().connect(SigC::slot(*this, &DemoWindow::on_treeview_row_activated));

  m_TreeView.expand_all();
}

DemoWindow::~DemoWindow()
{
  on_example_window_hide(); //delete the example window if there is one.
}

void DemoWindow::on_treeview_row_activated(const Gtk::TreePath& path, Gtk::TreeViewColumn* column)
{
  m_TreePath = path;

  if(m_pWindow_Example == 0) //Don't open a second window.
  {
    if(const Gtk::TreeIter row = m_TreeView.get_model()->get_iter(m_TreePath))
    {
      const DemoColumns& columns = demo_columns();

      type_slotDo *const pSlot = m_refTreeStore->get_value(row, columns.slot);
      if(pSlot && (m_pWindow_Example = (*pSlot)()))
      {
        m_refTreeStore->set_value(row, columns.italic, true);

        m_pWindow_Example->signal_hide().connect(SigC::slot(*this, &DemoWindow::on_example_window_hide));
        m_pWindow_Example->show();
      }
    }
  }
}

void DemoWindow::on_treeselection_changed()
{
  if(const Gtk::TreeIter row = m_refTreeSelection->get_selected())
  {
    load_file(m_refTreeStore->get_value(row, demo_columns().filename));
  }
}


bool DemoWindow::read_line (FILE *stream, GString *str)
{
  int n_read = 0;

  flockfile (stream);

  g_string_truncate (str, 0);

  while (1)
    {
      int c;

      c = getc_unlocked (stream);

      if (c == EOF)
	goto done;
      else
	n_read++;

      switch (c)
	{
	case '\r':
	case '\n':
	  {
	    int next_c = getc_unlocked (stream);
	
	    if (!(next_c == EOF ||
		  (c == '\r' && next_c == '\n') ||
		  (c == '\n' && next_c == '\r')))
	      ungetc (next_c, stream);
	
	    goto done;
	  }
	default:
	  g_string_append_c (str, c);
	}
    }

 done:

  funlockfile (stream);

  return n_read > 0;
}


void DemoWindow::load_file(const std::string& filename)
{
  if ( m_current_filename == filename )
  {
    return;
  }
  else
  {
    m_current_filename = filename;

    m_TextWidget_Info.wipe();
    m_TextWidget_Source.wipe();

    Glib::RefPtr<Gtk::TextBuffer> refBufferInfo = m_TextWidget_Info.get_buffer();
    Glib::RefPtr<Gtk::TextBuffer> refBufferSource = m_TextWidget_Source.get_buffer();

    FILE* file = fopen (filename.c_str(), "r");
    if (!file)
    {
      std::string installed = /* DEMOCODEDIR + G_DIR_SEPARATOR_S + */ filename;
      file = fopen (installed.c_str(), "r");
    }

    if (!file)
    {
      g_warning ("Cannot open %s: %s\n", filename.c_str(), g_strerror (errno));
      return;
    }

    GString *buffer = g_string_new (NULL);
    int state = 0;
    bool in_para = false;
    Gtk::TextIter start = refBufferInfo->get_iter_at_offset(0);
    while (read_line (file, buffer))
    {
      gchar *p = buffer->str;
      gchar *q = 0;
      gchar *r = 0;

      switch (state)
    	{
      	case 0:
      	  /* Reading title */
      	  while (*p == '/' || *p == '*' || isspace (*p))
      	    p++;

      	  r = p;

      	  while (*r != '/' && strlen (r))
      	    r++;
      	  if (strlen (r) > 0)
      	    p = r + 1;

      	  q = p + strlen (p);

      	  while (q > p && isspace (*(q - 1)))
      	    q--;

      	  if (q > p)
    	    {
    	      Gtk::TextIter end = start;

    	      g_assert (strlen (p) >= q - p); //TODO: What is this for?  daniel.

              const Glib::ustring strTemp (p, q);
    	      refBufferInfo->insert(end, strTemp);
    	      start = end;

    	      start.backward_chars(strTemp.length());
    	      refBufferInfo->apply_tag_by_name("title", start, end);

    	      start = end;
      	
    	      state++;
    	    }
    	    break;
  	
      	case 1:
      	  /* Reading body of info section */
      	  while (isspace (*p))
      	    p++;

      	  if (*p == '*' && *(p + 1) == '/')
    	    {
    	      start = refBufferSource->get_iter_at_offset(0);
    	      state++;
    	    }
      	  else
    	    {
    	      int len;
        	
    	      while (*p == '*' || isspace (*p))
    		      p++;

    	      len = strlen (p);
    	      while (isspace (*(p + len - 1)))
    		     len--;
        	
    	      if (len > 0)
                {
                  if (in_para)
                    {
                      refBufferInfo->insert(start, " ");
                    }

                  g_assert (strlen (p) >= len); //TODO: What is this for?  daniel.

                  refBufferInfo->insert(start, Glib::ustring(p, p + len));
                  in_para = 1;
                }
              else
                {
                  refBufferInfo->insert(start, "\n");
                  in_para = 0;
                }
      	    }
      	  break;

      	case 2:
      	  /* Skipping blank lines */
      	  while (isspace (*p))
      	    p++;

      	  if (*p)
    	    {
    	      p = buffer->str;
    	      state++;
    	      /* Fall through */
    	    }
      	  else
      	    break;
        	
      	case 3:
      	  /* Reading program body */
      	  refBufferSource->insert(start, p);
      	  refBufferSource->insert(start, "\n");
      	  break;
      	}
     }

    m_TextWidget_Source.fontify();
  }
}


void DemoWindow::on_example_window_hide()
{
  if(m_pWindow_Example)
  {
    if(const Gtk::TreeIter row = m_refTreeStore->get_iter(m_TreePath))
    {
      m_refTreeStore->set_value(row, demo_columns().italic, false);

      delete m_pWindow_Example;
      m_pWindow_Example = 0;
    }
  }
}

