// -*- c++ -*-

/* $Id: closure.cc,v 1.19 2002/02/25 02:21:14 murrayc Exp $ */

/* closure.h
 *
 * Copyright (C) 1998-2001 The Gtk-- Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <glibmm/closure.h>
#include <glibmm/object.h>
#include <glib-object.h>


namespace
{

void glibmm_closure_destroy_notifier_handler(gpointer data, GClosure* closure)
{
  Glib::GClosure_Glibmm* pClosure = reinterpret_cast<Glib::GClosure_Glibmm*>(closure);

  if(pClosure)
  {
    //pClosure->pCClosure_->sigc_slot_->invalid(); //TODO

    Glib::Closure* pCppClosure = pClosure->pCppClosure_;
    if(pCppClosure)
    {
      pCppClosure->discard_slot_data();
      pCppClosure->clear_gobj(); //Prevents the destructor from attempting a second disconnect.

      delete pCppClosure;
    }
  }

  //The GClosure_Glibmm will be freed by glib, via the GClosure* pointer.
}

} // anonymous namespace


namespace Glib
{

GSigConnectionNode::GSigConnectionNode(SigC::SlotNode* slot_data, Object* cppObject, gulong connection_id)
:
  SigC::ConnectionNode(slot_data),
  connection_id_(connection_id), pCppObject_(cppObject)
{}

GSigConnectionNode::~GSigConnectionNode()
{
  pCppObject_ = 0;
}

void GSigConnectionNode::notify(bool from_child)
{
  if(pCppObject_ && !pCppObject_->cpp_destruction_in_progress_)
  {
    g_signal_handler_disconnect(pCppObject_->gobj(), connection_id_);
    pCppObject_ = 0;
  }

  connection_id_ = 0;

  //Call base class:
  SigC::ConnectionNode::notify(from_child);

}


//Closure implementation:
Closure::Closure(Object* cppObject, const char* signal_name, GCallback callback_func,
                 const SigC::SlotBase& slot, bool after, GClosureMarshal marshal /* = 0 */)
{
  //Cause our Closure (this) to be passed as the data arg of callback_func.
  //It does seem a bit silly that the Closure is not more easily available to callbacks.
  //Unless we use a special marshaller, we don't really need to use a derived closure -
  //we could just pass a class through the data arg. But it's good design to use the
  //Closure mechanism for Closure (state) information..
  pCClosure_ = (GClosure_Glibmm*)g_closure_new_simple(sizeof(GClosure_Glibmm), this /* data */);

  if(pCClosure_)
  {
    //Set fields of the GClosure base class:
    ((GCClosure*)pCClosure_)->callback = (gpointer)callback_func;

    //Make sure that this C++ instance is destroyed when Glib wants to destroy the C instance:
    g_closure_add_finalize_notifier((GClosure*)pCClosure_, 0, &glibmm_closure_destroy_notifier_handler);

    if(marshal)
      g_closure_set_marshal((GClosure*)pCClosure_, marshal);

    //Set fields of our GClosure-derived class:
    pCClosure_->pCppClosure_ = this; //This will allow us to delete this when glibmm_closure_destroy_notifier_handler() is called.

    //We store SigC::SlotNode because we don't know type type of the SlotBase in glibmm_closure_destroy_notifier_handler().
    pCClosure_->sigc_slot_ = static_cast<SigC::SlotNode*>( const_cast<SigC::SlotBase&>(slot).impl() );
    pCClosure_->sigc_slot_->reference(); //It will actually be deleted in the last SlotNode::unreference()

    //This is like the implementation of gtk_signal_connect()
    GObject* gobject = cppObject->gobj();
    gulong connection_id = g_signal_connect_closure_by_id(
                                   gobject,
                                   g_signal_lookup (signal_name, G_OBJECT_TYPE (gobject)),
                                   0, //detail, whatever that is.
                                   gobj(), //Our C closure.
                                   after );
    pCClosure_->connection = SigC::Connection( new GSigConnectionNode(pCClosure_->sigc_slot_, cppObject, connection_id) );


    //FIXME: Comments needed here:
    //TODO:
    //obj->register_data(sd->sender());
    //slotData->connect();

    // register disconnection data
    //SigC::SlotList_& refList = slot_data->list_;
    //refList.insert_direct(refList.begin(), pClosure);
  }
}



Closure::~Closure()
{
  if(pCClosure_) //This is probably always 0, because this destructor is only called from glibmm_closure_destroy_notifier_handler().
  {
    if(gobj())
    {
      pCClosure_->connection.disconnect(); //calls g_signal_disconnect.
    }
  }
}

GClosure* Closure::gobj()
{
  return (GClosure*)pCClosure_;
}

SigC::Connection Closure::get_connection()
{
  return pCClosure_->connection;
}

void Closure::clear_gobj()
{
  pCClosure_ = 0;
}

void Closure::discard_slot_data()
{
  if(pCClosure_)
    pCClosure_->sigc_slot_->unreference(); //Will probably cause it be deleted.
}

/*
void* Closure::get_sigc_callback()
{
  if(pClosure && pCClosure_->slot)
  {
    return pCClosure_->sigc_slot_data_->data;
  }
  else
  {
    return (void*)0;
  }

}
*/

} /* namespace Glib */



