/*
 * Widget.java
 *
 * Copyright (c) 2006-2007 Operational Dynamics Consulting Pty Ltd, and Others
 * 
 * The code in this file, and the library it is a part of, are made available
 * to you by the authors under the terms of the "GNU General Public Licence,
 * version 2" plus the "Classpath Exception" (you may link to this code as a
 * library into other programs provided you don't make a derivation of it).
 * See the LICENCE file for the terms governing usage and redistribution.
 */
package org.gnome.gtk;

import org.gnome.gdk.EventExpose;
import org.gnome.gdk.EventKey;

/**
 * The base class of all GTK Widgets. Graphical user interface toolkits have
 * long been built up from individual controls and presentation mechanisms
 * that are nested together. These elements are collectively called Widgets.
 * Quite a lot of Widgets contain other Widgets; those are called
 * {@link Container Container}s.
 * 
 * @author Andrew Cowie
 * @since 4.0.0
 */
public abstract class Widget extends org.gnome.gtk.Object
{

    protected Widget(long pointer) {
        super(pointer);
    }

    /**
     * Cause this Widget to be mapped to the screen. Flags a widget to be
     * displayed. Any widget that isn't shown will not appear on the screen.
     * 
     * <p>
     * There are a bunch of quirks you need to be aware of:
     * 
     * <ul>
     * <li>You have to show the Containers containing a Widget, in addition
     * to the Widget itself, before it will appear on the display.
     * <li>When a toplevel Container is shown (ie, your {@link Window Window}),
     * it is immediately realized and mapped, and any Widgets within it that
     * are shown are then realized and mapped.
     * <li>You can't get information about the actual size that has been
     * allocated to a Widget until it gets mapped to the screen.
     * </ul>
     * 
     * <p>
     * If you want to show all the widgets in a container, it's actually much
     * easier to just call {@link #showAll()} on the container, rather than
     * calling show manually one each individual Widget you've added to it.
     * 
     * @since 4.0.0
     */
    public void show() {
        GtkWidget.show(this);
    }

    /**
     * Cause this Widget, and any Widgets it contains, to be mapped to the
     * screen. You typically call this on a {@link Window Window} after you've
     * finished all the work necessary to set it up.
     * <p>
     * Quite frequently you also want to cause a Window to appear on the
     * screen as well (ie, not be buried under a whole bunch of other
     * applications' Windows), so calling Window's
     * {@link Window#present() present()} is usually next.
     * 
     * <p>
     * <i>Don't be surprised if this takes a few hundred milliseconds.
     * Realizing and mapping all the zillion elements that ultimately make up
     * a Window is one of the most resource intensive operations that GTK,
     * GDK, Pango, your X server, and your kernel have to churn through.
     * Sometimes, you just gotta wait.</i>
     * 
     * @since 4.0.0
     */
    public void showAll() {
        GtkWidget.showAll(this);
    }

    /**
     * Signal emitted when the mouse enters the Widget
     * 
     * @author Andrew Cowie
     * @author Davyd Madeley
     * @since 4.0.2
     */
    public interface ENTER_NOTIFY_EVENT extends GtkWidget.ENTER_NOTIFY_EVENT
    {
        public boolean onEnterNotifyEvent(Widget source, Object event);
    }

    /**
     * Hook up a handler to receive "enter-notify-event" events on this
     * Widget.
     * 
     * @since 4.0.2
     */
    public void connect(ENTER_NOTIFY_EVENT handler) {
        GtkWidget.connect(this, handler);
    }

    /**
     * Signal emitted when the focus leaves this Widget. Focus is a concept
     * that is shared evenly between the widget toolkit and the window manager -
     * which can become apparent if you're wondering <i>why</i> you have lost
     * focus or regained it.
     * 
     * @author Andrew Cowie
     * @since 4.0.2
     */
    public interface FOCUS_OUT_EVENT extends GtkWidget.FOCUS_OUT_EVENT
    {
        public boolean onFocusOutEvent(Widget source, Object event);
    }

    /**
     * Hook up a handler to receive "focus-out-event" events on this Widget
     * 
     * @since 4.0.2
     */
    public void connect(FOCUS_OUT_EVENT handler) {
        GtkWidget.connect(this, handler);
    }

    public interface EXPOSE_EVENT extends GtkWidget.EXPOSE_EVENT
    {
        public boolean onExposeEvent(Widget source, EventExpose event);
    }

    /**
     * Hook up a handler to receive "expose-event" events on this Widget
     * 
     * @since 4.0.2
     */
    public void connect(EXPOSE_EVENT handler) {
        GtkWidget.connect(this, handler);
    }

    /**
     * Handler interface for key press events. The user <i>pressing</i> a key
     * is generally less interesting than them <i>releasing</i> a key as
     * that, by the conventions of modern graphical user interfaces, is when a
     * program should take action; note for example that if they press a key
     * but then move the mouse out the release won't cause that Button to
     * activate.
     * 
     * @since 4.0.3
     * @see Widget.KEY_RELEASE_EVENT
     */
    public interface KEY_PRESS_EVENT extends GtkWidget.KEY_PRESS_EVENT
    {
        public boolean onKeyPressEvent(Widget source, EventKey event);
    }

    /**
     * Hook up a handler to receive <code>key-press-event</code> signals on
     * this Widget. In general you <b>don't</b> want this.
     * 
     * @since 4.0.3
     */
    public void connect(KEY_PRESS_EVENT handler) {
        GtkWidget.connect(this, handler);
    }

    /**
     * Handler interface for key release events.
     * 
     * @since 4.0.3
     */
    public interface KEY_RELEASE_EVENT extends GtkWidget.KEY_RELEASE_EVENT
    {
        public boolean onKeyReleaseEvent(Widget source, EventKey event);
    }

    /**
     * Hook up a handler to receive <code>key-release-event</code> signals
     * on this Widget
     * 
     * @since 4.0.3
     */
    public void connect(KEY_RELEASE_EVENT handler) {
        GtkWidget.connect(this, handler);
    }

    /*
     * Temporary: testing full downcasting.
     */
    public Container getParent() {
        return (Container) getPropertyObject("parent");
    }
}
