/* this file is part of libccc, criawips' cairo-based canvas
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2005 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <ccc/cc-text.h>

#include <pango/pangocairo.h>
#include <gtk/gtk.h>
#include <ccc/cc-utils.h>
#include "pango-helpers.h"

CcItem*
cc_text_new(gchar const* text) {
	return g_object_new(CC_TYPE_TEXT, "text", text, NULL);
}

static gdouble
ct_get_x_offset(CcText* self, gdouble x, gdouble width) {
	switch(self->anchor) {
	case GTK_ANCHOR_NE:
	case GTK_ANCHOR_E:
	case GTK_ANCHOR_SE:
		return x - width;
	case GTK_ANCHOR_N:
	case GTK_ANCHOR_CENTER:
	case GTK_ANCHOR_S:
		return x - 0.5 * width;
	case GTK_ANCHOR_NW:
	case GTK_ANCHOR_W:
	case GTK_ANCHOR_SW:
	default:
		return x;
	}
}

static gdouble
ct_get_y_offset(CcText* self, gdouble y, gdouble height) {
	switch(self->anchor) {
	case GTK_ANCHOR_SW:
	case GTK_ANCHOR_S:
	case GTK_ANCHOR_SE:
		return y - height;
	case GTK_ANCHOR_W:
	case GTK_ANCHOR_CENTER:
	case GTK_ANCHOR_E:
		return y - 0.5 * height;
	case GTK_ANCHOR_NW:
	case GTK_ANCHOR_N:
	case GTK_ANCHOR_NE:
	default:
		return y;
	}
}

void
cc_text_set_anchor(CcText* self, gdouble x, gdouble y) {
	CcItem* item = CC_ITEM(self);

	self->x = x;
	self->y = y;

	cc_item_update_bounds(item, NULL);
}

void
cc_text_set_anchor_type(CcText* self, GtkAnchorType anchor) {
	if(self->anchor == anchor) {
		return;
	}

	self->anchor = anchor;
	cc_item_update_bounds(CC_ITEM(self), NULL);

	g_object_notify(G_OBJECT(self), "anchor");
}

/**
 * cc_text_set_font_description:
 * @self: a #CcText
 * @desc: a #PangoFontDescription
 *
 * Sets the font description to be used for rendering.
 */
void
cc_text_set_font_description(CcText* self, PangoFontDescription* desc) {
	pango_layout_set_font_description(self->layout, desc);
	// FIXME: forward the description-changed signal
	cc_item_update_bounds(CC_ITEM(self), NULL);
}

/**
 * cc_text_set_markup:
 * @self: a #CcText
 * @markup: the markup to be set
 *
 * Set the markup displayed by this item.
 */
void
cc_text_set_markup(CcText* text, gchar const* markup) {
	// FIXME: implement markup
	cc_text_set_text(text, markup);
}

/**
 * cc_text_set_size_pixels:
 * @self: a #Cctext
 * @size_pixels: the value to be set
 *
 * Specify whether the text size is given in pixels. If it is, the text won't
 * scale with the zoom level.
 */
void
cc_text_set_size_pixels(CcText* self, gboolean size_pixels) {
	g_return_if_fail(CC_IS_TEXT(self));

	if(self->size_pixels == size_pixels) {
		return;
	}

	self->size_pixels = size_pixels;

	g_object_notify(G_OBJECT(self), "size-pixels");
	cc_item_update_bounds(CC_ITEM(self), NULL);
}

/**
 * cc_text_set_text:
 * @self: a #CcText
 * @text: the text to be set
 *
 * Set the text to be displayed by a #CcText item.
 */
void
cc_text_set_text(CcText* self, gchar const* text) {
	pango_layout_set_text(self->layout, text, -1);
	cc_item_update_bounds(CC_ITEM(self), NULL);
}

/* GType stuff */
G_DEFINE_TYPE(CcText, cc_text, CC_TYPE_ITEM);

enum {
	PROP_0,
	PROP_ANCHOR,
	PROP_SIZE_PIXELS,
	PROP_TEXT
};

static void
ct_layout_notify_text(CcText* self, GParamSpec* pspec, PangoLayout* layout) {
	g_object_notify(G_OBJECT(self), "text");

	cc_item_update_bounds(CC_ITEM(self), NULL);
}

static void
cc_text_init(CcText* self) {
	self->layout = pango_layout_new_cairo();
	g_signal_connect_swapped(self->layout, "notify::text",
				 G_CALLBACK(ct_layout_notify_text), self);
}

static void
ct_dispose(GObject* object) {
	CcText* self = CC_TEXT(object);

	if(CC_ITEM_DISPOSED(self)) {
		return;
	}
	CC_ITEM_SET_FLAGS(self, CC_DISPOSED);

	if(self->layout) {
		g_signal_handlers_disconnect_by_func(self->layout, ct_layout_notify_text, self);
		g_object_unref(self->layout);
		self->layout = NULL;
	}

	G_OBJECT_CLASS(cc_text_parent_class)->dispose(object);
}

static void
ct_render(CcItem* item, CcView* view, cairo_t* cr) {
	CcText     * self   = CC_TEXT(item);
	PangoLayout* layout = self->layout;
	CcDRect    * bounds = cc_hash_map_lookup(item->bounds, view);
	gdouble      x = ct_get_x_offset(self, self->x, bounds->x2 - bounds->x1),
		     y = ct_get_y_offset(self, self->y, bounds->y2 - bounds->y1);
	PangoFontDescription const* desc_old = pango_layout_get_font_description(layout);
	PangoFontDescription      * backup   = pango_font_description_copy(desc_old ? desc_old : pango_context_get_font_description(pango_cairo_font_map_create_context(PANGO_CAIRO_FONT_MAP(pango_cairo_font_map_get_default()))));
	PangoFontDescription      * desc_new = pango_font_description_copy_static(backup);
	gboolean      size_absolute = FALSE;
	gdouble       old_size = 0.0,
		      new_size = 0.0;

	cc_view_world_to_window(view, &x, &y);

	if(CC_ITEM_GRID_ALIGNED(item)) {
		gdouble width = 1.0;
		cc_point_grid_align(&x, &y, &width);
	}

	if(!self->size_pixels) {
		size_absolute = pango_font_description_get_size_is_absolute(backup);
		new_size = old_size = pango_font_description_get_size(backup);

		cc_view_world_to_window_distance(view, &new_size, NULL);

		if(size_absolute) {
			pango_font_description_set_absolute_size(desc_new, new_size);
		} else {
			pango_font_description_set_size(desc_new, new_size);
		}

		pango_layout_set_font_description(layout, desc_new);
	}

	cairo_save(cr);
	  cairo_move_to(cr, x, y);
	  pango_cairo_update_layout(cr, layout);
	  pango_cairo_show_layout(cr, layout);
	cairo_restore(cr);

	if(!self->size_pixels) {
		pango_layout_set_font_description(layout, desc_old);
	}

	pango_font_description_free(desc_new);

	CC_ITEM_CLASS(cc_text_parent_class)->render(item, view, cr);
}

static void
ct_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
	CcText* self = CC_TEXT(object);

	switch(prop_id) {
	case PROP_ANCHOR:
		g_value_set_enum(value, self->anchor);
		break;
	case PROP_SIZE_PIXELS:
		g_value_set_boolean(value, self->size_pixels);
		break;
	case PROP_TEXT:
		g_value_set_string(value, pango_layout_get_text(self->layout));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}

static void
ct_set_property(GObject* object, guint prop_id, GValue const* value, GParamSpec* pspec) {
	CcText* self = CC_TEXT(object);

	switch(prop_id) {
	case PROP_ANCHOR:
		cc_text_set_anchor_type(self, g_value_get_enum(value));
		break;
	case PROP_SIZE_PIXELS:
		cc_text_set_size_pixels(self, g_value_get_boolean(value));
		break;
	case PROP_TEXT:
		cc_text_set_text(self, g_value_get_string(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}

static void
ct_update_bounds(CcItem* item, CcView const* view, gpointer data) {
	PangoRectangle rect;
	CcDRect new_bounds;
	CcText* self = CC_TEXT(item);
	CcDRect* bounds = cc_hash_map_lookup(item->bounds, view);

	pango_layout_get_extents(self->layout, NULL, &rect);

	new_bounds.x1 = ct_get_x_offset(self, self->x, (1.0 * rect.width  / PANGO_SCALE)) + (1.0 * rect.x / PANGO_SCALE);
	new_bounds.y1 = ct_get_y_offset(self, self->y, (1.0 * rect.height / PANGO_SCALE)) + (1.0 * rect.y / PANGO_SCALE);
	new_bounds.x2 = new_bounds.x1         + (1.0 * rect.width  / PANGO_SCALE);
	new_bounds.y2 = new_bounds.y1         + (1.0 * rect.height / PANGO_SCALE);

	if(!bounds || !cc_d_rect_equal(new_bounds, *bounds)) {
		if(bounds) {
			cc_item_dirty(item, view, *bounds);
		}

		cc_hash_map_remove(item->bounds, view);
		cc_hash_map_insert(item->bounds, (gpointer)view, cc_d_rect_copy(&new_bounds));
		cc_item_dirty(item, view, new_bounds);
		cc_item_bounds_changed(item, view);
	}
}

static void
cc_text_class_init(CcTextClass* self_class) {
	GObjectClass* go_class;
	CcItemClass * ci_class;

	/* GObjectClass */
	go_class = G_OBJECT_CLASS(self_class);
	go_class->dispose      = ct_dispose;
	go_class->get_property = ct_get_property;
	go_class->set_property = ct_set_property;

	g_object_class_install_property(go_class,
					PROP_ANCHOR,
					g_param_spec_enum("anchor",
							  "Anchor",
							  "The location of the anchor point of the text element",
							  GTK_TYPE_ANCHOR_TYPE,
							  GTK_ANCHOR_NW,
							  G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(go_class,
					PROP_SIZE_PIXELS,
					g_param_spec_boolean("size-pixels",
							     "Size in Pixels",
							     "Specifies whether the given size is in pixels or in canvas units",
							     FALSE,
							     G_PARAM_READWRITE));
	g_object_class_install_property(go_class,
					PROP_TEXT,
					g_param_spec_string("text",
							    "Text",
							    "The displayed text",
							    NULL,
							    G_PARAM_READWRITE));

	/* CcItemClass */
	ci_class = CC_ITEM_CLASS(self_class);
	ci_class->render        = ct_render;
	ci_class->update_bounds = ct_update_bounds;
}

