'''
Defines an L{AEEvent} indicating that the selector has moved.

@author: Pete Brunet
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import Base, Constants

class SelectorChange(Base.AccessEngineEvent):
  '''
  Event that fires when the selector moves.
  
  This class registers its name and whether it should be monitored by default in
  an L{AEMonitor.EventMonitor} using the L{Constants.registerEventType} function
  when this module is first imported. The L{AEMonitor.EventMonitor} can use this
  information to build its menus.
  
  @ivar item_text: The accessible text or name of item at the selector
  @type item_text: string
  '''
  Constants.registerEventType('SelectorChange', True)
  def __init__(self, por, item_text, **kwargs):
    '''
    Calls the base class (which set the event priority) and then stores the item
    to be passed along to the Tier as part of the event.
    
    @param por: Point-of-regard
    @type por: L{POR}
    @param item_text: the accessible text or name of the item at the selector
    @type item_text: string
    '''
    Base.AccessEngineEvent.__init__(self, **kwargs)
    self.item_text = item_text
    self.por = por
    
  def __str__(self):
    '''
    Returns a human readable representation of this event including its name,
    its L{POR}, and its item text.
    
    @return: Information about this event
    @rtype: string
    '''
    name = Base.AccessEngineEvent.__str__(self)
    return '%s:\n\tPOR: %s\n\titem text: %s' % (name, self.por, self.item_text)
  
  def execute(self, tier_manager, **kwargs):
    '''
    Contacts the L{TierManager} so it can manage the selector change event.
    
    @param tier_manager: TierManager that will handle the event
    @type tier_manager: L{TierManager}
    @param kwargs: Packed references to other managers not of interest here
    @type kwargs: dictionary
    @return: Always True to indicate the event executed properly
    @rtype: boolean
    '''
    tier_manager.manageEvent(self)
    return True

  def getDataForTask(self):
    '''
    Fetches data out of this L{SelectorChange} for use by a
    L{Task.SelectorTask}.
    
    @return: Dictionary of parameters to be passed to a
      L{Task.SelectorTask} as follows:
        - por:  The L{POR} of the accessible in which the selector moved
        - item_text: The text for the item at the L{POR}
    @rtype: dictionary
    '''
    return {'por':self.getPOR(), 'item_text':self.item_text}