'''
Defines the abstract base class for all L{AEInput} subclasses.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import logging

log = logging.getLogger('Input')

class AEInput(object):
  '''  
  Most abstract base class for all L{AEInput} devices. Maintains a collection of
  L{Gesture} listeners that can be notified using L{_notifyInputListeners}.
  Defines simple L{init} and L{close} methods that change the state of the
  L{ready} flag.
  
  This class is abstract as most of its methods raise NotImplementedError and 
  need to be overriden with input device specific code in subclasses.

  @ivar listeners: Collection of listeners that are notified about L{Gesture}s
    on an input device
  @type listeners: list of callable
  @ivar ready: Is the input device initialized?
  @type ready: boolean
  '''
  def __init__(self):
    '''
    Initializes the listeners list to empty and sets the ready flag to False.
    '''
    self.listeners = []
    self.ready = False

  def init(self): 
    '''
    Sets the ready flag to True.
    '''
    self.ready = True
    
  def close(self): 
    '''
    Sets the ready flag to False.
    '''
    self.ready = False
    
  def isReady(self): 
    '''
    Gets the state of the ready flag.
    
    @return: Is the input device initialized?
    @rtype: boolean
    '''
    return self.ready
    
  def addInputListener(self, listener):
    '''
    Adds a listener to be notified whenever a L{Gesture} occurs on an input
    device. Listeners are called in the order they are added and are given the
    L{Gesture} detected.
    
    @param listener: Object to call when a L{Gesture} occurs
    @type listener: callable
    '''
    self.listeners.append(listener)
    
  def removeInputListener(self, listener):
    '''
    Removes an existing listener for L{Gesture}s.
    
    @param listener: Object to remove from the listener list
    @type listener: callable
    @raise ValueError: When removing a listener that is not registered
    '''
    self.listeners.remove(listener)
    
  def inputListenersExist(self):
    '''
    Gets if there are any registered L{Gesture} listeners for this device.
    
    @return: Is there at least one listener registered?
    @rtype: boolean
    '''
    return len(self.listeners) > 0
    
  def _notifyInputListeners(self, gesture):
    '''
    Notifies registered listeners about a L{Gesture} seen on the input device. 
    Catches all exceptions from the callback and logs them.
    
    @param gesture: L{Gesture} to send to listeners
    @type gesture: L{Gesture}
    '''
    for listener in self.listeners:
      try:
        listener(gesture)
      except Exception:
        log.exception('AEInput exception')
      
  def addFilter(self, gestures):
    '''
    Abstact method. Adds a L{GestureList} as a filter to this device. How the
    filter is stored and used is determined by a subclass.
    
    @param gestures: L{Gesture}s to possibly filter
    @type gestures: L{GestureList}
    @raise NotImplementedError: When this method is not overridden by a subclass
    '''
    raise NotImplementedError
    
  def removeFilter(self, gestures):
    '''
    Abstract method. Removes a L{GestureList} as a filter from this device. How
    the filter is removed is determined by a subclass.
    
    @param gestures: L{Gesture}s to no longer filter
    @type gestures: L{GestureList}
    @raise NotImplementedError: When this method is not overridden by a subclass
    '''
    raise NotImplementedError
  
  def clearFilters(self):
    '''    
    Abstract method. Removes all L{GestureList} filters from the device. How the
    filters are removed is determined by a subclass.
    
    @raise NotImplementedError: When this method is not overridden by a subclass
    '''
    raise NotImplementedError
    
  def sortGesture(self, gesture):
    '''    
    Abstract method. Sort the provided L{Gesture} according to a sort order
    specific to the type of input device.
    
    @param gesture: L{Gesture} object to sort
    @type gesture: L{Gesture}
    @return: List of action codes sorted according to the specific device 
        implementation
    @rtype: list of integer
    @raise NotImplementedError: When this method is not overridden by a subclass
    '''
    raise NotImplementedError
  
  def getMaxActions(self):
    '''
    Abstract method. Gets the maximum number of actions that can be in a 
    L{Gesture} on this input device.
    
    @return: Maximum number of actions per L{Gesture} supported by this device
    @rtype: integer
    @raise NotImplementedError: When this method is not overridden by a subclass
    '''
    raise NotImplementedError
    
  def asString(self, gesture): 
    ''' 
    Abstract method. Gets a human readable representation of the given
    L{Gesture}.
    
    @param gesture: L{Gesture} object to render as text
    @type gesture: L{Gesture}
    @return: Text representation of the L{Gesture} 
    @rtype: string
    @raise NotImplementedError: When this method is not overridden by a subclass
    '''
    raise NotImplementedError
    
  def getName(self):
    '''
    Abstract method. Gets the name of the input device.
    
    @return: Name of the device
    @rtype: string
    @raise NotImplementedError: When this method is not overridden by a subclass
    '''
    raise NotImplementedError
    
  def setLocale(self, loc):
    '''
    Abstract method. Sets the locale used for handling input from the user.

    When the exact locale provided is not supported, this will attempt to set
    an available locale based on the language. And will use the device default
    when the language is not available. This returns the exact locale set.

    Depending on the device, this should also modify the text content gotten
    from the user. For Braille devices, this can change which table is
    used.
    
    When a supported locale is specified, the strings returned from
    L{asString} will reflect this command.
    
    @param loc: Name of the locale to set
    @type loc: string
    @raise NotImplementedError: When this method is not overridden by a subclass
    '''
    raise NotImplementedError
    
  def getLocale(self): 
    '''
    Abstract method. Gets the current locale in use for the input device.

    @return: The name of the current locale
    @rtype: string
    @raise NotImplementedError: When not overridden in a subclass
    @see: L{setLocale}
    '''
    raise NotImplementedError