'''
Defines the base class for all speech output devices.

@author: Larry Weiss
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import logging, threading, time, Queue
import Base, Constants

log = logging.getLogger('Output')

class Speech(Base.AEOutput):
  '''
  Defines the base class for all speech output devices.

  This overrides most of the methods defined by L{AEOutput.Base}, but further
  defines and uses methods that raise L{NotImplementedError} to ensure that
  derived device specific classes create speech specific implementions.
  
  @ivar running: True when the device is initialized and ready to receive
    output.
  @type running: boolean
  @ivar cmd_chars: The string of chars that may be interpreted as a command
  @type cmd_chars: string
  @ivar write_thread: The thread that uses the device specific methods to send
    buffered content and commands to the speech device.
  @type write_thread: L{threading.Thread}
  @ivar listeners: Index marker listeners
  @type listeners: list
  '''
  def __init__(self):
    '''
    Initializes the instance variables to their default values.
    '''
    self.running = False
    self.cmd_chars = ''
    self.write_thread = None
    self.listeners = []
    
  def useThread(self):
    '''
    States whether the devices wishes to use a Python thread to buffer speech
    commands or not. If this method returns True, all speech requests are 
    buffered, dispatched, and executed in a thread separate from the rest of
    the LSR system. This is beneficial if the device blocks until each speech
    command completes or if parameter commands take effect immediately instead
    of at the time when they are executed. If this method returns False, all
    speech requests are sent immediately to the device. This mode is beneficial
    if calls to the device are already asynchronous and parameter commands do
    take effect in sequence with all other commands
    
    @return: True if a secondary thread is desired, False if not
    @rtype: boolean
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError

  def deviceInit(self):
    '''
    Initializes the specific speech device.

    @raise AEOutput.InitError: When the device can not be initialized
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError

  def deviceClose(self):
    '''
    Closes the specific speech device.

    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def writeStop(self):
    '''
    Writes a stop command to the device. Blocks until the device is ready to
    receive more data.

    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def writeTalk(self):
    '''
    Writes a talk command to the device. Blocks until the device is ready to
    receive more data.

    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def writeString(self, string):
    '''
    Writes a string to the device. Blocks until the device is ready to receive
    more data.
  
    @param string: Text to write to the device
    @type string: string
    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def writeCommand(self, cmd, value):
    '''
    Writes a new value to the given command on the device. Blocks until the
    device is ready to receive more data.

    @param cmd: Command to set on the device
    @type cmd: object
    @param value: Value to set
    @type value: object
    @raise NotImplementedError: When not overridden in a subclass.
    @raise ValueError: When the command or value is invalid
    '''
    raise NotImplementedError

  def deviceSpeaking(self):
    '''
    Indicates whether the device is active.

    @return: True when the speech device is actively synthesizing
    @rtype: boolean
    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def init(self):
    '''
    Called to initialize the speech output interface. Sets up a L{_WriteThread}
    that may or may not actually start a secondary thread. Calls L{deviceInit} 
    before creating the thread to ensure that the device is working correctly 
    before data is sent.
    
    This method should not be overriden to do device specific initialization.
    Override L{deviceInit} instead.
    
    Overrides L{AEOutput.Base.AEOutput.init}

    @raise InitError: When raised by L{deviceInit}
    @raise NotImplementedError: When L{deviceInit} is not overridden in a
      subclass.
    '''
    if not self.running:
      # run the device specific init
      self.deviceInit()

      # fill in the chars that should not be sent "as is"
      self.cmd_chars = self.getCommandChars()

      # ready for data when we've gotten this far
      self.running = True
      
      # create write_thread, let it start itself
      self.write_thread = _WriteThread(self)

  def close(self):
    '''
    Stops the writing queue and calls L{deviceClose}.

    Overrides L{AEOutput.Base.AEOutput.close}

    @raise NotImplementedError: When L{deviceClose} is not overridden in a
      subclass.
    '''
    if self.running:
      self.running = False
      # send a sentinal to write_thread to kill it
      self.write_thread.put(Constants.BUFFER_OBJECT)

  def sendString(self, string):
    '''
    Sends a string asynchronously to the device. Characters in the string 
    matching those specified in L{cmd_chars} will be replaced with a space.

    Overrides L{AEOutput.Base.AEOutput.sendString}

    @param string: Text string to buffer
    @type string: string
    '''
    for c in self.cmd_chars:
      string = string.replace(self.cmd_chars, ' ')
    self.write_thread.put(Constants.BUFFER_STRING, string)

  def sendCommand(self, cmd, value):
    '''
    Sends a command to set a parameter to the current value asynchronously.

    Overrides L{AEOutput.Base.AEOutput.sendCommand}

    @param cmd: The device specific command to send
    @type cmd: object
    @param value: The value of the specified command to send
    @type value: object
    '''
    self.write_thread.put(Constants.BUFFER_COMMAND, cmd, value)

  def sendStop(self):
    '''
    Sends a stop command asynchronously. The stop command will be processed with
    higher priority than other commands so that it stops speech ASAP.

    Overrides L{AEOutput.Base.AEOutput.sendStop}
    '''
    self.write_thread.put(Constants.BUFFER_COMMAND, Constants.DEVICE_STOP)

  def sendTalk(self):
    '''
    Sends a command to starting speaking asynchronously.

    Overrides L{AEOutput.Base.AEOutput.sendTalk}
    '''
    self.write_thread.put(Constants.BUFFER_COMMAND, Constants.DEVICE_TALK)

  def isSpeaking(self):
    '''
    Indicates whether the device is active.

    @return: True when content is buffered or the speech device is
      synthesizing
    @rtype: boolean
    @raise NotImplementedError: When L{deviceSpeaking} is not overridden in a
      subclass.
    '''
    if self.write_thread.hasContent():
      return True
    else:
      return self.deviceSpeaking()

  def sendStringSync(self, string, stop=True):
    '''
    Buffers a complete string to send to the device synchronously.

    This should B{not} be used in place of L{sendString} since this will not
    return until the string is finished being output. This is provided B{only}
    for the convenience of utility writers. Uses L{sendStop}, L{sendString}, 
    L{sendTalk}, and then sleeps until L{isSpeaking} returns False.

    Overrides L{AEOutput.Base.AEOutput.sendString}

    @param string: The string to buffer
    @type string: string
    @param stop: True by default to interrupt current output
    @type stop: boolean
    '''
    if stop:
      self.sendStop()
    self.sendString(string)
    self.sendTalk()

    # wait until done speaking; this blocks the main thread -- be careful
    while self.isSpeaking():
      time.sleep(0.01 * self.write_thread.contentSize())
      
  def sendIndex(self, marker):
    '''
    When a subclass implements indexing (marker notification), it should 
    override this method with C{self.sendCommand(Constants.DEVICE_INDEX, 
    marker)}.

    @param marker: The value that should be returned when speech has been 
      processed up to this point
    @type marker: integer
    @raise NotImplementedError: When not overridden in a subclass.
    '''
    raise NotImplementedError

  def addIndexListener(self, listener):
    '''
    Adds a listener that should be notified when speech has progressed to the 
    point where a marker was inserted with L{sendIndex}.
    
    @param listener: The method that should be called when markers received.
    @type listener: callable
    '''
    self.listeners.append(listener)

  def removeIndexListener(self, listener):
    '''
    Removes the specified listener.
    
    @param listener: The method that should no longer be called when markers 
      received.
    @type listener: callable
    @raise ValueError: When the given listener is not already registered
    '''
    self.listeners.remove(listener)

class _WriteThread(threading.Thread):
  '''
  Calls methods on a L{AEOutput.Speech} device based on received commands. The 
  method calls may or may not be performed in a separate thread based on the
  value of the L{AEOutput.Speech.useThread} method. This class supports both
  immediate calls in the main thread and buffered calls in a secondary thread.
  
  @ivar device: Device on which to invoke write* methods
  @type device: L{AEOutput.Speech}
  @ivar want_stop: Flag indicating a stop is requested
  @type want_stop: boolean
  @ivar just_stopped: Flag indicating that the last written command was a stop.
      Used to avoid unnecessary stops
  @type just_stopped: boolean
  @ivar lock: Semaphore used to ensure no commands are buffered while the buffer
      is being reset after L{Constants.DEVICE_STOP} is received
  @type lock: threading.Sempahore
  @ivar data_buffer: Buffer of commands to be sent to the output device
  @type data_buffer: Queue.Queue
  '''
  def __init__(self, device):
    '''
    Initializes the parent class and stores the device reference. Creates a 
    queue that will buffer commands to the speech device. Creates flags used
    for indicating whether a stop is requested or has been requested recently.
    Creates a semaphore used to ensure that no commands can be added to the 
    buffer while it is being reset by a stop command. Starts a secondary thread
    if L{AEOutput.Speech.useThread} returns True.

    @param device: The device reference to use for writing commands
    @type device: L{AEOutput.Speech}
    '''
    threading.Thread.__init__(self)
    self.device = device
    self.alive = True
    self.want_stop = False
    self.just_stopped = False
    self.lock = threading.Semaphore()
    self.data_buffer = Queue.Queue()
    # start a secondary thread running if a thread is desired
    if self.device.useThread():
      self.start()

  def hasContent(self):
    '''
    Does the data buffer contain any commands?
    
    @note: The implementation of L{Queue.Queue} is such that it does not 
        guarantee an accurate report of whether it is empty or not. It may have
        just transitioned from empty to not empty and a subsequent call may
        return a different result. OK for our purposes.
    @return: True if it does, False if not
    @rtype: boolean
    '''
    return not self.data_buffer.empty()

  def contentSize(self):
    '''
    Gets the number of commands in the data buffer. Returns 1 if a secondary
    thread is not being used.
    
    @note: The implementation of L{Queue.Queue} is such that it does not 
      guarantee an accurate report of the number of items it holds. It may 
      have just transitioned from count of x to count of y and a subsequent 
      call may return a different result. OK for our purposes.
    @return: Number of items in the buffer
    @rtype: integer
    '''
    if self.device.useThread():
      return self.data_buffer.qsize()
    else:
      return 1
  
  def put(self, cmd, *args):
    '''
    Determines whether to handle a command immediately in the main thread or
    buffer it for later to be handled by a secondary speech device thread.
    
    @param cmd: L{AEOutput.Constants} buffer command
    @type cmd: integer
    @param args: Additional positional arguments to be passed to the device
      when the command is executed
    @type args: list
    '''
    if self.device.useThread():
      self._putBuffer(cmd, args)
    else:
      self._handleCommand(cmd, args)
  
  def _putBuffer(self, cmd, args):
    '''
    Buffers any non-stop command in the queue and returns immediately. Sets the
    L{want_stop} flag for a stop command if neither L{want_stop} nor 
    L{just_stopped} is set. Blocks on entry if the thread is busy clearing out
    the buffer in response to a previous stop command. Leaves the lock set if
    a new stop command is buffered. It will be unset when the command is
    processed.
    
    @param cmd: L{AEOutput.Constants} buffer command
    @type cmd: integer
    @param args: Additional positional arguments to be passed to the device
        when the command is executed
    @type args: list
    '''
    self.lock.acquire()
    if (cmd == Constants.BUFFER_COMMAND and args[0] == Constants.DEVICE_STOP):
      if not self.want_stop and not self.just_stopped:
        # only queue the stop and set the flag if it's not already set
        self.want_stop = True
        # add the stop command to the buffer in case the thread is sleeping
        self.data_buffer.put_nowait((cmd, args))
        # IMPORTANT: do not release the semaphore, the thread will do it when
        # it processes the stop command; it must be held so that further 
        # invocations of this method block until the buffer has been emptied
        return
    else:
      self.data_buffer.put_nowait((cmd, args))
      # as soon as something is buffered, stop is no longer last.
      self.just_stopped = False
    self.lock.release()
    
  def _handleCommand(self, cmd, args):
    '''
    Handle a speech command by invoking the appropriate write method on the 
    L{device}.
    
    @todo: LW: need to handle context commands

    @param cmd: L{AEOutput.Constants} buffer command
    @type cmd: integer
    @param args: Additional positional arguments to be passed to the device
        when the command is executed
    @type args: list
    '''
    if cmd == Constants.BUFFER_STRING:
      # handle buffered chars
      self.device.writeString(*args)
    elif cmd == Constants.BUFFER_COMMAND:
      if args[0] == Constants.DEVICE_STOP:
        # handle device stop (shouldn't happen when using the thread)
        self.device.writeStop()
      elif args[0] == Constants.DEVICE_TALK:
        # handle device talk
        self.device.writeTalk()
      else:
        # write out a device specific command
        self.device.writeCommand(*args)
    elif cmd == Constants.BUFFER_OBJECT:
      # handle death of this thread by closing the device
      self.device.deviceClose()
      self.alive = False
    
  def run(self):
    '''
    Runs the thread until a L{Constants.BUFFER_OBJECT} is received. Calls
    L{_handleCommand} to process anything other than a L{Constants.DEVICE_STOP}.
    The stop is handled immediately by flushing the buffer, calling writeStop
    on the device, flipping L{want_stop} and L{just_stopped}, and releasing the 
    L{lock} semaphore.
    '''
    while self.alive:
      # Queue.get() blocks for data
      cmd, args = self.data_buffer.get()

      # if want to stop, it doesn't matter what the buffer element was
      if self.want_stop:
        # reset the buffer immediately
        self.data_buffer = Queue.Queue()
        # send the device specific stop command
        self.device.writeStop()
        # reset the stop flag
        self.want_stop = False
        # indicate we've just stopped
        self.just_stopped = True
        # IMPORTANT: release the semaphore so the put method can continue
        self.lock.release()
        continue
      # handle the command
      self._handleCommand(cmd, args)
      
def isSpeech(obj):
  '''
  Returns if the given object derives from L{Speech} or not. Works for instances
  or classes.
  
  @deprecated: only here until Task.Tools.Output is refactored
  @note: We'd like to use Speech as an interface, but we currently require 
  that all classes derive from it, not just implement it.
  @param obj: Any object to test
  @type obj: object
  @return: True if the object derives from L{Speech}, False if not
  '''
  try:
    return issubclass(obj, Speech)
  except TypeError:
    return isinstance(obj, Speech)