'''
Defines an abstract base class for all persistent and/or configurable setting
classes.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import Setting

class AEState(object):
  '''
  Abstract base class for objects containing data that will be serialized to 
  disk or configured by the user. The L{getSettings} method must be overridden
  to support user configuration. The L{getNonPersistent} and/or 
  L{initNonPersistent} methods might be overridden to customize persistence.
  '''
  def __getstate__(self):
    '''
    Removes all non-persistent instance variables from a copy of the instance
    dictionary before persisting the instance.

    @return: Values to persist
    @rtype: dictionary
    '''
    try:
      names = self.getNonPersistent()
    except NotImplementedError:
      return self.__dict__
    d = self.__dict__.copy()
    for name in names:
      del d[name]
    return d
  
  def __setstate__(self, dict):
    '''
    Restores the state of this object using the given dictionary of values. 
    Calls L{initNonPersistent} to allow the object to reinitialize all 
    non-persistent instance variables.
    
    @param dict: Values to restore
    @type dict: dictionary
    '''
    self.__dict__.update(dict)
    try:
      self.initNonPersistent()
    except NotImplementedError:
      pass
  
  def _newGroup(self, name=None):
    '''
    Creates a new L{AEState.Setting.Group} object with the given name. 
    Associates the group with this instance so that it may modify variables in
    this instance.
    
    @param name: Name of the group
    @type name: string
    @return: New group object
    @rtype: L{AEState.Setting.Group}
    '''
    return Setting.Group(self, name)
  
  def initNonPersistent(self):
    '''
    Re-initializes non-persistent instance variables when this object is 
    loaded from persistent storage. Not required if this class will not be
    persisted to disk or has no non-persistent instance variables to 
    reinitialize.
    '''
    raise NotImplementedError
  
  def getNonPersistent(self):
    '''
    Gets the names of the instance variables in this object that should not be
    persisted to disk with the object. Anything not listed here will be written
    to disk. Not required if this class will not be persisted to disk or has
    no non-persistent instance variables.
    
    @return: List of names of instance variables
    @rtype: list of string
    @raise NotImplementedError: When not implemented in a subclass
    '''
    raise NotImplementedError
    
  def getSettings(self):
    '''    
    Gets a L{AEState.Setting.Group} of L{Setting} objects representing the 
    values that should be configurable by the user. L{Setting} objects are used
    because they provide additional metadata useful in constructing
    configuration dialogs.
    
    The L{_newGroup} method can be used to instantiate the root group. Methods
    in that group can then be used to instantiate L{Setting}s.
    
    Any changes made to the settings objects will be immediately reflected in
    the instance variables in this object. Use L{AEState.Setting.Group.save} and 
    L{Setting.Group.restore}.
    
    @return: Collection of L{Setting} objects
    @rtype: L{AEState.Setting.Group}
    @raise NotImplementedError: When not implemented in a subclass
    '''
    raise NotImplementedError