'''
Defines the default user interface for LSR.

@author: Peter Parente
@author: Pete Brunet
@author: Larry Weiss
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import Perk, Task
from POR import POR
from i18n import _

class DefaultPerk(Perk.Perk):
  '''
  Defines a default user interface that makes LSR act like a typical screen 
  reader. Event handlers are registered for focus, view, caret, and state change
  events. Various commands for controling LSR are defined, but not mapped to
  any particular input device.
  
  The following are stored in the L{perk_state} object for reference in all
  classes in this L{Perk}.
  
  @ivar last_caret: Stores the previous caret L{POR}
  @type last_caret: L{POR}
  @ivar last_role: Stores the previous role to avoid duplicate announcements
  @type last_role: string
  @ivar last_level: Stores the previous level to avoid duplicate announcements
  @type last_level: integer
  '''
  def init(self):
    '''
    Registers L{HandleFocusChange}, L{HandleViewChange}, L{HandleCaretChange},
    and L{HandleSelectorChange} objects to handle the associated events.
    Registers named L{Task}s that can be mapped to input gestures.
    '''
    # create varibles referenced through self.perk
    self.resetLasts()
    # register event handlers
    self.registerEventTask(HandleFocusChange(self))
    self.registerEventTask(HandleViewChange(self))
    self.registerEventTask(HandleCaretChange(self))
    self.registerEventTask(HandleSelectorChange(self))
    self.registerEventTask(HandleStateChange(self))
    # register named tasks
    # these will be mapped to input gestures in other Perks
    self.registerNamedTask(Stop(self), 'Stop now')
    self.registerNamedTask(Mute(self), 'Mute/Unmute now')
    self.registerNamedTask(IncreaseRate(self), 'Increase rate')
    self.registerNamedTask(DecreaseRate(self), 'Decrease rate')
    self.registerNamedTask(PreviousItem(self), 'Previous item')
    self.registerNamedTask(CurrentItem(self), 'Current item')
    self.registerNamedTask(NextItem(self), 'Next item')
    self.registerNamedTask(PreviousWord(self), 'Previous word')
    self.registerNamedTask(CurrentWord(self), 'Current word')
    self.registerNamedTask(NextWord(self), 'Next word')
    self.registerNamedTask(PreviousChar(self), 'Previous char')
    self.registerNamedTask(CurrentChar(self), 'Current char')
    self.registerNamedTask(NextChar(self), 'Next char')
    self.registerNamedTask(WhereAmI(self), 'Where am I')
    self.registerNamedTask(FocusToPointer(self), 'Focus to pointer')
    self.registerNamedTask(PointerToFocus(self), 'Pointer to focus')
    self.registerNamedTask(ReadTop(self), 'Read top')
    self.registerNamedTask(ReadTextAttributes(self), 'Read text attributes')
    
  def sayNewMenu(self, task, por=None):
    '''
    Speaks the text of a menu at the given L{POR} or at the L{task_por} if no
    L{POR} is given only if it is different than the last menu spoken by this
    method.
    
    @param task: The task to use for calling L{Task.Tools} methods
    @type task: L{Task.Base}
    @param por: Point of regard to an accessible that should be announced if it
      is a new menu.
    @type por: L{POR}
    @return: Was a new role announced?
    @rtype: boolean
    '''
    if task.hasAccRole('menu', por):
      self.perk_state.last_menu = por
      return False
    por = task.getParentAcc(por)
    if task.hasAccRole('menu', por) and por != self.perk_state.last_menu:
      self.sayNewRole(task, por)
      task.sayItem(por)
      self.perk_state.last_menu = por
      return True
    return False
       
  def sayNewRole(self, task, por=None):
    '''
    Speaks the role of the given L{POR} or the L{task_por} if no L{POR} is given
    only if it is different than the last role spoken by this method.
    
    @param task: The task to use for calling L{Task.Tools} methods
    @type task: L{Task.Base}
    @param por: Point of regard to the accessible whose role should be said, 
      defaults to L{task_por} if None
    @type por: L{POR}
    @return: Was a new role announced?
    @rtype: boolean
    '''
    role = task.getAccRoleName(por)
    if role != self.perk_state.last_role:
      task.sayRole(text=role)
      self.perk_state.last_role = role
      return True
    return False
  
  def sayNewLevel(self, task, por=None):
    '''
    Speaks the level of the given L{POR} or the L{task_por} if no L{POR} is 
    given only if it is different than the last level spoken by this method.
    
    @param task: The task to use for calling L{Task.Tools} methods
    @type task: L{Task.Base}
    @param por: Point of regard to the accessible whose role should be said, 
      defaults to L{task_por} if None
    @type por: L{POR}
    @return: Was a new level announced?
    @rtype: boolean
    '''
    level = task.getAccLevel(por)
    if level is not None and level != self.perk_state.last_level:
      task.sayLevel(text=level, template=_('Level %d'))
      self.perk_state.last_level = level
      return True
    return False
  
  def resetLasts(self):
    '''
    Resets variables tracking last announced role, level, and menu as well as
    the last caret position.
    '''
    self.perk_state.last_caret = POR(item_offset=0)
    self.perk_state.last_role = None
    self.perk_state.last_level = None
    self.perk_state.last_menu = POR()

#
# Tasks executed by keys
#

class Mute(Task.InputTask):
  '''
  Task to mute output indefinitely or unmute it if it has already been muted.
  '''
  def execute(self, **kwargs):
    self.stopAll()
    mute = self.getMute() 
    if mute:
      self.setMute(False)
      self.sayInfo(text=_('Unmuted'))
    else:
      self.sayInfo(text=_('Muted'))
      self.setMute(True)

class Stop(Task.InputTask):
  '''
  Task to stop speech immediately, ignoring the value of the Stopping setting.
  '''
  def execute(self, **kwargs):
    self.stopAll()

class IncreaseRate(Task.InputTask):
  '''
  Increase the speech rate. The maximum rate is announced when reached.

  @see: L{DecreaseRate}
  '''
  def execute(self, **kwargs):
    self.stopNow()
    rate = self.getRate()
    maxRate = self.getMaxRate()
    if rate < maxRate:
      self.setRate(rate+10)
      self.msg(_('rate %i') % (rate+10))
    else:
      self.msg(_('maximum rate'))

class DecreaseRate(Task.InputTask):
  '''
  Decrease the speech rate. The minimum rate is announced when reached.

  @see: L{IncreaseRate}
  '''
  def execute(self, **kwargs):
    self.stopNow()
    rate = self.getRate()
    # don't allow rate < 70 WPM for sanity
    if rate > 70:
      self.setRate(rate-10)
      self.msg(_('rate %i') % (rate-10))
    else:
      self.msg(_('minimum rate'))
      
class PreviousItem(Task.InputTask):
  '''
  Moves the POR to the beginning of the previous Item and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    self.moveToPrevItem()
    self.perk.sayNewRole(self)
    self.perk.sayNewLevel(self)
    self.sayItem()
    self.sayState()

class CurrentItem(Task.InputTask):
  '''
  Moves the POR to the beginning of the current Item and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    self.moveToCurrItem()
    self.sayItem()
    self.sayState()

class NextItem(Task.InputTask):
  '''
  Moves the POR to the beginning of the next Item and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    self.moveToNextItem()
    self.perk.sayNewRole(self)
    self.perk.sayNewLevel(self)
    self.sayItem()
    self.sayState()
    
class PreviousWord(Task.InputTask):
  '''
  Moves the POR to the beginning of the previous word and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    moved = self.moveToPrevWord()
    if moved:
      self.perk.sayNewRole(self)
      self.perk.sayNewLevel(self)
    self.sayWord()
    if moved:
      self.perk.sayState(self)

class CurrentWord(Task.InputTask):
  '''
  Moves the POR to the beginning of the current word and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    self.moveToCurrWord()
    self.sayWord()

class NextWord(Task.InputTask):
  '''
  Moves the POR to the beginning of the next word and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    moved = self.moveToNextWord()
    if moved:
      self.perk.sayNewRole(self)
      self.perk.sayNewLevel(self)
    self.sayWord()
    if moved:
      self.sayState()
    
class PreviousChar(Task.InputTask):
  '''
  Moves the POR to the beginning of the previous word and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    moved = self.moveToPrevChar()
    if moved:
      self.perk.sayNewRole(self)
      self.perk.sayNewLevel(self)
    self.sayChar()
    if moved:
      self.sayState()

class CurrentChar(Task.InputTask):
  '''
  Moves the POR to the beginning of the current word and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    self.moveToCurrChar()
    self.sayChar()

class NextChar(Task.InputTask):
  '''
  Moves the POR to the beginning of the next word and speaks it.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    moved = self.moveToNextChar()
    if moved:
      self.perk.sayNewRole(self)
      self.perk.sayNewLevel(self)
    self.sayChar()
    if moved:
      self.sayState()
    
class WhereAmI(Task.InputTask):
  '''
  Reports the location of the current L{task_por} by announcing all control and
  container names in child-to-parent order.
  '''
  def execute(self, **kwargs):
    # stop now and indicate the next stop is fine
    self.stopNow()
    pors = [self.task_por] + self.getAncestorAccs()
    for curr in pors[:-1]:
      # announce role and text
      self.sayRole(curr)
      self.sayItem(curr)
    # announce window title in its appropriate semantic
    curr = pors[-1]
    self.sayRole(curr)
    self.sayWindow(curr)
    # also announce the name of the application
    self.sayRole(text=_('application'))
    self.sayApp()

class FocusToPointer(Task.InputTask):
  '''
  Attempts to switch focus to try to set focus to L{POR} of accessible that was
  last reviewed in passive mode.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    self.setFocus()
    
class PointerToFocus(Task.InputTask):
  '''
  Jumps the L{task_por} back to the L{POR} of the last focus event.
  '''
  def execute(self, **kwargs):
    # set the Task POR to the focus POR
    self.task_por = self.getFocus()
    por = self.task_por
    # stop all output and pretend a focus event happened to give context
    self.mayStop()
    # announce the label when present
    self.sayLabel(por)
    # announce parent menus when changing menus
    self.perk.sayNewMenu(self, por)
    # only announce role when it has changed
    self.perk.sayNewRole(self, por)
    # say the item text too
    self.sayItem(por)

class ReadTop(Task.InputTask):
  '''
  Read top of the view, typically the title of the foreground window.
  '''
  def execute(self, **kwargs):
    self.stopNow()
    self.sayWindow()
      
class ReadTextAttributes(Task.InputTask):
  '''
  Read font attributes (size, style, color, etc).
  '''             
  def execute(self, **kwargs):
    self.stopNow()
    self.sayTextAttrs()
      
# Tasks executed by events
#

class HandleViewChange(Task.ViewTask):
  '''
  Task that handles a L{AEEvent.ViewChange}.
  
  @todo: PP: Pause before saying 'No view'
  '''
  def executeLost(self, por, title, **kwargs):  
    #self.stopNow()
    #self.msg(_('No view'))
    return True
  
  def executeGained(self, por, title, **kwargs):
    '''
    Announces the title of the newly activated window. Inhibits the next stop
    to avoid never announcing the title because of another immediate event
    following this one (i.e. focus).
    
    @param por: Point of regard to the root of the new view
    @type por: L{POR}
    @param title: Title of the newly activated window
    @type title: string
    '''
    # trying stopping output
    self.stopNow()
    # say the title of the new view
    if not self.sayWindow(text=title):
      self.sayWindow(text=_('No title!'))
    # prevent the next focus event from stopping this announcement
    self.inhibitMayStop()
    # reset stateful vars
    self.perk.resetLasts()
    return True
    
class HandleFocusChange(Task.FocusTask):
  '''  
  Task that handles a L{AEEvent.FocusChange}.
  '''  
  def executeGained(self, por, **kwargs):
    '''
    Annouces the label and role of the newly focused widget. The role is output
    only if it is different from the last one announced.
    
    @param por: Point of regard for the new focus
    @type por: L{POR}
    '''
    self.mayStop()
    # announce the label when present if it is not the same as the name
    label = self.getAccLabel(por)
    if label and label != self.getItemText(por):
      self.sayLabel(por)
    # announce parent menus when changing menus
    self.perk.sayNewMenu(self, por)
    # only announce role when it has changed
    self.perk.sayNewRole(self, por)
    # prevent the next selector or caret from stopping this announcement
    self.inhibitMayStop()
    # always reset last caret position
    self.perk_state.last_caret = POR(item_offset=0)
    return True

class HandleSelectorChange(Task.SelectorTask):
  '''  
  Task that handles a L{AEEvent.SelectorChange}.
  '''  
  def executeActive(self, por, text, **kwargs):
    '''
    Announces the text of an actively selected item.
    
    @param por: Point of regard where the selection event occurred
    @type por: L{POR}
    @param text: The text of the selected item
    @type text: string
    '''
    self.mayStop()
    # only announce role when it has changed
    self.perk.sayNewRole(self, por)
    # try to announce the level
    self.perk.sayNewLevel(self, por)
    # announce the selected text
    self.sayItem(text=text)
    # see if there are any states worth reporting
    self.sayState(por)
    return True

class HandleCaretChange(Task.CaretTask):
  '''  
  Task that handles a L{AEEvent.CaretChange}. The L{execute} method performs
  actions that should be done regardless of whether text was inserted or deleted
  or the caret moved. The other three methods handle these more specific cases.
  
  @todo: PP: think about semantic here for insert, move, and delete; is it echo?
    is it char/word/item?
  '''  
  def execute(self, por, text, text_offset, added, **kwargs):
    '''
    Stores the point of regard to the caret event location. Tries to stop
    current output.
    
    @param por: Point of regard where the caret event occurred
    @type por: L{POR}
    @param text: The text inserted, deleted or the line of the caret
    @type text: string
    @param text_offset: The offset of the inserted/deleted text or the line 
      offset when movement only
    @type text_offset: integer
    @param added: True when text added, False when text deleted, and None 
      (the default) when event is for caret movement only
    @type added: boolean
    '''
    if not self.mayStop() and por.isSameAcc(self.perk_state.last_caret):
      # opt: bail out if we're trying to queue some other text event
      return True
    # let the base class call the appropriate method for inset, move, delete
    rv = Task.CaretTask.execute(self, por, text, text_offset, added, **kwargs)
    # store for next comparison
    self.perk_state.last_caret = por
    return rv
    
  def executeInserted(self, por, text, text_offset, **kwargs):
    '''
    Announces the inserted text.
    
    @param por: Point of regard where the caret event occurred
    @type por: L{POR}
    @param text: The text inserted
    @type text: string
    @param text_offset: The offset of the inserted text
    @type text_offset: integer
    '''
    # inhibit the next stop
    self.inhibitMayStop()
    self.sayItem(text=text)
    return True
    
  def executeMoved(self, por, text, text_offset, **kwargs):
    '''
    Announces the text passed in the caret movement.
    
    @param por: Point of regard where the caret event occurred
    @type por: L{POR}
    @param text: The text passed during the move
    @type text: string
    @param text_offset: The offset of the new caret position
    @type text_offset: integer
    '''
    # say line when item_offset changes
    if (not por.isSameItem(self.perk_state.last_caret) or
        not por.isSameAcc(self.perk_state.last_caret)):
      self.sayItem(text=text)
    # respond to caret movement on same line
    else: 
      # moved left, say text between positions
      if por.isCharBefore(self.perk_state.last_caret):
        self.sayItem(text=text[por.char_offset : 
                     self.perk_state.last_caret.char_offset])
      # moved right, say text between positions
      elif por.isCharAfter(self.perk_state.last_caret):
        self.sayItem(text=text[self.perk_state.last_caret.char_offset : 
                      por.char_offset])
    return True
    
  def executeDeleted(self, por, text, text_offset, **kwargs):
    '''
    Announces the deleted text. Prepends delete and backspace text to indicate
    which way the caret moved.
    
    @param por: Point of regard where the caret event occurred
    @type por: L{POR}
    @param text: The text delete
    @type text: string
    @param text_offset: The offset of the delete text
    @type text_offset: integer
    '''   
    if (self.perk_state.last_caret.item_offset + 
        self.perk_state.last_caret.char_offset) > text_offset:
      # inhibit the next stop
      self.inhibitMayStop()
      # say backspace if we're backing up
      self.sayItem(text=text, template=_('backspace %s'))
    else:
      # say delete if we're deleting the current
      self.sayItem(text=text, template=_('delete %s'))
    return True

  def update(self, por, text, text_offset, added, **kwargs):
    '''
    Updates the L{DefaultPerk}.last_caret with the current L{por}.
    
    @param por: Point of regard where the caret event occurred
    @type por: L{POR}
    @param text: The text inserted, deleted or the line of the caret
    @type text: string
    @param text_offset: The offset of the inserted/deleted text or the line 
      offset when movement only
    @type text_offset: integer
    @param added: True when text added, False when text deleted, and None 
      (the default) when event is for caret movement only
    @type added: boolean
    '''
    # store for next comparison
    self.perk_state.last_caret = por
  
class HandleStateChange(Task.StateTask):
  '''  
  Task that handles a L{AEEvent.StateChange}.
  '''
  def execute(self, por, name, value, **kwargs):
    '''
    Announces state changes according to the ones of interest defined by
    L{getStateText}.
    
    @param por: Point of regard where the state change occurred
    @type por: L{POR}
    @param name: Name of the state that changed
    @type name: string
    @param value: Is the state set (True) or unset (False)?
    @type value: boolean
    '''
    text = self.getStateText(por, name, value)
    if text:
      # try a may stop
      self.mayStop()
      self.sayState(text=text)
    return True
