'''
Defines a L{Task} to execute when a L{AEEvent.InputGesture} occurs in order to
start an L{AEChooser} dialog with the user.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import InputTask, Constants
from AEChooser import AEChooser

class ChooserTask(InputTask.InputTask):
  '''
  Executes when an input gesture occurs or a chooser changes state.

  This class registers its name and whether it should be monitored by default
  in an L{AEMonitor.EventMonitor} using the L{Constants.registerTaskType}
  function when this module is first imported. The L{AEMonitor.EventMonitor}
  can use this information to build its menus.
  
  All class variables are replicated from L{AEChooser.Base.AEChooser} for 
  advanced L{Perk} writer convenience.
  '''
  OK = AEChooser.OK
  APPLY = AEChooser.APPLY
  CANCEL = AEChooser.CANCEL
  Constants.registerTaskType('InputTask', True)

  def execute(self, chooser=None, kind=None, **kwargs):
    '''
    Executes this L{Task} in response to input on some L{AEInput} device. Called 
    by L{Tier.Tier._executeTask}.
    
    @return: Should processing continue? Always returns True by default.
    @rtype: boolean
    '''
    if chooser is None:
      return self.executeInput(**kwargs)
    elif kind == self.OK:
      return self.executeOK(chooser=chooser, **kwargs)
    elif kind == self.APPLY:
      return self.executeApply(chooser=chooser, **kwargs)
    else:
      return self.executeCancel(chooser=chooser, **kwargs)

  def executeInput(self, **kwargs):
    '''
    Executes this L{Task} in response to input on some L{AEInput} device. 
    Called by L{Tier.Tier._executeTask}.
    
    @return: Should processing continue? Always returns True by default.
    @rtype: boolean
    '''
    return True
  
  def executeOK(self, chooser, **kwargs):
    '''
    Executes this L{Task} in response to a chooser completion. Called by
    L{Tier.Tier._executeTask}.

    @param chooser: Chooser that fired the event
    @type chooser: L{AEChooser}
    @param kwargs: Arbitrary keyword arguments returned by the chooser
    @type kwargs: dictionary
    '''
    return True
  
  def executeCancel(self, chooser, **kwargs):
    '''
    Executes this L{Task} in response to a chooser cancellation. Called by
    L{Tier.Tier._executeTask}.

    @param chooser: Chooser that fired the event
    @type chooser: L{AEChooser}
    @param kwargs: Arbitrary keyword arguments returned by the chooser
    @type kwargs: dictionary
    '''
    return True
  
  def executeApply(self, chooser, **kwargs):
    '''  
    Executes this L{Task} in response to a chooser request to have its current
    options immediately applied with its completion. Called by
    L{Tier.Tier._executeTask}.
    
    @param chooser: Chooser that fired the event
    @type chooser: L{AEChooser}
    @param kwargs: Arbitrary keyword arguments returned by the chooser
    @type kwargs: dictionary
    '''
    return True
