'''
Defines L{Tools} for registering, unregistering, introspecting, and otherwise
manging L{Tier}s, L{Perk}s, and L{Task}s.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import Base
import LSRConstants, UIRegistrar

class System(Base.TaskTools):
  '''
  Provides methods for managing L{Tier}s, L{Perk}s, and L{Task}s.
  '''
  def getDescription(self):
    '''
    @return: Description of this task. Defaults to the first paragraph (all 
      lines up to the first blank line) in its class docstring.
    @rtype: string
    '''
    desc = []
    for line, i in enumerate(self.__class__.__doc__.split('\n')):
      if i != 0 and not line.strip():
        break
      else:
        desc.append(line)
    return ' '.join(desc)
    
  def getName(self):
    '''
    @return: Name of this task. Defaults to its class name.
    @rtype: string
    '''
    return str(self.__class__.__name__)  
  
  def pushPerk(self, *perks):
    '''
    Adds one or more L{Perk}s to the top of the stack. If more than one L{Perk}
    is specified, the last specified L{Perk} will be at the top of the stack 
    when this method completes. That is, the behavior of pushing more than one
    L{Perk} at a time is the same as if each L{Perk} were pushed individually.
    
    @param perks: L{Perk}s to add
    @type perks: list of L{Perk}
    '''
    self.tier.pushPerk(self.acc_eng, *perks)
  
  def popPerk(self, *perks):
    '''
    Removes one or more L{Perk}s from the stack.
    
    @param perks: L{Perk}s to remove
    @type perks: list of L{Perk}
    @raise IndexError: When the stack is empty
    @raise ValueError: When specified L{Perk} is not found
    '''
    self.tier.popPerk(*perks)
    
  def reloadPerks(self):
    '''
    Reloads all L{Perk}s in the current L{Tier}. Any changes made to the 
    L{Perk} logic will take effect immediately after the reload. This method
    does not refer to the L{Perk}s associated with the current profile. It 
    simply reloads all currently loaded L{Perk}s.
    '''
    reg = UIRegistrar.UIRegistrar()
    # get official registered Perk names for all existing Perks in this Tier
    names = [perk.__class__.__name__ for perk in self.tier.getPerks()]
    names.reverse()
    # load new copies of all existing Perks
    perks = [reg.loadOne(UIRegistrar.PERK, name) for name in names]
    # clear out existing Perks
    self.tier.clearPerks()
    # push all new Perks
    self.pushPerk(*perks)
    
  def getPerkNames(self):
    '''
    Gets the names of all L{Perk}s in this L{Tier} in the order in which they
    are executed.
    
    @return: Names of all L{Perk}s
    @rtype: list of string
    '''
    return [perk.getName() for perk in self.tier.getPerks()]
  
  def registerState(self, state):
    '''
    Registers a new L{AEState} object that will store user configurable and/or
    persistable settings for the L{Perk} that invoked this method.
    
    @param state: New state object
    @type state: L{AEState.AEState}
    '''
    self.perk.registerState(state)

  def registerCommandTask(self, device, codes, name, propogate=False):
    '''
    Registers a L{Task} in this L{Perk} to be executed in response to an 
    L{AEEvent} indicating that the given action codes were input on the given 
    L{AEInput} device. 
    
    @param device: Input device to monitor
    @type device: L{AEInput}
    @param codes: List of lists of action codes forming the L{AEInput.Gesture} 
      that will trigger the execution of the named L{Task}. For example, 
      codes=[[Keyboard.AEK_CTRL, Keyboard.AEK_TILDE]] indicates the single
      gesture of simultaneously pressing Ctrl and ~ on the keyboard device.
    @type codes: list of list of integer
    @param name: Name of the L{Task} registered via L{registerNamedTask} to 
      execute when the input gesture is detected on the device
    @type name: string
    @param propogate: Should the input gesture be allowed to propogate to the 
      OS after we receive it?
    @type propogate: boolean
    @raise ValueError: When a L{Task} with the given name is not registered
    '''
    self.perk.registerCommandTask(device, codes, name, propogate)
    
  def registerNamedTask(self, task, name):
    '''
    Registers a new L{Task} under the given name if no L{Task} is already 
    registered with that name in this L{Tier}.
    
    Only one L{Task} can be registered under a name in a L{Tier}. If a L{Task} 
    is already registered under the given name, any other registration with that
    name is ignored.
    
    @param task: L{Task} to register
    @type task: L{Task.Base.Task}
    @param name: Name to associate with the L{Task}
    @type name: string
    @raise ValueError: When a L{Task} with the given name is already registered
      in this L{Tier}
    '''
    self.perk.registerNamedTask(task, name)
    
  def registerEventTask(self, task, focus=False, tier=False, background=False):
    '''
    Registers a new L{Task} in this L{Perk} under the L{Task} type. The type
    determines which kind of L{AEEvent} will trigger the execution of the
    registered L{Task}. If one or more Tasks are already registered for this 
    type, the given L{Task} will be inserted at the top of the registered stack
    of L{Task}s (i.e. it will be executed first for the appropriate event).
    
    The L{focus}, L{tier}, and L{background} parameters specify on which layer
    the L{Task} will handle events. If focus is True, the L{Task} will be 
    executed in response to an event from a focused control within this L{Tier}.
    If tier is True, the L{Task} will be executed in response to an event from
    an unfocused control within this L{Tier}. If background is True, the L{Task}
    will be executed in response to an event from any control within the tier 
    when the L{Tier} is not active.
    
    The three layers are mutually exclusive. You may set any combination of
    focus, tier, and background to True to register the given L{task} on each
    selected layer in one call. If all three parameters are False, the 
    registration defaults to the focus layer.
    
    The L{Task} passed to this method must implement L{Task.Base.Task.getType}.

    @param task: L{Task} to register
    @type task: L{Task.Base.Task}
    @param focus: Should this L{Task} handle events from focused accessibles in 
      this L{Tier}?
    @type focus: boolean
    @param tier: Should this L{Task} handle events from unfocused accessibles in 
      this L{Tier}?
    @type tier: boolean
    @param background: Should this L{Task} handle events from any accessible in 
      this L{Tier} when the L{Tier} is inactive?
    @type background: boolean
    @raise NotImplementedError: When the L{Task} does not implement 
      L{Task.Base.Task.getType}
    '''
    self.perk.registerEventTask(task, focus, tier, background)
    
  def unregisterCommandTask(self, device, codes):
    '''
    Unregisters a L{Task} set to execute in response to the given action codes 
    on the given device B{from this L{Perk} only}.
    
    @param device: Input device to monitor
    @type device: L{AEInput}
    @param codes: List of lists of action codes forming the L{AEInput.Gesture} 
      that will trigger the execution of the named L{Task}. For example, 
      codes=[[Keyboard.AEK_CTRL, Keyboard.AEK_TILDE]] indicates the single
      gesture of simultaneously pressing Ctrl and ~ on the keyboard device.
    @type codes:  list of list of integer
    @raise KeyError: When a L{AEInput.GestureList} is not registered
    '''
    self.perk.unregisterCommandTask(self, device, codes)
    
  def unregisterNamedTask(self, name):
    '''
    Unregisters a named L{Task} B{from this L{Perk} only}. If a Task with the 
    given name is not found in this L{Perk}, an exception is raised.
    
    @param name: Name of the L{Task} to unregister
    @type name: string
    @raise KeyError: When a L{Task} with the given name is not registered
    '''
    del self.named_tasks[name]
    
  def unregisterEventTask(self, task, focus=False, tier=False,
                          background=False):
    '''
    Unregisters the given L{Task} instance B{from this L{Perk} only}. If the 
    given L{Task} instance was not registered for an event in this L{Perk}, an 
    exception is raised. The L{focus}, L{tier}, and L{background} parameters
    state from which layer(s) this L{Task} should be unregistered.
    
    The L{Task} passed to this method must implement L{Task.Base.Task.getType}.
    
    @param task: L{Task} to unregister
    @type task: L{Task.Base.Task}
    @raise KeyError: When there are no L{Task}s registered with the type of the 
        given L{Task}
    @raise ValueError: When the given L{Task} is not registered on one of the
      specified layers
    @raise NotImplementedError: When the L{Task} does not implement 
      L{Task.Base.Task.getType}
    @see: L{registerEventTask}
    '''
    self.perk.unregisterEventTaskLocal(task, focus, tier, background)
    
  def getTierNamedTask(self, name):
    '''
    Gets the L{Task} with the given name if it is registered anywhere in the
    current L{Tier}. If no L{Task} is registered under the given name, returns 
    None.
    
    Calls L{Tier.Tier.getNamedTask} to search across all L{Perk}s in the owning
    L{Tier}.
    
    @param name: Name of the L{Task} to locate
    @type name: string
    @return: L{Task} with the given name or None
    @rtype: L{Task.Base.Task}
    '''
    return self.tier.getNamedTask(name)
    
  def getTierEventTasks(self, event_type, task_layer):
    '''
    Gets all registered L{Task}s registered to handle the given L{AEEvent} type
    on the given layer from the current L{Tier}. 
    
    Calls L{Tier.Tier.getEventTasks} to search across all L{Perk}s in the 
    owning L{Tier}.  
    
    @param event_type: Desired type of L{AEEvent}
    @type event_type: L{AEEvent} class
    @param task_layer: Layer on which the desired L{Task}s are registered, one 
      of L{Task.FOCUS_LAYER}, L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type task_layer: integer
    @return: List of all L{Task}s that handle the given event type and on 
      the given layer in the L{Tier} which owns this L{Perk}
    @rtype: list of L{Task.Base.Task}
    '''
    return self.tier.getEventTasks(event_type, task_layer)
  
  def getNamedTask(self, name):
    '''
    Gets the L{Task} with the given name if it is registered B{in this L{Perk}
    only}. If no L{Task} is registered under the given name in this L{Perk}, 
    returns None.
    
    @param name: Name of the L{Task} to locate
    @type name: string
    @return: L{Task} with the given name or None
    @rtype: L{Task.Base.Task}
    @see: L{getTierNamedTask}
    '''
    return self.perk.getNamedTask(name)
    
  def getEventTasks(self, event_type, task_layer):
    '''
    Get all registered L{Task}s registered to handle the given L{AEEvent} type 
    B{in this L{Perk} only}.
    
    @param event_type: Desired type of L{AEEvent}
    @type event_type: L{AEEvent} class
    @param task_layer: Layer on which the desired L{Task}s are registered, one 
      of L{Task.FOCUS_LAYER}, L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type task_layer: integer
    @return: List of all L{Task}s that handle the given event type on the given
      layer in this L{Perk}
    @rtype: list of L{Task.Base.Task}
    @see: L{getTierEventTasks}
    '''
    return self.perk.getEventTasks(event_type, task_layer)
  
  def loadMonitors(self):
    '''
    Loads and shows all L{AEMonitor}s associated with this profile.
    
    @return: Was at least one monitor loaded?
    @rtype: boolean
    '''
    loaded = False
    reg = UIRegistrar.UIRegistrar()
    mons = reg.loadAssociated(UIRegistrar.MONITOR, self.acc_eng.getProfile())
    if not len(self.event_man.getMonitors()):
      self.event_man.addMonitors(*mons)
      loaded = True
    if not len(self.tier_man.getMonitors()):
      self.tier_man.addMonitors(*mons)
      loaded = True
    return loaded
  
  def unloadMonitors(self):
    '''
    Hides and unloads all L{AEMonitor}s associated with this profile.
    
    @return: Was at least one monitor unloaded?
    @rtype: boolean
    '''
    unloaded = False
    if len(self.event_man.getMonitors()):
      self.event_man.removeAllMonitors()
      unloaded = True
    if len(self.tier_man.getMonitors()):
      self.tier_man.removeAllMonitors()
      unloaded = True
    return unloaded