'''
Defines a class unifying navigation over all accessibles and their items.

@todo: PP: need robust strategy for handling cycles (mark nodes as visited 
  somehow?)

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import Base
from LSRInterfaces import *

class AccessibleItemWalker(Base.Walker):
  '''
  Walks the accessible hierarchy by accessibles and their items returning
  L{POR}s. The walk order is a pre-order traversal of the subtree of the
  accessible hierarchy rooted at a top-level window accessible. The subtree is
  assumed to have no loops, though logic could be added to detect them. The
  traversal is stateless so that the walker can begin at any node in the tree
  and walk in any direction.

  Accessibles that are not visibile are skipped along with their items and their
  children. Accessibles that are trivial are skipped along with their items, but
  their child accessibles are still considered. This class uses the
  L{LSRInterfaces} to leave the decision of what constitutes an item, what
  constitutes a trivial accessible, and what constitutes and invisible
  accessible to the objects that implement the interfaces.
  
  @ivar allow_invisible: Stop on invisible L{POR}s as well as visible?
  @type allow_invisible: boolean
  @ivar allow_trivial: Stop on trivial L{POR}s as well as non-trivial?
  @type allow_trivial: boolean
  @ivar por: The starting L{POR} for the L{Walker}
  @type por: L{POR}
  '''
  def __init__(self, por, only_visible=True, allow_trivial=False):
    '''
    Stores the starting L{POR}.
    
    @param only_visible: Stop only on visible L{POR}s?
    @type only_visible: boolean
    @param allow_trivial: Stop on trivial L{POR}s as well as non-trivial?
    @type allow_trivial: boolean
    @param por: The current L{POR} for the L{Walker}
    @type por: L{POR}
    '''
    self.allow_invisible = not only_visible
    self.allow_trivial = allow_trivial
    self.por = por
    
  def _getNextItem(self, por):
    '''
    Gets the next item in the current accessible in the given L{POR}. Returns
    that next item if it exists. If not or if the accessible in the given L{POR}
    is not visible, returns the current L{POR} and the L{_getFirstChild} as the 
    method to call to continue the search.
    
    @param por: Initial L{POR}
    @type por: L{POR}
    @return: L{POR} and the next method to call to continue the search, or 
      L{POR} and None to indicate the search is complete
    @rtype: 2-tuple of L{POR}, callable
    '''
    if self.allow_invisible or IAccessibleInfo(por).isAccVisible():
      try:
        # try to get the next item
        item = IItemNav(por).getNextItem()
        return (item, None)
      except (LookupError, IndexError):
        # there's no next item
        pass
    return (por, self._getFirstChild)
            
  def _getFirstChild(self, por):
    '''    
    Gets the first child of the current accessible in the given L{POR}. Returns
    the child accessible if it exists, is visible, and is not trivial. If it
    does not exist or is invisible, returns the given L{POR} and L{_getNextPeer}
    as the method to call to continue the search. If it is trivial, returns the
    child accessible and L{_getFirstChild} as the method to call to continue the
    search.
    
    @param por: Initial L{POR}
    @type por: L{POR}
    @return: L{POR} and the next method to call to continue the search, or 
      L{POR} and None to indicate the search is complete
    @rtype: 2-tuple of L{POR}, callable
    '''
    try:
      # try to get first child accessible
      child = IAccessibleNav(por).getFirstAccChild()
      ai = IAccessibleInfo(child)
      if not self.allow_invisible and not ai.isAccVisible():
        # don't use this child or its children if it is not visible
        return (child, self._getNextPeer)
      elif not self.allow_trivial and ai.isAccTrivial():
        # skip this child and move a level deeper if it is trivial
        return (child, self._getFirstChild)
      else:
        # use this child
        return (child, None)
    except (LookupError, IndexError):
      # there's no child accessible
      return (por, self._getNextPeer)
  
  def _getNextPeer(self, por):
    '''    
    Gets the next peer of the current accessible in the given L{POR}. Returns
    the peer accessible if it exists, is visible, and is not trivial. If it does
    not exist, returns the given L{POR} and L{_getParentNextPeer} as the method
    to call to continue the search. If it is invisible, returns the peer L{POR}
    and L{_getNextPeer} as the method to call to continue the search. If it is
    trivial, returns the peer and L{_getFirstChild} as the method to call to
    continue the search. If it is trivial, returns the child accessible and
    L{_getFirstChild} as the method to call to continue the search.

    @param por: Initial L{POR}
    @type por: L{POR}
    @return: L{POR} and the next method to call to continue the search, or 
      L{POR} and None to indicate the search is complete
    @rtype: 2-tuple of L{POR}, callable
    '''
    try:
      # try getting the next peer accessible
      next = IAccessibleNav(por).getNextAcc()
      ai = IAccessibleInfo(next)
      if not self.allow_invisible and not ai.isAccVisible():
        # skip to the next peer if this one is not visible
        return (next, self._getNextPeer)
      elif not self.allow_trivial and ai.isAccTrivial():
        # skip this peer and move to its children if it is trivial
        return (next, self._getFirstChild)
      else:
        # use this peer
        return (next, None)
    except (LookupError, IndexError):
      # there's no next peer accessible
      return (por, self._getParentNextPeer)

  def _getParentNextPeer(self, por):
    '''
    Gets the parent accessible of the current accessible in the given L{POR}. 
    Returns the parent and L{_getNextPeer} as the method to call to continue the
    search if the parent exists. Returns a sentinel (None, None) if there is no
    parent indicating the given L{POR} is the root of the subtree containing
    the starting L{POR}.
    
    @param por: Initial L{POR}
    @type por: L{POR}
    @return: L{POR} and the next method to call to continue the search, or 
      L{POR} and None to indicate the search is complete
    @rtype: 2-tuple of L{POR}, callable
    '''
    try:
      parent = IAccessibleNav(por).getParentAcc()
      if IAccessibleInfo(parent).isAccTopLevelWindow():
        # stop if the parent is the root of the active window
        return (None, None)
      else:
        # get the next peer of the parent accessible
        return (parent, self._getNextPeer)
    except (LookupError, IndexError):
      # there's no parent, so bail because we're on the last accessible
      return (None, None)
  
  def _getPrevItem(self, por):
    '''
    Gets the previous item in the current accessible in the given L{POR}.
    Returns that previous item if it exists. If not or if the accessible in the
    given L{POR} is not visible, returns the current L{POR} and the
    L{_getPrevPeer} as the method to call to continue the search.
    
    @param por: Initial L{POR}
    @type por: L{POR}
    @return: L{POR} and the next method to call to continue the search, or 
      L{POR} and None to indicate the search is complete
    @rtype: 2-tuple of L{POR}, callable
    '''
    if self.allow_invisible or IAccessibleInfo(por).isAccVisible():
      try:
        # try to get the previous item
        item = IItemNav(por).getPrevItem()
        return (item, None)
      except (LookupError, IndexError):
        # there's no previous item
        pass
    return (por, self._getPrevPeer)
  
  def _getPrevPeer(self, por):
    '''
    Gets the previous peer of the current accessible in the given L{POR}. If it
    does not exist, returns the given L{POR} and L{_getParent} as the method to
    call to continue the search. If it is not visible, returns the peer
    accessible and L{_getPrevPeer} as the method to call to continue the search.
    Otherwise, returns the peer accessible and L{_getLastChild} as the method to
    call to continue the search.
    
    @param por: Initial L{POR}
    @type por: L{POR}
    @return: L{POR} and the next method to call to continue the search, or 
      L{POR} and None to indicate the search is complete
    @rtype: 2-tuple of L{POR}, callable
    '''
    try:
      # try to get the previous peer
      prev = IAccessibleNav(por).getPrevAcc()
      ai = IAccessibleInfo(prev)
      if ai.isAccTopLevelWindow():
        # stop if the parent is the root of the active window
        return (None, None)
      elif self.allow_invisible or ai.isAccVisible():
        # get the deepest last child
        return (prev, self._getLastChild)  
      else:
        # skip this peer and its children
        return (prev, self._getPrevPeer)
    except (LookupError, IndexError):
      # there's no previous peer accessible
      return (por, self._getParent)

  def _getLastChild(self, por):
    '''
    Gets the last child of the accessible in the given L{POR}. If it does not
    exist, checks if the given L{POR} is invisible or trivial. If so, returns 
    the given L{POR} and L{_getPrevPeer} to continue the search. If not, returns
    a L{POR} to the last item in the given L{POR} as the result. 

    If the last child does exist, checks if it is visible. If so, returns the
    child and L{_getLastChild} to continue the search. If not, returns the
    child and L{_getPrevPeer} to continue the search.
    
    @param por: Initial L{POR}
    @type por: L{POR}
    @return: L{POR} and the next method to call to continue the search, or 
      L{POR} and None to indicate the search is complete
    @rtype: 2-tuple of L{POR}, callable
    '''
    try:
      # try to get the last child
      child = IAccessibleNav(por).getLastAccChild()
      ai = IAccessibleInfo(child)
      if self.allow_invisible or ai.isAccVisible():
        # try for the next deeper last child
        return (child, self._getLastChild)        
      else:
        # try for the previous peer of the invisible child
        return (child, self._getPrevPeer)
    except (LookupError, IndexError):
      ai = IAccessibleInfo(por)
      if ((not self.allow_invisible and not ai.isAccVisible()) or 
           (not self.allow_trivial and ai.isAccTrivial())):
        # try the previous peer of the invisible or trivial accessible
        return (por, self._getPrevPeer)
      else:
        # use the last item in this accessible
        return (self._getLastItem(por), None)
    
  def _getParent(self, por):
    '''
    Gets the parent accessible of the one in the given L{POR}. Returns the last
    item in the parent if it exists. If it does not exist, Returns a sentinel
    (None, None) indicating the given L{POR} is the root of the subtree
    containing the starting L{POR}. If the parent is invisible or trivial,
    returns the parent and L{_getPrevPeer} as the method to call to continue the
    search.
    
    @param por: Initial L{POR}
    @type por: L{POR}
    @return: L{POR} and the next method to call to continue the search, or 
      L{POR} and None to indicate the search is complete, or None and None to
      indicate we're at the root
    @rtype: 2-tuple of L{POR}, callable
    '''
    try:
      # try to get the parent
      parent = IAccessibleNav(por).getParentAcc()
      ai = IAccessibleInfo(parent)
      if ((not self.allow_invisible and not ai.isAccVisible()) or 
          (not self.allow_trivial and ai.isAccTrivial())):
        # skip the parent and move to its previous peer
        return (parent, self._getPrevPeer)
      else:
        # use the parent, but get its last item
        return (self._getLastItem(parent), None)
    except (LookupError, IndexError):
      # there's no parent, so bail because we're at the root
      return (None, None)
    
  def _getLastItem(self, por):
    '''
    Gets the last visible item of the given L{POR}. Returns the given L{POR} if
    any errors occur.
    
    @param por: Initial L{POR}
    @type por: L{POR}
    @return: L{POR} pointing to the last item of the accessible in the given
      L{POR}
    @rtype: L{POR}
    '''
    try:
      return IItemNav(por).getLastItem()
    except LookupError:
      # just use the current por if there is an error
      return por
    
  def getCurrPOR(self):
    '''
    Gets the current L{POR} held by the walker which serves as the anchor for
    the next navigation.
    
    @return: Current L{POR}
    @rtype: L{POR}
    '''
    return self.por

  def getNextPOR(self):
    '''
    Gets the next L{POR} in the walk order. Calls L{_getNextItem}, 
    L{_getFirstChild}, L{_getNextPeer}, and L{_getParentNextPeer} to attempt to 
    get the next valid L{POR}. Each method determines whether the L{POR} is 
    valid as the next L{POR}, and, if not, which call to make next. Each method
    potentially returns a L{POR} and the next method to call to continue the
    search for the next L{POR}.
    
    @return: Next L{POR} or None if this is the last L{POR}
    @rtype: L{POR} or None
    '''
    state = self._getNextItem
    por = self.por
    while state is not None:
      por, state = state(por)
    if por is not None:
      self.por = por
    return por

  def getPrevPOR(self):
    '''
    Gets the previous L{POR} in the walk order. Calls L{_getPrevItem},
    L{_getPrevPeer}, L{_getLastChild}, and L{_getParent} to attempt to get the
    previous valid L{POR}. Each method determines whether the L{POR} is valid as
    the previous L{POR}, and, if not, which call to make next. Each method
    potentially returns a L{POR} and the next method to call to continue the
    search for the previous L{POR}.
    
    @return: Previous L{POR} or None if this is the first L{POR}
    @rtype: L{POR} or None
    '''
    state = self._getPrevItem
    por = self.por
    while state is not None:
      por, state = state(por)
    if por is not None:
      self.por = por
    return por
  
  def getFirstPOR(self):
    '''
    Gets the first L{POR} in the walk order. Searches up the accessible 
    hierarchy until an accessible with no parent is encountered. The last 
    visited child of that accessible is the first L{POR} (i.e. the top-level
    window containing the starting L{POR}).
    
    @return: First L{POR}
    @rtype: L{POR}
    '''
    por = self.por
    child = self.por
    while 1:
      try:
        parent = IAccessibleNav(por).getParentAcc()
      except LookupError:
        self.por = child
        return child
      else:
        child = por
        por = parent
        
  def getLastPOR(self):
    '''
    Gets the last L{POR} in the walk order. Searches down the last child branch
    of the accessible hierarchy to the deepest accessible. The search then
    proceeds through previous L{POR}s in the walk order until the first visible,
    non-trivial accessible is encountered. The last item of that accessible is
    the last L{POR}.
    
    @return: Last L{POR}
    @rtype: L{POR}
    '''
    # get to the first POR first so we can traverse the very last branch
    self.getFirstPOR()
    while 1:
      try:
        self.por = IAccessibleNav(self.por).getLastAccChild()
      except LookupError:
        ai = IAccessibleInfo(self.por)
        if ((not ai.isAccTrivial() or self.allow_trivial) and 
            (ai.isAccVisible() or self.allow_invisible)):
          # get the last item
          return self._getLastItem(self.por)
        else:
          # back off until we find a non-trivial accessible
          return self.getPrevPOR()
        
  def getParentPOR(self):
    '''
    Gets the parent L{POR} of the current L{POR}. Searches up the accessible
    hierarchy to find the first non-trivial, visible containing element.
    
    @return: Parent L{POR} or None if at root
    @rtype: L{POR}
    '''
    while 1:
      try:
        # try to get the parent
        self.por = IAccessibleNav(self.por).getParentAcc()
        ai = IAccessibleInfo(self.por)
        if ((self.allow_invisible or ai.isAccVisible()) and 
            (not ai.isAccTrivial() or self.allow_trivial)):
          return self.por
      except (LookupError, IndexError):
        # there's no parent, so bail because we're at the root
        return None