/* sel-onetable.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "sel-onetable.h"
#include <libgnomedb/libgnomedb.h>
#include "mg-server-data-type.h"
#include "mg-database.h"
#include "mg-db-table.h"
#include "mg-db-field.h"
#include "mg-db-constraint.h"
#include "mg-entity.h"
#include "mg-field.h"

/*
 *
 * Modules for the fields of a given MgDbTable
 *
 */
static void         module_onetable_fill_model (Module *module);
static void         module_onetable_free (Module *module);
static const gchar *module_onetable_col_name (Module *module, guint colno);
static void         module_onetable_model_store_data (Module *module, GtkTreeIter *iter);

static GSList      *module_onetable_get_objects_list (Module *module);
static gchar       *module_onetable_get_extended_name (GObject *obj);

Module *sel_module_onetable_new (MgSelector *mgsel, gboolean insert_header, 
				 GtkTreeIter *iter, gpointer data)
{
	Module *module;
	MgDbTable *table;
	GdkPixbuf *pixbuf_field = NULL;

	g_assert (data && IS_MG_DB_TABLE (data));
	pixbuf_field = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-field_16x16.png");

	/* module structure */
	table = MG_DB_TABLE (data);
	module = g_new0 (Module, 1);
	module->selector = mgsel;
	module->fill_model = module_onetable_fill_model;
	module->free = module_onetable_free;
	module->col_name = module_onetable_col_name;
	module->obj_manager = NULL;
	module->model_store_data = module_onetable_model_store_data;
	module->mod_data = NULL;
	module->iter = NULL;
	module->parent_module = NULL;
	module->sub_modules = NULL;

	/* Module's private data */
	module->mod_data = g_new0 (ModNameGroupData, 1);
	GROUP_DATA (module)->manager = G_OBJECT (table);
	GROUP_DATA (module)->manager_weak_refed = FALSE;
	GROUP_DATA (module)->obj_pixbuf = pixbuf_field;
	GROUP_DATA (module)->get_objects_list = module_onetable_get_objects_list;
	GROUP_DATA (module)->get_extended_name = module_onetable_get_extended_name;
	
	/* model settings */
	if (insert_header) {
		GdkPixbuf *pixbuf = NULL;
		GtkTreeModel *model = mgsel->priv->model;

		module->iter = g_new0 (GtkTreeIter, 1);
		gtk_tree_store_append (GTK_TREE_STORE (model), module->iter, iter);
		gtk_tree_store_set (GTK_TREE_STORE (model), module->iter, 
				    NAME_COLUMN, mg_base_get_name (MG_BASE (table)), 
				    PIXBUF_COLUMN, pixbuf, 
				    CONTENTS_COLUMN, CONTENTS_TOP_CATEGORY, 
				    SUB_MODULE_COLUMN, NULL, -1);
	}
	else {
		if (iter)
			module->iter = gtk_tree_iter_copy (iter);
	}

	return module;	
}

static void module_onetable_constraint_any_cb (MgDatabase *db, MgDbConstraint *cstr, Module *module);
static void
module_onetable_fill_model (Module *module)
{
	GObject *manager, *db;
	GtkTreeModel *model;

	manager = GROUP_DATA (module)->manager;

	/* Initial model filling */
	model = module->selector->priv->model;
	name_group_init_model_fill (module, model);

	/* Signals handlers */
	g_signal_connect (manager, "field_added",
			  G_CALLBACK (name_group_obj_added_cb), module);
	g_signal_connect (manager, "field_removed",
			  G_CALLBACK (name_group_obj_removed_cb), module);
	g_signal_connect (manager, "field_updated",
			  G_CALLBACK (name_group_obj_updated_cb), module);
	/* REM: we don't need to take care of the "fields_order_changed" signal since the MgDbTable DOES NOT
	 emit it. */

	/* signals from the Database object */
	db = (GObject *) mg_db_table_get_database (MG_DB_TABLE (manager));
	g_assert (db && IS_MG_DATABASE (db));
	g_signal_connect (db, "constraint_added",
			  G_CALLBACK (module_onetable_constraint_any_cb), module);
	g_signal_connect (db, "constraint_removed",
			  G_CALLBACK (module_onetable_constraint_any_cb), module);
	g_signal_connect (db, "constraint_updated",
			  G_CALLBACK (module_onetable_constraint_any_cb), module);
}

static void
module_onetable_constraint_any_cb (MgDatabase *db, MgDbConstraint *cstr, Module *module)
{
	if (mg_db_constraint_get_table (cstr) == (MgDbTable *) (GROUP_DATA (module)->manager)) {
		GSList *fields, *list;

		fields = mg_entity_get_all_fields (MG_ENTITY (GROUP_DATA (module)->manager));
		list = fields;
		while (list) {
			if (mg_db_constraint_uses_field (cstr, MG_DB_FIELD (list->data)))
				name_group_obj_updated_cb (GROUP_DATA (module)->manager, G_OBJECT (list->data),
							   module);
			list = g_slist_next (list);
		}
		g_slist_free (fields);
	}
}

static GSList *
module_onetable_get_objects_list (Module *module)
{
	g_return_val_if_fail (GROUP_DATA (module)->manager, NULL);
	g_return_val_if_fail (IS_MG_DB_TABLE (GROUP_DATA (module)->manager), NULL);

	return mg_entity_get_all_fields (MG_ENTITY (GROUP_DATA (module)->manager));
}

static gchar *
module_onetable_get_extended_name (GObject *obj)
{
	g_return_val_if_fail (obj && IS_MG_DB_FIELD (obj), NULL);

	return g_strdup (mg_base_get_name (MG_BASE (obj)));
}

static void
module_onetable_free (Module *module)
{
	GObject *manager = GROUP_DATA (module)->manager;
	GObject *db;

	/* free this module */
	g_assert (manager);
	g_signal_handlers_disconnect_by_func (manager,
					      G_CALLBACK (name_group_obj_added_cb), module);
	g_signal_handlers_disconnect_by_func (manager,
					      G_CALLBACK (name_group_obj_removed_cb), module);
	g_signal_handlers_disconnect_by_func (manager,
					      G_CALLBACK (name_group_obj_updated_cb), module);
	
	db = (GObject *) mg_db_table_get_database (MG_DB_TABLE (manager));
	g_assert (db && IS_MG_DATABASE (db));
	g_signal_handlers_disconnect_by_func (db, 
					      G_CALLBACK (module_onetable_constraint_any_cb), module);
	g_signal_handlers_disconnect_by_func (db, 
					      G_CALLBACK (module_onetable_constraint_any_cb), module);
	g_signal_handlers_disconnect_by_func (db,
					      G_CALLBACK (module_onetable_constraint_any_cb), module);

	if (module->iter)
		gtk_tree_iter_free (module->iter);
	name_group_free_mod_data (module);
	g_free (module->mod_data);
	module->mod_data = NULL;
}


static const gchar *
module_onetable_col_name (Module *module, guint colno)
{
	switch (colno) {
	case 0:
		return _("Field");
		break;
	case EXTRA1_COLUMN:
		return _("Type");
		break;
	case EXTRA2_COLUMN:
		return _("Length");
		break;
	case EXTRA3_COLUMN:
		return _("Not NULL?");
		break;
	case EXTRA5_COLUMN:
		return _("Default value");
		break;
	default:
		return NULL;
		break;
	}
}

static void
module_onetable_model_store_data (Module *module, GtkTreeIter *iter)
{
	GObject *obj;
	GtkTreeModel *model;

	model = module->selector->priv->model;
	gtk_tree_model_get (model, iter, OBJ_COLUMN, &obj, -1);

	if (obj && IS_MG_DB_FIELD (obj)) {
		MgDbField *field;
		const gchar *str1;
		gchar *str2, *str3;
		gboolean bool;
		MgServerDataType *type;
		const GdaValue *value;
		gint length, scale;
		
		field = MG_DB_FIELD (obj);

		/* data type */
		type = mg_field_get_data_type (MG_FIELD (obj));
		if (type)
			str1 = mg_server_data_type_get_sqlname (type);
		else
			str1 = _("Unknown");
			
		/* length */
		length = mg_db_field_get_length (field);
		scale = mg_db_field_get_scale (field);
		if (length != -1) {
			if (scale != 0)
				str2 = g_strdup_printf ("(%d, %d)", length, scale);
			else
				str2 = g_strdup_printf ("%d", length);
		}
		else
			str2 = NULL;
		
		/* NOT NULL */
		bool = mg_db_field_is_null_allowed (field) ? FALSE : TRUE;
		
		/* Default value */
		value = mg_db_field_get_default_value (field);
		if (value)
			str3 = gda_value_stringify (value);
		else
			str3 = g_strdup ("");

		gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
				    EXTRA1_COLUMN, str1,
				    EXTRA2_COLUMN, str2,
				    EXTRA3_COLUMN, bool, 
				    EXTRA4_COLUMN, TRUE, 
				    EXTRA5_COLUMN, str3,
				    -1);
		if (str2) g_free (str2);
		g_free (str3);
	}
}



