# This file is part of pybliographer
# 
# Copyright (C) 1998 Frederic GOBRY
# Email : gobry@idiap.ch
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# $Id: BibTeX.py,v 1.23 1999/08/18 14:05:21 gobry Exp $

# Extension module for BibTeX files

import Pyblio.Base, _bibtex
import os,sys, tempfile, pwd, time, traceback, re, string
import Pyblio.Open

from types import *
from Pyblio.Fields import *
from Pyblio import Types

def _nativify (field, type):
	obj = _bibtex.reverse (type, field)
	return _bibtex.native (obj)


def entry_write (entry, output):
	""" Print an entry as BiBTeX """

	native = (entry.id == 'BibTeX')


	# get the entry description
	tp = entry.type
	
	output.write ('@%s{%s,\n' % (tp.name, entry.name))
	dico = entry.keys ()
	
	for f in tp.mandatory:
		field = string.lower (f.name)
		
		if entry.has_key (field):
			if native:
				text = entry.get_native (field)
			else:
				type = Types.gettype (tp, field)
				text = _nativify (entry [field], type)
				
			output.write ('  ' + Pyblio.Base.format (
				'%-14s = %s' % (f.name, text),
                                75, 0, 19) + ',\n')
			dico.remove (field)

	for f in dico:
		if native:
			text = entry.get_native (f)
		else:
			type = Types.gettype (tp, f)
			text = _nativify (entry [f], type)
			
		output.write ('  ' + Pyblio.Base.format (
			'%-14s = %s' % (f, text), 75, 0, 19) +
                              ',\n')
		
        output.write ('}\n\n')
	return


def my_write (database, output):
	need_header = []
	if database.id == 'BibTeX':
		need_header = [database]
	else:
		# ugly hack to get the BibTeX databases involved
		db = {}
		def check_entry (entry, need, database = database):
			# guess what, we are in a RefDB !
			if entry.id == 'BibTeX':
				base = database.base (entry)
				# ensure unicity
				need [base.key] = base
			return
		database.foreach (check_entry, db)

		need_header = db.values ()

	if len (need_header) > 0:
		for db in need_header:
			parser = db.get_parser ()
		
			# write the list of strings
			dict = _bibtex.get_dict (parser)
		
			if len (dict.keys ()) > 0:
				output.write ('@string{\n')
				for k in dict.keys ():
					value = _bibtex.native (dict [k])
					output.write ("  %s \t= %s,\n" % (k, value))
				output.write ('}\n\n')

	# write all the entries
	database.foreach (entry_write, output)
	return

# --------------------------------------------------
# Register a method to open BibTeX files
# --------------------------------------------------

def my_open (entity, check):
	
	method, address, file, p, q, f = entity
	base = None
	
	if (not check) or (method == 'file' and file [-4:] == '.bib'):
		base = DataBase (file)
		
	return base


Pyblio.Open.register ('BibTeX', {'open'  : my_open,
				 'write' : my_write })

	
# --------------------------------------------------
#  Entry inherits from pyb
# --------------------------------------------------

class Entry (Pyblio.Base.Entry):

	id = 'BibTeX'

	def __init__ (self, key, name, type, content, parser):
		Pyblio.Base.Entry.__init__ (self, key, name, type, content)
		self.__text = {}

		self.parser = parser
		return
	
	
	def __delitem__ (self, key):
		# First, eventually remove from cache
		if self.__text.has_key (key):
			del self.__text [key]

		del self.__dict [key]
		return

	def __setitem__ (self, key, value):
		# First, eventually remove from cache
		if self.__text.has_key (key):
			del self.__text [key]

		type = Types.gettype (self.type, key)
		self.__dict [key] = _bibtex.reverse (type, value)
		return

	
	def get_native (self, key):
		""" Return object in its native format """

		if self.__text.has_key (key):
			obj = self.__text [key] [0]
			if obj.modified ():
				type = Types.gettype (self.type, key)
				self.__dict [key] = \
					    _bibtex.reverse (type, obj)
				# update cache
				del self.__text [key]
			
		obj = self.__dict [key]
		return _bibtex.native (obj)
	

	def set_native (self, key, value):
		""" set in native format """
		
		if self.__text.has_key (key):
			del self.__text [key]

		type = Types.gettype (self.type, key)
		self.__dict [key] =  _bibtex.reverse (type, value)
		return
	
		
	def text (self, key):
		# look in the cache first
		if self.__text.has_key (key):
			return self.__text [key]
		
		obj = self.__dict [key]

		# search its declared type

		type = Types.gettype (self.type, key)
		ret = _bibtex.expand (self.parser, obj, type)
		
		if ret [0] == Types.TypeAuthor:
			# Author
			val = AuthorGroup ()
			for aut in ret [3]:
				val.append (Author (aut))
			
		elif ret [0] == Types.TypeDate:
			# Date
			val = Date ((ret [3], None, None))
			
		else:
			# Any other text
			val = Text (ret [2])

		val.clean ()
		self.__text [key] = (val, ret [1])
		
		return (val, ret [1])



# --------------------------------------------------
# Une base de references bibliographiques,
# comme un dictionnaire avec des extensions...
# --------------------------------------------------

class DataBase (Pyblio.Base.DataBase):

	id = 'BibTeX'
	
	properties = {
		'edit'        : 1,
		'change_id'   : 1,
		'change_type' : 1,
		'add'         : 1,
		'remove'      : 1,
		'has_extra'   : 1,
		}
	
	def __parsefile__ (self):
		self.__parser = None
		self.__recursive = 0
		self.__current = 0
		
		self.__dict = {}
		
		# Ouvrir le fichier associe
		self.__parser = _bibtex.open (self.name)
		
		finished = 0
		errors = []
		
		# Creer la base de cles
		while 1:
			try:
				retval = _bibtex.next (self.__parser)
			except IOError, err:
				errors.append (str (err))
				continue
			
			if retval == None: break

			name, type, offset, line, object = retval
			
			key = Pyblio.Base.Key (self, name)

			if not self.__dict.has_key (key):
				type = Types.getentry (type)
				entry = Entry (key, name, type, object, self.__parser)
				self.__dict [key] = entry
				continue
				
			errors.append ("%s:%d: key `%s' already defined" % (
				self.name, line, name))

		if len (errors) > 0:
			raise IOError, string.join (errors, "\n")

		
	def __init__ (self, basename):
		"Initialisation"
		Pyblio.Base.DataBase.__init__ (self, basename)
		self.__parsefile__ ()
		return

	
	def __del__ (self):
		pass

	def get_parser (self):
		return self.__parser
	

	def __repr__ (self):
		""
		return "<BibTeX database `%s' (%d entries)>" % \
		       (self.name, len (self))

	# ==================================================

	def update (self):
		""" save the database """

		# backup file
		os.rename (self.name, self.name + '.bak')
		
		tmpfile = open (self.name, 'w')
		my_write (self, tmpfile)
		tmpfile.close ()

		self.__parsefile__ ()
		return

	
