/*
 * XML Catalog Manager (xmlcatmgr)
 * $Id: compat.c,v 1.2 2004/08/31 21:25:47 jmmv Exp $
 *
 * Copyright (c) 2003, 2004 Julio M. Merino Vidal.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the author nor the names of contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * This file provides compatibility stuff for system functions that may
 * be missing.  See their respective manpages for more information.
 */

#include "system.h"

#ifndef lint
__RCSID("$Id: compat.c,v 1.2 2004/08/31 21:25:47 jmmv Exp $");
#endif

#if !defined(HAVE_GETPROGNAME) || !defined(HAVE_SETPROGNAME)
static const char *ProgName = NULL;
#endif

/* --------------------------------------------------------------------- */

#if !defined(HAVE_GETPROGNAME) || !defined(HAVE_SETPROGNAME)
const char *
compat_getprogname(void)
{
    assert(ProgName != NULL);
    return ProgName;
}
#endif

/* --------------------------------------------------------------------- */

#if !defined(HAVE_LOCKF)
int
compat_lockf(int filedes, int function, off_t size)
{
    int cmd;
    struct flock fl;

    fl.l_start = 0;
    fl.l_len = size;
    fl.l_whence = SEEK_CUR;

    switch (function) {
        case F_ULOCK:
            cmd = F_SETLK;
            fl.l_type = F_UNLCK;
            break;
        case F_LOCK:
            cmd = F_SETLKW;
            fl.l_type = F_WRLCK;
            break;
        case F_TLOCK:
        case F_TEST:
            assert(false);
            break;
        default:
            errno = EINVAL;
            return -1;
    }

    return fcntl(filedes, cmd, &fl);
}
#endif

/* --------------------------------------------------------------------- */

#if !defined(HAVE_GETPROGNAME) || !defined(HAVE_SETPROGNAME)
void
compat_setprogname(const char *name)
{
    assert(ProgName == NULL);

    if ((ProgName = strrchr(name, '/')) != NULL)
        ProgName++;
    else
        ProgName = name;
}
#endif

/* --------------------------------------------------------------------- */

#if !defined(HAVE_WARN)
void
compat_warn(const char *fmt, ...)
{
    va_list va;

    fprintf(stderr, "%s: ", getprogname());
    va_start(va, fmt);
    vfprintf(stderr, fmt, va);
    va_end(va);
    fprintf(stderr, ": %s\n", strerror(errno));
}
#endif

/* --------------------------------------------------------------------- */

#if !defined(HAVE_WARNX)
void
compat_warnx(const char *fmt, ...)
{
    va_list va;

    fprintf(stderr, "%s: ", getprogname());
    va_start(va, fmt);
    vfprintf(stderr, fmt, va);
    va_end(va);
    fprintf(stderr, "\n");
}
#endif

/*
 * Local Variables: ***
 * mode: c ***
 * c-file-style: "stroustrup" ***
 * End: ***
 * vim: syntax=c:expandtab:shiftwidth=4:softtabstop=4
 */
