/*
 *   Copyright (C) 2007 Ivan Cukic <ivan.cukic+kde@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser/Library General Public License version 2,
 *   or (at your option) any later version, as published by the Free
 *   Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser/Library General Public License for more details
 *
 *   You should have received a copy of the GNU Lesser/Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "ExtenderButton.h"

#include <QRectF>
#include <KDebug>
#include <QIcon>

#define ACTIVATION_TIME 300

namespace Lancelot {

// Inner class ExtenderObject
class ExtenderObject : public BasicWidget {
public:
    ExtenderObject(Plasma::Svg * icon,
            ExtenderButton * parent)
      : BasicWidget(icon, "", "")
    {
        setParentItem(parent);
        setInnerOrientation(Qt::Vertical);
        setAlignment(Qt::AlignCenter);
    }

    // Needed because of a Qt bug - making it public
    // so that ExtenderButton can invoke it directly
    void hoverLeaveEvent(QGraphicsSceneHoverEvent * event)
    {
        BasicWidget::hoverLeaveEvent(event);
    }
};


//
/* class ExtenderObject : public BasicWidget {
public:
    ExtenderObject(Plasma::Svg * icon,
            ExtenderButton * parent)
      : BasicWidget(icon, "", ""),
        m_parent(parent)
    {
        setParentItem(parent);
        setInnerOrientation(Qt::Vertical);
        setAlignment(Qt::AlignCenter);
    }

    void hoverEnterEvent(QGraphicsSceneHoverEvent * event)
    {
        // TODO: Replace this with signals/slots
        BasicWidget::hoverEnterEvent(event);
        startTimer();
    }

    void hoverLeaveEvent(QGraphicsSceneHoverEvent * event)
    {
        // TODO: Replace this with signals/slots
        BasicWidget::hoverLeaveEvent(event);
        stopTimer();
    }

    static QBasicTimer timer;

    void startTimer()
    {
        timer.start(ACTIVATION_TIME, m_parent);
    }

    void stopTimer()
    {
        timer.stop();
    }

    void paint(QPainter * painter,
            const QStyleOptionGraphicsItem * option, QWidget * widget)
    {
        if (Plasma::PanelSvg * svg = group()->backgroundSvg()) {
            svg->setEnabledBorders(borders);
        }
        BasicWidget::paint(painter, option, widget);
    }

    Plasma::PanelSvg::EnabledBorders borders;

private:
    ExtenderButton * m_parent;
};*/

// ExtenderButton
class ExtenderButton::Private {
public:
    Private(ExtenderButton * parent)
      : q(parent),
        extender(NULL),
        extenderPosition(NoExtender),
        activationMethod(ClickActivate),
        checkable(false),
        checked(false)
    {
        if (!extenderIconSvg) {
            extenderIconSvg = new Plasma::Svg(q);
            extenderIconSvg->setImagePath("lancelot/extender-button-icon");
        }

        extender = new ExtenderObject(extenderIconSvg, parent);
        extender->setVisible(false);

        extender->setIconSize(QSize(16, 16));

        timer.setInterval(ACTIVATION_TIME);
        timer.setSingleShot(true);
        connect(
                extender, SIGNAL(mouseHoverEnter()),
                &timer, SLOT(start())
               );
        connect(
                extender, SIGNAL(mouseHoverLeave()),
                &timer, SLOT(stop())
               );
        connect(
                &timer, SIGNAL(timeout()),
                parent, SLOT(activate())
               );

    }

    ~Private()
    {
        delete extender;
    }

    void relayoutExtender()
    {
        if (!extender) return;
        QRectF geometry = QRectF(QPointF(0, 0), q->size());

        Plasma::PanelSvg::EnabledBorders extender_borders = Plasma::PanelSvg::AllBorders;
        borders = Plasma::PanelSvg::AllBorders;

        switch (extenderPosition) {
        case TopExtender:
            geometry.setHeight(EXTENDER_SIZE);
            geometry.moveTop(- EXTENDER_SIZE);
            borders &= ~ Plasma::PanelSvg::TopBorder;
            extender_borders &= ~ Plasma::PanelSvg::BottomBorder;
            break;
        case BottomExtender:
            geometry.moveTop(geometry.bottom());
            geometry.setHeight(EXTENDER_SIZE);
            borders &= ~ Plasma::PanelSvg::BottomBorder;
            extender_borders &= ~ Plasma::PanelSvg::TopBorder;
            break;
        case LeftExtender:
            geometry.setWidth(EXTENDER_SIZE);
            geometry.moveLeft(- EXTENDER_SIZE);
            borders &= ~ Plasma::PanelSvg::LeftBorder;
            extender_borders &= ~ Plasma::PanelSvg::RightBorder;
            break;
        case RightExtender:
            geometry.moveLeft(geometry.right());
            geometry.setWidth(EXTENDER_SIZE);
            borders &= ~ Plasma::PanelSvg::RightBorder;
            extender_borders &= ~ Plasma::PanelSvg::LeftBorder;
            break;
        case NoExtender:
            break;
        }
        extender->setGeometry(geometry);
        extender->setPreferredSize(geometry.size());

        if (Plasma::PanelSvg * svg = extender->group()->backgroundSvg()) {
            svg->setEnabledBorders(extender_borders);
        }

        // qreal left, top, right, bottom;
        // q->getContentsMargins(&left, &top, &right, &bottom);
    }

    ExtenderButton * q;

    ExtenderPosition extenderPosition;
    ActivationMethod activationMethod;

    static Plasma::Svg * extenderIconSvg;
    Plasma::PanelSvg::EnabledBorders borders;

    ExtenderObject * extender;
    QTimer timer;
    bool checkable : 1;
    bool checked : 1;

};

Plasma::Svg * ExtenderButton::Private::extenderIconSvg = NULL;
// QBasicTimer ExtenderObject::timer = QBasicTimer();

/*void ExtenderButton::timerEvent(QTimerEvent * event)
{
    if (event->timerId() == ExtenderObject::timer.timerId()) {
        d->extender->stopTimer();
        toggle();
        d->extender->hide();
        // Qt bug... - element is hidden but doesn't receive hoverLeaveEvent
        hoverLeaveEvent(0);
        d->extender->hoverLeaveEvent(0);

        // Sending the activate signal
        activate();
    }
    QObject::timerEvent(event);
}*/

ExtenderButton::ExtenderButton(QGraphicsItem * parent)
  : BasicWidget(parent),
    d(new Private(this))
{
    setGroupByName("ExtenderButton");
    connect(this, SIGNAL(clicked()), this, SLOT(activate()));
    L_WIDGET_SET_INITIALIZED;
}

ExtenderButton::ExtenderButton(QString title,
        QString description, QGraphicsItem * parent)
  : BasicWidget(title, description, parent),
    d(new Private(this))
{
    setGroupByName("ExtenderButton");
    connect(this, SIGNAL(clicked()), this, SLOT(activate()));
    L_WIDGET_SET_INITIALIZED;
}

ExtenderButton::ExtenderButton(QIcon icon, QString title,
        QString description, QGraphicsItem * parent)
  : BasicWidget(icon, title, description, parent),
    d(new Private(this))
{
    setGroupByName("ExtenderButton");
    connect(this, SIGNAL(clicked()), this, SLOT(activate()));
    L_WIDGET_SET_INITIALIZED;
}

ExtenderButton::ExtenderButton(Plasma::Svg * icon, QString title,
        QString description, QGraphicsItem * parent)
  : BasicWidget(icon, title, description, parent),
    d(new Private(this))
{
    setGroupByName("ExtenderButton");
    connect(this, SIGNAL(clicked()), this, SLOT(activate()));
    L_WIDGET_SET_INITIALIZED;
}

void ExtenderButton::setGroup(WidgetGroup * g)
{
    BasicWidget::setGroup(g);
    d->extender->setGroupByName(
            group()->name() + "-Extender"
            );
}

void ExtenderButton::groupUpdated()
{
    BasicWidget::groupUpdated();
    if (group()->hasProperty("ExtenderPosition")) {
        setExtenderPosition((ExtenderPosition)(group()->property("ExtenderPosition").toInt()));
    }
    if (group()->hasProperty("ActivationMethod")) {
        setActivationMethod((ActivationMethod)(group()->property("ActivationMethod").toInt()));
    }
}

ExtenderButton::~ExtenderButton()
{
    delete d;
}

QRectF ExtenderButton::boundingRect() const
{
    return QRectF(0, 0, size().width(), size().height());
}

void ExtenderButton::hoverEnterEvent(QGraphicsSceneHoverEvent * event)
{
    if (d->extenderPosition != NoExtender) {
        d->extender->setVisible(true);
    } else if (d->activationMethod == HoverActivate) {
        d->timer.start();
    }
    BasicWidget::hoverEnterEvent(event);
}

void ExtenderButton::hoverLeaveEvent(QGraphicsSceneHoverEvent * event)
{
    d->extender->setVisible(false);
    d->timer.stop();
    BasicWidget::hoverLeaveEvent(event);
}

void ExtenderButton::setExtenderPosition(ExtenderPosition position)
{
    if (d->extenderPosition == position)
        return;
    d->extenderPosition = position;
    d->relayoutExtender();
}

ExtenderPosition ExtenderButton::extenderPosition()
{
    return d->extenderPosition;
}

void ExtenderButton::setActivationMethod(ActivationMethod method)
{
    if (d->activationMethod == method)
        return;
    d->activationMethod = method;
    if (d->activationMethod != ExtenderActivate)
        setExtenderPosition(NoExtender);
}

ActivationMethod ExtenderButton::activationMethod()
{
    return d->activationMethod;
}

void ExtenderButton::activate()
{
    toggle();
    d->timer.stop();
    d->extender->hide();
    // Qt bug... - element is hidden but doesn't receive hoverLeaveEvent
    hoverLeaveEvent(0);
    //d->extender->hoverLeaveEvent(0);
    update();
    emit activated();
}

void ExtenderButton::geometryUpdated()
{
    BasicWidget::geometryUpdated();
    d->relayoutExtender();
}

void ExtenderButton::paint(QPainter * painter,
        const QStyleOptionGraphicsItem * option, QWidget * widget)
{
    Q_UNUSED(option);
    Q_UNUSED(widget);

    if (Plasma::PanelSvg * svg = group()->backgroundSvg()) {
        if (isHovered() && (d->extenderPosition != NoExtender)) {
            svg->setEnabledBorders(d->borders);
        } else {
            svg->setEnabledBorders(Plasma::PanelSvg::AllBorders);
        }
    }

    if (d->checked && !isDown()) {
        paintBackground(painter, "checked");
    } else {
        paintBackground(painter);
    }
    paintForeground(painter);
}

void ExtenderButton::setChecked(bool checked)
{
    if (!d->checkable) return;
    if (d->checked == checked) return;
    d->checked = checked;
    update();
    emit toggled(d->checked);
}

bool ExtenderButton::isChecked()
{
    return d->checked;
}

void ExtenderButton::toggle()
{
    if (!d->checkable) return;
    d->checked = !d->checked;
    update();
    emit toggled(d->checked);
}

void ExtenderButton::setCheckable(bool checkable)
{
    if (d->checkable == checkable) return;
    d->checkable = checkable;
    if (!checkable) {
        d->checked = false;
    }
    update();
}

bool ExtenderButton::isCheckable()
{
    return d->checkable;
}

} // namespace Lancelot

