/***************************************************************************
 *   notification.cpp                                                      *
 *                                                                         *
 *   Copyright (C) 2008 Jason Stubbs <jasonbstubbs@gmail.com>              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
 ***************************************************************************/

#include "notification.h"
#include "task.h"

#include <QImage>
#include <QtCore/QTimer>

#include <KDebug>


namespace SystemTray
{


class Notification::Private
{
public:
    Private()
        : timeout(0),
          owner(0),
          hideTimer(0),
          expired(false),
          read(false)
    {
    }

    QString identifier;
    QString applicationName;
    QIcon applicationIcon;
    QString message;
    QString summary;
    int timeout;
    QImage image;
    Task *owner;
    QTimer *deleteTimer;
    QTimer *hideTimer;
    bool expired;
    bool read;

    QHash<QString, QString> actions;
    QStringList actionOrder;
};


Notification::Notification(QObject *parent)
    : QObject(parent),
      d(new Private)
{
    d->deleteTimer = new QTimer(this);
    d->deleteTimer->setSingleShot(true);
    connect(d->deleteTimer, SIGNAL(timeout()), this, SLOT(deleteLater()));
}


Notification::~Notification()
{
    emit notificationDestroyed(this);
    delete d;
}


QString Notification::applicationName() const
{
    return d->applicationName;
}


void Notification::setApplicationName(const QString &applicationName)
{
    d->applicationName = applicationName;
}


QIcon Notification::applicationIcon() const
{
    return d->applicationIcon;
}


void Notification::setApplicationIcon(const QIcon &applicationIcon)
{
    d->applicationIcon = applicationIcon;
}


QString Notification::message() const
{
    return d->message;
}


void Notification::setMessage(const QString &message)
{
    d->message = message;
}


QString Notification::summary() const
{
    return d->summary;
}


void Notification::setSummary(const QString &summary)
{
    d->summary = summary;
}


int Notification::timeout() const
{
    return d->timeout;
}

QImage Notification::image() const
{
    return d->image;
}

void Notification::setImage(QImage image)
{
    d->image = image;
}

void Notification::setTimeout(int timeout)
{
    d->timeout = timeout;
    //keep it available for 30 minutes
    d->deleteTimer->start(30*60*1000);

    if (timeout) {
        if (!d->hideTimer) {
            d->hideTimer = new QTimer(this);
            d->hideTimer->setSingleShot(true);
            connect(d->hideTimer, SIGNAL(timeout()), this, SLOT(hide()));
        }
        d->hideTimer->start(d->timeout);
    }
}


QHash<QString, QString> Notification::actions() const
{
    return d->actions;
}


void Notification::setActions(const QHash<QString, QString> &actions)
{
    d->actions = actions;
}


QStringList Notification::actionOrder() const
{
    return d->actionOrder;
}


void Notification::setActionOrder(const QStringList &actionOrder)
{
    d->actionOrder = actionOrder;
}


void Notification::triggerAction(const QString &actionId)
{
    Q_UNUSED(actionId);
    kDebug() << "action triggered but no handler implemented";
}

void Notification::remove()
{
    kDebug() << "remove requested but no handler implemented";
}

void Notification::hide()
{
    d->expired = true;
    emit expired(this);
}

//temporary bug fix only for 4.4
void Notification::show()
{
    d->expired = false;
}

bool Notification::isExpired() const
{
    return d->expired;
}

void Notification::setRead(const bool read)
{
    d->read = read;
}

bool Notification::isRead() const
{
    return d->read;
}

void Notification::setDeleteTimeout(const int time)
{
    if (d->deleteTimer->interval() != time) {
        d->deleteTimer->start(time);
    }
}

int Notification::deleteTimeOut() const
{
    return d->deleteTimer->interval();
}

}


#include "notification.moc"
