
#include <klocale.h>
#include <kmessagebox.h>
#include <kfiledialog.h>

#include <qstring.h>
#include <klistview.h>
#include <qradiobutton.h>
#include <qbutton.h>
#include <qbuttongroup.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <qstringlist.h>
#include <qcheckbox.h>
#include <qdir.h>

#include "setup.h"

Setup :: Setup(QWidget *_parent, const char* _name) :
  Setup_Base(_parent, _name) 
{

  gpInterface = new Gphoto_Interface();

  getCameraListings();
  getPortListings();

  KListView_cameras->setFullWidth();
  QButtonGroup_port->setExclusive(true);
  QComboBox_downloadDir->insertItem(QDir::currentDirPath(),0);
  QComboBox_downloadDir->insertItem(QDir::homeDirPath());

  
  loadCameraSettings();
  loadGeneralSettings();

  connect(this, SIGNAL(okClicked()),
	  this, SLOT(slot_ok()) );
  connect(this, SIGNAL(cancelClicked()),
	  this, SLOT(slot_cancel()) );
  connect(this, SIGNAL(applyClicked()),
	  this, SLOT(slot_apply()) );


}

Setup::~Setup(){
  
  delete gpInterface;
  delete config;

}

void Setup::getCameraListings() {

  QStringList cameraList; 
  int numSupported;

  gpInterface->getSupportedCameras(cameraList, numSupported);

  for (int i=0; i<numSupported; i++) {
    new KListViewItem(KListView_cameras, cameraList[i]);
  }

}

void Setup::getPortListings() {

  QStringList portList;
  gpInterface->getSupportedPorts(portList);

  for (unsigned int i=0; i<portList.count(); i++) {
    if ((portList[i]).startsWith("serial:"))
      supportedSerialPorts.append(portList[i]);
  }

}


void Setup::loadCameraSettings() {

  QString portPath;

  gpInterface->getCameraSettings(my_cameraModel, my_cameraPort);

  if (my_cameraModel != "" && my_cameraPort != "") {
    
    KListViewItem *item = (KListViewItem*) KListView_cameras->firstChild();
    do {
      if (item->text(0) == my_cameraModel) {
	KListView_cameras->setSelected(item, true);
	KListView_cameras->ensureItemVisible(item);
	break;
      }
    } while( item = (KListViewItem*) item->nextSibling());


    if (my_cameraPort.startsWith("serial:")) {
      slot_setPort(QButtonGroup_port->id(QRadioButton_serial));
      QComboBox_portPath->insertItem(my_cameraPort,0);
    }
    else if (my_cameraPort.startsWith("usb:")) 
      slot_setPort(QButtonGroup_port->id(QRadioButton_usb));
    
  }

}

void Setup::loadGeneralSettings() {

  config = new KSimpleConfig("digikamrc");
  config->setGroup("General Settings");

  if (config->readEntry("Default Download Directory") != NULL) 
    QComboBox_downloadDir->insertItem(config->readEntry("Default Download Directory"), 0);

  if (config->readBoolEntry("No ThumbNails on Startup"))
    QCheckBox_thumbnails->setChecked(false);
  else
    QCheckBox_thumbnails->setChecked(true);
    
  if (config->readBoolEntry("Connect on Startup"))
    QCheckBox_connect->setChecked(true);
  else
    QCheckBox_connect->setChecked(false);
  
  if (config->readBoolEntry("Show All Images on Startup"))
    QCheckBox_showAllImages->setChecked(true);
  else
    QCheckBox_showAllImages->setChecked(false);

  if (config->readBoolEntry("Don't Show Camera Folders"))
    QCheckBox_hideCameraFolders->setChecked(true);
  else
    QCheckBox_hideCameraFolders->setChecked(false);

}




void Setup::saveCameraSettings() {


  if (KListView_cameras->selectedItem()) {

    my_cameraModel=KListView_cameras->selectedItem()->text(0);
    
    if (QButtonGroup_port->selected()) {

      QString type = QButtonGroup_port->selected()->text();
    
      if (type == i18n("Serial")) { 
	my_cameraPort=QComboBox_portPath->currentText();
      }
      else if(type == i18n("Parallel"))
	my_cameraPort="parallel:";
      else if(type == i18n("USB"))
	my_cameraPort="usb:";
      else if(type == i18n("FireWire"))
	my_cameraPort="ieee1394:";
    
    }

    else 
      my_cameraPort="";

    gpInterface->saveCameraSettings(my_cameraModel, my_cameraPort);

  }
}

void Setup::saveGeneralSettings() {

  config->setGroup("General Settings");

  if (QComboBox_downloadDir->currentText()) 
    config->writeEntry("Default Download Directory",QComboBox_downloadDir->currentText());
  
  if (QCheckBox_thumbnails->isChecked())
    config->writeEntry("No ThumbNails on Startup", false);
  else
    config->writeEntry("No ThumbNails on Startup", true);
  
  if (QCheckBox_connect->isChecked())
    config->writeEntry("Connect on Startup", true);
  else
    config->writeEntry("Connect on Startup", false);

  if (QCheckBox_showAllImages->isChecked())
    config->writeEntry("Show All Images on Startup", true);
  else
    config->writeEntry("Show All Images on Startup", false);

  if (QCheckBox_hideCameraFolders->isChecked())
    config->writeEntry("Don't Show Camera Folders", true);
  else
    config->writeEntry("Don't Show Camera Folders", false);

  config->sync();

}


void Setup::slot_cancel() {
  close();
}

void Setup::slot_ok() {
  saveCameraSettings();
  saveGeneralSettings();
  close();
}

void Setup::slot_apply() {
  saveCameraSettings();
  saveGeneralSettings();
}

void Setup::slot_autoDetect() {
  
  QString cameraModel;
  QString cameraPort;

  if (gpInterface->autoDetectCamera(cameraModel, cameraPort) != 0) {
    KMessageBox::error(this,"Failed to auto-detect camera!\n"
		       "Please rety or try setting manually.");
    return;
  }

  QListViewItem *item = KListView_cameras->firstChild();
  do {
    if (item->text(0) == cameraModel) {
      KListView_cameras->setSelected(item, true);
      KListView_cameras->ensureItemVisible(item);
      break;
    }
  } while ( item = item->nextSibling());

  if (cameraPort.startsWith("parallel:")) {
    slot_setPort(QButtonGroup_port->id(QRadioButton_parallel));
    QComboBox_portPath->clear();
    QComboBox_portPath->insertItem(cameraPort,0);
  }
  else if (cameraPort.startsWith("serial:")) {
    slot_setPort(QButtonGroup_port->id(QRadioButton_serial));
    QComboBox_portPath->clear();
    QComboBox_portPath->insertItem(cameraPort,0);
  }
  else if (cameraPort.startsWith("usb:")) {
    QComboBox_portPath->clear();
    slot_setPort(QButtonGroup_port->id(QRadioButton_usb));
  }
  else if (cameraPort.startsWith("ieee1394:")) {
    QComboBox_portPath->clear();
    slot_setPort(QButtonGroup_port->id(QRadioButton_firewire));
  }

  QString infoStr;
  infoStr = infoStr+"Detected a " +  cameraModel + " with port " + cameraPort;
  KMessageBox::information(this,infoStr);
  

}

void Setup::slot_setModel(QListViewItem *item) {
  

  QString cameraModel = item->text(0);
  QStringList supportedPorts;

  gpInterface->getCameraPorts(cameraModel, supportedPorts);

  if (supportedPorts.contains("serial")) {
    QRadioButton_serial->setEnabled(true);
  }
  else {
    QRadioButton_serial->setChecked(false);
    QRadioButton_serial->setEnabled(false);
  }

  if (supportedPorts.contains("usb")) {
    QRadioButton_usb->setEnabled(true);
  }
  else {
    QRadioButton_usb->setChecked(false);
    QRadioButton_usb->setEnabled(false);
  }

  QRadioButton_parallel->setEnabled(false);
  QRadioButton_firewire->setEnabled(false);

  if ( supportedPorts.contains("serial") + supportedPorts.contains("usb") == 1) {
    if (supportedPorts.contains("serial")) 
      slot_setPort(QButtonGroup_port->id(QRadioButton_serial));
    else if (supportedPorts.contains("usb")) 
      slot_setPort(QButtonGroup_port->id(QRadioButton_usb));
  }

}


  
void Setup::slot_setPort(int portType) {
  
   QButtonGroup_port->setButton(portType);
   
   slot_setPortPathLabel(portType);

}


void Setup::slot_setPortPathLabel(int portType) {

  if (portType==QButtonGroup_port->id(QRadioButton_serial)) {
     QTextLabel_portPath->setText("Set Port Path for \n"
				  "serial port");
     QComboBox_portPath->setEditable(true);
     
     QString current = QComboBox_portPath->currentText();
     QComboBox_portPath->clear();
     if (supportedSerialPorts.count() > 0)
       QComboBox_portPath->insertStringList(supportedSerialPorts);
     for (int i=0; i<QComboBox_portPath->count(); i++) {
       if (QComboBox_portPath->text(i) == current) {
	 QComboBox_portPath->setCurrentItem(i);
	 break;
       }
     }
     return;
  }

  if (portType==QButtonGroup_port->id(QRadioButton_parallel)) {
     QTextLabel_portPath->setText("Set Port Path for \n"
				  "parallel port");
     QComboBox_portPath->setEditable(true);
     QComboBox_portPath->clear();
     return;
  }


  if (portType==QButtonGroup_port->id(QRadioButton_usb)) {
    QTextLabel_portPath->setText("No further configuration\n"
				 "needed for usb");
    QComboBox_portPath->clear();
    QComboBox_portPath->setEditable(false);
    return;
  }
 
}



void Setup::slot_browse() {
  
  QString result = 
    KFileDialog::getExistingDirectory( QComboBox_downloadDir->text(QComboBox_downloadDir->currentItem()),
				       this );
  if (!result.isEmpty()) {
    for (int i = 0; i < QComboBox_downloadDir->count(); i++)
      if (result == QComboBox_downloadDir->text(i)) {
	QComboBox_downloadDir->setCurrentItem(i);
	return;
      }
    QComboBox_downloadDir->insertItem(result, 0);
    QComboBox_downloadDir->setCurrentItem(0);
  }
  
}
