/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2009-08-08
 * Description : an option to add a sequence number to the parser
 *
 * Copyright (C) 2009 by Andi Clemens <andi dot clemens at gmx dot net>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include "sequencenumberoption.moc"

// Qt includes

#include <QGridLayout>
#include <QGroupBox>
#include <QLabel>
#include <QPointer>

// KDE includes

#include <kiconloader.h>
#include <klocale.h>
#include <knuminput.h>

// Local includes

#include "ui_sequencenumberoptiondialogwidget.h"

namespace Digikam
{

SequenceNumberDialog::SequenceNumberDialog(Parseable* parent)
                    : ParseableDialog(parent), ui(new Ui::SequenceNumberOptionDialogWidget())
{
    QWidget* mainWidget = new QWidget(this);
    ui->setupUi(mainWidget);
    setSettingsWidget(mainWidget);
    ui->digits->setFocus();
}

SequenceNumberDialog::~SequenceNumberDialog()
{
    delete ui;
}

// --------------------------------------------------------

SequenceNumberOption::SequenceNumberOption()
                    : Option(i18nc("Sequence Number", "Number..."), i18n("Add a sequence number"),
                             SmallIcon("accessories-calculator"))
{
    addToken("#",                                   i18n("Sequence number"));
    addToken("#[||options||]",                      i18n("Sequence number (||options||: ||e|| = extension aware)"));
    addToken("#[||options||,||start||]",            i18n("Sequence number (custom start)"));
    addToken("#[||options||,||start||,||step||]",   i18n("Sequence number (custom start + step)"));

    QRegExp reg("(#+)(\\[(e,?)?((\\d+)(,(\\d+))?)?\\])?");
    setRegExp(reg);
}

void SequenceNumberOption::slotTokenTriggered(const QString& token)
{
    Q_UNUSED(token)

    QPointer<SequenceNumberDialog> dlg = new SequenceNumberDialog(this);

    QString result;
    if (dlg->exec() == KDialog::Accepted)
    {
        int digits          = dlg->ui->digits->value();
        int start           = dlg->ui->start->value();
        int step            = dlg->ui->step->value();
        bool extensionAware = dlg->ui->extensionAware->isChecked();

        result = QString("%1").arg("#", digits, QChar('#'));

        if (start > 1 || step > 1 || extensionAware)
        {
            result.append(QChar('['));

            if (extensionAware)
            {
                result.append(QChar('e'));
            }

            if (start > 1 || step > 1)
            {
                if (extensionAware)
                {
                    result.append(QChar(','));
                }
                result.append(QString::number(start));
            }

            if (step > 1)
            {
                result.append(QString(",%1").arg(QString::number(step)));
            }

            result.append(QChar(']'));
        }
    }

    delete dlg;

    emit signalTokenTriggered(result);
}

QString SequenceNumberOption::parseOperation(ParseSettings& settings)
{
    QString result;
    const QRegExp& reg = regExp();

    int slength = 0;
    int start   = 0;
    int step    = 0;
    int number  = 0;
    int index   = 0;
//    settings.currentIndex;

    if (settings.manager)
    {
        if (reg.cap(3).isEmpty())
        {
            index = settings.manager->indexOfFile(settings.fileUrl.toLocalFile());
        }
        else
        {
            index = settings.manager->indexOfFileGroup(settings.fileUrl.toLocalFile());
        }
    }

    // --------------------------------------------------------

    slength            = reg.cap(1).length();
    start              = reg.cap(5).isEmpty() ? settings.startIndex : reg.cap(5).toInt();
    step               = reg.cap(7).isEmpty() ? 1 : reg.cap(7).toInt();

    number             = start + ((index - 1) * step);
    result             = QString("%1").arg(number, slength, 10, QChar('0'));

    return result;
}

} // namespace Digikam
