/*
    $Id: calendarimap.cpp,v 1.1.2.66 2003/06/20 07:22:25 rogowski Exp $
    This file is part of libkcal.
    Copyright (c) 2002 Klarlvdalens Datakonsult AB <info@klaralvdalens-datakonsult.se>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include "calendarimap.h"
#include "calendarlocal.h"
// Just for some enums
#include "scheduler.h"

#include "journal.h"

#include <qtimer.h>

#include <kdebug.h>
#include <kprocess.h>
#include <kapplication.h>
#include <kstandarddirs.h>
#include <kconfig.h>
#include <kinstance.h>
#include <kglobal.h>
#include <dcopclient.h>
#include <qfile.h>
#include <qdatastream.h>
#include <stdlib.h>

using namespace KCal;

CalendarIMAP::CalendarIMAP( const QString& organizerEmail ) : mTarget(0)
{
  mLocalCalendar = new CalendarLocal();
  init( organizerEmail );
}

CalendarIMAP::CalendarIMAP( const QString& organizerEmail, const QString& timeZoneId )
  : Calendar( timeZoneId ), mTarget(0)
{
  mLocalCalendar = new CalendarLocal( timeZoneId );
  init( organizerEmail );
}

CalendarIMAP::~CalendarIMAP()
{
  delete mLocalCalendar;
}

void CalendarIMAP::init( const QString& organizerEmail )
{
  mOrganizerEmail = organizerEmail;
  mSilent = false;
  mForceAddDelete = false;
  mCurrentUID = QString::null;
  mBulkUpdate = 0;
  mUpdateViewDeferred = false;
  mEventsChangedDeferred = false;
  mSetupAlarmsDeferred = false;
  mIncidenceTimer = new QTimer( this );
  mAlarmTimer = new QTimer( this );
  mTempEventList = 0;
  mTempTaskList = 0;
  connect( mIncidenceTimer, SIGNAL( timeout() ),
	   this, SLOT( slotDoIncidenceUpdated() ) );
  connect( mAlarmTimer, SIGNAL( timeout() ),
	   this, SLOT( slotDoAlarmUpdate() ) );
}

void CalendarIMAP::setTarget( QObject* target )
{
  if( mTarget ) {
    // Disconnect the old target
    disconnect( mTarget );
    mTarget->disconnect( this );
  }

  mTarget = target;

  if( !connect( this, SIGNAL( incidenceUpdated( const QString&, const QString&,
						const QString& ) ),
		mTarget, SLOT( slotUpdateIncidence( const QString&, const QString&,
						    const QString& ) ) ) )
    qFatal("Connection between the KOrganizer part and KMail could not be made");
  if( !connect( this, SIGNAL( incidenceDeleted( const QString&, const QString& ) ),
		mTarget, SLOT( slotDeleteIncidence( const QString&, const QString& ) ) ) )
    qFatal("Connection between the KOrganizer part and KMail could not be made");
  if( !connect( this, SIGNAL( doSendICalMessage( const QString&, const QString&,
						 const QStringList&, const QString& ) ),
		mTarget, SLOT( slotSendICalMessage( const QString&, const QString&,
						    const QStringList&, const QString& ) ) ) )
    qFatal("Connection between the KOrganizer part and KMail could not be made");
  if( !connect( this, SIGNAL( getStripAlarmsForSending( bool& ) ),
		mTarget, SLOT( slotGetStripAlarmsForSending( bool& ) ) ) )
    qFatal("Connection between the KOrganizer part and KMail could not be made");
   if( !connect( this, SIGNAL( conflict( const QString& ) ),
  	mTarget, SLOT( slotConflictResolved( const QString& ) ) ) )
   qFatal("Connection between the KOrganizer part and KMail could not be made");
  if( !connect( mTarget, SIGNAL( incidenceAdded(const QString&,const QString&, bool& ) ),
		this, SLOT( slotAddIncidence(const QString&,const QString&, bool& ) ) ) )
    qFatal("Connection between the KOrganizer part and KMail could not be made");
  if( !connect( mTarget, SIGNAL( incidenceDeleted(const QString&,const QString&) ),
		this, SLOT( slotDeleteIncidence(const QString&,const QString&) ) ) )
    qFatal("Connection between the KOrganizer part and KMail could not be made");
  if( !connect( this, SIGNAL( initIncidences( const QString& ) ),
		mTarget, SLOT( slotInitIncidences( const QString& ) ) ) )
    qFatal("Connection between the KOrganizer part and KMail could not be made");
  if( !connect( mTarget, SIGNAL( signalCalendarFolderExpunged() ),
		this, SLOT( slotCalendarFolderExpunged() ) ) )
    qFatal("Connection between the KOrganizer part and KMail could not be made");
  if( !connect( mTarget, SIGNAL( signalTasksFolderExpunged() ),
		this, SLOT( slotTasksFolderExpunged() ) ) )
    qFatal("Connection between the KOrganizer part and KMail could not be made");
  if( !connect( mTarget, SIGNAL( signalRefreshAll() ), this, SLOT( slotRefreshAll() ) ) )
    qFatal("Connection between the KOrganizer part and KMail could not be made");

  // Re-read everything
  slotRefreshAll();
}


void CalendarIMAP::slotRefreshAll()
{
  startBulkUpdate();

  // Delete everything
  slotCalendarFolderExpunged();
  slotTasksFolderExpunged();

  // Now get all appointments and tasks
  emit initIncidences( "Calendar" );
  emit initIncidences( "Task" );
  // emit initIncidences( "Journal" );

  stopBulkUpdate();
}

void CalendarIMAP::close()
{
  if( mLocalCalendar )
    mLocalCalendar->close();
}


void CalendarIMAP::addEvent( Event* aEvent )
{
  internalAddEvent( aEvent);
}

bool CalendarIMAP::internalAddEvent( Event* aEvent )
{
  if( aEvent == 0 )
    qFatal( "CalendarIMAP::addEvent( 0 )" );

  bool addDuplicate = false;
  bool silent = mSilent;
  if ( mForceAddDelete ) {
    // introduced for syncing with palm pilot
    // we do not want to be asked for every synced item
    // If an event with this uid was already here, delete it
    Event* ev;
    ev = event( aEvent->uid() );
    if( ev ) {
      if ( mTempEventList ) {
	mTempEventList->remove( ev );
      }
      // Potential conflict situation
      if( *ev == *aEvent ) {
	delete aEvent;
	return false;
      } else {
     }
    } else {
      // new event from the palm
      // check if it is private and then set Transparent
      // such that the event is not published on freebusy list per default
      if ( aEvent->secrecy() == Event::SecrecyPrivate ) {
	aEvent->setTransparency( Event::Transparent );
      }
      if (aEvent->organizer() == "unknown@nowhere" )
	aEvent->setOrganizer( mOrganizerEmail );
    }
    deleteEvent ( event( aEvent->uid() ) );
  } else {
    Event* e = event( aEvent->uid() );
    if( e ) {
      // Potential conflict situation
      if( *e == *aEvent ) {
	// No conflict, the events are the same
	delete aEvent;
	return false;
      }

      kdDebug(5800) << "Conflict: " << aEvent->uid() << ", " << e->uid() << endl;
      // Mark the folder for re-syncing
      emit conflict( "Calendar" );

      int result = 0;
      emit incidenceConflict( aEvent, e, result );
      switch( result ) {
	// See korganizer/eventconflictdialog.h for these numbers
      case 1:
	kdDebug(5800) << "Conflict: Deleting server event\n";
	mSilent = false;
	// Delete the old one
	deleteEvent( e );
	// Delete the new one
	emitDeletedIncidence( "Calendar", aEvent );
	// Add the new one below
	break;
      case 2:
	kdDebug(5800) << "Conflict: Deleting local event\n";
	delete aEvent;
	return false;
      default:
	// Give the new one a new UID and add it like a completely new appointment
	kdDebug(5800) << "Conflict: Keeping both\n";
	addDuplicate = true;
	mSilent = false;
	aEvent->setUid( CalFormat::createUniqueId() );
      }
    }
  }

  // Add the new one
  mLocalCalendar->addEvent( aEvent );
  aEvent->registerObserver( this );

  // Tell mail client about the change
  if( !mSilent || addDuplicate ) {
    mCurrentUID = aEvent->uid();
    emitUpdatedIncidence( "Calendar", aEvent );
    mCurrentUID = QString::null;
  }
  emitEventsChanged();

  mSilent = silent;

  // If we're adding a duplicate, we want KMail to ditch the old version with the
  // duplicate UID
  return !addDuplicate;
}


void CalendarIMAP::deleteEvent( Event* aEvent )
{
  if( !aEvent )
    return;

  mUpdatedIncidences.removeRef(aEvent);
  if( !mSilent ) {
    mCurrentUID = aEvent->uid();
    // Tell mail client about the deletion
    emitDeletedIncidence( "Calendar", aEvent );
    mCurrentUID = QString::null;
  }
  mLocalCalendar->deleteEvent( aEvent );
  emitEventsChanged();
}

Event* CalendarIMAP::event( const QString& uniqueStr )
{
  return mLocalCalendar->event( uniqueStr );
}


QPtrList<Event> CalendarIMAP::events()
{
  return mLocalCalendar->events();
}


QPtrList<Event> CalendarIMAP::rawEvents()
{
  return mLocalCalendar->rawEvents();
}


int CalendarIMAP::numEvents( const QDate& qd )
{
  return mLocalCalendar->numEvents( qd );
}


void CalendarIMAP::addTodo( Todo* aTodo )
{
  internalAddTodo( aTodo);
}

bool CalendarIMAP::internalAddTodo( Todo* aTodo )
{
  if( aTodo == 0 )
    qFatal( "CalendarIMAP::addTodo( 0 )" );

  // If a todo with this uid was already here, delete it 
  bool addDuplicate = false;
  bool silent = mSilent;
  if ( mForceAddDelete ) {
    // introduced for syncing with palm pilot
    // we do not want to be asked for every synced item
    // If a todo with this uid was already here, delete it
    Todo* t = todo( aTodo->uid() );
    if( t ) {
      if ( mTempTaskList ) {
	mTempTaskList->remove( t );
      }
      if( *t == *aTodo ) {
	// No conflict, the todos are the same
	delete aTodo;
	return false; 
      }
    }
    deleteTodo( todo( aTodo->uid() ) );
  } else {
    Todo* t = todo( aTodo->uid() );
    if( t ) {
      // Potential conflict situation
      if( *t == *aTodo ) {
	// No conflict, the todos are the same
	delete aTodo;
	return false;
      }
      kdDebug(5800) << "Conflict: " << aTodo->uid() << ", " << t->uid() << endl;
    // Mark the folder for re-syncing
    emit conflict( "Todo" );

    int result = 0;
    emit incidenceConflict( aTodo, t, result );
    switch( result ) {
      // See korganizer/incidenceconflictdialog.h for these numbers
    case 1:
      kdDebug(5800) << "Conflict: Deleting server task\n";
      mSilent = false;
      // Delete the old one
      deleteTodo( t );
      // Delete the new one
      emitDeletedIncidence( "Calendar", aTodo );
      // Add the new one below
      break;
    case 2:
      kdDebug(5800) << "Conflict: Deleting new todo\n";
      delete aTodo;
      return false;
    default:
      // Give the new one a new UID and add it like a completely new todo
      kdDebug(5800) << "Conflict: Keeping both\n";
      addDuplicate = true;
      mSilent = false;
      aTodo->setUid( CalFormat::createUniqueId() );
    }
  }
}
  // Add the new one
  mLocalCalendar->addTodo( aTodo );
  aTodo->registerObserver( this );

  // Tell mail client about the change
  if( !mSilent || addDuplicate ) {
    mCurrentUID = aTodo->uid();
    emitUpdatedIncidence( "Task", aTodo );
    mCurrentUID = QString::null;
  }

  setModified( true );
  emitUpdateView();

  mSilent = silent;

  // If we're adding a duplicate, we want KMail to ditch the old version with the
  // duplicate UID
  return !addDuplicate;
}


void CalendarIMAP::deleteTodo( Todo* aTodo )
{
  if( !aTodo )
    return;

  // Tell mail client about the deletion
  mUpdatedIncidences.removeRef(aTodo);
  if( !mSilent ) {
    mCurrentUID = aTodo->uid();
    emitDeletedIncidence( "Task", aTodo );
    mCurrentUID = QString::null;
  }

  mLocalCalendar->deleteTodo( aTodo );

  setModified( true );
  emitUpdateView();
}

QPtrList<Todo> CalendarIMAP::todos()
{
  return mLocalCalendar->todos();
}


Todo* CalendarIMAP::todo( const QString& uid )
{
  return mLocalCalendar->todo( uid );
}


QPtrList<Todo> CalendarIMAP::todos( const QDate& date )
{
  return mLocalCalendar->todos( date );
}


QPtrList<Todo> CalendarIMAP::rawTodos() const
{
  return mLocalCalendar->rawTodos();
}


void CalendarIMAP::addJournal( Journal* journal )
{
  mLocalCalendar->addJournal( journal );
  journal->registerObserver( this );
  emitUpdateView();
}

Journal* CalendarIMAP::journal( const QDate& date )
{
  return mLocalCalendar->journal( date );
}


Journal* CalendarIMAP::journal( const QString& UID )
{
  return mLocalCalendar->journal( UID );
}


QPtrList<Journal> CalendarIMAP::journals()
{
  return mLocalCalendar->journals();
}


Alarm::List CalendarIMAP::alarms( const QDateTime &from,
                                  const QDateTime &to )
{
  return mLocalCalendar->alarms( from, to );
}


QPtrList<Event> CalendarIMAP::rawEventsForDate( const QDateTime &qdt )
{
  return mLocalCalendar->rawEventsForDate( qdt );
}


QPtrList<Event> CalendarIMAP::rawEventsForDate( const QDate &date,
                                                bool sorted )
{
  return mLocalCalendar->rawEventsForDate( date, sorted );
}


QPtrList<Event> CalendarIMAP::rawEvents( const QDate &start, const QDate &end,
                                         bool inclusive )
{
  return mLocalCalendar->rawEvents( start, end, inclusive );
}


/*!
  This method passes a new or updated event that has been created in
  KOrganizer to KMail for storage or sending to the recipients.
*/

void CalendarIMAP::emitUpdatedIncidence( const QString& type,
					 Incidence* incidence )
{
  mFormat.setTimeZone( timeZoneId(), !isLocalTime() );

  // The method is always Request, never Refresh. At least, that's
  // what Outlook does.
  QString messageText( mFormat.createScheduleMessage(incidence,
						     Scheduler::Request) );
  emit incidenceUpdated( type, incidence->uid(), messageText );
}

void CalendarIMAP::emitDeletedIncidence( const QString& type, Incidence* incidence )
{
  emit incidenceDeleted( type, incidence->uid() );
}

void CalendarIMAP::sendICalMessage( Scheduler::Method method, Incidence* i )
{
  mFormat.setTimeZone( timeZoneId(), !isLocalTime() );

  Incidence* incidence = i->clone();

  bool stripAlarms = false;
  emit getStripAlarmsForSending( stripAlarms );
  if( stripAlarms )
    // Strip alarms from the send
    incidence->clearAlarms();

  QStringList attendeeMailAddresses;
  if( mOrganizerEmail != incidence->organizer() && method == Scheduler::Reply ) {
    // reply to organizer
    attendeeMailAddresses << incidence->organizer();
  } else {
    QPtrList<Attendee> attendees = incidence->attendees();
    for( Attendee* attendee = attendees.first(); attendee; attendee = attendees.next() ) {
      // Omit the organizer, so that we don't send an email to ourselves
      if( attendee->email() != mOrganizerEmail )
	attendeeMailAddresses.append( attendee->email() );
    }
  }
  // The method is always Request, never Refresh. At least, that's
  // what Outlook does.
  QString messageText( mFormat.createScheduleMessage( incidence, method) );

  QString methodStr = Scheduler::methodName( method );
  emit doSendICalMessage( methodStr, incidence->summary(), attendeeMailAddresses,
			  messageText );
  delete incidence;
}
void CalendarIMAP::loadAndDelete( const QString& filename )
{
  QPtrList<Event> myEvents = mLocalCalendar->events();
  mTempEventList = &myEvents;
  QPtrList<Todo> myTasks = mLocalCalendar->todos();
  mTempTaskList = &myTasks;
  load( filename );
  if (  mTempEventList->count() ) {
    // kdDebug(5800) << "events in list %d " << mTempEventList->count() << endl;
    for( Event* e = mTempEventList->first(); e; e = mTempEventList->next() ) {
      deleteEvent( e );
    }
  }
  if (  mTempTaskList->count() ) {
    // kdDebug(5800) << "tasks in list %d " << mTempTaskList->count() << endl;
    for( Todo* t = mTempTaskList->first(); t; t = mTempTaskList->next() ) {
      deleteTodo( t );
    }
  }
  mTempTaskList = 0;
  mTempEventList = 0;
}
/*!
  Loads calendar data into the local calendar.
  This does not clear the current calendar.
*/

bool CalendarIMAP::load( const QString& filename )
{
  CalendarLocal tmpCalendar( timeZoneId() );

  if( !tmpCalendar.load( filename ) )
    return false;
  startBulkUpdate();
  bool force = mForceAddDelete;
  mForceAddDelete = true;
  QPtrList<Event> events = tmpCalendar.rawEvents();
  for( Event* event = events.first(); event; event = events.next() ) {
    addEvent( event->clone() );
  }
  
  QPtrList<Todo> todos = tmpCalendar.rawTodos();
  for( Todo* todo = todos.first(); todo; todo = todos.next() )
    addTodo( todo->clone() );

#if 0
  // TODO: journals
  QPtrList<Journal> journals = tmpCalendar.rawJournals();
  for( Journal* journal = journals.first(); journal; journal = journals.next() )
    addJournal( journal->clone() );
#endif
  mForceAddDelete = force;
  stopBulkUpdate();
  return true;
}


/*!
  Saves the calendar data in the local calendar into a file.
*/

bool CalendarIMAP::save( const QString& filename, CalFormat* format )
{
  return mLocalCalendar->save( filename, format );
}


void CalendarIMAP::setupAlarm()
{
  if( mBulkUpdate != 0 ) {
    // Deferring the setup
    mSetupAlarmsDeferred = true;
    return;
  }
  mSetupAlarmsDeferred = false;

  kdDebug(5800) << "Starting alarm timer\n";
  mAlarmTimer->start( 5000, true );
}

void CalendarIMAP::slotDoAlarmUpdate()
{
  kdDebug(5800) << "Running alarm timer\n";
  static bool firstRun = true;
  static bool isInSetupAlarm = false;
  if( isInSetupAlarm )
    return;

  isInSetupAlarm = true;

  // Store the calendar in a file
  QString calFile = locateLocal( "appdata", "kmailalarms.ics" );

  if( save( calFile ) ) {
    DCOPClient* dcopClient = kapp->dcopClient();
    QByteArray params;
    QDataStream ds( params, IO_WriteOnly );
    ds << QCString( "korgac" ) << calFile;

    bool rc;
    if( firstRun ) {
      rc = dcopClient->send( "kalarmd", "ad", "addCal(QCString,QString)", params );
      firstRun = false;
    } else
      rc = dcopClient->send( "kalarmd", "ad", "reloadCal(QCString,QString)", params );
    if( !rc )
      kdDebug(5800) << "Could not send reloadCal() DCOP call to kalarmd" << endl;
  }
  isInSetupAlarm = false;
}

void CalendarIMAP::incidenceUpdated( KCal::IncidenceBase* aIncidenceBase )
{
  if( !aIncidenceBase ) return;
  if( !mUpdatedIncidences.contains(aIncidenceBase) ) mUpdatedIncidences.append(aIncidenceBase);
  mIncidenceTimer->start( 0, true );
}

// Invoked by the timer started by incidenceUpdated()
void CalendarIMAP::slotDoIncidenceUpdated()
{
  IncidenceBase* aIncidenceBase = 0;

  startBulkUpdate();
  while( ( aIncidenceBase = mUpdatedIncidences.first() ) ) {
    mUpdatedIncidences.removeFirst();
    QString type = aIncidenceBase->type();
    // Tell mail client about the change
    emitUpdatedIncidence( type, static_cast<Incidence*>(aIncidenceBase) );

    if( type == "Event" )
      emitEventsChanged();
    else
      emitUpdateView();
  }
  stopBulkUpdate();
}

KCal::Incidence* CalendarIMAP::parseIncidence( const QString& str )
{
  mFormat.setTimeZone( timeZoneId(), !isLocalTime() );
  Incidence* i = mFormat.fromString( str );
  return i;
}


void CalendarIMAP::slotAddIncidence( const QString& type, const QString& ical, bool& accepted )
{
  // kdDebug(5800) << "CalendarIMAP::addIncidence( " << type << ", " << ical << " )" << endl;
  Incidence* i = parseIncidence( ical );
  if( !i ) {
    accepted = false;
    return;
  }

  // Ignore events that come from us
  if( !mCurrentUID.isNull() && mCurrentUID == i->uid() ) {
    accepted = true;
    return;
  }

  bool oldSilent = mSilent;
  mSilent = true;
  // Only add an iCal coming from the right folder type
  if( type == "Calendar" && i->type() == "Event" )
    accepted = internalAddEvent( static_cast<Event*>(i) );
  else if( type == "Task" && i->type() == "Todo" )
    accepted = internalAddTodo( static_cast<Todo*>(i) );
  else {
    accepted = false;
    delete i;
  }
  mSilent = oldSilent;
}

void CalendarIMAP::slotDeleteIncidence( const QString& type, const QString& uid )
{
  // kdDebug(5800) << "CalendarIMAP::deleteIncidence( " << type << ", " << uid << " )" << endl;
  // Ignore events that come from us
  if( !mCurrentUID.isNull() && mCurrentUID == uid ) return;

  bool oldSilent = mSilent;
  mSilent = true;
  if( type == "Calendar" ) {
    Event* e = event(uid);
    if( e ) deleteEvent(e);
  } else if( type == "Task" ) {
    Todo* t = todo(uid);
    if( t ) deleteTodo(t);
  } 
  else if( type == "Journal" ) {
    // TODO: Journals
    qFatal( "Journals are not implemented yet" );
#if 0
    Journal* j = journal(uid);
    if( j ) deleteJournal(j);
#endif
  }
  mSilent = oldSilent;
}

void CalendarIMAP::slotCalendarFolderExpunged()
{
  QPtrList<Event> eventList = events();

  for ( Event* e = eventList.first(); e; e = eventList.next() )
    mLocalCalendar->deleteEvent( e );
  emitUpdateView();
}

void CalendarIMAP::slotTasksFolderExpunged()
{
  QPtrList<Todo> tasks = todos();

  for ( Todo* t = tasks.first(); t; t = tasks.next() )
    mLocalCalendar->deleteTodo( t );
  emitUpdateView();
}

void CalendarIMAP::startBulkUpdate()
{
   ++mBulkUpdate;
}

void CalendarIMAP::stopBulkUpdate()
{
  if( mBulkUpdate == 0 )
    // Strange
    return;

  if( --mBulkUpdate == 0 ) {
    if( mEventsChangedDeferred )
      emitEventsChanged();
    if( mUpdateViewDeferred )
      emitUpdateView();
    if( mSetupAlarmsDeferred )
      setupAlarm();
  }
}

void CalendarIMAP::emitUpdateView()
{
  if( mBulkUpdate == 0 ) {
    mUpdateViewDeferred = false;
    emit updateView();
    setupAlarm();
  } else
    mUpdateViewDeferred = true;
}

void CalendarIMAP::emitEventsChanged()
{
  if( mBulkUpdate == 0 ) {
    mEventsChangedDeferred = false;
    emit eventsChanged();
    emitUpdateView();
  } else
    mEventsChangedDeferred = true;
}

void CalendarIMAP::slotSyncRunning( const QString& type, bool running )
{
  if( type == "Event" || type == "Calendar" ||
      type == "Task" || type == "Todo" ) {
    if( running )
      startBulkUpdate();
    else
      stopBulkUpdate();
  }
}

#include "calendarimap.moc"
