<?php
/**
 * The RPC_xmlrpc class provides an XMLRPC implementation of the
 * Horde RPC system.
 *
 * $Horde: horde/lib/RPC/xmlrpc.php,v 1.5 2003/05/15 14:36:59 chuck Exp $
 *
 * Copyright 2002-2003 Jan Schneider <jan@horde.org>
 *
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * @author  Jan Schneider <jan@horde.org>
 * @version $Revision: 1.5 $
 * @since   Horde 3.0
 * @package horde.rpc
 */
class RPC_xmlrpc extends RPC {

    /**
     * Resource handler for the RPC server.
     * @var object $server
     */
    var $_server;

    /**
     * XMLRPC server constructor
     *
     * @access private
     * @return object   A RPC server instance
     */
    function RPC_xmlrpc()
    {
        parent::RPC();

        $this->_server = xmlrpc_server_create();

        foreach ($GLOBALS['registry']->listMethods() as $method) {
            xmlrpc_server_register_method($this->_server, str_replace('/', '.', $method), array('RPC', '_dispatcher'));
        }
    }

    /**
     * Cleans up the RPC server.
     */
    function shutdown()
    {
        xmlrpc_server_destroy($this->_server);
    }

    /**
     * Sends an RPC request to the server and returns the result.
     *
     * @param string    The raw request string.
     *
     * @return string   The XML encoded response from the server.
     */
    function getResponse($request)
    {
        $response = null;
        return xmlrpc_server_call_method($this->_server, $request, $response);
    }

    /**
     * Builds an XMLRPC request and sends it to the XMLRPC server.
     *
     * This statically called method is actually the XMLRPC client.
     *
     * @param string $url       The path to the XMLRPC server on the called host.
     * @param string $method    The method to call.
     * @param array $params     (optional) A hash containing any necessary
     *                          parameters for the method call.
     * @param $options  Optional associative array of parameters which can be:
     *                  user           - Basic Auth username
     *                  pass           - Basic Auth password
     *                  proxy_host     - Proxy server host
     *                  proxy_port     - Proxy server port
     *                  proxy_user     - Proxy auth username
     *                  proxy_pass     - Proxy auth password
     *                  timeout        - Connection timeout in seconds.
     *                  allowRedirects - Whether to follow redirects or not
     *                  maxRedirects   - Max number of redirects to follow
     *
     * @return mixed            The returned result from the method or a PEAR
     *                          error object on failure.
     */
    function request($url, $method, $params = null, $options = array())
    {
        $options['method'] = 'POST';
        $language = isset($GLOBALS['language']) ? $GLOBALS['language'] :
                    (isset($_SERVER['LANG']) ? $_SERVER['LANG'] : '');

        if (!isset($options['timeout'])) {
            $options['timeout'] = 5;
        }
        if (!isset($options['allowRedirects'])) {
            $options['allowRedirects'] = true;
            $options['maxRedirects'] = 3;
        }

        require_once 'HTTP/Request.php';
        require_once dirname(__FILE__) . '/../version.php';

        $http = &new HTTP_Request($url, $options);
        if (!empty($language)) {
            $http->addHeader('Accept-Language', $language);
        }
        $http->addHeader('User-Agent', 'Horde ' . HORDE_VERSION . ' RPC client');
        $http->addHeader('Content-Type', 'text/xml');
        $http->addRawPostData(xmlrpc_encode_request($method, $params));

        $http->sendRequest();
        if ($http->getResponseCode() != 200) {
            return PEAR::raiseError(_("Request couldn't be answered. Returned errorcode: ") . $http->getResponseCode(), 'horde.error');
        } elseif (!strstr($http->getResponseBody(), '<?xml')) {
            return PEAR::raiseError(_("No valid XML data returned"), 'horde.error', null, null, $http->getResponseBody());
        } else {
            $response = @xmlrpc_decode(substr($http->getResponseBody(), strpos($http->getResponseBody(), '<?xml')));
            if (is_array($response) && isset($response[0]['faultString'])) {
                return PEAR::raiseError($response[0]['faultString'], 'horde.error');
            }
            return $response;
        }
    }

}
