# frozen_string_literal: true

require 'json'

module GitlabQuality
  module TestTooling
    module CodeCoverage
      class TestMap
        SEPARATOR = '/'
        MARKER = 1

        # @param [Hash] compact_map A nested hash structure where keys are
        #   source files and values are tree structures with test file paths
        # @example Example of compact_map for a file tested by 3 spec files
        #   {
        #     "app/models/user.rb" => {
        #       "spec" => {
        #         "models" => {
        #           "user_spec.rb" => 1  # MARKER (1) indicates a leaf node
        #         }
        #       },
        #       "ee" => {
        #         "spec" => {
        #           "lib" => {
        #              "ee" => {
        #                "gitlab" => {
        #                  "background_migration" => {
        #                    "delete_invalid_epic_issues_spec.rb"=>1,
        #                    "backfill_security_policies_spec.rb"=>1
        #                  }
        #                }
        #              }
        #           }
        #         }
        #       }
        #     }
        #   }
        def initialize(compact_map)
          @compact_map = compact_map
        end

        # @return [Hash<String, Array<String>>] Source files mapped to all test
        #   files testing them
        # @example Return value
        #   {
        #     "path/to/file1.rb" => [
        #       "spec/path/to/file1_spec.rb",
        #       "spec/path/to/another/file1_spec.rb"
        #     ],
        #     ...
        #   }
        def source_to_tests
          @source_to_tests ||= @compact_map.transform_values { |tree| traverse(tree).to_a.uniq }
        end

        # @return [Hash<String, Array<String>>] Test files mapped to all source
        #   files tested by them
        # @example Return value
        #   {
        #     "spec/path/to/file1_spec.rb" => [
        #       "path/to/file1.rb",
        #       "path/to/file2.rb"
        #     ],
        #     ...
        #   }
        def test_to_sources
          @test_to_sources ||= begin
            test_to_sources = Hash.new { |hash, key| hash[key] = [] }

            @compact_map.each do |source_file, tree|
              traverse(tree).to_a.each do |test_file|
                test_to_sources[test_file] << source_file
              end
            end

            test_to_sources
          end
        end

        private

        def traverse(tree, segments = [], &block)
          return to_enum(__method__, tree, segments) unless block
          return yield segments.join(SEPARATOR) if tree == MARKER && !segments.empty?

          tree.each do |key, value|
            traverse(value, segments + [key], &block)
          end
        end
      end
    end
  end
end
