# frozen_string_literal: true

module GitlabQuality
  module TestTooling
    module Report
      module GroupIssues
        class IssueFormatter
          def generate_issue_title(grouped_failure)
            case grouped_failure[:pattern_name]
            when /http_500/ then "Environment Issue: HTTP 500 Internal Server Errors"
            when /http_400/ then "Environment Issue: Backend Connection Failures"
            when /http_503/ then "Environment Issue: Service Unavailable (503)"
            when /timeout/ then "Environment Issue: Timeout Failures"
            when /git_rpc|repository/ then "Environment Issue: Repository/Git Operation Failures"
            else "Environment Issue: Multiple Similar Failures"
            end
          end

          def generate_issue_description(grouped_failure, options = {})
            active_incidents = IncidentChecker.get_active_incidents(token: options[:token])
            incident_section = IncidentChecker.format_incidents_for_issue(active_incidents)

            <<~MARKDOWN
            ## Environment Issue: #{grouped_failure[:pattern_name]}

            Multiple tests have failed with similar error patterns, indicating an environment-related issue affecting multiple test cases.

            ### Error Pattern
            ```
            #{grouped_failure[:normalized_message]}
            ```

            ### Affected Tests (#{grouped_failure[:failures].size} failures)
            #{format_affected_tests(grouped_failure[:failures])}

            ### Pipeline Information
            #{format_pipeline_info(grouped_failure[:failures].first)}

            ### Recommended Actions
            #{generate_recommended_actions(grouped_failure)}

            #{incident_section}
            ---
            <!-- grouped-failure-fingerprint:#{grouped_failure[:fingerprint]} -->
            MARKDOWN
          end

          def format_affected_tests(failures)
            failures.map do |failure|
              job_name = failure[:ci_job_url] || failure.dig(:ci_job, :name) || 'unknown_job'
              spec_file = failure[:file_path] || failure[:file] || 'unknown_spec'
              line_number = failure[:line_number] || failure[:line]
              test_name = failure[:description] || failure[:test_name] || 'Unknown test'

              spec_with_line = line_number.to_s.empty? ? spec_file : "#{spec_file}:#{line_number}"
              "- **#{test_name}** (Job: `#{job_name}`, Spec: `#{spec_with_line}`)"
            end.join("\n")
          end

          def format_pipeline_info(failure)
            pipeline_url = ENV['CI_PIPELINE_URL'] || "Pipeline #{ENV.fetch('CI_PIPELINE_ID', 'unknown')}"
            job_url = failure[:ci_job_url] || 'Unknown job'

            "- **Pipeline**: #{pipeline_url}\n- **Job**: #{job_url}"
          end

          def generate_recommended_actions(grouped_failure)
            case grouped_failure[:pattern_name]
            when /http_500/
              "1. Check GitLab instance status and logs\n2. Verify database connectivity\n3. Review application server health"
            when /timeout/
              "1. Check network connectivity\n2. Review timeout configurations\n3. Monitor system resources"
            when /git_rpc|repository/
              "1. Verify Git repository accessibility\n2. Check Gitaly service status\n3. Review storage capacity"
            else
              "1. Check if there are ongoing incidents affecting the GitLab instance\n2. Verify API endpoints are responding correctly\n3. Review system logs for related errors"
            end
          end
        end
      end
    end
  end
end
