# font_row.py
#
# Copyright 2024 Hari Rana (TheEvilSkeleton)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later


from typing import Any, cast

from gi.repository import GObject, Adw, Gtk, Gio, Pango, GLib

from refine.widgets.action_row import RefineActionRow
from refine.compositions import RefineCompositions

import logging


@Gtk.Template(resource_path="/page/tesk/Refine/widgets/font-row.ui")
class RefineFontRow(RefineActionRow):
    """
    A `RefineActionRow` for displaying and interacting with font.
    It is also used for launching the `Gtk.FontDialog`.
    """

    __gtype_name__ = "RefineFontRow"

    revert_button = Gtk.Template.Child()
    revealer = Gtk.Template.Child()

    is_default = GObject.Property(type=bool, default=True)

    def __init__(self, **kwargs: Any) -> None:
        super().__init__(**kwargs)

        self.connect("notify::key", self.__key_set)
        self.connect("activated", self.__row_activated)

    def __update_label(self) -> None:
        font_string = self.settings.get_string(self.key)
        default_font_string = self.settings.get_default_value(self.key).get_string()

        self.is_default = default_font_string != font_string

        font_description = Pango.FontDescription.from_string(font_string)
        font_family = font_description.get_family()

        self.set_subtitle(f"<span face='{font_family}'>{font_string}</span>")

    def __key_set(self, *args: Any) -> None:
        if not self.is_valid_setting:
            return

        self.__update_label()
        self.bind_property(
            "is-default",
            self.revert_button,
            "visible",
            GObject.BindingFlags.SYNC_CREATE,
        )
        self.revert_button.connect("clicked", self.__revert_button_clicked)
        RefineCompositions.set_up_separator_revealer(
            self.revealer, self.revert_button, "visible"
        )

    def __row_activated(self, *args: Any) -> None:
        def choose_font_and_features_callback(
            _dialog: Gtk.FontDialog, result: Gio.AsyncResult
        ) -> None:
            try:
                font = font_dialog.choose_font_and_features_finish(result)
                font_string = font[1].to_string()
            except (IndexError, GLib.GError):
                return

            self.settings.set_string(self.key, font_string)
            self.value_set(self.settings.get_string(self.key))
            self.__update_label()

        font_string = self.settings.get_string(self.key)
        font_description = Pango.FontDescription.from_string(font_string)
        font_dialog = Gtk.FontDialog()

        if window := cast(Adw.Window | None, self.get_root()):
            font_dialog.choose_font_and_features(
                window, font_description, callback=choose_font_and_features_callback
            )

    def __revert_button_clicked(self, *args: Any) -> None:
        logging.debug("Revert button pressed")
        self.settings.reset(self.key)
        self.__update_label()
