﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/connect/ConnectRequest.h>
#include <aws/connect/Connect_EXPORTS.h>
#include <aws/connect/model/RuleAction.h>
#include <aws/connect/model/RulePublishStatus.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Connect {
namespace Model {

/**
 */
class UpdateRuleRequest : public ConnectRequest {
 public:
  AWS_CONNECT_API UpdateRuleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateRule"; }

  AWS_CONNECT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A unique identifier for the rule.</p>
   */
  inline const Aws::String& GetRuleId() const { return m_ruleId; }
  inline bool RuleIdHasBeenSet() const { return m_ruleIdHasBeenSet; }
  template <typename RuleIdT = Aws::String>
  void SetRuleId(RuleIdT&& value) {
    m_ruleIdHasBeenSet = true;
    m_ruleId = std::forward<RuleIdT>(value);
  }
  template <typename RuleIdT = Aws::String>
  UpdateRuleRequest& WithRuleId(RuleIdT&& value) {
    SetRuleId(std::forward<RuleIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Amazon Connect instance. You can <a
   * href="https://docs.aws.amazon.com/connect/latest/adminguide/find-instance-arn.html">find
   * the instance ID</a> in the Amazon Resource Name (ARN) of the instance.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  UpdateRuleRequest& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the rule. You can change the name only if
   * <code>TriggerEventSource</code> is one of the following values:
   * <code>OnZendeskTicketCreate</code> | <code>OnZendeskTicketStatusUpdate</code> |
   * <code>OnSalesforceCaseCreate</code> </p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateRuleRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The conditions of the rule.</p>
   */
  inline const Aws::String& GetFunction() const { return m_function; }
  inline bool FunctionHasBeenSet() const { return m_functionHasBeenSet; }
  template <typename FunctionT = Aws::String>
  void SetFunction(FunctionT&& value) {
    m_functionHasBeenSet = true;
    m_function = std::forward<FunctionT>(value);
  }
  template <typename FunctionT = Aws::String>
  UpdateRuleRequest& WithFunction(FunctionT&& value) {
    SetFunction(std::forward<FunctionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of actions to be run when the rule is triggered.</p>
   */
  inline const Aws::Vector<RuleAction>& GetActions() const { return m_actions; }
  inline bool ActionsHasBeenSet() const { return m_actionsHasBeenSet; }
  template <typename ActionsT = Aws::Vector<RuleAction>>
  void SetActions(ActionsT&& value) {
    m_actionsHasBeenSet = true;
    m_actions = std::forward<ActionsT>(value);
  }
  template <typename ActionsT = Aws::Vector<RuleAction>>
  UpdateRuleRequest& WithActions(ActionsT&& value) {
    SetActions(std::forward<ActionsT>(value));
    return *this;
  }
  template <typename ActionsT = RuleAction>
  UpdateRuleRequest& AddActions(ActionsT&& value) {
    m_actionsHasBeenSet = true;
    m_actions.emplace_back(std::forward<ActionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The publish status of the rule.</p>
   */
  inline RulePublishStatus GetPublishStatus() const { return m_publishStatus; }
  inline bool PublishStatusHasBeenSet() const { return m_publishStatusHasBeenSet; }
  inline void SetPublishStatus(RulePublishStatus value) {
    m_publishStatusHasBeenSet = true;
    m_publishStatus = value;
  }
  inline UpdateRuleRequest& WithPublishStatus(RulePublishStatus value) {
    SetPublishStatus(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_ruleId;

  Aws::String m_instanceId;

  Aws::String m_name;

  Aws::String m_function;

  Aws::Vector<RuleAction> m_actions;

  RulePublishStatus m_publishStatus{RulePublishStatus::NOT_SET};
  bool m_ruleIdHasBeenSet = false;
  bool m_instanceIdHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_functionHasBeenSet = false;
  bool m_actionsHasBeenSet = false;
  bool m_publishStatusHasBeenSet = false;
};

}  // namespace Model
}  // namespace Connect
}  // namespace Aws
