﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/personalize/PersonalizeRequest.h>
#include <aws/personalize/Personalize_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Personalize {
namespace Model {

/**
 */
class ListRecommendersRequest : public PersonalizeRequest {
 public:
  AWS_PERSONALIZE_API ListRecommendersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListRecommenders"; }

  AWS_PERSONALIZE_API Aws::String SerializePayload() const override;

  AWS_PERSONALIZE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Domain dataset group to list the
   * recommenders for. When a Domain dataset group is not specified, all the
   * recommenders associated with the account are listed.</p>
   */
  inline const Aws::String& GetDatasetGroupArn() const { return m_datasetGroupArn; }
  inline bool DatasetGroupArnHasBeenSet() const { return m_datasetGroupArnHasBeenSet; }
  template <typename DatasetGroupArnT = Aws::String>
  void SetDatasetGroupArn(DatasetGroupArnT&& value) {
    m_datasetGroupArnHasBeenSet = true;
    m_datasetGroupArn = std::forward<DatasetGroupArnT>(value);
  }
  template <typename DatasetGroupArnT = Aws::String>
  ListRecommendersRequest& WithDatasetGroupArn(DatasetGroupArnT&& value) {
    SetDatasetGroupArn(std::forward<DatasetGroupArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token returned from the previous call to <code>ListRecommenders</code> for
   * getting the next set of recommenders (if they exist).</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListRecommendersRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of recommenders to return.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListRecommendersRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_datasetGroupArn;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_datasetGroupArnHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Personalize
}  // namespace Aws
