/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.security.configuration;

import com.google.common.collect.ImmutableMap;
import java.io.IOException;
import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.ExceptionsHelper;
import org.opensearch.ResourceAlreadyExistsException;
import org.opensearch.action.admin.cluster.health.ClusterHealthRequest;
import org.opensearch.action.admin.cluster.health.ClusterHealthResponse;
import org.opensearch.action.admin.indices.create.CreateIndexRequest;
import org.opensearch.action.admin.indices.create.CreateIndexResponse;
import org.opensearch.action.index.IndexRequest;
import org.opensearch.action.support.WriteRequest;
import org.opensearch.cluster.health.ClusterHealthStatus;
import org.opensearch.common.settings.Settings;
import org.opensearch.common.util.concurrent.ThreadContext;
import org.opensearch.common.xcontent.XContentType;
import org.opensearch.core.xcontent.MediaType;
import org.opensearch.index.engine.VersionConflictEngineException;
import org.opensearch.security.DefaultObjectMapper;
import org.opensearch.security.configuration.ClusterInfoHolder;
import org.opensearch.security.configuration.ConfigurationChangeListener;
import org.opensearch.security.configuration.ConfigurationMap;
import org.opensearch.security.configuration.ConfigurationRepository;
import org.opensearch.security.configuration.SecurityConfigDiffCalculator;
import org.opensearch.security.configuration.SecurityConfigVersionDocument;
import org.opensearch.security.configuration.SecurityConfigVersionsLoader;
import org.opensearch.security.securityconf.impl.CType;
import org.opensearch.security.securityconf.impl.SecurityDynamicConfiguration;
import org.opensearch.security.user.User;
import org.opensearch.threadpool.ThreadPool;
import org.opensearch.transport.client.Client;

public class SecurityConfigVersionHandler
implements ConfigurationChangeListener {
    private final int maxVersionsToKeep;
    private static final int MAX_RETRIES = 3;
    private static final long BASE_DELAY_MS = 200L;
    private static final Logger log = LogManager.getLogger(SecurityConfigVersionHandler.class);
    private final Client client;
    private final String securityConfigVersionsIndex;
    private final SecurityConfigVersionsLoader configVersionsLoader;
    private final ClusterInfoHolder clusterInfoHolder;
    private final ConfigurationRepository configurationRepository;
    private final Settings settings;
    private final ThreadContext threadContext;
    private final ThreadPool threadPool;

    public SecurityConfigVersionHandler(ConfigurationRepository configurationRepository, Settings settings, ThreadContext threadContext, ThreadPool threadPool, Client client, ClusterInfoHolder clusterInfoHolder) {
        this.configurationRepository = configurationRepository;
        this.settings = settings;
        this.threadContext = threadContext;
        this.client = client;
        this.securityConfigVersionsIndex = settings.get("plugins.security.config_versions_index_name", ".opensearch_security_config_versions");
        this.configVersionsLoader = new SecurityConfigVersionsLoader(client, settings);
        this.threadPool = threadPool;
        this.maxVersionsToKeep = settings.getAsInt("plugins.security.config_version.retention_count", Integer.valueOf(10));
        this.clusterInfoHolder = clusterInfoHolder;
    }

    @Override
    public void onChange(ConfigurationMap typeToConfig) {
        if (!Boolean.TRUE.equals(this.clusterInfoHolder.isLocalNodeElectedClusterManager())) {
            return;
        }
        if (!SecurityConfigVersionHandler.isVersionIndexEnabled(this.settings)) {
            return;
        }
        this.threadPool.generic().execute(() -> {
            ThreadContext threadContext = this.threadPool.getThreadContext();
            try (ThreadContext.StoredContext ctx = threadContext.stashContext();){
                log.debug("Initializing version index ({})", (Object)this.securityConfigVersionsIndex);
                if (!this.createOpendistroSecurityConfigVersionsIndexIfAbsent()) {
                    log.debug("Version index already exists, skipping initialization.");
                }
                this.waitForOpendistroSecurityConfigVersionsIndexToBeAtLeastYellow();
                String nextVersionId = this.fetchNextVersionId();
                User user = (User)threadContext.getTransient("_opendistro_security_user");
                String userinfo = user != null ? user.getName() : "system";
                SecurityConfigVersionDocument.Version<?> version = this.buildVersionFromSecurityIndex(nextVersionId, userinfo);
                this.saveCurrentVersionToSystemIndex(version);
            }
            catch (Exception e) {
                log.error("Failed to initialize config version index", (Throwable)e);
            }
        });
    }

    boolean createOpendistroSecurityConfigVersionsIndexIfAbsent() {
        try {
            ImmutableMap indexSettings = ImmutableMap.of((Object)"index.number_of_shards", (Object)1, (Object)"index.auto_expand_replicas", (Object)"0-all");
            Map<String, Map<String, Map<String, Map<String, Map<String, Boolean>>>>> mappings = Map.of("properties", Map.of("versions", Map.of("type", "object", "properties", Map.of("version_id", Map.of("type", "keyword"), "timestamp", Map.of("type", "date"), "modified_by", Map.of("type", "keyword"), "security_configs", Map.of("type", "object", "enabled", false)))));
            log.debug("Index request for {}", (Object)this.securityConfigVersionsIndex);
            CreateIndexRequest createIndexRequest = new CreateIndexRequest(this.securityConfigVersionsIndex).settings((Map)indexSettings).mapping(mappings);
            boolean ok = ((CreateIndexResponse)this.client.admin().indices().create(createIndexRequest).actionGet()).isAcknowledged();
            log.info("Index {} created?: {}", (Object)this.securityConfigVersionsIndex, (Object)ok);
            return ok;
        }
        catch (ResourceAlreadyExistsException resourceAlreadyExistsException) {
            log.debug("Index {} already exists", (Object)this.securityConfigVersionsIndex);
            return false;
        }
        catch (Exception e) {
            log.error("Failed to create index {}", (Object)this.securityConfigVersionsIndex, (Object)e);
            throw e;
        }
    }

    void waitForOpendistroSecurityConfigVersionsIndexToBeAtLeastYellow() {
        log.info("Node started, try to initialize it. Wait for at least yellow cluster state....");
        ClusterHealthResponse response = null;
        try {
            response = (ClusterHealthResponse)this.client.admin().cluster().health(new ClusterHealthRequest(new String[]{this.securityConfigVersionsIndex}).waitForActiveShards(1).waitForYellowStatus()).actionGet();
        }
        catch (Exception e) {
            log.debug("Caught a {} but we just try again ...", (Object)e.toString());
        }
        while (response == null || response.isTimedOut() || response.getStatus() == ClusterHealthStatus.RED) {
            log.debug("index '{}' not healthy yet, we try again ... (Reason: {})", (Object)this.securityConfigVersionsIndex, (Object)(response == null ? "no response" : (response.isTimedOut() ? "timeout" : "other, maybe red cluster")));
            try {
                TimeUnit.MILLISECONDS.sleep(500L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            try {
                response = (ClusterHealthResponse)this.client.admin().cluster().health(new ClusterHealthRequest(new String[]{this.securityConfigVersionsIndex}).waitForYellowStatus()).actionGet();
            }
            catch (Exception e) {
                log.debug("Caught again a {} but we just try again ...", (Object)e.toString());
            }
        }
    }

    public static boolean isVersionIndexEnabled(Settings settings) {
        return settings.getAsBoolean("plugins.security.configurations_versions.enabled", Boolean.valueOf(false));
    }

    public String fetchNextVersionId() {
        try {
            SecurityConfigVersionDocument.Version<?> latestVersion = this.configVersionsLoader.loadLatestVersion();
            if (latestVersion == null || latestVersion.getVersion_id() == null || !latestVersion.getVersion_id().startsWith("v")) {
                return "v1";
            }
            int currentVersionNumber = Integer.parseInt(latestVersion.getVersion_id().substring(1));
            return "v" + (currentVersionNumber + 1);
        }
        catch (Exception e) {
            log.error("Failed to fetch latest version from {}", (Object)this.securityConfigVersionsIndex, (Object)e);
            throw new RuntimeException("Failed to fetch next version id", e);
        }
    }

    public SecurityConfigVersionDocument.Version<?> buildVersionFromSecurityIndex(String versionId, String modified_by) throws IOException {
        Instant now = Instant.now();
        String timestamp = now.toString();
        SecurityConfigVersionDocument.Version version = new SecurityConfigVersionDocument.Version(versionId, timestamp, new HashMap(), modified_by);
        ConfigurationMap allConfigs = this.configurationRepository.getConfigurationsFromIndex(CType.values(), false);
        for (CType<?> cType : CType.values()) {
            SecurityDynamicConfiguration<?> dynamicConfig = allConfigs.get(cType);
            TreeMap configData = new TreeMap();
            if (dynamicConfig != null) {
                if (dynamicConfig.getCEntries() != null) {
                    configData.putAll(dynamicConfig.getCEntries());
                }
                if (dynamicConfig.get_meta() != null) {
                    Map metaMap = (Map)DefaultObjectMapper.objectMapper.convertValue((Object)dynamicConfig.get_meta(), Map.class);
                    configData.put("_meta", metaMap);
                }
            }
            version.addSecurityConfig(cType.toLCString(), new SecurityConfigVersionDocument.HistoricSecurityConfig(timestamp, configData));
        }
        return version;
    }

    public <T> void saveCurrentVersionToSystemIndex(SecurityConfigVersionDocument.Version<T> version) {
        try {
            SecurityConfigVersionDocument document = this.configVersionsLoader.loadFullDocument();
            if (this.shouldSkipVersionUpdate(document, version)) {
                return;
            }
            document.addVersion(version);
            this.writeSecurityConfigVersion(document, document.getSeqNo(), document.getPrimaryTerm());
            log.info("Successfully saved version {} to {}", (Object)version.getVersion_id(), (Object)this.securityConfigVersionsIndex);
            this.threadPool.generic().submit(() -> {
                try {
                    this.applySecurityConfigVersionIndexRetentionPolicy();
                }
                catch (Exception e) {
                    log.warn("Retention policy async failed", (Throwable)e);
                }
            });
        }
        catch (VersionConflictEngineException conflictEx) {
            log.warn("Concurrent update detected on {}", (Object)this.securityConfigVersionsIndex);
        }
        catch (Exception e) {
            log.error("Failed to save version to {}", (Object)this.securityConfigVersionsIndex, (Object)e);
            throw ExceptionsHelper.convertToOpenSearchException((Exception)e);
        }
    }

    private boolean shouldSkipVersionUpdate(SecurityConfigVersionDocument document, SecurityConfigVersionDocument.Version<?> newVersion) {
        Map<String, SecurityConfigVersionDocument.HistoricSecurityConfig<?>> newConfigMap;
        SecurityConfigVersionDocument.Version<?> latestVersion;
        Map<String, SecurityConfigVersionDocument.HistoricSecurityConfig<?>> latestConfigMap;
        SecurityConfigVersionsLoader.sortVersionsById(document.getVersions());
        if (!document.getVersions().isEmpty() && !SecurityConfigDiffCalculator.hasSecurityConfigChanged(latestConfigMap = (latestVersion = document.getVersions().get(document.getVersions().size() - 1)).getSecurity_configs(), newConfigMap = newVersion.getSecurity_configs())) {
            log.info("No changes detected in security configuration. Skipping version update.");
            return true;
        }
        return false;
    }

    private void writeSecurityConfigVersion(SecurityConfigVersionDocument document, long currentSeqNo, long currentPrimaryTerm) throws IOException {
        Map<String, Object> updatedDocMap = document.toMap();
        String json = DefaultObjectMapper.objectMapper.writeValueAsString(updatedDocMap);
        IndexRequest indexRequest = (IndexRequest)new IndexRequest(this.securityConfigVersionsIndex).id("opensearch_security_config_versions").source(json, (MediaType)XContentType.JSON).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        if (currentSeqNo >= 0L && currentPrimaryTerm > 0L) {
            indexRequest.setIfSeqNo(currentSeqNo);
            indexRequest.setIfPrimaryTerm(currentPrimaryTerm);
        }
        int attempt = 0;
        while (true) {
            try {
                this.client.index(indexRequest).actionGet();
                return;
            }
            catch (Exception e) {
                if (attempt >= 3) {
                    throw new IOException(e);
                }
                ++attempt;
                log.debug("writeSecurityConfigVersion failed, retrying again");
                long delay = 200L;
                try {
                    Thread.sleep(delay);
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    throw new IOException(ie);
                }
            }
        }
    }

    public void applySecurityConfigVersionIndexRetentionPolicy() {
        SecurityConfigVersionDocument document = this.configVersionsLoader.loadFullDocument();
        List<SecurityConfigVersionDocument.Version<?>> versions = document.getVersions();
        SecurityConfigVersionsLoader.sortVersionsById(versions);
        if (versions.size() > this.maxVersionsToKeep) {
            int numVersionsToDelete = versions.size() - this.maxVersionsToKeep;
            log.info("Applying retention policy: deleting {} old security config versions", (Object)numVersionsToDelete);
            for (int i = 0; i < numVersionsToDelete; ++i) {
                versions.remove(0);
            }
            try {
                this.writeSecurityConfigVersion(document, document.getSeqNo(), document.getPrimaryTerm());
            }
            catch (Exception e) {
                log.warn("Failed to write document after pruning old versions", (Throwable)e);
            }
        }
    }
}

