"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.exportsForTesting = void 0;
exports.findNvim = findNvim;
const node_child_process_1 = require("node:child_process");
const node_path_1 = require("node:path");
const node_fs_1 = require("node:fs");
const versionRegex = /^(\d+)\.(\d+)\.(\d+)(?:-(.+))?$/;
const nvimVersionRegex = /^NVIM\s+v(.+)$/m;
const buildTypeRegex = /^Build\s+type:\s+(.+)$/m;
const luaJitVersionRegex = /^LuaJIT\s+(.+)$/m;
const windows = process.platform === 'win32';
function parseVersion(version) {
    if (typeof version !== 'string') {
        throw new TypeError('Invalid version format: not a string');
    }
    const match = version.match(versionRegex);
    if (!match) {
        return undefined;
    }
    const [, major, minor, patch, prerelease] = match;
    const majorNumber = Number(major);
    const minorNumber = Number(minor);
    const patchNumber = Number(patch);
    const versionParts = [
        majorNumber,
        minorNumber,
        patchNumber,
    ];
    if (prerelease !== undefined) {
        versionParts.push(prerelease);
    }
    else {
        versionParts.push('zzz');
    }
    return versionParts;
}
/**
 * Compares two versions.
 * @param a - First version to compare.
 * @param b - Second version to compare.
 * @returns -1 if a < b, 0 if a == b, 1 if a > b.
 * @throws {TypeError} If the versions are not valid.
 *
 * Format could be:
 * - 0.9.1
 * - 0.10.0-dev-658+g06694203e-Homebrew
 */
function compareVersions(a, b) {
    var _a, _b, _c, _d, _e, _f;
    const versionA = parseVersion(a);
    const versionB = parseVersion(b);
    const length = Math.min((_a = versionA === null || versionA === void 0 ? void 0 : versionA.length) !== null && _a !== void 0 ? _a : 0, (_b = versionB === null || versionB === void 0 ? void 0 : versionB.length) !== null && _b !== void 0 ? _b : 0);
    for (let i = 0; i < length; i = i + 1) {
        const partA = (_c = versionA === null || versionA === void 0 ? void 0 : versionA[i]) !== null && _c !== void 0 ? _c : 0;
        const partB = (_d = versionB === null || versionB === void 0 ? void 0 : versionB[i]) !== null && _d !== void 0 ? _d : 0;
        if (partA < partB) {
            return -1;
        }
        if (partA > partB) {
            return 1;
        }
    }
    if (((_e = versionB === null || versionB === void 0 ? void 0 : versionB.length) !== null && _e !== void 0 ? _e : 0) > ((_f = versionA === null || versionA === void 0 ? void 0 : versionA.length) !== null && _f !== void 0 ? _f : 0)) {
        return -1;
    }
    return 0;
}
/**
 * Tries to find a usable `nvim` binary on the current system.
 *
 * @param opt.minVersion See {@link FindNvimOptions.minVersion}
 * @param opt.orderBy See {@link FindNvimOptions.orderBy}
 */
function findNvim(opt = {}) {
    var _a, _b, _c;
    const paths = (_b = (_a = process.env.PATH) === null || _a === void 0 ? void 0 : _a.split(node_path_1.delimiter)) !== null && _b !== void 0 ? _b : [];
    const pathLength = paths.length;
    const matches = new Array();
    const invalid = new Array();
    for (let i = 0; i !== pathLength; i = i + 1) {
        const nvimPath = (0, node_path_1.join)(paths[i], windows ? 'nvim.exe' : 'nvim');
        if ((0, node_fs_1.existsSync)(nvimPath)) {
            try {
                (0, node_fs_1.accessSync)(nvimPath, node_fs_1.constants.X_OK);
                const nvimVersionFull = (0, node_child_process_1.execFileSync)(nvimPath, [
                    '--version',
                ]).toString();
                const nvimVersionMatch = nvimVersionRegex.exec(nvimVersionFull);
                const buildTypeMatch = buildTypeRegex.exec(nvimVersionFull);
                const luaJitVersionMatch = luaJitVersionRegex.exec(nvimVersionFull);
                if (nvimVersionMatch && buildTypeMatch && luaJitVersionMatch) {
                    if ('minVersion' in opt &&
                        compareVersions((_c = opt.minVersion) !== null && _c !== void 0 ? _c : '0.0.0', nvimVersionMatch[1]) ===
                            1) {
                        invalid.push({
                            nvimVersion: nvimVersionMatch[1],
                            path: nvimPath,
                            buildType: buildTypeMatch[1],
                            luaJitVersion: luaJitVersionMatch[1],
                        });
                    }
                    else {
                        matches.push({
                            nvimVersion: nvimVersionMatch[1],
                            path: nvimPath,
                            buildType: buildTypeMatch[1],
                            luaJitVersion: luaJitVersionMatch[1],
                        });
                    }
                }
            }
            catch (e) {
                invalid.push({
                    path: nvimPath,
                    error: e,
                });
            }
        }
    }
    if (opt.orderBy === undefined || opt.orderBy === 'desc') {
        matches.sort((a, b) => { var _a, _b; return compareVersions((_a = b.nvimVersion) !== null && _a !== void 0 ? _a : '0.0.0', (_b = a.nvimVersion) !== null && _b !== void 0 ? _b : '0.0.0'); });
    }
    return {
        matches,
        invalid,
    };
}
// jest sets NODE_ENV=test.
if (process.env.NODE_ENV === 'test') {
    // These functions are intentionally not exported. After `nvim` is found, clients can use Nvim's
    // own `vim.version` module, so node-client shouldn't expose a half-baked "semver" implementation.
    exports.exportsForTesting = {
        parseVersion,
        compareVersions,
    };
}
