%feature("docstring") OT::KarhunenLoeveP1Algorithm
"Computation of Karhunen-Loeve decomposition using P1 approximation.

Parameters
----------
mesh : :class:`~openturns.Mesh`
    The mesh :math:`\\cD_N` that discretizes the domain :math:`\\cD`.
covariance : :class:`~openturns.CovarianceModel`
    The covariance function to decompose.
s : float, :math:`\\geq0`
    The threshold used to select the most significant eigenmodes, defined in  :class:`~openturns.KarhunenLoeveAlgorithm`.

Notes
-----
The Karhunen-Loeve :math:`P_1` algorithm solves the Fredholm problem  associated to the covariance function :math:`C`: see :class:`~openturns.KarhunenLoeveAlgorithm` to get the notations.

The Karhunen-Loeve :math:`P_1` approximation uses the :math:`P_1` functional basis :math:`(\\theta_p)_{1 \\leq p \\leq N}` where  :math:`\\theta_p: \\cD_N \\mapsto \\Rset` are the basis functions of the :math:`P_1` finite element space associated to :math:`\\cD_N`, which vertices are :math:`(\\vect{s}_i)_{1 \\leq i \\leq N}`.

The covariance function :math:`\\mat{C}` is approximated by its :math:`P_1` approximation :math:`\\hat{\\mat{C}}` on :math:`\\cD_N`:

.. math::

   \\hat{\\mat{C}}(\\vect{s},\\vect{t})=\\sum_{\\vect{s}_i,\\vect{s}_j\\in\\cV_N}\\mat{C}(\\vect{s}_i,\\vect{s}_j)\\theta_i(\\vect{s})\\theta_j(\\vect{t}), \\quad  \\forall \\vect{s},\\vect{t}\\in\\cD_N

The Galerkin approach and the collocation one are equivalent in the :math:`P_1` approach and both lead to the following formulation:

.. math::

    \\mat{C}\\, \\mat{G}\\, \\mat{\\Phi}  =   \\mat{\\Phi}\\, \\mat{\\Lambda}


where :math:`\\mat{G} = (G_{ij})_{1\\leq i,j \\leq N}` with :math:`G_{i\\ell}= \\int_{\\cD} \\theta_i(\\vect{s})\\theta_\\ell(\\vect{s})\\,  d\\vect{s}`, :math:`\\mat{\\Lambda}=diag(\\vect{\\lambda})`.

Though the eigenvalues computation is performed by LAPACK by default, several other solvers can be used. The solver to use must be specified by setting ``KarhunenLoeveP1Algorithm-EigenvaluesSolver`` key in :class:`~openturns.ResourceMap`. Some solvers set limits to the number of eigenvalues they can compute with regards to the size of the problem. The following table summarizes the supported solvers and their maximum number of eigenvalues for a problem of size :math:`n`: 

+-------------+---------------------+----------------------+
| Solver      | ResourceMap key     | Maximum EV number    |
+=============+=====================+======================+
| Lapack      | LAPACK              | :math:`n`            |
+-------------+---------------------+----------------------+
| Spectra     | SPECTRA             | :math:`n-2`          |
+-------------+---------------------+----------------------+

For large dimension problems, it can be useful to generate the covariance matrix of the problem as a :class:`~openturns.HMatrix` instead of a standard dense matrix. In this case, one can set ``KarhunenLoeveP1Algorithm-CovarianceMatrixStorage`` key in :class:`~openturns.ResourceMap` to ``HMAT`` (instead of ``DENSE``).

Examples
--------
Create a Karhunen-Loeve P1 algorithm:

>>> import openturns as ot
>>> mesh = ot.IntervalMesher([10]*2).build(ot.Interval([-1.0]*2, [1.0]*2))
>>> s = 0.01
>>> model = ot.AbsoluteExponential([1.0]*2)
>>> algorithm = ot.KarhunenLoeveP1Algorithm(mesh, model, s)

Run it!

>>> algorithm.run()
>>> result = algorithm.getResult()"

// ---------------------------------------------------------------------

%feature("docstring") OT::KarhunenLoeveP1Algorithm::run
"Computation of the eigenvalues and eigenfunctions values at nodes.

Notes
-----
Runs the algorithm and creates the result structure :class:`~openturns.KarhunenLoeveResult`."

// ---------------------------------------------------------------------

%feature("docstring") OT::KarhunenLoeveP1Algorithm::getMesh
"Accessor to the mesh.

Returns
-------
mesh : :class:`~openturns.Mesh`
    The mesh :math:`\\cD_N` that discretizes the domain :math:`\\cD`."

