//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//============================================================================
// Copyright (c) 2018, The Regents of the University of California, through
// Lawrence Berkeley National Laboratory (subject to receipt of any required approvals
// from the U.S. Dept. of Energy).  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification,
// are permitted provided that the following conditions are met:
//
// (1) Redistributions of source code must retain the above copyright notice, this
//     list of conditions and the following disclaimer.
//
// (2) Redistributions in binary form must reproduce the above copyright notice,
//     this list of conditions and the following disclaimer in the documentation
//     and/or other materials provided with the distribution.
//
// (3) Neither the name of the University of California, Lawrence Berkeley National
//     Laboratory, U.S. Dept. of Energy nor the names of its contributors may be
//     used to endorse or promote products derived from this software without
//     specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
// BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
//
//=============================================================================
//
//  This code is an extension of the algorithm presented in the paper:
//  Parallel Peak Pruning for Scalable SMP Contour Tree Computation.
//  Hamish Carr, Gunther Weber, Christopher Sewell, and James Ahrens.
//  Proceedings of the IEEE Symposium on Large Data Analysis and Visualization
//  (LDAV), October 2016, Baltimore, Maryland.
//
//  The PPP2 algorithm and software were jointly developed by
//  Hamish Carr (University of Leeds), Gunther H. Weber (LBNL), and
//  Oliver Ruebel (LBNL)
//==============================================================================

#ifndef vtk_m_worklet_contourtree_distributed_bract_maker_boundary_tree_node_comparator_h
#define vtk_m_worklet_contourtree_distributed_bract_maker_boundary_tree_node_comparator_h

#include <vtkm/cont/ArrayHandle.h>
#include <vtkm/cont/ArrayHandleMultiplexer.h>
#include <vtkm/cont/ExecutionObjectBase.h>
#include <vtkm/filter/scalar_topology/worklet/contourtree_augmented/Types.h>


namespace vtkm
{
namespace worklet
{
namespace contourtree_distributed
{
namespace bract_maker
{


/// device implementation of the comparator used for sorting hyperarcs
class BoundaryTreeNodeComparatorImpl
{
public:
  using IdArrayPortalType = vtkm::cont::ArrayHandle<vtkm::Id>::ReadPortalType;
  // The ContourTreeMesh uses a smart ArrayHandleIndex instead of a regular IdArrayType array.
  // That is why we use a ArrayHandleMultiplexer here.
  using SortIndexPortalType =
    vtkm::cont::ArrayHandleMultiplexer<vtkm::cont::ArrayHandle<vtkm::Id>,
                                       vtkm::cont::ArrayHandleIndex>::ReadPortalType;

  // constructor - takes vectors as parameters
  VTKM_CONT
  BoundaryTreeNodeComparatorImpl(const IdArrayPortalType& regularIdPortal,
                                 const SortIndexPortalType& meshSortIndexPortal)
    : RegularIdPortal(regularIdPortal)
    , MeshSortIndexPortal(meshSortIndexPortal)
  { // constructor
  } // constructor

  // () operator - gets called to do comparison
  VTKM_EXEC
  bool operator()(const vtkm::Id& i, const vtkm::Id& j) const
  { // operator()
    // retrieve the sort order
    vtkm::Id sortIndexI = MeshSortIndexPortal.Get(RegularIdPortal.Get(i));
    vtkm::Id sortIndexJ = MeshSortIndexPortal.Get(RegularIdPortal.Get(j));

    // now test on that
    if (sortIndexI < sortIndexJ)
    {
      return true;
    }
    else if (sortIndexJ < sortIndexI)
    {
      return false;
    }

    // fallback out of paranoia
    return false;

  } // operator()

private:
  IdArrayPortalType RegularIdPortal;
  SortIndexPortalType MeshSortIndexPortal;
}; // BoundaryTreeNodeComparatorImpl

/// comparator used to compare hyperarcs for sort
class BoundaryTreeNodeComparator : public vtkm::cont::ExecutionObjectBase
{
public:
  // constructor - takes vectors as parameters
  VTKM_CONT
  BoundaryTreeNodeComparator(
    const vtkm::worklet::contourtree_augmented::IdArrayType& regularId,
    const vtkm::cont::ArrayHandleMultiplexer<vtkm::cont::ArrayHandle<vtkm::Id>,
                                             vtkm::cont::ArrayHandleIndex>& meshSortIndex)
    : RegularId(regularId)
    , MeshSortIndex(meshSortIndex)
  { // constructor
  } // constructor

  VTKM_CONT BoundaryTreeNodeComparatorImpl PrepareForExecution(vtkm::cont::DeviceAdapterId device,
                                                               vtkm::cont::Token& token) const
  {
    return BoundaryTreeNodeComparatorImpl(this->RegularId.PrepareForInput(device, token),
                                          this->MeshSortIndex.PrepareForInput(device, token));
  }

private:
  vtkm::worklet::contourtree_augmented::IdArrayType RegularId;
  /// The ContourTreeMesh uses a smart ArrayHandleIndex instead of a regular IdArrayType array that is why we use a ArrayHandleMultiplexer here
  vtkm::cont::ArrayHandleMultiplexer<vtkm::cont::ArrayHandle<vtkm::Id>,
                                     vtkm::cont::ArrayHandleIndex>
    MeshSortIndex;
}; // BoundaryTreeNodeComparator

} // namespace bract_maker
} // namespace contourtree_distributed
} // namespace worklet
} // namespace vtkm

#endif
