# Copyright (c) 2012 Intel Corporation.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""
UUID related utilities and helper functions.

.. versionadded:: 1.1
"""

from typing import Any
import uuid


def generate_uuid(dashed: bool = True) -> str:
    """Creates a random uuid string.

    :param dashed: Generate uuid with dashes or not
    :type dashed: bool
    :returns: string
    """
    if dashed:
        return str(uuid.uuid4())
    return uuid.uuid4().hex


def _format_uuid_string(string: str) -> str:
    return (
        string.replace('urn:', '')
        .replace('uuid:', '')
        .strip('{}')
        .replace('-', '')
        .lower()
    )


def is_uuid_like(val: Any) -> bool:
    """Returns validation of a value as a UUID.

    :param val: Value to verify
    :type val: string
    :returns: bool

    .. versionchanged:: 1.1.1
       Support non-lowercase UUIDs.
    """
    try:
        return str(uuid.UUID(val)).replace('-', '') == _format_uuid_string(val)
    except (TypeError, ValueError, AttributeError):
        return False
