/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TGUI - Texus' Graphical User Interface
// Copyright (C) 2012-2023 Bruno Van de Velde (vdv_b@tgui.eu)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


#ifndef TGUI_TAB_HPP
#define TGUI_TAB_HPP


#include <TGUI/Widget.hpp>
#include <TGUI/Renderers/TabsRenderer.hpp>
#include <TGUI/Text.hpp>

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

TGUI_MODULE_EXPORT namespace tgui
{
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief Tabs widget
    ///
    /// If you are looking for something that allow holds multiple panels,
    /// displayed one panel at a time then check out the TabContainer class.
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    class TGUI_API Tabs : public Widget
    {
    public:

        using Ptr = std::shared_ptr<Tabs>; //!< Shared widget pointer
        using ConstPtr = std::shared_ptr<const Tabs>; //!< Shared constant widget pointer

        static constexpr const char StaticWidgetType[] = "Tabs"; //!< Type name of the widget


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /// @brief Constructor
        /// @param typeName     Type of the widget
        /// @param initRenderer Should the renderer be initialized? Should be true unless a derived class initializes it.
        /// @see create
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        Tabs(const char* typeName = StaticWidgetType, bool initRenderer = true);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Creates a new tabs widget
        ///
        /// @return The new tabs
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD static Tabs::Ptr create();


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Makes a copy of another tab
        ///
        /// @param tab  The other tab
        ///
        /// @return The new tab
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD static Tabs::Ptr copy(const Tabs::ConstPtr& tab);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the renderer, which gives access to functions that determine how the widget is displayed
        /// @return Temporary pointer to the renderer that may be shared with other widgets using the same renderer
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD TabsRenderer* getSharedRenderer() override;
        TGUI_NODISCARD const TabsRenderer* getSharedRenderer() const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the renderer, which gives access to functions that determine how the widget is displayed
        /// @return Temporary pointer to the renderer
        /// @warning After calling this function, the widget has its own copy of the renderer and it will no longer be shared.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD TabsRenderer* getRenderer() override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the area used by the tabs
        ///
        /// @param size  Total size of the tabs
        ///
        /// When this function is called, the tabs will no longer be auto-sizing, but each tab will get an equal width such that
        /// the total width corresponds to the size passed to this function.
        ///
        /// @see setAutoSize
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setSize(const Layout2d& size) override;
        using Widget::setSize;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Enables or disables the widget
        /// @param enabled  Is the widget enabled?
        ///
        /// The disabled widget will no longer receive events and thus no longer send callbacks.
        /// All widgets are enabled by default.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setEnabled(bool enabled) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes whether the tabs are auto-sized or not
        ///
        /// @param autoSize  Should the size of the tabs depend on their content?
        ///
        /// When the tabs are in auto-size mode, the width of the tabs will be depend on the text which they contain.
        /// Otherwise, all tabs have an equal width to fill the size of the widget.
        ///
        /// The tabs are auto-sized by default.
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setAutoSize(bool autoSize);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns whether the tabs are auto-sized or not
        ///
        /// @return Is the width of the tabs dependent on their content?
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD bool getAutoSize() const;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Adds a new tab
        ///
        /// @param text    The text of the tab that will be drawn on top of it
        /// @param select  Do you want the new tab to be selected immediately?
        ///
        /// @return  The index of the tab in the list
        ///
        /// @warning The index returned by this function may no longer be correct when a tab is removed
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        std::size_t add(const String& text, bool select = true);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Inserts a new tab somewhere between the other tabs
        ///
        /// @param index   The index where the tab gets inserted. 0 means before the first tab, 1 means behind the first tab
        /// @param text    The text of the tab that will be drawn on top of it
        /// @param select  Do you want the new tab to be selected immediately?
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void insert(std::size_t index, const String& text, bool select = true);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Gets the text of one of the tabs
        ///
        /// @param index  The index of the tab. The first tab has index 0
        ///
        /// @return The text on the tab or an empty string when index was too high
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD String getText(std::size_t index) const;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the text of one of the tabs
        ///
        /// @param index  The index of the tab to be changed. The first tab has index 0
        /// @param text   The new text of the tab that will be drawn on top of it
        ///
        /// @return True when text was successfully changed, false when index was too high
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool changeText(std::size_t index, const String& text);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Selects the tab with a given text
        /// @param text  The text of the tab to select
        /// @return Whether a tab was selected, false is returned if tab doesn't exist or is invisible or disabled
        /// @see select(int)
        ///
        /// If there are multiple tabs with the same text then the first one will be selected.
        /// When false is returned, the selected tab will still be deselected.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool select(const String& text);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Selects the tab with a given index
        /// @param index  The index of the tab to select
        /// @return Whether a tab was selected, false is returned if the index was too high or if tab is invisible or disabled
        /// @see select(String)
        ///
        /// When false is returned, the selected tab will still be deselected.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool select(std::size_t index);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Deselects the selected tab
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void deselect();


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Removes a tab with a given text
        /// @param text  The text on the tab to remove
        /// @return Whether a tab was removed, false is returned when the text didn't match any tab
        /// @see remove(std::size_t)
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool remove(const String& text);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Removes a tab with a given index
        /// @param index  The index of the tab to remove
        /// @return Whether a tab was removed, false is returned when the index was too high
        /// @see remove(String)
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool remove(std::size_t index);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Removes all tabs
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void removeAll();


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Gets the text that is drawn on the currently selected tab
        ///
        /// @return The text on the tab.
        ///         When no tab is selected then this function returns an empty string
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD String getSelected() const;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Gets the index of the currently selected tab
        ///
        /// @return The index of the tab.
        ///         When no tab is selected then this function returns -1
        ///
        /// @warning The index returned by this function may no longer be correct when a tab is removed
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD int getSelectedIndex() const;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes whether a tab is visible
        /// @param index   The index of the tab to show or hide
        /// @param visible Is the tab shown?
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setTabVisible(std::size_t index, bool visible);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns whether the tab is shown or hidden
        /// @param index  The index of the tab
        /// @return Whether the tab is visible
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD bool getTabVisible(std::size_t index) const;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes whether a tab is enabled
        /// @param index   The index of the tab to enable or disable
        /// @param enabled Is the tab enabled?
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setTabEnabled(std::size_t index, bool enabled);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns whether the tab is enabled or disabled
        /// @param index  The index of the tab
        /// @return Whether the tab is enabled
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD bool getTabEnabled(std::size_t index) const;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the height of the tabs
        ///
        /// @param height  Height of the tabs
        ///
        /// When the tabs are auto-sizing, this function allows changing the height of the tabs without setting a fixed width
        /// like calling setSize would do.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setTabHeight(float height);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the maximum tab width of the tabs
        ///
        /// @param maximumWidth  Maximum width of a single tab
        ///
        /// This property only has effect when the tabs are auto-sizing.
        ///
        /// If the text on the tab is longer than this width then it will be cropped to fit inside the tab.
        /// By default, the maximum width is 0 which means that there is no limitation.
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setMaximumTabWidth(float maximumWidth);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the maximum tab width of the tabs
        ///
        /// @return Maximum tab width
        ///
        /// This property only has effect when the tabs are auto-sizing.
        ///
        /// If the text on the tab is longer than this width then it will be cropped to fit inside the tab.
        /// By default, the maximum width is 0 which means that there is no limitation.
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD float getMaximumTabWidth() const;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the minimum tab width of the tabs
        ///
        /// @param minimumWidth  Minimum width of a single tab
        ///
        /// This property only has effect when the tabs are auto-sizing.
        ///
        /// Every tab is at least as wide as this minimum or twice the distance to side.
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setMinimumTabWidth(float minimumWidth);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the minimum tab width of the tabs
        ///
        /// @return Minimum tab width
        ///
        /// This property only has effect when the tabs are auto-sizing.
        ///
        /// Every tab is at least as wide as this minimum or twice the distance to side.
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD float getMinimumTabWidth() const;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the amount of tabs
        ///
        /// @return Number of tabs
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD std::size_t getTabsCount() const;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns whether the mouse position (which is relative to the parent widget) lies on top of the widget
        ///
        /// @return Is the mouse on top of the widget?
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD bool isMouseOnWidget(Vector2f pos) const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool leftMousePressed(Vector2f pos) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void mouseMoved(Vector2f pos) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void mouseNoLongerOnWidget() override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Draw the widget to a render target
        ///
        /// @param target Render target to draw to
        /// @param states Current render states
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void draw(BackendRenderTarget& target, RenderStates states) const override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Retrieves a signal based on its name
        ///
        /// @param signalName  Name of the signal
        ///
        /// @return Signal that corresponds to the name
        ///
        /// @throw Exception when the name does not match any signal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Signal& getSignal(String signalName) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Recalculates the size of each tab image.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void recalculateTabsWidth();


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Update the colors of the text that is drawn on the tabs
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void updateTextColors();


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Function called when one of the properties of the renderer is changed
        ///
        /// @param property  Name of the property that was changed
        ///
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void rendererChanged(const String& property) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Saves the widget as a tree node in order to save it to a file
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD std::unique_ptr<DataIO::Node> save(SavingRenderersMap& renderers) const override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Loads the widget from a tree of nodes
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void load(const std::unique_ptr<DataIO::Node>& node, const LoadingRenderersMap& renderers) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Called when the text size is changed (either by setTextSize or via the renderer)
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void updateTextSize() override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Makes a copy of the widget
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Widget::Ptr clone() const override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    public:

        SignalString onTabSelect = {"TabSelected"}; //!< A tab that was selected. Optional parameter: selected item


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        float              m_maximumTabWidth = 0;
        float              m_minimumTabWidth = 0;
        int                m_selectedTab = -1;
        int                m_hoveringTab = -1;
        bool               m_autoSize = true;

        struct Tab
        {
            bool  visible;
            bool  enabled;
            float width;
            Text  text;
        };
        std::vector<Tab> m_tabs;

        Sprite m_spriteTab;
        Sprite m_spriteTabHover;
        Sprite m_spriteSelectedTab;
        Sprite m_spriteSelectedTabHover;
        Sprite m_spriteDisabledTab;

        // Cached renderer properties
        Borders   m_bordersCached;
        Color     m_borderColorCached;
        Color     m_borderColorHoverCached;
        Color     m_selectedBorderColorCached;
        Color     m_selectedBorderColorHoverCached;
        Color     m_backgroundColorCached;
        Color     m_backgroundColorHoverCached;
        Color     m_backgroundColorDisabledCached;
        Color     m_selectedBackgroundColorCached;
        Color     m_selectedBackgroundColorHoverCached;
        Color     m_textColorCached;
        Color     m_textColorHoverCached;
        Color     m_textColorDisabledCached;
        Color     m_selectedTextColorCached;
        Color     m_selectedTextColorHoverCached;
        float     m_distanceToSideCached = 0;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    };

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
}


/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#endif // TGUI_TAB_HPP
