%feature("docstring") OT::UserDefinedStationaryCovarianceModel
"Stationary covariance model defined by the User.

Parameters
----------
mesh : :class:`~openturns.RegularGrid`
    Time grid of size :math:`N` associated with the process. Negative vertices are ignored.
sample : :class:`~openturns.SquareMatrixCollection`
    A collection of :math:`N` :class:`~openturns.SquareMatrix` objects with the same dimension.

Notes
-----
The covariance model is built as follows. Let :math:`d` be the dimension of the square matrices.

We consider a process :math:`X: \\Omega \\times \\Rset \\mapsto \\Rset^d`.

We note :math:`(t_0,\\dots, t_{N-1})` the vertices of :math:`\\cM \\subset \\Rset` and :math:`(\\mat{C}_{k})_{0 \\leq  k \\leq N-1}` the collection of square matrices provided by the user.

The class builds a stationary covariance function :math:`C^{stat}` as a piecewise constant function defined on :math:`\\Rset` by:

.. math::

    \\forall \\tau \\in \\Rset, \\, \\quad C^{stat}(\\tau) =  \\mat{C}_k


where :math:`k` is such that :math:`t_k` is the vertex of :math:`\\cM` closest to :math:`|\\tau|`.

Examples
--------
Create a time grid:

>>> import openturns as ot
>>> t0 = 0.0
>>> dt = 0.5
>>> N = int((20.0 - t0)/ dt)
>>> myShiftMesh =  ot.RegularGrid(t0, dt, N)

Create a stationary covariance function:

>>> def gamma(tau):
...     return 1.0 / (1.0 + tau * tau)

Create a collection of :math:`N` covariance values contained in 1x1 symmetric matrices:

>>> myCovarianceCollection = ot.SquareMatrixCollection()
>>> for k in range(N):
...     t = myShiftMesh.getValue(k)
...     matrix = ot.SquareMatrix(1)
...     matrix[0, 0] = gamma(t)
...     myCovarianceCollection.add(matrix)

Create the user-defined stationary covariance model:

>>> myCovarianceModel = ot.UserDefinedStationaryCovarianceModel(myShiftMesh, myCovarianceCollection)

Compute the covariance function at a specific vertex tau:

>>> tau = 1.5
>>> myCovModelMatrix = myCovarianceModel(tau)
"

// ---------------------------------------------------------------------
%feature("docstring") OT::UserDefinedStationaryCovarianceModel::getTimeGrid
"Accessor to the time grid.

Returns
-------
mesh : :class:`~openturns.RegularGrid`
    The time grid associated to the collection of covariance matrices.
"
