(*
 * Copyright (C) 2014-2022, AdaCore
 * SPDX-License-Identifier: Apache-2.0
 *)

open Ctypes
open Foreign

(* Under Linux, disable GNAT's handling of SIGSEGV, which is incompatible with
   what the OCaml runtime is already doing. *)
let () =
  if Sys.unix then
    ignore
      (Dl.dlopen
        ~filename:"liblangkit_sigsegv_handler.so"
        ~flags:[Dl.RTLD_NOW]
        : Dl.library)

let so_ext = if Sys.unix then "so" else "dll"
let c_lib_name = Format.sprintf "libadalang.%s" so_ext
let c_lib = Dl.dlopen ~filename:c_lib_name ~flags:[Dl.RTLD_NOW]

let add_gc_link ~from ~to_ =
  let r = ref (Some (Obj.repr to_)) in
  let finaliser _ = r := None in
  Gc.finalise finaliser from

exception FileReadError of string

exception BadTypeError of string

exception OutOfBoundsError of string

exception InvalidInput of string

exception InvalidSymbolError of string

exception InvalidUnitNameError of string

exception NativeException of string

exception PreconditionFailure of string

exception PropertyError of string

exception TemplateArgsError of string

exception TemplateFormatError of string

exception TemplateInstantiationError of string

exception StaleReferenceError of string

exception SyntaxError of string

exception UnknownCharset of string

exception MalformedTreeError of string

exception InvalidProject of string

exception UnsupportedViewError of string





let char_ptr_of_string str =
  coerce string (ptr char) str

let string_of_char_ptr str =
  coerce (ptr char) string str

module Exception = struct

  type t = {
    kind : int;
    information : string;
  }

  let c_struct : t structure typ = structure "exception"
  let kind = field c_struct "kind" int
  let information = field c_struct "information" (ptr char)
  let () = seal c_struct

  let wrap c_value_ptr =
    if is_null c_value_ptr then
      None
    else
      let c_value = !@ c_value_ptr in
      Some {
        kind = getf c_value kind;
        information = string_of_char_ptr (getf c_value information);
      }

  let unwrap value =
    match value with
    | None ->
        from_voidp c_struct null
    | Some value ->
        let c_value = make c_struct in
        let c_information = char_ptr_of_string value.information in
        setf c_value kind value.kind;
        add_gc_link ~from:c_value ~to_:c_information;
        setf c_value information c_information;
        allocate c_struct c_value

  let c_type = view (ptr c_struct) ~read:wrap ~write:unwrap

end

let get_last_exception = foreign ~from:c_lib
  "ada_get_last_exception"
  (void @-> returning Exception.c_type)

(* When declaring an imported function with foreign, use raisable instead of
 returning, to check get_last_exception before returning *)
let raisable typ =
  let read value =
    match get_last_exception () with
    | None -> value
    | Some exc ->
        (match exc.kind with
         | 0 ->
             raise (FileReadError exc.information)
         | 1 ->
             raise (BadTypeError exc.information)
         | 2 ->
             raise (OutOfBoundsError exc.information)
         | 3 ->
             raise (InvalidInput exc.information)
         | 4 ->
             raise (InvalidSymbolError exc.information)
         | 5 ->
             raise (InvalidUnitNameError exc.information)
         | 6 ->
             raise (NativeException exc.information)
         | 7 ->
             raise (PreconditionFailure exc.information)
         | 8 ->
             raise (PropertyError exc.information)
         | 9 ->
             raise (TemplateArgsError exc.information)
         | 10 ->
             raise (TemplateFormatError exc.information)
         | 11 ->
             raise (TemplateInstantiationError exc.information)
         | 12 ->
             raise (StaleReferenceError exc.information)
         | 13 ->
             raise (SyntaxError exc.information)
         | 14 ->
             raise (UnknownCharset exc.information)
         | 15 ->
             raise (MalformedTreeError exc.information)
         | 16 ->
             raise (InvalidProject exc.information)
         | 17 ->
             raise (UnsupportedViewError exc.information)
         | _ -> assert false)
  in
  let write value = value in
  let new_typ = view typ ~read ~write in
  returning new_typ

(* Module used to encode/decode UTF32 strings *)

(* Camomile needs to know the location of its standard library to work,
   so we use the following heuristic:
   - if the directory chosen at build time exists, we assume the installation
     is ok
   - otherwise we look for a directory 'share/camomile' next to the binary
   - otherwise we fail
*)

module type CamomileConfig = module type of CamomileLibrary.DefaultConfig

module CamomileDefaultConfig : CamomileConfig = CamomileLibrary.DefaultConfig

let ( ^/ ) = Filename.concat

let build_camomile_config root_path = (module struct
  let share_dir = root_path ^/ "share" ^/ "camomile"

  let datadir = share_dir ^/ "database"

  let localedir = share_dir ^/ "locales"

  let charmapdir = share_dir ^/ "charmaps"

  let unimapdir = share_dir ^/ "mappings"

  end : CamomileConfig)

module CamomileShareConfig =
  (val build_camomile_config
    (Filename.dirname Sys.executable_name ^/ Filename.parent_dir_name)
    : CamomileConfig)

(* In case we are building through an opam-installed env, find
   Camomile's stdlib through the appropriate opam env variable *)
module CamomileOpamConfig =
  (val
    let opam_dir = try Sys.getenv "OPAM_SWITCH_PREFIX" with _ -> "DUMMY" in
    build_camomile_config opam_dir : CamomileConfig)

let camomile_config =
  if Sys.file_exists CamomileDefaultConfig.datadir then
    (module CamomileDefaultConfig : CamomileConfig )
  else if Sys.file_exists CamomileShareConfig.datadir then
    (module CamomileShareConfig : CamomileConfig )
  else if Sys.file_exists CamomileOpamConfig.datadir then
    (module CamomileOpamConfig : CamomileConfig)
  else failwith "no camomile library found"

module CamomileConfig = (val camomile_config)

module Camomile = CamomileLibrary.Make (CamomileConfig)

module Text = struct
  type t = string

  let c_struct : t structure typ = structure "text"

  let chars = field c_struct "chars" (ptr uint32_t)

  let length = field c_struct "length" size_t

  let is_allocated = field c_struct "is_allocated" bool

  let () = seal c_struct

  let destroy_text = foreign ~from:c_lib "ada_destroy_text"
    (ptr c_struct @-> raisable void)

  module UCS4Encoding = Camomile.CharEncoding.Make (Camomile.UCS4)

  let wrap (c_value : t structure) : t =
    let open Unsigned.Size_t in
    let open Camomile in
    let length = to_int (getf c_value length) in
    let chars = getf c_value chars in
    let f i =
      UChar.chr_of_uint (Unsigned.UInt32.to_int !@ (chars +@ i))
    in
    let result = UCS4.init length f in
    (* Now that the value is fully transformed to an ocaml value, we can
      free it by calling destroy_text *)
    destroy_text (addr c_value) ;
    UCS4Encoding.encode CharEncoding.utf8 result

  let unwrap (value : t) : t structure =
    let open Unsigned in
    let open Camomile in
    let text = UCS4Encoding.decode CharEncoding.utf8 value in
    let struct_length = Size_t.of_int (UCS4.length text) in
    let struct_chars = allocate_n uint32_t ~count:(UCS4.length text) in
    let i = ref 0 in
    let f c =
      struct_chars +@ !i <-@ (UInt32.of_int (UChar.code c));
      i := !i + 1
    in
    UCS4.iter f text ;
    let c_value = make c_struct in
    setf c_value length struct_length ;
    add_gc_link ~from:c_value ~to_:struct_chars;
    setf c_value chars struct_chars ;
    setf c_value is_allocated false ;
    (* We don't need to care about calling destroy_text here since we
     manually allocated the pointer, ctypes will take care of freeing the
     memory *)
    c_value

  let c_type = view c_struct ~read:wrap ~write:unwrap
end

module Character = struct
  (* Characters are encoded as strings because ocaml char are not unicode
   characters *)
  type t = string

  module UCharEncoding = Camomile.CharEncoding.Make (Camomile.UText)

  let of_int i =
    let open Camomile in
    let uchar = UChar.chr i in
    UCharEncoding.encode CharEncoding.utf8 (UText.init 1 (fun _ -> uchar))

  let of_int32 i =
    of_int (Unsigned.UInt32.to_int i)

  let wrap (c_value : Unsigned.UInt32.t) : t =
    of_int32 c_value

  let unwrap (value : string) : Unsigned.UInt32.t =
    let open Camomile in
    let text = UCharEncoding.decode CharEncoding.utf8 value in
    let uchar = UText.get text 0 in
    Unsigned.UInt32.of_int (UChar.code uchar)

  let c_type = view uint32_t ~read:wrap ~write:unwrap
end

module StringType = struct
  type t = string

  let c_struct : t structure typ = structure "string"
  let length_field = field c_struct "length" int
  let _ = field c_struct "ref_count" int
  (* Langkit strings are encoded in UTF-32 (native endianity). *)
  let content_field = field c_struct "content" uint32_t
  let () = seal c_struct

  let buffer_ptr_type = ptr uint32_t
  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "ada_create_string"
    (buffer_ptr_type @-> int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "ada_string_dec_ref"
    (c_type @-> raisable void)

  module UCharEncoding = Camomile.CharEncoding.Make (Camomile.UText)

  let wrap c_value_ptr =
    let open Text in
    let open Camomile in
    let c_value = !@ c_value_ptr in
    let length = getf c_value length_field in
    let content = c_value @. content_field in
    (* We use Camomile to encode utf32 strings to an ocaml string *)
    let f i = UChar.chr_of_uint (Unsigned.UInt32.to_int !@(content +@ i)) in
    let result =
      UCS4Encoding.encode CharEncoding.utf8 (UCS4.init length f)
    in
    dec_ref c_value_ptr;
    result

  let unwrap value =
    let open Text in
    let open Camomile in

    (* Create a buffer to contain the UTF-32 encoded string. *)
    let text = UCS4Encoding.decode CharEncoding.utf8 value in
    let length = UCS4.length text in
    let buffer = allocate_n uint32_t ~count:length in
    let i = ref 0 in
    let f c =
      buffer +@ !i <-@ (Unsigned.UInt32.of_int (UChar.code c));
      i := !i + 1
    in
    UCS4.iter f text ;

    (* ctypes is supposed to take care of freeing "buffer" before returning. *)
    create buffer length
end

module BigInteger = struct

  type t = Z.t

  let c_type = ptr void

  let create = foreign ~from:c_lib "ada_create_big_integer"
    (ptr Text.c_type @-> raisable c_type)

  let text = foreign ~from:c_lib "ada_big_integer_text"
    (c_type @-> ptr Text.c_type @-> raisable void)

  let decref = foreign ~from:c_lib "ada_big_integer_decref"
    (c_type @-> raisable void)

  let wrap (c_value : unit ptr) : t =
    let c_text_ptr = allocate_n Text.c_type ~count:1 in
    text c_value c_text_ptr;
    decref c_value;
    Z.of_string (!@ c_text_ptr)

  let unwrap (value : t) : unit ptr =
    create (allocate Text.c_type (Z.to_string value))
end

module AnalysisUnitKind = struct
  type t =
  | UnitSpecification
  | UnitBody

  let name () = "AnalysisUnitKind"

  let wrap (c_value : int) : t =
    match c_value with
    | 0 -> UnitSpecification
    | 1 -> UnitBody
    | _ -> assert false

  let unwrap (value : t) : int =
    match value with
    | UnitSpecification -> 0
    | UnitBody -> 1

   let c_type = view int ~read:wrap ~write:unwrap
end

module LookupKind = struct
  type t =
  | Recursive
  | Flat
  | Minimal

  let name () = "LookupKind"

  let wrap (c_value : int) : t =
    match c_value with
    | 0 -> Recursive
    | 1 -> Flat
    | 2 -> Minimal
    | _ -> assert false

  let unwrap (value : t) : int =
    match value with
    | Recursive -> 0
    | Flat -> 1
    | Minimal -> 2

   let c_type = view int ~read:wrap ~write:unwrap
end

module DesignatedEnvKind = struct
  type t =
  | None
  | CurrentEnv
  | NamedEnv
  | DirectEnv

  let name () = "DesignatedEnvKind"

  let wrap (c_value : int) : t =
    match c_value with
    | 0 -> None
    | 1 -> CurrentEnv
    | 2 -> NamedEnv
    | 3 -> DirectEnv
    | _ -> assert false

  let unwrap (value : t) : int =
    match value with
    | None -> 0
    | CurrentEnv -> 1
    | NamedEnv -> 2
    | DirectEnv -> 3

   let c_type = view int ~read:wrap ~write:unwrap
end

module RefResultKind = struct
  type t =
  | NoRef
  | Precise
  | Imprecise
  | Error

  let name () = "RefResultKind"

  let wrap (c_value : int) : t =
    match c_value with
    | 0 -> NoRef
    | 1 -> Precise
    | 2 -> Imprecise
    | 3 -> Error
    | _ -> assert false

  let unwrap (value : t) : int =
    match value with
    | NoRef -> 0
    | Precise -> 1
    | Imprecise -> 2
    | Error -> 3

   let c_type = view int ~read:wrap ~write:unwrap
end

module CallExprKind = struct
  type t =
  | Call
  | ArraySlice
  | ArrayIndex
  | TypeConversion

  let name () = "CallExprKind"

  let wrap (c_value : int) : t =
    match c_value with
    | 0 -> Call
    | 1 -> ArraySlice
    | 2 -> ArrayIndex
    | 3 -> TypeConversion
    | _ -> assert false

  let unwrap (value : t) : int =
    match value with
    | Call -> 0
    | ArraySlice -> 1
    | ArrayIndex -> 2
    | TypeConversion -> 3

   let c_type = view int ~read:wrap ~write:unwrap
end

module GrammarRule = struct
  type t =
  | ParentListRule
  | ProtectedTypeDeclRule
  | ProtectedOpRule
  | ProtectedElRule
  | ProtectedDefRule
  | ProtectedDeclRule
  | TaskItemRule
  | TaskDefRule
  | TaskTypeDeclRule
  | SubtypeDeclRule
  | InterfaceTypeDefRule
  | UnconstrainedIndexRule
  | ArrayTypeDefRule
  | DiscreteSubtypeDefinitionRule
  | ConstraintListRule
  | SignedIntTypeDefRule
  | ModIntTypeDefRule
  | DerivedTypeDefRule
  | CompositeConstraintAssocRule
  | CompositeConstraintRule
  | DigitsConstraintRule
  | DeltaConstraintRule
  | RangeConstraintRule
  | ConstraintRule
  | DiscriminantSpecRule
  | DiscrSpecListRule
  | DiscriminantPartRule
  | EnumLiteralDeclRule
  | FormalDiscreteTypeDefRule
  | RecordDefRule
  | RangeSpecRule
  | RealTypeDefRule
  | SexprOrBoxRule
  | OrdinaryFixedPointDefRule
  | DecimalFixedPointDefRule
  | FloatingPointDefRule
  | RecordTypeDefRule
  | AccessDefRule
  | EnumTypeDefRule
  | TypeDefRule
  | VariantRule
  | AnonymousTypeDeclRule
  | IncompleteTypeDeclRule
  | TypeDeclRule
  | VariantPartRule
  | ComponentDefRule
  | ComponentItemRule
  | ComponentDeclRule
  | ComponentListRule
  | GenericDeclRule
  | GenericFormalPartRule
  | GenericFormalDeclRule
  | FormalTypeDeclRule
  | FormalSubpDeclRule
  | RenamingClauseRule
  | GenericRenamingDeclRule
  | GenericInstantiationRule
  | ExceptionDeclRule
  | BasicDeclsRule
  | PackageRenamingDeclRule
  | PackageDeclRule
  | BasicDeclRule
  | ObjectDeclRule
  | SubObjectDeclRule
  | NoTypeObjectRenamingDeclRule
  | ExtRetStmtObjectDeclRule
  | DefiningIdListRule
  | NumberDeclRule
  | ContractCaseAssocRule
  | ContractCasesExprRule
  | AbstractStateDeclRule
  | MultiAbstractStateDeclRule
  | AspectAssocRule
  | AspectSpecRule
  | SingleTaskDeclRule
  | OverridingIndicatorRule
  | EntryDeclRule
  | ComponentClauseRule
  | AspectClauseRule
  | ParamSpecRule
  | ParamSpecsRule
  | SubpSpecRule
  | ExprFnRule
  | NullSubpDeclRule
  | AbstractSubpDeclRule
  | SubpRenamingDeclRule
  | SimpleSubpDeclRule
  | SubpDeclRule
  | WithClauseRule
  | ContextItemRule
  | UseClauseRule
  | UsePackageClauseRule
  | UseTypeClauseRule
  | SubtypeIndicationRule
  | DiscreteSubtypeIndicationRule
  | ConstrainedSubtypeIndicationRule
  | TypeExprRule
  | AnonymousTypeRule
  | ModeRule
  | PragmaArgumentRule
  | PragmaRule
  | SubunitRule
  | LibraryUnitBodyRule
  | LibraryUnitRenamingDeclRule
  | LibraryItemRule
  | CompilationUnitRule
  | CompilationRule
  | DeclPartRule
  | EntryBodyRule
  | ProtectedBodyRule
  | ProtectedBodyStubRule
  | TaskBodyRule
  | TaskBodyStubRule
  | PackageBodyStubRule
  | PackageBodyRule
  | TerminateAlternativeRule
  | SelectStmtRule
  | AcceptStmtRule
  | CaseAltRule
  | CaseStmtRule
  | ExtReturnStmtRule
  | IblockStmtRule
  | BlockStmtRule
  | WhileLoopSpecRule
  | IloopStmtRule
  | LoopStmtRule
  | CompoundStmtRule
  | ElsifPartRule
  | IfStmtRule
  | RaiseStmtRule
  | DelayStmtRule
  | AbortStmtRule
  | BodyRule
  | BodyStubRule
  | SubpBodyStubRule
  | RecovDeclPartRule
  | SubpBodyRule
  | HandledStmtsRule
  | ExceptionHandlerRule
  | StmtsRule
  | LabelRule
  | StmtRule
  | CallStmtRule
  | SimpleStmtRule
  | NullStmtRule
  | AssignmentStmtRule
  | GotoStmtRule
  | ExitStmtRule
  | ReturnStmtRule
  | RequeueStmtRule
  | IdentifierRule
  | CharLiteralRule
  | StringLiteralRule
  | DefiningIdRule
  | DecLiteralRule
  | IntLiteralRule
  | NumLiteralRule
  | NullLiteralRule
  | AllocatorRule
  | ForLoopParamSpecRule
  | QuantifiedExprRule
  | CaseExprRule
  | CaseExprAltRule
  | RaiseExprRule
  | IfExprRule
  | ConditionalExprRule
  | BoxExprRule
  | OthersDesignatorRule
  | IteratedAssocRule
  | AggregateAssocRule
  | RegularAggregateRule
  | BracketAggregateRule
  | AggregateRule
  | DirectNameRule
  | ParamAssocRule
  | CallSuffixRule
  | AttrSuffixRule
  | QualifiedNameRule
  | QualNameInternalRule
  | ValueSequenceRule
  | NameRule
  | DefiningNameRule
  | DirectNameOrTargetNameRule
  | TargetNameRule
  | UpdateAttrAggregateRule
  | UpdateAttrContentRule
  | MultidimArrayAssocRule
  | SubtypeNameRule
  | StaticNameRule
  | PrimaryRule
  | ParenExprRule
  | DeclareExprRule
  | FactorRule
  | TermRule
  | UnopTermRule
  | SimpleExprRule
  | BooleanOpRule
  | DiscreteRangeRule
  | ChoiceRule
  | ChoiceListRule
  | RelOpRule
  | MembershipChoiceRule
  | MembershipChoiceListRule
  | RelationRule
  | ExprRule
  | PpDirectiveRule
  | PpThenRule
  | PpExprRule
  | PpTermRule

  let name () = "GrammarRule"

  let wrap (c_value : int) : t =
    match c_value with
    | 0 -> ParentListRule
    | 1 -> ProtectedTypeDeclRule
    | 2 -> ProtectedOpRule
    | 3 -> ProtectedElRule
    | 4 -> ProtectedDefRule
    | 5 -> ProtectedDeclRule
    | 6 -> TaskItemRule
    | 7 -> TaskDefRule
    | 8 -> TaskTypeDeclRule
    | 9 -> SubtypeDeclRule
    | 10 -> InterfaceTypeDefRule
    | 11 -> UnconstrainedIndexRule
    | 12 -> ArrayTypeDefRule
    | 13 -> DiscreteSubtypeDefinitionRule
    | 14 -> ConstraintListRule
    | 15 -> SignedIntTypeDefRule
    | 16 -> ModIntTypeDefRule
    | 17 -> DerivedTypeDefRule
    | 18 -> CompositeConstraintAssocRule
    | 19 -> CompositeConstraintRule
    | 20 -> DigitsConstraintRule
    | 21 -> DeltaConstraintRule
    | 22 -> RangeConstraintRule
    | 23 -> ConstraintRule
    | 24 -> DiscriminantSpecRule
    | 25 -> DiscrSpecListRule
    | 26 -> DiscriminantPartRule
    | 27 -> EnumLiteralDeclRule
    | 28 -> FormalDiscreteTypeDefRule
    | 29 -> RecordDefRule
    | 30 -> RangeSpecRule
    | 31 -> RealTypeDefRule
    | 32 -> SexprOrBoxRule
    | 33 -> OrdinaryFixedPointDefRule
    | 34 -> DecimalFixedPointDefRule
    | 35 -> FloatingPointDefRule
    | 36 -> RecordTypeDefRule
    | 37 -> AccessDefRule
    | 38 -> EnumTypeDefRule
    | 39 -> TypeDefRule
    | 40 -> VariantRule
    | 41 -> AnonymousTypeDeclRule
    | 42 -> IncompleteTypeDeclRule
    | 43 -> TypeDeclRule
    | 44 -> VariantPartRule
    | 45 -> ComponentDefRule
    | 46 -> ComponentItemRule
    | 47 -> ComponentDeclRule
    | 48 -> ComponentListRule
    | 49 -> GenericDeclRule
    | 50 -> GenericFormalPartRule
    | 51 -> GenericFormalDeclRule
    | 52 -> FormalTypeDeclRule
    | 53 -> FormalSubpDeclRule
    | 54 -> RenamingClauseRule
    | 55 -> GenericRenamingDeclRule
    | 56 -> GenericInstantiationRule
    | 57 -> ExceptionDeclRule
    | 58 -> BasicDeclsRule
    | 59 -> PackageRenamingDeclRule
    | 60 -> PackageDeclRule
    | 61 -> BasicDeclRule
    | 62 -> ObjectDeclRule
    | 63 -> SubObjectDeclRule
    | 64 -> NoTypeObjectRenamingDeclRule
    | 65 -> ExtRetStmtObjectDeclRule
    | 66 -> DefiningIdListRule
    | 67 -> NumberDeclRule
    | 68 -> ContractCaseAssocRule
    | 69 -> ContractCasesExprRule
    | 70 -> AbstractStateDeclRule
    | 71 -> MultiAbstractStateDeclRule
    | 72 -> AspectAssocRule
    | 73 -> AspectSpecRule
    | 74 -> SingleTaskDeclRule
    | 75 -> OverridingIndicatorRule
    | 76 -> EntryDeclRule
    | 77 -> ComponentClauseRule
    | 78 -> AspectClauseRule
    | 79 -> ParamSpecRule
    | 80 -> ParamSpecsRule
    | 81 -> SubpSpecRule
    | 82 -> ExprFnRule
    | 83 -> NullSubpDeclRule
    | 84 -> AbstractSubpDeclRule
    | 85 -> SubpRenamingDeclRule
    | 86 -> SimpleSubpDeclRule
    | 87 -> SubpDeclRule
    | 88 -> WithClauseRule
    | 89 -> ContextItemRule
    | 90 -> UseClauseRule
    | 91 -> UsePackageClauseRule
    | 92 -> UseTypeClauseRule
    | 93 -> SubtypeIndicationRule
    | 94 -> DiscreteSubtypeIndicationRule
    | 95 -> ConstrainedSubtypeIndicationRule
    | 96 -> TypeExprRule
    | 97 -> AnonymousTypeRule
    | 98 -> ModeRule
    | 99 -> PragmaArgumentRule
    | 100 -> PragmaRule
    | 101 -> SubunitRule
    | 102 -> LibraryUnitBodyRule
    | 103 -> LibraryUnitRenamingDeclRule
    | 104 -> LibraryItemRule
    | 105 -> CompilationUnitRule
    | 106 -> CompilationRule
    | 107 -> DeclPartRule
    | 108 -> EntryBodyRule
    | 109 -> ProtectedBodyRule
    | 110 -> ProtectedBodyStubRule
    | 111 -> TaskBodyRule
    | 112 -> TaskBodyStubRule
    | 113 -> PackageBodyStubRule
    | 114 -> PackageBodyRule
    | 115 -> TerminateAlternativeRule
    | 116 -> SelectStmtRule
    | 117 -> AcceptStmtRule
    | 118 -> CaseAltRule
    | 119 -> CaseStmtRule
    | 120 -> ExtReturnStmtRule
    | 121 -> IblockStmtRule
    | 122 -> BlockStmtRule
    | 123 -> WhileLoopSpecRule
    | 124 -> IloopStmtRule
    | 125 -> LoopStmtRule
    | 126 -> CompoundStmtRule
    | 127 -> ElsifPartRule
    | 128 -> IfStmtRule
    | 129 -> RaiseStmtRule
    | 130 -> DelayStmtRule
    | 131 -> AbortStmtRule
    | 132 -> BodyRule
    | 133 -> BodyStubRule
    | 134 -> SubpBodyStubRule
    | 135 -> RecovDeclPartRule
    | 136 -> SubpBodyRule
    | 137 -> HandledStmtsRule
    | 138 -> ExceptionHandlerRule
    | 139 -> StmtsRule
    | 140 -> LabelRule
    | 141 -> StmtRule
    | 142 -> CallStmtRule
    | 143 -> SimpleStmtRule
    | 144 -> NullStmtRule
    | 145 -> AssignmentStmtRule
    | 146 -> GotoStmtRule
    | 147 -> ExitStmtRule
    | 148 -> ReturnStmtRule
    | 149 -> RequeueStmtRule
    | 150 -> IdentifierRule
    | 151 -> CharLiteralRule
    | 152 -> StringLiteralRule
    | 153 -> DefiningIdRule
    | 154 -> DecLiteralRule
    | 155 -> IntLiteralRule
    | 156 -> NumLiteralRule
    | 157 -> NullLiteralRule
    | 158 -> AllocatorRule
    | 159 -> ForLoopParamSpecRule
    | 160 -> QuantifiedExprRule
    | 161 -> CaseExprRule
    | 162 -> CaseExprAltRule
    | 163 -> RaiseExprRule
    | 164 -> IfExprRule
    | 165 -> ConditionalExprRule
    | 166 -> BoxExprRule
    | 167 -> OthersDesignatorRule
    | 168 -> IteratedAssocRule
    | 169 -> AggregateAssocRule
    | 170 -> RegularAggregateRule
    | 171 -> BracketAggregateRule
    | 172 -> AggregateRule
    | 173 -> DirectNameRule
    | 174 -> ParamAssocRule
    | 175 -> CallSuffixRule
    | 176 -> AttrSuffixRule
    | 177 -> QualifiedNameRule
    | 178 -> QualNameInternalRule
    | 179 -> ValueSequenceRule
    | 180 -> NameRule
    | 181 -> DefiningNameRule
    | 182 -> DirectNameOrTargetNameRule
    | 183 -> TargetNameRule
    | 184 -> UpdateAttrAggregateRule
    | 185 -> UpdateAttrContentRule
    | 186 -> MultidimArrayAssocRule
    | 187 -> SubtypeNameRule
    | 188 -> StaticNameRule
    | 189 -> PrimaryRule
    | 190 -> ParenExprRule
    | 191 -> DeclareExprRule
    | 192 -> FactorRule
    | 193 -> TermRule
    | 194 -> UnopTermRule
    | 195 -> SimpleExprRule
    | 196 -> BooleanOpRule
    | 197 -> DiscreteRangeRule
    | 198 -> ChoiceRule
    | 199 -> ChoiceListRule
    | 200 -> RelOpRule
    | 201 -> MembershipChoiceRule
    | 202 -> MembershipChoiceListRule
    | 203 -> RelationRule
    | 204 -> ExprRule
    | 205 -> PpDirectiveRule
    | 206 -> PpThenRule
    | 207 -> PpExprRule
    | 208 -> PpTermRule
    | _ -> assert false

  let unwrap (value : t) : int =
    match value with
    | ParentListRule -> 0
    | ProtectedTypeDeclRule -> 1
    | ProtectedOpRule -> 2
    | ProtectedElRule -> 3
    | ProtectedDefRule -> 4
    | ProtectedDeclRule -> 5
    | TaskItemRule -> 6
    | TaskDefRule -> 7
    | TaskTypeDeclRule -> 8
    | SubtypeDeclRule -> 9
    | InterfaceTypeDefRule -> 10
    | UnconstrainedIndexRule -> 11
    | ArrayTypeDefRule -> 12
    | DiscreteSubtypeDefinitionRule -> 13
    | ConstraintListRule -> 14
    | SignedIntTypeDefRule -> 15
    | ModIntTypeDefRule -> 16
    | DerivedTypeDefRule -> 17
    | CompositeConstraintAssocRule -> 18
    | CompositeConstraintRule -> 19
    | DigitsConstraintRule -> 20
    | DeltaConstraintRule -> 21
    | RangeConstraintRule -> 22
    | ConstraintRule -> 23
    | DiscriminantSpecRule -> 24
    | DiscrSpecListRule -> 25
    | DiscriminantPartRule -> 26
    | EnumLiteralDeclRule -> 27
    | FormalDiscreteTypeDefRule -> 28
    | RecordDefRule -> 29
    | RangeSpecRule -> 30
    | RealTypeDefRule -> 31
    | SexprOrBoxRule -> 32
    | OrdinaryFixedPointDefRule -> 33
    | DecimalFixedPointDefRule -> 34
    | FloatingPointDefRule -> 35
    | RecordTypeDefRule -> 36
    | AccessDefRule -> 37
    | EnumTypeDefRule -> 38
    | TypeDefRule -> 39
    | VariantRule -> 40
    | AnonymousTypeDeclRule -> 41
    | IncompleteTypeDeclRule -> 42
    | TypeDeclRule -> 43
    | VariantPartRule -> 44
    | ComponentDefRule -> 45
    | ComponentItemRule -> 46
    | ComponentDeclRule -> 47
    | ComponentListRule -> 48
    | GenericDeclRule -> 49
    | GenericFormalPartRule -> 50
    | GenericFormalDeclRule -> 51
    | FormalTypeDeclRule -> 52
    | FormalSubpDeclRule -> 53
    | RenamingClauseRule -> 54
    | GenericRenamingDeclRule -> 55
    | GenericInstantiationRule -> 56
    | ExceptionDeclRule -> 57
    | BasicDeclsRule -> 58
    | PackageRenamingDeclRule -> 59
    | PackageDeclRule -> 60
    | BasicDeclRule -> 61
    | ObjectDeclRule -> 62
    | SubObjectDeclRule -> 63
    | NoTypeObjectRenamingDeclRule -> 64
    | ExtRetStmtObjectDeclRule -> 65
    | DefiningIdListRule -> 66
    | NumberDeclRule -> 67
    | ContractCaseAssocRule -> 68
    | ContractCasesExprRule -> 69
    | AbstractStateDeclRule -> 70
    | MultiAbstractStateDeclRule -> 71
    | AspectAssocRule -> 72
    | AspectSpecRule -> 73
    | SingleTaskDeclRule -> 74
    | OverridingIndicatorRule -> 75
    | EntryDeclRule -> 76
    | ComponentClauseRule -> 77
    | AspectClauseRule -> 78
    | ParamSpecRule -> 79
    | ParamSpecsRule -> 80
    | SubpSpecRule -> 81
    | ExprFnRule -> 82
    | NullSubpDeclRule -> 83
    | AbstractSubpDeclRule -> 84
    | SubpRenamingDeclRule -> 85
    | SimpleSubpDeclRule -> 86
    | SubpDeclRule -> 87
    | WithClauseRule -> 88
    | ContextItemRule -> 89
    | UseClauseRule -> 90
    | UsePackageClauseRule -> 91
    | UseTypeClauseRule -> 92
    | SubtypeIndicationRule -> 93
    | DiscreteSubtypeIndicationRule -> 94
    | ConstrainedSubtypeIndicationRule -> 95
    | TypeExprRule -> 96
    | AnonymousTypeRule -> 97
    | ModeRule -> 98
    | PragmaArgumentRule -> 99
    | PragmaRule -> 100
    | SubunitRule -> 101
    | LibraryUnitBodyRule -> 102
    | LibraryUnitRenamingDeclRule -> 103
    | LibraryItemRule -> 104
    | CompilationUnitRule -> 105
    | CompilationRule -> 106
    | DeclPartRule -> 107
    | EntryBodyRule -> 108
    | ProtectedBodyRule -> 109
    | ProtectedBodyStubRule -> 110
    | TaskBodyRule -> 111
    | TaskBodyStubRule -> 112
    | PackageBodyStubRule -> 113
    | PackageBodyRule -> 114
    | TerminateAlternativeRule -> 115
    | SelectStmtRule -> 116
    | AcceptStmtRule -> 117
    | CaseAltRule -> 118
    | CaseStmtRule -> 119
    | ExtReturnStmtRule -> 120
    | IblockStmtRule -> 121
    | BlockStmtRule -> 122
    | WhileLoopSpecRule -> 123
    | IloopStmtRule -> 124
    | LoopStmtRule -> 125
    | CompoundStmtRule -> 126
    | ElsifPartRule -> 127
    | IfStmtRule -> 128
    | RaiseStmtRule -> 129
    | DelayStmtRule -> 130
    | AbortStmtRule -> 131
    | BodyRule -> 132
    | BodyStubRule -> 133
    | SubpBodyStubRule -> 134
    | RecovDeclPartRule -> 135
    | SubpBodyRule -> 136
    | HandledStmtsRule -> 137
    | ExceptionHandlerRule -> 138
    | StmtsRule -> 139
    | LabelRule -> 140
    | StmtRule -> 141
    | CallStmtRule -> 142
    | SimpleStmtRule -> 143
    | NullStmtRule -> 144
    | AssignmentStmtRule -> 145
    | GotoStmtRule -> 146
    | ExitStmtRule -> 147
    | ReturnStmtRule -> 148
    | RequeueStmtRule -> 149
    | IdentifierRule -> 150
    | CharLiteralRule -> 151
    | StringLiteralRule -> 152
    | DefiningIdRule -> 153
    | DecLiteralRule -> 154
    | IntLiteralRule -> 155
    | NumLiteralRule -> 156
    | NullLiteralRule -> 157
    | AllocatorRule -> 158
    | ForLoopParamSpecRule -> 159
    | QuantifiedExprRule -> 160
    | CaseExprRule -> 161
    | CaseExprAltRule -> 162
    | RaiseExprRule -> 163
    | IfExprRule -> 164
    | ConditionalExprRule -> 165
    | BoxExprRule -> 166
    | OthersDesignatorRule -> 167
    | IteratedAssocRule -> 168
    | AggregateAssocRule -> 169
    | RegularAggregateRule -> 170
    | BracketAggregateRule -> 171
    | AggregateRule -> 172
    | DirectNameRule -> 173
    | ParamAssocRule -> 174
    | CallSuffixRule -> 175
    | AttrSuffixRule -> 176
    | QualifiedNameRule -> 177
    | QualNameInternalRule -> 178
    | ValueSequenceRule -> 179
    | NameRule -> 180
    | DefiningNameRule -> 181
    | DirectNameOrTargetNameRule -> 182
    | TargetNameRule -> 183
    | UpdateAttrAggregateRule -> 184
    | UpdateAttrContentRule -> 185
    | MultidimArrayAssocRule -> 186
    | SubtypeNameRule -> 187
    | StaticNameRule -> 188
    | PrimaryRule -> 189
    | ParenExprRule -> 190
    | DeclareExprRule -> 191
    | FactorRule -> 192
    | TermRule -> 193
    | UnopTermRule -> 194
    | SimpleExprRule -> 195
    | BooleanOpRule -> 196
    | DiscreteRangeRule -> 197
    | ChoiceRule -> 198
    | ChoiceListRule -> 199
    | RelOpRule -> 200
    | MembershipChoiceRule -> 201
    | MembershipChoiceListRule -> 202
    | RelationRule -> 203
    | ExprRule -> 204
    | PpDirectiveRule -> 205
    | PpThenRule -> 206
    | PpExprRule -> 207
    | PpTermRule -> 208

   let c_type = view int ~read:wrap ~write:unwrap
end


let free = foreign ~from:c_lib
  "ada_free"
  (ptr void @-> returning void)

(** Assuming char_ptr is a valid char*, convert it to a native Ocaml
  * string and free the C pointer.
  *)
let unwrap_str char_ptr =
  let str = Ctypes.coerce (ptr char) string char_ptr in
  free (Ctypes.coerce (ptr char) (ptr void) char_ptr);
  str


let default_grammar_rule = GrammarRule.CompilationRule

module Sloc = struct
  type t = {
    line : int;
    column : int;
  }

  let c_struct : t structure typ = structure "sloc"
  let line = field c_struct "line" uint32_t
  let column = field c_struct "column" uint16_t
  let () = seal c_struct

  let wrap (c_value : t structure) : t = {
    line = Unsigned.UInt32.to_int (getf c_value line);
    column = Unsigned.UInt16.to_int (getf c_value column);
  }

  let unwrap (value : t) : t structure =
    let c_value = make c_struct in
    setf c_value line (Unsigned.UInt32.of_int (value.line));
    setf c_value column (Unsigned.UInt16.of_int (value.column));
    c_value

  let c_type = view c_struct ~read:wrap ~write:unwrap
end

module SlocRange = struct
  type t = {
    loc_start : Sloc.t;
    loc_end : Sloc.t;
  }

  let c_struct : t structure typ = structure "sloc_range"
  let loc_start = field c_struct "loc_start" Sloc.c_type
  let loc_end = field c_struct "loc_end" Sloc.c_type
  let () = seal c_struct

  let wrap (c_value : t structure) : t = {
    loc_start = getf c_value loc_start;
    loc_end = getf c_value loc_end;
  }

  let unwrap (value : t) : t structure =
    let c_value = make c_struct in
    setf c_value loc_start value.loc_start;
    setf c_value loc_end value.loc_end;
    c_value

  let c_type = view c_struct ~read:wrap ~write:unwrap

  let pp fmt sloc_range =
    Format.fprintf fmt "<SlocRange %d:%d-%d:%d>"
      sloc_range.loc_start.line
      sloc_range.loc_start.column
      sloc_range.loc_end.line
      sloc_range.loc_end.column
end

module Diagnostic = struct
  type t = {
    sloc_range : SlocRange.t;
    message : string
  }

  let c_struct : t structure typ = structure "diagnostic"
  let sloc_range = field c_struct "sloc_range" SlocRange.c_type
  let message = field c_struct "message" Text.c_struct
  let () = seal c_struct

  let wrap (c_value : t structure) : t = {
    sloc_range = getf c_value sloc_range;
    message = Text.wrap (getf c_value message);
  }

  let unwrap (value : t) : t structure =
    let c_value = make c_struct in
    (* sloc_range is not a pointer, thus we have a copy here which is safe. *)
    setf c_value sloc_range value.sloc_range;
    (* message is not a pointer, thus we have a copy here which is safe.
       HOWEVER, there is a link from value.message to another pointer which can
       be freed by the GC if we don't propagate this link. *)
    let c_value_message = Text.unwrap value.message in
    add_gc_link ~from:c_value ~to_:c_value_message;
    setf c_value message c_value_message;
    c_value

  let c_type = view c_struct ~read:wrap ~write:unwrap
end

module TokenData = struct
  type t = unit ptr
end

module Token = struct
  (* We don't have access to AnalysisContextStruct at this point. We don't need
     to do anything with the context value except pass it around, so map it as
     an opaque pointer instead. *)
  type dummy_context = unit ptr

  type t = {
    context : dummy_context;
    token_data : TokenData.t;
    token_index : int;
    trivia_index : int;
    kind : int;
    text : string;
    sloc_range : SlocRange.t;
  }

  let c_type : t structure typ = structure "token"
  let context = field c_type "context" (ptr void)
  let token_data = field c_type "token_data" (ptr void)
  let token_index = field c_type "token_index" int
  let trivia_index = field c_type "trivia_index" int
  let () = seal c_type

  let _token_get_kind = foreign ~from:c_lib
    "ada_token_get_kind"
    (ptr c_type @-> raisable int)

  let _token_kind_name = foreign ~from:c_lib
    "ada_token_kind_name"
    (int @-> raisable (ptr char))

  let _token_sloc_range = foreign ~from:c_lib
    "ada_token_sloc_range"
    (ptr c_type @-> ptr SlocRange.c_type @-> raisable void)

  let token_kind_name kind =
    unwrap_str (_token_kind_name kind)

  let token_range_text = foreign ~from:c_lib
    "ada_token_range_text"
    (ptr c_type @-> ptr c_type @-> ptr Text.c_type @-> raisable int)

  let wrap (c_value : t structure) : t option =
  let token_data = getf c_value token_data in
  if is_null token_data then
    None
  else
    Some {
      context = getf c_value context;
      token_data;
      token_index = getf c_value token_index;
      trivia_index = getf c_value trivia_index;
      kind = _token_get_kind (addr c_value);
      text =
        (let c_result_ptr = allocate_n Text.c_type ~count:1 in
         let _ = token_range_text (addr c_value) (addr c_value) c_result_ptr in
         !@ c_result_ptr);
      sloc_range =
        (let c_result_ptr = allocate_n SlocRange.c_type ~count:1 in
         let _ = _token_sloc_range (addr c_value) c_result_ptr in
         !@ c_result_ptr);
    }

  let unwrap (value : t) : t structure =
    let c_value = make c_type in
    setf c_value context value.context;
    setf c_value token_data value.token_data;
    setf c_value token_index value.token_index;
    setf c_value trivia_index value.trivia_index;
    c_value

  let kind_name token = token_kind_name (_token_get_kind (addr (unwrap token)))

  let sloc_range token =
    let c_result_ptr = allocate_n SlocRange.c_type ~count:1 in
    let _ = _token_sloc_range (addr (unwrap token)) c_result_ptr in
    !@ c_result_ptr

  let token_next = foreign ~from:c_lib
    "ada_token_next"
    (ptr c_type @-> ptr c_type @-> raisable void)

  let token_previous = foreign ~from:c_lib
    "ada_token_previous"
    (ptr c_type @-> ptr c_type @-> raisable void)

  let is_equivalent = foreign ~from:c_lib
    "ada_token_is_equivalent"
    (ptr c_type @-> ptr c_type @-> raisable bool)

  let pp fmt token =
    let pp_text fmt = function
      | "" -> Format.pp_print_string fmt ""
      | _ as text -> Format.fprintf fmt " %S" text
    in
    Format.fprintf fmt "<Token %s%a at %a>"
      (kind_name token)
      pp_text token.text
      SlocRange.pp token.sloc_range

  let text_range token_first token_last =
    let c_result_ptr = allocate_n Text.c_type ~count:1 in
    let res =
      token_range_text
        (addr (unwrap token_first))
        (addr (unwrap token_last))
        c_result_ptr
    in
    if res = 0 then
      raise (Invalid_argument
        (Format.asprintf "%a and %a come from different units"
          pp token_first
          pp token_last));
    !@ c_result_ptr

  let text token = text_range token token

  let next token =
    let c_next_token_ptr = allocate_n c_type ~count:1 in
    token_next (addr (unwrap token)) c_next_token_ptr ;
    wrap (!@ c_next_token_ptr)

  let previous token =
    let c_next_token_ptr = allocate_n c_type ~count:1 in
    token_previous (addr (unwrap token)) c_next_token_ptr ;
    wrap (!@ c_next_token_ptr)

  let is_trivia token =
    token.trivia_index != 0

  let index token =
    match token.trivia_index with
    | 0 ->
        token.token_index - 1
    | _ ->
        token.trivia_index - 1

  let compare one other =
    let open Stdlib in
    let compare_token_data = compare one.token_data other.token_data in
    if compare_token_data = 0 then
      let compare_token_index = compare one.token_index other.token_index in
      if compare_token_index = 0 then
        compare one.trivia_index other.trivia_index
      else
        compare_token_index
    else
      compare_token_data

  let equal one other =
    compare one other = 0

  let hash token =
    Hashtbl.hash
      (token.token_data
       , token.token_index
       , token.trivia_index)

  let is_equivalent one other =
    is_equivalent (addr (unwrap one)) (addr (unwrap other))

end

module BareNode = struct
  type t = unit ptr
end

module Rebindings = struct
  type t = unit ptr
end

module FileReader = struct
  (* Use a pointer to pointer to register a finaliser when creating a value of
     this type. *)
  type t = unit ptr ptr

  let dec_ref =
    foreign ~from:c_lib "ada_dec_ref_file_reader"
      (ptr void @-> raisable void)

  let read v =
    let finalise arg =
      dec_ref (!@ arg)
    in
    allocate (ptr void) ~finalise v

  let c_type = view (ptr void) ~read ~write:(!@)
end

module UnitProvider = struct
  (* The real C type of a context is a void*. But we use a pointer to this
     type, to be able to allocate a value of t and attach a finalizer to it. *)
  type t = unit ptr ptr

  let c_type = ptr void

  let null = allocate c_type null

  
        let dec_ref =
    let f =
      foreign ~from:c_lib "ada_dec_ref_unit_provider"
        (c_type @-> raisable void)
    in
    (fun v ->
      f (!@ v))

  (* Keep is a special argument designed to extend the lifetime of an OCaml
     value. We put in a ref which is set to [None] only in the finalization
     function. This guarantees it won't be collected before. *)
  let wrap ?keep v =
    let ref_keep = ref keep in
    let finalise arg =
      ref_keep := None;
      dec_ref arg
    in
    allocate ~finalise (ptr void) v

  let create_auto_provider =
    foreign ~from:c_lib "ada_create_auto_provider"
      (ptr (ptr char) @-> string @-> raisable c_type)

  let auto input_files =
    (* Convert the names of the input files into pointers to C strings. We used
       to use the high-level type [Ctypes.string] type, but this was causing
       memory corruption problems. We switched to [ptr char] instead. *)
    let cstrings =
      List.map (fun f -> CArray.(start (of_string f))) input_files
    in
    (* Add a null pointer at the end. This is part of the LAL calling
       convention. *)
    let null_ptr = from_voidp char Ctypes.null in
    let cstrings_null = List.rev_append cstrings [null_ptr] in
    (* Create an array with all these pointers *)
    let array = CArray.of_list (ptr char) cstrings_null in
    let ptr = CArray.start array in
    let result = create_auto_provider ptr "" in
    (* Extend the lifetime of cstrings here, to make sure it is not garbage
       collected while [create_auto_provided] executes, nor after. (It is not
       clear whether LAL keeps internal references to this C object after the
       call to create_auto_provider, but simpler fixes do not work anyway. *)
    wrap ~keep:cstrings result


end



      
module MetadataStruct = struct
  type t

   
  let c_type : t structure typ = structure "metadata"
  let dottable_subp =
    field c_type "dottable_subp" bool
  let primitive =
    field c_type "primitive" (ptr void)
  let primitive_real_type =
    field c_type "primitive_real_type" (ptr void)
  let () = seal c_type

end

      
module EntityInfoStruct = struct
  type t

   
  let c_type : t structure typ = structure "entity_info"
  let md =
    field c_type "md" MetadataStruct.c_type
  let rebindings =
    field c_type "rebindings" (ptr void)
  let from_rebound =
    field c_type "from_rebound" bool
  let () = seal c_type

end

         
module EntityStruct = struct
  type t

   
  let c_type : t structure typ = structure "ada_node"
  let node =
    field c_type "node" (ptr void)
  let info =
    field c_type "info" EntityInfoStruct.c_type
  let () = seal c_type

end

      
module AnalysisUnitStruct : sig
  type t = unit ptr

  val c_type : t typ

  val unit_root : t -> EntityStruct.t structure ptr -> unit

  val unit_diagnostic_count : t -> int

  val unit_diagnostic : t -> int -> Diagnostic.t ptr -> int

  val unit_filename : t -> char ptr

  val unit_reparse_from_file : t -> string -> int

  val unit_reparse_from_buffer :
    t -> string -> string -> Unsigned.size_t -> int

  val unit_first_token : t -> Token.t structure ptr -> unit

  val unit_last_token : t -> Token.t structure ptr -> unit

  val unit_token_count : t -> int

  val unit_trivia_count : t -> int
end = struct
  (* Module defining the c structure of an analysis unit *)

  type t = unit ptr
  let c_type = ptr void

  let unit_root = foreign ~from:c_lib "ada_unit_root"
    (c_type @-> ptr EntityStruct.c_type @-> raisable void)

  let unit_diagnostic_count = foreign ~from:c_lib
    "ada_unit_diagnostic_count"
    (c_type @-> raisable int)

  let unit_diagnostic = foreign ~from:c_lib
    "ada_unit_diagnostic"
    (c_type @-> int @-> ptr Diagnostic.c_type @-> raisable int)

  let unit_filename = foreign ~from:c_lib
    "ada_unit_filename"
    (c_type @-> raisable (ptr char))

  let unit_reparse_from_file = foreign ~from:c_lib
    "ada_unit_reparse_from_file"
    (c_type
     @-> string
     @-> raisable int)

  let unit_reparse_from_buffer = foreign ~from:c_lib
    "ada_unit_reparse_from_buffer"
    (c_type
     @-> string
     @-> string
     @-> size_t
     @-> raisable int)

  let unit_first_token = foreign ~from:c_lib
    "ada_unit_first_token"
    (c_type
     @-> ptr Token.c_type
     @-> raisable void)

  let unit_last_token = foreign ~from:c_lib
    "ada_unit_last_token"
    (c_type
     @-> ptr Token.c_type
     @-> raisable void)

  let unit_token_count = foreign ~from:c_lib
    "ada_unit_token_count"
    (c_type @-> raisable int)

  let unit_trivia_count = foreign ~from:c_lib
    "ada_unit_trivia_count"
    (c_type @-> raisable int)
end


      
module AnalysisContextStruct : sig
  type t

  val c_type : t typ

  val allocate_analysis_context : ?keep:'a -> unit -> t
  val initialize_analysis_context :
    t -> string -> unit ptr -> unit ptr -> unit ptr -> bool -> int -> unit

  val get_analysis_unit_from_file :
    t -> string -> string -> bool -> GrammarRule.t -> AnalysisUnitStruct.t

  val get_analysis_unit_from_buffer :
    t
    -> string (* Filename *)
    -> string (* Charset *)
    -> string (* Buffer *)
    -> Unsigned.size_t (* Buffer size *)
    -> GrammarRule.t
    -> AnalysisUnitStruct.t
end = struct
  (* Module defining the c structure of an analysis context *)

  (* The real C type of a context is a void*. But we use a pointer to this
     type, to be able to allocate a value of t and attach a finalizer to it.
     See wrap function *)
  type t = unit ptr ptr

  let unwrap (value : t) : unit ptr = !@value

  (* The read part is not required as the only function returning a c_type is
     allocate_analysis_context which is manually written to take an object
     to keep alive as argument *)
  let c_type = view (ptr void) ~read:(fun _ -> assert false) ~write:unwrap

  let context_decref =
    foreign ~from:c_lib "ada_context_decref"
      (c_type @-> raisable void)

  let c_allocate_analysis_context =
    foreign ~from:c_lib "ada_allocate_analysis_context"
      ( void @-> raisable (ptr void) )

  let allocate_analysis_context ?keep () =
    (* To deallocate cleanly the context, we need to call context_decref.
       Allocate a value and attach a finalizer to it. Use the keep option
       to keep an object alive while the analysis context is. *)
    let ref_keep = ref keep in
    let finalise arg =
      ref_keep := None;
      context_decref arg
    in
    let c_value = c_allocate_analysis_context () in
    allocate ~finalise (ptr void) c_value

  let initialize_analysis_context =
    foreign ~from:c_lib "ada_initialize_analysis_context"
      ( c_type @-> string @-> ptr void @-> UnitProvider.c_type @-> ptr void
      @-> bool @-> int @-> raisable void )

  let get_analysis_unit_from_file =
    foreign ~from:c_lib "ada_get_analysis_unit_from_file"
      ( c_type @-> string @-> string @-> bool @-> GrammarRule.c_type
      @-> raisable AnalysisUnitStruct.c_type )

  let get_analysis_unit_from_buffer =
    foreign ~from:c_lib "ada_get_analysis_unit_from_buffer"
      ( c_type @-> string (* Filename *) @-> string (* Charset *)
      @-> string (* Buffer *) @-> size_t (* Buffer size *)
      @-> GrammarRule.c_type
      @-> raisable AnalysisUnitStruct.c_type )

end

      
module Symbol : sig
  type t = string

  val c_type : t structure typ

  val wrap : (t structure) -> t

  val unwrap : AnalysisContextStruct.t -> t -> (t structure)

  val symbol_text : t structure ptr -> string ptr -> unit

  val context_symbol :
    AnalysisContextStruct.t -> string ptr -> t structure ptr -> int
end = struct
  type t = string

  let c_type : t structure typ = structure "symbol"
  let data = field c_type "data" (ptr void)
  let bounds = field c_type "bounds" (ptr void)
  let () = seal c_type

  let symbol_text = foreign ~from:c_lib "ada_symbol_text"
    (ptr c_type @-> ptr Text.c_type @-> raisable void)

  let wrap (c_value : t structure) : t =
    let c_result_ptr = allocate_n Text.c_type ~count:1 in
    symbol_text (addr c_value) c_result_ptr;
    !@ c_result_ptr

  let context_symbol = foreign ~from:c_lib "ada_context_symbol"
    (AnalysisContextStruct.c_type
     @-> ptr Text.c_type
     @-> ptr c_type
     @-> raisable int)

  let unwrap (ctx : AnalysisContextStruct.t) (value : t) : t structure =
    let result = make c_type in
    let code =
      context_symbol ctx (allocate Text.c_type value) (addr result)
    in
    if code = 0 then
      raise (InvalidSymbolError value) ;
    result
end

      
module AspectStruct = struct
  type t

   
  let c_type : t structure typ = structure "aspect"
  let exists =
    field c_type "exists" bool
  let node =
    field c_type "node" EntityStruct.c_type
  let value =
    field c_type "value" EntityStruct.c_type
  let inherited =
    field c_type "inherited" bool
  let () = seal c_type

end

      
module CompletionItemStruct = struct
  type t

   
  let c_type : t structure typ = structure "completion_item"
  let decl =
    field c_type "decl" EntityStruct.c_type
  let is_dot_call =
    field c_type "is_dot_call" bool
  let is_visible =
    field c_type "is_visible" bool
  let weight =
    field c_type "weight" int
  let () = seal c_type

end

      
module DiscreteRangeStruct = struct
  type t

   
  let c_type : t structure typ = structure "discrete_range"
  let low_bound =
    field c_type "low_bound" EntityStruct.c_type
  let high_bound =
    field c_type "high_bound" EntityStruct.c_type
  let () = seal c_type

end

      
module DiscriminantValuesStruct = struct
  type t

   
  let c_type : t structure typ = structure "discriminant_values"
  let discriminant =
    field c_type "discriminant" EntityStruct.c_type
  let values =
    field c_type "values" EntityStruct.c_type
  let () = seal c_type

end

      
module DocAnnotationStruct = struct
  type t

   
  let c_type : t structure typ = structure "doc_annotation"
  let key =
    field c_type "key" StringType.c_type
  let value =
    field c_type "value" StringType.c_type
  let () = seal c_type

end

      
module ParamActualStruct = struct
  type t

   
  let c_type : t structure typ = structure "param_actual"
  let param =
    field c_type "param" EntityStruct.c_type
  let actual =
    field c_type "actual" EntityStruct.c_type
  let () = seal c_type

end

      
module RefResultStruct = struct
  type t

   
  let c_type : t structure typ = structure "ref_result"
  let ref_ =
    field c_type "ref_" EntityStruct.c_type
  let kind =
    field c_type "kind" RefResultKind.c_type
  let () = seal c_type

end

      
module RefdDeclStruct = struct
  type t

   
  let c_type : t structure typ = structure "refd_decl"
  let decl =
    field c_type "decl" EntityStruct.c_type
  let kind =
    field c_type "kind" RefResultKind.c_type
  let () = seal c_type

end

      
module RefdDefStruct = struct
  type t

   
  let c_type : t structure typ = structure "refd_def"
  let def_name =
    field c_type "def_name" EntityStruct.c_type
  let kind =
    field c_type "kind" RefResultKind.c_type
  let () = seal c_type

end

         
module AdaNodeArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "ada_node_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    EntityStruct.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "ada_ada_node_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "ada_ada_node_array_dec_ref"
    (c_type @-> raisable void)

end

         
module DiscriminantValuesArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "discriminant_values_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    DiscriminantValuesStruct.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "ada_discriminant_values_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "ada_discriminant_values_array_dec_ref"
    (c_type @-> raisable void)

end

      
module ShapeStruct = struct
  type t

   
  let c_type : t structure typ = structure "shape"
  let components =
    field c_type "components" AdaNodeArrayStruct.c_type
  let discriminants_values =
    field c_type "discriminants_values" DiscriminantValuesArrayStruct.c_type
  let () = seal c_type

end

      
module SubstitutionStruct = struct
  type t

   
  let c_type : t structure typ = structure "substitution"
  let from_decl =
    field c_type "from_decl" EntityStruct.c_type
  let to_value =
    field c_type "to_value" BigInteger.c_type
  let value_type =
    field c_type "value_type" EntityStruct.c_type
  let () = seal c_type

end

         
module DocAnnotationArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "doc_annotation_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    DocAnnotationStruct.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "ada_doc_annotation_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "ada_doc_annotation_array_dec_ref"
    (c_type @-> raisable void)

end

         
module ParamActualArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "param_actual_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    ParamActualStruct.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "ada_param_actual_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "ada_param_actual_array_dec_ref"
    (c_type @-> raisable void)

end

         
module RefResultArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "ref_result_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    RefResultStruct.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "ada_ref_result_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "ada_ref_result_array_dec_ref"
    (c_type @-> raisable void)

end

         
module ShapeArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "shape_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    ShapeStruct.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "ada_shape_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "ada_shape_array_dec_ref"
    (c_type @-> raisable void)

end

         
module SubstitutionArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "substitution_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    SubstitutionStruct.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "ada_substitution_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "ada_substitution_array_dec_ref"
    (c_type @-> raisable void)

end

         
module AnalysisUnitArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "analysis_unit_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    AnalysisUnitStruct.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "ada_analysis_unit_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "ada_analysis_unit_array_dec_ref"
    (c_type @-> raisable void)

end

         
module UnboundedTextTypeArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "unbounded_text_type_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    Symbol.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "ada_unbounded_text_type_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "ada_unbounded_text_type_array_dec_ref"
    (c_type @-> raisable void)

end


module CFunctions = struct
  let node_kind = foreign ~from:c_lib "ada_node_kind"
    (ptr EntityStruct.c_type @-> raisable int)

  let image = foreign ~from:c_lib "ada_node_image"
    (ptr EntityStruct.c_type
     @-> ptr Text.c_type
     @-> raisable void)

  let node_sloc_range = foreign ~from:c_lib
    "ada_node_sloc_range"
    (ptr EntityStruct.c_type
     @-> ptr SlocRange.c_type
     @-> raisable void)

  let lookup_in_node = foreign ~from:c_lib
    "ada_lookup_in_node"
    (ptr EntityStruct.c_type
     @-> ptr Sloc.c_type
     @-> ptr EntityStruct.c_type
     @-> raisable void)

  let node_is_token_node = foreign ~from:c_lib
    "ada_node_is_token_node"
    (ptr EntityStruct.c_type
     @-> raisable bool)

  let ada_node_p_declarative_scope = foreign ~from:c_lib
    "ada_ada_node_p_declarative_scope"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_p_enclosing_compilation_unit = foreign ~from:c_lib
    "ada_ada_node_p_enclosing_compilation_unit"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_p_get_uninstantiated_node = foreign ~from:c_lib
    "ada_ada_node_p_get_uninstantiated_node"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)


  let ada_node_p_valid_keywords = foreign ~from:c_lib
    "ada_ada_node_p_valid_keywords"
    (ptr EntityStruct.c_type
    @-> ptr UnboundedTextTypeArrayStruct.c_type
    @-> raisable int)

  let ada_node_p_generic_instantiations = foreign ~from:c_lib
    "ada_ada_node_p_generic_instantiations"
    (ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let ada_node_p_semantic_parent = foreign ~from:c_lib
    "ada_ada_node_p_semantic_parent"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_p_parent_basic_decl = foreign ~from:c_lib
    "ada_ada_node_p_parent_basic_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_p_filter_is_imported_by = foreign ~from:c_lib
    "ada_ada_node_p_filter_is_imported_by"
    (ptr EntityStruct.c_type
        
    @-> AnalysisUnitArrayStruct.c_type
        
    @-> bool
    @-> ptr AnalysisUnitArrayStruct.c_type
    @-> raisable int)

  let ada_node_p_xref_entry_point = foreign ~from:c_lib
    "ada_ada_node_p_xref_entry_point"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let ada_node_p_resolve_names = foreign ~from:c_lib
    "ada_ada_node_p_resolve_names"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let ada_node_p_standard_unit = foreign ~from:c_lib
    "ada_ada_node_p_standard_unit"
    (ptr EntityStruct.c_type
    @-> ptr AnalysisUnitStruct.c_type
    @-> raisable int)

  let ada_node_p_std_entity = foreign ~from:c_lib
    "ada_ada_node_p_std_entity"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_p_bool_type = foreign ~from:c_lib
    "ada_ada_node_p_bool_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_p_int_type = foreign ~from:c_lib
    "ada_ada_node_p_int_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_p_universal_int_type = foreign ~from:c_lib
    "ada_ada_node_p_universal_int_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_p_universal_real_type = foreign ~from:c_lib
    "ada_ada_node_p_universal_real_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_p_std_char_type = foreign ~from:c_lib
    "ada_ada_node_p_std_char_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_p_std_wide_char_type = foreign ~from:c_lib
    "ada_ada_node_p_std_wide_char_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_p_std_wide_wide_char_type = foreign ~from:c_lib
    "ada_ada_node_p_std_wide_wide_char_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_p_top_level_decl = foreign ~from:c_lib
    "ada_ada_node_p_top_level_decl"
    (ptr EntityStruct.c_type
        
    @-> AnalysisUnitStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_p_choice_match = foreign ~from:c_lib
    "ada_ada_node_p_choice_match"
    (ptr EntityStruct.c_type
        
    @-> ptr BigInteger.c_type
    @-> ptr bool
    @-> raisable int)

  let ada_node_p_gnat_xref = foreign ~from:c_lib
    "ada_ada_node_p_gnat_xref"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_parent = foreign ~from:c_lib
    "ada_ada_node_parent"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_parents = foreign ~from:c_lib
    "ada_ada_node_parents"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let ada_node_children = foreign ~from:c_lib
    "ada_ada_node_children"
    (ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let ada_node_token_start = foreign ~from:c_lib
    "ada_ada_node_token_start"
    (ptr EntityStruct.c_type
    @-> ptr Token.c_type
    @-> raisable int)

  let ada_node_token_end = foreign ~from:c_lib
    "ada_ada_node_token_end"
    (ptr EntityStruct.c_type
    @-> ptr Token.c_type
    @-> raisable int)

  let ada_node_child_index = foreign ~from:c_lib
    "ada_ada_node_child_index"
    (ptr EntityStruct.c_type
    @-> ptr int
    @-> raisable int)

  let ada_node_previous_sibling = foreign ~from:c_lib
    "ada_ada_node_previous_sibling"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_next_sibling = foreign ~from:c_lib
    "ada_ada_node_next_sibling"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ada_node_unit = foreign ~from:c_lib
    "ada_ada_node_unit"
    (ptr EntityStruct.c_type
    @-> ptr AnalysisUnitStruct.c_type
    @-> raisable int)

  let ada_node_is_ghost = foreign ~from:c_lib
    "ada_ada_node_is_ghost"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let ada_node_full_sloc_image = foreign ~from:c_lib
    "ada_ada_node_full_sloc_image"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let abort_node_p_as_bool = foreign ~from:c_lib
    "ada_abort_node_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let abstract_node_p_as_bool = foreign ~from:c_lib
    "ada_abstract_node_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let assoc_list_p_zip_with_params = foreign ~from:c_lib
    "ada_assoc_list_p_zip_with_params"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr ParamActualArrayStruct.c_type
    @-> raisable int)

  let aliased_node_p_as_bool = foreign ~from:c_lib
    "ada_aliased_node_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let all_node_p_as_bool = foreign ~from:c_lib
    "ada_all_node_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let constrained_array_indices_f_list = foreign ~from:c_lib
    "ada_constrained_array_indices_f_list"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let unconstrained_array_indices_f_types = foreign ~from:c_lib
    "ada_unconstrained_array_indices_f_types"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let aspect_assoc_f_id = foreign ~from:c_lib
    "ada_aspect_assoc_f_id"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let aspect_assoc_f_expr = foreign ~from:c_lib
    "ada_aspect_assoc_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let aspect_assoc_p_is_ghost_code = foreign ~from:c_lib
    "ada_aspect_assoc_p_is_ghost_code"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let at_clause_f_name = foreign ~from:c_lib
    "ada_at_clause_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let at_clause_f_expr = foreign ~from:c_lib
    "ada_at_clause_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let attribute_def_clause_f_attribute_expr = foreign ~from:c_lib
    "ada_attribute_def_clause_f_attribute_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let attribute_def_clause_f_expr = foreign ~from:c_lib
    "ada_attribute_def_clause_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let enum_rep_clause_f_type_name = foreign ~from:c_lib
    "ada_enum_rep_clause_f_type_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let enum_rep_clause_f_aggregate = foreign ~from:c_lib
    "ada_enum_rep_clause_f_aggregate"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let enum_rep_clause_p_params = foreign ~from:c_lib
    "ada_enum_rep_clause_p_params"
    (ptr EntityStruct.c_type
    @-> ptr ParamActualArrayStruct.c_type
    @-> raisable int)

  let record_rep_clause_f_name = foreign ~from:c_lib
    "ada_record_rep_clause_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let record_rep_clause_f_at_expr = foreign ~from:c_lib
    "ada_record_rep_clause_f_at_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let record_rep_clause_f_components = foreign ~from:c_lib
    "ada_record_rep_clause_f_components"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let aspect_spec_f_aspect_assocs = foreign ~from:c_lib
    "ada_aspect_spec_f_aspect_assocs"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_assoc_p_assoc_expr = foreign ~from:c_lib
    "ada_base_assoc_p_assoc_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let contract_case_assoc_f_guard = foreign ~from:c_lib
    "ada_contract_case_assoc_f_guard"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let contract_case_assoc_f_consequence = foreign ~from:c_lib
    "ada_contract_case_assoc_f_consequence"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let pragma_argument_assoc_f_name = foreign ~from:c_lib
    "ada_pragma_argument_assoc_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let pragma_argument_assoc_f_expr = foreign ~from:c_lib
    "ada_pragma_argument_assoc_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_formal_param_holder_p_abstract_formal_params = foreign ~from:c_lib
    "ada_base_formal_param_holder_p_abstract_formal_params"
    (ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let base_formal_param_holder_p_formal_params = foreign ~from:c_lib
    "ada_base_formal_param_holder_p_formal_params"
    (ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let base_formal_param_holder_p_nb_min_params = foreign ~from:c_lib
    "ada_base_formal_param_holder_p_nb_min_params"
    (ptr EntityStruct.c_type
    @-> ptr int
    @-> raisable int)

  let base_formal_param_holder_p_nb_max_params = foreign ~from:c_lib
    "ada_base_formal_param_holder_p_nb_max_params"
    (ptr EntityStruct.c_type
    @-> ptr int
    @-> raisable int)

  let base_formal_param_holder_p_param_types = foreign ~from:c_lib
    "ada_base_formal_param_holder_p_param_types"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let base_subp_spec_p_returns = foreign ~from:c_lib
    "ada_base_subp_spec_p_returns"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_subp_spec_p_params = foreign ~from:c_lib
    "ada_base_subp_spec_p_params"
    (ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let base_subp_spec_p_primitive_subp_types = foreign ~from:c_lib
    "ada_base_subp_spec_p_primitive_subp_types"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let base_subp_spec_p_primitive_subp_first_type = foreign ~from:c_lib
    "ada_base_subp_spec_p_primitive_subp_first_type"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_subp_spec_p_primitive_subp_tagged_type = foreign ~from:c_lib
    "ada_base_subp_spec_p_primitive_subp_tagged_type"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_subp_spec_p_return_type = foreign ~from:c_lib
    "ada_base_subp_spec_p_return_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_spec_f_entry_name = foreign ~from:c_lib
    "ada_entry_spec_f_entry_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_spec_f_family_type = foreign ~from:c_lib
    "ada_entry_spec_f_family_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_spec_f_entry_params = foreign ~from:c_lib
    "ada_entry_spec_f_entry_params"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subp_spec_f_subp_kind = foreign ~from:c_lib
    "ada_subp_spec_f_subp_kind"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subp_spec_f_subp_name = foreign ~from:c_lib
    "ada_subp_spec_f_subp_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subp_spec_f_subp_params = foreign ~from:c_lib
    "ada_subp_spec_f_subp_params"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subp_spec_f_subp_returns = foreign ~from:c_lib
    "ada_subp_spec_f_subp_returns"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let synthetic_binary_spec_f_left_param = foreign ~from:c_lib
    "ada_synthetic_binary_spec_f_left_param"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let synthetic_binary_spec_f_right_param = foreign ~from:c_lib
    "ada_synthetic_binary_spec_f_right_param"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let synthetic_binary_spec_f_return_type_expr = foreign ~from:c_lib
    "ada_synthetic_binary_spec_f_return_type_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let synthetic_unary_spec_f_right_param = foreign ~from:c_lib
    "ada_synthetic_unary_spec_f_right_param"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let synthetic_unary_spec_f_return_type_expr = foreign ~from:c_lib
    "ada_synthetic_unary_spec_f_return_type_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let component_list_f_components = foreign ~from:c_lib
    "ada_component_list_f_components"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let component_list_f_variant_part = foreign ~from:c_lib
    "ada_component_list_f_variant_part"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let known_discriminant_part_f_discr_specs = foreign ~from:c_lib
    "ada_known_discriminant_part_f_discr_specs"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_completion_formal_params_f_params = foreign ~from:c_lib
    "ada_entry_completion_formal_params_f_params"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_formal_part_f_decls = foreign ~from:c_lib
    "ada_generic_formal_part_f_decls"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_record_def_f_components = foreign ~from:c_lib
    "ada_base_record_def_f_components"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_assoc_p_get_params = foreign ~from:c_lib
    "ada_basic_assoc_p_get_params"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let aggregate_assoc_f_designators = foreign ~from:c_lib
    "ada_aggregate_assoc_f_designators"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let aggregate_assoc_f_r_expr = foreign ~from:c_lib
    "ada_aggregate_assoc_f_r_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let composite_constraint_assoc_f_ids = foreign ~from:c_lib
    "ada_composite_constraint_assoc_f_ids"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let composite_constraint_assoc_f_constraint_expr = foreign ~from:c_lib
    "ada_composite_constraint_assoc_f_constraint_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let iterated_assoc_f_spec = foreign ~from:c_lib
    "ada_iterated_assoc_f_spec"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let iterated_assoc_f_r_expr = foreign ~from:c_lib
    "ada_iterated_assoc_f_r_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let param_assoc_f_designator = foreign ~from:c_lib
    "ada_param_assoc_f_designator"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let param_assoc_f_r_expr = foreign ~from:c_lib
    "ada_param_assoc_f_r_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_is_formal = foreign ~from:c_lib
    "ada_basic_decl_p_is_formal"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let basic_decl_p_doc_annotations = foreign ~from:c_lib
    "ada_basic_decl_p_doc_annotations"
    (ptr EntityStruct.c_type
    @-> ptr DocAnnotationArrayStruct.c_type
    @-> raisable int)

  let basic_decl_p_doc = foreign ~from:c_lib
    "ada_basic_decl_p_doc"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let basic_decl_p_previous_part_for_decl = foreign ~from:c_lib
    "ada_basic_decl_p_previous_part_for_decl"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_canonical_part = foreign ~from:c_lib
    "ada_basic_decl_p_canonical_part"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_all_parts = foreign ~from:c_lib
    "ada_basic_decl_p_all_parts"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let basic_decl_p_is_static_decl = foreign ~from:c_lib
    "ada_basic_decl_p_is_static_decl"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr bool
    @-> raisable int)

  let basic_decl_f_aspects = foreign ~from:c_lib
    "ada_basic_decl_f_aspects"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_get_aspect_assoc = foreign ~from:c_lib
    "ada_basic_decl_p_get_aspect_assoc"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_get_aspect_spec_expr = foreign ~from:c_lib
    "ada_basic_decl_p_get_aspect_spec_expr"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_get_aspect = foreign ~from:c_lib
    "ada_basic_decl_p_get_aspect"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
        
    @-> bool
    @-> ptr AspectStruct.c_type
    @-> raisable int)

  let basic_decl_p_has_aspect = foreign ~from:c_lib
    "ada_basic_decl_p_has_aspect"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
        
    @-> bool
    @-> ptr bool
    @-> raisable int)

  let basic_decl_p_get_pragma = foreign ~from:c_lib
    "ada_basic_decl_p_get_pragma"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_get_representation_clause = foreign ~from:c_lib
    "ada_basic_decl_p_get_representation_clause"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_get_at_clause = foreign ~from:c_lib
    "ada_basic_decl_p_get_at_clause"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_is_imported = foreign ~from:c_lib
    "ada_basic_decl_p_is_imported"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let basic_decl_p_is_ghost_code = foreign ~from:c_lib
    "ada_basic_decl_p_is_ghost_code"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let basic_decl_p_is_compilation_unit_root = foreign ~from:c_lib
    "ada_basic_decl_p_is_compilation_unit_root"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let basic_decl_p_is_visible = foreign ~from:c_lib
    "ada_basic_decl_p_is_visible"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let basic_decl_p_base_subp_declarations = foreign ~from:c_lib
    "ada_basic_decl_p_base_subp_declarations"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let basic_decl_p_root_subp_declarations = foreign ~from:c_lib
    "ada_basic_decl_p_root_subp_declarations"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let basic_decl_p_find_all_overrides = foreign ~from:c_lib
    "ada_basic_decl_p_find_all_overrides"
    (ptr EntityStruct.c_type
        
    @-> AnalysisUnitArrayStruct.c_type
        
    @-> bool
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let basic_decl_p_defining_names = foreign ~from:c_lib
    "ada_basic_decl_p_defining_names"
    (ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let basic_decl_p_defining_name = foreign ~from:c_lib
    "ada_basic_decl_p_defining_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_type_expression = foreign ~from:c_lib
    "ada_basic_decl_p_type_expression"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_subp_spec_or_null = foreign ~from:c_lib
    "ada_basic_decl_p_subp_spec_or_null"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_is_subprogram = foreign ~from:c_lib
    "ada_basic_decl_p_is_subprogram"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let basic_decl_p_relative_name = foreign ~from:c_lib
    "ada_basic_decl_p_relative_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_relative_name_text = foreign ~from:c_lib
    "ada_basic_decl_p_relative_name_text"
    (ptr EntityStruct.c_type
    @-> ptr Symbol.c_type
    @-> raisable int)

  let basic_decl_p_next_part_for_decl = foreign ~from:c_lib
    "ada_basic_decl_p_next_part_for_decl"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_body_part_for_decl = foreign ~from:c_lib
    "ada_basic_decl_p_body_part_for_decl"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_most_visible_part = foreign ~from:c_lib
    "ada_basic_decl_p_most_visible_part"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_decl_p_fully_qualified_name_array = foreign ~from:c_lib
    "ada_basic_decl_p_fully_qualified_name_array"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr UnboundedTextTypeArrayStruct.c_type
    @-> raisable int)

  let basic_decl_p_fully_qualified_name = foreign ~from:c_lib
    "ada_basic_decl_p_fully_qualified_name"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let basic_decl_p_canonical_fully_qualified_name = foreign ~from:c_lib
    "ada_basic_decl_p_canonical_fully_qualified_name"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let basic_decl_p_unique_identifying_name = foreign ~from:c_lib
    "ada_basic_decl_p_unique_identifying_name"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let basic_decl_p_is_constant_object = foreign ~from:c_lib
    "ada_basic_decl_p_is_constant_object"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let abstract_state_decl_f_name = foreign ~from:c_lib
    "ada_abstract_state_decl_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let anonymous_expr_decl_f_expr = foreign ~from:c_lib
    "ada_anonymous_expr_decl_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let anonymous_expr_decl_p_get_formal = foreign ~from:c_lib
    "ada_anonymous_expr_decl_p_get_formal"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_formal_param_decl_p_formal_type = foreign ~from:c_lib
    "ada_base_formal_param_decl_p_formal_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let component_decl_f_ids = foreign ~from:c_lib
    "ada_component_decl_f_ids"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let component_decl_f_component_def = foreign ~from:c_lib
    "ada_component_decl_f_component_def"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let component_decl_f_default_expr = foreign ~from:c_lib
    "ada_component_decl_f_default_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let discriminant_spec_f_ids = foreign ~from:c_lib
    "ada_discriminant_spec_f_ids"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let discriminant_spec_f_type_expr = foreign ~from:c_lib
    "ada_discriminant_spec_f_type_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let discriminant_spec_f_default_expr = foreign ~from:c_lib
    "ada_discriminant_spec_f_default_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_formal_f_decl = foreign ~from:c_lib
    "ada_generic_formal_f_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let param_spec_f_ids = foreign ~from:c_lib
    "ada_param_spec_f_ids"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let param_spec_f_has_aliased = foreign ~from:c_lib
    "ada_param_spec_f_has_aliased"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let param_spec_f_mode = foreign ~from:c_lib
    "ada_param_spec_f_mode"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let param_spec_f_type_expr = foreign ~from:c_lib
    "ada_param_spec_f_type_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let param_spec_f_default_expr = foreign ~from:c_lib
    "ada_param_spec_f_default_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let synthetic_formal_param_decl_f_param_type = foreign ~from:c_lib
    "ada_synthetic_formal_param_decl_f_param_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_package_decl_f_package_name = foreign ~from:c_lib
    "ada_base_package_decl_f_package_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_package_decl_f_public_part = foreign ~from:c_lib
    "ada_base_package_decl_f_public_part"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_package_decl_f_private_part = foreign ~from:c_lib
    "ada_base_package_decl_f_private_part"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_package_decl_f_end_name = foreign ~from:c_lib
    "ada_base_package_decl_f_end_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_package_decl_p_body_part = foreign ~from:c_lib
    "ada_base_package_decl_p_body_part"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_f_name = foreign ~from:c_lib
    "ada_base_type_decl_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_base_subtype = foreign ~from:c_lib
    "ada_base_type_decl_p_base_subtype"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_private_completion = foreign ~from:c_lib
    "ada_base_type_decl_p_private_completion"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_is_inherited_primitive = foreign ~from:c_lib
    "ada_base_type_decl_p_is_inherited_primitive"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_get_record_representation_clause = foreign ~from:c_lib
    "ada_base_type_decl_p_get_record_representation_clause"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_get_enum_representation_clause = foreign ~from:c_lib
    "ada_base_type_decl_p_get_enum_representation_clause"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_is_record_type = foreign ~from:c_lib
    "ada_base_type_decl_p_is_record_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_is_array_type = foreign ~from:c_lib
    "ada_base_type_decl_p_is_array_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_find_derived_types = foreign ~from:c_lib
    "ada_base_type_decl_p_find_derived_types"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let base_type_decl_p_is_real_type = foreign ~from:c_lib
    "ada_base_type_decl_p_is_real_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_is_float_type = foreign ~from:c_lib
    "ada_base_type_decl_p_is_float_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_is_fixed_point = foreign ~from:c_lib
    "ada_base_type_decl_p_is_fixed_point"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_is_scalar_type = foreign ~from:c_lib
    "ada_base_type_decl_p_is_scalar_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_is_enum_type = foreign ~from:c_lib
    "ada_base_type_decl_p_is_enum_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_is_access_type = foreign ~from:c_lib
    "ada_base_type_decl_p_is_access_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_specific_type = foreign ~from:c_lib
    "ada_base_type_decl_p_specific_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_is_char_type = foreign ~from:c_lib
    "ada_base_type_decl_p_is_char_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_classwide_type = foreign ~from:c_lib
    "ada_base_type_decl_p_classwide_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_discrete_range = foreign ~from:c_lib
    "ada_base_type_decl_p_discrete_range"
    (ptr EntityStruct.c_type
    @-> ptr DiscreteRangeStruct.c_type
    @-> raisable int)

  let base_type_decl_p_is_discrete_type = foreign ~from:c_lib
    "ada_base_type_decl_p_is_discrete_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_is_int_type = foreign ~from:c_lib
    "ada_base_type_decl_p_is_int_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_accessed_type = foreign ~from:c_lib
    "ada_base_type_decl_p_accessed_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_is_tagged_type = foreign ~from:c_lib
    "ada_base_type_decl_p_is_tagged_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_base_type = foreign ~from:c_lib
    "ada_base_type_decl_p_base_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_base_types = foreign ~from:c_lib
    "ada_base_type_decl_p_base_types"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let base_type_decl_p_find_all_derived_types = foreign ~from:c_lib
    "ada_base_type_decl_p_find_all_derived_types"
    (ptr EntityStruct.c_type
        
    @-> AnalysisUnitArrayStruct.c_type
        
    @-> bool
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let base_type_decl_p_comp_type = foreign ~from:c_lib
    "ada_base_type_decl_p_comp_type"
    (ptr EntityStruct.c_type
        
    @-> bool
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_index_type = foreign ~from:c_lib
    "ada_base_type_decl_p_index_type"
    (ptr EntityStruct.c_type
        
    @-> int
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_is_derived_type = foreign ~from:c_lib
    "ada_base_type_decl_p_is_derived_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_is_interface_type = foreign ~from:c_lib
    "ada_base_type_decl_p_is_interface_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_matching_type = foreign ~from:c_lib
    "ada_base_type_decl_p_matching_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_canonical_type = foreign ~from:c_lib
    "ada_base_type_decl_p_canonical_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_previous_part = foreign ~from:c_lib
    "ada_base_type_decl_p_previous_part"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_next_part = foreign ~from:c_lib
    "ada_base_type_decl_p_next_part"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_full_view = foreign ~from:c_lib
    "ada_base_type_decl_p_full_view"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_is_definite_subtype = foreign ~from:c_lib
    "ada_base_type_decl_p_is_definite_subtype"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_is_private = foreign ~from:c_lib
    "ada_base_type_decl_p_is_private"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_type_decl_p_discriminants_list = foreign ~from:c_lib
    "ada_base_type_decl_p_discriminants_list"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let base_type_decl_p_root_type = foreign ~from:c_lib
    "ada_base_type_decl_p_root_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_type_decl_p_shapes = foreign ~from:c_lib
    "ada_base_type_decl_p_shapes"
    (ptr EntityStruct.c_type
        
    @-> bool
        
    @-> ptr EntityStruct.c_type
    @-> ptr ShapeArrayStruct.c_type
    @-> raisable int)

  let base_subtype_decl_p_get_type = foreign ~from:c_lib
    "ada_base_subtype_decl_p_get_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subtype_decl_f_subtype = foreign ~from:c_lib
    "ada_subtype_decl_f_subtype"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let incomplete_type_decl_f_discriminants = foreign ~from:c_lib
    "ada_incomplete_type_decl_f_discriminants"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let incomplete_formal_type_decl_f_is_tagged = foreign ~from:c_lib
    "ada_incomplete_formal_type_decl_f_is_tagged"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let incomplete_formal_type_decl_f_default_type = foreign ~from:c_lib
    "ada_incomplete_formal_type_decl_f_default_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let incomplete_tagged_type_decl_f_has_abstract = foreign ~from:c_lib
    "ada_incomplete_tagged_type_decl_f_has_abstract"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let protected_type_decl_f_discriminants = foreign ~from:c_lib
    "ada_protected_type_decl_f_discriminants"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let protected_type_decl_f_interfaces = foreign ~from:c_lib
    "ada_protected_type_decl_f_interfaces"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let protected_type_decl_f_definition = foreign ~from:c_lib
    "ada_protected_type_decl_f_definition"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let task_type_decl_f_discriminants = foreign ~from:c_lib
    "ada_task_type_decl_f_discriminants"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let task_type_decl_f_definition = foreign ~from:c_lib
    "ada_task_type_decl_f_definition"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_decl_f_discriminants = foreign ~from:c_lib
    "ada_type_decl_f_discriminants"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_decl_f_type_def = foreign ~from:c_lib
    "ada_type_decl_f_type_def"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_decl_p_get_primitives = foreign ~from:c_lib
    "ada_type_decl_p_get_primitives"
    (ptr EntityStruct.c_type
        
    @-> bool
        
    @-> bool
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let formal_type_decl_f_default_type = foreign ~from:c_lib
    "ada_formal_type_decl_f_default_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let basic_subp_decl_p_has_spark_mode_on = foreign ~from:c_lib
    "ada_basic_subp_decl_p_has_spark_mode_on"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let basic_subp_decl_p_subp_decl_spec = foreign ~from:c_lib
    "ada_basic_subp_decl_p_subp_decl_spec"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let classic_subp_decl_f_overriding = foreign ~from:c_lib
    "ada_classic_subp_decl_f_overriding"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let classic_subp_decl_f_subp_spec = foreign ~from:c_lib
    "ada_classic_subp_decl_f_subp_spec"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let classic_subp_decl_p_body_part = foreign ~from:c_lib
    "ada_classic_subp_decl_p_body_part"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let formal_subp_decl_f_default_expr = foreign ~from:c_lib
    "ada_formal_subp_decl_f_default_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_decl_f_overriding = foreign ~from:c_lib
    "ada_entry_decl_f_overriding"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_decl_f_spec = foreign ~from:c_lib
    "ada_entry_decl_f_spec"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_decl_p_body_part = foreign ~from:c_lib
    "ada_entry_decl_p_body_part"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_decl_p_accept_stmts = foreign ~from:c_lib
    "ada_entry_decl_p_accept_stmts"
    (ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let enum_literal_decl_f_name = foreign ~from:c_lib
    "ada_enum_literal_decl_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let enum_literal_decl_p_enum_type = foreign ~from:c_lib
    "ada_enum_literal_decl_p_enum_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let enum_literal_decl_p_enum_rep = foreign ~from:c_lib
    "ada_enum_literal_decl_p_enum_rep"
    (ptr EntityStruct.c_type
    @-> ptr BigInteger.c_type
    @-> raisable int)

  let synthetic_char_enum_lit_p_expr = foreign ~from:c_lib
    "ada_synthetic_char_enum_lit_p_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_subp_internal_f_subp_spec = foreign ~from:c_lib
    "ada_generic_subp_internal_f_subp_spec"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let synthetic_subp_decl_f_spec = foreign ~from:c_lib
    "ada_synthetic_subp_decl_f_spec"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let body_node_p_previous_part = foreign ~from:c_lib
    "ada_body_node_p_previous_part"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let body_node_p_decl_part = foreign ~from:c_lib
    "ada_body_node_p_decl_part"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let body_node_p_subunit_root = foreign ~from:c_lib
    "ada_body_node_p_subunit_root"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let accept_stmt_body_f_name = foreign ~from:c_lib
    "ada_accept_stmt_body_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_subp_body_f_overriding = foreign ~from:c_lib
    "ada_base_subp_body_f_overriding"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_subp_body_f_subp_spec = foreign ~from:c_lib
    "ada_base_subp_body_f_subp_spec"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_subp_body_p_is_subject_to_proof = foreign ~from:c_lib
    "ada_base_subp_body_p_is_subject_to_proof"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let base_subp_body_p_has_spark_mode_on = foreign ~from:c_lib
    "ada_base_subp_body_p_has_spark_mode_on"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let expr_function_f_expr = foreign ~from:c_lib
    "ada_expr_function_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subp_body_f_decls = foreign ~from:c_lib
    "ada_subp_body_f_decls"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subp_body_f_stmts = foreign ~from:c_lib
    "ada_subp_body_f_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subp_body_f_end_name = foreign ~from:c_lib
    "ada_subp_body_f_end_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subp_renaming_decl_f_renames = foreign ~from:c_lib
    "ada_subp_renaming_decl_f_renames"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let body_stub_p_syntactic_fully_qualified_name = foreign ~from:c_lib
    "ada_body_stub_p_syntactic_fully_qualified_name"
    (ptr EntityStruct.c_type
    @-> ptr UnboundedTextTypeArrayStruct.c_type
    @-> raisable int)

  let package_body_stub_f_name = foreign ~from:c_lib
    "ada_package_body_stub_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let protected_body_stub_f_name = foreign ~from:c_lib
    "ada_protected_body_stub_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subp_body_stub_f_overriding = foreign ~from:c_lib
    "ada_subp_body_stub_f_overriding"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subp_body_stub_f_subp_spec = foreign ~from:c_lib
    "ada_subp_body_stub_f_subp_spec"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let task_body_stub_f_name = foreign ~from:c_lib
    "ada_task_body_stub_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_body_f_entry_name = foreign ~from:c_lib
    "ada_entry_body_f_entry_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_body_f_index_spec = foreign ~from:c_lib
    "ada_entry_body_f_index_spec"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_body_f_params = foreign ~from:c_lib
    "ada_entry_body_f_params"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_body_f_barrier = foreign ~from:c_lib
    "ada_entry_body_f_barrier"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_body_f_decls = foreign ~from:c_lib
    "ada_entry_body_f_decls"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_body_f_stmts = foreign ~from:c_lib
    "ada_entry_body_f_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_body_f_end_name = foreign ~from:c_lib
    "ada_entry_body_f_end_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let package_body_f_package_name = foreign ~from:c_lib
    "ada_package_body_f_package_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let package_body_f_decls = foreign ~from:c_lib
    "ada_package_body_f_decls"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let package_body_f_stmts = foreign ~from:c_lib
    "ada_package_body_f_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let package_body_f_end_name = foreign ~from:c_lib
    "ada_package_body_f_end_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let protected_body_f_name = foreign ~from:c_lib
    "ada_protected_body_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let protected_body_f_decls = foreign ~from:c_lib
    "ada_protected_body_f_decls"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let protected_body_f_end_name = foreign ~from:c_lib
    "ada_protected_body_f_end_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let task_body_f_name = foreign ~from:c_lib
    "ada_task_body_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let task_body_f_decls = foreign ~from:c_lib
    "ada_task_body_f_decls"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let task_body_f_stmts = foreign ~from:c_lib
    "ada_task_body_f_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let task_body_f_end_name = foreign ~from:c_lib
    "ada_task_body_f_end_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_index_spec_f_id = foreign ~from:c_lib
    "ada_entry_index_spec_f_id"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let entry_index_spec_f_subtype = foreign ~from:c_lib
    "ada_entry_index_spec_f_subtype"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let exception_decl_f_ids = foreign ~from:c_lib
    "ada_exception_decl_f_ids"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let exception_decl_f_renames = foreign ~from:c_lib
    "ada_exception_decl_f_renames"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let exception_handler_f_exception_name = foreign ~from:c_lib
    "ada_exception_handler_f_exception_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let exception_handler_f_handled_exceptions = foreign ~from:c_lib
    "ada_exception_handler_f_handled_exceptions"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let exception_handler_f_stmts = foreign ~from:c_lib
    "ada_exception_handler_f_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let for_loop_var_decl_f_id = foreign ~from:c_lib
    "ada_for_loop_var_decl_f_id"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let for_loop_var_decl_f_id_type = foreign ~from:c_lib
    "ada_for_loop_var_decl_f_id_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_decl_f_formal_part = foreign ~from:c_lib
    "ada_generic_decl_f_formal_part"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_package_decl_f_package_decl = foreign ~from:c_lib
    "ada_generic_package_decl_f_package_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_package_decl_p_body_part = foreign ~from:c_lib
    "ada_generic_package_decl_p_body_part"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_subp_decl_f_subp_decl = foreign ~from:c_lib
    "ada_generic_subp_decl_f_subp_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_subp_decl_p_body_part = foreign ~from:c_lib
    "ada_generic_subp_decl_p_body_part"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_instantiation_p_designated_generic_decl = foreign ~from:c_lib
    "ada_generic_instantiation_p_designated_generic_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_instantiation_p_inst_params = foreign ~from:c_lib
    "ada_generic_instantiation_p_inst_params"
    (ptr EntityStruct.c_type
    @-> ptr ParamActualArrayStruct.c_type
    @-> raisable int)

  let generic_package_instantiation_f_name = foreign ~from:c_lib
    "ada_generic_package_instantiation_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_package_instantiation_f_generic_pkg_name = foreign ~from:c_lib
    "ada_generic_package_instantiation_f_generic_pkg_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_package_instantiation_f_params = foreign ~from:c_lib
    "ada_generic_package_instantiation_f_params"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_subp_instantiation_f_overriding = foreign ~from:c_lib
    "ada_generic_subp_instantiation_f_overriding"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_subp_instantiation_f_kind = foreign ~from:c_lib
    "ada_generic_subp_instantiation_f_kind"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_subp_instantiation_f_subp_name = foreign ~from:c_lib
    "ada_generic_subp_instantiation_f_subp_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_subp_instantiation_f_generic_subp_name = foreign ~from:c_lib
    "ada_generic_subp_instantiation_f_generic_subp_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_subp_instantiation_f_params = foreign ~from:c_lib
    "ada_generic_subp_instantiation_f_params"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_subp_instantiation_p_designated_subp = foreign ~from:c_lib
    "ada_generic_subp_instantiation_p_designated_subp"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_package_renaming_decl_f_name = foreign ~from:c_lib
    "ada_generic_package_renaming_decl_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_package_renaming_decl_f_renames = foreign ~from:c_lib
    "ada_generic_package_renaming_decl_f_renames"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_subp_renaming_decl_f_kind = foreign ~from:c_lib
    "ada_generic_subp_renaming_decl_f_kind"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_subp_renaming_decl_f_name = foreign ~from:c_lib
    "ada_generic_subp_renaming_decl_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_subp_renaming_decl_f_renames = foreign ~from:c_lib
    "ada_generic_subp_renaming_decl_f_renames"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let label_decl_f_name = foreign ~from:c_lib
    "ada_label_decl_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let named_stmt_decl_f_name = foreign ~from:c_lib
    "ada_named_stmt_decl_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let number_decl_f_ids = foreign ~from:c_lib
    "ada_number_decl_f_ids"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let number_decl_f_expr = foreign ~from:c_lib
    "ada_number_decl_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let object_decl_f_ids = foreign ~from:c_lib
    "ada_object_decl_f_ids"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let object_decl_f_has_aliased = foreign ~from:c_lib
    "ada_object_decl_f_has_aliased"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let object_decl_f_has_constant = foreign ~from:c_lib
    "ada_object_decl_f_has_constant"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let object_decl_f_mode = foreign ~from:c_lib
    "ada_object_decl_f_mode"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let object_decl_f_type_expr = foreign ~from:c_lib
    "ada_object_decl_f_type_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let object_decl_f_default_expr = foreign ~from:c_lib
    "ada_object_decl_f_default_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let object_decl_f_renaming_clause = foreign ~from:c_lib
    "ada_object_decl_f_renaming_clause"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let object_decl_p_private_part_decl = foreign ~from:c_lib
    "ada_object_decl_p_private_part_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let object_decl_p_public_part_decl = foreign ~from:c_lib
    "ada_object_decl_p_public_part_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let package_renaming_decl_f_name = foreign ~from:c_lib
    "ada_package_renaming_decl_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let package_renaming_decl_f_renames = foreign ~from:c_lib
    "ada_package_renaming_decl_f_renames"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let package_renaming_decl_p_renamed_package = foreign ~from:c_lib
    "ada_package_renaming_decl_p_renamed_package"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let package_renaming_decl_p_final_renamed_package = foreign ~from:c_lib
    "ada_package_renaming_decl_p_final_renamed_package"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let single_protected_decl_f_name = foreign ~from:c_lib
    "ada_single_protected_decl_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let single_protected_decl_f_interfaces = foreign ~from:c_lib
    "ada_single_protected_decl_f_interfaces"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let single_protected_decl_f_definition = foreign ~from:c_lib
    "ada_single_protected_decl_f_definition"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let single_task_decl_f_task_type = foreign ~from:c_lib
    "ada_single_task_decl_f_task_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let case_stmt_alternative_f_choices = foreign ~from:c_lib
    "ada_case_stmt_alternative_f_choices"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let case_stmt_alternative_f_stmts = foreign ~from:c_lib
    "ada_case_stmt_alternative_f_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let compilation_unit_f_prelude = foreign ~from:c_lib
    "ada_compilation_unit_f_prelude"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let compilation_unit_f_body = foreign ~from:c_lib
    "ada_compilation_unit_f_body"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let compilation_unit_f_pragmas = foreign ~from:c_lib
    "ada_compilation_unit_f_pragmas"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let compilation_unit_p_syntactic_fully_qualified_name = foreign ~from:c_lib
    "ada_compilation_unit_p_syntactic_fully_qualified_name"
    (ptr EntityStruct.c_type
    @-> ptr UnboundedTextTypeArrayStruct.c_type
    @-> raisable int)

  let compilation_unit_p_unit_kind = foreign ~from:c_lib
    "ada_compilation_unit_p_unit_kind"
    (ptr EntityStruct.c_type
    @-> ptr AnalysisUnitKind.c_type
    @-> raisable int)

  let compilation_unit_p_withed_units = foreign ~from:c_lib
    "ada_compilation_unit_p_withed_units"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let compilation_unit_p_imported_units = foreign ~from:c_lib
    "ada_compilation_unit_p_imported_units"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let compilation_unit_p_unit_dependencies = foreign ~from:c_lib
    "ada_compilation_unit_p_unit_dependencies"
    (ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let compilation_unit_p_decl = foreign ~from:c_lib
    "ada_compilation_unit_p_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let compilation_unit_p_is_preelaborable = foreign ~from:c_lib
    "ada_compilation_unit_p_is_preelaborable"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr bool
    @-> raisable int)

  let compilation_unit_p_other_part = foreign ~from:c_lib
    "ada_compilation_unit_p_other_part"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let compilation_unit_p_has_restriction = foreign ~from:c_lib
    "ada_compilation_unit_p_has_restriction"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
    @-> ptr bool
    @-> raisable int)

  let compilation_unit_p_all_config_pragmas = foreign ~from:c_lib
    "ada_compilation_unit_p_all_config_pragmas"
    (ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let compilation_unit_p_config_pragmas = foreign ~from:c_lib
    "ada_compilation_unit_p_config_pragmas"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let component_clause_f_id = foreign ~from:c_lib
    "ada_component_clause_f_id"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let component_clause_f_position = foreign ~from:c_lib
    "ada_component_clause_f_position"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let component_clause_f_range = foreign ~from:c_lib
    "ada_component_clause_f_range"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let component_def_f_has_aliased = foreign ~from:c_lib
    "ada_component_def_f_has_aliased"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let component_def_f_has_constant = foreign ~from:c_lib
    "ada_component_def_f_has_constant"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let component_def_f_type_expr = foreign ~from:c_lib
    "ada_component_def_f_type_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let constant_node_p_as_bool = foreign ~from:c_lib
    "ada_constant_node_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let composite_constraint_f_constraints = foreign ~from:c_lib
    "ada_composite_constraint_f_constraints"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let composite_constraint_p_is_index_constraint = foreign ~from:c_lib
    "ada_composite_constraint_p_is_index_constraint"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let composite_constraint_p_is_discriminant_constraint = foreign ~from:c_lib
    "ada_composite_constraint_p_is_discriminant_constraint"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let delta_constraint_f_digits = foreign ~from:c_lib
    "ada_delta_constraint_f_digits"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let delta_constraint_f_range = foreign ~from:c_lib
    "ada_delta_constraint_f_range"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let digits_constraint_f_digits = foreign ~from:c_lib
    "ada_digits_constraint_f_digits"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let digits_constraint_f_range = foreign ~from:c_lib
    "ada_digits_constraint_f_range"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let range_constraint_f_range = foreign ~from:c_lib
    "ada_range_constraint_f_range"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let declarative_part_f_decls = foreign ~from:c_lib
    "ada_declarative_part_f_decls"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let elsif_expr_part_f_cond_expr = foreign ~from:c_lib
    "ada_elsif_expr_part_f_cond_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let elsif_expr_part_f_then_expr = foreign ~from:c_lib
    "ada_elsif_expr_part_f_then_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let elsif_stmt_part_f_cond_expr = foreign ~from:c_lib
    "ada_elsif_stmt_part_f_cond_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let elsif_stmt_part_f_stmts = foreign ~from:c_lib
    "ada_elsif_stmt_part_f_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let expr_p_expression_type = foreign ~from:c_lib
    "ada_expr_p_expression_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let expr_p_expected_expression_type = foreign ~from:c_lib
    "ada_expr_p_expected_expression_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let expr_p_is_dynamically_tagged = foreign ~from:c_lib
    "ada_expr_p_is_dynamically_tagged"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr bool
    @-> raisable int)

  let expr_p_is_dispatching_call = foreign ~from:c_lib
    "ada_expr_p_is_dispatching_call"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr bool
    @-> raisable int)

  let expr_p_is_static_expr = foreign ~from:c_lib
    "ada_expr_p_is_static_expr"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr bool
    @-> raisable int)

  let expr_p_first_corresponding_decl = foreign ~from:c_lib
    "ada_expr_p_first_corresponding_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let expr_p_eval_as_int = foreign ~from:c_lib
    "ada_expr_p_eval_as_int"
    (ptr EntityStruct.c_type
    @-> ptr BigInteger.c_type
    @-> raisable int)

  let expr_p_eval_as_int_in_env = foreign ~from:c_lib
    "ada_expr_p_eval_as_int_in_env"
    (ptr EntityStruct.c_type
        
    @-> SubstitutionArrayStruct.c_type
    @-> ptr BigInteger.c_type
    @-> raisable int)

  let expr_p_eval_as_string = foreign ~from:c_lib
    "ada_expr_p_eval_as_string"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let expr_p_eval_as_string_in_env = foreign ~from:c_lib
    "ada_expr_p_eval_as_string_in_env"
    (ptr EntityStruct.c_type
        
    @-> SubstitutionArrayStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let expr_p_matching_nodes = foreign ~from:c_lib
    "ada_expr_p_matching_nodes"
    (ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let abstract_state_decl_expr_f_state_decl = foreign ~from:c_lib
    "ada_abstract_state_decl_expr_f_state_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let allocator_f_subpool = foreign ~from:c_lib
    "ada_allocator_f_subpool"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let allocator_f_type_or_expr = foreign ~from:c_lib
    "ada_allocator_f_type_or_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let allocator_p_get_allocated_type = foreign ~from:c_lib
    "ada_allocator_p_get_allocated_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_aggregate_f_ancestor_expr = foreign ~from:c_lib
    "ada_base_aggregate_f_ancestor_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_aggregate_f_assocs = foreign ~from:c_lib
    "ada_base_aggregate_f_assocs"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_aggregate_p_aggregate_params = foreign ~from:c_lib
    "ada_base_aggregate_p_aggregate_params"
    (ptr EntityStruct.c_type
    @-> ptr ParamActualArrayStruct.c_type
    @-> raisable int)

  let base_aggregate_p_is_subaggregate = foreign ~from:c_lib
    "ada_base_aggregate_p_is_subaggregate"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let bin_op_f_left = foreign ~from:c_lib
    "ada_bin_op_f_left"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let bin_op_f_op = foreign ~from:c_lib
    "ada_bin_op_f_op"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let bin_op_f_right = foreign ~from:c_lib
    "ada_bin_op_f_right"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let case_expr_alternative_f_choices = foreign ~from:c_lib
    "ada_case_expr_alternative_f_choices"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let case_expr_alternative_f_expr = foreign ~from:c_lib
    "ada_case_expr_alternative_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let concat_op_f_first_operand = foreign ~from:c_lib
    "ada_concat_op_f_first_operand"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let concat_op_f_other_operands = foreign ~from:c_lib
    "ada_concat_op_f_other_operands"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let concat_op_p_operands = foreign ~from:c_lib
    "ada_concat_op_p_operands"
    (ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let concat_operand_f_operator = foreign ~from:c_lib
    "ada_concat_operand_f_operator"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let concat_operand_f_operand = foreign ~from:c_lib
    "ada_concat_operand_f_operand"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let cond_expr_p_dependent_exprs = foreign ~from:c_lib
    "ada_cond_expr_p_dependent_exprs"
    (ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let case_expr_f_expr = foreign ~from:c_lib
    "ada_case_expr_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let case_expr_f_cases = foreign ~from:c_lib
    "ada_case_expr_f_cases"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let if_expr_f_cond_expr = foreign ~from:c_lib
    "ada_if_expr_f_cond_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let if_expr_f_then_expr = foreign ~from:c_lib
    "ada_if_expr_f_then_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let if_expr_f_alternatives = foreign ~from:c_lib
    "ada_if_expr_f_alternatives"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let if_expr_f_else_expr = foreign ~from:c_lib
    "ada_if_expr_f_else_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let contract_cases_f_contract_cases = foreign ~from:c_lib
    "ada_contract_cases_f_contract_cases"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decl_expr_f_decls = foreign ~from:c_lib
    "ada_decl_expr_f_decls"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decl_expr_f_expr = foreign ~from:c_lib
    "ada_decl_expr_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let membership_expr_f_expr = foreign ~from:c_lib
    "ada_membership_expr_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let membership_expr_f_op = foreign ~from:c_lib
    "ada_membership_expr_f_op"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let membership_expr_f_membership_exprs = foreign ~from:c_lib
    "ada_membership_expr_f_membership_exprs"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let name_p_enclosing_defining_name = foreign ~from:c_lib
    "ada_name_p_enclosing_defining_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let name_p_is_defining = foreign ~from:c_lib
    "ada_name_p_is_defining"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let name_p_name_is = foreign ~from:c_lib
    "ada_name_p_name_is"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
    @-> ptr bool
    @-> raisable int)

  let name_p_is_direct_call = foreign ~from:c_lib
    "ada_name_p_is_direct_call"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let name_p_is_access_call = foreign ~from:c_lib
    "ada_name_p_is_access_call"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let name_p_is_call = foreign ~from:c_lib
    "ada_name_p_is_call"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let name_p_is_dot_call = foreign ~from:c_lib
    "ada_name_p_is_dot_call"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr bool
    @-> raisable int)

  let name_p_failsafe_referenced_def_name = foreign ~from:c_lib
    "ada_name_p_failsafe_referenced_def_name"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr RefdDefStruct.c_type
    @-> raisable int)

  let name_p_referenced_defining_name = foreign ~from:c_lib
    "ada_name_p_referenced_defining_name"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let name_p_all_env_elements = foreign ~from:c_lib
    "ada_name_p_all_env_elements"
    (ptr EntityStruct.c_type
        
    @-> bool
        
    @-> ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let name_p_called_subp_spec = foreign ~from:c_lib
    "ada_name_p_called_subp_spec"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let name_p_referenced_decl = foreign ~from:c_lib
    "ada_name_p_referenced_decl"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let name_p_failsafe_referenced_decl = foreign ~from:c_lib
    "ada_name_p_failsafe_referenced_decl"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr RefdDeclStruct.c_type
    @-> raisable int)

  let name_p_name_designated_type = foreign ~from:c_lib
    "ada_name_p_name_designated_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let name_p_is_static_subtype = foreign ~from:c_lib
    "ada_name_p_is_static_subtype"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr bool
    @-> raisable int)

  let name_p_name_matches = foreign ~from:c_lib
    "ada_name_p_name_matches"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let name_p_relative_name = foreign ~from:c_lib
    "ada_name_p_relative_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let name_p_is_operator_name = foreign ~from:c_lib
    "ada_name_p_is_operator_name"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let name_p_is_write_reference = foreign ~from:c_lib
    "ada_name_p_is_write_reference"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr bool
    @-> raisable int)

  let name_p_is_static_call = foreign ~from:c_lib
    "ada_name_p_is_static_call"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr bool
    @-> raisable int)

  let name_p_as_symbol_array = foreign ~from:c_lib
    "ada_name_p_as_symbol_array"
    (ptr EntityStruct.c_type
    @-> ptr UnboundedTextTypeArrayStruct.c_type
    @-> raisable int)

  let name_p_canonical_text = foreign ~from:c_lib
    "ada_name_p_canonical_text"
    (ptr EntityStruct.c_type
    @-> ptr Symbol.c_type
    @-> raisable int)

  let name_p_is_constant = foreign ~from:c_lib
    "ada_name_p_is_constant"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let name_p_call_params = foreign ~from:c_lib
    "ada_name_p_call_params"
    (ptr EntityStruct.c_type
    @-> ptr ParamActualArrayStruct.c_type
    @-> raisable int)

  let attribute_ref_f_prefix = foreign ~from:c_lib
    "ada_attribute_ref_f_prefix"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let attribute_ref_f_attribute = foreign ~from:c_lib
    "ada_attribute_ref_f_attribute"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let attribute_ref_f_args = foreign ~from:c_lib
    "ada_attribute_ref_f_args"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let call_expr_f_name = foreign ~from:c_lib
    "ada_call_expr_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let call_expr_f_suffix = foreign ~from:c_lib
    "ada_call_expr_f_suffix"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let call_expr_p_kind = foreign ~from:c_lib
    "ada_call_expr_p_kind"
    (ptr EntityStruct.c_type
    @-> ptr CallExprKind.c_type
    @-> raisable int)

  let call_expr_p_is_array_slice = foreign ~from:c_lib
    "ada_call_expr_p_is_array_slice"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let defining_name_f_name = foreign ~from:c_lib
    "ada_defining_name_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let defining_name_p_canonical_fully_qualified_name = foreign ~from:c_lib
    "ada_defining_name_p_canonical_fully_qualified_name"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let defining_name_p_unique_identifying_name = foreign ~from:c_lib
    "ada_defining_name_p_unique_identifying_name"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let defining_name_p_fully_qualified_name_array = foreign ~from:c_lib
    "ada_defining_name_p_fully_qualified_name_array"
    (ptr EntityStruct.c_type
    @-> ptr UnboundedTextTypeArrayStruct.c_type
    @-> raisable int)

  let defining_name_p_fully_qualified_name = foreign ~from:c_lib
    "ada_defining_name_p_fully_qualified_name"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let defining_name_p_basic_decl = foreign ~from:c_lib
    "ada_defining_name_p_basic_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let defining_name_p_find_refs = foreign ~from:c_lib
    "ada_defining_name_p_find_refs"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr RefResultArrayStruct.c_type
    @-> raisable int)

  let defining_name_p_find_all_references = foreign ~from:c_lib
    "ada_defining_name_p_find_all_references"
    (ptr EntityStruct.c_type
        
    @-> AnalysisUnitArrayStruct.c_type
        
    @-> bool
        
    @-> bool
    @-> ptr RefResultArrayStruct.c_type
    @-> raisable int)

  let defining_name_p_find_all_calls = foreign ~from:c_lib
    "ada_defining_name_p_find_all_calls"
    (ptr EntityStruct.c_type
        
    @-> AnalysisUnitArrayStruct.c_type
        
    @-> bool
        
    @-> bool
    @-> ptr RefResultArrayStruct.c_type
    @-> raisable int)

  let defining_name_p_next_part = foreign ~from:c_lib
    "ada_defining_name_p_next_part"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let defining_name_p_previous_part = foreign ~from:c_lib
    "ada_defining_name_p_previous_part"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let defining_name_p_canonical_part = foreign ~from:c_lib
    "ada_defining_name_p_canonical_part"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let defining_name_p_most_visible_part = foreign ~from:c_lib
    "ada_defining_name_p_most_visible_part"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let defining_name_p_all_parts = foreign ~from:c_lib
    "ada_defining_name_p_all_parts"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let defining_name_p_get_aspect = foreign ~from:c_lib
    "ada_defining_name_p_get_aspect"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
        
    @-> bool
    @-> ptr AspectStruct.c_type
    @-> raisable int)

  let defining_name_p_has_aspect = foreign ~from:c_lib
    "ada_defining_name_p_has_aspect"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
        
    @-> bool
    @-> ptr bool
    @-> raisable int)

  let defining_name_p_get_pragma = foreign ~from:c_lib
    "ada_defining_name_p_get_pragma"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let defining_name_p_get_representation_clause = foreign ~from:c_lib
    "ada_defining_name_p_get_representation_clause"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let defining_name_p_get_at_clause = foreign ~from:c_lib
    "ada_defining_name_p_get_at_clause"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let defining_name_p_is_imported = foreign ~from:c_lib
    "ada_defining_name_p_is_imported"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let defining_name_p_is_ghost_code = foreign ~from:c_lib
    "ada_defining_name_p_is_ghost_code"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let discrete_subtype_name_f_subtype = foreign ~from:c_lib
    "ada_discrete_subtype_name_f_subtype"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let dotted_name_f_prefix = foreign ~from:c_lib
    "ada_dotted_name_f_prefix"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let dotted_name_f_suffix = foreign ~from:c_lib
    "ada_dotted_name_f_suffix"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let end_name_f_name = foreign ~from:c_lib
    "ada_end_name_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let end_name_p_basic_decl = foreign ~from:c_lib
    "ada_end_name_p_basic_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let explicit_deref_f_prefix = foreign ~from:c_lib
    "ada_explicit_deref_f_prefix"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let qual_expr_f_prefix = foreign ~from:c_lib
    "ada_qual_expr_f_prefix"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let qual_expr_f_suffix = foreign ~from:c_lib
    "ada_qual_expr_f_suffix"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let reduce_attribute_ref_f_prefix = foreign ~from:c_lib
    "ada_reduce_attribute_ref_f_prefix"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let reduce_attribute_ref_f_attribute = foreign ~from:c_lib
    "ada_reduce_attribute_ref_f_attribute"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let reduce_attribute_ref_f_args = foreign ~from:c_lib
    "ada_reduce_attribute_ref_f_args"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let char_literal_p_denoted_value = foreign ~from:c_lib
    "ada_char_literal_p_denoted_value"
    (ptr EntityStruct.c_type
    @-> ptr Character.c_type
    @-> raisable int)

  let string_literal_p_denoted_value = foreign ~from:c_lib
    "ada_string_literal_p_denoted_value"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let int_literal_p_denoted_value = foreign ~from:c_lib
    "ada_int_literal_p_denoted_value"
    (ptr EntityStruct.c_type
    @-> ptr BigInteger.c_type
    @-> raisable int)

  let update_attribute_ref_f_prefix = foreign ~from:c_lib
    "ada_update_attribute_ref_f_prefix"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let update_attribute_ref_f_attribute = foreign ~from:c_lib
    "ada_update_attribute_ref_f_attribute"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let update_attribute_ref_f_values = foreign ~from:c_lib
    "ada_update_attribute_ref_f_values"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let paren_expr_f_expr = foreign ~from:c_lib
    "ada_paren_expr_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let quantified_expr_f_quantifier = foreign ~from:c_lib
    "ada_quantified_expr_f_quantifier"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let quantified_expr_f_loop_spec = foreign ~from:c_lib
    "ada_quantified_expr_f_loop_spec"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let quantified_expr_f_expr = foreign ~from:c_lib
    "ada_quantified_expr_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let raise_expr_f_exception_name = foreign ~from:c_lib
    "ada_raise_expr_f_exception_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let raise_expr_f_error_message = foreign ~from:c_lib
    "ada_raise_expr_f_error_message"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let un_op_f_op = foreign ~from:c_lib
    "ada_un_op_f_op"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let un_op_f_expr = foreign ~from:c_lib
    "ada_un_op_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let handled_stmts_f_stmts = foreign ~from:c_lib
    "ada_handled_stmts_f_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let handled_stmts_f_exceptions = foreign ~from:c_lib
    "ada_handled_stmts_f_exceptions"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let library_item_f_has_private = foreign ~from:c_lib
    "ada_library_item_f_has_private"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let library_item_f_item = foreign ~from:c_lib
    "ada_library_item_f_item"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let limited_node_p_as_bool = foreign ~from:c_lib
    "ada_limited_node_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let for_loop_spec_f_var_decl = foreign ~from:c_lib
    "ada_for_loop_spec_f_var_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let for_loop_spec_f_loop_type = foreign ~from:c_lib
    "ada_for_loop_spec_f_loop_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let for_loop_spec_f_has_reverse = foreign ~from:c_lib
    "ada_for_loop_spec_f_has_reverse"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let for_loop_spec_f_iter_expr = foreign ~from:c_lib
    "ada_for_loop_spec_f_iter_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let for_loop_spec_f_iter_filter = foreign ~from:c_lib
    "ada_for_loop_spec_f_iter_filter"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let while_loop_spec_f_expr = foreign ~from:c_lib
    "ada_while_loop_spec_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let multi_abstract_state_decl_f_decls = foreign ~from:c_lib
    "ada_multi_abstract_state_decl_f_decls"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let not_null_p_as_bool = foreign ~from:c_lib
    "ada_not_null_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let params_f_params = foreign ~from:c_lib
    "ada_params_f_params"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let paren_abstract_state_decl_f_decl = foreign ~from:c_lib
    "ada_paren_abstract_state_decl_f_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let pp_elsif_directive_f_expr = foreign ~from:c_lib
    "ada_pp_elsif_directive_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let pp_elsif_directive_f_then_kw = foreign ~from:c_lib
    "ada_pp_elsif_directive_f_then_kw"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let pp_if_directive_f_expr = foreign ~from:c_lib
    "ada_pp_if_directive_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let pp_if_directive_f_then_kw = foreign ~from:c_lib
    "ada_pp_if_directive_f_then_kw"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let pragma_node_f_id = foreign ~from:c_lib
    "ada_pragma_node_f_id"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let pragma_node_f_args = foreign ~from:c_lib
    "ada_pragma_node_f_args"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let pragma_node_p_is_ghost_code = foreign ~from:c_lib
    "ada_pragma_node_p_is_ghost_code"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let pragma_node_p_associated_entities = foreign ~from:c_lib
    "ada_pragma_node_p_associated_entities"
    (ptr EntityStruct.c_type
    @-> ptr AdaNodeArrayStruct.c_type
    @-> raisable int)

  let private_node_p_as_bool = foreign ~from:c_lib
    "ada_private_node_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let protected_def_f_public_part = foreign ~from:c_lib
    "ada_protected_def_f_public_part"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let protected_def_f_private_part = foreign ~from:c_lib
    "ada_protected_def_f_private_part"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let protected_def_f_end_name = foreign ~from:c_lib
    "ada_protected_def_f_end_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let protected_node_p_as_bool = foreign ~from:c_lib
    "ada_protected_node_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let range_spec_f_range = foreign ~from:c_lib
    "ada_range_spec_f_range"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let renaming_clause_f_renamed_object = foreign ~from:c_lib
    "ada_renaming_clause_f_renamed_object"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let reverse_node_p_as_bool = foreign ~from:c_lib
    "ada_reverse_node_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let select_when_part_f_cond_expr = foreign ~from:c_lib
    "ada_select_when_part_f_cond_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let select_when_part_f_stmts = foreign ~from:c_lib
    "ada_select_when_part_f_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let stmt_p_is_ghost_code = foreign ~from:c_lib
    "ada_stmt_p_is_ghost_code"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let accept_stmt_f_body_decl = foreign ~from:c_lib
    "ada_accept_stmt_f_body_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let accept_stmt_f_entry_index_expr = foreign ~from:c_lib
    "ada_accept_stmt_f_entry_index_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let accept_stmt_f_params = foreign ~from:c_lib
    "ada_accept_stmt_f_params"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let accept_stmt_p_corresponding_entry = foreign ~from:c_lib
    "ada_accept_stmt_p_corresponding_entry"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let accept_stmt_with_stmts_f_stmts = foreign ~from:c_lib
    "ada_accept_stmt_with_stmts_f_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let accept_stmt_with_stmts_f_end_name = foreign ~from:c_lib
    "ada_accept_stmt_with_stmts_f_end_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_loop_stmt_f_spec = foreign ~from:c_lib
    "ada_base_loop_stmt_f_spec"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_loop_stmt_f_stmts = foreign ~from:c_lib
    "ada_base_loop_stmt_f_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_loop_stmt_f_end_name = foreign ~from:c_lib
    "ada_base_loop_stmt_f_end_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let begin_block_f_stmts = foreign ~from:c_lib
    "ada_begin_block_f_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let begin_block_f_end_name = foreign ~from:c_lib
    "ada_begin_block_f_end_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decl_block_f_decls = foreign ~from:c_lib
    "ada_decl_block_f_decls"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decl_block_f_stmts = foreign ~from:c_lib
    "ada_decl_block_f_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decl_block_f_end_name = foreign ~from:c_lib
    "ada_decl_block_f_end_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let case_stmt_f_expr = foreign ~from:c_lib
    "ada_case_stmt_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let case_stmt_f_pragmas = foreign ~from:c_lib
    "ada_case_stmt_f_pragmas"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let case_stmt_f_alternatives = foreign ~from:c_lib
    "ada_case_stmt_f_alternatives"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let extended_return_stmt_f_decl = foreign ~from:c_lib
    "ada_extended_return_stmt_f_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let extended_return_stmt_f_stmts = foreign ~from:c_lib
    "ada_extended_return_stmt_f_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let if_stmt_f_cond_expr = foreign ~from:c_lib
    "ada_if_stmt_f_cond_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let if_stmt_f_then_stmts = foreign ~from:c_lib
    "ada_if_stmt_f_then_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let if_stmt_f_alternatives = foreign ~from:c_lib
    "ada_if_stmt_f_alternatives"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let if_stmt_f_else_stmts = foreign ~from:c_lib
    "ada_if_stmt_f_else_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let named_stmt_f_decl = foreign ~from:c_lib
    "ada_named_stmt_f_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let named_stmt_f_stmt = foreign ~from:c_lib
    "ada_named_stmt_f_stmt"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let select_stmt_f_guards = foreign ~from:c_lib
    "ada_select_stmt_f_guards"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let select_stmt_f_else_stmts = foreign ~from:c_lib
    "ada_select_stmt_f_else_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let select_stmt_f_abort_stmts = foreign ~from:c_lib
    "ada_select_stmt_f_abort_stmts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let abort_stmt_f_names = foreign ~from:c_lib
    "ada_abort_stmt_f_names"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let assign_stmt_f_dest = foreign ~from:c_lib
    "ada_assign_stmt_f_dest"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let assign_stmt_f_expr = foreign ~from:c_lib
    "ada_assign_stmt_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let call_stmt_f_call = foreign ~from:c_lib
    "ada_call_stmt_f_call"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let delay_stmt_f_has_until = foreign ~from:c_lib
    "ada_delay_stmt_f_has_until"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let delay_stmt_f_expr = foreign ~from:c_lib
    "ada_delay_stmt_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let exit_stmt_f_loop_name = foreign ~from:c_lib
    "ada_exit_stmt_f_loop_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let exit_stmt_f_cond_expr = foreign ~from:c_lib
    "ada_exit_stmt_f_cond_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let goto_stmt_f_label_name = foreign ~from:c_lib
    "ada_goto_stmt_f_label_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let label_f_decl = foreign ~from:c_lib
    "ada_label_f_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let raise_stmt_f_exception_name = foreign ~from:c_lib
    "ada_raise_stmt_f_exception_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let raise_stmt_f_error_message = foreign ~from:c_lib
    "ada_raise_stmt_f_error_message"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let requeue_stmt_f_call_name = foreign ~from:c_lib
    "ada_requeue_stmt_f_call_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let requeue_stmt_f_has_abort = foreign ~from:c_lib
    "ada_requeue_stmt_f_has_abort"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let return_stmt_f_return_expr = foreign ~from:c_lib
    "ada_return_stmt_f_return_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subunit_f_name = foreign ~from:c_lib
    "ada_subunit_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subunit_f_body = foreign ~from:c_lib
    "ada_subunit_f_body"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subunit_p_body_root = foreign ~from:c_lib
    "ada_subunit_p_body_root"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let synchronized_node_p_as_bool = foreign ~from:c_lib
    "ada_synchronized_node_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let tagged_node_p_as_bool = foreign ~from:c_lib
    "ada_tagged_node_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let task_def_f_interfaces = foreign ~from:c_lib
    "ada_task_def_f_interfaces"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let task_def_f_public_part = foreign ~from:c_lib
    "ada_task_def_f_public_part"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let task_def_f_private_part = foreign ~from:c_lib
    "ada_task_def_f_private_part"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let task_def_f_end_name = foreign ~from:c_lib
    "ada_task_def_f_end_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let access_def_f_has_not_null = foreign ~from:c_lib
    "ada_access_def_f_has_not_null"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let access_to_subp_def_f_has_protected = foreign ~from:c_lib
    "ada_access_to_subp_def_f_has_protected"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let access_to_subp_def_f_subp_spec = foreign ~from:c_lib
    "ada_access_to_subp_def_f_subp_spec"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let anonymous_type_access_def_f_type_decl = foreign ~from:c_lib
    "ada_anonymous_type_access_def_f_type_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_access_def_f_has_all = foreign ~from:c_lib
    "ada_type_access_def_f_has_all"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_access_def_f_has_constant = foreign ~from:c_lib
    "ada_type_access_def_f_has_constant"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_access_def_f_subtype_indication = foreign ~from:c_lib
    "ada_type_access_def_f_subtype_indication"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let array_type_def_f_indices = foreign ~from:c_lib
    "ada_array_type_def_f_indices"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let array_type_def_f_component_type = foreign ~from:c_lib
    "ada_array_type_def_f_component_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let derived_type_def_f_has_abstract = foreign ~from:c_lib
    "ada_derived_type_def_f_has_abstract"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let derived_type_def_f_has_limited = foreign ~from:c_lib
    "ada_derived_type_def_f_has_limited"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let derived_type_def_f_has_synchronized = foreign ~from:c_lib
    "ada_derived_type_def_f_has_synchronized"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let derived_type_def_f_subtype_indication = foreign ~from:c_lib
    "ada_derived_type_def_f_subtype_indication"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let derived_type_def_f_interfaces = foreign ~from:c_lib
    "ada_derived_type_def_f_interfaces"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let derived_type_def_f_record_extension = foreign ~from:c_lib
    "ada_derived_type_def_f_record_extension"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let derived_type_def_f_has_with_private = foreign ~from:c_lib
    "ada_derived_type_def_f_has_with_private"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let enum_type_def_f_enum_literals = foreign ~from:c_lib
    "ada_enum_type_def_f_enum_literals"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let interface_type_def_f_interface_kind = foreign ~from:c_lib
    "ada_interface_type_def_f_interface_kind"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let interface_type_def_f_interfaces = foreign ~from:c_lib
    "ada_interface_type_def_f_interfaces"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let mod_int_type_def_f_expr = foreign ~from:c_lib
    "ada_mod_int_type_def_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let private_type_def_f_has_abstract = foreign ~from:c_lib
    "ada_private_type_def_f_has_abstract"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let private_type_def_f_has_tagged = foreign ~from:c_lib
    "ada_private_type_def_f_has_tagged"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let private_type_def_f_has_limited = foreign ~from:c_lib
    "ada_private_type_def_f_has_limited"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decimal_fixed_point_def_f_delta = foreign ~from:c_lib
    "ada_decimal_fixed_point_def_f_delta"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decimal_fixed_point_def_f_digits = foreign ~from:c_lib
    "ada_decimal_fixed_point_def_f_digits"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decimal_fixed_point_def_f_range = foreign ~from:c_lib
    "ada_decimal_fixed_point_def_f_range"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let floating_point_def_f_num_digits = foreign ~from:c_lib
    "ada_floating_point_def_f_num_digits"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let floating_point_def_f_range = foreign ~from:c_lib
    "ada_floating_point_def_f_range"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ordinary_fixed_point_def_f_delta = foreign ~from:c_lib
    "ada_ordinary_fixed_point_def_f_delta"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ordinary_fixed_point_def_f_range = foreign ~from:c_lib
    "ada_ordinary_fixed_point_def_f_range"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let record_type_def_f_has_abstract = foreign ~from:c_lib
    "ada_record_type_def_f_has_abstract"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let record_type_def_f_has_tagged = foreign ~from:c_lib
    "ada_record_type_def_f_has_tagged"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let record_type_def_f_has_limited = foreign ~from:c_lib
    "ada_record_type_def_f_has_limited"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let record_type_def_f_record_def = foreign ~from:c_lib
    "ada_record_type_def_f_record_def"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let signed_int_type_def_f_range = foreign ~from:c_lib
    "ada_signed_int_type_def_f_range"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_expr_p_type_name = foreign ~from:c_lib
    "ada_type_expr_p_type_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_expr_p_designated_type_decl = foreign ~from:c_lib
    "ada_type_expr_p_designated_type_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_expr_p_designated_type_decl_from = foreign ~from:c_lib
    "ada_type_expr_p_designated_type_decl_from"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let anonymous_type_f_type_decl = foreign ~from:c_lib
    "ada_anonymous_type_f_type_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subtype_indication_f_has_not_null = foreign ~from:c_lib
    "ada_subtype_indication_f_has_not_null"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subtype_indication_f_name = foreign ~from:c_lib
    "ada_subtype_indication_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subtype_indication_f_constraint = foreign ~from:c_lib
    "ada_subtype_indication_f_constraint"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subtype_indication_p_subtype_constraints = foreign ~from:c_lib
    "ada_subtype_indication_p_subtype_constraints"
    (ptr EntityStruct.c_type
    @-> ptr ParamActualArrayStruct.c_type
    @-> raisable int)

  let subtype_indication_p_is_static_subtype = foreign ~from:c_lib
    "ada_subtype_indication_p_is_static_subtype"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr bool
    @-> raisable int)

  let synthetic_type_expr_f_target_type = foreign ~from:c_lib
    "ada_synthetic_type_expr_f_target_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let unconstrained_array_index_f_subtype_indication = foreign ~from:c_lib
    "ada_unconstrained_array_index_f_subtype_indication"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let until_node_p_as_bool = foreign ~from:c_lib
    "ada_until_node_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let use_package_clause_f_packages = foreign ~from:c_lib
    "ada_use_package_clause_f_packages"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let use_type_clause_f_has_all = foreign ~from:c_lib
    "ada_use_type_clause_f_has_all"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let use_type_clause_f_types = foreign ~from:c_lib
    "ada_use_type_clause_f_types"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let value_sequence_f_iter_assoc = foreign ~from:c_lib
    "ada_value_sequence_f_iter_assoc"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let variant_f_choices = foreign ~from:c_lib
    "ada_variant_f_choices"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let variant_f_components = foreign ~from:c_lib
    "ada_variant_f_components"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let variant_part_f_discr_name = foreign ~from:c_lib
    "ada_variant_part_f_discr_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let variant_part_f_variant = foreign ~from:c_lib
    "ada_variant_part_f_variant"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let with_clause_f_has_limited = foreign ~from:c_lib
    "ada_with_clause_f_has_limited"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let with_clause_f_has_private = foreign ~from:c_lib
    "ada_with_clause_f_has_private"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let with_clause_f_packages = foreign ~from:c_lib
    "ada_with_clause_f_packages"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let with_private_p_as_bool = foreign ~from:c_lib
    "ada_with_private_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)


end

type analysis_context = {
  c_value : AnalysisContextStruct.t;
  unit_provider : UnitProvider.t
}

and analysis_unit = {
  c_value : AnalysisUnitStruct.t;
  context : analysis_context
}

and entity = EntityStruct.t structure


  and entity_info = {
      md :
         metadata;
      rebindings :
         Rebindings.t;
      from_rebound :
         bool;
  }



  and metadata = {
      dottable_subp :
         bool;
      primitive :
         BareNode.t;
      primitive_real_type :
         BareNode.t;
  }


  
   
  (**
    * abort_node
    * abstract_node
    * ada_list
    * aliased_node
    * all_node
    * array_indices
    * aspect_assoc
    * aspect_clause
    * aspect_spec
    * base_assoc
    * base_formal_param_holder
    * base_record_def
    * basic_assoc
    * basic_decl
    * case_stmt_alternative
    * compilation_unit
    * component_clause
    * component_def
    * constant_node
    * constraint_node
    * declarative_part
    * elsif_expr_part
    * elsif_stmt_part
    * expr
    * handled_stmts
    * interface_kind
    * iter_type
    * library_item
    * limited_node
    * loop_spec
    * mode
    * multi_abstract_state_decl
    * not_null
    * null_component_decl
    * others_designator
    * overriding_node
    * params
    * paren_abstract_state_decl
    * pp_directive
    * pp_then_kw
    * pragma_node
    * private_node
    * protected_def
    * protected_node
    * quantifier
    * range_spec
    * renaming_clause
    * reverse_node
    * select_when_part
    * stmt
    * subp_kind
    * subunit
    * synchronized_node
    * tagged_node
    * task_def
    * type_attributes_repository
    * type_def
    * type_expr
    * unconstrained_array_index
    * until_node
    * use_clause
    * value_sequence
    * variant
    * variant_part
    * with_clause
    * with_private
    *)
  and ada_node =
    [
    | `AbortAbsent
        of abort_absent_fields
    | `AbortPresent
        of abort_present_fields
    | `AbstractAbsent
        of abstract_absent_fields
    | `AbstractPresent
        of abstract_present_fields
    | `AdaNodeList
        of ada_node_list_fields
    | `AbstractStateDeclList
        of abstract_state_decl_list_fields
    | `AlternativesList
        of alternatives_list_fields
    | `ConstraintList
        of constraint_list_fields
    | `DeclList
        of decl_list_fields
    | `StmtList
        of stmt_list_fields
    | `AspectAssocList
        of aspect_assoc_list_fields
    | `BaseAssocList
        of base_assoc_list_fields
    | `AssocList
        of assoc_list_fields
    | `BasicDeclList
        of basic_decl_list_fields
    | `CaseExprAlternativeList
        of case_expr_alternative_list_fields
    | `CaseStmtAlternativeList
        of case_stmt_alternative_list_fields
    | `CompilationUnitList
        of compilation_unit_list_fields
    | `ConcatOperandList
        of concat_operand_list_fields
    | `ContractCaseAssocList
        of contract_case_assoc_list_fields
    | `DefiningNameList
        of defining_name_list_fields
    | `DiscriminantSpecList
        of discriminant_spec_list_fields
    | `ElsifExprPartList
        of elsif_expr_part_list_fields
    | `ElsifStmtPartList
        of elsif_stmt_part_list_fields
    | `EnumLiteralDeclList
        of enum_literal_decl_list_fields
    | `ExprAlternativesList
        of expr_alternatives_list_fields
    | `DiscriminantChoiceList
        of discriminant_choice_list_fields
    | `NameList
        of name_list_fields
    | `ParentList
        of parent_list_fields
    | `ParamSpecList
        of param_spec_list_fields
    | `PragmaNodeList
        of pragma_node_list_fields
    | `SelectWhenPartList
        of select_when_part_list_fields
    | `UnconstrainedArrayIndexList
        of unconstrained_array_index_list_fields
    | `VariantList
        of variant_list_fields
    | `AliasedAbsent
        of aliased_absent_fields
    | `AliasedPresent
        of aliased_present_fields
    | `AllAbsent
        of all_absent_fields
    | `AllPresent
        of all_present_fields
    | `ConstrainedArrayIndices
        of constrained_array_indices_fields
    | `UnconstrainedArrayIndices
        of unconstrained_array_indices_fields
    | `AspectAssoc
        of aspect_assoc_fields
    | `AtClause
        of at_clause_fields
    | `AttributeDefClause
        of attribute_def_clause_fields
    | `EnumRepClause
        of enum_rep_clause_fields
    | `RecordRepClause
        of record_rep_clause_fields
    | `AspectSpec
        of aspect_spec_fields
    | `ContractCaseAssoc
        of contract_case_assoc_fields
    | `PragmaArgumentAssoc
        of pragma_argument_assoc_fields
    | `EntrySpec
        of entry_spec_fields
    | `EnumSubpSpec
        of enum_subp_spec_fields
    | `SubpSpec
        of subp_spec_fields
    | `SyntheticBinarySpec
        of synthetic_binary_spec_fields
    | `SyntheticUnarySpec
        of synthetic_unary_spec_fields
    | `ComponentList
        of component_list_fields
    | `KnownDiscriminantPart
        of known_discriminant_part_fields
    | `UnknownDiscriminantPart
        of unknown_discriminant_part_fields
    | `EntryCompletionFormalParams
        of entry_completion_formal_params_fields
    | `GenericFormalPart
        of generic_formal_part_fields
    | `NullRecordDef
        of null_record_def_fields
    | `RecordDef
        of record_def_fields
    | `AggregateAssoc
        of aggregate_assoc_fields
    | `MultiDimArrayAssoc
        of multi_dim_array_assoc_fields
    | `CompositeConstraintAssoc
        of composite_constraint_assoc_fields
    | `IteratedAssoc
        of iterated_assoc_fields
    | `ParamAssoc
        of param_assoc_fields
    | `AbstractStateDecl
        of abstract_state_decl_fields
    | `AnonymousExprDecl
        of anonymous_expr_decl_fields
    | `ComponentDecl
        of component_decl_fields
    | `DiscriminantSpec
        of discriminant_spec_fields
    | `GenericFormalObjDecl
        of generic_formal_obj_decl_fields
    | `GenericFormalPackage
        of generic_formal_package_fields
    | `GenericFormalSubpDecl
        of generic_formal_subp_decl_fields
    | `GenericFormalTypeDecl
        of generic_formal_type_decl_fields
    | `ParamSpec
        of param_spec_fields
    | `SyntheticFormalParamDecl
        of synthetic_formal_param_decl_fields
    | `GenericPackageInternal
        of generic_package_internal_fields
    | `PackageDecl
        of package_decl_fields
    | `DiscreteBaseSubtypeDecl
        of discrete_base_subtype_decl_fields
    | `SubtypeDecl
        of subtype_decl_fields
    | `ClasswideTypeDecl
        of classwide_type_decl_fields
    | `IncompleteTypeDecl
        of incomplete_type_decl_fields
    | `IncompleteFormalTypeDecl
        of incomplete_formal_type_decl_fields
    | `IncompleteTaggedTypeDecl
        of incomplete_tagged_type_decl_fields
    | `ProtectedTypeDecl
        of protected_type_decl_fields
    | `TaskTypeDecl
        of task_type_decl_fields
    | `SingleTaskTypeDecl
        of single_task_type_decl_fields
    | `AnonymousTypeDecl
        of anonymous_type_decl_fields
    | `SynthAnonymousTypeDecl
        of synth_anonymous_type_decl_fields
    | `ConcreteTypeDecl
        of concrete_type_decl_fields
    | `FormalTypeDecl
        of formal_type_decl_fields
    | `AbstractSubpDecl
        of abstract_subp_decl_fields
    | `AbstractFormalSubpDecl
        of abstract_formal_subp_decl_fields
    | `ConcreteFormalSubpDecl
        of concrete_formal_subp_decl_fields
    | `SubpDecl
        of subp_decl_fields
    | `EntryDecl
        of entry_decl_fields
    | `EnumLiteralDecl
        of enum_literal_decl_fields
    | `SyntheticCharEnumLit
        of synthetic_char_enum_lit_fields
    | `GenericSubpInternal
        of generic_subp_internal_fields
    | `SyntheticSubpDecl
        of synthetic_subp_decl_fields
    | `AcceptStmtBody
        of accept_stmt_body_fields
    | `ExprFunction
        of expr_function_fields
    | `NullSubpDecl
        of null_subp_decl_fields
    | `SubpBody
        of subp_body_fields
    | `SubpRenamingDecl
        of subp_renaming_decl_fields
    | `PackageBodyStub
        of package_body_stub_fields
    | `ProtectedBodyStub
        of protected_body_stub_fields
    | `SubpBodyStub
        of subp_body_stub_fields
    | `TaskBodyStub
        of task_body_stub_fields
    | `EntryBody
        of entry_body_fields
    | `PackageBody
        of package_body_fields
    | `ProtectedBody
        of protected_body_fields
    | `TaskBody
        of task_body_fields
    | `EntryIndexSpec
        of entry_index_spec_fields
    | `ErrorDecl
        of error_decl_fields
    | `ExceptionDecl
        of exception_decl_fields
    | `ExceptionHandler
        of exception_handler_fields
    | `ForLoopVarDecl
        of for_loop_var_decl_fields
    | `GenericPackageDecl
        of generic_package_decl_fields
    | `GenericSubpDecl
        of generic_subp_decl_fields
    | `GenericPackageInstantiation
        of generic_package_instantiation_fields
    | `GenericSubpInstantiation
        of generic_subp_instantiation_fields
    | `GenericPackageRenamingDecl
        of generic_package_renaming_decl_fields
    | `GenericSubpRenamingDecl
        of generic_subp_renaming_decl_fields
    | `LabelDecl
        of label_decl_fields
    | `NamedStmtDecl
        of named_stmt_decl_fields
    | `NumberDecl
        of number_decl_fields
    | `ObjectDecl
        of object_decl_fields
    | `ExtendedReturnStmtObjectDecl
        of extended_return_stmt_object_decl_fields
    | `NoTypeObjectRenamingDecl
        of no_type_object_renaming_decl_fields
    | `PackageRenamingDecl
        of package_renaming_decl_fields
    | `SingleProtectedDecl
        of single_protected_decl_fields
    | `SingleTaskDecl
        of single_task_decl_fields
    | `CaseStmtAlternative
        of case_stmt_alternative_fields
    | `CompilationUnit
        of compilation_unit_fields
    | `ComponentClause
        of component_clause_fields
    | `ComponentDef
        of component_def_fields
    | `ConstantAbsent
        of constant_absent_fields
    | `ConstantPresent
        of constant_present_fields
    | `CompositeConstraint
        of composite_constraint_fields
    | `DeltaConstraint
        of delta_constraint_fields
    | `DigitsConstraint
        of digits_constraint_fields
    | `RangeConstraint
        of range_constraint_fields
    | `DeclarativePart
        of declarative_part_fields
    | `PrivatePart
        of private_part_fields
    | `PublicPart
        of public_part_fields
    | `ElsifExprPart
        of elsif_expr_part_fields
    | `ElsifStmtPart
        of elsif_stmt_part_fields
    | `AbstractStateDeclExpr
        of abstract_state_decl_expr_fields
    | `Allocator
        of allocator_fields
    | `Aggregate
        of aggregate_fields
    | `BracketAggregate
        of bracket_aggregate_fields
    | `DeltaAggregate
        of delta_aggregate_fields
    | `BracketDeltaAggregate
        of bracket_delta_aggregate_fields
    | `NullRecordAggregate
        of null_record_aggregate_fields
    | `BinOp
        of bin_op_fields
    | `RelationOp
        of relation_op_fields
    | `BoxExpr
        of box_expr_fields
    | `CaseExprAlternative
        of case_expr_alternative_fields
    | `ConcatOp
        of concat_op_fields
    | `ConcatOperand
        of concat_operand_fields
    | `CaseExpr
        of case_expr_fields
    | `IfExpr
        of if_expr_fields
    | `ContractCases
        of contract_cases_fields
    | `DeclExpr
        of decl_expr_fields
    | `MembershipExpr
        of membership_expr_fields
    | `AttributeRef
        of attribute_ref_fields
    | `CallExpr
        of call_expr_fields
    | `DefiningName
        of defining_name_fields
    | `SyntheticDefiningName
        of synthetic_defining_name_fields
    | `DiscreteSubtypeName
        of discrete_subtype_name_fields
    | `DottedName
        of dotted_name_fields
    | `EndName
        of end_name_fields
    | `ExplicitDeref
        of explicit_deref_fields
    | `QualExpr
        of qual_expr_fields
    | `ReduceAttributeRef
        of reduce_attribute_ref_fields
    | `CharLiteral
        of char_literal_fields
    | `Identifier
        of identifier_fields
    | `OpAbs
        of op_abs_fields
    | `OpAnd
        of op_and_fields
    | `OpAndThen
        of op_and_then_fields
    | `OpConcat
        of op_concat_fields
    | `OpDiv
        of op_div_fields
    | `OpDoubleDot
        of op_double_dot_fields
    | `OpEq
        of op_eq_fields
    | `OpGt
        of op_gt_fields
    | `OpGte
        of op_gte_fields
    | `OpIn
        of op_in_fields
    | `OpLt
        of op_lt_fields
    | `OpLte
        of op_lte_fields
    | `OpMinus
        of op_minus_fields
    | `OpMod
        of op_mod_fields
    | `OpMult
        of op_mult_fields
    | `OpNeq
        of op_neq_fields
    | `OpNot
        of op_not_fields
    | `OpNotIn
        of op_not_in_fields
    | `OpOr
        of op_or_fields
    | `OpOrElse
        of op_or_else_fields
    | `OpPlus
        of op_plus_fields
    | `OpPow
        of op_pow_fields
    | `OpRem
        of op_rem_fields
    | `OpXor
        of op_xor_fields
    | `StringLiteral
        of string_literal_fields
    | `NullLiteral
        of null_literal_fields
    | `IntLiteral
        of int_literal_fields
    | `RealLiteral
        of real_literal_fields
    | `SyntheticIdentifier
        of synthetic_identifier_fields
    | `TargetName
        of target_name_fields
    | `UpdateAttributeRef
        of update_attribute_ref_fields
    | `ParenExpr
        of paren_expr_fields
    | `QuantifiedExpr
        of quantified_expr_fields
    | `RaiseExpr
        of raise_expr_fields
    | `UnOp
        of un_op_fields
    | `HandledStmts
        of handled_stmts_fields
    | `InterfaceKindLimited
        of interface_kind_limited_fields
    | `InterfaceKindProtected
        of interface_kind_protected_fields
    | `InterfaceKindSynchronized
        of interface_kind_synchronized_fields
    | `InterfaceKindTask
        of interface_kind_task_fields
    | `IterTypeIn
        of iter_type_in_fields
    | `IterTypeOf
        of iter_type_of_fields
    | `LibraryItem
        of library_item_fields
    | `LimitedAbsent
        of limited_absent_fields
    | `LimitedPresent
        of limited_present_fields
    | `ForLoopSpec
        of for_loop_spec_fields
    | `WhileLoopSpec
        of while_loop_spec_fields
    | `ModeDefault
        of mode_default_fields
    | `ModeIn
        of mode_in_fields
    | `ModeInOut
        of mode_in_out_fields
    | `ModeOut
        of mode_out_fields
    | `MultiAbstractStateDecl
        of multi_abstract_state_decl_fields
    | `NotNullAbsent
        of not_null_absent_fields
    | `NotNullPresent
        of not_null_present_fields
    | `NullComponentDecl
        of null_component_decl_fields
    | `OthersDesignator
        of others_designator_fields
    | `OverridingNotOverriding
        of overriding_not_overriding_fields
    | `OverridingOverriding
        of overriding_overriding_fields
    | `OverridingUnspecified
        of overriding_unspecified_fields
    | `Params
        of params_fields
    | `ParenAbstractStateDecl
        of paren_abstract_state_decl_fields
    | `PpElseDirective
        of pp_else_directive_fields
    | `PpElsifDirective
        of pp_elsif_directive_fields
    | `PpEndIfDirective
        of pp_end_if_directive_fields
    | `PpIfDirective
        of pp_if_directive_fields
    | `PpThenKw
        of pp_then_kw_fields
    | `PragmaNode
        of pragma_node_fields
    | `PrivateAbsent
        of private_absent_fields
    | `PrivatePresent
        of private_present_fields
    | `ProtectedDef
        of protected_def_fields
    | `ProtectedAbsent
        of protected_absent_fields
    | `ProtectedPresent
        of protected_present_fields
    | `QuantifierAll
        of quantifier_all_fields
    | `QuantifierSome
        of quantifier_some_fields
    | `RangeSpec
        of range_spec_fields
    | `RenamingClause
        of renaming_clause_fields
    | `SyntheticRenamingClause
        of synthetic_renaming_clause_fields
    | `ReverseAbsent
        of reverse_absent_fields
    | `ReversePresent
        of reverse_present_fields
    | `SelectWhenPart
        of select_when_part_fields
    | `AcceptStmt
        of accept_stmt_fields
    | `AcceptStmtWithStmts
        of accept_stmt_with_stmts_fields
    | `ForLoopStmt
        of for_loop_stmt_fields
    | `LoopStmt
        of loop_stmt_fields
    | `WhileLoopStmt
        of while_loop_stmt_fields
    | `BeginBlock
        of begin_block_fields
    | `DeclBlock
        of decl_block_fields
    | `CaseStmt
        of case_stmt_fields
    | `ExtendedReturnStmt
        of extended_return_stmt_fields
    | `IfStmt
        of if_stmt_fields
    | `NamedStmt
        of named_stmt_fields
    | `SelectStmt
        of select_stmt_fields
    | `ErrorStmt
        of error_stmt_fields
    | `AbortStmt
        of abort_stmt_fields
    | `AssignStmt
        of assign_stmt_fields
    | `CallStmt
        of call_stmt_fields
    | `DelayStmt
        of delay_stmt_fields
    | `ExitStmt
        of exit_stmt_fields
    | `GotoStmt
        of goto_stmt_fields
    | `Label
        of label_fields
    | `NullStmt
        of null_stmt_fields
    | `RaiseStmt
        of raise_stmt_fields
    | `RequeueStmt
        of requeue_stmt_fields
    | `ReturnStmt
        of return_stmt_fields
    | `TerminateAlternative
        of terminate_alternative_fields
    | `SubpKindFunction
        of subp_kind_function_fields
    | `SubpKindProcedure
        of subp_kind_procedure_fields
    | `Subunit
        of subunit_fields
    | `SynchronizedAbsent
        of synchronized_absent_fields
    | `SynchronizedPresent
        of synchronized_present_fields
    | `TaggedAbsent
        of tagged_absent_fields
    | `TaggedPresent
        of tagged_present_fields
    | `TaskDef
        of task_def_fields
    | `TypeAttributesRepository
        of type_attributes_repository_fields
    | `AccessToSubpDef
        of access_to_subp_def_fields
    | `AnonymousTypeAccessDef
        of anonymous_type_access_def_fields
    | `TypeAccessDef
        of type_access_def_fields
    | `ArrayTypeDef
        of array_type_def_fields
    | `DerivedTypeDef
        of derived_type_def_fields
    | `EnumTypeDef
        of enum_type_def_fields
    | `FormalDiscreteTypeDef
        of formal_discrete_type_def_fields
    | `InterfaceTypeDef
        of interface_type_def_fields
    | `ModIntTypeDef
        of mod_int_type_def_fields
    | `PrivateTypeDef
        of private_type_def_fields
    | `DecimalFixedPointDef
        of decimal_fixed_point_def_fields
    | `FloatingPointDef
        of floating_point_def_fields
    | `OrdinaryFixedPointDef
        of ordinary_fixed_point_def_fields
    | `RecordTypeDef
        of record_type_def_fields
    | `SignedIntTypeDef
        of signed_int_type_def_fields
    | `AnonymousType
        of anonymous_type_fields
    | `EnumLitSynthTypeExpr
        of enum_lit_synth_type_expr_fields
    | `SubtypeIndication
        of subtype_indication_fields
    | `ConstrainedSubtypeIndication
        of constrained_subtype_indication_fields
    | `DiscreteSubtypeIndication
        of discrete_subtype_indication_fields
    | `SyntheticTypeExpr
        of synthetic_type_expr_fields
    | `UnconstrainedArrayIndex
        of unconstrained_array_index_fields
    | `UntilAbsent
        of until_absent_fields
    | `UntilPresent
        of until_present_fields
    | `UsePackageClause
        of use_package_clause_fields
    | `UseTypeClause
        of use_type_clause_fields
    | `ValueSequence
        of value_sequence_fields
    | `Variant
        of variant_fields
    | `VariantPart
        of variant_part_fields
    | `WithClause
        of with_clause_fields
    | `WithPrivateAbsent
        of with_private_absent_fields
    | `WithPrivatePresent
        of with_private_present_fields
    ]

  
   
  and abort_node =
    [
    | `AbortAbsent
        of abort_absent_fields
    | `AbortPresent
        of abort_present_fields
    ]

  
   
  and abort_absent =
    [
    | `AbortAbsent
        of abort_absent_fields
    ]
  and abort_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and abort_present =
    [
    | `AbortPresent
        of abort_present_fields
    ]
  and abort_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and abstract_node =
    [
    | `AbstractAbsent
        of abstract_absent_fields
    | `AbstractPresent
        of abstract_present_fields
    ]

  
   
  and abstract_absent =
    [
    | `AbstractAbsent
        of abstract_absent_fields
    ]
  and abstract_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and abstract_present =
    [
    | `AbstractPresent
        of abstract_present_fields
    ]
  and abstract_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * ada_node_list
    * aspect_assoc_list
    * base_assoc_list
    * basic_assoc_list
    * basic_decl_list
    * case_expr_alternative_list
    * case_stmt_alternative_list
    * compilation_unit_list
    * concat_operand_list
    * contract_case_assoc_list
    * defining_name_list
    * discriminant_spec_list
    * elsif_expr_part_list
    * elsif_stmt_part_list
    * enum_literal_decl_list
    * expr_list
    * identifier_list
    * name_list
    * param_spec_list
    * pragma_node_list
    * select_when_part_list
    * unconstrained_array_index_list
    * variant_list
    *)
  and ada_list =
    [
    | `AdaNodeList
        of ada_node_list_fields
    | `AbstractStateDeclList
        of abstract_state_decl_list_fields
    | `AlternativesList
        of alternatives_list_fields
    | `ConstraintList
        of constraint_list_fields
    | `DeclList
        of decl_list_fields
    | `StmtList
        of stmt_list_fields
    | `AspectAssocList
        of aspect_assoc_list_fields
    | `BaseAssocList
        of base_assoc_list_fields
    | `AssocList
        of assoc_list_fields
    | `BasicDeclList
        of basic_decl_list_fields
    | `CaseExprAlternativeList
        of case_expr_alternative_list_fields
    | `CaseStmtAlternativeList
        of case_stmt_alternative_list_fields
    | `CompilationUnitList
        of compilation_unit_list_fields
    | `ConcatOperandList
        of concat_operand_list_fields
    | `ContractCaseAssocList
        of contract_case_assoc_list_fields
    | `DefiningNameList
        of defining_name_list_fields
    | `DiscriminantSpecList
        of discriminant_spec_list_fields
    | `ElsifExprPartList
        of elsif_expr_part_list_fields
    | `ElsifStmtPartList
        of elsif_stmt_part_list_fields
    | `EnumLiteralDeclList
        of enum_literal_decl_list_fields
    | `ExprAlternativesList
        of expr_alternatives_list_fields
    | `DiscriminantChoiceList
        of discriminant_choice_list_fields
    | `NameList
        of name_list_fields
    | `ParentList
        of parent_list_fields
    | `ParamSpecList
        of param_spec_list_fields
    | `PragmaNodeList
        of pragma_node_list_fields
    | `SelectWhenPartList
        of select_when_part_list_fields
    | `UnconstrainedArrayIndexList
        of unconstrained_array_index_list_fields
    | `VariantList
        of variant_list_fields
    ]

  
   
  and ada_node_list =
    [
    | `AdaNodeList
        of ada_node_list_fields
    | `AbstractStateDeclList
        of abstract_state_decl_list_fields
    | `AlternativesList
        of alternatives_list_fields
    | `ConstraintList
        of constraint_list_fields
    | `DeclList
        of decl_list_fields
    | `StmtList
        of stmt_list_fields
    ]
  and ada_node_list_fields = 
  {
    list : ada_node list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and abstract_state_decl_list =
    [
    | `AbstractStateDeclList
        of abstract_state_decl_list_fields
    ]
  and abstract_state_decl_list_fields = 
  {
    list : ada_node list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and alternatives_list =
    [
    | `AlternativesList
        of alternatives_list_fields
    ]
  and alternatives_list_fields = 
  {
    list : ada_node list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and constraint_list =
    [
    | `ConstraintList
        of constraint_list_fields
    ]
  and constraint_list_fields = 
  {
    list : ada_node list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and decl_list =
    [
    | `DeclList
        of decl_list_fields
    ]
  and decl_list_fields = 
  {
    list : ada_node list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and stmt_list =
    [
    | `StmtList
        of stmt_list_fields
    ]
  and stmt_list_fields = 
  {
    list : ada_node list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and aspect_assoc_list =
    [
    | `AspectAssocList
        of aspect_assoc_list_fields
    ]
  and aspect_assoc_list_fields = 
  {
    list : aspect_assoc list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_assoc_list =
    [
    | `BaseAssocList
        of base_assoc_list_fields
    ]
  and base_assoc_list_fields = 
  {
    list : base_assoc list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and basic_assoc_list =
    [
    | `AssocList
        of assoc_list_fields
    ]

  
   
  and assoc_list =
    [
    | `AssocList
        of assoc_list_fields
    ]
  and assoc_list_fields = 
  {
    list : basic_assoc list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and basic_decl_list =
    [
    | `BasicDeclList
        of basic_decl_list_fields
    ]
  and basic_decl_list_fields = 
  {
    list : basic_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and case_expr_alternative_list =
    [
    | `CaseExprAlternativeList
        of case_expr_alternative_list_fields
    ]
  and case_expr_alternative_list_fields = 
  {
    list : case_expr_alternative list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and case_stmt_alternative_list =
    [
    | `CaseStmtAlternativeList
        of case_stmt_alternative_list_fields
    ]
  and case_stmt_alternative_list_fields = 
  {
    list : case_stmt_alternative list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and compilation_unit_list =
    [
    | `CompilationUnitList
        of compilation_unit_list_fields
    ]
  and compilation_unit_list_fields = 
  {
    list : compilation_unit list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and concat_operand_list =
    [
    | `ConcatOperandList
        of concat_operand_list_fields
    ]
  and concat_operand_list_fields = 
  {
    list : concat_operand list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and contract_case_assoc_list =
    [
    | `ContractCaseAssocList
        of contract_case_assoc_list_fields
    ]
  and contract_case_assoc_list_fields = 
  {
    list : contract_case_assoc list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and defining_name_list =
    [
    | `DefiningNameList
        of defining_name_list_fields
    ]
  and defining_name_list_fields = 
  {
    list : defining_name list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and discriminant_spec_list =
    [
    | `DiscriminantSpecList
        of discriminant_spec_list_fields
    ]
  and discriminant_spec_list_fields = 
  {
    list : discriminant_spec list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and elsif_expr_part_list =
    [
    | `ElsifExprPartList
        of elsif_expr_part_list_fields
    ]
  and elsif_expr_part_list_fields = 
  {
    list : elsif_expr_part list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and elsif_stmt_part_list =
    [
    | `ElsifStmtPartList
        of elsif_stmt_part_list_fields
    ]
  and elsif_stmt_part_list_fields = 
  {
    list : elsif_stmt_part list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_literal_decl_list =
    [
    | `EnumLiteralDeclList
        of enum_literal_decl_list_fields
    ]
  and enum_literal_decl_list_fields = 
  {
    list : enum_literal_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and expr_list =
    [
    | `ExprAlternativesList
        of expr_alternatives_list_fields
    ]

  
   
  and expr_alternatives_list =
    [
    | `ExprAlternativesList
        of expr_alternatives_list_fields
    ]
  and expr_alternatives_list_fields = 
  {
    list : expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and identifier_list =
    [
    | `DiscriminantChoiceList
        of discriminant_choice_list_fields
    ]

  
   
  and discriminant_choice_list =
    [
    | `DiscriminantChoiceList
        of discriminant_choice_list_fields
    ]
  and discriminant_choice_list_fields = 
  {
    list : identifier list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and name_list =
    [
    | `NameList
        of name_list_fields
    | `ParentList
        of parent_list_fields
    ]
  and name_list_fields = 
  {
    list : name list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and parent_list =
    [
    | `ParentList
        of parent_list_fields
    ]
  and parent_list_fields = 
  {
    list : name list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and param_spec_list =
    [
    | `ParamSpecList
        of param_spec_list_fields
    ]
  and param_spec_list_fields = 
  {
    list : param_spec list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and pragma_node_list =
    [
    | `PragmaNodeList
        of pragma_node_list_fields
    ]
  and pragma_node_list_fields = 
  {
    list : pragma_node list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and select_when_part_list =
    [
    | `SelectWhenPartList
        of select_when_part_list_fields
    ]
  and select_when_part_list_fields = 
  {
    list : select_when_part list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and unconstrained_array_index_list =
    [
    | `UnconstrainedArrayIndexList
        of unconstrained_array_index_list_fields
    ]
  and unconstrained_array_index_list_fields = 
  {
    list : unconstrained_array_index list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and variant_list =
    [
    | `VariantList
        of variant_list_fields
    ]
  and variant_list_fields = 
  {
    list : variant list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and aliased_node =
    [
    | `AliasedAbsent
        of aliased_absent_fields
    | `AliasedPresent
        of aliased_present_fields
    ]

  
   
  and aliased_absent =
    [
    | `AliasedAbsent
        of aliased_absent_fields
    ]
  and aliased_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and aliased_present =
    [
    | `AliasedPresent
        of aliased_present_fields
    ]
  and aliased_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and all_node =
    [
    | `AllAbsent
        of all_absent_fields
    | `AllPresent
        of all_present_fields
    ]

  
   
  and all_absent =
    [
    | `AllAbsent
        of all_absent_fields
    ]
  and all_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and all_present =
    [
    | `AllPresent
        of all_present_fields
    ]
  and all_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and array_indices =
    [
    | `ConstrainedArrayIndices
        of constrained_array_indices_fields
    | `UnconstrainedArrayIndices
        of unconstrained_array_indices_fields
    ]

  
   
  and constrained_array_indices =
    [
    | `ConstrainedArrayIndices
        of constrained_array_indices_fields
    ]
  and constrained_array_indices_fields = 
  {
         
    f_list: constraint_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and unconstrained_array_indices =
    [
    | `UnconstrainedArrayIndices
        of unconstrained_array_indices_fields
    ]
  and unconstrained_array_indices_fields = 
  {
         
    f_types: unconstrained_array_index_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and aspect_assoc =
    [
    | `AspectAssoc
        of aspect_assoc_fields
    ]
  and aspect_assoc_fields = 
  {
         
    f_id: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_expr: [
      | `AbstractStateDeclExpr
          of abstract_state_decl_expr_fields
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `ContractCases
          of contract_cases_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and aspect_clause =
    [
    | `AtClause
        of at_clause_fields
    | `AttributeDefClause
        of attribute_def_clause_fields
    | `EnumRepClause
        of enum_rep_clause_fields
    | `RecordRepClause
        of record_rep_clause_fields
    ]

  
   
  and at_clause =
    [
    | `AtClause
        of at_clause_fields
    ]
  and at_clause_fields = 
  {
         
    f_name: [
      | `CharLiteral
          of char_literal_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and attribute_def_clause =
    [
    | `AttributeDefClause
        of attribute_def_clause_fields
    ]
  and attribute_def_clause_fields = 
  {
         
    f_attribute_expr: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_rep_clause =
    [
    | `EnumRepClause
        of enum_rep_clause_fields
    ]
  and enum_rep_clause_fields = 
  {
         
    f_type_name: [
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
         
    f_aggregate: base_aggregate
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and record_rep_clause =
    [
    | `RecordRepClause
        of record_rep_clause_fields
    ]
  and record_rep_clause_fields = 
  {
         
    f_name: [
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
         
    f_at_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_components: ada_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and aspect_spec =
    [
    | `AspectSpec
        of aspect_spec_fields
    ]
  and aspect_spec_fields = 
  {
         
    f_aspect_assocs: aspect_assoc_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_assoc =
    [
    | `ContractCaseAssoc
        of contract_case_assoc_fields
    | `PragmaArgumentAssoc
        of pragma_argument_assoc_fields
    ]

  
   
  and contract_case_assoc =
    [
    | `ContractCaseAssoc
        of contract_case_assoc_fields
    ]
  and contract_case_assoc_fields = 
  {
         
    f_guard: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `OthersDesignator
          of others_designator_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_consequence: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and pragma_argument_assoc =
    [
    | `PragmaArgumentAssoc
        of pragma_argument_assoc_fields
    ]
  and pragma_argument_assoc_fields = 
  {
         
    f_name: [
      | `AttributeRef
          of attribute_ref_fields
      | `Identifier
          of identifier_fields
    ]
    option
    Lazy.t;
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * base_subp_spec
    * component_list
    * discriminant_part
    * entry_completion_formal_params
    * generic_formal_part
    *)
  and base_formal_param_holder =
    [
    | `EntrySpec
        of entry_spec_fields
    | `EnumSubpSpec
        of enum_subp_spec_fields
    | `SubpSpec
        of subp_spec_fields
    | `SyntheticBinarySpec
        of synthetic_binary_spec_fields
    | `SyntheticUnarySpec
        of synthetic_unary_spec_fields
    | `ComponentList
        of component_list_fields
    | `KnownDiscriminantPart
        of known_discriminant_part_fields
    | `UnknownDiscriminantPart
        of unknown_discriminant_part_fields
    | `EntryCompletionFormalParams
        of entry_completion_formal_params_fields
    | `GenericFormalPart
        of generic_formal_part_fields
    ]

  
   
  and base_subp_spec =
    [
    | `EntrySpec
        of entry_spec_fields
    | `EnumSubpSpec
        of enum_subp_spec_fields
    | `SubpSpec
        of subp_spec_fields
    | `SyntheticBinarySpec
        of synthetic_binary_spec_fields
    | `SyntheticUnarySpec
        of synthetic_unary_spec_fields
    ]

  
   
  and entry_spec =
    [
    | `EntrySpec
        of entry_spec_fields
    ]
  and entry_spec_fields = 
  {
         
    f_entry_name: defining_name
    Lazy.t;
         
    f_family_type: [
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConstrainedSubtypeIndication
          of constrained_subtype_indication_fields
      | `DiscreteSubtypeIndication
          of discrete_subtype_indication_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `SubtypeIndication
          of subtype_indication_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_entry_params: params
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_subp_spec =
    [
    | `EnumSubpSpec
        of enum_subp_spec_fields
    ]
  and enum_subp_spec_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and subp_spec =
    [
    | `SubpSpec
        of subp_spec_fields
    ]
  and subp_spec_fields = 
  {
         
    f_subp_kind: subp_kind
    Lazy.t;
         
    f_subp_name: defining_name
    option
    Lazy.t;
         
    f_subp_params: params
    option
    Lazy.t;
         
    f_subp_returns: [
      | `AnonymousType
          of anonymous_type_fields
      | `ConstrainedSubtypeIndication
          of constrained_subtype_indication_fields
      | `DiscreteSubtypeIndication
          of discrete_subtype_indication_fields
      | `SubtypeIndication
          of subtype_indication_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and synthetic_binary_spec =
    [
    | `SyntheticBinarySpec
        of synthetic_binary_spec_fields
    ]
  and synthetic_binary_spec_fields = 
  {
         
    f_left_param: synthetic_formal_param_decl
    Lazy.t;
         
    f_right_param: synthetic_formal_param_decl
    Lazy.t;
         
    f_return_type_expr: type_expr
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and synthetic_unary_spec =
    [
    | `SyntheticUnarySpec
        of synthetic_unary_spec_fields
    ]
  and synthetic_unary_spec_fields = 
  {
         
    f_right_param: synthetic_formal_param_decl
    Lazy.t;
         
    f_return_type_expr: synthetic_type_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and component_list =
    [
    | `ComponentList
        of component_list_fields
    ]
  and component_list_fields = 
  {
         
    f_components: ada_node_list
    Lazy.t;
         
    f_variant_part: variant_part
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and discriminant_part =
    [
    | `KnownDiscriminantPart
        of known_discriminant_part_fields
    | `UnknownDiscriminantPart
        of unknown_discriminant_part_fields
    ]

  
   
  and known_discriminant_part =
    [
    | `KnownDiscriminantPart
        of known_discriminant_part_fields
    ]
  and known_discriminant_part_fields = 
  {
         
    f_discr_specs: discriminant_spec_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and unknown_discriminant_part =
    [
    | `UnknownDiscriminantPart
        of unknown_discriminant_part_fields
    ]
  and unknown_discriminant_part_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and entry_completion_formal_params =
    [
    | `EntryCompletionFormalParams
        of entry_completion_formal_params_fields
    ]
  and entry_completion_formal_params_fields = 
  {
         
    f_params: params
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_formal_part =
    [
    | `GenericFormalPart
        of generic_formal_part_fields
    ]
  and generic_formal_part_fields = 
  {
         
    f_decls: ada_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_record_def =
    [
    | `NullRecordDef
        of null_record_def_fields
    | `RecordDef
        of record_def_fields
    ]

  
   
  and null_record_def =
    [
    | `NullRecordDef
        of null_record_def_fields
    ]
  and null_record_def_fields = 
  {
         
    f_components: component_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and record_def =
    [
    | `RecordDef
        of record_def_fields
    ]
  and record_def_fields = 
  {
         
    f_components: component_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * aggregate_assoc
    * composite_constraint_assoc
    * iterated_assoc
    * param_assoc
    *)
  and basic_assoc =
    [
    | `AggregateAssoc
        of aggregate_assoc_fields
    | `MultiDimArrayAssoc
        of multi_dim_array_assoc_fields
    | `CompositeConstraintAssoc
        of composite_constraint_assoc_fields
    | `IteratedAssoc
        of iterated_assoc_fields
    | `ParamAssoc
        of param_assoc_fields
    ]

  
   
  and aggregate_assoc =
    [
    | `AggregateAssoc
        of aggregate_assoc_fields
    | `MultiDimArrayAssoc
        of multi_dim_array_assoc_fields
    ]
  and aggregate_assoc_fields = 
  {
         
    f_designators: alternatives_list
    Lazy.t;
         
    f_r_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BoxExpr
          of box_expr_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and multi_dim_array_assoc =
    [
    | `MultiDimArrayAssoc
        of multi_dim_array_assoc_fields
    ]
  and multi_dim_array_assoc_fields = 
  {
         
    f_designators: alternatives_list
    Lazy.t;
         
    f_r_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BoxExpr
          of box_expr_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and composite_constraint_assoc =
    [
    | `CompositeConstraintAssoc
        of composite_constraint_assoc_fields
    ]
  and composite_constraint_assoc_fields = 
  {
         
    f_ids: identifier_list
    Lazy.t;
         
    f_constraint_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DiscreteSubtypeIndication
          of discrete_subtype_indication_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and iterated_assoc =
    [
    | `IteratedAssoc
        of iterated_assoc_fields
    ]
  and iterated_assoc_fields = 
  {
         
    f_spec: for_loop_spec
    Lazy.t;
         
    f_r_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and param_assoc =
    [
    | `ParamAssoc
        of param_assoc_fields
    ]
  and param_assoc_fields = 
  {
         
    f_designator: [
      | `Identifier
          of identifier_fields
      | `OthersDesignator
          of others_designator_fields
      | `StringLiteral
          of string_literal_fields
    ]
    option
    Lazy.t;
         
    f_r_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BoxExpr
          of box_expr_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * abstract_state_decl
    * anonymous_expr_decl
    * base_formal_param_decl
    * base_package_decl
    * base_type_decl
    * basic_subp_decl
    * body_node
    * entry_index_spec
    * error_decl
    * exception_decl
    * exception_handler
    * for_loop_var_decl
    * generic_decl
    * generic_instantiation
    * generic_renaming_decl
    * label_decl
    * named_stmt_decl
    * number_decl
    * object_decl
    * package_renaming_decl
    * single_protected_decl
    * single_task_decl
    *)
  and basic_decl =
    [
    | `AbstractStateDecl
        of abstract_state_decl_fields
    | `AnonymousExprDecl
        of anonymous_expr_decl_fields
    | `ComponentDecl
        of component_decl_fields
    | `DiscriminantSpec
        of discriminant_spec_fields
    | `GenericFormalObjDecl
        of generic_formal_obj_decl_fields
    | `GenericFormalPackage
        of generic_formal_package_fields
    | `GenericFormalSubpDecl
        of generic_formal_subp_decl_fields
    | `GenericFormalTypeDecl
        of generic_formal_type_decl_fields
    | `ParamSpec
        of param_spec_fields
    | `SyntheticFormalParamDecl
        of synthetic_formal_param_decl_fields
    | `GenericPackageInternal
        of generic_package_internal_fields
    | `PackageDecl
        of package_decl_fields
    | `DiscreteBaseSubtypeDecl
        of discrete_base_subtype_decl_fields
    | `SubtypeDecl
        of subtype_decl_fields
    | `ClasswideTypeDecl
        of classwide_type_decl_fields
    | `IncompleteTypeDecl
        of incomplete_type_decl_fields
    | `IncompleteFormalTypeDecl
        of incomplete_formal_type_decl_fields
    | `IncompleteTaggedTypeDecl
        of incomplete_tagged_type_decl_fields
    | `ProtectedTypeDecl
        of protected_type_decl_fields
    | `TaskTypeDecl
        of task_type_decl_fields
    | `SingleTaskTypeDecl
        of single_task_type_decl_fields
    | `AnonymousTypeDecl
        of anonymous_type_decl_fields
    | `SynthAnonymousTypeDecl
        of synth_anonymous_type_decl_fields
    | `ConcreteTypeDecl
        of concrete_type_decl_fields
    | `FormalTypeDecl
        of formal_type_decl_fields
    | `AbstractSubpDecl
        of abstract_subp_decl_fields
    | `AbstractFormalSubpDecl
        of abstract_formal_subp_decl_fields
    | `ConcreteFormalSubpDecl
        of concrete_formal_subp_decl_fields
    | `SubpDecl
        of subp_decl_fields
    | `EntryDecl
        of entry_decl_fields
    | `EnumLiteralDecl
        of enum_literal_decl_fields
    | `SyntheticCharEnumLit
        of synthetic_char_enum_lit_fields
    | `GenericSubpInternal
        of generic_subp_internal_fields
    | `SyntheticSubpDecl
        of synthetic_subp_decl_fields
    | `AcceptStmtBody
        of accept_stmt_body_fields
    | `ExprFunction
        of expr_function_fields
    | `NullSubpDecl
        of null_subp_decl_fields
    | `SubpBody
        of subp_body_fields
    | `SubpRenamingDecl
        of subp_renaming_decl_fields
    | `PackageBodyStub
        of package_body_stub_fields
    | `ProtectedBodyStub
        of protected_body_stub_fields
    | `SubpBodyStub
        of subp_body_stub_fields
    | `TaskBodyStub
        of task_body_stub_fields
    | `EntryBody
        of entry_body_fields
    | `PackageBody
        of package_body_fields
    | `ProtectedBody
        of protected_body_fields
    | `TaskBody
        of task_body_fields
    | `EntryIndexSpec
        of entry_index_spec_fields
    | `ErrorDecl
        of error_decl_fields
    | `ExceptionDecl
        of exception_decl_fields
    | `ExceptionHandler
        of exception_handler_fields
    | `ForLoopVarDecl
        of for_loop_var_decl_fields
    | `GenericPackageDecl
        of generic_package_decl_fields
    | `GenericSubpDecl
        of generic_subp_decl_fields
    | `GenericPackageInstantiation
        of generic_package_instantiation_fields
    | `GenericSubpInstantiation
        of generic_subp_instantiation_fields
    | `GenericPackageRenamingDecl
        of generic_package_renaming_decl_fields
    | `GenericSubpRenamingDecl
        of generic_subp_renaming_decl_fields
    | `LabelDecl
        of label_decl_fields
    | `NamedStmtDecl
        of named_stmt_decl_fields
    | `NumberDecl
        of number_decl_fields
    | `ObjectDecl
        of object_decl_fields
    | `ExtendedReturnStmtObjectDecl
        of extended_return_stmt_object_decl_fields
    | `NoTypeObjectRenamingDecl
        of no_type_object_renaming_decl_fields
    | `PackageRenamingDecl
        of package_renaming_decl_fields
    | `SingleProtectedDecl
        of single_protected_decl_fields
    | `SingleTaskDecl
        of single_task_decl_fields
    ]

  
   
  and abstract_state_decl =
    [
    | `AbstractStateDecl
        of abstract_state_decl_fields
    ]
  and abstract_state_decl_fields = 
  {
         
    f_name: defining_name
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and anonymous_expr_decl =
    [
    | `AnonymousExprDecl
        of anonymous_expr_decl_fields
    ]
  and anonymous_expr_decl_fields = 
  {
         
    f_expr: expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * component_decl
    * discriminant_spec
    * generic_formal
    * param_spec
    * synthetic_formal_param_decl
    *)
  and base_formal_param_decl =
    [
    | `ComponentDecl
        of component_decl_fields
    | `DiscriminantSpec
        of discriminant_spec_fields
    | `GenericFormalObjDecl
        of generic_formal_obj_decl_fields
    | `GenericFormalPackage
        of generic_formal_package_fields
    | `GenericFormalSubpDecl
        of generic_formal_subp_decl_fields
    | `GenericFormalTypeDecl
        of generic_formal_type_decl_fields
    | `ParamSpec
        of param_spec_fields
    | `SyntheticFormalParamDecl
        of synthetic_formal_param_decl_fields
    ]

  
   
  and component_decl =
    [
    | `ComponentDecl
        of component_decl_fields
    ]
  and component_decl_fields = 
  {
         
    f_ids: defining_name_list
    Lazy.t;
         
    f_component_def: component_def
    Lazy.t;
         
    f_default_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and discriminant_spec =
    [
    | `DiscriminantSpec
        of discriminant_spec_fields
    ]
  and discriminant_spec_fields = 
  {
         
    f_ids: defining_name_list
    Lazy.t;
         
    f_type_expr: [
      | `AnonymousType
          of anonymous_type_fields
      | `ConstrainedSubtypeIndication
          of constrained_subtype_indication_fields
      | `DiscreteSubtypeIndication
          of discrete_subtype_indication_fields
      | `SubtypeIndication
          of subtype_indication_fields
    ]
    Lazy.t;
         
    f_default_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_formal =
    [
    | `GenericFormalObjDecl
        of generic_formal_obj_decl_fields
    | `GenericFormalPackage
        of generic_formal_package_fields
    | `GenericFormalSubpDecl
        of generic_formal_subp_decl_fields
    | `GenericFormalTypeDecl
        of generic_formal_type_decl_fields
    ]

  
   
  and generic_formal_obj_decl =
    [
    | `GenericFormalObjDecl
        of generic_formal_obj_decl_fields
    ]
  and generic_formal_obj_decl_fields = 
  {
         
    f_decl: [
      | `AbstractFormalSubpDecl
          of abstract_formal_subp_decl_fields
      | `ConcreteFormalSubpDecl
          of concrete_formal_subp_decl_fields
      | `ExtendedReturnStmtObjectDecl
          of extended_return_stmt_object_decl_fields
      | `FormalTypeDecl
          of formal_type_decl_fields
      | `GenericPackageInstantiation
          of generic_package_instantiation_fields
      | `GenericSubpInstantiation
          of generic_subp_instantiation_fields
      | `IncompleteFormalTypeDecl
          of incomplete_formal_type_decl_fields
      | `NoTypeObjectRenamingDecl
          of no_type_object_renaming_decl_fields
      | `NumberDecl
          of number_decl_fields
      | `ObjectDecl
          of object_decl_fields
      | `SingleProtectedDecl
          of single_protected_decl_fields
      | `SingleTaskDecl
          of single_task_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_formal_package =
    [
    | `GenericFormalPackage
        of generic_formal_package_fields
    ]
  and generic_formal_package_fields = 
  {
         
    f_decl: [
      | `AbstractFormalSubpDecl
          of abstract_formal_subp_decl_fields
      | `ConcreteFormalSubpDecl
          of concrete_formal_subp_decl_fields
      | `ExtendedReturnStmtObjectDecl
          of extended_return_stmt_object_decl_fields
      | `FormalTypeDecl
          of formal_type_decl_fields
      | `GenericPackageInstantiation
          of generic_package_instantiation_fields
      | `GenericSubpInstantiation
          of generic_subp_instantiation_fields
      | `IncompleteFormalTypeDecl
          of incomplete_formal_type_decl_fields
      | `NoTypeObjectRenamingDecl
          of no_type_object_renaming_decl_fields
      | `NumberDecl
          of number_decl_fields
      | `ObjectDecl
          of object_decl_fields
      | `SingleProtectedDecl
          of single_protected_decl_fields
      | `SingleTaskDecl
          of single_task_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_formal_subp_decl =
    [
    | `GenericFormalSubpDecl
        of generic_formal_subp_decl_fields
    ]
  and generic_formal_subp_decl_fields = 
  {
         
    f_decl: [
      | `AbstractFormalSubpDecl
          of abstract_formal_subp_decl_fields
      | `ConcreteFormalSubpDecl
          of concrete_formal_subp_decl_fields
      | `ExtendedReturnStmtObjectDecl
          of extended_return_stmt_object_decl_fields
      | `FormalTypeDecl
          of formal_type_decl_fields
      | `GenericPackageInstantiation
          of generic_package_instantiation_fields
      | `GenericSubpInstantiation
          of generic_subp_instantiation_fields
      | `IncompleteFormalTypeDecl
          of incomplete_formal_type_decl_fields
      | `NoTypeObjectRenamingDecl
          of no_type_object_renaming_decl_fields
      | `NumberDecl
          of number_decl_fields
      | `ObjectDecl
          of object_decl_fields
      | `SingleProtectedDecl
          of single_protected_decl_fields
      | `SingleTaskDecl
          of single_task_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_formal_type_decl =
    [
    | `GenericFormalTypeDecl
        of generic_formal_type_decl_fields
    ]
  and generic_formal_type_decl_fields = 
  {
         
    f_decl: [
      | `AbstractFormalSubpDecl
          of abstract_formal_subp_decl_fields
      | `ConcreteFormalSubpDecl
          of concrete_formal_subp_decl_fields
      | `ExtendedReturnStmtObjectDecl
          of extended_return_stmt_object_decl_fields
      | `FormalTypeDecl
          of formal_type_decl_fields
      | `GenericPackageInstantiation
          of generic_package_instantiation_fields
      | `GenericSubpInstantiation
          of generic_subp_instantiation_fields
      | `IncompleteFormalTypeDecl
          of incomplete_formal_type_decl_fields
      | `NoTypeObjectRenamingDecl
          of no_type_object_renaming_decl_fields
      | `NumberDecl
          of number_decl_fields
      | `ObjectDecl
          of object_decl_fields
      | `SingleProtectedDecl
          of single_protected_decl_fields
      | `SingleTaskDecl
          of single_task_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and param_spec =
    [
    | `ParamSpec
        of param_spec_fields
    ]
  and param_spec_fields = 
  {
         
    f_ids: defining_name_list
    Lazy.t;
         
    f_has_aliased: aliased_node
    Lazy.t;
         
    f_mode: mode
    option
    Lazy.t;
         
    f_type_expr: [
      | `AnonymousType
          of anonymous_type_fields
      | `ConstrainedSubtypeIndication
          of constrained_subtype_indication_fields
      | `DiscreteSubtypeIndication
          of discrete_subtype_indication_fields
      | `SubtypeIndication
          of subtype_indication_fields
    ]
    Lazy.t;
         
    f_default_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and synthetic_formal_param_decl =
    [
    | `SyntheticFormalParamDecl
        of synthetic_formal_param_decl_fields
    ]
  and synthetic_formal_param_decl_fields = 
  {
         
    f_param_type: type_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_package_decl =
    [
    | `GenericPackageInternal
        of generic_package_internal_fields
    | `PackageDecl
        of package_decl_fields
    ]

  
   
  and generic_package_internal =
    [
    | `GenericPackageInternal
        of generic_package_internal_fields
    ]
  and generic_package_internal_fields = 
  {
         
    f_package_name: defining_name
    Lazy.t;
         
    f_public_part: public_part
    Lazy.t;
         
    f_private_part: private_part
    option
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and package_decl =
    [
    | `PackageDecl
        of package_decl_fields
    ]
  and package_decl_fields = 
  {
         
    f_package_name: defining_name
    Lazy.t;
         
    f_public_part: public_part
    Lazy.t;
         
    f_private_part: private_part
    option
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * base_subtype_decl
    * classwide_type_decl
    * incomplete_type_decl
    * protected_type_decl
    * task_type_decl
    * type_decl
    *)
  and base_type_decl =
    [
    | `DiscreteBaseSubtypeDecl
        of discrete_base_subtype_decl_fields
    | `SubtypeDecl
        of subtype_decl_fields
    | `ClasswideTypeDecl
        of classwide_type_decl_fields
    | `IncompleteTypeDecl
        of incomplete_type_decl_fields
    | `IncompleteFormalTypeDecl
        of incomplete_formal_type_decl_fields
    | `IncompleteTaggedTypeDecl
        of incomplete_tagged_type_decl_fields
    | `ProtectedTypeDecl
        of protected_type_decl_fields
    | `TaskTypeDecl
        of task_type_decl_fields
    | `SingleTaskTypeDecl
        of single_task_type_decl_fields
    | `AnonymousTypeDecl
        of anonymous_type_decl_fields
    | `SynthAnonymousTypeDecl
        of synth_anonymous_type_decl_fields
    | `ConcreteTypeDecl
        of concrete_type_decl_fields
    | `FormalTypeDecl
        of formal_type_decl_fields
    ]

  
   
  and base_subtype_decl =
    [
    | `DiscreteBaseSubtypeDecl
        of discrete_base_subtype_decl_fields
    | `SubtypeDecl
        of subtype_decl_fields
    ]

  
   
  and discrete_base_subtype_decl =
    [
    | `DiscreteBaseSubtypeDecl
        of discrete_base_subtype_decl_fields
    ]
  and discrete_base_subtype_decl_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and subtype_decl =
    [
    | `SubtypeDecl
        of subtype_decl_fields
    ]
  and subtype_decl_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
         
    f_subtype: subtype_indication
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and classwide_type_decl =
    [
    | `ClasswideTypeDecl
        of classwide_type_decl_fields
    ]
  and classwide_type_decl_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and incomplete_type_decl =
    [
    | `IncompleteTypeDecl
        of incomplete_type_decl_fields
    | `IncompleteFormalTypeDecl
        of incomplete_formal_type_decl_fields
    | `IncompleteTaggedTypeDecl
        of incomplete_tagged_type_decl_fields
    ]
  and incomplete_type_decl_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
         
    f_discriminants: discriminant_part
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and incomplete_formal_type_decl =
    [
    | `IncompleteFormalTypeDecl
        of incomplete_formal_type_decl_fields
    ]
  and incomplete_formal_type_decl_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
         
    f_discriminants: discriminant_part
    option
    Lazy.t;
         
    f_is_tagged: tagged_node
    option
    Lazy.t;
         
    f_default_type: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and incomplete_tagged_type_decl =
    [
    | `IncompleteTaggedTypeDecl
        of incomplete_tagged_type_decl_fields
    ]
  and incomplete_tagged_type_decl_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
         
    f_discriminants: discriminant_part
    option
    Lazy.t;
         
    f_has_abstract: abstract_node
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and protected_type_decl =
    [
    | `ProtectedTypeDecl
        of protected_type_decl_fields
    ]
  and protected_type_decl_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
         
    f_discriminants: discriminant_part
    option
    Lazy.t;
         
    f_interfaces: parent_list
    Lazy.t;
         
    f_definition: protected_def
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and task_type_decl =
    [
    | `TaskTypeDecl
        of task_type_decl_fields
    | `SingleTaskTypeDecl
        of single_task_type_decl_fields
    ]
  and task_type_decl_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
         
    f_discriminants: discriminant_part
    option
    Lazy.t;
         
    f_definition: task_def
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and single_task_type_decl =
    [
    | `SingleTaskTypeDecl
        of single_task_type_decl_fields
    ]
  and single_task_type_decl_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
         
    f_discriminants: discriminant_part
    option
    Lazy.t;
         
    f_definition: task_def
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * anonymous_type_decl
    * concrete_type_decl
    * formal_type_decl
    *)
  and type_decl =
    [
    | `AnonymousTypeDecl
        of anonymous_type_decl_fields
    | `SynthAnonymousTypeDecl
        of synth_anonymous_type_decl_fields
    | `ConcreteTypeDecl
        of concrete_type_decl_fields
    | `FormalTypeDecl
        of formal_type_decl_fields
    ]

  
   
  and anonymous_type_decl =
    [
    | `AnonymousTypeDecl
        of anonymous_type_decl_fields
    | `SynthAnonymousTypeDecl
        of synth_anonymous_type_decl_fields
    ]
  and anonymous_type_decl_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
         
    f_discriminants: discriminant_part
    option
    Lazy.t;
         
    f_type_def: type_def
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and synth_anonymous_type_decl =
    [
    | `SynthAnonymousTypeDecl
        of synth_anonymous_type_decl_fields
    ]
  and synth_anonymous_type_decl_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
         
    f_discriminants: discriminant_part
    option
    Lazy.t;
         
    f_type_def: type_def
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and concrete_type_decl =
    [
    | `ConcreteTypeDecl
        of concrete_type_decl_fields
    ]
  and concrete_type_decl_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
         
    f_discriminants: discriminant_part
    option
    Lazy.t;
         
    f_type_def: type_def
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and formal_type_decl =
    [
    | `FormalTypeDecl
        of formal_type_decl_fields
    ]
  and formal_type_decl_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
         
    f_discriminants: discriminant_part
    option
    Lazy.t;
         
    f_type_def: type_def
    Lazy.t;
         
    f_default_type: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * classic_subp_decl
    * entry_decl
    * enum_literal_decl
    * generic_subp_internal
    * synthetic_subp_decl
    *)
  and basic_subp_decl =
    [
    | `AbstractSubpDecl
        of abstract_subp_decl_fields
    | `AbstractFormalSubpDecl
        of abstract_formal_subp_decl_fields
    | `ConcreteFormalSubpDecl
        of concrete_formal_subp_decl_fields
    | `SubpDecl
        of subp_decl_fields
    | `EntryDecl
        of entry_decl_fields
    | `EnumLiteralDecl
        of enum_literal_decl_fields
    | `SyntheticCharEnumLit
        of synthetic_char_enum_lit_fields
    | `GenericSubpInternal
        of generic_subp_internal_fields
    | `SyntheticSubpDecl
        of synthetic_subp_decl_fields
    ]

  
   
  (**
    * abstract_subp_decl
    * formal_subp_decl
    * subp_decl
    *)
  and classic_subp_decl =
    [
    | `AbstractSubpDecl
        of abstract_subp_decl_fields
    | `AbstractFormalSubpDecl
        of abstract_formal_subp_decl_fields
    | `ConcreteFormalSubpDecl
        of concrete_formal_subp_decl_fields
    | `SubpDecl
        of subp_decl_fields
    ]

  
   
  and abstract_subp_decl =
    [
    | `AbstractSubpDecl
        of abstract_subp_decl_fields
    ]
  and abstract_subp_decl_fields = 
  {
         
    f_overriding: overriding_node
    Lazy.t;
         
    f_subp_spec: subp_spec
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and formal_subp_decl =
    [
    | `AbstractFormalSubpDecl
        of abstract_formal_subp_decl_fields
    | `ConcreteFormalSubpDecl
        of concrete_formal_subp_decl_fields
    ]

  
   
  and abstract_formal_subp_decl =
    [
    | `AbstractFormalSubpDecl
        of abstract_formal_subp_decl_fields
    ]
  and abstract_formal_subp_decl_fields = 
  {
         
    f_overriding: overriding_node
    Lazy.t;
         
    f_subp_spec: subp_spec
    Lazy.t;
         
    f_default_expr: [
      | `AttributeRef
          of attribute_ref_fields
      | `BoxExpr
          of box_expr_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `NullLiteral
          of null_literal_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and concrete_formal_subp_decl =
    [
    | `ConcreteFormalSubpDecl
        of concrete_formal_subp_decl_fields
    ]
  and concrete_formal_subp_decl_fields = 
  {
         
    f_overriding: overriding_node
    Lazy.t;
         
    f_subp_spec: subp_spec
    Lazy.t;
         
    f_default_expr: [
      | `AttributeRef
          of attribute_ref_fields
      | `BoxExpr
          of box_expr_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `NullLiteral
          of null_literal_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and subp_decl =
    [
    | `SubpDecl
        of subp_decl_fields
    ]
  and subp_decl_fields = 
  {
         
    f_overriding: overriding_node
    Lazy.t;
         
    f_subp_spec: subp_spec
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and entry_decl =
    [
    | `EntryDecl
        of entry_decl_fields
    ]
  and entry_decl_fields = 
  {
         
    f_overriding: overriding_node
    Lazy.t;
         
    f_spec: entry_spec
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_literal_decl =
    [
    | `EnumLiteralDecl
        of enum_literal_decl_fields
    | `SyntheticCharEnumLit
        of synthetic_char_enum_lit_fields
    ]
  and enum_literal_decl_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and synthetic_char_enum_lit =
    [
    | `SyntheticCharEnumLit
        of synthetic_char_enum_lit_fields
    ]
  and synthetic_char_enum_lit_fields = 
  {
         
    f_name: defining_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_subp_internal =
    [
    | `GenericSubpInternal
        of generic_subp_internal_fields
    ]
  and generic_subp_internal_fields = 
  {
         
    f_subp_spec: subp_spec
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and synthetic_subp_decl =
    [
    | `SyntheticSubpDecl
        of synthetic_subp_decl_fields
    ]
  and synthetic_subp_decl_fields = 
  {
         
    f_spec: [
      | `SyntheticBinarySpec
          of synthetic_binary_spec_fields
      | `SyntheticUnarySpec
          of synthetic_unary_spec_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * accept_stmt_body
    * base_subp_body
    * body_stub
    * entry_body
    * package_body
    * protected_body
    * task_body
    *)
  and body_node =
    [
    | `AcceptStmtBody
        of accept_stmt_body_fields
    | `ExprFunction
        of expr_function_fields
    | `NullSubpDecl
        of null_subp_decl_fields
    | `SubpBody
        of subp_body_fields
    | `SubpRenamingDecl
        of subp_renaming_decl_fields
    | `PackageBodyStub
        of package_body_stub_fields
    | `ProtectedBodyStub
        of protected_body_stub_fields
    | `SubpBodyStub
        of subp_body_stub_fields
    | `TaskBodyStub
        of task_body_stub_fields
    | `EntryBody
        of entry_body_fields
    | `PackageBody
        of package_body_fields
    | `ProtectedBody
        of protected_body_fields
    | `TaskBody
        of task_body_fields
    ]

  
   
  and accept_stmt_body =
    [
    | `AcceptStmtBody
        of accept_stmt_body_fields
    ]
  and accept_stmt_body_fields = 
  {
         
    f_name: defining_name
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_subp_body =
    [
    | `ExprFunction
        of expr_function_fields
    | `NullSubpDecl
        of null_subp_decl_fields
    | `SubpBody
        of subp_body_fields
    | `SubpRenamingDecl
        of subp_renaming_decl_fields
    ]

  
   
  and expr_function =
    [
    | `ExprFunction
        of expr_function_fields
    ]
  and expr_function_fields = 
  {
         
    f_overriding: overriding_node
    Lazy.t;
         
    f_subp_spec: subp_spec
    Lazy.t;
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and null_subp_decl =
    [
    | `NullSubpDecl
        of null_subp_decl_fields
    ]
  and null_subp_decl_fields = 
  {
         
    f_overriding: overriding_node
    Lazy.t;
         
    f_subp_spec: subp_spec
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and subp_body =
    [
    | `SubpBody
        of subp_body_fields
    ]
  and subp_body_fields = 
  {
         
    f_overriding: overriding_node
    Lazy.t;
         
    f_subp_spec: subp_spec
    Lazy.t;
         
    f_decls: declarative_part
    Lazy.t;
         
    f_stmts: handled_stmts
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and subp_renaming_decl =
    [
    | `SubpRenamingDecl
        of subp_renaming_decl_fields
    ]
  and subp_renaming_decl_fields = 
  {
         
    f_overriding: overriding_node
    Lazy.t;
         
    f_subp_spec: subp_spec
    Lazy.t;
         
    f_renames: renaming_clause
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and body_stub =
    [
    | `PackageBodyStub
        of package_body_stub_fields
    | `ProtectedBodyStub
        of protected_body_stub_fields
    | `SubpBodyStub
        of subp_body_stub_fields
    | `TaskBodyStub
        of task_body_stub_fields
    ]

  
   
  and package_body_stub =
    [
    | `PackageBodyStub
        of package_body_stub_fields
    ]
  and package_body_stub_fields = 
  {
         
    f_name: defining_name
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and protected_body_stub =
    [
    | `ProtectedBodyStub
        of protected_body_stub_fields
    ]
  and protected_body_stub_fields = 
  {
         
    f_name: defining_name
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and subp_body_stub =
    [
    | `SubpBodyStub
        of subp_body_stub_fields
    ]
  and subp_body_stub_fields = 
  {
         
    f_overriding: overriding_node
    Lazy.t;
         
    f_subp_spec: subp_spec
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and task_body_stub =
    [
    | `TaskBodyStub
        of task_body_stub_fields
    ]
  and task_body_stub_fields = 
  {
         
    f_name: defining_name
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and entry_body =
    [
    | `EntryBody
        of entry_body_fields
    ]
  and entry_body_fields = 
  {
         
    f_entry_name: defining_name
    Lazy.t;
         
    f_index_spec: entry_index_spec
    option
    Lazy.t;
         
    f_params: entry_completion_formal_params
    Lazy.t;
         
    f_barrier: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_decls: declarative_part
    Lazy.t;
         
    f_stmts: handled_stmts
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and package_body =
    [
    | `PackageBody
        of package_body_fields
    ]
  and package_body_fields = 
  {
         
    f_package_name: defining_name
    Lazy.t;
         
    f_decls: declarative_part
    Lazy.t;
         
    f_stmts: handled_stmts
    option
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and protected_body =
    [
    | `ProtectedBody
        of protected_body_fields
    ]
  and protected_body_fields = 
  {
         
    f_name: defining_name
    Lazy.t;
         
    f_decls: declarative_part
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and task_body =
    [
    | `TaskBody
        of task_body_fields
    ]
  and task_body_fields = 
  {
         
    f_name: defining_name
    Lazy.t;
         
    f_decls: declarative_part
    Lazy.t;
         
    f_stmts: handled_stmts
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and entry_index_spec =
    [
    | `EntryIndexSpec
        of entry_index_spec_fields
    ]
  and entry_index_spec_fields = 
  {
         
    f_id: defining_name
    Lazy.t;
         
    f_subtype: [
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConstrainedSubtypeIndication
          of constrained_subtype_indication_fields
      | `DiscreteSubtypeIndication
          of discrete_subtype_indication_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `SubtypeIndication
          of subtype_indication_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and error_decl =
    [
    | `ErrorDecl
        of error_decl_fields
    ]
  and error_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and exception_decl =
    [
    | `ExceptionDecl
        of exception_decl_fields
    ]
  and exception_decl_fields = 
  {
         
    f_ids: defining_name_list
    Lazy.t;
         
    f_renames: renaming_clause
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and exception_handler =
    [
    | `ExceptionHandler
        of exception_handler_fields
    ]
  and exception_handler_fields = 
  {
         
    f_exception_name: defining_name
    option
    Lazy.t;
         
    f_handled_exceptions: alternatives_list
    Lazy.t;
         
    f_stmts: stmt_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and for_loop_var_decl =
    [
    | `ForLoopVarDecl
        of for_loop_var_decl_fields
    ]
  and for_loop_var_decl_fields = 
  {
         
    f_id: defining_name
    Lazy.t;
         
    f_id_type: [
      | `AnonymousType
          of anonymous_type_fields
      | `ConstrainedSubtypeIndication
          of constrained_subtype_indication_fields
      | `DiscreteSubtypeIndication
          of discrete_subtype_indication_fields
      | `SubtypeIndication
          of subtype_indication_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_decl =
    [
    | `GenericPackageDecl
        of generic_package_decl_fields
    | `GenericSubpDecl
        of generic_subp_decl_fields
    ]

  
   
  and generic_package_decl =
    [
    | `GenericPackageDecl
        of generic_package_decl_fields
    ]
  and generic_package_decl_fields = 
  {
         
    f_formal_part: generic_formal_part
    Lazy.t;
         
    f_package_decl: generic_package_internal
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_subp_decl =
    [
    | `GenericSubpDecl
        of generic_subp_decl_fields
    ]
  and generic_subp_decl_fields = 
  {
         
    f_formal_part: generic_formal_part
    Lazy.t;
         
    f_subp_decl: generic_subp_internal
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_instantiation =
    [
    | `GenericPackageInstantiation
        of generic_package_instantiation_fields
    | `GenericSubpInstantiation
        of generic_subp_instantiation_fields
    ]

  
   
  and generic_package_instantiation =
    [
    | `GenericPackageInstantiation
        of generic_package_instantiation_fields
    ]
  and generic_package_instantiation_fields = 
  {
         
    f_name: defining_name
    Lazy.t;
         
    f_generic_pkg_name: [
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
         
    f_params: basic_assoc_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_subp_instantiation =
    [
    | `GenericSubpInstantiation
        of generic_subp_instantiation_fields
    ]
  and generic_subp_instantiation_fields = 
  {
         
    f_overriding: overriding_node
    Lazy.t;
         
    f_kind: subp_kind
    Lazy.t;
         
    f_subp_name: defining_name
    Lazy.t;
         
    f_generic_subp_name: [
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
         
    f_params: basic_assoc_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_renaming_decl =
    [
    | `GenericPackageRenamingDecl
        of generic_package_renaming_decl_fields
    | `GenericSubpRenamingDecl
        of generic_subp_renaming_decl_fields
    ]

  
   
  and generic_package_renaming_decl =
    [
    | `GenericPackageRenamingDecl
        of generic_package_renaming_decl_fields
    ]
  and generic_package_renaming_decl_fields = 
  {
         
    f_name: defining_name
    Lazy.t;
         
    f_renames: [
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_subp_renaming_decl =
    [
    | `GenericSubpRenamingDecl
        of generic_subp_renaming_decl_fields
    ]
  and generic_subp_renaming_decl_fields = 
  {
         
    f_kind: subp_kind
    Lazy.t;
         
    f_name: defining_name
    Lazy.t;
         
    f_renames: [
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and label_decl =
    [
    | `LabelDecl
        of label_decl_fields
    ]
  and label_decl_fields = 
  {
         
    f_name: defining_name
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and named_stmt_decl =
    [
    | `NamedStmtDecl
        of named_stmt_decl_fields
    ]
  and named_stmt_decl_fields = 
  {
         
    f_name: defining_name
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and number_decl =
    [
    | `NumberDecl
        of number_decl_fields
    ]
  and number_decl_fields = 
  {
         
    f_ids: defining_name_list
    Lazy.t;
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and object_decl =
    [
    | `ObjectDecl
        of object_decl_fields
    | `ExtendedReturnStmtObjectDecl
        of extended_return_stmt_object_decl_fields
    | `NoTypeObjectRenamingDecl
        of no_type_object_renaming_decl_fields
    ]
  and object_decl_fields = 
  {
         
    f_ids: defining_name_list
    Lazy.t;
         
    f_has_aliased: aliased_node
    option
    Lazy.t;
         
    f_has_constant: constant_node
    option
    Lazy.t;
         
    f_mode: mode
    option
    Lazy.t;
         
    f_type_expr: [
      | `AnonymousType
          of anonymous_type_fields
      | `ConstrainedSubtypeIndication
          of constrained_subtype_indication_fields
      | `DiscreteSubtypeIndication
          of discrete_subtype_indication_fields
      | `SubtypeIndication
          of subtype_indication_fields
    ]
    option
    Lazy.t;
         
    f_default_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_renaming_clause: renaming_clause
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and extended_return_stmt_object_decl =
    [
    | `ExtendedReturnStmtObjectDecl
        of extended_return_stmt_object_decl_fields
    ]
  and extended_return_stmt_object_decl_fields = 
  {
         
    f_ids: defining_name_list
    Lazy.t;
         
    f_has_aliased: aliased_node
    option
    Lazy.t;
         
    f_has_constant: constant_node
    option
    Lazy.t;
         
    f_mode: mode
    option
    Lazy.t;
         
    f_type_expr: [
      | `AnonymousType
          of anonymous_type_fields
      | `ConstrainedSubtypeIndication
          of constrained_subtype_indication_fields
      | `DiscreteSubtypeIndication
          of discrete_subtype_indication_fields
      | `SubtypeIndication
          of subtype_indication_fields
    ]
    option
    Lazy.t;
         
    f_default_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_renaming_clause: renaming_clause
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and no_type_object_renaming_decl =
    [
    | `NoTypeObjectRenamingDecl
        of no_type_object_renaming_decl_fields
    ]
  and no_type_object_renaming_decl_fields = 
  {
         
    f_ids: defining_name_list
    Lazy.t;
         
    f_has_aliased: aliased_node
    option
    Lazy.t;
         
    f_has_constant: constant_node
    option
    Lazy.t;
         
    f_mode: mode
    option
    Lazy.t;
         
    f_type_expr: [
      | `AnonymousType
          of anonymous_type_fields
      | `ConstrainedSubtypeIndication
          of constrained_subtype_indication_fields
      | `DiscreteSubtypeIndication
          of discrete_subtype_indication_fields
      | `SubtypeIndication
          of subtype_indication_fields
    ]
    option
    Lazy.t;
         
    f_default_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_renaming_clause: renaming_clause
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and package_renaming_decl =
    [
    | `PackageRenamingDecl
        of package_renaming_decl_fields
    ]
  and package_renaming_decl_fields = 
  {
         
    f_name: defining_name
    Lazy.t;
         
    f_renames: renaming_clause
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and single_protected_decl =
    [
    | `SingleProtectedDecl
        of single_protected_decl_fields
    ]
  and single_protected_decl_fields = 
  {
         
    f_name: defining_name
    Lazy.t;
         
    f_interfaces: parent_list
    Lazy.t;
         
    f_definition: protected_def
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and single_task_decl =
    [
    | `SingleTaskDecl
        of single_task_decl_fields
    ]
  and single_task_decl_fields = 
  {
         
    f_task_type: single_task_type_decl
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and case_stmt_alternative =
    [
    | `CaseStmtAlternative
        of case_stmt_alternative_fields
    ]
  and case_stmt_alternative_fields = 
  {
         
    f_choices: alternatives_list
    Lazy.t;
         
    f_stmts: stmt_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and compilation_unit =
    [
    | `CompilationUnit
        of compilation_unit_fields
    ]
  and compilation_unit_fields = 
  {
         
    f_prelude: ada_node_list
    Lazy.t;
         
    f_body: [
      | `LibraryItem
          of library_item_fields
      | `Subunit
          of subunit_fields
    ]
    Lazy.t;
         
    f_pragmas: pragma_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and component_clause =
    [
    | `ComponentClause
        of component_clause_fields
    ]
  and component_clause_fields = 
  {
         
    f_id: identifier
    Lazy.t;
         
    f_position: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_range: range_spec
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and component_def =
    [
    | `ComponentDef
        of component_def_fields
    ]
  and component_def_fields = 
  {
         
    f_has_aliased: aliased_node
    Lazy.t;
         
    f_has_constant: constant_node
    Lazy.t;
         
    f_type_expr: [
      | `AnonymousType
          of anonymous_type_fields
      | `ConstrainedSubtypeIndication
          of constrained_subtype_indication_fields
      | `DiscreteSubtypeIndication
          of discrete_subtype_indication_fields
      | `SubtypeIndication
          of subtype_indication_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and constant_node =
    [
    | `ConstantAbsent
        of constant_absent_fields
    | `ConstantPresent
        of constant_present_fields
    ]

  
   
  and constant_absent =
    [
    | `ConstantAbsent
        of constant_absent_fields
    ]
  and constant_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and constant_present =
    [
    | `ConstantPresent
        of constant_present_fields
    ]
  and constant_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and constraint_node =
    [
    | `CompositeConstraint
        of composite_constraint_fields
    | `DeltaConstraint
        of delta_constraint_fields
    | `DigitsConstraint
        of digits_constraint_fields
    | `RangeConstraint
        of range_constraint_fields
    ]

  
   
  and composite_constraint =
    [
    | `CompositeConstraint
        of composite_constraint_fields
    ]
  and composite_constraint_fields = 
  {
         
    f_constraints: basic_assoc_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and delta_constraint =
    [
    | `DeltaConstraint
        of delta_constraint_fields
    ]
  and delta_constraint_fields = 
  {
         
    f_digits: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_range: range_spec
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and digits_constraint =
    [
    | `DigitsConstraint
        of digits_constraint_fields
    ]
  and digits_constraint_fields = 
  {
         
    f_digits: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_range: range_spec
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and range_constraint =
    [
    | `RangeConstraint
        of range_constraint_fields
    ]
  and range_constraint_fields = 
  {
         
    f_range: range_spec
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and declarative_part =
    [
    | `DeclarativePart
        of declarative_part_fields
    | `PrivatePart
        of private_part_fields
    | `PublicPart
        of public_part_fields
    ]
  and declarative_part_fields = 
  {
         
    f_decls: ada_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and private_part =
    [
    | `PrivatePart
        of private_part_fields
    ]
  and private_part_fields = 
  {
         
    f_decls: ada_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and public_part =
    [
    | `PublicPart
        of public_part_fields
    ]
  and public_part_fields = 
  {
         
    f_decls: ada_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and elsif_expr_part =
    [
    | `ElsifExprPart
        of elsif_expr_part_fields
    ]
  and elsif_expr_part_fields = 
  {
         
    f_cond_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_then_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and elsif_stmt_part =
    [
    | `ElsifStmtPart
        of elsif_stmt_part_fields
    ]
  and elsif_stmt_part_fields = 
  {
         
    f_cond_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_stmts: stmt_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * abstract_state_decl_expr
    * allocator
    * base_aggregate
    * bin_op
    * box_expr
    * case_expr_alternative
    * concat_op
    * concat_operand
    * cond_expr
    * contract_cases
    * decl_expr
    * membership_expr
    * name
    * paren_expr
    * quantified_expr
    * raise_expr
    * un_op
    *)
  and expr =
    [
    | `AbstractStateDeclExpr
        of abstract_state_decl_expr_fields
    | `Allocator
        of allocator_fields
    | `Aggregate
        of aggregate_fields
    | `BracketAggregate
        of bracket_aggregate_fields
    | `DeltaAggregate
        of delta_aggregate_fields
    | `BracketDeltaAggregate
        of bracket_delta_aggregate_fields
    | `NullRecordAggregate
        of null_record_aggregate_fields
    | `BinOp
        of bin_op_fields
    | `RelationOp
        of relation_op_fields
    | `BoxExpr
        of box_expr_fields
    | `CaseExprAlternative
        of case_expr_alternative_fields
    | `ConcatOp
        of concat_op_fields
    | `ConcatOperand
        of concat_operand_fields
    | `CaseExpr
        of case_expr_fields
    | `IfExpr
        of if_expr_fields
    | `ContractCases
        of contract_cases_fields
    | `DeclExpr
        of decl_expr_fields
    | `MembershipExpr
        of membership_expr_fields
    | `AttributeRef
        of attribute_ref_fields
    | `CallExpr
        of call_expr_fields
    | `DefiningName
        of defining_name_fields
    | `SyntheticDefiningName
        of synthetic_defining_name_fields
    | `DiscreteSubtypeName
        of discrete_subtype_name_fields
    | `DottedName
        of dotted_name_fields
    | `EndName
        of end_name_fields
    | `ExplicitDeref
        of explicit_deref_fields
    | `QualExpr
        of qual_expr_fields
    | `ReduceAttributeRef
        of reduce_attribute_ref_fields
    | `CharLiteral
        of char_literal_fields
    | `Identifier
        of identifier_fields
    | `OpAbs
        of op_abs_fields
    | `OpAnd
        of op_and_fields
    | `OpAndThen
        of op_and_then_fields
    | `OpConcat
        of op_concat_fields
    | `OpDiv
        of op_div_fields
    | `OpDoubleDot
        of op_double_dot_fields
    | `OpEq
        of op_eq_fields
    | `OpGt
        of op_gt_fields
    | `OpGte
        of op_gte_fields
    | `OpIn
        of op_in_fields
    | `OpLt
        of op_lt_fields
    | `OpLte
        of op_lte_fields
    | `OpMinus
        of op_minus_fields
    | `OpMod
        of op_mod_fields
    | `OpMult
        of op_mult_fields
    | `OpNeq
        of op_neq_fields
    | `OpNot
        of op_not_fields
    | `OpNotIn
        of op_not_in_fields
    | `OpOr
        of op_or_fields
    | `OpOrElse
        of op_or_else_fields
    | `OpPlus
        of op_plus_fields
    | `OpPow
        of op_pow_fields
    | `OpRem
        of op_rem_fields
    | `OpXor
        of op_xor_fields
    | `StringLiteral
        of string_literal_fields
    | `NullLiteral
        of null_literal_fields
    | `IntLiteral
        of int_literal_fields
    | `RealLiteral
        of real_literal_fields
    | `SyntheticIdentifier
        of synthetic_identifier_fields
    | `TargetName
        of target_name_fields
    | `UpdateAttributeRef
        of update_attribute_ref_fields
    | `ParenExpr
        of paren_expr_fields
    | `QuantifiedExpr
        of quantified_expr_fields
    | `RaiseExpr
        of raise_expr_fields
    | `UnOp
        of un_op_fields
    ]

  
   
  and abstract_state_decl_expr =
    [
    | `AbstractStateDeclExpr
        of abstract_state_decl_expr_fields
    ]
  and abstract_state_decl_expr_fields = 
  {
         
    f_state_decl: [
      | `AbstractStateDecl
          of abstract_state_decl_fields
      | `MultiAbstractStateDecl
          of multi_abstract_state_decl_fields
      | `ParenAbstractStateDecl
          of paren_abstract_state_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and allocator =
    [
    | `Allocator
        of allocator_fields
    ]
  and allocator_fields = 
  {
         
    f_subpool: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_type_or_expr: [
      | `ConstrainedSubtypeIndication
          of constrained_subtype_indication_fields
      | `DiscreteSubtypeIndication
          of discrete_subtype_indication_fields
      | `QualExpr
          of qual_expr_fields
      | `SubtypeIndication
          of subtype_indication_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * aggregate
    * delta_aggregate
    * null_record_aggregate
    *)
  and base_aggregate =
    [
    | `Aggregate
        of aggregate_fields
    | `BracketAggregate
        of bracket_aggregate_fields
    | `DeltaAggregate
        of delta_aggregate_fields
    | `BracketDeltaAggregate
        of bracket_delta_aggregate_fields
    | `NullRecordAggregate
        of null_record_aggregate_fields
    ]

  
   
  and aggregate =
    [
    | `Aggregate
        of aggregate_fields
    | `BracketAggregate
        of bracket_aggregate_fields
    ]
  and aggregate_fields = 
  {
         
    f_ancestor_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_assocs: basic_assoc_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and bracket_aggregate =
    [
    | `BracketAggregate
        of bracket_aggregate_fields
    ]
  and bracket_aggregate_fields = 
  {
         
    f_ancestor_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_assocs: basic_assoc_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and delta_aggregate =
    [
    | `DeltaAggregate
        of delta_aggregate_fields
    | `BracketDeltaAggregate
        of bracket_delta_aggregate_fields
    ]
  and delta_aggregate_fields = 
  {
         
    f_ancestor_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_assocs: basic_assoc_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and bracket_delta_aggregate =
    [
    | `BracketDeltaAggregate
        of bracket_delta_aggregate_fields
    ]
  and bracket_delta_aggregate_fields = 
  {
         
    f_ancestor_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_assocs: basic_assoc_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and null_record_aggregate =
    [
    | `NullRecordAggregate
        of null_record_aggregate_fields
    ]
  and null_record_aggregate_fields = 
  {
         
    f_ancestor_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_assocs: basic_assoc_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and bin_op =
    [
    | `BinOp
        of bin_op_fields
    | `RelationOp
        of relation_op_fields
    ]
  and bin_op_fields = 
  {
         
    f_left: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_op: [
      | `OpAnd
          of op_and_fields
      | `OpAndThen
          of op_and_then_fields
      | `OpDiv
          of op_div_fields
      | `OpDoubleDot
          of op_double_dot_fields
      | `OpEq
          of op_eq_fields
      | `OpGt
          of op_gt_fields
      | `OpGte
          of op_gte_fields
      | `OpLt
          of op_lt_fields
      | `OpLte
          of op_lte_fields
      | `OpMinus
          of op_minus_fields
      | `OpMod
          of op_mod_fields
      | `OpMult
          of op_mult_fields
      | `OpNeq
          of op_neq_fields
      | `OpOr
          of op_or_fields
      | `OpOrElse
          of op_or_else_fields
      | `OpPlus
          of op_plus_fields
      | `OpPow
          of op_pow_fields
      | `OpRem
          of op_rem_fields
      | `OpXor
          of op_xor_fields
    ]
    Lazy.t;
         
    f_right: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and relation_op =
    [
    | `RelationOp
        of relation_op_fields
    ]
  and relation_op_fields = 
  {
         
    f_left: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_op: [
      | `OpAnd
          of op_and_fields
      | `OpAndThen
          of op_and_then_fields
      | `OpDiv
          of op_div_fields
      | `OpDoubleDot
          of op_double_dot_fields
      | `OpEq
          of op_eq_fields
      | `OpGt
          of op_gt_fields
      | `OpGte
          of op_gte_fields
      | `OpLt
          of op_lt_fields
      | `OpLte
          of op_lte_fields
      | `OpMinus
          of op_minus_fields
      | `OpMod
          of op_mod_fields
      | `OpMult
          of op_mult_fields
      | `OpNeq
          of op_neq_fields
      | `OpOr
          of op_or_fields
      | `OpOrElse
          of op_or_else_fields
      | `OpPlus
          of op_plus_fields
      | `OpPow
          of op_pow_fields
      | `OpRem
          of op_rem_fields
      | `OpXor
          of op_xor_fields
    ]
    Lazy.t;
         
    f_right: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and box_expr =
    [
    | `BoxExpr
        of box_expr_fields
    ]
  and box_expr_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and case_expr_alternative =
    [
    | `CaseExprAlternative
        of case_expr_alternative_fields
    ]
  and case_expr_alternative_fields = 
  {
         
    f_choices: alternatives_list
    Lazy.t;
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and concat_op =
    [
    | `ConcatOp
        of concat_op_fields
    ]
  and concat_op_fields = 
  {
         
    f_first_operand: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_other_operands: concat_operand_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and concat_operand =
    [
    | `ConcatOperand
        of concat_operand_fields
    ]
  and concat_operand_fields = 
  {
         
    f_operator: op_concat
    Lazy.t;
         
    f_operand: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and cond_expr =
    [
    | `CaseExpr
        of case_expr_fields
    | `IfExpr
        of if_expr_fields
    ]

  
   
  and case_expr =
    [
    | `CaseExpr
        of case_expr_fields
    ]
  and case_expr_fields = 
  {
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_cases: case_expr_alternative_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and if_expr =
    [
    | `IfExpr
        of if_expr_fields
    ]
  and if_expr_fields = 
  {
         
    f_cond_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_then_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_alternatives: elsif_expr_part_list
    Lazy.t;
         
    f_else_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and contract_cases =
    [
    | `ContractCases
        of contract_cases_fields
    ]
  and contract_cases_fields = 
  {
         
    f_contract_cases: contract_case_assoc_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and decl_expr =
    [
    | `DeclExpr
        of decl_expr_fields
    ]
  and decl_expr_fields = 
  {
         
    f_decls: basic_decl_list
    Lazy.t;
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and membership_expr =
    [
    | `MembershipExpr
        of membership_expr_fields
    ]
  and membership_expr_fields = 
  {
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_op: [
      | `OpIn
          of op_in_fields
      | `OpNotIn
          of op_not_in_fields
    ]
    Lazy.t;
         
    f_membership_exprs: expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * attribute_ref
    * call_expr
    * defining_name
    * discrete_subtype_name
    * dotted_name
    * end_name
    * explicit_deref
    * qual_expr
    * reduce_attribute_ref
    * single_tok_node
    * synthetic_identifier
    * target_name
    * update_attribute_ref
    *)
  and name =
    [
    | `AttributeRef
        of attribute_ref_fields
    | `CallExpr
        of call_expr_fields
    | `DefiningName
        of defining_name_fields
    | `SyntheticDefiningName
        of synthetic_defining_name_fields
    | `DiscreteSubtypeName
        of discrete_subtype_name_fields
    | `DottedName
        of dotted_name_fields
    | `EndName
        of end_name_fields
    | `ExplicitDeref
        of explicit_deref_fields
    | `QualExpr
        of qual_expr_fields
    | `ReduceAttributeRef
        of reduce_attribute_ref_fields
    | `CharLiteral
        of char_literal_fields
    | `Identifier
        of identifier_fields
    | `OpAbs
        of op_abs_fields
    | `OpAnd
        of op_and_fields
    | `OpAndThen
        of op_and_then_fields
    | `OpConcat
        of op_concat_fields
    | `OpDiv
        of op_div_fields
    | `OpDoubleDot
        of op_double_dot_fields
    | `OpEq
        of op_eq_fields
    | `OpGt
        of op_gt_fields
    | `OpGte
        of op_gte_fields
    | `OpIn
        of op_in_fields
    | `OpLt
        of op_lt_fields
    | `OpLte
        of op_lte_fields
    | `OpMinus
        of op_minus_fields
    | `OpMod
        of op_mod_fields
    | `OpMult
        of op_mult_fields
    | `OpNeq
        of op_neq_fields
    | `OpNot
        of op_not_fields
    | `OpNotIn
        of op_not_in_fields
    | `OpOr
        of op_or_fields
    | `OpOrElse
        of op_or_else_fields
    | `OpPlus
        of op_plus_fields
    | `OpPow
        of op_pow_fields
    | `OpRem
        of op_rem_fields
    | `OpXor
        of op_xor_fields
    | `StringLiteral
        of string_literal_fields
    | `NullLiteral
        of null_literal_fields
    | `IntLiteral
        of int_literal_fields
    | `RealLiteral
        of real_literal_fields
    | `SyntheticIdentifier
        of synthetic_identifier_fields
    | `TargetName
        of target_name_fields
    | `UpdateAttributeRef
        of update_attribute_ref_fields
    ]

  
   
  and attribute_ref =
    [
    | `AttributeRef
        of attribute_ref_fields
    ]
  and attribute_ref_fields = 
  {
         
    f_prefix: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_attribute: identifier
    Lazy.t;
         
    f_args: basic_assoc_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and call_expr =
    [
    | `CallExpr
        of call_expr_fields
    ]
  and call_expr_fields = 
  {
         
    f_name: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_suffix: [
      | `AssocList
          of assoc_list_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DiscreteSubtypeIndication
          of discrete_subtype_indication_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and defining_name =
    [
    | `DefiningName
        of defining_name_fields
    | `SyntheticDefiningName
        of synthetic_defining_name_fields
    ]
  and defining_name_fields = 
  {
         
    f_name: [
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
      | `SyntheticIdentifier
          of synthetic_identifier_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and synthetic_defining_name =
    [
    | `SyntheticDefiningName
        of synthetic_defining_name_fields
    ]
  and synthetic_defining_name_fields = 
  {
         
    f_name: [
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
      | `SyntheticIdentifier
          of synthetic_identifier_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and discrete_subtype_name =
    [
    | `DiscreteSubtypeName
        of discrete_subtype_name_fields
    ]
  and discrete_subtype_name_fields = 
  {
         
    f_subtype: discrete_subtype_indication
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and dotted_name =
    [
    | `DottedName
        of dotted_name_fields
    ]
  and dotted_name_fields = 
  {
         
    f_prefix: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_suffix: [
      | `CharLiteral
          of char_literal_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and end_name =
    [
    | `EndName
        of end_name_fields
    ]
  and end_name_fields = 
  {
         
    f_name: [
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and explicit_deref =
    [
    | `ExplicitDeref
        of explicit_deref_fields
    ]
  and explicit_deref_fields = 
  {
         
    f_prefix: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and qual_expr =
    [
    | `QualExpr
        of qual_expr_fields
    ]
  and qual_expr_fields = 
  {
         
    f_prefix: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_suffix: [
      | `Aggregate
          of aggregate_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and reduce_attribute_ref =
    [
    | `ReduceAttributeRef
        of reduce_attribute_ref_fields
    ]
  and reduce_attribute_ref_fields = 
  {
         
    f_prefix: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
      | `ValueSequence
          of value_sequence_fields
    ]
    Lazy.t;
         
    f_attribute: identifier
    Lazy.t;
         
    f_args: basic_assoc_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * base_id
    * null_literal
    * num_literal
    *)
  and single_tok_node =
    [
    | `CharLiteral
        of char_literal_fields
    | `Identifier
        of identifier_fields
    | `OpAbs
        of op_abs_fields
    | `OpAnd
        of op_and_fields
    | `OpAndThen
        of op_and_then_fields
    | `OpConcat
        of op_concat_fields
    | `OpDiv
        of op_div_fields
    | `OpDoubleDot
        of op_double_dot_fields
    | `OpEq
        of op_eq_fields
    | `OpGt
        of op_gt_fields
    | `OpGte
        of op_gte_fields
    | `OpIn
        of op_in_fields
    | `OpLt
        of op_lt_fields
    | `OpLte
        of op_lte_fields
    | `OpMinus
        of op_minus_fields
    | `OpMod
        of op_mod_fields
    | `OpMult
        of op_mult_fields
    | `OpNeq
        of op_neq_fields
    | `OpNot
        of op_not_fields
    | `OpNotIn
        of op_not_in_fields
    | `OpOr
        of op_or_fields
    | `OpOrElse
        of op_or_else_fields
    | `OpPlus
        of op_plus_fields
    | `OpPow
        of op_pow_fields
    | `OpRem
        of op_rem_fields
    | `OpXor
        of op_xor_fields
    | `StringLiteral
        of string_literal_fields
    | `NullLiteral
        of null_literal_fields
    | `IntLiteral
        of int_literal_fields
    | `RealLiteral
        of real_literal_fields
    ]

  
   
  (**
    * char_literal
    * identifier
    * op
    * string_literal
    *)
  and base_id =
    [
    | `CharLiteral
        of char_literal_fields
    | `Identifier
        of identifier_fields
    | `OpAbs
        of op_abs_fields
    | `OpAnd
        of op_and_fields
    | `OpAndThen
        of op_and_then_fields
    | `OpConcat
        of op_concat_fields
    | `OpDiv
        of op_div_fields
    | `OpDoubleDot
        of op_double_dot_fields
    | `OpEq
        of op_eq_fields
    | `OpGt
        of op_gt_fields
    | `OpGte
        of op_gte_fields
    | `OpIn
        of op_in_fields
    | `OpLt
        of op_lt_fields
    | `OpLte
        of op_lte_fields
    | `OpMinus
        of op_minus_fields
    | `OpMod
        of op_mod_fields
    | `OpMult
        of op_mult_fields
    | `OpNeq
        of op_neq_fields
    | `OpNot
        of op_not_fields
    | `OpNotIn
        of op_not_in_fields
    | `OpOr
        of op_or_fields
    | `OpOrElse
        of op_or_else_fields
    | `OpPlus
        of op_plus_fields
    | `OpPow
        of op_pow_fields
    | `OpRem
        of op_rem_fields
    | `OpXor
        of op_xor_fields
    | `StringLiteral
        of string_literal_fields
    ]

  
   
  and char_literal =
    [
    | `CharLiteral
        of char_literal_fields
    ]
  and char_literal_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and identifier =
    [
    | `Identifier
        of identifier_fields
    ]
  and identifier_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op =
    [
    | `OpAbs
        of op_abs_fields
    | `OpAnd
        of op_and_fields
    | `OpAndThen
        of op_and_then_fields
    | `OpConcat
        of op_concat_fields
    | `OpDiv
        of op_div_fields
    | `OpDoubleDot
        of op_double_dot_fields
    | `OpEq
        of op_eq_fields
    | `OpGt
        of op_gt_fields
    | `OpGte
        of op_gte_fields
    | `OpIn
        of op_in_fields
    | `OpLt
        of op_lt_fields
    | `OpLte
        of op_lte_fields
    | `OpMinus
        of op_minus_fields
    | `OpMod
        of op_mod_fields
    | `OpMult
        of op_mult_fields
    | `OpNeq
        of op_neq_fields
    | `OpNot
        of op_not_fields
    | `OpNotIn
        of op_not_in_fields
    | `OpOr
        of op_or_fields
    | `OpOrElse
        of op_or_else_fields
    | `OpPlus
        of op_plus_fields
    | `OpPow
        of op_pow_fields
    | `OpRem
        of op_rem_fields
    | `OpXor
        of op_xor_fields
    ]

  
   
  and op_abs =
    [
    | `OpAbs
        of op_abs_fields
    ]
  and op_abs_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_and =
    [
    | `OpAnd
        of op_and_fields
    ]
  and op_and_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_and_then =
    [
    | `OpAndThen
        of op_and_then_fields
    ]
  and op_and_then_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_concat =
    [
    | `OpConcat
        of op_concat_fields
    ]
  and op_concat_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_div =
    [
    | `OpDiv
        of op_div_fields
    ]
  and op_div_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_double_dot =
    [
    | `OpDoubleDot
        of op_double_dot_fields
    ]
  and op_double_dot_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_eq =
    [
    | `OpEq
        of op_eq_fields
    ]
  and op_eq_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_gt =
    [
    | `OpGt
        of op_gt_fields
    ]
  and op_gt_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_gte =
    [
    | `OpGte
        of op_gte_fields
    ]
  and op_gte_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_in =
    [
    | `OpIn
        of op_in_fields
    ]
  and op_in_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_lt =
    [
    | `OpLt
        of op_lt_fields
    ]
  and op_lt_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_lte =
    [
    | `OpLte
        of op_lte_fields
    ]
  and op_lte_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_minus =
    [
    | `OpMinus
        of op_minus_fields
    ]
  and op_minus_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_mod =
    [
    | `OpMod
        of op_mod_fields
    ]
  and op_mod_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_mult =
    [
    | `OpMult
        of op_mult_fields
    ]
  and op_mult_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_neq =
    [
    | `OpNeq
        of op_neq_fields
    ]
  and op_neq_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_not =
    [
    | `OpNot
        of op_not_fields
    ]
  and op_not_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_not_in =
    [
    | `OpNotIn
        of op_not_in_fields
    ]
  and op_not_in_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_or =
    [
    | `OpOr
        of op_or_fields
    ]
  and op_or_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_or_else =
    [
    | `OpOrElse
        of op_or_else_fields
    ]
  and op_or_else_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_plus =
    [
    | `OpPlus
        of op_plus_fields
    ]
  and op_plus_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_pow =
    [
    | `OpPow
        of op_pow_fields
    ]
  and op_pow_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_rem =
    [
    | `OpRem
        of op_rem_fields
    ]
  and op_rem_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_xor =
    [
    | `OpXor
        of op_xor_fields
    ]
  and op_xor_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and string_literal =
    [
    | `StringLiteral
        of string_literal_fields
    ]
  and string_literal_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and null_literal =
    [
    | `NullLiteral
        of null_literal_fields
    ]
  and null_literal_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and num_literal =
    [
    | `IntLiteral
        of int_literal_fields
    | `RealLiteral
        of real_literal_fields
    ]

  
   
  and int_literal =
    [
    | `IntLiteral
        of int_literal_fields
    ]
  and int_literal_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and real_literal =
    [
    | `RealLiteral
        of real_literal_fields
    ]
  and real_literal_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and synthetic_identifier =
    [
    | `SyntheticIdentifier
        of synthetic_identifier_fields
    ]
  and synthetic_identifier_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and target_name =
    [
    | `TargetName
        of target_name_fields
    ]
  and target_name_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and update_attribute_ref =
    [
    | `UpdateAttributeRef
        of update_attribute_ref_fields
    ]
  and update_attribute_ref_fields = 
  {
         
    f_prefix: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_attribute: identifier
    Lazy.t;
         
    f_values: base_aggregate
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and paren_expr =
    [
    | `ParenExpr
        of paren_expr_fields
    ]
  and paren_expr_fields = 
  {
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and quantified_expr =
    [
    | `QuantifiedExpr
        of quantified_expr_fields
    ]
  and quantified_expr_fields = 
  {
         
    f_quantifier: quantifier
    Lazy.t;
         
    f_loop_spec: for_loop_spec
    Lazy.t;
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and raise_expr =
    [
    | `RaiseExpr
        of raise_expr_fields
    ]
  and raise_expr_fields = 
  {
         
    f_exception_name: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_error_message: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and un_op =
    [
    | `UnOp
        of un_op_fields
    ]
  and un_op_fields = 
  {
         
    f_op: [
      | `OpAbs
          of op_abs_fields
      | `OpMinus
          of op_minus_fields
      | `OpNot
          of op_not_fields
      | `OpPlus
          of op_plus_fields
    ]
    Lazy.t;
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and handled_stmts =
    [
    | `HandledStmts
        of handled_stmts_fields
    ]
  and handled_stmts_fields = 
  {
         
    f_stmts: stmt_list
    Lazy.t;
         
    f_exceptions: ada_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and interface_kind =
    [
    | `InterfaceKindLimited
        of interface_kind_limited_fields
    | `InterfaceKindProtected
        of interface_kind_protected_fields
    | `InterfaceKindSynchronized
        of interface_kind_synchronized_fields
    | `InterfaceKindTask
        of interface_kind_task_fields
    ]

  
   
  and interface_kind_limited =
    [
    | `InterfaceKindLimited
        of interface_kind_limited_fields
    ]
  and interface_kind_limited_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and interface_kind_protected =
    [
    | `InterfaceKindProtected
        of interface_kind_protected_fields
    ]
  and interface_kind_protected_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and interface_kind_synchronized =
    [
    | `InterfaceKindSynchronized
        of interface_kind_synchronized_fields
    ]
  and interface_kind_synchronized_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and interface_kind_task =
    [
    | `InterfaceKindTask
        of interface_kind_task_fields
    ]
  and interface_kind_task_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and iter_type =
    [
    | `IterTypeIn
        of iter_type_in_fields
    | `IterTypeOf
        of iter_type_of_fields
    ]

  
   
  and iter_type_in =
    [
    | `IterTypeIn
        of iter_type_in_fields
    ]
  and iter_type_in_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and iter_type_of =
    [
    | `IterTypeOf
        of iter_type_of_fields
    ]
  and iter_type_of_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and library_item =
    [
    | `LibraryItem
        of library_item_fields
    ]
  and library_item_fields = 
  {
         
    f_has_private: private_node
    Lazy.t;
         
    f_item: [
      | `AbstractSubpDecl
          of abstract_subp_decl_fields
      | `ErrorDecl
          of error_decl_fields
      | `ExprFunction
          of expr_function_fields
      | `GenericPackageDecl
          of generic_package_decl_fields
      | `GenericPackageInstantiation
          of generic_package_instantiation_fields
      | `GenericPackageRenamingDecl
          of generic_package_renaming_decl_fields
      | `GenericSubpDecl
          of generic_subp_decl_fields
      | `GenericSubpInstantiation
          of generic_subp_instantiation_fields
      | `GenericSubpRenamingDecl
          of generic_subp_renaming_decl_fields
      | `NullSubpDecl
          of null_subp_decl_fields
      | `PackageBody
          of package_body_fields
      | `PackageDecl
          of package_decl_fields
      | `PackageRenamingDecl
          of package_renaming_decl_fields
      | `SubpBody
          of subp_body_fields
      | `SubpDecl
          of subp_decl_fields
      | `SubpRenamingDecl
          of subp_renaming_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and limited_node =
    [
    | `LimitedAbsent
        of limited_absent_fields
    | `LimitedPresent
        of limited_present_fields
    ]

  
   
  and limited_absent =
    [
    | `LimitedAbsent
        of limited_absent_fields
    ]
  and limited_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and limited_present =
    [
    | `LimitedPresent
        of limited_present_fields
    ]
  and limited_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and loop_spec =
    [
    | `ForLoopSpec
        of for_loop_spec_fields
    | `WhileLoopSpec
        of while_loop_spec_fields
    ]

  
   
  and for_loop_spec =
    [
    | `ForLoopSpec
        of for_loop_spec_fields
    ]
  and for_loop_spec_fields = 
  {
         
    f_var_decl: for_loop_var_decl
    Lazy.t;
         
    f_loop_type: iter_type
    Lazy.t;
         
    f_has_reverse: reverse_node
    Lazy.t;
         
    f_iter_expr: [
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DiscreteSubtypeIndication
          of discrete_subtype_indication_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_iter_filter: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and while_loop_spec =
    [
    | `WhileLoopSpec
        of while_loop_spec_fields
    ]
  and while_loop_spec_fields = 
  {
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and mode =
    [
    | `ModeDefault
        of mode_default_fields
    | `ModeIn
        of mode_in_fields
    | `ModeInOut
        of mode_in_out_fields
    | `ModeOut
        of mode_out_fields
    ]

  
   
  and mode_default =
    [
    | `ModeDefault
        of mode_default_fields
    ]
  and mode_default_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and mode_in =
    [
    | `ModeIn
        of mode_in_fields
    ]
  and mode_in_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and mode_in_out =
    [
    | `ModeInOut
        of mode_in_out_fields
    ]
  and mode_in_out_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and mode_out =
    [
    | `ModeOut
        of mode_out_fields
    ]
  and mode_out_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and multi_abstract_state_decl =
    [
    | `MultiAbstractStateDecl
        of multi_abstract_state_decl_fields
    ]
  and multi_abstract_state_decl_fields = 
  {
         
    f_decls: abstract_state_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and not_null =
    [
    | `NotNullAbsent
        of not_null_absent_fields
    | `NotNullPresent
        of not_null_present_fields
    ]

  
   
  and not_null_absent =
    [
    | `NotNullAbsent
        of not_null_absent_fields
    ]
  and not_null_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and not_null_present =
    [
    | `NotNullPresent
        of not_null_present_fields
    ]
  and not_null_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and null_component_decl =
    [
    | `NullComponentDecl
        of null_component_decl_fields
    ]
  and null_component_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and others_designator =
    [
    | `OthersDesignator
        of others_designator_fields
    ]
  and others_designator_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and overriding_node =
    [
    | `OverridingNotOverriding
        of overriding_not_overriding_fields
    | `OverridingOverriding
        of overriding_overriding_fields
    | `OverridingUnspecified
        of overriding_unspecified_fields
    ]

  
   
  and overriding_not_overriding =
    [
    | `OverridingNotOverriding
        of overriding_not_overriding_fields
    ]
  and overriding_not_overriding_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and overriding_overriding =
    [
    | `OverridingOverriding
        of overriding_overriding_fields
    ]
  and overriding_overriding_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and overriding_unspecified =
    [
    | `OverridingUnspecified
        of overriding_unspecified_fields
    ]
  and overriding_unspecified_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and params =
    [
    | `Params
        of params_fields
    ]
  and params_fields = 
  {
         
    f_params: param_spec_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and paren_abstract_state_decl =
    [
    | `ParenAbstractStateDecl
        of paren_abstract_state_decl_fields
    ]
  and paren_abstract_state_decl_fields = 
  {
         
    f_decl: [
      | `AbstractStateDecl
          of abstract_state_decl_fields
      | `ParenAbstractStateDecl
          of paren_abstract_state_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and pp_directive =
    [
    | `PpElseDirective
        of pp_else_directive_fields
    | `PpElsifDirective
        of pp_elsif_directive_fields
    | `PpEndIfDirective
        of pp_end_if_directive_fields
    | `PpIfDirective
        of pp_if_directive_fields
    ]

  
   
  and pp_else_directive =
    [
    | `PpElseDirective
        of pp_else_directive_fields
    ]
  and pp_else_directive_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and pp_elsif_directive =
    [
    | `PpElsifDirective
        of pp_elsif_directive_fields
    ]
  and pp_elsif_directive_fields = 
  {
         
    f_expr: [
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `Identifier
          of identifier_fields
      | `ParenExpr
          of paren_expr_fields
      | `RelationOp
          of relation_op_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_then_kw: pp_then_kw
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and pp_end_if_directive =
    [
    | `PpEndIfDirective
        of pp_end_if_directive_fields
    ]
  and pp_end_if_directive_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and pp_if_directive =
    [
    | `PpIfDirective
        of pp_if_directive_fields
    ]
  and pp_if_directive_fields = 
  {
         
    f_expr: [
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `Identifier
          of identifier_fields
      | `ParenExpr
          of paren_expr_fields
      | `RelationOp
          of relation_op_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_then_kw: pp_then_kw
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and pp_then_kw =
    [
    | `PpThenKw
        of pp_then_kw_fields
    ]
  and pp_then_kw_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and pragma_node =
    [
    | `PragmaNode
        of pragma_node_fields
    ]
  and pragma_node_fields = 
  {
         
    f_id: identifier
    Lazy.t;
         
    f_args: base_assoc_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and private_node =
    [
    | `PrivateAbsent
        of private_absent_fields
    | `PrivatePresent
        of private_present_fields
    ]

  
   
  and private_absent =
    [
    | `PrivateAbsent
        of private_absent_fields
    ]
  and private_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and private_present =
    [
    | `PrivatePresent
        of private_present_fields
    ]
  and private_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and protected_def =
    [
    | `ProtectedDef
        of protected_def_fields
    ]
  and protected_def_fields = 
  {
         
    f_public_part: public_part
    Lazy.t;
         
    f_private_part: private_part
    option
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and protected_node =
    [
    | `ProtectedAbsent
        of protected_absent_fields
    | `ProtectedPresent
        of protected_present_fields
    ]

  
   
  and protected_absent =
    [
    | `ProtectedAbsent
        of protected_absent_fields
    ]
  and protected_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and protected_present =
    [
    | `ProtectedPresent
        of protected_present_fields
    ]
  and protected_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and quantifier =
    [
    | `QuantifierAll
        of quantifier_all_fields
    | `QuantifierSome
        of quantifier_some_fields
    ]

  
   
  and quantifier_all =
    [
    | `QuantifierAll
        of quantifier_all_fields
    ]
  and quantifier_all_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and quantifier_some =
    [
    | `QuantifierSome
        of quantifier_some_fields
    ]
  and quantifier_some_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and range_spec =
    [
    | `RangeSpec
        of range_spec_fields
    ]
  and range_spec_fields = 
  {
         
    f_range: [
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BoxExpr
          of box_expr_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and renaming_clause =
    [
    | `RenamingClause
        of renaming_clause_fields
    | `SyntheticRenamingClause
        of synthetic_renaming_clause_fields
    ]
  and renaming_clause_fields = 
  {
         
    f_renamed_object: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and synthetic_renaming_clause =
    [
    | `SyntheticRenamingClause
        of synthetic_renaming_clause_fields
    ]
  and synthetic_renaming_clause_fields = 
  {
         
    f_renamed_object: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and reverse_node =
    [
    | `ReverseAbsent
        of reverse_absent_fields
    | `ReversePresent
        of reverse_present_fields
    ]

  
   
  and reverse_absent =
    [
    | `ReverseAbsent
        of reverse_absent_fields
    ]
  and reverse_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and reverse_present =
    [
    | `ReversePresent
        of reverse_present_fields
    ]
  and reverse_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and select_when_part =
    [
    | `SelectWhenPart
        of select_when_part_fields
    ]
  and select_when_part_fields = 
  {
         
    f_cond_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_stmts: stmt_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * composite_stmt
    * error_stmt
    * simple_stmt
    *)
  and stmt =
    [
    | `AcceptStmt
        of accept_stmt_fields
    | `AcceptStmtWithStmts
        of accept_stmt_with_stmts_fields
    | `ForLoopStmt
        of for_loop_stmt_fields
    | `LoopStmt
        of loop_stmt_fields
    | `WhileLoopStmt
        of while_loop_stmt_fields
    | `BeginBlock
        of begin_block_fields
    | `DeclBlock
        of decl_block_fields
    | `CaseStmt
        of case_stmt_fields
    | `ExtendedReturnStmt
        of extended_return_stmt_fields
    | `IfStmt
        of if_stmt_fields
    | `NamedStmt
        of named_stmt_fields
    | `SelectStmt
        of select_stmt_fields
    | `ErrorStmt
        of error_stmt_fields
    | `AbortStmt
        of abort_stmt_fields
    | `AssignStmt
        of assign_stmt_fields
    | `CallStmt
        of call_stmt_fields
    | `DelayStmt
        of delay_stmt_fields
    | `ExitStmt
        of exit_stmt_fields
    | `GotoStmt
        of goto_stmt_fields
    | `Label
        of label_fields
    | `NullStmt
        of null_stmt_fields
    | `RaiseStmt
        of raise_stmt_fields
    | `RequeueStmt
        of requeue_stmt_fields
    | `ReturnStmt
        of return_stmt_fields
    | `TerminateAlternative
        of terminate_alternative_fields
    ]

  
   
  (**
    * accept_stmt
    * base_loop_stmt
    * block_stmt
    * case_stmt
    * extended_return_stmt
    * if_stmt
    * named_stmt
    * select_stmt
    *)
  and composite_stmt =
    [
    | `AcceptStmt
        of accept_stmt_fields
    | `AcceptStmtWithStmts
        of accept_stmt_with_stmts_fields
    | `ForLoopStmt
        of for_loop_stmt_fields
    | `LoopStmt
        of loop_stmt_fields
    | `WhileLoopStmt
        of while_loop_stmt_fields
    | `BeginBlock
        of begin_block_fields
    | `DeclBlock
        of decl_block_fields
    | `CaseStmt
        of case_stmt_fields
    | `ExtendedReturnStmt
        of extended_return_stmt_fields
    | `IfStmt
        of if_stmt_fields
    | `NamedStmt
        of named_stmt_fields
    | `SelectStmt
        of select_stmt_fields
    ]

  
   
  and accept_stmt =
    [
    | `AcceptStmt
        of accept_stmt_fields
    | `AcceptStmtWithStmts
        of accept_stmt_with_stmts_fields
    ]
  and accept_stmt_fields = 
  {
         
    f_body_decl: accept_stmt_body
    Lazy.t;
         
    f_entry_index_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_params: entry_completion_formal_params
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and accept_stmt_with_stmts =
    [
    | `AcceptStmtWithStmts
        of accept_stmt_with_stmts_fields
    ]
  and accept_stmt_with_stmts_fields = 
  {
         
    f_body_decl: accept_stmt_body
    Lazy.t;
         
    f_entry_index_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_params: entry_completion_formal_params
    Lazy.t;
         
    f_stmts: handled_stmts
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_loop_stmt =
    [
    | `ForLoopStmt
        of for_loop_stmt_fields
    | `LoopStmt
        of loop_stmt_fields
    | `WhileLoopStmt
        of while_loop_stmt_fields
    ]

  
   
  and for_loop_stmt =
    [
    | `ForLoopStmt
        of for_loop_stmt_fields
    ]
  and for_loop_stmt_fields = 
  {
         
    f_spec: loop_spec
    option
    Lazy.t;
         
    f_stmts: stmt_list
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and loop_stmt =
    [
    | `LoopStmt
        of loop_stmt_fields
    ]
  and loop_stmt_fields = 
  {
         
    f_spec: loop_spec
    option
    Lazy.t;
         
    f_stmts: stmt_list
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and while_loop_stmt =
    [
    | `WhileLoopStmt
        of while_loop_stmt_fields
    ]
  and while_loop_stmt_fields = 
  {
         
    f_spec: loop_spec
    option
    Lazy.t;
         
    f_stmts: stmt_list
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and block_stmt =
    [
    | `BeginBlock
        of begin_block_fields
    | `DeclBlock
        of decl_block_fields
    ]

  
   
  and begin_block =
    [
    | `BeginBlock
        of begin_block_fields
    ]
  and begin_block_fields = 
  {
         
    f_stmts: handled_stmts
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and decl_block =
    [
    | `DeclBlock
        of decl_block_fields
    ]
  and decl_block_fields = 
  {
         
    f_decls: declarative_part
    Lazy.t;
         
    f_stmts: handled_stmts
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and case_stmt =
    [
    | `CaseStmt
        of case_stmt_fields
    ]
  and case_stmt_fields = 
  {
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_pragmas: pragma_node_list
    Lazy.t;
         
    f_alternatives: case_stmt_alternative_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and extended_return_stmt =
    [
    | `ExtendedReturnStmt
        of extended_return_stmt_fields
    ]
  and extended_return_stmt_fields = 
  {
         
    f_decl: extended_return_stmt_object_decl
    Lazy.t;
         
    f_stmts: handled_stmts
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and if_stmt =
    [
    | `IfStmt
        of if_stmt_fields
    ]
  and if_stmt_fields = 
  {
         
    f_cond_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_then_stmts: stmt_list
    Lazy.t;
         
    f_alternatives: elsif_stmt_part_list
    Lazy.t;
         
    f_else_stmts: stmt_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and named_stmt =
    [
    | `NamedStmt
        of named_stmt_fields
    ]
  and named_stmt_fields = 
  {
         
    f_decl: named_stmt_decl
    Lazy.t;
         
    f_stmt: [
      | `BeginBlock
          of begin_block_fields
      | `DeclBlock
          of decl_block_fields
      | `ForLoopStmt
          of for_loop_stmt_fields
      | `LoopStmt
          of loop_stmt_fields
      | `WhileLoopStmt
          of while_loop_stmt_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and select_stmt =
    [
    | `SelectStmt
        of select_stmt_fields
    ]
  and select_stmt_fields = 
  {
         
    f_guards: select_when_part_list
    Lazy.t;
         
    f_else_stmts: stmt_list
    Lazy.t;
         
    f_abort_stmts: stmt_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and error_stmt =
    [
    | `ErrorStmt
        of error_stmt_fields
    ]
  and error_stmt_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and simple_stmt =
    [
    | `AbortStmt
        of abort_stmt_fields
    | `AssignStmt
        of assign_stmt_fields
    | `CallStmt
        of call_stmt_fields
    | `DelayStmt
        of delay_stmt_fields
    | `ExitStmt
        of exit_stmt_fields
    | `GotoStmt
        of goto_stmt_fields
    | `Label
        of label_fields
    | `NullStmt
        of null_stmt_fields
    | `RaiseStmt
        of raise_stmt_fields
    | `RequeueStmt
        of requeue_stmt_fields
    | `ReturnStmt
        of return_stmt_fields
    | `TerminateAlternative
        of terminate_alternative_fields
    ]

  
   
  and abort_stmt =
    [
    | `AbortStmt
        of abort_stmt_fields
    ]
  and abort_stmt_fields = 
  {
         
    f_names: name_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and assign_stmt =
    [
    | `AssignStmt
        of assign_stmt_fields
    ]
  and assign_stmt_fields = 
  {
         
    f_dest: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and call_stmt =
    [
    | `CallStmt
        of call_stmt_fields
    ]
  and call_stmt_fields = 
  {
         
    f_call: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and delay_stmt =
    [
    | `DelayStmt
        of delay_stmt_fields
    ]
  and delay_stmt_fields = 
  {
         
    f_has_until: until_node
    Lazy.t;
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and exit_stmt =
    [
    | `ExitStmt
        of exit_stmt_fields
    ]
  and exit_stmt_fields = 
  {
         
    f_loop_name: [
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    option
    Lazy.t;
         
    f_cond_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and goto_stmt =
    [
    | `GotoStmt
        of goto_stmt_fields
    ]
  and goto_stmt_fields = 
  {
         
    f_label_name: [
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and label =
    [
    | `Label
        of label_fields
    ]
  and label_fields = 
  {
         
    f_decl: label_decl
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and null_stmt =
    [
    | `NullStmt
        of null_stmt_fields
    ]
  and null_stmt_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and raise_stmt =
    [
    | `RaiseStmt
        of raise_stmt_fields
    ]
  and raise_stmt_fields = 
  {
         
    f_exception_name: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
         
    f_error_message: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and requeue_stmt =
    [
    | `RequeueStmt
        of requeue_stmt_fields
    ]
  and requeue_stmt_fields = 
  {
         
    f_call_name: [
      | `AttributeRef
          of attribute_ref_fields
      | `CallExpr
          of call_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `QualExpr
          of qual_expr_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_has_abort: abort_node
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and return_stmt =
    [
    | `ReturnStmt
        of return_stmt_fields
    ]
  and return_stmt_fields = 
  {
         
    f_return_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `MembershipExpr
          of membership_expr_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and terminate_alternative =
    [
    | `TerminateAlternative
        of terminate_alternative_fields
    ]
  and terminate_alternative_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and subp_kind =
    [
    | `SubpKindFunction
        of subp_kind_function_fields
    | `SubpKindProcedure
        of subp_kind_procedure_fields
    ]

  
   
  and subp_kind_function =
    [
    | `SubpKindFunction
        of subp_kind_function_fields
    ]
  and subp_kind_function_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and subp_kind_procedure =
    [
    | `SubpKindProcedure
        of subp_kind_procedure_fields
    ]
  and subp_kind_procedure_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and subunit =
    [
    | `Subunit
        of subunit_fields
    ]
  and subunit_fields = 
  {
         
    f_name: [
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
         
    f_body: [
      | `PackageBody
          of package_body_fields
      | `ProtectedBody
          of protected_body_fields
      | `SubpBody
          of subp_body_fields
      | `TaskBody
          of task_body_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and synchronized_node =
    [
    | `SynchronizedAbsent
        of synchronized_absent_fields
    | `SynchronizedPresent
        of synchronized_present_fields
    ]

  
   
  and synchronized_absent =
    [
    | `SynchronizedAbsent
        of synchronized_absent_fields
    ]
  and synchronized_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and synchronized_present =
    [
    | `SynchronizedPresent
        of synchronized_present_fields
    ]
  and synchronized_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and tagged_node =
    [
    | `TaggedAbsent
        of tagged_absent_fields
    | `TaggedPresent
        of tagged_present_fields
    ]

  
   
  and tagged_absent =
    [
    | `TaggedAbsent
        of tagged_absent_fields
    ]
  and tagged_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and tagged_present =
    [
    | `TaggedPresent
        of tagged_present_fields
    ]
  and tagged_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and task_def =
    [
    | `TaskDef
        of task_def_fields
    ]
  and task_def_fields = 
  {
         
    f_interfaces: parent_list
    Lazy.t;
         
    f_public_part: public_part
    Lazy.t;
         
    f_private_part: private_part
    option
    Lazy.t;
         
    f_end_name: end_name
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and type_attributes_repository =
    [
    | `TypeAttributesRepository
        of type_attributes_repository_fields
    ]
  and type_attributes_repository_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * access_def
    * array_type_def
    * derived_type_def
    * enum_type_def
    * formal_discrete_type_def
    * interface_type_def
    * mod_int_type_def
    * private_type_def
    * real_type_def
    * record_type_def
    * signed_int_type_def
    *)
  and type_def =
    [
    | `AccessToSubpDef
        of access_to_subp_def_fields
    | `AnonymousTypeAccessDef
        of anonymous_type_access_def_fields
    | `TypeAccessDef
        of type_access_def_fields
    | `ArrayTypeDef
        of array_type_def_fields
    | `DerivedTypeDef
        of derived_type_def_fields
    | `EnumTypeDef
        of enum_type_def_fields
    | `FormalDiscreteTypeDef
        of formal_discrete_type_def_fields
    | `InterfaceTypeDef
        of interface_type_def_fields
    | `ModIntTypeDef
        of mod_int_type_def_fields
    | `PrivateTypeDef
        of private_type_def_fields
    | `DecimalFixedPointDef
        of decimal_fixed_point_def_fields
    | `FloatingPointDef
        of floating_point_def_fields
    | `OrdinaryFixedPointDef
        of ordinary_fixed_point_def_fields
    | `RecordTypeDef
        of record_type_def_fields
    | `SignedIntTypeDef
        of signed_int_type_def_fields
    ]

  
   
  (**
    * access_to_subp_def
    * base_type_access_def
    *)
  and access_def =
    [
    | `AccessToSubpDef
        of access_to_subp_def_fields
    | `AnonymousTypeAccessDef
        of anonymous_type_access_def_fields
    | `TypeAccessDef
        of type_access_def_fields
    ]

  
   
  and access_to_subp_def =
    [
    | `AccessToSubpDef
        of access_to_subp_def_fields
    ]
  and access_to_subp_def_fields = 
  {
         
    f_has_not_null: not_null
    option
    Lazy.t;
         
    f_has_protected: protected_node
    Lazy.t;
         
    f_subp_spec: subp_spec
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_type_access_def =
    [
    | `AnonymousTypeAccessDef
        of anonymous_type_access_def_fields
    | `TypeAccessDef
        of type_access_def_fields
    ]

  
   
  and anonymous_type_access_def =
    [
    | `AnonymousTypeAccessDef
        of anonymous_type_access_def_fields
    ]
  and anonymous_type_access_def_fields = 
  {
         
    f_has_not_null: not_null
    option
    Lazy.t;
         
    f_type_decl: base_type_decl
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and type_access_def =
    [
    | `TypeAccessDef
        of type_access_def_fields
    ]
  and type_access_def_fields = 
  {
         
    f_has_not_null: not_null
    option
    Lazy.t;
         
    f_has_all: all_node
    Lazy.t;
         
    f_has_constant: constant_node
    Lazy.t;
         
    f_subtype_indication: subtype_indication
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and array_type_def =
    [
    | `ArrayTypeDef
        of array_type_def_fields
    ]
  and array_type_def_fields = 
  {
         
    f_indices: array_indices
    Lazy.t;
         
    f_component_type: component_def
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and derived_type_def =
    [
    | `DerivedTypeDef
        of derived_type_def_fields
    ]
  and derived_type_def_fields = 
  {
         
    f_has_abstract: abstract_node
    Lazy.t;
         
    f_has_limited: limited_node
    Lazy.t;
         
    f_has_synchronized: synchronized_node
    Lazy.t;
         
    f_subtype_indication: subtype_indication
    Lazy.t;
         
    f_interfaces: parent_list
    Lazy.t;
         
    f_record_extension: base_record_def
    option
    Lazy.t;
         
    f_has_with_private: with_private
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_type_def =
    [
    | `EnumTypeDef
        of enum_type_def_fields
    ]
  and enum_type_def_fields = 
  {
         
    f_enum_literals: enum_literal_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and formal_discrete_type_def =
    [
    | `FormalDiscreteTypeDef
        of formal_discrete_type_def_fields
    ]
  and formal_discrete_type_def_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and interface_type_def =
    [
    | `InterfaceTypeDef
        of interface_type_def_fields
    ]
  and interface_type_def_fields = 
  {
         
    f_interface_kind: interface_kind
    option
    Lazy.t;
         
    f_interfaces: parent_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and mod_int_type_def =
    [
    | `ModIntTypeDef
        of mod_int_type_def_fields
    ]
  and mod_int_type_def_fields = 
  {
         
    f_expr: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BoxExpr
          of box_expr_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and private_type_def =
    [
    | `PrivateTypeDef
        of private_type_def_fields
    ]
  and private_type_def_fields = 
  {
         
    f_has_abstract: abstract_node
    Lazy.t;
         
    f_has_tagged: tagged_node
    Lazy.t;
         
    f_has_limited: limited_node
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and real_type_def =
    [
    | `DecimalFixedPointDef
        of decimal_fixed_point_def_fields
    | `FloatingPointDef
        of floating_point_def_fields
    | `OrdinaryFixedPointDef
        of ordinary_fixed_point_def_fields
    ]

  
   
  and decimal_fixed_point_def =
    [
    | `DecimalFixedPointDef
        of decimal_fixed_point_def_fields
    ]
  and decimal_fixed_point_def_fields = 
  {
         
    f_delta: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BoxExpr
          of box_expr_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_digits: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BoxExpr
          of box_expr_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_range: range_spec
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and floating_point_def =
    [
    | `FloatingPointDef
        of floating_point_def_fields
    ]
  and floating_point_def_fields = 
  {
         
    f_num_digits: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BoxExpr
          of box_expr_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_range: range_spec
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and ordinary_fixed_point_def =
    [
    | `OrdinaryFixedPointDef
        of ordinary_fixed_point_def_fields
    ]
  and ordinary_fixed_point_def_fields = 
  {
         
    f_delta: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BoxExpr
          of box_expr_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_range: range_spec
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and record_type_def =
    [
    | `RecordTypeDef
        of record_type_def_fields
    ]
  and record_type_def_fields = 
  {
         
    f_has_abstract: abstract_node
    Lazy.t;
         
    f_has_tagged: tagged_node
    Lazy.t;
         
    f_has_limited: limited_node
    Lazy.t;
         
    f_record_def: base_record_def
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and signed_int_type_def =
    [
    | `SignedIntTypeDef
        of signed_int_type_def_fields
    ]
  and signed_int_type_def_fields = 
  {
         
    f_range: range_spec
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * anonymous_type
    * enum_lit_synth_type_expr
    * subtype_indication
    * synthetic_type_expr
    *)
  and type_expr =
    [
    | `AnonymousType
        of anonymous_type_fields
    | `EnumLitSynthTypeExpr
        of enum_lit_synth_type_expr_fields
    | `SubtypeIndication
        of subtype_indication_fields
    | `ConstrainedSubtypeIndication
        of constrained_subtype_indication_fields
    | `DiscreteSubtypeIndication
        of discrete_subtype_indication_fields
    | `SyntheticTypeExpr
        of synthetic_type_expr_fields
    ]

  
   
  and anonymous_type =
    [
    | `AnonymousType
        of anonymous_type_fields
    ]
  and anonymous_type_fields = 
  {
         
    f_type_decl: anonymous_type_decl
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_lit_synth_type_expr =
    [
    | `EnumLitSynthTypeExpr
        of enum_lit_synth_type_expr_fields
    ]
  and enum_lit_synth_type_expr_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and subtype_indication =
    [
    | `SubtypeIndication
        of subtype_indication_fields
    | `ConstrainedSubtypeIndication
        of constrained_subtype_indication_fields
    | `DiscreteSubtypeIndication
        of discrete_subtype_indication_fields
    ]
  and subtype_indication_fields = 
  {
         
    f_has_not_null: not_null
    Lazy.t;
         
    f_name: [
      | `AttributeRef
          of attribute_ref_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
         
    f_constraint: constraint_node
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and constrained_subtype_indication =
    [
    | `ConstrainedSubtypeIndication
        of constrained_subtype_indication_fields
    ]
  and constrained_subtype_indication_fields = 
  {
         
    f_has_not_null: not_null
    Lazy.t;
         
    f_name: [
      | `AttributeRef
          of attribute_ref_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
         
    f_constraint: constraint_node
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and discrete_subtype_indication =
    [
    | `DiscreteSubtypeIndication
        of discrete_subtype_indication_fields
    ]
  and discrete_subtype_indication_fields = 
  {
         
    f_has_not_null: not_null
    Lazy.t;
         
    f_name: [
      | `AttributeRef
          of attribute_ref_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
         
    f_constraint: constraint_node
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and synthetic_type_expr =
    [
    | `SyntheticTypeExpr
        of synthetic_type_expr_fields
    ]
  and synthetic_type_expr_fields = 
  {
         
    f_target_type: base_type_decl
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and unconstrained_array_index =
    [
    | `UnconstrainedArrayIndex
        of unconstrained_array_index_fields
    ]
  and unconstrained_array_index_fields = 
  {
         
    f_subtype_indication: subtype_indication
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and until_node =
    [
    | `UntilAbsent
        of until_absent_fields
    | `UntilPresent
        of until_present_fields
    ]

  
   
  and until_absent =
    [
    | `UntilAbsent
        of until_absent_fields
    ]
  and until_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and until_present =
    [
    | `UntilPresent
        of until_present_fields
    ]
  and until_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and use_clause =
    [
    | `UsePackageClause
        of use_package_clause_fields
    | `UseTypeClause
        of use_type_clause_fields
    ]

  
   
  and use_package_clause =
    [
    | `UsePackageClause
        of use_package_clause_fields
    ]
  and use_package_clause_fields = 
  {
         
    f_packages: name_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and use_type_clause =
    [
    | `UseTypeClause
        of use_type_clause_fields
    ]
  and use_type_clause_fields = 
  {
         
    f_has_all: all_node
    Lazy.t;
         
    f_types: name_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and value_sequence =
    [
    | `ValueSequence
        of value_sequence_fields
    ]
  and value_sequence_fields = 
  {
         
    f_iter_assoc: iterated_assoc
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and variant =
    [
    | `Variant
        of variant_fields
    ]
  and variant_fields = 
  {
         
    f_choices: alternatives_list
    Lazy.t;
         
    f_components: component_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and variant_part =
    [
    | `VariantPart
        of variant_part_fields
    ]
  and variant_part_fields = 
  {
         
    f_discr_name: identifier
    Lazy.t;
         
    f_variant: variant_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and with_clause =
    [
    | `WithClause
        of with_clause_fields
    ]
  and with_clause_fields = 
  {
         
    f_has_limited: limited_node
    Lazy.t;
         
    f_has_private: private_node
    Lazy.t;
         
    f_packages: name_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and with_private =
    [
    | `WithPrivateAbsent
        of with_private_absent_fields
    | `WithPrivatePresent
        of with_private_present_fields
    ]

  
   
  and with_private_absent =
    [
    | `WithPrivateAbsent
        of with_private_absent_fields
    ]
  and with_private_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and with_private_present =
    [
    | `WithPrivatePresent
        of with_private_present_fields
    ]
  and with_private_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }




      module GPRProject = struct
  type project_scenario_variable

  let c_project_scenario_variable : project_scenario_variable structure typ =
    structure "project_scenario_variable"

  let scenario_var_name = field c_project_scenario_variable "name" (ptr char)

  let scenario_var_value = field c_project_scenario_variable "value" (ptr char)

  let () = seal c_project_scenario_variable

  let create_scenario_variable scenario_vars =
    (* One more to store (null, null) *)
    let scenario_vars_len = List.length scenario_vars + 1 in
    let c_scenario_vars =
      (* Allocates a fresh array with given size, fill with zeros. Thus,
         the last cell is already (null, null) *)
      allocate_n c_project_scenario_variable ~count:scenario_vars_len
    in
    let fill_scenario_vars i (name, value) =
      let c_struct = make c_project_scenario_variable in
      let name = char_ptr_of_string name in
      let value = char_ptr_of_string value in
      setf c_struct scenario_var_name name ;
      add_gc_link ~from:c_scenario_vars ~to_:name ;
      setf c_struct scenario_var_value value ;
      add_gc_link ~from:c_scenario_vars ~to_:value ;
      c_scenario_vars +@ i <-@ c_struct
    in
    List.iteri fill_scenario_vars scenario_vars ;
    c_scenario_vars

  (* project_scenario_variable is never wrapped *)
  let project_scenario_variable =
    view
      (ptr c_project_scenario_variable)
      ~read:(fun _ -> assert false)
      ~write:create_scenario_variable

  type string_array

  let c_string_array : string_array structure typ =
    structure "string_array"

  let array_length = field c_string_array "length" int

  let array_c_ptr = field c_string_array "c_ptr" (ptr string)

  let () = seal c_string_array

  let c_free_string_array =
    foreign ~from:c_lib "ada_free_string_array"
      (ptr c_string_array @-> raisable void)

  let read_string_array c_value =
    let result =
      List.init
        (!@ (c_value |-> array_length))
        (fun i -> (!@ ((!@ (c_value |-> array_c_ptr)) +@ i)))
    in
    c_free_string_array c_value ;
    result

  let write_string_array _value =
    (* Not used for now *)
    assert false

  let string_array = view (ptr c_string_array) ~read:read_string_array ~write:write_string_array

  type t = unit ptr ptr

  let gpr_project_free =
    foreign ~from:c_lib "ada_gpr_project_free"
      (ptr void @-> raisable void)

  let read c_value =
    let finalise arg =
      gpr_project_free (!@ arg)
    in
    allocate ~finalise (ptr void) c_value

  let write value = (!@ value)

  let c_type = view (ptr void) ~read ~write

  let gpr_project_load =
    foreign ~from:c_lib "ada_gpr_project_load"
    (string @-> project_scenario_variable @-> string
     @-> string @-> ptr c_type @-> ptr string_array @-> raisable void)

  let gpr_project_create_preprocessor =
    foreign ~from:c_lib "ada_gpr_project_create_preprocessor"
      (c_type @-> string @-> ptr int @-> raisable FileReader.c_type)

  let load
      ?(scenario_vars = [])
      ?(target = "")
      ?(runtime = "")
      project_file : t =
    (* Use allocate_n to avoid having to give it an initial value *)
    let result = allocate_n ~count:1 c_type in
    let errors = allocate_n ~count:1 string_array in
    gpr_project_load project_file scenario_vars target runtime result errors ;
    (* Not sure what to do with errors here as we already have an exception *)
    !@ result

  let c_source_files =
    foreign ~from:c_lib "ada_gpr_project_source_files"
      (c_type @-> int @-> ptr void @-> raisable string_array)

  type source_file_mode =
    | Default
    | RootProject
    | WholeProject
    | WholeProjectWithRuntime

  let write_source_file_mode = function
    | Default -> 0
    | RootProject -> 1
    | WholeProject -> 2
    | WholeProjectWithRuntime -> 3

  let source_files ?(mode = Default) gpr_project =
    c_source_files gpr_project (write_source_file_mode mode) null

  type line_mode =
    | DeleteLines
    | BlankLines
    | CommentLines

  let create_preprocessor ?(project = "") ?line_mode gpr_project =
    let line_mode_c =
      match line_mode with
      | Some DeleteLines ->
          allocate int 0
      | Some BlankLines ->
          allocate int 1
      | Some CommentLines ->
          allocate int 2
      | None ->
         from_voidp int null
    in
    gpr_project_create_preprocessor gpr_project project line_mode_c

  let gpr_project_create_unit_provider =
    foreign ~from:c_lib "ada_gpr_project_create_unit_provider"
      (c_type @-> string @-> raisable UnitProvider.c_type)

  let create_unit_provider ?(project = "") gpr_project =
    let result = gpr_project_create_unit_provider gpr_project project in
    (* The unit provider keeps an internal reference to the project. Use
       the keep argument to simulate the same behaviour and avoid freeing
       the project file too early. *)
    UnitProvider.wrap ~keep:gpr_project result

  let gpr_project_initialize_context =
    foreign ~from:c_lib "ada_gpr_project_initialize_context"
      ( c_type @-> AnalysisContextStruct.c_type @-> string
        @-> ptr void @-> bool @-> int @-> raisable void )

  let create_analysis_context
      ?(with_trivia = true)
      ?(tab_stop = 8)
      ?(project = "")
      gpr_project : analysis_context =
    if tab_stop < 1 then
      raise (Invalid_argument "Invalid tab_stop (positive integer expected)") ;
    let c_context =
      AnalysisContextStruct.allocate_analysis_context ~keep:gpr_project ()
    in
    gpr_project_initialize_context
      gpr_project
      c_context
      project
      Ctypes.null (* TODO: bind the event handlers API to OCaml *)
      with_trivia
      tab_stop ;
    { c_value= c_context
      ; unit_provider= UnitProvider.null }
end



let rec unwrap_ada_node value =
  (* This is the unique unwrap function that can be called for any node. *)
  match (value :> ada_node) with
  | `AbortAbsent fields -> fields.c_value
  | `AbortPresent fields -> fields.c_value
  | `AbstractAbsent fields -> fields.c_value
  | `AbstractPresent fields -> fields.c_value
  | `AdaNodeList fields -> fields.c_value
  | `AbstractStateDeclList fields -> fields.c_value
  | `AlternativesList fields -> fields.c_value
  | `ConstraintList fields -> fields.c_value
  | `DeclList fields -> fields.c_value
  | `StmtList fields -> fields.c_value
  | `AspectAssocList fields -> fields.c_value
  | `BaseAssocList fields -> fields.c_value
  | `AssocList fields -> fields.c_value
  | `BasicDeclList fields -> fields.c_value
  | `CaseExprAlternativeList fields -> fields.c_value
  | `CaseStmtAlternativeList fields -> fields.c_value
  | `CompilationUnitList fields -> fields.c_value
  | `ConcatOperandList fields -> fields.c_value
  | `ContractCaseAssocList fields -> fields.c_value
  | `DefiningNameList fields -> fields.c_value
  | `DiscriminantSpecList fields -> fields.c_value
  | `ElsifExprPartList fields -> fields.c_value
  | `ElsifStmtPartList fields -> fields.c_value
  | `EnumLiteralDeclList fields -> fields.c_value
  | `ExprAlternativesList fields -> fields.c_value
  | `DiscriminantChoiceList fields -> fields.c_value
  | `NameList fields -> fields.c_value
  | `ParentList fields -> fields.c_value
  | `ParamSpecList fields -> fields.c_value
  | `PragmaNodeList fields -> fields.c_value
  | `SelectWhenPartList fields -> fields.c_value
  | `UnconstrainedArrayIndexList fields -> fields.c_value
  | `VariantList fields -> fields.c_value
  | `AliasedAbsent fields -> fields.c_value
  | `AliasedPresent fields -> fields.c_value
  | `AllAbsent fields -> fields.c_value
  | `AllPresent fields -> fields.c_value
  | `ConstrainedArrayIndices fields -> fields.c_value
  | `UnconstrainedArrayIndices fields -> fields.c_value
  | `AspectAssoc fields -> fields.c_value
  | `AtClause fields -> fields.c_value
  | `AttributeDefClause fields -> fields.c_value
  | `EnumRepClause fields -> fields.c_value
  | `RecordRepClause fields -> fields.c_value
  | `AspectSpec fields -> fields.c_value
  | `ContractCaseAssoc fields -> fields.c_value
  | `PragmaArgumentAssoc fields -> fields.c_value
  | `EntrySpec fields -> fields.c_value
  | `EnumSubpSpec fields -> fields.c_value
  | `SubpSpec fields -> fields.c_value
  | `SyntheticBinarySpec fields -> fields.c_value
  | `SyntheticUnarySpec fields -> fields.c_value
  | `ComponentList fields -> fields.c_value
  | `KnownDiscriminantPart fields -> fields.c_value
  | `UnknownDiscriminantPart fields -> fields.c_value
  | `EntryCompletionFormalParams fields -> fields.c_value
  | `GenericFormalPart fields -> fields.c_value
  | `NullRecordDef fields -> fields.c_value
  | `RecordDef fields -> fields.c_value
  | `AggregateAssoc fields -> fields.c_value
  | `MultiDimArrayAssoc fields -> fields.c_value
  | `CompositeConstraintAssoc fields -> fields.c_value
  | `IteratedAssoc fields -> fields.c_value
  | `ParamAssoc fields -> fields.c_value
  | `AbstractStateDecl fields -> fields.c_value
  | `AnonymousExprDecl fields -> fields.c_value
  | `ComponentDecl fields -> fields.c_value
  | `DiscriminantSpec fields -> fields.c_value
  | `GenericFormalObjDecl fields -> fields.c_value
  | `GenericFormalPackage fields -> fields.c_value
  | `GenericFormalSubpDecl fields -> fields.c_value
  | `GenericFormalTypeDecl fields -> fields.c_value
  | `ParamSpec fields -> fields.c_value
  | `SyntheticFormalParamDecl fields -> fields.c_value
  | `GenericPackageInternal fields -> fields.c_value
  | `PackageDecl fields -> fields.c_value
  | `DiscreteBaseSubtypeDecl fields -> fields.c_value
  | `SubtypeDecl fields -> fields.c_value
  | `ClasswideTypeDecl fields -> fields.c_value
  | `IncompleteTypeDecl fields -> fields.c_value
  | `IncompleteFormalTypeDecl fields -> fields.c_value
  | `IncompleteTaggedTypeDecl fields -> fields.c_value
  | `ProtectedTypeDecl fields -> fields.c_value
  | `TaskTypeDecl fields -> fields.c_value
  | `SingleTaskTypeDecl fields -> fields.c_value
  | `AnonymousTypeDecl fields -> fields.c_value
  | `SynthAnonymousTypeDecl fields -> fields.c_value
  | `ConcreteTypeDecl fields -> fields.c_value
  | `FormalTypeDecl fields -> fields.c_value
  | `AbstractSubpDecl fields -> fields.c_value
  | `AbstractFormalSubpDecl fields -> fields.c_value
  | `ConcreteFormalSubpDecl fields -> fields.c_value
  | `SubpDecl fields -> fields.c_value
  | `EntryDecl fields -> fields.c_value
  | `EnumLiteralDecl fields -> fields.c_value
  | `SyntheticCharEnumLit fields -> fields.c_value
  | `GenericSubpInternal fields -> fields.c_value
  | `SyntheticSubpDecl fields -> fields.c_value
  | `AcceptStmtBody fields -> fields.c_value
  | `ExprFunction fields -> fields.c_value
  | `NullSubpDecl fields -> fields.c_value
  | `SubpBody fields -> fields.c_value
  | `SubpRenamingDecl fields -> fields.c_value
  | `PackageBodyStub fields -> fields.c_value
  | `ProtectedBodyStub fields -> fields.c_value
  | `SubpBodyStub fields -> fields.c_value
  | `TaskBodyStub fields -> fields.c_value
  | `EntryBody fields -> fields.c_value
  | `PackageBody fields -> fields.c_value
  | `ProtectedBody fields -> fields.c_value
  | `TaskBody fields -> fields.c_value
  | `EntryIndexSpec fields -> fields.c_value
  | `ErrorDecl fields -> fields.c_value
  | `ExceptionDecl fields -> fields.c_value
  | `ExceptionHandler fields -> fields.c_value
  | `ForLoopVarDecl fields -> fields.c_value
  | `GenericPackageDecl fields -> fields.c_value
  | `GenericSubpDecl fields -> fields.c_value
  | `GenericPackageInstantiation fields -> fields.c_value
  | `GenericSubpInstantiation fields -> fields.c_value
  | `GenericPackageRenamingDecl fields -> fields.c_value
  | `GenericSubpRenamingDecl fields -> fields.c_value
  | `LabelDecl fields -> fields.c_value
  | `NamedStmtDecl fields -> fields.c_value
  | `NumberDecl fields -> fields.c_value
  | `ObjectDecl fields -> fields.c_value
  | `ExtendedReturnStmtObjectDecl fields -> fields.c_value
  | `NoTypeObjectRenamingDecl fields -> fields.c_value
  | `PackageRenamingDecl fields -> fields.c_value
  | `SingleProtectedDecl fields -> fields.c_value
  | `SingleTaskDecl fields -> fields.c_value
  | `CaseStmtAlternative fields -> fields.c_value
  | `CompilationUnit fields -> fields.c_value
  | `ComponentClause fields -> fields.c_value
  | `ComponentDef fields -> fields.c_value
  | `ConstantAbsent fields -> fields.c_value
  | `ConstantPresent fields -> fields.c_value
  | `CompositeConstraint fields -> fields.c_value
  | `DeltaConstraint fields -> fields.c_value
  | `DigitsConstraint fields -> fields.c_value
  | `RangeConstraint fields -> fields.c_value
  | `DeclarativePart fields -> fields.c_value
  | `PrivatePart fields -> fields.c_value
  | `PublicPart fields -> fields.c_value
  | `ElsifExprPart fields -> fields.c_value
  | `ElsifStmtPart fields -> fields.c_value
  | `AbstractStateDeclExpr fields -> fields.c_value
  | `Allocator fields -> fields.c_value
  | `Aggregate fields -> fields.c_value
  | `BracketAggregate fields -> fields.c_value
  | `DeltaAggregate fields -> fields.c_value
  | `BracketDeltaAggregate fields -> fields.c_value
  | `NullRecordAggregate fields -> fields.c_value
  | `BinOp fields -> fields.c_value
  | `RelationOp fields -> fields.c_value
  | `BoxExpr fields -> fields.c_value
  | `CaseExprAlternative fields -> fields.c_value
  | `ConcatOp fields -> fields.c_value
  | `ConcatOperand fields -> fields.c_value
  | `CaseExpr fields -> fields.c_value
  | `IfExpr fields -> fields.c_value
  | `ContractCases fields -> fields.c_value
  | `DeclExpr fields -> fields.c_value
  | `MembershipExpr fields -> fields.c_value
  | `AttributeRef fields -> fields.c_value
  | `CallExpr fields -> fields.c_value
  | `DefiningName fields -> fields.c_value
  | `SyntheticDefiningName fields -> fields.c_value
  | `DiscreteSubtypeName fields -> fields.c_value
  | `DottedName fields -> fields.c_value
  | `EndName fields -> fields.c_value
  | `ExplicitDeref fields -> fields.c_value
  | `QualExpr fields -> fields.c_value
  | `ReduceAttributeRef fields -> fields.c_value
  | `CharLiteral fields -> fields.c_value
  | `Identifier fields -> fields.c_value
  | `OpAbs fields -> fields.c_value
  | `OpAnd fields -> fields.c_value
  | `OpAndThen fields -> fields.c_value
  | `OpConcat fields -> fields.c_value
  | `OpDiv fields -> fields.c_value
  | `OpDoubleDot fields -> fields.c_value
  | `OpEq fields -> fields.c_value
  | `OpGt fields -> fields.c_value
  | `OpGte fields -> fields.c_value
  | `OpIn fields -> fields.c_value
  | `OpLt fields -> fields.c_value
  | `OpLte fields -> fields.c_value
  | `OpMinus fields -> fields.c_value
  | `OpMod fields -> fields.c_value
  | `OpMult fields -> fields.c_value
  | `OpNeq fields -> fields.c_value
  | `OpNot fields -> fields.c_value
  | `OpNotIn fields -> fields.c_value
  | `OpOr fields -> fields.c_value
  | `OpOrElse fields -> fields.c_value
  | `OpPlus fields -> fields.c_value
  | `OpPow fields -> fields.c_value
  | `OpRem fields -> fields.c_value
  | `OpXor fields -> fields.c_value
  | `StringLiteral fields -> fields.c_value
  | `NullLiteral fields -> fields.c_value
  | `IntLiteral fields -> fields.c_value
  | `RealLiteral fields -> fields.c_value
  | `SyntheticIdentifier fields -> fields.c_value
  | `TargetName fields -> fields.c_value
  | `UpdateAttributeRef fields -> fields.c_value
  | `ParenExpr fields -> fields.c_value
  | `QuantifiedExpr fields -> fields.c_value
  | `RaiseExpr fields -> fields.c_value
  | `UnOp fields -> fields.c_value
  | `HandledStmts fields -> fields.c_value
  | `InterfaceKindLimited fields -> fields.c_value
  | `InterfaceKindProtected fields -> fields.c_value
  | `InterfaceKindSynchronized fields -> fields.c_value
  | `InterfaceKindTask fields -> fields.c_value
  | `IterTypeIn fields -> fields.c_value
  | `IterTypeOf fields -> fields.c_value
  | `LibraryItem fields -> fields.c_value
  | `LimitedAbsent fields -> fields.c_value
  | `LimitedPresent fields -> fields.c_value
  | `ForLoopSpec fields -> fields.c_value
  | `WhileLoopSpec fields -> fields.c_value
  | `ModeDefault fields -> fields.c_value
  | `ModeIn fields -> fields.c_value
  | `ModeInOut fields -> fields.c_value
  | `ModeOut fields -> fields.c_value
  | `MultiAbstractStateDecl fields -> fields.c_value
  | `NotNullAbsent fields -> fields.c_value
  | `NotNullPresent fields -> fields.c_value
  | `NullComponentDecl fields -> fields.c_value
  | `OthersDesignator fields -> fields.c_value
  | `OverridingNotOverriding fields -> fields.c_value
  | `OverridingOverriding fields -> fields.c_value
  | `OverridingUnspecified fields -> fields.c_value
  | `Params fields -> fields.c_value
  | `ParenAbstractStateDecl fields -> fields.c_value
  | `PpElseDirective fields -> fields.c_value
  | `PpElsifDirective fields -> fields.c_value
  | `PpEndIfDirective fields -> fields.c_value
  | `PpIfDirective fields -> fields.c_value
  | `PpThenKw fields -> fields.c_value
  | `PragmaNode fields -> fields.c_value
  | `PrivateAbsent fields -> fields.c_value
  | `PrivatePresent fields -> fields.c_value
  | `ProtectedDef fields -> fields.c_value
  | `ProtectedAbsent fields -> fields.c_value
  | `ProtectedPresent fields -> fields.c_value
  | `QuantifierAll fields -> fields.c_value
  | `QuantifierSome fields -> fields.c_value
  | `RangeSpec fields -> fields.c_value
  | `RenamingClause fields -> fields.c_value
  | `SyntheticRenamingClause fields -> fields.c_value
  | `ReverseAbsent fields -> fields.c_value
  | `ReversePresent fields -> fields.c_value
  | `SelectWhenPart fields -> fields.c_value
  | `AcceptStmt fields -> fields.c_value
  | `AcceptStmtWithStmts fields -> fields.c_value
  | `ForLoopStmt fields -> fields.c_value
  | `LoopStmt fields -> fields.c_value
  | `WhileLoopStmt fields -> fields.c_value
  | `BeginBlock fields -> fields.c_value
  | `DeclBlock fields -> fields.c_value
  | `CaseStmt fields -> fields.c_value
  | `ExtendedReturnStmt fields -> fields.c_value
  | `IfStmt fields -> fields.c_value
  | `NamedStmt fields -> fields.c_value
  | `SelectStmt fields -> fields.c_value
  | `ErrorStmt fields -> fields.c_value
  | `AbortStmt fields -> fields.c_value
  | `AssignStmt fields -> fields.c_value
  | `CallStmt fields -> fields.c_value
  | `DelayStmt fields -> fields.c_value
  | `ExitStmt fields -> fields.c_value
  | `GotoStmt fields -> fields.c_value
  | `Label fields -> fields.c_value
  | `NullStmt fields -> fields.c_value
  | `RaiseStmt fields -> fields.c_value
  | `RequeueStmt fields -> fields.c_value
  | `ReturnStmt fields -> fields.c_value
  | `TerminateAlternative fields -> fields.c_value
  | `SubpKindFunction fields -> fields.c_value
  | `SubpKindProcedure fields -> fields.c_value
  | `Subunit fields -> fields.c_value
  | `SynchronizedAbsent fields -> fields.c_value
  | `SynchronizedPresent fields -> fields.c_value
  | `TaggedAbsent fields -> fields.c_value
  | `TaggedPresent fields -> fields.c_value
  | `TaskDef fields -> fields.c_value
  | `TypeAttributesRepository fields -> fields.c_value
  | `AccessToSubpDef fields -> fields.c_value
  | `AnonymousTypeAccessDef fields -> fields.c_value
  | `TypeAccessDef fields -> fields.c_value
  | `ArrayTypeDef fields -> fields.c_value
  | `DerivedTypeDef fields -> fields.c_value
  | `EnumTypeDef fields -> fields.c_value
  | `FormalDiscreteTypeDef fields -> fields.c_value
  | `InterfaceTypeDef fields -> fields.c_value
  | `ModIntTypeDef fields -> fields.c_value
  | `PrivateTypeDef fields -> fields.c_value
  | `DecimalFixedPointDef fields -> fields.c_value
  | `FloatingPointDef fields -> fields.c_value
  | `OrdinaryFixedPointDef fields -> fields.c_value
  | `RecordTypeDef fields -> fields.c_value
  | `SignedIntTypeDef fields -> fields.c_value
  | `AnonymousType fields -> fields.c_value
  | `EnumLitSynthTypeExpr fields -> fields.c_value
  | `SubtypeIndication fields -> fields.c_value
  | `ConstrainedSubtypeIndication fields -> fields.c_value
  | `DiscreteSubtypeIndication fields -> fields.c_value
  | `SyntheticTypeExpr fields -> fields.c_value
  | `UnconstrainedArrayIndex fields -> fields.c_value
  | `UntilAbsent fields -> fields.c_value
  | `UntilPresent fields -> fields.c_value
  | `UsePackageClause fields -> fields.c_value
  | `UseTypeClause fields -> fields.c_value
  | `ValueSequence fields -> fields.c_value
  | `Variant fields -> fields.c_value
  | `VariantPart fields -> fields.c_value
  | `WithClause fields -> fields.c_value
  | `WithPrivateAbsent fields -> fields.c_value
  | `WithPrivatePresent fields -> fields.c_value


   

  and unwrap_entity_info value =
    let c_value = make EntityInfoStruct.c_type in
    let field_c_value =
      unwrap_metadata (value.md)
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      EntityInfoStruct.md
      field_c_value;
    let field_c_value =
      value.rebindings
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      EntityInfoStruct.rebindings
      field_c_value;
    let field_c_value =
      value.from_rebound
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      EntityInfoStruct.from_rebound
      field_c_value;
    c_value




   

  and unwrap_metadata value =
    let c_value = make MetadataStruct.c_type in
    let field_c_value =
      value.dottable_subp
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      MetadataStruct.dottable_subp
      field_c_value;
    let field_c_value =
      value.primitive
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      MetadataStruct.primitive
      field_c_value;
    let field_c_value =
      value.primitive_real_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      MetadataStruct.primitive_real_type
      field_c_value;
    c_value



and unwrap_analysis_unit
  (unit : analysis_unit) = unit.c_value

let rec wrap_ada_node context c_value =
  (* Top level wrap function that dispatch to wrap function of concrete types
     depending on the node kind *)
  if is_null (getf c_value EntityStruct.node) then
    raise (SyntaxError "null node")
  else
    let kind = CFunctions.node_kind (addr c_value) in
    match kind with
    | 1 ->
        (wrap_abort_absent context (c_value)
         :> ada_node)
    | 2 ->
        (wrap_abort_present context (c_value)
         :> ada_node)
    | 3 ->
        (wrap_abstract_absent context (c_value)
         :> ada_node)
    | 4 ->
        (wrap_abstract_present context (c_value)
         :> ada_node)
    | 5 ->
        (wrap_ada_node_list context (c_value)
         :> ada_node)
    | 6 ->
        (wrap_abstract_state_decl_list context (c_value)
         :> ada_node)
    | 7 ->
        (wrap_alternatives_list context (c_value)
         :> ada_node)
    | 8 ->
        (wrap_constraint_list context (c_value)
         :> ada_node)
    | 9 ->
        (wrap_decl_list context (c_value)
         :> ada_node)
    | 10 ->
        (wrap_stmt_list context (c_value)
         :> ada_node)
    | 11 ->
        (wrap_aspect_assoc_list context (c_value)
         :> ada_node)
    | 12 ->
        (wrap_base_assoc_list context (c_value)
         :> ada_node)
    | 13 ->
        (wrap_assoc_list context (c_value)
         :> ada_node)
    | 14 ->
        (wrap_basic_decl_list context (c_value)
         :> ada_node)
    | 15 ->
        (wrap_case_expr_alternative_list context (c_value)
         :> ada_node)
    | 16 ->
        (wrap_case_stmt_alternative_list context (c_value)
         :> ada_node)
    | 17 ->
        (wrap_compilation_unit_list context (c_value)
         :> ada_node)
    | 18 ->
        (wrap_concat_operand_list context (c_value)
         :> ada_node)
    | 19 ->
        (wrap_contract_case_assoc_list context (c_value)
         :> ada_node)
    | 20 ->
        (wrap_defining_name_list context (c_value)
         :> ada_node)
    | 21 ->
        (wrap_discriminant_spec_list context (c_value)
         :> ada_node)
    | 22 ->
        (wrap_elsif_expr_part_list context (c_value)
         :> ada_node)
    | 23 ->
        (wrap_elsif_stmt_part_list context (c_value)
         :> ada_node)
    | 24 ->
        (wrap_enum_literal_decl_list context (c_value)
         :> ada_node)
    | 25 ->
        (wrap_expr_alternatives_list context (c_value)
         :> ada_node)
    | 26 ->
        (wrap_discriminant_choice_list context (c_value)
         :> ada_node)
    | 27 ->
        (wrap_name_list context (c_value)
         :> ada_node)
    | 28 ->
        (wrap_parent_list context (c_value)
         :> ada_node)
    | 29 ->
        (wrap_param_spec_list context (c_value)
         :> ada_node)
    | 30 ->
        (wrap_pragma_node_list context (c_value)
         :> ada_node)
    | 31 ->
        (wrap_select_when_part_list context (c_value)
         :> ada_node)
    | 32 ->
        (wrap_unconstrained_array_index_list context (c_value)
         :> ada_node)
    | 33 ->
        (wrap_variant_list context (c_value)
         :> ada_node)
    | 34 ->
        (wrap_aliased_absent context (c_value)
         :> ada_node)
    | 35 ->
        (wrap_aliased_present context (c_value)
         :> ada_node)
    | 36 ->
        (wrap_all_absent context (c_value)
         :> ada_node)
    | 37 ->
        (wrap_all_present context (c_value)
         :> ada_node)
    | 38 ->
        (wrap_constrained_array_indices context (c_value)
         :> ada_node)
    | 39 ->
        (wrap_unconstrained_array_indices context (c_value)
         :> ada_node)
    | 40 ->
        (wrap_aspect_assoc context (c_value)
         :> ada_node)
    | 41 ->
        (wrap_at_clause context (c_value)
         :> ada_node)
    | 42 ->
        (wrap_attribute_def_clause context (c_value)
         :> ada_node)
    | 43 ->
        (wrap_enum_rep_clause context (c_value)
         :> ada_node)
    | 44 ->
        (wrap_record_rep_clause context (c_value)
         :> ada_node)
    | 45 ->
        (wrap_aspect_spec context (c_value)
         :> ada_node)
    | 46 ->
        (wrap_contract_case_assoc context (c_value)
         :> ada_node)
    | 47 ->
        (wrap_pragma_argument_assoc context (c_value)
         :> ada_node)
    | 48 ->
        (wrap_entry_spec context (c_value)
         :> ada_node)
    | 49 ->
        (wrap_enum_subp_spec context (c_value)
         :> ada_node)
    | 50 ->
        (wrap_subp_spec context (c_value)
         :> ada_node)
    | 51 ->
        (wrap_synthetic_binary_spec context (c_value)
         :> ada_node)
    | 52 ->
        (wrap_synthetic_unary_spec context (c_value)
         :> ada_node)
    | 53 ->
        (wrap_component_list context (c_value)
         :> ada_node)
    | 54 ->
        (wrap_known_discriminant_part context (c_value)
         :> ada_node)
    | 55 ->
        (wrap_unknown_discriminant_part context (c_value)
         :> ada_node)
    | 56 ->
        (wrap_entry_completion_formal_params context (c_value)
         :> ada_node)
    | 57 ->
        (wrap_generic_formal_part context (c_value)
         :> ada_node)
    | 58 ->
        (wrap_null_record_def context (c_value)
         :> ada_node)
    | 59 ->
        (wrap_record_def context (c_value)
         :> ada_node)
    | 60 ->
        (wrap_aggregate_assoc context (c_value)
         :> ada_node)
    | 61 ->
        (wrap_multi_dim_array_assoc context (c_value)
         :> ada_node)
    | 62 ->
        (wrap_composite_constraint_assoc context (c_value)
         :> ada_node)
    | 63 ->
        (wrap_iterated_assoc context (c_value)
         :> ada_node)
    | 64 ->
        (wrap_param_assoc context (c_value)
         :> ada_node)
    | 65 ->
        (wrap_abstract_state_decl context (c_value)
         :> ada_node)
    | 66 ->
        (wrap_anonymous_expr_decl context (c_value)
         :> ada_node)
    | 67 ->
        (wrap_component_decl context (c_value)
         :> ada_node)
    | 68 ->
        (wrap_discriminant_spec context (c_value)
         :> ada_node)
    | 69 ->
        (wrap_generic_formal_obj_decl context (c_value)
         :> ada_node)
    | 70 ->
        (wrap_generic_formal_package context (c_value)
         :> ada_node)
    | 71 ->
        (wrap_generic_formal_subp_decl context (c_value)
         :> ada_node)
    | 72 ->
        (wrap_generic_formal_type_decl context (c_value)
         :> ada_node)
    | 73 ->
        (wrap_param_spec context (c_value)
         :> ada_node)
    | 74 ->
        (wrap_synthetic_formal_param_decl context (c_value)
         :> ada_node)
    | 75 ->
        (wrap_generic_package_internal context (c_value)
         :> ada_node)
    | 76 ->
        (wrap_package_decl context (c_value)
         :> ada_node)
    | 77 ->
        (wrap_discrete_base_subtype_decl context (c_value)
         :> ada_node)
    | 78 ->
        (wrap_subtype_decl context (c_value)
         :> ada_node)
    | 79 ->
        (wrap_classwide_type_decl context (c_value)
         :> ada_node)
    | 80 ->
        (wrap_incomplete_type_decl context (c_value)
         :> ada_node)
    | 81 ->
        (wrap_incomplete_formal_type_decl context (c_value)
         :> ada_node)
    | 82 ->
        (wrap_incomplete_tagged_type_decl context (c_value)
         :> ada_node)
    | 83 ->
        (wrap_protected_type_decl context (c_value)
         :> ada_node)
    | 84 ->
        (wrap_task_type_decl context (c_value)
         :> ada_node)
    | 85 ->
        (wrap_single_task_type_decl context (c_value)
         :> ada_node)
    | 86 ->
        (wrap_anonymous_type_decl context (c_value)
         :> ada_node)
    | 87 ->
        (wrap_synth_anonymous_type_decl context (c_value)
         :> ada_node)
    | 88 ->
        (wrap_concrete_type_decl context (c_value)
         :> ada_node)
    | 89 ->
        (wrap_formal_type_decl context (c_value)
         :> ada_node)
    | 90 ->
        (wrap_abstract_subp_decl context (c_value)
         :> ada_node)
    | 91 ->
        (wrap_abstract_formal_subp_decl context (c_value)
         :> ada_node)
    | 92 ->
        (wrap_concrete_formal_subp_decl context (c_value)
         :> ada_node)
    | 93 ->
        (wrap_subp_decl context (c_value)
         :> ada_node)
    | 94 ->
        (wrap_entry_decl context (c_value)
         :> ada_node)
    | 95 ->
        (wrap_enum_literal_decl context (c_value)
         :> ada_node)
    | 96 ->
        (wrap_synthetic_char_enum_lit context (c_value)
         :> ada_node)
    | 97 ->
        (wrap_generic_subp_internal context (c_value)
         :> ada_node)
    | 98 ->
        (wrap_synthetic_subp_decl context (c_value)
         :> ada_node)
    | 99 ->
        (wrap_accept_stmt_body context (c_value)
         :> ada_node)
    | 100 ->
        (wrap_expr_function context (c_value)
         :> ada_node)
    | 101 ->
        (wrap_null_subp_decl context (c_value)
         :> ada_node)
    | 102 ->
        (wrap_subp_body context (c_value)
         :> ada_node)
    | 103 ->
        (wrap_subp_renaming_decl context (c_value)
         :> ada_node)
    | 104 ->
        (wrap_package_body_stub context (c_value)
         :> ada_node)
    | 105 ->
        (wrap_protected_body_stub context (c_value)
         :> ada_node)
    | 106 ->
        (wrap_subp_body_stub context (c_value)
         :> ada_node)
    | 107 ->
        (wrap_task_body_stub context (c_value)
         :> ada_node)
    | 108 ->
        (wrap_entry_body context (c_value)
         :> ada_node)
    | 109 ->
        (wrap_package_body context (c_value)
         :> ada_node)
    | 110 ->
        (wrap_protected_body context (c_value)
         :> ada_node)
    | 111 ->
        (wrap_task_body context (c_value)
         :> ada_node)
    | 112 ->
        (wrap_entry_index_spec context (c_value)
         :> ada_node)
    | 113 ->
        (wrap_error_decl context (c_value)
         :> ada_node)
    | 114 ->
        (wrap_exception_decl context (c_value)
         :> ada_node)
    | 115 ->
        (wrap_exception_handler context (c_value)
         :> ada_node)
    | 116 ->
        (wrap_for_loop_var_decl context (c_value)
         :> ada_node)
    | 117 ->
        (wrap_generic_package_decl context (c_value)
         :> ada_node)
    | 118 ->
        (wrap_generic_subp_decl context (c_value)
         :> ada_node)
    | 119 ->
        (wrap_generic_package_instantiation context (c_value)
         :> ada_node)
    | 120 ->
        (wrap_generic_subp_instantiation context (c_value)
         :> ada_node)
    | 121 ->
        (wrap_generic_package_renaming_decl context (c_value)
         :> ada_node)
    | 122 ->
        (wrap_generic_subp_renaming_decl context (c_value)
         :> ada_node)
    | 123 ->
        (wrap_label_decl context (c_value)
         :> ada_node)
    | 124 ->
        (wrap_named_stmt_decl context (c_value)
         :> ada_node)
    | 125 ->
        (wrap_number_decl context (c_value)
         :> ada_node)
    | 126 ->
        (wrap_object_decl context (c_value)
         :> ada_node)
    | 127 ->
        (wrap_extended_return_stmt_object_decl context (c_value)
         :> ada_node)
    | 128 ->
        (wrap_no_type_object_renaming_decl context (c_value)
         :> ada_node)
    | 129 ->
        (wrap_package_renaming_decl context (c_value)
         :> ada_node)
    | 130 ->
        (wrap_single_protected_decl context (c_value)
         :> ada_node)
    | 131 ->
        (wrap_single_task_decl context (c_value)
         :> ada_node)
    | 132 ->
        (wrap_case_stmt_alternative context (c_value)
         :> ada_node)
    | 133 ->
        (wrap_compilation_unit context (c_value)
         :> ada_node)
    | 134 ->
        (wrap_component_clause context (c_value)
         :> ada_node)
    | 135 ->
        (wrap_component_def context (c_value)
         :> ada_node)
    | 136 ->
        (wrap_constant_absent context (c_value)
         :> ada_node)
    | 137 ->
        (wrap_constant_present context (c_value)
         :> ada_node)
    | 138 ->
        (wrap_composite_constraint context (c_value)
         :> ada_node)
    | 139 ->
        (wrap_delta_constraint context (c_value)
         :> ada_node)
    | 140 ->
        (wrap_digits_constraint context (c_value)
         :> ada_node)
    | 141 ->
        (wrap_range_constraint context (c_value)
         :> ada_node)
    | 142 ->
        (wrap_declarative_part context (c_value)
         :> ada_node)
    | 143 ->
        (wrap_private_part context (c_value)
         :> ada_node)
    | 144 ->
        (wrap_public_part context (c_value)
         :> ada_node)
    | 145 ->
        (wrap_elsif_expr_part context (c_value)
         :> ada_node)
    | 146 ->
        (wrap_elsif_stmt_part context (c_value)
         :> ada_node)
    | 147 ->
        (wrap_abstract_state_decl_expr context (c_value)
         :> ada_node)
    | 148 ->
        (wrap_allocator context (c_value)
         :> ada_node)
    | 149 ->
        (wrap_aggregate context (c_value)
         :> ada_node)
    | 150 ->
        (wrap_bracket_aggregate context (c_value)
         :> ada_node)
    | 151 ->
        (wrap_delta_aggregate context (c_value)
         :> ada_node)
    | 152 ->
        (wrap_bracket_delta_aggregate context (c_value)
         :> ada_node)
    | 153 ->
        (wrap_null_record_aggregate context (c_value)
         :> ada_node)
    | 154 ->
        (wrap_bin_op context (c_value)
         :> ada_node)
    | 155 ->
        (wrap_relation_op context (c_value)
         :> ada_node)
    | 156 ->
        (wrap_box_expr context (c_value)
         :> ada_node)
    | 157 ->
        (wrap_case_expr_alternative context (c_value)
         :> ada_node)
    | 158 ->
        (wrap_concat_op context (c_value)
         :> ada_node)
    | 159 ->
        (wrap_concat_operand context (c_value)
         :> ada_node)
    | 160 ->
        (wrap_case_expr context (c_value)
         :> ada_node)
    | 161 ->
        (wrap_if_expr context (c_value)
         :> ada_node)
    | 162 ->
        (wrap_contract_cases context (c_value)
         :> ada_node)
    | 163 ->
        (wrap_decl_expr context (c_value)
         :> ada_node)
    | 164 ->
        (wrap_membership_expr context (c_value)
         :> ada_node)
    | 165 ->
        (wrap_attribute_ref context (c_value)
         :> ada_node)
    | 166 ->
        (wrap_call_expr context (c_value)
         :> ada_node)
    | 167 ->
        (wrap_defining_name context (c_value)
         :> ada_node)
    | 168 ->
        (wrap_synthetic_defining_name context (c_value)
         :> ada_node)
    | 169 ->
        (wrap_discrete_subtype_name context (c_value)
         :> ada_node)
    | 170 ->
        (wrap_dotted_name context (c_value)
         :> ada_node)
    | 171 ->
        (wrap_end_name context (c_value)
         :> ada_node)
    | 172 ->
        (wrap_explicit_deref context (c_value)
         :> ada_node)
    | 173 ->
        (wrap_qual_expr context (c_value)
         :> ada_node)
    | 174 ->
        (wrap_reduce_attribute_ref context (c_value)
         :> ada_node)
    | 175 ->
        (wrap_char_literal context (c_value)
         :> ada_node)
    | 176 ->
        (wrap_identifier context (c_value)
         :> ada_node)
    | 177 ->
        (wrap_op_abs context (c_value)
         :> ada_node)
    | 178 ->
        (wrap_op_and context (c_value)
         :> ada_node)
    | 179 ->
        (wrap_op_and_then context (c_value)
         :> ada_node)
    | 180 ->
        (wrap_op_concat context (c_value)
         :> ada_node)
    | 181 ->
        (wrap_op_div context (c_value)
         :> ada_node)
    | 182 ->
        (wrap_op_double_dot context (c_value)
         :> ada_node)
    | 183 ->
        (wrap_op_eq context (c_value)
         :> ada_node)
    | 184 ->
        (wrap_op_gt context (c_value)
         :> ada_node)
    | 185 ->
        (wrap_op_gte context (c_value)
         :> ada_node)
    | 186 ->
        (wrap_op_in context (c_value)
         :> ada_node)
    | 187 ->
        (wrap_op_lt context (c_value)
         :> ada_node)
    | 188 ->
        (wrap_op_lte context (c_value)
         :> ada_node)
    | 189 ->
        (wrap_op_minus context (c_value)
         :> ada_node)
    | 190 ->
        (wrap_op_mod context (c_value)
         :> ada_node)
    | 191 ->
        (wrap_op_mult context (c_value)
         :> ada_node)
    | 192 ->
        (wrap_op_neq context (c_value)
         :> ada_node)
    | 193 ->
        (wrap_op_not context (c_value)
         :> ada_node)
    | 194 ->
        (wrap_op_not_in context (c_value)
         :> ada_node)
    | 195 ->
        (wrap_op_or context (c_value)
         :> ada_node)
    | 196 ->
        (wrap_op_or_else context (c_value)
         :> ada_node)
    | 197 ->
        (wrap_op_plus context (c_value)
         :> ada_node)
    | 198 ->
        (wrap_op_pow context (c_value)
         :> ada_node)
    | 199 ->
        (wrap_op_rem context (c_value)
         :> ada_node)
    | 200 ->
        (wrap_op_xor context (c_value)
         :> ada_node)
    | 201 ->
        (wrap_string_literal context (c_value)
         :> ada_node)
    | 202 ->
        (wrap_null_literal context (c_value)
         :> ada_node)
    | 203 ->
        (wrap_int_literal context (c_value)
         :> ada_node)
    | 204 ->
        (wrap_real_literal context (c_value)
         :> ada_node)
    | 205 ->
        (wrap_synthetic_identifier context (c_value)
         :> ada_node)
    | 206 ->
        (wrap_target_name context (c_value)
         :> ada_node)
    | 207 ->
        (wrap_update_attribute_ref context (c_value)
         :> ada_node)
    | 208 ->
        (wrap_paren_expr context (c_value)
         :> ada_node)
    | 209 ->
        (wrap_quantified_expr context (c_value)
         :> ada_node)
    | 210 ->
        (wrap_raise_expr context (c_value)
         :> ada_node)
    | 211 ->
        (wrap_un_op context (c_value)
         :> ada_node)
    | 212 ->
        (wrap_handled_stmts context (c_value)
         :> ada_node)
    | 213 ->
        (wrap_interface_kind_limited context (c_value)
         :> ada_node)
    | 214 ->
        (wrap_interface_kind_protected context (c_value)
         :> ada_node)
    | 215 ->
        (wrap_interface_kind_synchronized context (c_value)
         :> ada_node)
    | 216 ->
        (wrap_interface_kind_task context (c_value)
         :> ada_node)
    | 217 ->
        (wrap_iter_type_in context (c_value)
         :> ada_node)
    | 218 ->
        (wrap_iter_type_of context (c_value)
         :> ada_node)
    | 219 ->
        (wrap_library_item context (c_value)
         :> ada_node)
    | 220 ->
        (wrap_limited_absent context (c_value)
         :> ada_node)
    | 221 ->
        (wrap_limited_present context (c_value)
         :> ada_node)
    | 222 ->
        (wrap_for_loop_spec context (c_value)
         :> ada_node)
    | 223 ->
        (wrap_while_loop_spec context (c_value)
         :> ada_node)
    | 224 ->
        (wrap_mode_default context (c_value)
         :> ada_node)
    | 225 ->
        (wrap_mode_in context (c_value)
         :> ada_node)
    | 226 ->
        (wrap_mode_in_out context (c_value)
         :> ada_node)
    | 227 ->
        (wrap_mode_out context (c_value)
         :> ada_node)
    | 228 ->
        (wrap_multi_abstract_state_decl context (c_value)
         :> ada_node)
    | 229 ->
        (wrap_not_null_absent context (c_value)
         :> ada_node)
    | 230 ->
        (wrap_not_null_present context (c_value)
         :> ada_node)
    | 231 ->
        (wrap_null_component_decl context (c_value)
         :> ada_node)
    | 232 ->
        (wrap_others_designator context (c_value)
         :> ada_node)
    | 233 ->
        (wrap_overriding_not_overriding context (c_value)
         :> ada_node)
    | 234 ->
        (wrap_overriding_overriding context (c_value)
         :> ada_node)
    | 235 ->
        (wrap_overriding_unspecified context (c_value)
         :> ada_node)
    | 236 ->
        (wrap_params context (c_value)
         :> ada_node)
    | 237 ->
        (wrap_paren_abstract_state_decl context (c_value)
         :> ada_node)
    | 238 ->
        (wrap_pp_else_directive context (c_value)
         :> ada_node)
    | 239 ->
        (wrap_pp_elsif_directive context (c_value)
         :> ada_node)
    | 240 ->
        (wrap_pp_end_if_directive context (c_value)
         :> ada_node)
    | 241 ->
        (wrap_pp_if_directive context (c_value)
         :> ada_node)
    | 242 ->
        (wrap_pp_then_kw context (c_value)
         :> ada_node)
    | 243 ->
        (wrap_pragma_node context (c_value)
         :> ada_node)
    | 244 ->
        (wrap_private_absent context (c_value)
         :> ada_node)
    | 245 ->
        (wrap_private_present context (c_value)
         :> ada_node)
    | 246 ->
        (wrap_protected_def context (c_value)
         :> ada_node)
    | 247 ->
        (wrap_protected_absent context (c_value)
         :> ada_node)
    | 248 ->
        (wrap_protected_present context (c_value)
         :> ada_node)
    | 249 ->
        (wrap_quantifier_all context (c_value)
         :> ada_node)
    | 250 ->
        (wrap_quantifier_some context (c_value)
         :> ada_node)
    | 251 ->
        (wrap_range_spec context (c_value)
         :> ada_node)
    | 252 ->
        (wrap_renaming_clause context (c_value)
         :> ada_node)
    | 253 ->
        (wrap_synthetic_renaming_clause context (c_value)
         :> ada_node)
    | 254 ->
        (wrap_reverse_absent context (c_value)
         :> ada_node)
    | 255 ->
        (wrap_reverse_present context (c_value)
         :> ada_node)
    | 256 ->
        (wrap_select_when_part context (c_value)
         :> ada_node)
    | 257 ->
        (wrap_accept_stmt context (c_value)
         :> ada_node)
    | 258 ->
        (wrap_accept_stmt_with_stmts context (c_value)
         :> ada_node)
    | 259 ->
        (wrap_for_loop_stmt context (c_value)
         :> ada_node)
    | 260 ->
        (wrap_loop_stmt context (c_value)
         :> ada_node)
    | 261 ->
        (wrap_while_loop_stmt context (c_value)
         :> ada_node)
    | 262 ->
        (wrap_begin_block context (c_value)
         :> ada_node)
    | 263 ->
        (wrap_decl_block context (c_value)
         :> ada_node)
    | 264 ->
        (wrap_case_stmt context (c_value)
         :> ada_node)
    | 265 ->
        (wrap_extended_return_stmt context (c_value)
         :> ada_node)
    | 266 ->
        (wrap_if_stmt context (c_value)
         :> ada_node)
    | 267 ->
        (wrap_named_stmt context (c_value)
         :> ada_node)
    | 268 ->
        (wrap_select_stmt context (c_value)
         :> ada_node)
    | 269 ->
        (wrap_error_stmt context (c_value)
         :> ada_node)
    | 270 ->
        (wrap_abort_stmt context (c_value)
         :> ada_node)
    | 271 ->
        (wrap_assign_stmt context (c_value)
         :> ada_node)
    | 272 ->
        (wrap_call_stmt context (c_value)
         :> ada_node)
    | 273 ->
        (wrap_delay_stmt context (c_value)
         :> ada_node)
    | 274 ->
        (wrap_exit_stmt context (c_value)
         :> ada_node)
    | 275 ->
        (wrap_goto_stmt context (c_value)
         :> ada_node)
    | 276 ->
        (wrap_label context (c_value)
         :> ada_node)
    | 277 ->
        (wrap_null_stmt context (c_value)
         :> ada_node)
    | 278 ->
        (wrap_raise_stmt context (c_value)
         :> ada_node)
    | 279 ->
        (wrap_requeue_stmt context (c_value)
         :> ada_node)
    | 280 ->
        (wrap_return_stmt context (c_value)
         :> ada_node)
    | 281 ->
        (wrap_terminate_alternative context (c_value)
         :> ada_node)
    | 282 ->
        (wrap_subp_kind_function context (c_value)
         :> ada_node)
    | 283 ->
        (wrap_subp_kind_procedure context (c_value)
         :> ada_node)
    | 284 ->
        (wrap_subunit context (c_value)
         :> ada_node)
    | 285 ->
        (wrap_synchronized_absent context (c_value)
         :> ada_node)
    | 286 ->
        (wrap_synchronized_present context (c_value)
         :> ada_node)
    | 287 ->
        (wrap_tagged_absent context (c_value)
         :> ada_node)
    | 288 ->
        (wrap_tagged_present context (c_value)
         :> ada_node)
    | 289 ->
        (wrap_task_def context (c_value)
         :> ada_node)
    | 290 ->
        (wrap_type_attributes_repository context (c_value)
         :> ada_node)
    | 291 ->
        (wrap_access_to_subp_def context (c_value)
         :> ada_node)
    | 292 ->
        (wrap_anonymous_type_access_def context (c_value)
         :> ada_node)
    | 293 ->
        (wrap_type_access_def context (c_value)
         :> ada_node)
    | 294 ->
        (wrap_array_type_def context (c_value)
         :> ada_node)
    | 295 ->
        (wrap_derived_type_def context (c_value)
         :> ada_node)
    | 296 ->
        (wrap_enum_type_def context (c_value)
         :> ada_node)
    | 297 ->
        (wrap_formal_discrete_type_def context (c_value)
         :> ada_node)
    | 298 ->
        (wrap_interface_type_def context (c_value)
         :> ada_node)
    | 299 ->
        (wrap_mod_int_type_def context (c_value)
         :> ada_node)
    | 300 ->
        (wrap_private_type_def context (c_value)
         :> ada_node)
    | 301 ->
        (wrap_decimal_fixed_point_def context (c_value)
         :> ada_node)
    | 302 ->
        (wrap_floating_point_def context (c_value)
         :> ada_node)
    | 303 ->
        (wrap_ordinary_fixed_point_def context (c_value)
         :> ada_node)
    | 304 ->
        (wrap_record_type_def context (c_value)
         :> ada_node)
    | 305 ->
        (wrap_signed_int_type_def context (c_value)
         :> ada_node)
    | 306 ->
        (wrap_anonymous_type context (c_value)
         :> ada_node)
    | 307 ->
        (wrap_enum_lit_synth_type_expr context (c_value)
         :> ada_node)
    | 308 ->
        (wrap_subtype_indication context (c_value)
         :> ada_node)
    | 309 ->
        (wrap_constrained_subtype_indication context (c_value)
         :> ada_node)
    | 310 ->
        (wrap_discrete_subtype_indication context (c_value)
         :> ada_node)
    | 311 ->
        (wrap_synthetic_type_expr context (c_value)
         :> ada_node)
    | 312 ->
        (wrap_unconstrained_array_index context (c_value)
         :> ada_node)
    | 313 ->
        (wrap_until_absent context (c_value)
         :> ada_node)
    | 314 ->
        (wrap_until_present context (c_value)
         :> ada_node)
    | 315 ->
        (wrap_use_package_clause context (c_value)
         :> ada_node)
    | 316 ->
        (wrap_use_type_clause context (c_value)
         :> ada_node)
    | 317 ->
        (wrap_value_sequence context (c_value)
         :> ada_node)
    | 318 ->
        (wrap_variant context (c_value)
         :> ada_node)
    | 319 ->
        (wrap_variant_part context (c_value)
         :> ada_node)
    | 320 ->
        (wrap_with_clause context (c_value)
         :> ada_node)
    | 321 ->
        (wrap_with_private_absent context (c_value)
         :> ada_node)
    | 322 ->
        (wrap_with_private_present context (c_value)
         :> ada_node)
    | _ -> assert false

      

  and wrap_abort_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AbortAbsent _
      | `AbortPresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_abort_absent context c_value
   : abort_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AbortAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_abort_present context c_value
   : abort_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AbortPresent {
        c_value = c_value;
        context = context
      }

      

  and wrap_abstract_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AbstractAbsent _
      | `AbstractPresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_abstract_absent context c_value
   : abstract_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AbstractAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_abstract_present context c_value
   : abstract_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AbstractPresent {
        c_value = c_value;
        context = context
      }

      

  and wrap_ada_list context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AdaNodeList _
      | `AbstractStateDeclList _
      | `AlternativesList _
      | `ConstraintList _
      | `DeclList _
      | `StmtList _
      | `AspectAssocList _
      | `BaseAssocList _
      | `AssocList _
      | `BasicDeclList _
      | `CaseExprAlternativeList _
      | `CaseStmtAlternativeList _
      | `CompilationUnitList _
      | `ConcatOperandList _
      | `ContractCaseAssocList _
      | `DefiningNameList _
      | `DiscriminantSpecList _
      | `ElsifExprPartList _
      | `ElsifStmtPartList _
      | `EnumLiteralDeclList _
      | `ExprAlternativesList _
      | `DiscriminantChoiceList _
      | `NameList _
      | `ParentList _
      | `ParamSpecList _
      | `PragmaNodeList _
      | `SelectWhenPartList _
      | `UnconstrainedArrayIndexList _
      | `VariantList _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_ada_node_list context c_value
   : ada_node_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_ada_node context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AdaNodeList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_abstract_state_decl_list context c_value
   : abstract_state_decl_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_ada_node context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AbstractStateDeclList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_alternatives_list context c_value
   : alternatives_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_ada_node context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AlternativesList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_constraint_list context c_value
   : constraint_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_ada_node context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ConstraintList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_decl_list context c_value
   : decl_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_ada_node context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DeclList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_stmt_list context c_value
   : stmt_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_ada_node context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `StmtList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_aspect_assoc_list context c_value
   : aspect_assoc_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_aspect_assoc context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AspectAssocList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_base_assoc_list context c_value
   : base_assoc_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_base_assoc context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BaseAssocList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      

  and wrap_basic_assoc_list context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AssocList _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_assoc_list context c_value
   : assoc_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_basic_assoc context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AssocList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_basic_decl_list context c_value
   : basic_decl_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_basic_decl context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BasicDeclList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_case_expr_alternative_list context c_value
   : case_expr_alternative_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_case_expr_alternative context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CaseExprAlternativeList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_case_stmt_alternative_list context c_value
   : case_stmt_alternative_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_case_stmt_alternative context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CaseStmtAlternativeList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_compilation_unit_list context c_value
   : compilation_unit_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_compilation_unit context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CompilationUnitList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_concat_operand_list context c_value
   : concat_operand_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_concat_operand context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ConcatOperandList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_contract_case_assoc_list context c_value
   : contract_case_assoc_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_contract_case_assoc context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ContractCaseAssocList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_defining_name_list context c_value
   : defining_name_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_defining_name context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DefiningNameList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_discriminant_spec_list context c_value
   : discriminant_spec_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_discriminant_spec context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DiscriminantSpecList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_elsif_expr_part_list context c_value
   : elsif_expr_part_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_elsif_expr_part context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ElsifExprPartList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_elsif_stmt_part_list context c_value
   : elsif_stmt_part_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_elsif_stmt_part context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ElsifStmtPartList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_enum_literal_decl_list context c_value
   : enum_literal_decl_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_enum_literal_decl context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnumLiteralDeclList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      

  and wrap_expr_list context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `ExprAlternativesList _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_expr_alternatives_list context c_value
   : expr_alternatives_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_expr context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ExprAlternativesList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      

  and wrap_identifier_list context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `DiscriminantChoiceList _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_discriminant_choice_list context c_value
   : discriminant_choice_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_identifier context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DiscriminantChoiceList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_name_list context c_value
   : name_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_name context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NameList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_parent_list context c_value
   : parent_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_name context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ParentList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_param_spec_list context c_value
   : param_spec_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_param_spec context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ParamSpecList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_pragma_node_list context c_value
   : pragma_node_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_pragma_node context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PragmaNodeList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_select_when_part_list context c_value
   : select_when_part_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_select_when_part context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SelectWhenPartList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_unconstrained_array_index_list context c_value
   : unconstrained_array_index_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_unconstrained_array_index context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `UnconstrainedArrayIndexList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_variant_list context c_value
   : variant_list =
    let list () =
      let c_value_ptr =
        allocate_n AdaNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.ada_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value AdaNodeArrayStruct.n in
      let items = c_value @. AdaNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_variant context ((!@ fresh))
      in
      let result = List.init length f in
      AdaNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `VariantList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      

  and wrap_aliased_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AliasedAbsent _
      | `AliasedPresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_aliased_absent context c_value
   : aliased_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AliasedAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_aliased_present context c_value
   : aliased_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AliasedPresent {
        c_value = c_value;
        context = context
      }

      

  and wrap_all_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AllAbsent _
      | `AllPresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_all_absent context c_value
   : all_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AllAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_all_present context c_value
   : all_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AllPresent {
        c_value = c_value;
        context = context
      }

      

  and wrap_array_indices context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `ConstrainedArrayIndices _
      | `UnconstrainedArrayIndices _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_constrained_array_indices context c_value
   : constrained_array_indices =
    let f_list () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.constrained_array_indices_f_list
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_constraint_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ConstrainedArrayIndices {
        f_list
          = Lazy.from_fun f_list ;
        c_value = c_value;
        context = context
      }

      


  and wrap_unconstrained_array_indices context c_value
   : unconstrained_array_indices =
    let f_types () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.unconstrained_array_indices_f_types
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_unconstrained_array_index_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `UnconstrainedArrayIndices {
        f_types
          = Lazy.from_fun f_types ;
        c_value = c_value;
        context = context
      }

      


  and wrap_aspect_assoc context c_value
   : aspect_assoc =
    let f_id () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.aspect_assoc_f_id
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.aspect_assoc_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `AbstractStateDeclExpr _
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `ContractCases _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AspectAssoc {
        f_id
          = Lazy.from_fun f_id ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      

  and wrap_aspect_clause context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AtClause _
      | `AttributeDefClause _
      | `EnumRepClause _
      | `RecordRepClause _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_at_clause context c_value
   : at_clause =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.at_clause_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_base_id context (field_c_value)
      in
         

      match node with
            
      | `CharLiteral _
      | `Identifier _
      | `StringLiteral _
      as e -> e
      | _ -> assert false
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.at_clause_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AtClause {
        f_name
          = Lazy.from_fun f_name ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_attribute_def_clause context c_value
   : attribute_def_clause =
    let f_attribute_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.attribute_def_clause_f_attribute_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.attribute_def_clause_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AttributeDefClause {
        f_attribute_expr
          = Lazy.from_fun f_attribute_expr ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_enum_rep_clause context c_value
   : enum_rep_clause =
    let f_type_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.enum_rep_clause_f_type_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `CharLiteral _
      | `DottedName _
      | `Identifier _
      | `StringLiteral _
      as e -> e
      | _ -> assert false
    in
    let f_aggregate () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.enum_rep_clause_f_aggregate
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_base_aggregate context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnumRepClause {
        f_type_name
          = Lazy.from_fun f_type_name ;
        f_aggregate
          = Lazy.from_fun f_aggregate ;
        c_value = c_value;
        context = context
      }

      


  and wrap_record_rep_clause context c_value
   : record_rep_clause =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.record_rep_clause_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `CharLiteral _
      | `DottedName _
      | `Identifier _
      | `StringLiteral _
      as e -> e
      | _ -> assert false
    in
    let f_at_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.record_rep_clause_f_at_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_components () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.record_rep_clause_f_components
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RecordRepClause {
        f_name
          = Lazy.from_fun f_name ;
        f_at_expr
          = Lazy.from_fun f_at_expr ;
        f_components
          = Lazy.from_fun f_components ;
        c_value = c_value;
        context = context
      }

      


  and wrap_aspect_spec context c_value
   : aspect_spec =
    let f_aspect_assocs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.aspect_spec_f_aspect_assocs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_aspect_assoc_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AspectSpec {
        f_aspect_assocs
          = Lazy.from_fun f_aspect_assocs ;
        c_value = c_value;
        context = context
      }

      

  and wrap_base_assoc context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `ContractCaseAssoc _
      | `PragmaArgumentAssoc _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_contract_case_assoc context c_value
   : contract_case_assoc =
    let f_guard () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.contract_case_assoc_f_guard
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `OthersDesignator _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_consequence () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.contract_case_assoc_f_consequence
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ContractCaseAssoc {
        f_guard
          = Lazy.from_fun f_guard ;
        f_consequence
          = Lazy.from_fun f_consequence ;
        c_value = c_value;
        context = context
      }

      


  and wrap_pragma_argument_assoc context c_value
   : pragma_argument_assoc =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.pragma_argument_assoc_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_name context (field_c_value))
      in
         

      match node with
            
      | Some `AttributeRef _
      | Some `Identifier _
      | None as e -> e
      | _ -> assert false
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.pragma_argument_assoc_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PragmaArgumentAssoc {
        f_name
          = Lazy.from_fun f_name ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      

  and wrap_base_formal_param_holder context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `EntrySpec _
      | `EnumSubpSpec _
      | `SubpSpec _
      | `SyntheticBinarySpec _
      | `SyntheticUnarySpec _
      | `ComponentList _
      | `KnownDiscriminantPart _
      | `UnknownDiscriminantPart _
      | `EntryCompletionFormalParams _
      | `GenericFormalPart _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      

  and wrap_base_subp_spec context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `EntrySpec _
      | `EnumSubpSpec _
      | `SubpSpec _
      | `SyntheticBinarySpec _
      | `SyntheticUnarySpec _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_entry_spec context c_value
   : entry_spec =
    let f_entry_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_spec_f_entry_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_family_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_spec_f_family_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_ada_node context (field_c_value))
      in
         

      match node with
            
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `CallExpr _
      | Some `CharLiteral _
      | Some `ConstrainedSubtypeIndication _
      | Some `DiscreteSubtypeIndication _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `QualExpr _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `SubtypeIndication _
      | Some `TargetName _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_entry_params () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_spec_f_entry_params
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_params context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EntrySpec {
        f_entry_name
          = Lazy.from_fun f_entry_name ;
        f_family_type
          = Lazy.from_fun f_family_type ;
        f_entry_params
          = Lazy.from_fun f_entry_params ;
        c_value = c_value;
        context = context
      }

      


  and wrap_enum_subp_spec context c_value
   : enum_subp_spec =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnumSubpSpec {
        c_value = c_value;
        context = context
      }

      


  and wrap_subp_spec context c_value
   : subp_spec =
    let f_subp_kind () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subp_spec_f_subp_kind
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subp_kind context (field_c_value)
      in
         

      node
    in
    let f_subp_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subp_spec_f_subp_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    let f_subp_params () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subp_spec_f_subp_params
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_params context (field_c_value))
      in
         

      node
    in
    let f_subp_returns () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subp_spec_f_subp_returns
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_type_expr context (field_c_value))
      in
         

      match node with
            
      | Some `AnonymousType _
      | Some `ConstrainedSubtypeIndication _
      | Some `DiscreteSubtypeIndication _
      | Some `SubtypeIndication _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SubpSpec {
        f_subp_kind
          = Lazy.from_fun f_subp_kind ;
        f_subp_name
          = Lazy.from_fun f_subp_name ;
        f_subp_params
          = Lazy.from_fun f_subp_params ;
        f_subp_returns
          = Lazy.from_fun f_subp_returns ;
        c_value = c_value;
        context = context
      }

      


  and wrap_synthetic_binary_spec context c_value
   : synthetic_binary_spec =
    let f_left_param () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.synthetic_binary_spec_f_left_param
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_synthetic_formal_param_decl context (field_c_value)
      in
         

      node
    in
    let f_right_param () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.synthetic_binary_spec_f_right_param
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_synthetic_formal_param_decl context (field_c_value)
      in
         

      node
    in
    let f_return_type_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.synthetic_binary_spec_f_return_type_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_type_expr context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SyntheticBinarySpec {
        f_left_param
          = Lazy.from_fun f_left_param ;
        f_right_param
          = Lazy.from_fun f_right_param ;
        f_return_type_expr
          = Lazy.from_fun f_return_type_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_synthetic_unary_spec context c_value
   : synthetic_unary_spec =
    let f_right_param () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.synthetic_unary_spec_f_right_param
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_synthetic_formal_param_decl context (field_c_value)
      in
         

      node
    in
    let f_return_type_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.synthetic_unary_spec_f_return_type_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_synthetic_type_expr context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SyntheticUnarySpec {
        f_right_param
          = Lazy.from_fun f_right_param ;
        f_return_type_expr
          = Lazy.from_fun f_return_type_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_component_list context c_value
   : component_list =
    let f_components () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.component_list_f_components
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node_list context (field_c_value)
      in
         

      node
    in
    let f_variant_part () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.component_list_f_variant_part
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_variant_part context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ComponentList {
        f_components
          = Lazy.from_fun f_components ;
        f_variant_part
          = Lazy.from_fun f_variant_part ;
        c_value = c_value;
        context = context
      }

      

  and wrap_discriminant_part context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `KnownDiscriminantPart _
      | `UnknownDiscriminantPart _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_known_discriminant_part context c_value
   : known_discriminant_part =
    let f_discr_specs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.known_discriminant_part_f_discr_specs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_discriminant_spec_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `KnownDiscriminantPart {
        f_discr_specs
          = Lazy.from_fun f_discr_specs ;
        c_value = c_value;
        context = context
      }

      


  and wrap_unknown_discriminant_part context c_value
   : unknown_discriminant_part =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `UnknownDiscriminantPart {
        c_value = c_value;
        context = context
      }

      


  and wrap_entry_completion_formal_params context c_value
   : entry_completion_formal_params =
    let f_params () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_completion_formal_params_f_params
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_params context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EntryCompletionFormalParams {
        f_params
          = Lazy.from_fun f_params ;
        c_value = c_value;
        context = context
      }

      


  and wrap_generic_formal_part context c_value
   : generic_formal_part =
    let f_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_formal_part_f_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericFormalPart {
        f_decls
          = Lazy.from_fun f_decls ;
        c_value = c_value;
        context = context
      }

      

  and wrap_base_record_def context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `NullRecordDef _
      | `RecordDef _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_null_record_def context c_value
   : null_record_def =
    let f_components () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_record_def_f_components
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_component_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NullRecordDef {
        f_components
          = Lazy.from_fun f_components ;
        c_value = c_value;
        context = context
      }

      


  and wrap_record_def context c_value
   : record_def =
    let f_components () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_record_def_f_components
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_component_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RecordDef {
        f_components
          = Lazy.from_fun f_components ;
        c_value = c_value;
        context = context
      }

      

  and wrap_basic_assoc context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AggregateAssoc _
      | `MultiDimArrayAssoc _
      | `CompositeConstraintAssoc _
      | `IteratedAssoc _
      | `ParamAssoc _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_aggregate_assoc context c_value
   : aggregate_assoc =
    let f_designators () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.aggregate_assoc_f_designators
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_alternatives_list context (field_c_value)
      in
         

      node
    in
    let f_r_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.aggregate_assoc_f_r_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BoxExpr _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AggregateAssoc {
        f_designators
          = Lazy.from_fun f_designators ;
        f_r_expr
          = Lazy.from_fun f_r_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_multi_dim_array_assoc context c_value
   : multi_dim_array_assoc =
    let f_designators () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.aggregate_assoc_f_designators
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_alternatives_list context (field_c_value)
      in
         

      node
    in
    let f_r_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.aggregate_assoc_f_r_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BoxExpr _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `MultiDimArrayAssoc {
        f_designators
          = Lazy.from_fun f_designators ;
        f_r_expr
          = Lazy.from_fun f_r_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_composite_constraint_assoc context c_value
   : composite_constraint_assoc =
    let f_ids () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.composite_constraint_assoc_f_ids
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_discriminant_choice_list context (field_c_value)
      in
         

      node
    in
    let f_constraint_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.composite_constraint_assoc_f_constraint_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DiscreteSubtypeIndication _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CompositeConstraintAssoc {
        f_ids
          = Lazy.from_fun f_ids ;
        f_constraint_expr
          = Lazy.from_fun f_constraint_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_iterated_assoc context c_value
   : iterated_assoc =
    let f_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.iterated_assoc_f_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_for_loop_spec context (field_c_value)
      in
         

      node
    in
    let f_r_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.iterated_assoc_f_r_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `IteratedAssoc {
        f_spec
          = Lazy.from_fun f_spec ;
        f_r_expr
          = Lazy.from_fun f_r_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_param_assoc context c_value
   : param_assoc =
    let f_designator () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.param_assoc_f_designator
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_ada_node context (field_c_value))
      in
         

      match node with
            
      | Some `Identifier _
      | Some `OthersDesignator _
      | Some `StringLiteral _
      | None as e -> e
      | _ -> assert false
    in
    let f_r_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.param_assoc_f_r_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BoxExpr _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ParamAssoc {
        f_designator
          = Lazy.from_fun f_designator ;
        f_r_expr
          = Lazy.from_fun f_r_expr ;
        c_value = c_value;
        context = context
      }

      

  and wrap_basic_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AbstractStateDecl _
      | `AnonymousExprDecl _
      | `ComponentDecl _
      | `DiscriminantSpec _
      | `GenericFormalObjDecl _
      | `GenericFormalPackage _
      | `GenericFormalSubpDecl _
      | `GenericFormalTypeDecl _
      | `ParamSpec _
      | `SyntheticFormalParamDecl _
      | `GenericPackageInternal _
      | `PackageDecl _
      | `DiscreteBaseSubtypeDecl _
      | `SubtypeDecl _
      | `ClasswideTypeDecl _
      | `IncompleteTypeDecl _
      | `IncompleteFormalTypeDecl _
      | `IncompleteTaggedTypeDecl _
      | `ProtectedTypeDecl _
      | `TaskTypeDecl _
      | `SingleTaskTypeDecl _
      | `AnonymousTypeDecl _
      | `SynthAnonymousTypeDecl _
      | `ConcreteTypeDecl _
      | `FormalTypeDecl _
      | `AbstractSubpDecl _
      | `AbstractFormalSubpDecl _
      | `ConcreteFormalSubpDecl _
      | `SubpDecl _
      | `EntryDecl _
      | `EnumLiteralDecl _
      | `SyntheticCharEnumLit _
      | `GenericSubpInternal _
      | `SyntheticSubpDecl _
      | `AcceptStmtBody _
      | `ExprFunction _
      | `NullSubpDecl _
      | `SubpBody _
      | `SubpRenamingDecl _
      | `PackageBodyStub _
      | `ProtectedBodyStub _
      | `SubpBodyStub _
      | `TaskBodyStub _
      | `EntryBody _
      | `PackageBody _
      | `ProtectedBody _
      | `TaskBody _
      | `EntryIndexSpec _
      | `ErrorDecl _
      | `ExceptionDecl _
      | `ExceptionHandler _
      | `ForLoopVarDecl _
      | `GenericPackageDecl _
      | `GenericSubpDecl _
      | `GenericPackageInstantiation _
      | `GenericSubpInstantiation _
      | `GenericPackageRenamingDecl _
      | `GenericSubpRenamingDecl _
      | `LabelDecl _
      | `NamedStmtDecl _
      | `NumberDecl _
      | `ObjectDecl _
      | `ExtendedReturnStmtObjectDecl _
      | `NoTypeObjectRenamingDecl _
      | `PackageRenamingDecl _
      | `SingleProtectedDecl _
      | `SingleTaskDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_abstract_state_decl context c_value
   : abstract_state_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.abstract_state_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AbstractStateDecl {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_anonymous_expr_decl context c_value
   : anonymous_expr_decl =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.anonymous_expr_decl_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AnonymousExprDecl {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      

  and wrap_base_formal_param_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `ComponentDecl _
      | `DiscriminantSpec _
      | `GenericFormalObjDecl _
      | `GenericFormalPackage _
      | `GenericFormalSubpDecl _
      | `GenericFormalTypeDecl _
      | `ParamSpec _
      | `SyntheticFormalParamDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_component_decl context c_value
   : component_decl =
    let f_ids () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.component_decl_f_ids
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name_list context (field_c_value)
      in
         

      node
    in
    let f_component_def () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.component_decl_f_component_def
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_component_def context (field_c_value)
      in
         

      node
    in
    let f_default_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.component_decl_f_default_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ComponentDecl {
        f_ids
          = Lazy.from_fun f_ids ;
        f_component_def
          = Lazy.from_fun f_component_def ;
        f_default_expr
          = Lazy.from_fun f_default_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_discriminant_spec context c_value
   : discriminant_spec =
    let f_ids () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.discriminant_spec_f_ids
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name_list context (field_c_value)
      in
         

      node
    in
    let f_type_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.discriminant_spec_f_type_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_expr context (field_c_value)
      in
         

      match node with
            
      | `AnonymousType _
      | `ConstrainedSubtypeIndication _
      | `DiscreteSubtypeIndication _
      | `SubtypeIndication _
      as e -> e
      | _ -> assert false
    in
    let f_default_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.discriminant_spec_f_default_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DiscriminantSpec {
        f_ids
          = Lazy.from_fun f_ids ;
        f_type_expr
          = Lazy.from_fun f_type_expr ;
        f_default_expr
          = Lazy.from_fun f_default_expr ;
        c_value = c_value;
        context = context
      }

      

  and wrap_generic_formal context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `GenericFormalObjDecl _
      | `GenericFormalPackage _
      | `GenericFormalSubpDecl _
      | `GenericFormalTypeDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_generic_formal_obj_decl context c_value
   : generic_formal_obj_decl =
    let f_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_formal_f_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_basic_decl context (field_c_value)
      in
         

      match node with
            
      | `AbstractFormalSubpDecl _
      | `ConcreteFormalSubpDecl _
      | `ExtendedReturnStmtObjectDecl _
      | `FormalTypeDecl _
      | `GenericPackageInstantiation _
      | `GenericSubpInstantiation _
      | `IncompleteFormalTypeDecl _
      | `NoTypeObjectRenamingDecl _
      | `NumberDecl _
      | `ObjectDecl _
      | `SingleProtectedDecl _
      | `SingleTaskDecl _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericFormalObjDecl {
        f_decl
          = Lazy.from_fun f_decl ;
        c_value = c_value;
        context = context
      }

      


  and wrap_generic_formal_package context c_value
   : generic_formal_package =
    let f_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_formal_f_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_basic_decl context (field_c_value)
      in
         

      match node with
            
      | `AbstractFormalSubpDecl _
      | `ConcreteFormalSubpDecl _
      | `ExtendedReturnStmtObjectDecl _
      | `FormalTypeDecl _
      | `GenericPackageInstantiation _
      | `GenericSubpInstantiation _
      | `IncompleteFormalTypeDecl _
      | `NoTypeObjectRenamingDecl _
      | `NumberDecl _
      | `ObjectDecl _
      | `SingleProtectedDecl _
      | `SingleTaskDecl _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericFormalPackage {
        f_decl
          = Lazy.from_fun f_decl ;
        c_value = c_value;
        context = context
      }

      


  and wrap_generic_formal_subp_decl context c_value
   : generic_formal_subp_decl =
    let f_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_formal_f_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_basic_decl context (field_c_value)
      in
         

      match node with
            
      | `AbstractFormalSubpDecl _
      | `ConcreteFormalSubpDecl _
      | `ExtendedReturnStmtObjectDecl _
      | `FormalTypeDecl _
      | `GenericPackageInstantiation _
      | `GenericSubpInstantiation _
      | `IncompleteFormalTypeDecl _
      | `NoTypeObjectRenamingDecl _
      | `NumberDecl _
      | `ObjectDecl _
      | `SingleProtectedDecl _
      | `SingleTaskDecl _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericFormalSubpDecl {
        f_decl
          = Lazy.from_fun f_decl ;
        c_value = c_value;
        context = context
      }

      


  and wrap_generic_formal_type_decl context c_value
   : generic_formal_type_decl =
    let f_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_formal_f_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_basic_decl context (field_c_value)
      in
         

      match node with
            
      | `AbstractFormalSubpDecl _
      | `ConcreteFormalSubpDecl _
      | `ExtendedReturnStmtObjectDecl _
      | `FormalTypeDecl _
      | `GenericPackageInstantiation _
      | `GenericSubpInstantiation _
      | `IncompleteFormalTypeDecl _
      | `NoTypeObjectRenamingDecl _
      | `NumberDecl _
      | `ObjectDecl _
      | `SingleProtectedDecl _
      | `SingleTaskDecl _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericFormalTypeDecl {
        f_decl
          = Lazy.from_fun f_decl ;
        c_value = c_value;
        context = context
      }

      


  and wrap_param_spec context c_value
   : param_spec =
    let f_ids () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.param_spec_f_ids
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name_list context (field_c_value)
      in
         

      node
    in
    let f_has_aliased () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.param_spec_f_has_aliased
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_aliased_node context (field_c_value)
      in
         

      node
    in
    let f_mode () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.param_spec_f_mode
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_mode context (field_c_value))
      in
         

      node
    in
    let f_type_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.param_spec_f_type_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_expr context (field_c_value)
      in
         

      match node with
            
      | `AnonymousType _
      | `ConstrainedSubtypeIndication _
      | `DiscreteSubtypeIndication _
      | `SubtypeIndication _
      as e -> e
      | _ -> assert false
    in
    let f_default_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.param_spec_f_default_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ParamSpec {
        f_ids
          = Lazy.from_fun f_ids ;
        f_has_aliased
          = Lazy.from_fun f_has_aliased ;
        f_mode
          = Lazy.from_fun f_mode ;
        f_type_expr
          = Lazy.from_fun f_type_expr ;
        f_default_expr
          = Lazy.from_fun f_default_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_synthetic_formal_param_decl context c_value
   : synthetic_formal_param_decl =
    let f_param_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.synthetic_formal_param_decl_f_param_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_expr context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SyntheticFormalParamDecl {
        f_param_type
          = Lazy.from_fun f_param_type ;
        c_value = c_value;
        context = context
      }

      

  and wrap_base_package_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `GenericPackageInternal _
      | `PackageDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_generic_package_internal context c_value
   : generic_package_internal =
    let f_package_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_package_decl_f_package_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_public_part () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_package_decl_f_public_part
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_public_part context (field_c_value)
      in
         

      node
    in
    let f_private_part () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_package_decl_f_private_part
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_private_part context (field_c_value))
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_package_decl_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericPackageInternal {
        f_package_name
          = Lazy.from_fun f_package_name ;
        f_public_part
          = Lazy.from_fun f_public_part ;
        f_private_part
          = Lazy.from_fun f_private_part ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_package_decl context c_value
   : package_decl =
    let f_package_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_package_decl_f_package_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_public_part () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_package_decl_f_public_part
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_public_part context (field_c_value)
      in
         

      node
    in
    let f_private_part () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_package_decl_f_private_part
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_private_part context (field_c_value))
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_package_decl_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PackageDecl {
        f_package_name
          = Lazy.from_fun f_package_name ;
        f_public_part
          = Lazy.from_fun f_public_part ;
        f_private_part
          = Lazy.from_fun f_private_part ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      

  and wrap_base_type_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `DiscreteBaseSubtypeDecl _
      | `SubtypeDecl _
      | `ClasswideTypeDecl _
      | `IncompleteTypeDecl _
      | `IncompleteFormalTypeDecl _
      | `IncompleteTaggedTypeDecl _
      | `ProtectedTypeDecl _
      | `TaskTypeDecl _
      | `SingleTaskTypeDecl _
      | `AnonymousTypeDecl _
      | `SynthAnonymousTypeDecl _
      | `ConcreteTypeDecl _
      | `FormalTypeDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      

  and wrap_base_subtype_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `DiscreteBaseSubtypeDecl _
      | `SubtypeDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_discrete_base_subtype_decl context c_value
   : discrete_base_subtype_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_type_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DiscreteBaseSubtypeDecl {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_subtype_decl context c_value
   : subtype_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_type_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    let f_subtype () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subtype_decl_f_subtype
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subtype_indication context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SubtypeDecl {
        f_name
          = Lazy.from_fun f_name ;
        f_subtype
          = Lazy.from_fun f_subtype ;
        c_value = c_value;
        context = context
      }

      


  and wrap_classwide_type_decl context c_value
   : classwide_type_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_type_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ClasswideTypeDecl {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_incomplete_type_decl context c_value
   : incomplete_type_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_type_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    let f_discriminants () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.incomplete_type_decl_f_discriminants
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_discriminant_part context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `IncompleteTypeDecl {
        f_name
          = Lazy.from_fun f_name ;
        f_discriminants
          = Lazy.from_fun f_discriminants ;
        c_value = c_value;
        context = context
      }

      


  and wrap_incomplete_formal_type_decl context c_value
   : incomplete_formal_type_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_type_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    let f_discriminants () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.incomplete_type_decl_f_discriminants
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_discriminant_part context (field_c_value))
      in
         

      node
    in
    let f_is_tagged () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.incomplete_formal_type_decl_f_is_tagged
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_tagged_node context (field_c_value))
      in
         

      node
    in
    let f_default_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.incomplete_formal_type_decl_f_default_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_name context (field_c_value))
      in
         

      match node with
            
      | Some `AttributeRef _
      | Some `CallExpr _
      | Some `CharLiteral _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `QualExpr _
      | Some `ReduceAttributeRef _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `IncompleteFormalTypeDecl {
        f_name
          = Lazy.from_fun f_name ;
        f_discriminants
          = Lazy.from_fun f_discriminants ;
        f_is_tagged
          = Lazy.from_fun f_is_tagged ;
        f_default_type
          = Lazy.from_fun f_default_type ;
        c_value = c_value;
        context = context
      }

      


  and wrap_incomplete_tagged_type_decl context c_value
   : incomplete_tagged_type_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_type_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    let f_discriminants () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.incomplete_type_decl_f_discriminants
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_discriminant_part context (field_c_value))
      in
         

      node
    in
    let f_has_abstract () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.incomplete_tagged_type_decl_f_has_abstract
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_abstract_node context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `IncompleteTaggedTypeDecl {
        f_name
          = Lazy.from_fun f_name ;
        f_discriminants
          = Lazy.from_fun f_discriminants ;
        f_has_abstract
          = Lazy.from_fun f_has_abstract ;
        c_value = c_value;
        context = context
      }

      


  and wrap_protected_type_decl context c_value
   : protected_type_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_type_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    let f_discriminants () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.protected_type_decl_f_discriminants
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_discriminant_part context (field_c_value))
      in
         

      node
    in
    let f_interfaces () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.protected_type_decl_f_interfaces
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_parent_list context (field_c_value)
      in
         

      node
    in
    let f_definition () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.protected_type_decl_f_definition
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_protected_def context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ProtectedTypeDecl {
        f_name
          = Lazy.from_fun f_name ;
        f_discriminants
          = Lazy.from_fun f_discriminants ;
        f_interfaces
          = Lazy.from_fun f_interfaces ;
        f_definition
          = Lazy.from_fun f_definition ;
        c_value = c_value;
        context = context
      }

      


  and wrap_task_type_decl context c_value
   : task_type_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_type_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    let f_discriminants () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.task_type_decl_f_discriminants
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_discriminant_part context (field_c_value))
      in
         

      node
    in
    let f_definition () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.task_type_decl_f_definition
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_task_def context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TaskTypeDecl {
        f_name
          = Lazy.from_fun f_name ;
        f_discriminants
          = Lazy.from_fun f_discriminants ;
        f_definition
          = Lazy.from_fun f_definition ;
        c_value = c_value;
        context = context
      }

      


  and wrap_single_task_type_decl context c_value
   : single_task_type_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_type_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    let f_discriminants () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.task_type_decl_f_discriminants
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_discriminant_part context (field_c_value))
      in
         

      node
    in
    let f_definition () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.task_type_decl_f_definition
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_task_def context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SingleTaskTypeDecl {
        f_name
          = Lazy.from_fun f_name ;
        f_discriminants
          = Lazy.from_fun f_discriminants ;
        f_definition
          = Lazy.from_fun f_definition ;
        c_value = c_value;
        context = context
      }

      

  and wrap_type_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AnonymousTypeDecl _
      | `SynthAnonymousTypeDecl _
      | `ConcreteTypeDecl _
      | `FormalTypeDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_anonymous_type_decl context c_value
   : anonymous_type_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_type_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    let f_discriminants () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.type_decl_f_discriminants
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_discriminant_part context (field_c_value))
      in
         

      node
    in
    let f_type_def () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.type_decl_f_type_def
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_def context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AnonymousTypeDecl {
        f_name
          = Lazy.from_fun f_name ;
        f_discriminants
          = Lazy.from_fun f_discriminants ;
        f_type_def
          = Lazy.from_fun f_type_def ;
        c_value = c_value;
        context = context
      }

      


  and wrap_synth_anonymous_type_decl context c_value
   : synth_anonymous_type_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_type_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    let f_discriminants () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.type_decl_f_discriminants
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_discriminant_part context (field_c_value))
      in
         

      node
    in
    let f_type_def () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.type_decl_f_type_def
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_def context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SynthAnonymousTypeDecl {
        f_name
          = Lazy.from_fun f_name ;
        f_discriminants
          = Lazy.from_fun f_discriminants ;
        f_type_def
          = Lazy.from_fun f_type_def ;
        c_value = c_value;
        context = context
      }

      


  and wrap_concrete_type_decl context c_value
   : concrete_type_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_type_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    let f_discriminants () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.type_decl_f_discriminants
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_discriminant_part context (field_c_value))
      in
         

      node
    in
    let f_type_def () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.type_decl_f_type_def
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_def context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ConcreteTypeDecl {
        f_name
          = Lazy.from_fun f_name ;
        f_discriminants
          = Lazy.from_fun f_discriminants ;
        f_type_def
          = Lazy.from_fun f_type_def ;
        c_value = c_value;
        context = context
      }

      


  and wrap_formal_type_decl context c_value
   : formal_type_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_type_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    let f_discriminants () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.type_decl_f_discriminants
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_discriminant_part context (field_c_value))
      in
         

      node
    in
    let f_type_def () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.type_decl_f_type_def
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_def context (field_c_value)
      in
         

      node
    in
    let f_default_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.formal_type_decl_f_default_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_name context (field_c_value))
      in
         

      match node with
            
      | Some `AttributeRef _
      | Some `CallExpr _
      | Some `CharLiteral _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `QualExpr _
      | Some `ReduceAttributeRef _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `FormalTypeDecl {
        f_name
          = Lazy.from_fun f_name ;
        f_discriminants
          = Lazy.from_fun f_discriminants ;
        f_type_def
          = Lazy.from_fun f_type_def ;
        f_default_type
          = Lazy.from_fun f_default_type ;
        c_value = c_value;
        context = context
      }

      

  and wrap_basic_subp_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AbstractSubpDecl _
      | `AbstractFormalSubpDecl _
      | `ConcreteFormalSubpDecl _
      | `SubpDecl _
      | `EntryDecl _
      | `EnumLiteralDecl _
      | `SyntheticCharEnumLit _
      | `GenericSubpInternal _
      | `SyntheticSubpDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      

  and wrap_classic_subp_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AbstractSubpDecl _
      | `AbstractFormalSubpDecl _
      | `ConcreteFormalSubpDecl _
      | `SubpDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_abstract_subp_decl context c_value
   : abstract_subp_decl =
    let f_overriding () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.classic_subp_decl_f_overriding
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_overriding_node context (field_c_value)
      in
         

      node
    in
    let f_subp_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.classic_subp_decl_f_subp_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subp_spec context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AbstractSubpDecl {
        f_overriding
          = Lazy.from_fun f_overriding ;
        f_subp_spec
          = Lazy.from_fun f_subp_spec ;
        c_value = c_value;
        context = context
      }

      

  and wrap_formal_subp_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AbstractFormalSubpDecl _
      | `ConcreteFormalSubpDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_abstract_formal_subp_decl context c_value
   : abstract_formal_subp_decl =
    let f_overriding () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.classic_subp_decl_f_overriding
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_overriding_node context (field_c_value)
      in
         

      node
    in
    let f_subp_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.classic_subp_decl_f_subp_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subp_spec context (field_c_value)
      in
         

      node
    in
    let f_default_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.formal_subp_decl_f_default_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `AttributeRef _
      | Some `BoxExpr _
      | Some `CallExpr _
      | Some `CharLiteral _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `NullLiteral _
      | Some `QualExpr _
      | Some `ReduceAttributeRef _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AbstractFormalSubpDecl {
        f_overriding
          = Lazy.from_fun f_overriding ;
        f_subp_spec
          = Lazy.from_fun f_subp_spec ;
        f_default_expr
          = Lazy.from_fun f_default_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_concrete_formal_subp_decl context c_value
   : concrete_formal_subp_decl =
    let f_overriding () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.classic_subp_decl_f_overriding
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_overriding_node context (field_c_value)
      in
         

      node
    in
    let f_subp_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.classic_subp_decl_f_subp_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subp_spec context (field_c_value)
      in
         

      node
    in
    let f_default_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.formal_subp_decl_f_default_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `AttributeRef _
      | Some `BoxExpr _
      | Some `CallExpr _
      | Some `CharLiteral _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `NullLiteral _
      | Some `QualExpr _
      | Some `ReduceAttributeRef _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ConcreteFormalSubpDecl {
        f_overriding
          = Lazy.from_fun f_overriding ;
        f_subp_spec
          = Lazy.from_fun f_subp_spec ;
        f_default_expr
          = Lazy.from_fun f_default_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_subp_decl context c_value
   : subp_decl =
    let f_overriding () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.classic_subp_decl_f_overriding
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_overriding_node context (field_c_value)
      in
         

      node
    in
    let f_subp_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.classic_subp_decl_f_subp_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subp_spec context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SubpDecl {
        f_overriding
          = Lazy.from_fun f_overriding ;
        f_subp_spec
          = Lazy.from_fun f_subp_spec ;
        c_value = c_value;
        context = context
      }

      


  and wrap_entry_decl context c_value
   : entry_decl =
    let f_overriding () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_decl_f_overriding
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_overriding_node context (field_c_value)
      in
         

      node
    in
    let f_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_decl_f_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_entry_spec context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EntryDecl {
        f_overriding
          = Lazy.from_fun f_overriding ;
        f_spec
          = Lazy.from_fun f_spec ;
        c_value = c_value;
        context = context
      }

      


  and wrap_enum_literal_decl context c_value
   : enum_literal_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.enum_literal_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnumLiteralDecl {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_synthetic_char_enum_lit context c_value
   : synthetic_char_enum_lit =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.enum_literal_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SyntheticCharEnumLit {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_generic_subp_internal context c_value
   : generic_subp_internal =
    let f_subp_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_subp_internal_f_subp_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subp_spec context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericSubpInternal {
        f_subp_spec
          = Lazy.from_fun f_subp_spec ;
        c_value = c_value;
        context = context
      }

      


  and wrap_synthetic_subp_decl context c_value
   : synthetic_subp_decl =
    let f_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.synthetic_subp_decl_f_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_base_subp_spec context (field_c_value)
      in
         

      match node with
            
      | `SyntheticBinarySpec _
      | `SyntheticUnarySpec _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SyntheticSubpDecl {
        f_spec
          = Lazy.from_fun f_spec ;
        c_value = c_value;
        context = context
      }

      

  and wrap_body_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AcceptStmtBody _
      | `ExprFunction _
      | `NullSubpDecl _
      | `SubpBody _
      | `SubpRenamingDecl _
      | `PackageBodyStub _
      | `ProtectedBodyStub _
      | `SubpBodyStub _
      | `TaskBodyStub _
      | `EntryBody _
      | `PackageBody _
      | `ProtectedBody _
      | `TaskBody _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_accept_stmt_body context c_value
   : accept_stmt_body =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.accept_stmt_body_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AcceptStmtBody {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      

  and wrap_base_subp_body context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `ExprFunction _
      | `NullSubpDecl _
      | `SubpBody _
      | `SubpRenamingDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_expr_function context c_value
   : expr_function =
    let f_overriding () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_subp_body_f_overriding
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_overriding_node context (field_c_value)
      in
         

      node
    in
    let f_subp_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_subp_body_f_subp_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subp_spec context (field_c_value)
      in
         

      node
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.expr_function_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `DeltaAggregate _
      | `NullRecordAggregate _
      | `ParenExpr _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ExprFunction {
        f_overriding
          = Lazy.from_fun f_overriding ;
        f_subp_spec
          = Lazy.from_fun f_subp_spec ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_null_subp_decl context c_value
   : null_subp_decl =
    let f_overriding () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_subp_body_f_overriding
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_overriding_node context (field_c_value)
      in
         

      node
    in
    let f_subp_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_subp_body_f_subp_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subp_spec context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NullSubpDecl {
        f_overriding
          = Lazy.from_fun f_overriding ;
        f_subp_spec
          = Lazy.from_fun f_subp_spec ;
        c_value = c_value;
        context = context
      }

      


  and wrap_subp_body context c_value
   : subp_body =
    let f_overriding () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_subp_body_f_overriding
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_overriding_node context (field_c_value)
      in
         

      node
    in
    let f_subp_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_subp_body_f_subp_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subp_spec context (field_c_value)
      in
         

      node
    in
    let f_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subp_body_f_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_declarative_part context (field_c_value)
      in
         

      node
    in
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subp_body_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_handled_stmts context (field_c_value)
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subp_body_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SubpBody {
        f_overriding
          = Lazy.from_fun f_overriding ;
        f_subp_spec
          = Lazy.from_fun f_subp_spec ;
        f_decls
          = Lazy.from_fun f_decls ;
        f_stmts
          = Lazy.from_fun f_stmts ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_subp_renaming_decl context c_value
   : subp_renaming_decl =
    let f_overriding () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_subp_body_f_overriding
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_overriding_node context (field_c_value)
      in
         

      node
    in
    let f_subp_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_subp_body_f_subp_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subp_spec context (field_c_value)
      in
         

      node
    in
    let f_renames () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subp_renaming_decl_f_renames
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_renaming_clause context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SubpRenamingDecl {
        f_overriding
          = Lazy.from_fun f_overriding ;
        f_subp_spec
          = Lazy.from_fun f_subp_spec ;
        f_renames
          = Lazy.from_fun f_renames ;
        c_value = c_value;
        context = context
      }

      

  and wrap_body_stub context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `PackageBodyStub _
      | `ProtectedBodyStub _
      | `SubpBodyStub _
      | `TaskBodyStub _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_package_body_stub context c_value
   : package_body_stub =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.package_body_stub_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PackageBodyStub {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_protected_body_stub context c_value
   : protected_body_stub =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.protected_body_stub_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ProtectedBodyStub {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_subp_body_stub context c_value
   : subp_body_stub =
    let f_overriding () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subp_body_stub_f_overriding
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_overriding_node context (field_c_value)
      in
         

      node
    in
    let f_subp_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subp_body_stub_f_subp_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subp_spec context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SubpBodyStub {
        f_overriding
          = Lazy.from_fun f_overriding ;
        f_subp_spec
          = Lazy.from_fun f_subp_spec ;
        c_value = c_value;
        context = context
      }

      


  and wrap_task_body_stub context c_value
   : task_body_stub =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.task_body_stub_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TaskBodyStub {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_entry_body context c_value
   : entry_body =
    let f_entry_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_body_f_entry_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_index_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_body_f_index_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_entry_index_spec context (field_c_value))
      in
         

      node
    in
    let f_params () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_body_f_params
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_entry_completion_formal_params context (field_c_value)
      in
         

      node
    in
    let f_barrier () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_body_f_barrier
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_body_f_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_declarative_part context (field_c_value)
      in
         

      node
    in
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_body_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_handled_stmts context (field_c_value)
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_body_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EntryBody {
        f_entry_name
          = Lazy.from_fun f_entry_name ;
        f_index_spec
          = Lazy.from_fun f_index_spec ;
        f_params
          = Lazy.from_fun f_params ;
        f_barrier
          = Lazy.from_fun f_barrier ;
        f_decls
          = Lazy.from_fun f_decls ;
        f_stmts
          = Lazy.from_fun f_stmts ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_package_body context c_value
   : package_body =
    let f_package_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.package_body_f_package_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.package_body_f_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_declarative_part context (field_c_value)
      in
         

      node
    in
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.package_body_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_handled_stmts context (field_c_value))
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.package_body_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PackageBody {
        f_package_name
          = Lazy.from_fun f_package_name ;
        f_decls
          = Lazy.from_fun f_decls ;
        f_stmts
          = Lazy.from_fun f_stmts ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_protected_body context c_value
   : protected_body =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.protected_body_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.protected_body_f_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_declarative_part context (field_c_value)
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.protected_body_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ProtectedBody {
        f_name
          = Lazy.from_fun f_name ;
        f_decls
          = Lazy.from_fun f_decls ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_task_body context c_value
   : task_body =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.task_body_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.task_body_f_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_declarative_part context (field_c_value)
      in
         

      node
    in
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.task_body_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_handled_stmts context (field_c_value)
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.task_body_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TaskBody {
        f_name
          = Lazy.from_fun f_name ;
        f_decls
          = Lazy.from_fun f_decls ;
        f_stmts
          = Lazy.from_fun f_stmts ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_entry_index_spec context c_value
   : entry_index_spec =
    let f_id () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_index_spec_f_id
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_subtype () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.entry_index_spec_f_subtype
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `BinOp _
      | `CallExpr _
      | `CharLiteral _
      | `ConstrainedSubtypeIndication _
      | `DiscreteSubtypeIndication _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `SubtypeIndication _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EntryIndexSpec {
        f_id
          = Lazy.from_fun f_id ;
        f_subtype
          = Lazy.from_fun f_subtype ;
        c_value = c_value;
        context = context
      }

      


  and wrap_error_decl context c_value
   : error_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ErrorDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_exception_decl context c_value
   : exception_decl =
    let f_ids () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.exception_decl_f_ids
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name_list context (field_c_value)
      in
         

      node
    in
    let f_renames () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.exception_decl_f_renames
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_renaming_clause context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ExceptionDecl {
        f_ids
          = Lazy.from_fun f_ids ;
        f_renames
          = Lazy.from_fun f_renames ;
        c_value = c_value;
        context = context
      }

      


  and wrap_exception_handler context c_value
   : exception_handler =
    let f_exception_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.exception_handler_f_exception_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_defining_name context (field_c_value))
      in
         

      node
    in
    let f_handled_exceptions () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.exception_handler_f_handled_exceptions
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_alternatives_list context (field_c_value)
      in
         

      node
    in
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.exception_handler_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_stmt_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ExceptionHandler {
        f_exception_name
          = Lazy.from_fun f_exception_name ;
        f_handled_exceptions
          = Lazy.from_fun f_handled_exceptions ;
        f_stmts
          = Lazy.from_fun f_stmts ;
        c_value = c_value;
        context = context
      }

      


  and wrap_for_loop_var_decl context c_value
   : for_loop_var_decl =
    let f_id () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.for_loop_var_decl_f_id
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_id_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.for_loop_var_decl_f_id_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_type_expr context (field_c_value))
      in
         

      match node with
            
      | Some `AnonymousType _
      | Some `ConstrainedSubtypeIndication _
      | Some `DiscreteSubtypeIndication _
      | Some `SubtypeIndication _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ForLoopVarDecl {
        f_id
          = Lazy.from_fun f_id ;
        f_id_type
          = Lazy.from_fun f_id_type ;
        c_value = c_value;
        context = context
      }

      

  and wrap_generic_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `GenericPackageDecl _
      | `GenericSubpDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_generic_package_decl context c_value
   : generic_package_decl =
    let f_formal_part () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_decl_f_formal_part
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_generic_formal_part context (field_c_value)
      in
         

      node
    in
    let f_package_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_package_decl_f_package_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_generic_package_internal context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericPackageDecl {
        f_formal_part
          = Lazy.from_fun f_formal_part ;
        f_package_decl
          = Lazy.from_fun f_package_decl ;
        c_value = c_value;
        context = context
      }

      


  and wrap_generic_subp_decl context c_value
   : generic_subp_decl =
    let f_formal_part () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_decl_f_formal_part
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_generic_formal_part context (field_c_value)
      in
         

      node
    in
    let f_subp_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_subp_decl_f_subp_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_generic_subp_internal context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericSubpDecl {
        f_formal_part
          = Lazy.from_fun f_formal_part ;
        f_subp_decl
          = Lazy.from_fun f_subp_decl ;
        c_value = c_value;
        context = context
      }

      

  and wrap_generic_instantiation context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `GenericPackageInstantiation _
      | `GenericSubpInstantiation _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_generic_package_instantiation context c_value
   : generic_package_instantiation =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_package_instantiation_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_generic_pkg_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_package_instantiation_f_generic_pkg_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `CharLiteral _
      | `DottedName _
      | `Identifier _
      | `StringLiteral _
      as e -> e
      | _ -> assert false
    in
    let f_params () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_package_instantiation_f_params
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_assoc_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericPackageInstantiation {
        f_name
          = Lazy.from_fun f_name ;
        f_generic_pkg_name
          = Lazy.from_fun f_generic_pkg_name ;
        f_params
          = Lazy.from_fun f_params ;
        c_value = c_value;
        context = context
      }

      


  and wrap_generic_subp_instantiation context c_value
   : generic_subp_instantiation =
    let f_overriding () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_subp_instantiation_f_overriding
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_overriding_node context (field_c_value)
      in
         

      node
    in
    let f_kind () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_subp_instantiation_f_kind
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subp_kind context (field_c_value)
      in
         

      node
    in
    let f_subp_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_subp_instantiation_f_subp_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_generic_subp_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_subp_instantiation_f_generic_subp_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `CharLiteral _
      | `DottedName _
      | `Identifier _
      | `StringLiteral _
      as e -> e
      | _ -> assert false
    in
    let f_params () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_subp_instantiation_f_params
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_assoc_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericSubpInstantiation {
        f_overriding
          = Lazy.from_fun f_overriding ;
        f_kind
          = Lazy.from_fun f_kind ;
        f_subp_name
          = Lazy.from_fun f_subp_name ;
        f_generic_subp_name
          = Lazy.from_fun f_generic_subp_name ;
        f_params
          = Lazy.from_fun f_params ;
        c_value = c_value;
        context = context
      }

      

  and wrap_generic_renaming_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `GenericPackageRenamingDecl _
      | `GenericSubpRenamingDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_generic_package_renaming_decl context c_value
   : generic_package_renaming_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_package_renaming_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_renames () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_package_renaming_decl_f_renames
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `CharLiteral _
      | `DottedName _
      | `Identifier _
      | `StringLiteral _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericPackageRenamingDecl {
        f_name
          = Lazy.from_fun f_name ;
        f_renames
          = Lazy.from_fun f_renames ;
        c_value = c_value;
        context = context
      }

      


  and wrap_generic_subp_renaming_decl context c_value
   : generic_subp_renaming_decl =
    let f_kind () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_subp_renaming_decl_f_kind
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subp_kind context (field_c_value)
      in
         

      node
    in
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_subp_renaming_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_renames () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_subp_renaming_decl_f_renames
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `CharLiteral _
      | `DottedName _
      | `Identifier _
      | `StringLiteral _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericSubpRenamingDecl {
        f_kind
          = Lazy.from_fun f_kind ;
        f_name
          = Lazy.from_fun f_name ;
        f_renames
          = Lazy.from_fun f_renames ;
        c_value = c_value;
        context = context
      }

      


  and wrap_label_decl context c_value
   : label_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.label_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LabelDecl {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_named_stmt_decl context c_value
   : named_stmt_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.named_stmt_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NamedStmtDecl {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_number_decl context c_value
   : number_decl =
    let f_ids () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.number_decl_f_ids
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name_list context (field_c_value)
      in
         

      node
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.number_decl_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NumberDecl {
        f_ids
          = Lazy.from_fun f_ids ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_object_decl context c_value
   : object_decl =
    let f_ids () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_ids
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name_list context (field_c_value)
      in
         

      node
    in
    let f_has_aliased () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_has_aliased
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_aliased_node context (field_c_value))
      in
         

      node
    in
    let f_has_constant () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_has_constant
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_constant_node context (field_c_value))
      in
         

      node
    in
    let f_mode () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_mode
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_mode context (field_c_value))
      in
         

      node
    in
    let f_type_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_type_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_type_expr context (field_c_value))
      in
         

      match node with
            
      | Some `AnonymousType _
      | Some `ConstrainedSubtypeIndication _
      | Some `DiscreteSubtypeIndication _
      | Some `SubtypeIndication _
      | None as e -> e
      | _ -> assert false
    in
    let f_default_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_default_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_renaming_clause () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_renaming_clause
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_renaming_clause context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ObjectDecl {
        f_ids
          = Lazy.from_fun f_ids ;
        f_has_aliased
          = Lazy.from_fun f_has_aliased ;
        f_has_constant
          = Lazy.from_fun f_has_constant ;
        f_mode
          = Lazy.from_fun f_mode ;
        f_type_expr
          = Lazy.from_fun f_type_expr ;
        f_default_expr
          = Lazy.from_fun f_default_expr ;
        f_renaming_clause
          = Lazy.from_fun f_renaming_clause ;
        c_value = c_value;
        context = context
      }

      


  and wrap_extended_return_stmt_object_decl context c_value
   : extended_return_stmt_object_decl =
    let f_ids () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_ids
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name_list context (field_c_value)
      in
         

      node
    in
    let f_has_aliased () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_has_aliased
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_aliased_node context (field_c_value))
      in
         

      node
    in
    let f_has_constant () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_has_constant
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_constant_node context (field_c_value))
      in
         

      node
    in
    let f_mode () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_mode
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_mode context (field_c_value))
      in
         

      node
    in
    let f_type_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_type_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_type_expr context (field_c_value))
      in
         

      match node with
            
      | Some `AnonymousType _
      | Some `ConstrainedSubtypeIndication _
      | Some `DiscreteSubtypeIndication _
      | Some `SubtypeIndication _
      | None as e -> e
      | _ -> assert false
    in
    let f_default_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_default_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_renaming_clause () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_renaming_clause
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_renaming_clause context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ExtendedReturnStmtObjectDecl {
        f_ids
          = Lazy.from_fun f_ids ;
        f_has_aliased
          = Lazy.from_fun f_has_aliased ;
        f_has_constant
          = Lazy.from_fun f_has_constant ;
        f_mode
          = Lazy.from_fun f_mode ;
        f_type_expr
          = Lazy.from_fun f_type_expr ;
        f_default_expr
          = Lazy.from_fun f_default_expr ;
        f_renaming_clause
          = Lazy.from_fun f_renaming_clause ;
        c_value = c_value;
        context = context
      }

      


  and wrap_no_type_object_renaming_decl context c_value
   : no_type_object_renaming_decl =
    let f_ids () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_ids
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name_list context (field_c_value)
      in
         

      node
    in
    let f_has_aliased () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_has_aliased
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_aliased_node context (field_c_value))
      in
         

      node
    in
    let f_has_constant () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_has_constant
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_constant_node context (field_c_value))
      in
         

      node
    in
    let f_mode () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_mode
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_mode context (field_c_value))
      in
         

      node
    in
    let f_type_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_type_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_type_expr context (field_c_value))
      in
         

      match node with
            
      | Some `AnonymousType _
      | Some `ConstrainedSubtypeIndication _
      | Some `DiscreteSubtypeIndication _
      | Some `SubtypeIndication _
      | None as e -> e
      | _ -> assert false
    in
    let f_default_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_default_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_renaming_clause () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.object_decl_f_renaming_clause
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_renaming_clause context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NoTypeObjectRenamingDecl {
        f_ids
          = Lazy.from_fun f_ids ;
        f_has_aliased
          = Lazy.from_fun f_has_aliased ;
        f_has_constant
          = Lazy.from_fun f_has_constant ;
        f_mode
          = Lazy.from_fun f_mode ;
        f_type_expr
          = Lazy.from_fun f_type_expr ;
        f_default_expr
          = Lazy.from_fun f_default_expr ;
        f_renaming_clause
          = Lazy.from_fun f_renaming_clause ;
        c_value = c_value;
        context = context
      }

      


  and wrap_package_renaming_decl context c_value
   : package_renaming_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.package_renaming_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_renames () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.package_renaming_decl_f_renames
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_renaming_clause context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PackageRenamingDecl {
        f_name
          = Lazy.from_fun f_name ;
        f_renames
          = Lazy.from_fun f_renames ;
        c_value = c_value;
        context = context
      }

      


  and wrap_single_protected_decl context c_value
   : single_protected_decl =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.single_protected_decl_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_defining_name context (field_c_value)
      in
         

      node
    in
    let f_interfaces () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.single_protected_decl_f_interfaces
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_parent_list context (field_c_value)
      in
         

      node
    in
    let f_definition () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.single_protected_decl_f_definition
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_protected_def context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SingleProtectedDecl {
        f_name
          = Lazy.from_fun f_name ;
        f_interfaces
          = Lazy.from_fun f_interfaces ;
        f_definition
          = Lazy.from_fun f_definition ;
        c_value = c_value;
        context = context
      }

      


  and wrap_single_task_decl context c_value
   : single_task_decl =
    let f_task_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.single_task_decl_f_task_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_single_task_type_decl context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SingleTaskDecl {
        f_task_type
          = Lazy.from_fun f_task_type ;
        c_value = c_value;
        context = context
      }

      


  and wrap_case_stmt_alternative context c_value
   : case_stmt_alternative =
    let f_choices () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.case_stmt_alternative_f_choices
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_alternatives_list context (field_c_value)
      in
         

      node
    in
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.case_stmt_alternative_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_stmt_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CaseStmtAlternative {
        f_choices
          = Lazy.from_fun f_choices ;
        f_stmts
          = Lazy.from_fun f_stmts ;
        c_value = c_value;
        context = context
      }

      


  and wrap_compilation_unit context c_value
   : compilation_unit =
    let f_prelude () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.compilation_unit_f_prelude
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node_list context (field_c_value)
      in
         

      node
    in
    let f_body () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.compilation_unit_f_body
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node context (field_c_value)
      in
         

      match node with
            
      | `LibraryItem _
      | `Subunit _
      as e -> e
      | _ -> assert false
    in
    let f_pragmas () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.compilation_unit_f_pragmas
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pragma_node_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CompilationUnit {
        f_prelude
          = Lazy.from_fun f_prelude ;
        f_body
          = Lazy.from_fun f_body ;
        f_pragmas
          = Lazy.from_fun f_pragmas ;
        c_value = c_value;
        context = context
      }

      


  and wrap_component_clause context c_value
   : component_clause =
    let f_id () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.component_clause_f_id
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_identifier context (field_c_value)
      in
         

      node
    in
    let f_position () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.component_clause_f_position
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_range () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.component_clause_f_range
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_range_spec context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ComponentClause {
        f_id
          = Lazy.from_fun f_id ;
        f_position
          = Lazy.from_fun f_position ;
        f_range
          = Lazy.from_fun f_range ;
        c_value = c_value;
        context = context
      }

      


  and wrap_component_def context c_value
   : component_def =
    let f_has_aliased () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.component_def_f_has_aliased
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_aliased_node context (field_c_value)
      in
         

      node
    in
    let f_has_constant () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.component_def_f_has_constant
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_constant_node context (field_c_value)
      in
         

      node
    in
    let f_type_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.component_def_f_type_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_expr context (field_c_value)
      in
         

      match node with
            
      | `AnonymousType _
      | `ConstrainedSubtypeIndication _
      | `DiscreteSubtypeIndication _
      | `SubtypeIndication _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ComponentDef {
        f_has_aliased
          = Lazy.from_fun f_has_aliased ;
        f_has_constant
          = Lazy.from_fun f_has_constant ;
        f_type_expr
          = Lazy.from_fun f_type_expr ;
        c_value = c_value;
        context = context
      }

      

  and wrap_constant_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `ConstantAbsent _
      | `ConstantPresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_constant_absent context c_value
   : constant_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ConstantAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_constant_present context c_value
   : constant_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ConstantPresent {
        c_value = c_value;
        context = context
      }

      

  and wrap_constraint context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `CompositeConstraint _
      | `DeltaConstraint _
      | `DigitsConstraint _
      | `RangeConstraint _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_composite_constraint context c_value
   : composite_constraint =
    let f_constraints () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.composite_constraint_f_constraints
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_assoc_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CompositeConstraint {
        f_constraints
          = Lazy.from_fun f_constraints ;
        c_value = c_value;
        context = context
      }

      


  and wrap_delta_constraint context c_value
   : delta_constraint =
    let f_digits () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.delta_constraint_f_digits
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_range () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.delta_constraint_f_range
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_range_spec context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DeltaConstraint {
        f_digits
          = Lazy.from_fun f_digits ;
        f_range
          = Lazy.from_fun f_range ;
        c_value = c_value;
        context = context
      }

      


  and wrap_digits_constraint context c_value
   : digits_constraint =
    let f_digits () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.digits_constraint_f_digits
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_range () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.digits_constraint_f_range
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_range_spec context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DigitsConstraint {
        f_digits
          = Lazy.from_fun f_digits ;
        f_range
          = Lazy.from_fun f_range ;
        c_value = c_value;
        context = context
      }

      


  and wrap_range_constraint context c_value
   : range_constraint =
    let f_range () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.range_constraint_f_range
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_range_spec context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RangeConstraint {
        f_range
          = Lazy.from_fun f_range ;
        c_value = c_value;
        context = context
      }

      


  and wrap_declarative_part context c_value
   : declarative_part =
    let f_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.declarative_part_f_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DeclarativePart {
        f_decls
          = Lazy.from_fun f_decls ;
        c_value = c_value;
        context = context
      }

      


  and wrap_private_part context c_value
   : private_part =
    let f_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.declarative_part_f_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PrivatePart {
        f_decls
          = Lazy.from_fun f_decls ;
        c_value = c_value;
        context = context
      }

      


  and wrap_public_part context c_value
   : public_part =
    let f_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.declarative_part_f_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PublicPart {
        f_decls
          = Lazy.from_fun f_decls ;
        c_value = c_value;
        context = context
      }

      


  and wrap_elsif_expr_part context c_value
   : elsif_expr_part =
    let f_cond_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.elsif_expr_part_f_cond_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_then_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.elsif_expr_part_f_then_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ElsifExprPart {
        f_cond_expr
          = Lazy.from_fun f_cond_expr ;
        f_then_expr
          = Lazy.from_fun f_then_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_elsif_stmt_part context c_value
   : elsif_stmt_part =
    let f_cond_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.elsif_stmt_part_f_cond_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.elsif_stmt_part_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_stmt_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ElsifStmtPart {
        f_cond_expr
          = Lazy.from_fun f_cond_expr ;
        f_stmts
          = Lazy.from_fun f_stmts ;
        c_value = c_value;
        context = context
      }

      

  and wrap_expr context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AbstractStateDeclExpr _
      | `Allocator _
      | `Aggregate _
      | `BracketAggregate _
      | `DeltaAggregate _
      | `BracketDeltaAggregate _
      | `NullRecordAggregate _
      | `BinOp _
      | `RelationOp _
      | `BoxExpr _
      | `CaseExprAlternative _
      | `ConcatOp _
      | `ConcatOperand _
      | `CaseExpr _
      | `IfExpr _
      | `ContractCases _
      | `DeclExpr _
      | `MembershipExpr _
      | `AttributeRef _
      | `CallExpr _
      | `DefiningName _
      | `SyntheticDefiningName _
      | `DiscreteSubtypeName _
      | `DottedName _
      | `EndName _
      | `ExplicitDeref _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `CharLiteral _
      | `Identifier _
      | `OpAbs _
      | `OpAnd _
      | `OpAndThen _
      | `OpConcat _
      | `OpDiv _
      | `OpDoubleDot _
      | `OpEq _
      | `OpGt _
      | `OpGte _
      | `OpIn _
      | `OpLt _
      | `OpLte _
      | `OpMinus _
      | `OpMod _
      | `OpMult _
      | `OpNeq _
      | `OpNot _
      | `OpNotIn _
      | `OpOr _
      | `OpOrElse _
      | `OpPlus _
      | `OpPow _
      | `OpRem _
      | `OpXor _
      | `StringLiteral _
      | `NullLiteral _
      | `IntLiteral _
      | `RealLiteral _
      | `SyntheticIdentifier _
      | `TargetName _
      | `UpdateAttributeRef _
      | `ParenExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `UnOp _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_abstract_state_decl_expr context c_value
   : abstract_state_decl_expr =
    let f_state_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.abstract_state_decl_expr_f_state_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node context (field_c_value)
      in
         

      match node with
            
      | `AbstractStateDecl _
      | `MultiAbstractStateDecl _
      | `ParenAbstractStateDecl _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AbstractStateDeclExpr {
        f_state_decl
          = Lazy.from_fun f_state_decl ;
        c_value = c_value;
        context = context
      }

      


  and wrap_allocator context c_value
   : allocator =
    let f_subpool () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.allocator_f_subpool
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_name context (field_c_value))
      in
         

      match node with
            
      | Some `AttributeRef _
      | Some `CallExpr _
      | Some `CharLiteral _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `QualExpr _
      | Some `ReduceAttributeRef _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_type_or_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.allocator_f_type_or_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node context (field_c_value)
      in
         

      match node with
            
      | `ConstrainedSubtypeIndication _
      | `DiscreteSubtypeIndication _
      | `QualExpr _
      | `SubtypeIndication _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `Allocator {
        f_subpool
          = Lazy.from_fun f_subpool ;
        f_type_or_expr
          = Lazy.from_fun f_type_or_expr ;
        c_value = c_value;
        context = context
      }

      

  and wrap_base_aggregate context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `Aggregate _
      | `BracketAggregate _
      | `DeltaAggregate _
      | `BracketDeltaAggregate _
      | `NullRecordAggregate _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_aggregate context c_value
   : aggregate =
    let f_ancestor_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_aggregate_f_ancestor_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_assocs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_aggregate_f_assocs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_assoc_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `Aggregate {
        f_ancestor_expr
          = Lazy.from_fun f_ancestor_expr ;
        f_assocs
          = Lazy.from_fun f_assocs ;
        c_value = c_value;
        context = context
      }

      


  and wrap_bracket_aggregate context c_value
   : bracket_aggregate =
    let f_ancestor_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_aggregate_f_ancestor_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_assocs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_aggregate_f_assocs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_assoc_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BracketAggregate {
        f_ancestor_expr
          = Lazy.from_fun f_ancestor_expr ;
        f_assocs
          = Lazy.from_fun f_assocs ;
        c_value = c_value;
        context = context
      }

      


  and wrap_delta_aggregate context c_value
   : delta_aggregate =
    let f_ancestor_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_aggregate_f_ancestor_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_assocs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_aggregate_f_assocs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_assoc_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DeltaAggregate {
        f_ancestor_expr
          = Lazy.from_fun f_ancestor_expr ;
        f_assocs
          = Lazy.from_fun f_assocs ;
        c_value = c_value;
        context = context
      }

      


  and wrap_bracket_delta_aggregate context c_value
   : bracket_delta_aggregate =
    let f_ancestor_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_aggregate_f_ancestor_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_assocs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_aggregate_f_assocs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_assoc_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BracketDeltaAggregate {
        f_ancestor_expr
          = Lazy.from_fun f_ancestor_expr ;
        f_assocs
          = Lazy.from_fun f_assocs ;
        c_value = c_value;
        context = context
      }

      


  and wrap_null_record_aggregate context c_value
   : null_record_aggregate =
    let f_ancestor_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_aggregate_f_ancestor_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_assocs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_aggregate_f_assocs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_assoc_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NullRecordAggregate {
        f_ancestor_expr
          = Lazy.from_fun f_ancestor_expr ;
        f_assocs
          = Lazy.from_fun f_assocs ;
        c_value = c_value;
        context = context
      }

      


  and wrap_bin_op context c_value
   : bin_op =
    let f_left () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.bin_op_f_left
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_op () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.bin_op_f_op
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_op context (field_c_value)
      in
         

      match node with
            
      | `OpAnd _
      | `OpAndThen _
      | `OpDiv _
      | `OpDoubleDot _
      | `OpEq _
      | `OpGt _
      | `OpGte _
      | `OpLt _
      | `OpLte _
      | `OpMinus _
      | `OpMod _
      | `OpMult _
      | `OpNeq _
      | `OpOr _
      | `OpOrElse _
      | `OpPlus _
      | `OpPow _
      | `OpRem _
      | `OpXor _
      as e -> e
      | _ -> assert false
    in
    let f_right () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.bin_op_f_right
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BinOp {
        f_left
          = Lazy.from_fun f_left ;
        f_op
          = Lazy.from_fun f_op ;
        f_right
          = Lazy.from_fun f_right ;
        c_value = c_value;
        context = context
      }

      


  and wrap_relation_op context c_value
   : relation_op =
    let f_left () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.bin_op_f_left
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_op () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.bin_op_f_op
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_op context (field_c_value)
      in
         

      match node with
            
      | `OpAnd _
      | `OpAndThen _
      | `OpDiv _
      | `OpDoubleDot _
      | `OpEq _
      | `OpGt _
      | `OpGte _
      | `OpLt _
      | `OpLte _
      | `OpMinus _
      | `OpMod _
      | `OpMult _
      | `OpNeq _
      | `OpOr _
      | `OpOrElse _
      | `OpPlus _
      | `OpPow _
      | `OpRem _
      | `OpXor _
      as e -> e
      | _ -> assert false
    in
    let f_right () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.bin_op_f_right
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RelationOp {
        f_left
          = Lazy.from_fun f_left ;
        f_op
          = Lazy.from_fun f_op ;
        f_right
          = Lazy.from_fun f_right ;
        c_value = c_value;
        context = context
      }

      


  and wrap_box_expr context c_value
   : box_expr =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BoxExpr {
        c_value = c_value;
        context = context
      }

      


  and wrap_case_expr_alternative context c_value
   : case_expr_alternative =
    let f_choices () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.case_expr_alternative_f_choices
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_alternatives_list context (field_c_value)
      in
         

      node
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.case_expr_alternative_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CaseExprAlternative {
        f_choices
          = Lazy.from_fun f_choices ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_concat_op context c_value
   : concat_op =
    let f_first_operand () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.concat_op_f_first_operand
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_other_operands () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.concat_op_f_other_operands
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_concat_operand_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ConcatOp {
        f_first_operand
          = Lazy.from_fun f_first_operand ;
        f_other_operands
          = Lazy.from_fun f_other_operands ;
        c_value = c_value;
        context = context
      }

      


  and wrap_concat_operand context c_value
   : concat_operand =
    let f_operator () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.concat_operand_f_operator
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_op_concat context (field_c_value)
      in
         

      node
    in
    let f_operand () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.concat_operand_f_operand
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ConcatOperand {
        f_operator
          = Lazy.from_fun f_operator ;
        f_operand
          = Lazy.from_fun f_operand ;
        c_value = c_value;
        context = context
      }

      

  and wrap_cond_expr context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `CaseExpr _
      | `IfExpr _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_case_expr context c_value
   : case_expr =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.case_expr_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_cases () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.case_expr_f_cases
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_case_expr_alternative_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CaseExpr {
        f_expr
          = Lazy.from_fun f_expr ;
        f_cases
          = Lazy.from_fun f_cases ;
        c_value = c_value;
        context = context
      }

      


  and wrap_if_expr context c_value
   : if_expr =
    let f_cond_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.if_expr_f_cond_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_then_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.if_expr_f_then_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_alternatives () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.if_expr_f_alternatives
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_elsif_expr_part_list context (field_c_value)
      in
         

      node
    in
    let f_else_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.if_expr_f_else_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `IfExpr {
        f_cond_expr
          = Lazy.from_fun f_cond_expr ;
        f_then_expr
          = Lazy.from_fun f_then_expr ;
        f_alternatives
          = Lazy.from_fun f_alternatives ;
        f_else_expr
          = Lazy.from_fun f_else_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_contract_cases context c_value
   : contract_cases =
    let f_contract_cases () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.contract_cases_f_contract_cases
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_contract_case_assoc_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ContractCases {
        f_contract_cases
          = Lazy.from_fun f_contract_cases ;
        c_value = c_value;
        context = context
      }

      


  and wrap_decl_expr context c_value
   : decl_expr =
    let f_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.decl_expr_f_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_basic_decl_list context (field_c_value)
      in
         

      node
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.decl_expr_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DeclExpr {
        f_decls
          = Lazy.from_fun f_decls ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_membership_expr context c_value
   : membership_expr =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.membership_expr_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_op () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.membership_expr_f_op
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_op context (field_c_value)
      in
         

      match node with
            
      | `OpIn _
      | `OpNotIn _
      as e -> e
      | _ -> assert false
    in
    let f_membership_exprs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.membership_expr_f_membership_exprs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr_alternatives_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `MembershipExpr {
        f_expr
          = Lazy.from_fun f_expr ;
        f_op
          = Lazy.from_fun f_op ;
        f_membership_exprs
          = Lazy.from_fun f_membership_exprs ;
        c_value = c_value;
        context = context
      }

      

  and wrap_name context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AttributeRef _
      | `CallExpr _
      | `DefiningName _
      | `SyntheticDefiningName _
      | `DiscreteSubtypeName _
      | `DottedName _
      | `EndName _
      | `ExplicitDeref _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `CharLiteral _
      | `Identifier _
      | `OpAbs _
      | `OpAnd _
      | `OpAndThen _
      | `OpConcat _
      | `OpDiv _
      | `OpDoubleDot _
      | `OpEq _
      | `OpGt _
      | `OpGte _
      | `OpIn _
      | `OpLt _
      | `OpLte _
      | `OpMinus _
      | `OpMod _
      | `OpMult _
      | `OpNeq _
      | `OpNot _
      | `OpNotIn _
      | `OpOr _
      | `OpOrElse _
      | `OpPlus _
      | `OpPow _
      | `OpRem _
      | `OpXor _
      | `StringLiteral _
      | `NullLiteral _
      | `IntLiteral _
      | `RealLiteral _
      | `SyntheticIdentifier _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_attribute_ref context c_value
   : attribute_ref =
    let f_prefix () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.attribute_ref_f_prefix
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_attribute () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.attribute_ref_f_attribute
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_identifier context (field_c_value)
      in
         

      node
    in
    let f_args () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.attribute_ref_f_args
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_assoc_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AttributeRef {
        f_prefix
          = Lazy.from_fun f_prefix ;
        f_attribute
          = Lazy.from_fun f_attribute ;
        f_args
          = Lazy.from_fun f_args ;
        c_value = c_value;
        context = context
      }

      


  and wrap_call_expr context c_value
   : call_expr =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.call_expr_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_suffix () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.call_expr_f_suffix
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node context (field_c_value)
      in
         

      match node with
            
      | `AssocList _
      | `AttributeRef _
      | `BinOp _
      | `CallExpr _
      | `CharLiteral _
      | `DiscreteSubtypeIndication _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CallExpr {
        f_name
          = Lazy.from_fun f_name ;
        f_suffix
          = Lazy.from_fun f_suffix ;
        c_value = c_value;
        context = context
      }

      


  and wrap_defining_name context c_value
   : defining_name =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.defining_name_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `CharLiteral _
      | `DottedName _
      | `Identifier _
      | `StringLiteral _
      | `SyntheticIdentifier _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DefiningName {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_synthetic_defining_name context c_value
   : synthetic_defining_name =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.defining_name_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `CharLiteral _
      | `DottedName _
      | `Identifier _
      | `StringLiteral _
      | `SyntheticIdentifier _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SyntheticDefiningName {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_discrete_subtype_name context c_value
   : discrete_subtype_name =
    let f_subtype () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.discrete_subtype_name_f_subtype
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_discrete_subtype_indication context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DiscreteSubtypeName {
        f_subtype
          = Lazy.from_fun f_subtype ;
        c_value = c_value;
        context = context
      }

      


  and wrap_dotted_name context c_value
   : dotted_name =
    let f_prefix () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.dotted_name_f_prefix
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_suffix () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.dotted_name_f_suffix
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_base_id context (field_c_value)
      in
         

      match node with
            
      | `CharLiteral _
      | `Identifier _
      | `StringLiteral _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DottedName {
        f_prefix
          = Lazy.from_fun f_prefix ;
        f_suffix
          = Lazy.from_fun f_suffix ;
        c_value = c_value;
        context = context
      }

      


  and wrap_end_name context c_value
   : end_name =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.end_name_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `CharLiteral _
      | `DottedName _
      | `Identifier _
      | `StringLiteral _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EndName {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_explicit_deref context c_value
   : explicit_deref =
    let f_prefix () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.explicit_deref_f_prefix
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ExplicitDeref {
        f_prefix
          = Lazy.from_fun f_prefix ;
        c_value = c_value;
        context = context
      }

      


  and wrap_qual_expr context c_value
   : qual_expr =
    let f_prefix () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.qual_expr_f_prefix
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_suffix () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.qual_expr_f_suffix
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `DeltaAggregate _
      | `NullRecordAggregate _
      | `ParenExpr _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `QualExpr {
        f_prefix
          = Lazy.from_fun f_prefix ;
        f_suffix
          = Lazy.from_fun f_suffix ;
        c_value = c_value;
        context = context
      }

      


  and wrap_reduce_attribute_ref context c_value
   : reduce_attribute_ref =
    let f_prefix () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.reduce_attribute_ref_f_prefix
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      | `ValueSequence _
      as e -> e
      | _ -> assert false
    in
    let f_attribute () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.reduce_attribute_ref_f_attribute
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_identifier context (field_c_value)
      in
         

      node
    in
    let f_args () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.reduce_attribute_ref_f_args
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_assoc_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ReduceAttributeRef {
        f_prefix
          = Lazy.from_fun f_prefix ;
        f_attribute
          = Lazy.from_fun f_attribute ;
        f_args
          = Lazy.from_fun f_args ;
        c_value = c_value;
        context = context
      }

      

  and wrap_single_tok_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `CharLiteral _
      | `Identifier _
      | `OpAbs _
      | `OpAnd _
      | `OpAndThen _
      | `OpConcat _
      | `OpDiv _
      | `OpDoubleDot _
      | `OpEq _
      | `OpGt _
      | `OpGte _
      | `OpIn _
      | `OpLt _
      | `OpLte _
      | `OpMinus _
      | `OpMod _
      | `OpMult _
      | `OpNeq _
      | `OpNot _
      | `OpNotIn _
      | `OpOr _
      | `OpOrElse _
      | `OpPlus _
      | `OpPow _
      | `OpRem _
      | `OpXor _
      | `StringLiteral _
      | `NullLiteral _
      | `IntLiteral _
      | `RealLiteral _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      

  and wrap_base_id context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `CharLiteral _
      | `Identifier _
      | `OpAbs _
      | `OpAnd _
      | `OpAndThen _
      | `OpConcat _
      | `OpDiv _
      | `OpDoubleDot _
      | `OpEq _
      | `OpGt _
      | `OpGte _
      | `OpIn _
      | `OpLt _
      | `OpLte _
      | `OpMinus _
      | `OpMod _
      | `OpMult _
      | `OpNeq _
      | `OpNot _
      | `OpNotIn _
      | `OpOr _
      | `OpOrElse _
      | `OpPlus _
      | `OpPow _
      | `OpRem _
      | `OpXor _
      | `StringLiteral _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_char_literal context c_value
   : char_literal =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CharLiteral {
        c_value = c_value;
        context = context
      }

      


  and wrap_identifier context c_value
   : identifier =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `Identifier {
        c_value = c_value;
        context = context
      }

      

  and wrap_op context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `OpAbs _
      | `OpAnd _
      | `OpAndThen _
      | `OpConcat _
      | `OpDiv _
      | `OpDoubleDot _
      | `OpEq _
      | `OpGt _
      | `OpGte _
      | `OpIn _
      | `OpLt _
      | `OpLte _
      | `OpMinus _
      | `OpMod _
      | `OpMult _
      | `OpNeq _
      | `OpNot _
      | `OpNotIn _
      | `OpOr _
      | `OpOrElse _
      | `OpPlus _
      | `OpPow _
      | `OpRem _
      | `OpXor _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_op_abs context c_value
   : op_abs =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpAbs {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_and context c_value
   : op_and =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpAnd {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_and_then context c_value
   : op_and_then =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpAndThen {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_concat context c_value
   : op_concat =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpConcat {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_div context c_value
   : op_div =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpDiv {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_double_dot context c_value
   : op_double_dot =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpDoubleDot {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_eq context c_value
   : op_eq =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpEq {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_gt context c_value
   : op_gt =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpGt {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_gte context c_value
   : op_gte =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpGte {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_in context c_value
   : op_in =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpIn {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_lt context c_value
   : op_lt =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpLt {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_lte context c_value
   : op_lte =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpLte {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_minus context c_value
   : op_minus =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpMinus {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_mod context c_value
   : op_mod =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpMod {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_mult context c_value
   : op_mult =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpMult {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_neq context c_value
   : op_neq =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpNeq {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_not context c_value
   : op_not =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpNot {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_not_in context c_value
   : op_not_in =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpNotIn {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_or context c_value
   : op_or =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpOr {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_or_else context c_value
   : op_or_else =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpOrElse {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_plus context c_value
   : op_plus =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpPlus {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_pow context c_value
   : op_pow =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpPow {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_rem context c_value
   : op_rem =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpRem {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_xor context c_value
   : op_xor =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpXor {
        c_value = c_value;
        context = context
      }

      


  and wrap_string_literal context c_value
   : string_literal =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `StringLiteral {
        c_value = c_value;
        context = context
      }

      


  and wrap_null_literal context c_value
   : null_literal =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NullLiteral {
        c_value = c_value;
        context = context
      }

      

  and wrap_num_literal context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `IntLiteral _
      | `RealLiteral _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_int_literal context c_value
   : int_literal =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `IntLiteral {
        c_value = c_value;
        context = context
      }

      


  and wrap_real_literal context c_value
   : real_literal =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RealLiteral {
        c_value = c_value;
        context = context
      }

      


  and wrap_synthetic_identifier context c_value
   : synthetic_identifier =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SyntheticIdentifier {
        c_value = c_value;
        context = context
      }

      


  and wrap_target_name context c_value
   : target_name =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TargetName {
        c_value = c_value;
        context = context
      }

      


  and wrap_update_attribute_ref context c_value
   : update_attribute_ref =
    let f_prefix () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.update_attribute_ref_f_prefix
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_attribute () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.update_attribute_ref_f_attribute
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_identifier context (field_c_value)
      in
         

      node
    in
    let f_values () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.update_attribute_ref_f_values
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_base_aggregate context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `UpdateAttributeRef {
        f_prefix
          = Lazy.from_fun f_prefix ;
        f_attribute
          = Lazy.from_fun f_attribute ;
        f_values
          = Lazy.from_fun f_values ;
        c_value = c_value;
        context = context
      }

      


  and wrap_paren_expr context c_value
   : paren_expr =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.paren_expr_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ParenExpr {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_quantified_expr context c_value
   : quantified_expr =
    let f_quantifier () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.quantified_expr_f_quantifier
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_quantifier context (field_c_value)
      in
         

      node
    in
    let f_loop_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.quantified_expr_f_loop_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_for_loop_spec context (field_c_value)
      in
         

      node
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.quantified_expr_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `QuantifiedExpr {
        f_quantifier
          = Lazy.from_fun f_quantifier ;
        f_loop_spec
          = Lazy.from_fun f_loop_spec ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_raise_expr context c_value
   : raise_expr =
    let f_exception_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.raise_expr_f_exception_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_name context (field_c_value))
      in
         

      match node with
            
      | Some `AttributeRef _
      | Some `CallExpr _
      | Some `CharLiteral _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `QualExpr _
      | Some `ReduceAttributeRef _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_error_message () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.raise_expr_f_error_message
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RaiseExpr {
        f_exception_name
          = Lazy.from_fun f_exception_name ;
        f_error_message
          = Lazy.from_fun f_error_message ;
        c_value = c_value;
        context = context
      }

      


  and wrap_un_op context c_value
   : un_op =
    let f_op () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.un_op_f_op
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_op context (field_c_value)
      in
         

      match node with
            
      | `OpAbs _
      | `OpMinus _
      | `OpNot _
      | `OpPlus _
      as e -> e
      | _ -> assert false
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.un_op_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `UnOp {
        f_op
          = Lazy.from_fun f_op ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_handled_stmts context c_value
   : handled_stmts =
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.handled_stmts_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_stmt_list context (field_c_value)
      in
         

      node
    in
    let f_exceptions () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.handled_stmts_f_exceptions
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `HandledStmts {
        f_stmts
          = Lazy.from_fun f_stmts ;
        f_exceptions
          = Lazy.from_fun f_exceptions ;
        c_value = c_value;
        context = context
      }

      

  and wrap_interface_kind context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `InterfaceKindLimited _
      | `InterfaceKindProtected _
      | `InterfaceKindSynchronized _
      | `InterfaceKindTask _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_interface_kind_limited context c_value
   : interface_kind_limited =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `InterfaceKindLimited {
        c_value = c_value;
        context = context
      }

      


  and wrap_interface_kind_protected context c_value
   : interface_kind_protected =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `InterfaceKindProtected {
        c_value = c_value;
        context = context
      }

      


  and wrap_interface_kind_synchronized context c_value
   : interface_kind_synchronized =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `InterfaceKindSynchronized {
        c_value = c_value;
        context = context
      }

      


  and wrap_interface_kind_task context c_value
   : interface_kind_task =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `InterfaceKindTask {
        c_value = c_value;
        context = context
      }

      

  and wrap_iter_type context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `IterTypeIn _
      | `IterTypeOf _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_iter_type_in context c_value
   : iter_type_in =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `IterTypeIn {
        c_value = c_value;
        context = context
      }

      


  and wrap_iter_type_of context c_value
   : iter_type_of =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `IterTypeOf {
        c_value = c_value;
        context = context
      }

      


  and wrap_library_item context c_value
   : library_item =
    let f_has_private () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.library_item_f_has_private
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_private_node context (field_c_value)
      in
         

      node
    in
    let f_item () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.library_item_f_item
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_basic_decl context (field_c_value)
      in
         

      match node with
            
      | `AbstractSubpDecl _
      | `ErrorDecl _
      | `ExprFunction _
      | `GenericPackageDecl _
      | `GenericPackageInstantiation _
      | `GenericPackageRenamingDecl _
      | `GenericSubpDecl _
      | `GenericSubpInstantiation _
      | `GenericSubpRenamingDecl _
      | `NullSubpDecl _
      | `PackageBody _
      | `PackageDecl _
      | `PackageRenamingDecl _
      | `SubpBody _
      | `SubpDecl _
      | `SubpRenamingDecl _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LibraryItem {
        f_has_private
          = Lazy.from_fun f_has_private ;
        f_item
          = Lazy.from_fun f_item ;
        c_value = c_value;
        context = context
      }

      

  and wrap_limited_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `LimitedAbsent _
      | `LimitedPresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_limited_absent context c_value
   : limited_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LimitedAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_limited_present context c_value
   : limited_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LimitedPresent {
        c_value = c_value;
        context = context
      }

      

  and wrap_loop_spec context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `ForLoopSpec _
      | `WhileLoopSpec _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_for_loop_spec context c_value
   : for_loop_spec =
    let f_var_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.for_loop_spec_f_var_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_for_loop_var_decl context (field_c_value)
      in
         

      node
    in
    let f_loop_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.for_loop_spec_f_loop_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_iter_type context (field_c_value)
      in
         

      node
    in
    let f_has_reverse () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.for_loop_spec_f_has_reverse
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_reverse_node context (field_c_value)
      in
         

      node
    in
    let f_iter_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.for_loop_spec_f_iter_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `BinOp _
      | `CallExpr _
      | `CharLiteral _
      | `DiscreteSubtypeIndication _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_iter_filter () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.for_loop_spec_f_iter_filter
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ForLoopSpec {
        f_var_decl
          = Lazy.from_fun f_var_decl ;
        f_loop_type
          = Lazy.from_fun f_loop_type ;
        f_has_reverse
          = Lazy.from_fun f_has_reverse ;
        f_iter_expr
          = Lazy.from_fun f_iter_expr ;
        f_iter_filter
          = Lazy.from_fun f_iter_filter ;
        c_value = c_value;
        context = context
      }

      


  and wrap_while_loop_spec context c_value
   : while_loop_spec =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.while_loop_spec_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `WhileLoopSpec {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      

  and wrap_mode context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `ModeDefault _
      | `ModeIn _
      | `ModeInOut _
      | `ModeOut _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_mode_default context c_value
   : mode_default =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ModeDefault {
        c_value = c_value;
        context = context
      }

      


  and wrap_mode_in context c_value
   : mode_in =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ModeIn {
        c_value = c_value;
        context = context
      }

      


  and wrap_mode_in_out context c_value
   : mode_in_out =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ModeInOut {
        c_value = c_value;
        context = context
      }

      


  and wrap_mode_out context c_value
   : mode_out =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ModeOut {
        c_value = c_value;
        context = context
      }

      


  and wrap_multi_abstract_state_decl context c_value
   : multi_abstract_state_decl =
    let f_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.multi_abstract_state_decl_f_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_abstract_state_decl_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `MultiAbstractStateDecl {
        f_decls
          = Lazy.from_fun f_decls ;
        c_value = c_value;
        context = context
      }

      

  and wrap_not_null context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `NotNullAbsent _
      | `NotNullPresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_not_null_absent context c_value
   : not_null_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NotNullAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_not_null_present context c_value
   : not_null_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NotNullPresent {
        c_value = c_value;
        context = context
      }

      


  and wrap_null_component_decl context c_value
   : null_component_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NullComponentDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_others_designator context c_value
   : others_designator =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OthersDesignator {
        c_value = c_value;
        context = context
      }

      

  and wrap_overriding_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `OverridingNotOverriding _
      | `OverridingOverriding _
      | `OverridingUnspecified _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_overriding_not_overriding context c_value
   : overriding_not_overriding =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OverridingNotOverriding {
        c_value = c_value;
        context = context
      }

      


  and wrap_overriding_overriding context c_value
   : overriding_overriding =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OverridingOverriding {
        c_value = c_value;
        context = context
      }

      


  and wrap_overriding_unspecified context c_value
   : overriding_unspecified =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OverridingUnspecified {
        c_value = c_value;
        context = context
      }

      


  and wrap_params context c_value
   : params =
    let f_params () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.params_f_params
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_param_spec_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `Params {
        f_params
          = Lazy.from_fun f_params ;
        c_value = c_value;
        context = context
      }

      


  and wrap_paren_abstract_state_decl context c_value
   : paren_abstract_state_decl =
    let f_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.paren_abstract_state_decl_f_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ada_node context (field_c_value)
      in
         

      match node with
            
      | `AbstractStateDecl _
      | `ParenAbstractStateDecl _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ParenAbstractStateDecl {
        f_decl
          = Lazy.from_fun f_decl ;
        c_value = c_value;
        context = context
      }

      

  and wrap_pp_directive context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `PpElseDirective _
      | `PpElsifDirective _
      | `PpEndIfDirective _
      | `PpIfDirective _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_pp_else_directive context c_value
   : pp_else_directive =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PpElseDirective {
        c_value = c_value;
        context = context
      }

      


  and wrap_pp_elsif_directive context c_value
   : pp_elsif_directive =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.pp_elsif_directive_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `BinOp _
      | `Identifier _
      | `ParenExpr _
      | `RelationOp _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    let f_then_kw () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.pp_elsif_directive_f_then_kw
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_pp_then_kw context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PpElsifDirective {
        f_expr
          = Lazy.from_fun f_expr ;
        f_then_kw
          = Lazy.from_fun f_then_kw ;
        c_value = c_value;
        context = context
      }

      


  and wrap_pp_end_if_directive context c_value
   : pp_end_if_directive =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PpEndIfDirective {
        c_value = c_value;
        context = context
      }

      


  and wrap_pp_if_directive context c_value
   : pp_if_directive =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.pp_if_directive_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `BinOp _
      | `Identifier _
      | `ParenExpr _
      | `RelationOp _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    let f_then_kw () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.pp_if_directive_f_then_kw
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_pp_then_kw context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PpIfDirective {
        f_expr
          = Lazy.from_fun f_expr ;
        f_then_kw
          = Lazy.from_fun f_then_kw ;
        c_value = c_value;
        context = context
      }

      


  and wrap_pp_then_kw context c_value
   : pp_then_kw =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PpThenKw {
        c_value = c_value;
        context = context
      }

      


  and wrap_pragma_node context c_value
   : pragma_node =
    let f_id () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.pragma_node_f_id
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_identifier context (field_c_value)
      in
         

      node
    in
    let f_args () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.pragma_node_f_args
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_base_assoc_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PragmaNode {
        f_id
          = Lazy.from_fun f_id ;
        f_args
          = Lazy.from_fun f_args ;
        c_value = c_value;
        context = context
      }

      

  and wrap_private_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `PrivateAbsent _
      | `PrivatePresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_private_absent context c_value
   : private_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PrivateAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_private_present context c_value
   : private_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PrivatePresent {
        c_value = c_value;
        context = context
      }

      


  and wrap_protected_def context c_value
   : protected_def =
    let f_public_part () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.protected_def_f_public_part
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_public_part context (field_c_value)
      in
         

      node
    in
    let f_private_part () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.protected_def_f_private_part
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_private_part context (field_c_value))
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.protected_def_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ProtectedDef {
        f_public_part
          = Lazy.from_fun f_public_part ;
        f_private_part
          = Lazy.from_fun f_private_part ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      

  and wrap_protected_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `ProtectedAbsent _
      | `ProtectedPresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_protected_absent context c_value
   : protected_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ProtectedAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_protected_present context c_value
   : protected_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ProtectedPresent {
        c_value = c_value;
        context = context
      }

      

  and wrap_quantifier context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `QuantifierAll _
      | `QuantifierSome _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_quantifier_all context c_value
   : quantifier_all =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `QuantifierAll {
        c_value = c_value;
        context = context
      }

      


  and wrap_quantifier_some context c_value
   : quantifier_some =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `QuantifierSome {
        c_value = c_value;
        context = context
      }

      


  and wrap_range_spec context c_value
   : range_spec =
    let f_range () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.range_spec_f_range
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `BinOp _
      | `BoxExpr _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RangeSpec {
        f_range
          = Lazy.from_fun f_range ;
        c_value = c_value;
        context = context
      }

      


  and wrap_renaming_clause context c_value
   : renaming_clause =
    let f_renamed_object () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.renaming_clause_f_renamed_object
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RenamingClause {
        f_renamed_object
          = Lazy.from_fun f_renamed_object ;
        c_value = c_value;
        context = context
      }

      


  and wrap_synthetic_renaming_clause context c_value
   : synthetic_renaming_clause =
    let f_renamed_object () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.renaming_clause_f_renamed_object
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SyntheticRenamingClause {
        f_renamed_object
          = Lazy.from_fun f_renamed_object ;
        c_value = c_value;
        context = context
      }

      

  and wrap_reverse_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `ReverseAbsent _
      | `ReversePresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_reverse_absent context c_value
   : reverse_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ReverseAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_reverse_present context c_value
   : reverse_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ReversePresent {
        c_value = c_value;
        context = context
      }

      


  and wrap_select_when_part context c_value
   : select_when_part =
    let f_cond_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.select_when_part_f_cond_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.select_when_part_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_stmt_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SelectWhenPart {
        f_cond_expr
          = Lazy.from_fun f_cond_expr ;
        f_stmts
          = Lazy.from_fun f_stmts ;
        c_value = c_value;
        context = context
      }

      

  and wrap_stmt context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AcceptStmt _
      | `AcceptStmtWithStmts _
      | `ForLoopStmt _
      | `LoopStmt _
      | `WhileLoopStmt _
      | `BeginBlock _
      | `DeclBlock _
      | `CaseStmt _
      | `ExtendedReturnStmt _
      | `IfStmt _
      | `NamedStmt _
      | `SelectStmt _
      | `ErrorStmt _
      | `AbortStmt _
      | `AssignStmt _
      | `CallStmt _
      | `DelayStmt _
      | `ExitStmt _
      | `GotoStmt _
      | `Label _
      | `NullStmt _
      | `RaiseStmt _
      | `RequeueStmt _
      | `ReturnStmt _
      | `TerminateAlternative _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      

  and wrap_composite_stmt context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AcceptStmt _
      | `AcceptStmtWithStmts _
      | `ForLoopStmt _
      | `LoopStmt _
      | `WhileLoopStmt _
      | `BeginBlock _
      | `DeclBlock _
      | `CaseStmt _
      | `ExtendedReturnStmt _
      | `IfStmt _
      | `NamedStmt _
      | `SelectStmt _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_accept_stmt context c_value
   : accept_stmt =
    let f_body_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.accept_stmt_f_body_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_accept_stmt_body context (field_c_value)
      in
         

      node
    in
    let f_entry_index_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.accept_stmt_f_entry_index_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_params () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.accept_stmt_f_params
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_entry_completion_formal_params context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AcceptStmt {
        f_body_decl
          = Lazy.from_fun f_body_decl ;
        f_entry_index_expr
          = Lazy.from_fun f_entry_index_expr ;
        f_params
          = Lazy.from_fun f_params ;
        c_value = c_value;
        context = context
      }

      


  and wrap_accept_stmt_with_stmts context c_value
   : accept_stmt_with_stmts =
    let f_body_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.accept_stmt_f_body_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_accept_stmt_body context (field_c_value)
      in
         

      node
    in
    let f_entry_index_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.accept_stmt_f_entry_index_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_params () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.accept_stmt_f_params
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_entry_completion_formal_params context (field_c_value)
      in
         

      node
    in
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.accept_stmt_with_stmts_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_handled_stmts context (field_c_value)
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.accept_stmt_with_stmts_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AcceptStmtWithStmts {
        f_body_decl
          = Lazy.from_fun f_body_decl ;
        f_entry_index_expr
          = Lazy.from_fun f_entry_index_expr ;
        f_params
          = Lazy.from_fun f_params ;
        f_stmts
          = Lazy.from_fun f_stmts ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      

  and wrap_base_loop_stmt context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `ForLoopStmt _
      | `LoopStmt _
      | `WhileLoopStmt _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_for_loop_stmt context c_value
   : for_loop_stmt =
    let f_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_loop_stmt_f_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_loop_spec context (field_c_value))
      in
         

      node
    in
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_loop_stmt_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_stmt_list context (field_c_value)
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_loop_stmt_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ForLoopStmt {
        f_spec
          = Lazy.from_fun f_spec ;
        f_stmts
          = Lazy.from_fun f_stmts ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_loop_stmt context c_value
   : loop_stmt =
    let f_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_loop_stmt_f_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_loop_spec context (field_c_value))
      in
         

      node
    in
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_loop_stmt_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_stmt_list context (field_c_value)
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_loop_stmt_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LoopStmt {
        f_spec
          = Lazy.from_fun f_spec ;
        f_stmts
          = Lazy.from_fun f_stmts ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_while_loop_stmt context c_value
   : while_loop_stmt =
    let f_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_loop_stmt_f_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_loop_spec context (field_c_value))
      in
         

      node
    in
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_loop_stmt_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_stmt_list context (field_c_value)
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_loop_stmt_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `WhileLoopStmt {
        f_spec
          = Lazy.from_fun f_spec ;
        f_stmts
          = Lazy.from_fun f_stmts ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      

  and wrap_block_stmt context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `BeginBlock _
      | `DeclBlock _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_begin_block context c_value
   : begin_block =
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.begin_block_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_handled_stmts context (field_c_value)
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.begin_block_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BeginBlock {
        f_stmts
          = Lazy.from_fun f_stmts ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_decl_block context c_value
   : decl_block =
    let f_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.decl_block_f_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_declarative_part context (field_c_value)
      in
         

      node
    in
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.decl_block_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_handled_stmts context (field_c_value)
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.decl_block_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DeclBlock {
        f_decls
          = Lazy.from_fun f_decls ;
        f_stmts
          = Lazy.from_fun f_stmts ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_case_stmt context c_value
   : case_stmt =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.case_stmt_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_pragmas () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.case_stmt_f_pragmas
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pragma_node_list context (field_c_value)
      in
         

      node
    in
    let f_alternatives () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.case_stmt_f_alternatives
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_case_stmt_alternative_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CaseStmt {
        f_expr
          = Lazy.from_fun f_expr ;
        f_pragmas
          = Lazy.from_fun f_pragmas ;
        f_alternatives
          = Lazy.from_fun f_alternatives ;
        c_value = c_value;
        context = context
      }

      


  and wrap_extended_return_stmt context c_value
   : extended_return_stmt =
    let f_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.extended_return_stmt_f_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_extended_return_stmt_object_decl context (field_c_value)
      in
         

      node
    in
    let f_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.extended_return_stmt_f_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_handled_stmts context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ExtendedReturnStmt {
        f_decl
          = Lazy.from_fun f_decl ;
        f_stmts
          = Lazy.from_fun f_stmts ;
        c_value = c_value;
        context = context
      }

      


  and wrap_if_stmt context c_value
   : if_stmt =
    let f_cond_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.if_stmt_f_cond_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_then_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.if_stmt_f_then_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_stmt_list context (field_c_value)
      in
         

      node
    in
    let f_alternatives () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.if_stmt_f_alternatives
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_elsif_stmt_part_list context (field_c_value)
      in
         

      node
    in
    let f_else_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.if_stmt_f_else_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_stmt_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `IfStmt {
        f_cond_expr
          = Lazy.from_fun f_cond_expr ;
        f_then_stmts
          = Lazy.from_fun f_then_stmts ;
        f_alternatives
          = Lazy.from_fun f_alternatives ;
        f_else_stmts
          = Lazy.from_fun f_else_stmts ;
        c_value = c_value;
        context = context
      }

      


  and wrap_named_stmt context c_value
   : named_stmt =
    let f_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.named_stmt_f_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_named_stmt_decl context (field_c_value)
      in
         

      node
    in
    let f_stmt () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.named_stmt_f_stmt
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_composite_stmt context (field_c_value)
      in
         

      match node with
            
      | `BeginBlock _
      | `DeclBlock _
      | `ForLoopStmt _
      | `LoopStmt _
      | `WhileLoopStmt _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NamedStmt {
        f_decl
          = Lazy.from_fun f_decl ;
        f_stmt
          = Lazy.from_fun f_stmt ;
        c_value = c_value;
        context = context
      }

      


  and wrap_select_stmt context c_value
   : select_stmt =
    let f_guards () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.select_stmt_f_guards
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_select_when_part_list context (field_c_value)
      in
         

      node
    in
    let f_else_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.select_stmt_f_else_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_stmt_list context (field_c_value)
      in
         

      node
    in
    let f_abort_stmts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.select_stmt_f_abort_stmts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_stmt_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SelectStmt {
        f_guards
          = Lazy.from_fun f_guards ;
        f_else_stmts
          = Lazy.from_fun f_else_stmts ;
        f_abort_stmts
          = Lazy.from_fun f_abort_stmts ;
        c_value = c_value;
        context = context
      }

      


  and wrap_error_stmt context c_value
   : error_stmt =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ErrorStmt {
        c_value = c_value;
        context = context
      }

      

  and wrap_simple_stmt context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AbortStmt _
      | `AssignStmt _
      | `CallStmt _
      | `DelayStmt _
      | `ExitStmt _
      | `GotoStmt _
      | `Label _
      | `NullStmt _
      | `RaiseStmt _
      | `RequeueStmt _
      | `ReturnStmt _
      | `TerminateAlternative _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_abort_stmt context c_value
   : abort_stmt =
    let f_names () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.abort_stmt_f_names
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AbortStmt {
        f_names
          = Lazy.from_fun f_names ;
        c_value = c_value;
        context = context
      }

      


  and wrap_assign_stmt context c_value
   : assign_stmt =
    let f_dest () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.assign_stmt_f_dest
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.assign_stmt_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AssignStmt {
        f_dest
          = Lazy.from_fun f_dest ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_call_stmt context c_value
   : call_stmt =
    let f_call () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.call_stmt_f_call
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CallStmt {
        f_call
          = Lazy.from_fun f_call ;
        c_value = c_value;
        context = context
      }

      


  and wrap_delay_stmt context c_value
   : delay_stmt =
    let f_has_until () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.delay_stmt_f_has_until
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_until_node context (field_c_value)
      in
         

      node
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.delay_stmt_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `MembershipExpr _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DelayStmt {
        f_has_until
          = Lazy.from_fun f_has_until ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_exit_stmt context c_value
   : exit_stmt =
    let f_loop_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.exit_stmt_f_loop_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_name context (field_c_value))
      in
         

      match node with
            
      | Some `CharLiteral _
      | Some `DottedName _
      | Some `Identifier _
      | Some `StringLiteral _
      | None as e -> e
      | _ -> assert false
    in
    let f_cond_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.exit_stmt_f_cond_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ExitStmt {
        f_loop_name
          = Lazy.from_fun f_loop_name ;
        f_cond_expr
          = Lazy.from_fun f_cond_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_goto_stmt context c_value
   : goto_stmt =
    let f_label_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.goto_stmt_f_label_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `CharLiteral _
      | `DottedName _
      | `Identifier _
      | `StringLiteral _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GotoStmt {
        f_label_name
          = Lazy.from_fun f_label_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_label context c_value
   : label =
    let f_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.label_f_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_label_decl context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `Label {
        f_decl
          = Lazy.from_fun f_decl ;
        c_value = c_value;
        context = context
      }

      


  and wrap_null_stmt context c_value
   : null_stmt =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NullStmt {
        c_value = c_value;
        context = context
      }

      


  and wrap_raise_stmt context c_value
   : raise_stmt =
    let f_exception_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.raise_stmt_f_exception_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_name context (field_c_value))
      in
         

      match node with
            
      | Some `AttributeRef _
      | Some `CallExpr _
      | Some `CharLiteral _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `QualExpr _
      | Some `ReduceAttributeRef _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_error_message () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.raise_stmt_f_error_message
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RaiseStmt {
        f_exception_name
          = Lazy.from_fun f_exception_name ;
        f_error_message
          = Lazy.from_fun f_error_message ;
        c_value = c_value;
        context = context
      }

      


  and wrap_requeue_stmt context c_value
   : requeue_stmt =
    let f_call_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.requeue_stmt_f_call_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CallExpr _
      | `CharLiteral _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `QualExpr _
      | `ReduceAttributeRef _
      | `StringLiteral _
      | `TargetName _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_has_abort () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.requeue_stmt_f_has_abort
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_abort_node context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RequeueStmt {
        f_call_name
          = Lazy.from_fun f_call_name ;
        f_has_abort
          = Lazy.from_fun f_has_abort ;
        c_value = c_value;
        context = context
      }

      


  and wrap_return_stmt context c_value
   : return_stmt =
    let f_return_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.return_stmt_f_return_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `Aggregate _
      | Some `Allocator _
      | Some `AttributeRef _
      | Some `BinOp _
      | Some `BracketAggregate _
      | Some `BracketDeltaAggregate _
      | Some `CallExpr _
      | Some `CaseExpr _
      | Some `CharLiteral _
      | Some `ConcatOp _
      | Some `DeclExpr _
      | Some `DeltaAggregate _
      | Some `DottedName _
      | Some `ExplicitDeref _
      | Some `Identifier _
      | Some `IfExpr _
      | Some `IntLiteral _
      | Some `MembershipExpr _
      | Some `NullLiteral _
      | Some `NullRecordAggregate _
      | Some `ParenExpr _
      | Some `QualExpr _
      | Some `QuantifiedExpr _
      | Some `RaiseExpr _
      | Some `RealLiteral _
      | Some `ReduceAttributeRef _
      | Some `RelationOp _
      | Some `StringLiteral _
      | Some `TargetName _
      | Some `UnOp _
      | Some `UpdateAttributeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ReturnStmt {
        f_return_expr
          = Lazy.from_fun f_return_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_terminate_alternative context c_value
   : terminate_alternative =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TerminateAlternative {
        c_value = c_value;
        context = context
      }

      

  and wrap_subp_kind context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `SubpKindFunction _
      | `SubpKindProcedure _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_subp_kind_function context c_value
   : subp_kind_function =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SubpKindFunction {
        c_value = c_value;
        context = context
      }

      


  and wrap_subp_kind_procedure context c_value
   : subp_kind_procedure =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SubpKindProcedure {
        c_value = c_value;
        context = context
      }

      


  and wrap_subunit context c_value
   : subunit =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subunit_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `CharLiteral _
      | `DottedName _
      | `Identifier _
      | `StringLiteral _
      as e -> e
      | _ -> assert false
    in
    let f_body () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subunit_f_body
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_body_node context (field_c_value)
      in
         

      match node with
            
      | `PackageBody _
      | `ProtectedBody _
      | `SubpBody _
      | `TaskBody _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `Subunit {
        f_name
          = Lazy.from_fun f_name ;
        f_body
          = Lazy.from_fun f_body ;
        c_value = c_value;
        context = context
      }

      

  and wrap_synchronized_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `SynchronizedAbsent _
      | `SynchronizedPresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_synchronized_absent context c_value
   : synchronized_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SynchronizedAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_synchronized_present context c_value
   : synchronized_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SynchronizedPresent {
        c_value = c_value;
        context = context
      }

      

  and wrap_tagged_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `TaggedAbsent _
      | `TaggedPresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_tagged_absent context c_value
   : tagged_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TaggedAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_tagged_present context c_value
   : tagged_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TaggedPresent {
        c_value = c_value;
        context = context
      }

      


  and wrap_task_def context c_value
   : task_def =
    let f_interfaces () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.task_def_f_interfaces
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_parent_list context (field_c_value)
      in
         

      node
    in
    let f_public_part () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.task_def_f_public_part
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_public_part context (field_c_value)
      in
         

      node
    in
    let f_private_part () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.task_def_f_private_part
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_private_part context (field_c_value))
      in
         

      node
    in
    let f_end_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.task_def_f_end_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_end_name context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TaskDef {
        f_interfaces
          = Lazy.from_fun f_interfaces ;
        f_public_part
          = Lazy.from_fun f_public_part ;
        f_private_part
          = Lazy.from_fun f_private_part ;
        f_end_name
          = Lazy.from_fun f_end_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_type_attributes_repository context c_value
   : type_attributes_repository =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TypeAttributesRepository {
        c_value = c_value;
        context = context
      }

      

  and wrap_type_def context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AccessToSubpDef _
      | `AnonymousTypeAccessDef _
      | `TypeAccessDef _
      | `ArrayTypeDef _
      | `DerivedTypeDef _
      | `EnumTypeDef _
      | `FormalDiscreteTypeDef _
      | `InterfaceTypeDef _
      | `ModIntTypeDef _
      | `PrivateTypeDef _
      | `DecimalFixedPointDef _
      | `FloatingPointDef _
      | `OrdinaryFixedPointDef _
      | `RecordTypeDef _
      | `SignedIntTypeDef _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      

  and wrap_access_def context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AccessToSubpDef _
      | `AnonymousTypeAccessDef _
      | `TypeAccessDef _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_access_to_subp_def context c_value
   : access_to_subp_def =
    let f_has_not_null () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.access_def_f_has_not_null
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_not_null context (field_c_value))
      in
         

      node
    in
    let f_has_protected () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.access_to_subp_def_f_has_protected
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_protected_node context (field_c_value)
      in
         

      node
    in
    let f_subp_spec () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.access_to_subp_def_f_subp_spec
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subp_spec context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AccessToSubpDef {
        f_has_not_null
          = Lazy.from_fun f_has_not_null ;
        f_has_protected
          = Lazy.from_fun f_has_protected ;
        f_subp_spec
          = Lazy.from_fun f_subp_spec ;
        c_value = c_value;
        context = context
      }

      

  and wrap_base_type_access_def context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AnonymousTypeAccessDef _
      | `TypeAccessDef _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_anonymous_type_access_def context c_value
   : anonymous_type_access_def =
    let f_has_not_null () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.access_def_f_has_not_null
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_not_null context (field_c_value))
      in
         

      node
    in
    let f_type_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.anonymous_type_access_def_f_type_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_base_type_decl context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AnonymousTypeAccessDef {
        f_has_not_null
          = Lazy.from_fun f_has_not_null ;
        f_type_decl
          = Lazy.from_fun f_type_decl ;
        c_value = c_value;
        context = context
      }

      


  and wrap_type_access_def context c_value
   : type_access_def =
    let f_has_not_null () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.access_def_f_has_not_null
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_not_null context (field_c_value))
      in
         

      node
    in
    let f_has_all () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.type_access_def_f_has_all
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_all_node context (field_c_value)
      in
         

      node
    in
    let f_has_constant () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.type_access_def_f_has_constant
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_constant_node context (field_c_value)
      in
         

      node
    in
    let f_subtype_indication () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.type_access_def_f_subtype_indication
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subtype_indication context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TypeAccessDef {
        f_has_not_null
          = Lazy.from_fun f_has_not_null ;
        f_has_all
          = Lazy.from_fun f_has_all ;
        f_has_constant
          = Lazy.from_fun f_has_constant ;
        f_subtype_indication
          = Lazy.from_fun f_subtype_indication ;
        c_value = c_value;
        context = context
      }

      


  and wrap_array_type_def context c_value
   : array_type_def =
    let f_indices () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.array_type_def_f_indices
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_array_indices context (field_c_value)
      in
         

      node
    in
    let f_component_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.array_type_def_f_component_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_component_def context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ArrayTypeDef {
        f_indices
          = Lazy.from_fun f_indices ;
        f_component_type
          = Lazy.from_fun f_component_type ;
        c_value = c_value;
        context = context
      }

      


  and wrap_derived_type_def context c_value
   : derived_type_def =
    let f_has_abstract () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.derived_type_def_f_has_abstract
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_abstract_node context (field_c_value)
      in
         

      node
    in
    let f_has_limited () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.derived_type_def_f_has_limited
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_limited_node context (field_c_value)
      in
         

      node
    in
    let f_has_synchronized () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.derived_type_def_f_has_synchronized
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_synchronized_node context (field_c_value)
      in
         

      node
    in
    let f_subtype_indication () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.derived_type_def_f_subtype_indication
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subtype_indication context (field_c_value)
      in
         

      node
    in
    let f_interfaces () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.derived_type_def_f_interfaces
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_parent_list context (field_c_value)
      in
         

      node
    in
    let f_record_extension () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.derived_type_def_f_record_extension
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_base_record_def context (field_c_value))
      in
         

      node
    in
    let f_has_with_private () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.derived_type_def_f_has_with_private
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_with_private context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DerivedTypeDef {
        f_has_abstract
          = Lazy.from_fun f_has_abstract ;
        f_has_limited
          = Lazy.from_fun f_has_limited ;
        f_has_synchronized
          = Lazy.from_fun f_has_synchronized ;
        f_subtype_indication
          = Lazy.from_fun f_subtype_indication ;
        f_interfaces
          = Lazy.from_fun f_interfaces ;
        f_record_extension
          = Lazy.from_fun f_record_extension ;
        f_has_with_private
          = Lazy.from_fun f_has_with_private ;
        c_value = c_value;
        context = context
      }

      


  and wrap_enum_type_def context c_value
   : enum_type_def =
    let f_enum_literals () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.enum_type_def_f_enum_literals
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_enum_literal_decl_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnumTypeDef {
        f_enum_literals
          = Lazy.from_fun f_enum_literals ;
        c_value = c_value;
        context = context
      }

      


  and wrap_formal_discrete_type_def context c_value
   : formal_discrete_type_def =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `FormalDiscreteTypeDef {
        c_value = c_value;
        context = context
      }

      


  and wrap_interface_type_def context c_value
   : interface_type_def =
    let f_interface_kind () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.interface_type_def_f_interface_kind
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_interface_kind context (field_c_value))
      in
         

      node
    in
    let f_interfaces () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.interface_type_def_f_interfaces
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_parent_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `InterfaceTypeDef {
        f_interface_kind
          = Lazy.from_fun f_interface_kind ;
        f_interfaces
          = Lazy.from_fun f_interfaces ;
        c_value = c_value;
        context = context
      }

      


  and wrap_mod_int_type_def context c_value
   : mod_int_type_def =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.mod_int_type_def_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BoxExpr _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ModIntTypeDef {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_private_type_def context c_value
   : private_type_def =
    let f_has_abstract () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.private_type_def_f_has_abstract
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_abstract_node context (field_c_value)
      in
         

      node
    in
    let f_has_tagged () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.private_type_def_f_has_tagged
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_tagged_node context (field_c_value)
      in
         

      node
    in
    let f_has_limited () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.private_type_def_f_has_limited
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_limited_node context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PrivateTypeDef {
        f_has_abstract
          = Lazy.from_fun f_has_abstract ;
        f_has_tagged
          = Lazy.from_fun f_has_tagged ;
        f_has_limited
          = Lazy.from_fun f_has_limited ;
        c_value = c_value;
        context = context
      }

      

  and wrap_real_type_def context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `DecimalFixedPointDef _
      | `FloatingPointDef _
      | `OrdinaryFixedPointDef _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_decimal_fixed_point_def context c_value
   : decimal_fixed_point_def =
    let f_delta () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.decimal_fixed_point_def_f_delta
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BoxExpr _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_digits () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.decimal_fixed_point_def_f_digits
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BoxExpr _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_range () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.decimal_fixed_point_def_f_range
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_range_spec context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DecimalFixedPointDef {
        f_delta
          = Lazy.from_fun f_delta ;
        f_digits
          = Lazy.from_fun f_digits ;
        f_range
          = Lazy.from_fun f_range ;
        c_value = c_value;
        context = context
      }

      


  and wrap_floating_point_def context c_value
   : floating_point_def =
    let f_num_digits () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.floating_point_def_f_num_digits
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BoxExpr _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_range () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.floating_point_def_f_range
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_range_spec context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `FloatingPointDef {
        f_num_digits
          = Lazy.from_fun f_num_digits ;
        f_range
          = Lazy.from_fun f_range ;
        c_value = c_value;
        context = context
      }

      


  and wrap_ordinary_fixed_point_def context c_value
   : ordinary_fixed_point_def =
    let f_delta () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.ordinary_fixed_point_def_f_delta
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `Aggregate _
      | `Allocator _
      | `AttributeRef _
      | `BinOp _
      | `BoxExpr _
      | `BracketAggregate _
      | `BracketDeltaAggregate _
      | `CallExpr _
      | `CaseExpr _
      | `CharLiteral _
      | `ConcatOp _
      | `DeclExpr _
      | `DeltaAggregate _
      | `DottedName _
      | `ExplicitDeref _
      | `Identifier _
      | `IfExpr _
      | `IntLiteral _
      | `NullLiteral _
      | `NullRecordAggregate _
      | `ParenExpr _
      | `QualExpr _
      | `QuantifiedExpr _
      | `RaiseExpr _
      | `RealLiteral _
      | `ReduceAttributeRef _
      | `RelationOp _
      | `StringLiteral _
      | `TargetName _
      | `UnOp _
      | `UpdateAttributeRef _
      as e -> e
      | _ -> assert false
    in
    let f_range () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.ordinary_fixed_point_def_f_range
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_range_spec context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OrdinaryFixedPointDef {
        f_delta
          = Lazy.from_fun f_delta ;
        f_range
          = Lazy.from_fun f_range ;
        c_value = c_value;
        context = context
      }

      


  and wrap_record_type_def context c_value
   : record_type_def =
    let f_has_abstract () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.record_type_def_f_has_abstract
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_abstract_node context (field_c_value)
      in
         

      node
    in
    let f_has_tagged () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.record_type_def_f_has_tagged
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_tagged_node context (field_c_value)
      in
         

      node
    in
    let f_has_limited () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.record_type_def_f_has_limited
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_limited_node context (field_c_value)
      in
         

      node
    in
    let f_record_def () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.record_type_def_f_record_def
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_base_record_def context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RecordTypeDef {
        f_has_abstract
          = Lazy.from_fun f_has_abstract ;
        f_has_tagged
          = Lazy.from_fun f_has_tagged ;
        f_has_limited
          = Lazy.from_fun f_has_limited ;
        f_record_def
          = Lazy.from_fun f_record_def ;
        c_value = c_value;
        context = context
      }

      


  and wrap_signed_int_type_def context c_value
   : signed_int_type_def =
    let f_range () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.signed_int_type_def_f_range
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_range_spec context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SignedIntTypeDef {
        f_range
          = Lazy.from_fun f_range ;
        c_value = c_value;
        context = context
      }

      

  and wrap_type_expr context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `AnonymousType _
      | `EnumLitSynthTypeExpr _
      | `SubtypeIndication _
      | `ConstrainedSubtypeIndication _
      | `DiscreteSubtypeIndication _
      | `SyntheticTypeExpr _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_anonymous_type context c_value
   : anonymous_type =
    let f_type_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.anonymous_type_f_type_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_anonymous_type_decl context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AnonymousType {
        f_type_decl
          = Lazy.from_fun f_type_decl ;
        c_value = c_value;
        context = context
      }

      


  and wrap_enum_lit_synth_type_expr context c_value
   : enum_lit_synth_type_expr =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnumLitSynthTypeExpr {
        c_value = c_value;
        context = context
      }

      


  and wrap_subtype_indication context c_value
   : subtype_indication =
    let f_has_not_null () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subtype_indication_f_has_not_null
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_not_null context (field_c_value)
      in
         

      node
    in
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subtype_indication_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CharLiteral _
      | `DottedName _
      | `Identifier _
      | `StringLiteral _
      as e -> e
      | _ -> assert false
    in
    let f_constraint () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subtype_indication_f_constraint
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_constraint context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SubtypeIndication {
        f_has_not_null
          = Lazy.from_fun f_has_not_null ;
        f_name
          = Lazy.from_fun f_name ;
        f_constraint
          = Lazy.from_fun f_constraint ;
        c_value = c_value;
        context = context
      }

      


  and wrap_constrained_subtype_indication context c_value
   : constrained_subtype_indication =
    let f_has_not_null () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subtype_indication_f_has_not_null
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_not_null context (field_c_value)
      in
         

      node
    in
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subtype_indication_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CharLiteral _
      | `DottedName _
      | `Identifier _
      | `StringLiteral _
      as e -> e
      | _ -> assert false
    in
    let f_constraint () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subtype_indication_f_constraint
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_constraint context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ConstrainedSubtypeIndication {
        f_has_not_null
          = Lazy.from_fun f_has_not_null ;
        f_name
          = Lazy.from_fun f_name ;
        f_constraint
          = Lazy.from_fun f_constraint ;
        c_value = c_value;
        context = context
      }

      


  and wrap_discrete_subtype_indication context c_value
   : discrete_subtype_indication =
    let f_has_not_null () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subtype_indication_f_has_not_null
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_not_null context (field_c_value)
      in
         

      node
    in
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subtype_indication_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name context (field_c_value)
      in
         

      match node with
            
      | `AttributeRef _
      | `CharLiteral _
      | `DottedName _
      | `Identifier _
      | `StringLiteral _
      as e -> e
      | _ -> assert false
    in
    let f_constraint () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subtype_indication_f_constraint
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_constraint context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DiscreteSubtypeIndication {
        f_has_not_null
          = Lazy.from_fun f_has_not_null ;
        f_name
          = Lazy.from_fun f_name ;
        f_constraint
          = Lazy.from_fun f_constraint ;
        c_value = c_value;
        context = context
      }

      


  and wrap_synthetic_type_expr context c_value
   : synthetic_type_expr =
    let f_target_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.synthetic_type_expr_f_target_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_base_type_decl context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SyntheticTypeExpr {
        f_target_type
          = Lazy.from_fun f_target_type ;
        c_value = c_value;
        context = context
      }

      


  and wrap_unconstrained_array_index context c_value
   : unconstrained_array_index =
    let f_subtype_indication () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.unconstrained_array_index_f_subtype_indication
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_subtype_indication context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `UnconstrainedArrayIndex {
        f_subtype_indication
          = Lazy.from_fun f_subtype_indication ;
        c_value = c_value;
        context = context
      }

      

  and wrap_until_node context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `UntilAbsent _
      | `UntilPresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_until_absent context c_value
   : until_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `UntilAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_until_present context c_value
   : until_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `UntilPresent {
        c_value = c_value;
        context = context
      }

      

  and wrap_use_clause context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `UsePackageClause _
      | `UseTypeClause _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_use_package_clause context c_value
   : use_package_clause =
    let f_packages () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.use_package_clause_f_packages
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `UsePackageClause {
        f_packages
          = Lazy.from_fun f_packages ;
        c_value = c_value;
        context = context
      }

      


  and wrap_use_type_clause context c_value
   : use_type_clause =
    let f_has_all () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.use_type_clause_f_has_all
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_all_node context (field_c_value)
      in
         

      node
    in
    let f_types () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.use_type_clause_f_types
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `UseTypeClause {
        f_has_all
          = Lazy.from_fun f_has_all ;
        f_types
          = Lazy.from_fun f_types ;
        c_value = c_value;
        context = context
      }

      


  and wrap_value_sequence context c_value
   : value_sequence =
    let f_iter_assoc () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.value_sequence_f_iter_assoc
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_iterated_assoc context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ValueSequence {
        f_iter_assoc
          = Lazy.from_fun f_iter_assoc ;
        c_value = c_value;
        context = context
      }

      


  and wrap_variant context c_value
   : variant =
    let f_choices () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.variant_f_choices
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_alternatives_list context (field_c_value)
      in
         

      node
    in
    let f_components () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.variant_f_components
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_component_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `Variant {
        f_choices
          = Lazy.from_fun f_choices ;
        f_components
          = Lazy.from_fun f_components ;
        c_value = c_value;
        context = context
      }

      


  and wrap_variant_part context c_value
   : variant_part =
    let f_discr_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.variant_part_f_discr_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_identifier context (field_c_value)
      in
         

      node
    in
    let f_variant () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.variant_part_f_variant
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_variant_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `VariantPart {
        f_discr_name
          = Lazy.from_fun f_discr_name ;
        f_variant
          = Lazy.from_fun f_variant ;
        c_value = c_value;
        context = context
      }

      


  and wrap_with_clause context c_value
   : with_clause =
    let f_has_limited () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.with_clause_f_has_limited
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_limited_node context (field_c_value)
      in
         

      node
    in
    let f_has_private () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.with_clause_f_has_private
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_private_node context (field_c_value)
      in
         

      node
    in
    let f_packages () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.with_clause_f_packages
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_name_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `WithClause {
        f_has_limited
          = Lazy.from_fun f_has_limited ;
        f_has_private
          = Lazy.from_fun f_has_private ;
        f_packages
          = Lazy.from_fun f_packages ;
        c_value = c_value;
        context = context
      }

      

  and wrap_with_private context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_ada_node context (c_value) with
      | `WithPrivateAbsent _
      | `WithPrivatePresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_with_private_absent context c_value
   : with_private_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `WithPrivateAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_with_private_present context c_value
   : with_private_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `WithPrivatePresent {
        c_value = c_value;
        context = context
      }



   
  and wrap_entity_info c_value = {
    md = wrap_metadata ((getf c_value EntityInfoStruct.md));
    rebindings = (getf c_value EntityInfoStruct.rebindings);
    from_rebound = (getf c_value EntityInfoStruct.from_rebound);
  }



   
  and wrap_metadata c_value = {
    dottable_subp = (getf c_value MetadataStruct.dottable_subp);
    primitive = (getf c_value MetadataStruct.primitive);
    primitive_real_type = (getf c_value MetadataStruct.primitive_real_type);
  }


and wrap_analysis_unit context c_value
   : analysis_unit = {
 c_value=c_value;
 context=context;
}

module Entity = struct
  type t = entity

  let info value =
    wrap_entity_info (getf value EntityStruct.info)

  let compare e1 e2 =
    let open Stdlib in
    let compare_node =
      compare (getf e1 EntityStruct.node) (getf e2 EntityStruct.node)
    in
    if compare_node = 0 then
      compare
        (getf (getf e1 EntityStruct.info) EntityInfoStruct.rebindings)
        (getf (getf e2 EntityStruct.info) EntityInfoStruct.rebindings)
    else
      compare_node

  let equal e1 e2 =
    compare e1 e2 = 0

  let hash e =
    Hashtbl.hash
      ( getf e EntityStruct.node
      , getf (getf e EntityStruct.info) EntityInfoStruct.rebindings )
end

module AnalysisUnit = struct
  type t = analysis_unit

  let root (unit : t) =
    let c_value = make EntityStruct.c_type in
    AnalysisUnitStruct.unit_root
      (unwrap_analysis_unit (unit))
      (addr c_value);
    if is_null (getf c_value EntityStruct.node) then None else Some (wrap_ada_node unit.context (c_value))

  let diagnostics (unit : t) =
    let c_unit = unwrap_analysis_unit (unit) in
    let length = AnalysisUnitStruct.unit_diagnostic_count c_unit in
    let f i =
      let diag = allocate_n Diagnostic.c_type ~count:1 in
      let _ : int = AnalysisUnitStruct.unit_diagnostic c_unit i diag in
      !@ diag
    in
    List.init length f

  let filename (unit : t) =
    unwrap_str( AnalysisUnitStruct.unit_filename
      (unwrap_analysis_unit (unit)))

  let reparse ?charset:(charset="") ?buffer (unit : t) =
    match buffer with
    | None ->
        ignore
          (AnalysisUnitStruct.unit_reparse_from_file unit.c_value charset)
    | Some buffer ->
        ignore (AnalysisUnitStruct.unit_reparse_from_buffer unit.c_value
          charset buffer (Unsigned.Size_t.of_int (String.length buffer)))

  let first_token (unit : t) =
    let c_unit = unwrap_analysis_unit (unit) in
    let result_ptr = allocate_n Token.c_type ~count:1 in
    AnalysisUnitStruct.unit_first_token c_unit result_ptr ;
    Token.wrap (!@ result_ptr)

  let last_token (unit : t) =
    let c_unit = unwrap_analysis_unit (unit) in
    let result_ptr = allocate_n Token.c_type ~count:1 in
    AnalysisUnitStruct.unit_last_token c_unit result_ptr ;
    Token.wrap (!@ result_ptr)

  let token_count (unit : t) =
    AnalysisUnitStruct.unit_token_count
      (unwrap_analysis_unit (unit))

  let trivia_count (unit : t) =
    AnalysisUnitStruct.unit_trivia_count
      (unwrap_analysis_unit (unit))

  
  let fold_tokens f init node =
    match first_token node, last_token node with
    | Some tok_start, Some tok_end ->
        let rec aux acc tok_curr =
          let new_acc = f acc tok_curr in
          if Token.equal tok_curr tok_end then
            new_acc
          else (
            match Token.next tok_curr with
            | Some tok_next ->
                aux new_acc tok_next
            | None ->
                new_acc )
        in
        aux init tok_start
    | _ ->
        init

  let iter_tokens f node =
    match first_token node, last_token node with
    | Some tok_start, Some tok_end ->
        let rec aux tok_curr =
          f tok_curr;
          if not (Token.equal tok_curr tok_end) then (
            match Token.next tok_curr with
            | Some tok_next ->
                aux tok_next
            | None ->
                () )
        in
        aux tok_start
    | _ ->
        ()

  let map_tokens f node =
    match first_token node, last_token node with
    | Some tok_start, Some tok_end ->
        let rec aux tok_curr =
          let value = f tok_curr in
          if Token.equal tok_curr tok_end then
            [value]
          else (
            match Token.next tok_curr with
            | Some tok_next ->
                value :: aux tok_next
            | None ->
                [value] )
        in
        aux tok_start
    | _ ->
        []

  let tokens node =
    map_tokens (fun x -> x) node

end

module AnalysisContext = struct
  type t = analysis_context

  let create
    ?charset:(charset="")
    ?with_trivia:(with_trivia=true)
    ?tab_stop:(tab_stop=8)
    ?unit_provider:(unit_provider=UnitProvider.null)
    ?file_reader () : t =
    if tab_stop < 1 then
      raise (Invalid_argument "Invalid tab_stop (positive integer expected)") ;
    let c_context = AnalysisContextStruct.allocate_analysis_context () in
    AnalysisContextStruct.initialize_analysis_context
      c_context
      charset
      (match file_reader with Some v -> !@v | None -> null)
      (!@unit_provider)
      Ctypes.null (* TODO: bind the event handlers API to OCaml *)
      with_trivia
      tab_stop ;
    { c_value= c_context
      ; unit_provider= unit_provider }

  let get_from_file
    ?charset:(charset="")
    ?reparse:(reparse=false)
    ?grammar_rule:(grammar_rule=default_grammar_rule)
    (ctx : t)
    filename : AnalysisUnit.t =

    wrap_analysis_unit ctx (AnalysisContextStruct.get_analysis_unit_from_file ctx.c_value filename charset reparse grammar_rule)

  let get_from_buffer
    ?charset:(charset="")
    ?grammar_rule:(grammar_rule=default_grammar_rule)
    (ctx : t)
    filename
    buffer : AnalysisUnit.t =

    wrap_analysis_unit ctx (AnalysisContextStruct.get_analysis_unit_from_buffer ctx.c_value filename charset buffer (Unsigned.Size_t.of_int (String.length buffer)) grammar_rule)
end

   
module Aspect : sig
   
  
  type t = {
      exists :
         bool;
      node :
         ada_node option;
      value :
         expr option;
      inherited :
         bool;
  }


  val wrap : analysis_context -> AspectStruct.t structure -> t

  val unwrap : t -> AspectStruct.t structure


end = struct
   
  
  type t = {
      exists :
         bool;
      node :
         ada_node option;
      value :
         expr option;
      inherited :
         bool;
  }


  
   
  let wrap context c_value = {
    exists = (getf c_value AspectStruct.exists);
    node = if is_null (getf (getf c_value AspectStruct.node) EntityStruct.node) then None else Some (wrap_ada_node context ((getf c_value AspectStruct.node)));
    value = if is_null (getf (getf c_value AspectStruct.value) EntityStruct.node) then None else Some (wrap_expr context ((getf c_value AspectStruct.value)));
    inherited = (getf c_value AspectStruct.inherited);
  }


  
   

  let unwrap value =
    let c_value = make AspectStruct.c_type in
    let field_c_value =
      value.exists
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      AspectStruct.exists
      field_c_value;
    let field_c_value =
      match value.node with Some n -> unwrap_ada_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      AspectStruct.node
      field_c_value;
    let field_c_value =
      match value.value with Some n -> unwrap_ada_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      AspectStruct.value
      field_c_value;
    let field_c_value =
      value.inherited
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      AspectStruct.inherited
      field_c_value;
    c_value



end

   
module CompletionItem : sig
   
  
  type t = {
      decl :
         basic_decl option;
      is_dot_call :
         bool;
      is_visible :
         bool;
      weight :
         int;
  }


  val wrap : analysis_context -> CompletionItemStruct.t structure -> t

  val unwrap : t -> CompletionItemStruct.t structure


end = struct
   
  
  type t = {
      decl :
         basic_decl option;
      is_dot_call :
         bool;
      is_visible :
         bool;
      weight :
         int;
  }


  
   
  let wrap context c_value = {
    decl = if is_null (getf (getf c_value CompletionItemStruct.decl) EntityStruct.node) then None else Some (wrap_basic_decl context ((getf c_value CompletionItemStruct.decl)));
    is_dot_call = (getf c_value CompletionItemStruct.is_dot_call);
    is_visible = (getf c_value CompletionItemStruct.is_visible);
    weight = (getf c_value CompletionItemStruct.weight);
  }


  
   

  let unwrap value =
    let c_value = make CompletionItemStruct.c_type in
    let field_c_value =
      match value.decl with Some n -> unwrap_ada_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      CompletionItemStruct.decl
      field_c_value;
    let field_c_value =
      value.is_dot_call
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      CompletionItemStruct.is_dot_call
      field_c_value;
    let field_c_value =
      value.is_visible
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      CompletionItemStruct.is_visible
      field_c_value;
    let field_c_value =
      value.weight
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      CompletionItemStruct.weight
      field_c_value;
    c_value



end

   
module DiscreteRange : sig
   
  
  type t = {
      low_bound :
         expr option;
      high_bound :
         expr option;
  }


  val wrap : analysis_context -> DiscreteRangeStruct.t structure -> t

  val unwrap : t -> DiscreteRangeStruct.t structure


end = struct
   
  
  type t = {
      low_bound :
         expr option;
      high_bound :
         expr option;
  }


  
   
  let wrap context c_value = {
    low_bound = if is_null (getf (getf c_value DiscreteRangeStruct.low_bound) EntityStruct.node) then None else Some (wrap_expr context ((getf c_value DiscreteRangeStruct.low_bound)));
    high_bound = if is_null (getf (getf c_value DiscreteRangeStruct.high_bound) EntityStruct.node) then None else Some (wrap_expr context ((getf c_value DiscreteRangeStruct.high_bound)));
  }


  
   

  let unwrap value =
    let c_value = make DiscreteRangeStruct.c_type in
    let field_c_value =
      match value.low_bound with Some n -> unwrap_ada_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      DiscreteRangeStruct.low_bound
      field_c_value;
    let field_c_value =
      match value.high_bound with Some n -> unwrap_ada_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      DiscreteRangeStruct.high_bound
      field_c_value;
    c_value



end

   
module DiscriminantValues : sig
   
  
  type t = {
      discriminant :
         identifier option;
      values :
         alternatives_list option;
  }


  val wrap : analysis_context -> DiscriminantValuesStruct.t structure -> t

  val unwrap : t -> DiscriminantValuesStruct.t structure


end = struct
   
  
  type t = {
      discriminant :
         identifier option;
      values :
         alternatives_list option;
  }


  
   
  let wrap context c_value = {
    discriminant = if is_null (getf (getf c_value DiscriminantValuesStruct.discriminant) EntityStruct.node) then None else Some (wrap_identifier context ((getf c_value DiscriminantValuesStruct.discriminant)));
    values = if is_null (getf (getf c_value DiscriminantValuesStruct.values) EntityStruct.node) then None else Some (wrap_alternatives_list context ((getf c_value DiscriminantValuesStruct.values)));
  }


  
   

  let unwrap value =
    let c_value = make DiscriminantValuesStruct.c_type in
    let field_c_value =
      match value.discriminant with Some n -> unwrap_ada_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      DiscriminantValuesStruct.discriminant
      field_c_value;
    let field_c_value =
      match value.values with Some n -> unwrap_ada_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      DiscriminantValuesStruct.values
      field_c_value;
    c_value



end

   
module DocAnnotation : sig
   
  
  type t = {
      key :
         string;
      value :
         string;
  }


  val wrap : DocAnnotationStruct.t structure -> t

  val unwrap : t -> DocAnnotationStruct.t structure


end = struct
   
  
  type t = {
      key :
         string;
      value :
         string;
  }


  
   
  let wrap c_value = {
    key = StringType.wrap ((getf c_value DocAnnotationStruct.key));
    value = StringType.wrap ((getf c_value DocAnnotationStruct.value));
  }


  
   

  let unwrap value =
    let c_value = make DocAnnotationStruct.c_type in
    let field_c_value =
      StringType.unwrap (value.key)
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      DocAnnotationStruct.key
      field_c_value;
    let field_c_value =
      StringType.unwrap (value.value)
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      DocAnnotationStruct.value
      field_c_value;
    c_value



end

   
module ParamActual : sig
   
  
  type t = {
      param :
         defining_name option;
      actual :
         expr option;
  }


  val wrap : analysis_context -> ParamActualStruct.t structure -> t

  val unwrap : t -> ParamActualStruct.t structure


end = struct
   
  
  type t = {
      param :
         defining_name option;
      actual :
         expr option;
  }


  
   
  let wrap context c_value = {
    param = if is_null (getf (getf c_value ParamActualStruct.param) EntityStruct.node) then None else Some (wrap_defining_name context ((getf c_value ParamActualStruct.param)));
    actual = if is_null (getf (getf c_value ParamActualStruct.actual) EntityStruct.node) then None else Some (wrap_expr context ((getf c_value ParamActualStruct.actual)));
  }


  
   

  let unwrap value =
    let c_value = make ParamActualStruct.c_type in
    let field_c_value =
      match value.param with Some n -> unwrap_ada_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      ParamActualStruct.param
      field_c_value;
    let field_c_value =
      match value.actual with Some n -> unwrap_ada_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      ParamActualStruct.actual
      field_c_value;
    c_value



end

   
module RefResult : sig
   
  
  type t = {
      ref_ :
         base_id option;
      kind :
         RefResultKind.t;
  }


  val wrap : analysis_context -> RefResultStruct.t structure -> t

  val unwrap : t -> RefResultStruct.t structure


end = struct
   
  
  type t = {
      ref_ :
         base_id option;
      kind :
         RefResultKind.t;
  }


  
   
  let wrap context c_value = {
    ref_ = if is_null (getf (getf c_value RefResultStruct.ref_) EntityStruct.node) then None else Some (wrap_base_id context ((getf c_value RefResultStruct.ref_)));
    kind = (getf c_value RefResultStruct.kind);
  }


  
   

  let unwrap value =
    let c_value = make RefResultStruct.c_type in
    let field_c_value =
      match value.ref_ with Some n -> unwrap_ada_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      RefResultStruct.ref_
      field_c_value;
    let field_c_value =
      value.kind
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      RefResultStruct.kind
      field_c_value;
    c_value



end

   
module RefdDecl : sig
   
  
  type t = {
      decl :
         basic_decl option;
      kind :
         RefResultKind.t;
  }


  val wrap : analysis_context -> RefdDeclStruct.t structure -> t

  val unwrap : t -> RefdDeclStruct.t structure


end = struct
   
  
  type t = {
      decl :
         basic_decl option;
      kind :
         RefResultKind.t;
  }


  
   
  let wrap context c_value = {
    decl = if is_null (getf (getf c_value RefdDeclStruct.decl) EntityStruct.node) then None else Some (wrap_basic_decl context ((getf c_value RefdDeclStruct.decl)));
    kind = (getf c_value RefdDeclStruct.kind);
  }


  
   

  let unwrap value =
    let c_value = make RefdDeclStruct.c_type in
    let field_c_value =
      match value.decl with Some n -> unwrap_ada_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      RefdDeclStruct.decl
      field_c_value;
    let field_c_value =
      value.kind
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      RefdDeclStruct.kind
      field_c_value;
    c_value



end

   
module RefdDef : sig
   
  
  type t = {
      def_name :
         defining_name option;
      kind :
         RefResultKind.t;
  }


  val wrap : analysis_context -> RefdDefStruct.t structure -> t

  val unwrap : t -> RefdDefStruct.t structure


end = struct
   
  
  type t = {
      def_name :
         defining_name option;
      kind :
         RefResultKind.t;
  }


  
   
  let wrap context c_value = {
    def_name = if is_null (getf (getf c_value RefdDefStruct.def_name) EntityStruct.node) then None else Some (wrap_defining_name context ((getf c_value RefdDefStruct.def_name)));
    kind = (getf c_value RefdDefStruct.kind);
  }


  
   

  let unwrap value =
    let c_value = make RefdDefStruct.c_type in
    let field_c_value =
      match value.def_name with Some n -> unwrap_ada_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      RefdDefStruct.def_name
      field_c_value;
    let field_c_value =
      value.kind
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      RefdDefStruct.kind
      field_c_value;
    c_value



end

   
module BaseFormalParamDeclArray : sig
   
  type t = base_formal_param_decl list

  val wrap : analysis_context -> AdaNodeArrayStruct.t structure ptr -> t

  val unwrap : t -> AdaNodeArrayStruct.t structure ptr

end = struct
   
  type t = base_formal_param_decl list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value AdaNodeArrayStruct.n in
    let items = c_value @. AdaNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      wrap_base_formal_param_decl context (!@ fresh)
    in
    let result = List.init length f in
    AdaNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = AdaNodeArrayStruct.create (List.length value) in
    let items = result |-> AdaNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_ada_node (v)
    in
    List.iteri f value;
    result


end

   
module AdaNodeArray : sig
   
  type t = ada_node list

  val wrap : analysis_context -> AdaNodeArrayStruct.t structure ptr -> t

  val unwrap : t -> AdaNodeArrayStruct.t structure ptr

end = struct
   
  type t = ada_node list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value AdaNodeArrayStruct.n in
    let items = c_value @. AdaNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      wrap_ada_node context (!@ fresh)
    in
    let result = List.init length f in
    AdaNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = AdaNodeArrayStruct.create (List.length value) in
    let items = result |-> AdaNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_ada_node (v)
    in
    List.iteri f value;
    result


end

   
module DiscriminantValuesArray : sig
   
  type t = DiscriminantValues.t list

  val wrap : analysis_context -> DiscriminantValuesArrayStruct.t structure ptr -> t

  val unwrap : t -> DiscriminantValuesArrayStruct.t structure ptr

end = struct
   
  type t = DiscriminantValues.t list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value DiscriminantValuesArrayStruct.n in
    let items = c_value @. DiscriminantValuesArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate DiscriminantValuesStruct.c_type (!@ (items +@ i))
      in
      DiscriminantValues.wrap context (!@ fresh)
    in
    let result = List.init length f in
    DiscriminantValuesArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = DiscriminantValuesArrayStruct.create (List.length value) in
    let items = result |-> DiscriminantValuesArrayStruct.items in
    let f i v =
      items +@ i <-@
        DiscriminantValues.unwrap (v)
    in
    List.iteri f value;
    result


end

   
module Shape : sig
   
  
  type t = {
      components :
         base_formal_param_decl list;
      discriminants_values :
         DiscriminantValues.t list;
  }


  val wrap : analysis_context -> ShapeStruct.t structure -> t

  val unwrap : t -> ShapeStruct.t structure


end = struct
   
  
  type t = {
      components :
         base_formal_param_decl list;
      discriminants_values :
         DiscriminantValues.t list;
  }


  
   
  let wrap context c_value = {
    components = BaseFormalParamDeclArray.wrap context ((getf c_value ShapeStruct.components));
    discriminants_values = DiscriminantValuesArray.wrap context ((getf c_value ShapeStruct.discriminants_values));
  }


  
   

  let unwrap value =
    let c_value = make ShapeStruct.c_type in
    let field_c_value =
      BaseFormalParamDeclArray.unwrap (value.components)
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      ShapeStruct.components
      field_c_value;
    let field_c_value =
      DiscriminantValuesArray.unwrap (value.discriminants_values)
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      ShapeStruct.discriminants_values
      field_c_value;
    c_value



end

   
module Substitution : sig
   
  
  type t = {
      from_decl :
         basic_decl option;
      to_value :
         BigInteger.t;
      value_type :
         base_type_decl option;
  }


  val wrap : analysis_context -> SubstitutionStruct.t structure -> t

  val unwrap : t -> SubstitutionStruct.t structure


end = struct
   
  
  type t = {
      from_decl :
         basic_decl option;
      to_value :
         BigInteger.t;
      value_type :
         base_type_decl option;
  }


  
   
  let wrap context c_value = {
    from_decl = if is_null (getf (getf c_value SubstitutionStruct.from_decl) EntityStruct.node) then None else Some (wrap_basic_decl context ((getf c_value SubstitutionStruct.from_decl)));
    to_value = BigInteger.wrap ((getf c_value SubstitutionStruct.to_value));
    value_type = if is_null (getf (getf c_value SubstitutionStruct.value_type) EntityStruct.node) then None else Some (wrap_base_type_decl context ((getf c_value SubstitutionStruct.value_type)));
  }


  
   

  let unwrap value =
    let c_value = make SubstitutionStruct.c_type in
    let field_c_value =
      match value.from_decl with Some n -> unwrap_ada_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      SubstitutionStruct.from_decl
      field_c_value;
    let field_c_value =
      BigInteger.unwrap (value.to_value)
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      SubstitutionStruct.to_value
      field_c_value;
    let field_c_value =
      match value.value_type with Some n -> unwrap_ada_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      SubstitutionStruct.value_type
      field_c_value;
    c_value



end

   
module DocAnnotationArray : sig
   
  type t = DocAnnotation.t list

  val wrap : DocAnnotationArrayStruct.t structure ptr -> t

  val unwrap : t -> DocAnnotationArrayStruct.t structure ptr

end = struct
   
  type t = DocAnnotation.t list

  let wrap c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value DocAnnotationArrayStruct.n in
    let items = c_value @. DocAnnotationArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate DocAnnotationStruct.c_type (!@ (items +@ i))
      in
      DocAnnotation.wrap (!@ fresh)
    in
    let result = List.init length f in
    DocAnnotationArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = DocAnnotationArrayStruct.create (List.length value) in
    let items = result |-> DocAnnotationArrayStruct.items in
    let f i v =
      items +@ i <-@
        DocAnnotation.unwrap (v)
    in
    List.iteri f value;
    result


end

   
module AcceptStmtArray : sig
   
  type t = accept_stmt list

  val wrap : analysis_context -> AdaNodeArrayStruct.t structure ptr -> t

  val unwrap : t -> AdaNodeArrayStruct.t structure ptr

end = struct
   
  type t = accept_stmt list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value AdaNodeArrayStruct.n in
    let items = c_value @. AdaNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      wrap_accept_stmt context (!@ fresh)
    in
    let result = List.init length f in
    AdaNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = AdaNodeArrayStruct.create (List.length value) in
    let items = result |-> AdaNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_ada_node (v)
    in
    List.iteri f value;
    result


end

   
module BaseTypeDeclArray : sig
   
  type t = base_type_decl list

  val wrap : analysis_context -> AdaNodeArrayStruct.t structure ptr -> t

  val unwrap : t -> AdaNodeArrayStruct.t structure ptr

end = struct
   
  type t = base_type_decl list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value AdaNodeArrayStruct.n in
    let items = c_value @. AdaNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      wrap_base_type_decl context (!@ fresh)
    in
    let result = List.init length f in
    AdaNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = AdaNodeArrayStruct.create (List.length value) in
    let items = result |-> AdaNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_ada_node (v)
    in
    List.iteri f value;
    result


end

   
module BasicDeclArray : sig
   
  type t = basic_decl list

  val wrap : analysis_context -> AdaNodeArrayStruct.t structure ptr -> t

  val unwrap : t -> AdaNodeArrayStruct.t structure ptr

end = struct
   
  type t = basic_decl list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value AdaNodeArrayStruct.n in
    let items = c_value @. AdaNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      wrap_basic_decl context (!@ fresh)
    in
    let result = List.init length f in
    AdaNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = AdaNodeArrayStruct.create (List.length value) in
    let items = result |-> AdaNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_ada_node (v)
    in
    List.iteri f value;
    result


end

   
module CompilationUnitArray : sig
   
  type t = compilation_unit list

  val wrap : analysis_context -> AdaNodeArrayStruct.t structure ptr -> t

  val unwrap : t -> AdaNodeArrayStruct.t structure ptr

end = struct
   
  type t = compilation_unit list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value AdaNodeArrayStruct.n in
    let items = c_value @. AdaNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      wrap_compilation_unit context (!@ fresh)
    in
    let result = List.init length f in
    AdaNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = AdaNodeArrayStruct.create (List.length value) in
    let items = result |-> AdaNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_ada_node (v)
    in
    List.iteri f value;
    result


end

   
module DefiningNameArray : sig
   
  type t = defining_name list

  val wrap : analysis_context -> AdaNodeArrayStruct.t structure ptr -> t

  val unwrap : t -> AdaNodeArrayStruct.t structure ptr

end = struct
   
  type t = defining_name list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value AdaNodeArrayStruct.n in
    let items = c_value @. AdaNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      wrap_defining_name context (!@ fresh)
    in
    let result = List.init length f in
    AdaNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = AdaNodeArrayStruct.create (List.length value) in
    let items = result |-> AdaNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_ada_node (v)
    in
    List.iteri f value;
    result


end

   
module ExprArray : sig
   
  type t = expr list

  val wrap : analysis_context -> AdaNodeArrayStruct.t structure ptr -> t

  val unwrap : t -> AdaNodeArrayStruct.t structure ptr

end = struct
   
  type t = expr list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value AdaNodeArrayStruct.n in
    let items = c_value @. AdaNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      wrap_expr context (!@ fresh)
    in
    let result = List.init length f in
    AdaNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = AdaNodeArrayStruct.create (List.length value) in
    let items = result |-> AdaNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_ada_node (v)
    in
    List.iteri f value;
    result


end

   
module GenericInstantiationArray : sig
   
  type t = generic_instantiation list

  val wrap : analysis_context -> AdaNodeArrayStruct.t structure ptr -> t

  val unwrap : t -> AdaNodeArrayStruct.t structure ptr

end = struct
   
  type t = generic_instantiation list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value AdaNodeArrayStruct.n in
    let items = c_value @. AdaNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      wrap_generic_instantiation context (!@ fresh)
    in
    let result = List.init length f in
    AdaNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = AdaNodeArrayStruct.create (List.length value) in
    let items = result |-> AdaNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_ada_node (v)
    in
    List.iteri f value;
    result


end

   
module ParamSpecArray : sig
   
  type t = param_spec list

  val wrap : analysis_context -> AdaNodeArrayStruct.t structure ptr -> t

  val unwrap : t -> AdaNodeArrayStruct.t structure ptr

end = struct
   
  type t = param_spec list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value AdaNodeArrayStruct.n in
    let items = c_value @. AdaNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      wrap_param_spec context (!@ fresh)
    in
    let result = List.init length f in
    AdaNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = AdaNodeArrayStruct.create (List.length value) in
    let items = result |-> AdaNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_ada_node (v)
    in
    List.iteri f value;
    result


end

   
module PragmaNodeArray : sig
   
  type t = pragma_node list

  val wrap : analysis_context -> AdaNodeArrayStruct.t structure ptr -> t

  val unwrap : t -> AdaNodeArrayStruct.t structure ptr

end = struct
   
  type t = pragma_node list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value AdaNodeArrayStruct.n in
    let items = c_value @. AdaNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      wrap_pragma_node context (!@ fresh)
    in
    let result = List.init length f in
    AdaNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = AdaNodeArrayStruct.create (List.length value) in
    let items = result |-> AdaNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_ada_node (v)
    in
    List.iteri f value;
    result


end

   
module TypeDeclArray : sig
   
  type t = type_decl list

  val wrap : analysis_context -> AdaNodeArrayStruct.t structure ptr -> t

  val unwrap : t -> AdaNodeArrayStruct.t structure ptr

end = struct
   
  type t = type_decl list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value AdaNodeArrayStruct.n in
    let items = c_value @. AdaNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      wrap_type_decl context (!@ fresh)
    in
    let result = List.init length f in
    AdaNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = AdaNodeArrayStruct.create (List.length value) in
    let items = result |-> AdaNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_ada_node (v)
    in
    List.iteri f value;
    result


end

   
module ParamActualArray : sig
   
  type t = ParamActual.t list

  val wrap : analysis_context -> ParamActualArrayStruct.t structure ptr -> t

  val unwrap : t -> ParamActualArrayStruct.t structure ptr

end = struct
   
  type t = ParamActual.t list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value ParamActualArrayStruct.n in
    let items = c_value @. ParamActualArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate ParamActualStruct.c_type (!@ (items +@ i))
      in
      ParamActual.wrap context (!@ fresh)
    in
    let result = List.init length f in
    ParamActualArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = ParamActualArrayStruct.create (List.length value) in
    let items = result |-> ParamActualArrayStruct.items in
    let f i v =
      items +@ i <-@
        ParamActual.unwrap (v)
    in
    List.iteri f value;
    result


end

   
module RefResultArray : sig
   
  type t = RefResult.t list

  val wrap : analysis_context -> RefResultArrayStruct.t structure ptr -> t

  val unwrap : t -> RefResultArrayStruct.t structure ptr

end = struct
   
  type t = RefResult.t list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value RefResultArrayStruct.n in
    let items = c_value @. RefResultArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate RefResultStruct.c_type (!@ (items +@ i))
      in
      RefResult.wrap context (!@ fresh)
    in
    let result = List.init length f in
    RefResultArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = RefResultArrayStruct.create (List.length value) in
    let items = result |-> RefResultArrayStruct.items in
    let f i v =
      items +@ i <-@
        RefResult.unwrap (v)
    in
    List.iteri f value;
    result


end

   
module ShapeArray : sig
   
  type t = Shape.t list

  val wrap : analysis_context -> ShapeArrayStruct.t structure ptr -> t

  val unwrap : t -> ShapeArrayStruct.t structure ptr

end = struct
   
  type t = Shape.t list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value ShapeArrayStruct.n in
    let items = c_value @. ShapeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate ShapeStruct.c_type (!@ (items +@ i))
      in
      Shape.wrap context (!@ fresh)
    in
    let result = List.init length f in
    ShapeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = ShapeArrayStruct.create (List.length value) in
    let items = result |-> ShapeArrayStruct.items in
    let f i v =
      items +@ i <-@
        Shape.unwrap (v)
    in
    List.iteri f value;
    result


end

   
module SubstitutionArray : sig
   
  type t = Substitution.t list

  val wrap : analysis_context -> SubstitutionArrayStruct.t structure ptr -> t

  val unwrap : t -> SubstitutionArrayStruct.t structure ptr

end = struct
   
  type t = Substitution.t list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value SubstitutionArrayStruct.n in
    let items = c_value @. SubstitutionArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate SubstitutionStruct.c_type (!@ (items +@ i))
      in
      Substitution.wrap context (!@ fresh)
    in
    let result = List.init length f in
    SubstitutionArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = SubstitutionArrayStruct.create (List.length value) in
    let items = result |-> SubstitutionArrayStruct.items in
    let f i v =
      items +@ i <-@
        Substitution.unwrap (v)
    in
    List.iteri f value;
    result


end

   
module AnalysisUnitArray : sig
   
  type t = analysis_unit list

  val wrap : analysis_context -> AnalysisUnitArrayStruct.t structure ptr -> t

  val unwrap : t -> AnalysisUnitArrayStruct.t structure ptr

end = struct
   
  type t = analysis_unit list

  let wrap (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value AnalysisUnitArrayStruct.n in
    let items = c_value @. AnalysisUnitArrayStruct.items in
    let f i =
      wrap_analysis_unit context (!@ (items +@ i))
    in
    let result = List.init length f in
    AnalysisUnitArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = AnalysisUnitArrayStruct.create (List.length value) in
    let items = result |-> AnalysisUnitArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_analysis_unit (v)
    in
    List.iteri f value;
    result


end

   
module UnboundedTextTypeArray : sig
   
  type t = string list

  val wrap : UnboundedTextTypeArrayStruct.t structure ptr -> t

  val unwrap : analysis_context -> t -> UnboundedTextTypeArrayStruct.t structure ptr

end = struct
   
  type t = string list

  let wrap c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value UnboundedTextTypeArrayStruct.n in
    let items = c_value @. UnboundedTextTypeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate Symbol.c_type (!@ (items +@ i))
      in
      Symbol.wrap (!@ fresh)
    in
    let result = List.init length f in
    UnboundedTextTypeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap (context : analysis_context) value =
    let result = UnboundedTextTypeArrayStruct.create (List.length value) in
    let items = result |-> UnboundedTextTypeArrayStruct.items in
    let c_context = context.c_value in
    let f i v =
      items +@ i <-@
        Symbol.unwrap c_context (v)
    in
    List.iteri f value;
    result


end


let context node =
  (* Given any node, extract the context field *)
  match (node :> ada_node) with
  | `AbortAbsent fields -> fields.context
  | `AbortPresent fields -> fields.context
  | `AbstractAbsent fields -> fields.context
  | `AbstractPresent fields -> fields.context
  | `AdaNodeList fields -> fields.context
  | `AbstractStateDeclList fields -> fields.context
  | `AlternativesList fields -> fields.context
  | `ConstraintList fields -> fields.context
  | `DeclList fields -> fields.context
  | `StmtList fields -> fields.context
  | `AspectAssocList fields -> fields.context
  | `BaseAssocList fields -> fields.context
  | `AssocList fields -> fields.context
  | `BasicDeclList fields -> fields.context
  | `CaseExprAlternativeList fields -> fields.context
  | `CaseStmtAlternativeList fields -> fields.context
  | `CompilationUnitList fields -> fields.context
  | `ConcatOperandList fields -> fields.context
  | `ContractCaseAssocList fields -> fields.context
  | `DefiningNameList fields -> fields.context
  | `DiscriminantSpecList fields -> fields.context
  | `ElsifExprPartList fields -> fields.context
  | `ElsifStmtPartList fields -> fields.context
  | `EnumLiteralDeclList fields -> fields.context
  | `ExprAlternativesList fields -> fields.context
  | `DiscriminantChoiceList fields -> fields.context
  | `NameList fields -> fields.context
  | `ParentList fields -> fields.context
  | `ParamSpecList fields -> fields.context
  | `PragmaNodeList fields -> fields.context
  | `SelectWhenPartList fields -> fields.context
  | `UnconstrainedArrayIndexList fields -> fields.context
  | `VariantList fields -> fields.context
  | `AliasedAbsent fields -> fields.context
  | `AliasedPresent fields -> fields.context
  | `AllAbsent fields -> fields.context
  | `AllPresent fields -> fields.context
  | `ConstrainedArrayIndices fields -> fields.context
  | `UnconstrainedArrayIndices fields -> fields.context
  | `AspectAssoc fields -> fields.context
  | `AtClause fields -> fields.context
  | `AttributeDefClause fields -> fields.context
  | `EnumRepClause fields -> fields.context
  | `RecordRepClause fields -> fields.context
  | `AspectSpec fields -> fields.context
  | `ContractCaseAssoc fields -> fields.context
  | `PragmaArgumentAssoc fields -> fields.context
  | `EntrySpec fields -> fields.context
  | `EnumSubpSpec fields -> fields.context
  | `SubpSpec fields -> fields.context
  | `SyntheticBinarySpec fields -> fields.context
  | `SyntheticUnarySpec fields -> fields.context
  | `ComponentList fields -> fields.context
  | `KnownDiscriminantPart fields -> fields.context
  | `UnknownDiscriminantPart fields -> fields.context
  | `EntryCompletionFormalParams fields -> fields.context
  | `GenericFormalPart fields -> fields.context
  | `NullRecordDef fields -> fields.context
  | `RecordDef fields -> fields.context
  | `AggregateAssoc fields -> fields.context
  | `MultiDimArrayAssoc fields -> fields.context
  | `CompositeConstraintAssoc fields -> fields.context
  | `IteratedAssoc fields -> fields.context
  | `ParamAssoc fields -> fields.context
  | `AbstractStateDecl fields -> fields.context
  | `AnonymousExprDecl fields -> fields.context
  | `ComponentDecl fields -> fields.context
  | `DiscriminantSpec fields -> fields.context
  | `GenericFormalObjDecl fields -> fields.context
  | `GenericFormalPackage fields -> fields.context
  | `GenericFormalSubpDecl fields -> fields.context
  | `GenericFormalTypeDecl fields -> fields.context
  | `ParamSpec fields -> fields.context
  | `SyntheticFormalParamDecl fields -> fields.context
  | `GenericPackageInternal fields -> fields.context
  | `PackageDecl fields -> fields.context
  | `DiscreteBaseSubtypeDecl fields -> fields.context
  | `SubtypeDecl fields -> fields.context
  | `ClasswideTypeDecl fields -> fields.context
  | `IncompleteTypeDecl fields -> fields.context
  | `IncompleteFormalTypeDecl fields -> fields.context
  | `IncompleteTaggedTypeDecl fields -> fields.context
  | `ProtectedTypeDecl fields -> fields.context
  | `TaskTypeDecl fields -> fields.context
  | `SingleTaskTypeDecl fields -> fields.context
  | `AnonymousTypeDecl fields -> fields.context
  | `SynthAnonymousTypeDecl fields -> fields.context
  | `ConcreteTypeDecl fields -> fields.context
  | `FormalTypeDecl fields -> fields.context
  | `AbstractSubpDecl fields -> fields.context
  | `AbstractFormalSubpDecl fields -> fields.context
  | `ConcreteFormalSubpDecl fields -> fields.context
  | `SubpDecl fields -> fields.context
  | `EntryDecl fields -> fields.context
  | `EnumLiteralDecl fields -> fields.context
  | `SyntheticCharEnumLit fields -> fields.context
  | `GenericSubpInternal fields -> fields.context
  | `SyntheticSubpDecl fields -> fields.context
  | `AcceptStmtBody fields -> fields.context
  | `ExprFunction fields -> fields.context
  | `NullSubpDecl fields -> fields.context
  | `SubpBody fields -> fields.context
  | `SubpRenamingDecl fields -> fields.context
  | `PackageBodyStub fields -> fields.context
  | `ProtectedBodyStub fields -> fields.context
  | `SubpBodyStub fields -> fields.context
  | `TaskBodyStub fields -> fields.context
  | `EntryBody fields -> fields.context
  | `PackageBody fields -> fields.context
  | `ProtectedBody fields -> fields.context
  | `TaskBody fields -> fields.context
  | `EntryIndexSpec fields -> fields.context
  | `ErrorDecl fields -> fields.context
  | `ExceptionDecl fields -> fields.context
  | `ExceptionHandler fields -> fields.context
  | `ForLoopVarDecl fields -> fields.context
  | `GenericPackageDecl fields -> fields.context
  | `GenericSubpDecl fields -> fields.context
  | `GenericPackageInstantiation fields -> fields.context
  | `GenericSubpInstantiation fields -> fields.context
  | `GenericPackageRenamingDecl fields -> fields.context
  | `GenericSubpRenamingDecl fields -> fields.context
  | `LabelDecl fields -> fields.context
  | `NamedStmtDecl fields -> fields.context
  | `NumberDecl fields -> fields.context
  | `ObjectDecl fields -> fields.context
  | `ExtendedReturnStmtObjectDecl fields -> fields.context
  | `NoTypeObjectRenamingDecl fields -> fields.context
  | `PackageRenamingDecl fields -> fields.context
  | `SingleProtectedDecl fields -> fields.context
  | `SingleTaskDecl fields -> fields.context
  | `CaseStmtAlternative fields -> fields.context
  | `CompilationUnit fields -> fields.context
  | `ComponentClause fields -> fields.context
  | `ComponentDef fields -> fields.context
  | `ConstantAbsent fields -> fields.context
  | `ConstantPresent fields -> fields.context
  | `CompositeConstraint fields -> fields.context
  | `DeltaConstraint fields -> fields.context
  | `DigitsConstraint fields -> fields.context
  | `RangeConstraint fields -> fields.context
  | `DeclarativePart fields -> fields.context
  | `PrivatePart fields -> fields.context
  | `PublicPart fields -> fields.context
  | `ElsifExprPart fields -> fields.context
  | `ElsifStmtPart fields -> fields.context
  | `AbstractStateDeclExpr fields -> fields.context
  | `Allocator fields -> fields.context
  | `Aggregate fields -> fields.context
  | `BracketAggregate fields -> fields.context
  | `DeltaAggregate fields -> fields.context
  | `BracketDeltaAggregate fields -> fields.context
  | `NullRecordAggregate fields -> fields.context
  | `BinOp fields -> fields.context
  | `RelationOp fields -> fields.context
  | `BoxExpr fields -> fields.context
  | `CaseExprAlternative fields -> fields.context
  | `ConcatOp fields -> fields.context
  | `ConcatOperand fields -> fields.context
  | `CaseExpr fields -> fields.context
  | `IfExpr fields -> fields.context
  | `ContractCases fields -> fields.context
  | `DeclExpr fields -> fields.context
  | `MembershipExpr fields -> fields.context
  | `AttributeRef fields -> fields.context
  | `CallExpr fields -> fields.context
  | `DefiningName fields -> fields.context
  | `SyntheticDefiningName fields -> fields.context
  | `DiscreteSubtypeName fields -> fields.context
  | `DottedName fields -> fields.context
  | `EndName fields -> fields.context
  | `ExplicitDeref fields -> fields.context
  | `QualExpr fields -> fields.context
  | `ReduceAttributeRef fields -> fields.context
  | `CharLiteral fields -> fields.context
  | `Identifier fields -> fields.context
  | `OpAbs fields -> fields.context
  | `OpAnd fields -> fields.context
  | `OpAndThen fields -> fields.context
  | `OpConcat fields -> fields.context
  | `OpDiv fields -> fields.context
  | `OpDoubleDot fields -> fields.context
  | `OpEq fields -> fields.context
  | `OpGt fields -> fields.context
  | `OpGte fields -> fields.context
  | `OpIn fields -> fields.context
  | `OpLt fields -> fields.context
  | `OpLte fields -> fields.context
  | `OpMinus fields -> fields.context
  | `OpMod fields -> fields.context
  | `OpMult fields -> fields.context
  | `OpNeq fields -> fields.context
  | `OpNot fields -> fields.context
  | `OpNotIn fields -> fields.context
  | `OpOr fields -> fields.context
  | `OpOrElse fields -> fields.context
  | `OpPlus fields -> fields.context
  | `OpPow fields -> fields.context
  | `OpRem fields -> fields.context
  | `OpXor fields -> fields.context
  | `StringLiteral fields -> fields.context
  | `NullLiteral fields -> fields.context
  | `IntLiteral fields -> fields.context
  | `RealLiteral fields -> fields.context
  | `SyntheticIdentifier fields -> fields.context
  | `TargetName fields -> fields.context
  | `UpdateAttributeRef fields -> fields.context
  | `ParenExpr fields -> fields.context
  | `QuantifiedExpr fields -> fields.context
  | `RaiseExpr fields -> fields.context
  | `UnOp fields -> fields.context
  | `HandledStmts fields -> fields.context
  | `InterfaceKindLimited fields -> fields.context
  | `InterfaceKindProtected fields -> fields.context
  | `InterfaceKindSynchronized fields -> fields.context
  | `InterfaceKindTask fields -> fields.context
  | `IterTypeIn fields -> fields.context
  | `IterTypeOf fields -> fields.context
  | `LibraryItem fields -> fields.context
  | `LimitedAbsent fields -> fields.context
  | `LimitedPresent fields -> fields.context
  | `ForLoopSpec fields -> fields.context
  | `WhileLoopSpec fields -> fields.context
  | `ModeDefault fields -> fields.context
  | `ModeIn fields -> fields.context
  | `ModeInOut fields -> fields.context
  | `ModeOut fields -> fields.context
  | `MultiAbstractStateDecl fields -> fields.context
  | `NotNullAbsent fields -> fields.context
  | `NotNullPresent fields -> fields.context
  | `NullComponentDecl fields -> fields.context
  | `OthersDesignator fields -> fields.context
  | `OverridingNotOverriding fields -> fields.context
  | `OverridingOverriding fields -> fields.context
  | `OverridingUnspecified fields -> fields.context
  | `Params fields -> fields.context
  | `ParenAbstractStateDecl fields -> fields.context
  | `PpElseDirective fields -> fields.context
  | `PpElsifDirective fields -> fields.context
  | `PpEndIfDirective fields -> fields.context
  | `PpIfDirective fields -> fields.context
  | `PpThenKw fields -> fields.context
  | `PragmaNode fields -> fields.context
  | `PrivateAbsent fields -> fields.context
  | `PrivatePresent fields -> fields.context
  | `ProtectedDef fields -> fields.context
  | `ProtectedAbsent fields -> fields.context
  | `ProtectedPresent fields -> fields.context
  | `QuantifierAll fields -> fields.context
  | `QuantifierSome fields -> fields.context
  | `RangeSpec fields -> fields.context
  | `RenamingClause fields -> fields.context
  | `SyntheticRenamingClause fields -> fields.context
  | `ReverseAbsent fields -> fields.context
  | `ReversePresent fields -> fields.context
  | `SelectWhenPart fields -> fields.context
  | `AcceptStmt fields -> fields.context
  | `AcceptStmtWithStmts fields -> fields.context
  | `ForLoopStmt fields -> fields.context
  | `LoopStmt fields -> fields.context
  | `WhileLoopStmt fields -> fields.context
  | `BeginBlock fields -> fields.context
  | `DeclBlock fields -> fields.context
  | `CaseStmt fields -> fields.context
  | `ExtendedReturnStmt fields -> fields.context
  | `IfStmt fields -> fields.context
  | `NamedStmt fields -> fields.context
  | `SelectStmt fields -> fields.context
  | `ErrorStmt fields -> fields.context
  | `AbortStmt fields -> fields.context
  | `AssignStmt fields -> fields.context
  | `CallStmt fields -> fields.context
  | `DelayStmt fields -> fields.context
  | `ExitStmt fields -> fields.context
  | `GotoStmt fields -> fields.context
  | `Label fields -> fields.context
  | `NullStmt fields -> fields.context
  | `RaiseStmt fields -> fields.context
  | `RequeueStmt fields -> fields.context
  | `ReturnStmt fields -> fields.context
  | `TerminateAlternative fields -> fields.context
  | `SubpKindFunction fields -> fields.context
  | `SubpKindProcedure fields -> fields.context
  | `Subunit fields -> fields.context
  | `SynchronizedAbsent fields -> fields.context
  | `SynchronizedPresent fields -> fields.context
  | `TaggedAbsent fields -> fields.context
  | `TaggedPresent fields -> fields.context
  | `TaskDef fields -> fields.context
  | `TypeAttributesRepository fields -> fields.context
  | `AccessToSubpDef fields -> fields.context
  | `AnonymousTypeAccessDef fields -> fields.context
  | `TypeAccessDef fields -> fields.context
  | `ArrayTypeDef fields -> fields.context
  | `DerivedTypeDef fields -> fields.context
  | `EnumTypeDef fields -> fields.context
  | `FormalDiscreteTypeDef fields -> fields.context
  | `InterfaceTypeDef fields -> fields.context
  | `ModIntTypeDef fields -> fields.context
  | `PrivateTypeDef fields -> fields.context
  | `DecimalFixedPointDef fields -> fields.context
  | `FloatingPointDef fields -> fields.context
  | `OrdinaryFixedPointDef fields -> fields.context
  | `RecordTypeDef fields -> fields.context
  | `SignedIntTypeDef fields -> fields.context
  | `AnonymousType fields -> fields.context
  | `EnumLitSynthTypeExpr fields -> fields.context
  | `SubtypeIndication fields -> fields.context
  | `ConstrainedSubtypeIndication fields -> fields.context
  | `DiscreteSubtypeIndication fields -> fields.context
  | `SyntheticTypeExpr fields -> fields.context
  | `UnconstrainedArrayIndex fields -> fields.context
  | `UntilAbsent fields -> fields.context
  | `UntilPresent fields -> fields.context
  | `UsePackageClause fields -> fields.context
  | `UseTypeClause fields -> fields.context
  | `ValueSequence fields -> fields.context
  | `Variant fields -> fields.context
  | `VariantPart fields -> fields.context
  | `WithClause fields -> fields.context
  | `WithPrivateAbsent fields -> fields.context
  | `WithPrivatePresent fields -> fields.context

type _ node =
  | AdaNode :
      ada_node node
  | AbortNode :
      abort_node node
  | AbortAbsent :
      abort_absent node
  | AbortPresent :
      abort_present node
  | AbstractNode :
      abstract_node node
  | AbstractAbsent :
      abstract_absent node
  | AbstractPresent :
      abstract_present node
  | AdaList :
      ada_list node
  | AdaNodeList :
      ada_node_list node
  | AbstractStateDeclList :
      abstract_state_decl_list node
  | AlternativesList :
      alternatives_list node
  | ConstraintList :
      constraint_list node
  | DeclList :
      decl_list node
  | StmtList :
      stmt_list node
  | AspectAssocList :
      aspect_assoc_list node
  | BaseAssocList :
      base_assoc_list node
  | BasicAssocList :
      basic_assoc_list node
  | AssocList :
      assoc_list node
  | BasicDeclList :
      basic_decl_list node
  | CaseExprAlternativeList :
      case_expr_alternative_list node
  | CaseStmtAlternativeList :
      case_stmt_alternative_list node
  | CompilationUnitList :
      compilation_unit_list node
  | ConcatOperandList :
      concat_operand_list node
  | ContractCaseAssocList :
      contract_case_assoc_list node
  | DefiningNameList :
      defining_name_list node
  | DiscriminantSpecList :
      discriminant_spec_list node
  | ElsifExprPartList :
      elsif_expr_part_list node
  | ElsifStmtPartList :
      elsif_stmt_part_list node
  | EnumLiteralDeclList :
      enum_literal_decl_list node
  | ExprList :
      expr_list node
  | ExprAlternativesList :
      expr_alternatives_list node
  | IdentifierList :
      identifier_list node
  | DiscriminantChoiceList :
      discriminant_choice_list node
  | NameList :
      name_list node
  | ParentList :
      parent_list node
  | ParamSpecList :
      param_spec_list node
  | PragmaNodeList :
      pragma_node_list node
  | SelectWhenPartList :
      select_when_part_list node
  | UnconstrainedArrayIndexList :
      unconstrained_array_index_list node
  | VariantList :
      variant_list node
  | AliasedNode :
      aliased_node node
  | AliasedAbsent :
      aliased_absent node
  | AliasedPresent :
      aliased_present node
  | AllNode :
      all_node node
  | AllAbsent :
      all_absent node
  | AllPresent :
      all_present node
  | ArrayIndices :
      array_indices node
  | ConstrainedArrayIndices :
      constrained_array_indices node
  | UnconstrainedArrayIndices :
      unconstrained_array_indices node
  | AspectAssoc :
      aspect_assoc node
  | AspectClause :
      aspect_clause node
  | AtClause :
      at_clause node
  | AttributeDefClause :
      attribute_def_clause node
  | EnumRepClause :
      enum_rep_clause node
  | RecordRepClause :
      record_rep_clause node
  | AspectSpec :
      aspect_spec node
  | BaseAssoc :
      base_assoc node
  | ContractCaseAssoc :
      contract_case_assoc node
  | PragmaArgumentAssoc :
      pragma_argument_assoc node
  | BaseFormalParamHolder :
      base_formal_param_holder node
  | BaseSubpSpec :
      base_subp_spec node
  | EntrySpec :
      entry_spec node
  | EnumSubpSpec :
      enum_subp_spec node
  | SubpSpec :
      subp_spec node
  | SyntheticBinarySpec :
      synthetic_binary_spec node
  | SyntheticUnarySpec :
      synthetic_unary_spec node
  | ComponentList :
      component_list node
  | DiscriminantPart :
      discriminant_part node
  | KnownDiscriminantPart :
      known_discriminant_part node
  | UnknownDiscriminantPart :
      unknown_discriminant_part node
  | EntryCompletionFormalParams :
      entry_completion_formal_params node
  | GenericFormalPart :
      generic_formal_part node
  | BaseRecordDef :
      base_record_def node
  | NullRecordDef :
      null_record_def node
  | RecordDef :
      record_def node
  | BasicAssoc :
      basic_assoc node
  | AggregateAssoc :
      aggregate_assoc node
  | MultiDimArrayAssoc :
      multi_dim_array_assoc node
  | CompositeConstraintAssoc :
      composite_constraint_assoc node
  | IteratedAssoc :
      iterated_assoc node
  | ParamAssoc :
      param_assoc node
  | BasicDecl :
      basic_decl node
  | AbstractStateDecl :
      abstract_state_decl node
  | AnonymousExprDecl :
      anonymous_expr_decl node
  | BaseFormalParamDecl :
      base_formal_param_decl node
  | ComponentDecl :
      component_decl node
  | DiscriminantSpec :
      discriminant_spec node
  | GenericFormal :
      generic_formal node
  | GenericFormalObjDecl :
      generic_formal_obj_decl node
  | GenericFormalPackage :
      generic_formal_package node
  | GenericFormalSubpDecl :
      generic_formal_subp_decl node
  | GenericFormalTypeDecl :
      generic_formal_type_decl node
  | ParamSpec :
      param_spec node
  | SyntheticFormalParamDecl :
      synthetic_formal_param_decl node
  | BasePackageDecl :
      base_package_decl node
  | GenericPackageInternal :
      generic_package_internal node
  | PackageDecl :
      package_decl node
  | BaseTypeDecl :
      base_type_decl node
  | BaseSubtypeDecl :
      base_subtype_decl node
  | DiscreteBaseSubtypeDecl :
      discrete_base_subtype_decl node
  | SubtypeDecl :
      subtype_decl node
  | ClasswideTypeDecl :
      classwide_type_decl node
  | IncompleteTypeDecl :
      incomplete_type_decl node
  | IncompleteFormalTypeDecl :
      incomplete_formal_type_decl node
  | IncompleteTaggedTypeDecl :
      incomplete_tagged_type_decl node
  | ProtectedTypeDecl :
      protected_type_decl node
  | TaskTypeDecl :
      task_type_decl node
  | SingleTaskTypeDecl :
      single_task_type_decl node
  | TypeDecl :
      type_decl node
  | AnonymousTypeDecl :
      anonymous_type_decl node
  | SynthAnonymousTypeDecl :
      synth_anonymous_type_decl node
  | ConcreteTypeDecl :
      concrete_type_decl node
  | FormalTypeDecl :
      formal_type_decl node
  | BasicSubpDecl :
      basic_subp_decl node
  | ClassicSubpDecl :
      classic_subp_decl node
  | AbstractSubpDecl :
      abstract_subp_decl node
  | FormalSubpDecl :
      formal_subp_decl node
  | AbstractFormalSubpDecl :
      abstract_formal_subp_decl node
  | ConcreteFormalSubpDecl :
      concrete_formal_subp_decl node
  | SubpDecl :
      subp_decl node
  | EntryDecl :
      entry_decl node
  | EnumLiteralDecl :
      enum_literal_decl node
  | SyntheticCharEnumLit :
      synthetic_char_enum_lit node
  | GenericSubpInternal :
      generic_subp_internal node
  | SyntheticSubpDecl :
      synthetic_subp_decl node
  | BodyNode :
      body_node node
  | AcceptStmtBody :
      accept_stmt_body node
  | BaseSubpBody :
      base_subp_body node
  | ExprFunction :
      expr_function node
  | NullSubpDecl :
      null_subp_decl node
  | SubpBody :
      subp_body node
  | SubpRenamingDecl :
      subp_renaming_decl node
  | BodyStub :
      body_stub node
  | PackageBodyStub :
      package_body_stub node
  | ProtectedBodyStub :
      protected_body_stub node
  | SubpBodyStub :
      subp_body_stub node
  | TaskBodyStub :
      task_body_stub node
  | EntryBody :
      entry_body node
  | PackageBody :
      package_body node
  | ProtectedBody :
      protected_body node
  | TaskBody :
      task_body node
  | EntryIndexSpec :
      entry_index_spec node
  | ErrorDecl :
      error_decl node
  | ExceptionDecl :
      exception_decl node
  | ExceptionHandler :
      exception_handler node
  | ForLoopVarDecl :
      for_loop_var_decl node
  | GenericDecl :
      generic_decl node
  | GenericPackageDecl :
      generic_package_decl node
  | GenericSubpDecl :
      generic_subp_decl node
  | GenericInstantiation :
      generic_instantiation node
  | GenericPackageInstantiation :
      generic_package_instantiation node
  | GenericSubpInstantiation :
      generic_subp_instantiation node
  | GenericRenamingDecl :
      generic_renaming_decl node
  | GenericPackageRenamingDecl :
      generic_package_renaming_decl node
  | GenericSubpRenamingDecl :
      generic_subp_renaming_decl node
  | LabelDecl :
      label_decl node
  | NamedStmtDecl :
      named_stmt_decl node
  | NumberDecl :
      number_decl node
  | ObjectDecl :
      object_decl node
  | ExtendedReturnStmtObjectDecl :
      extended_return_stmt_object_decl node
  | NoTypeObjectRenamingDecl :
      no_type_object_renaming_decl node
  | PackageRenamingDecl :
      package_renaming_decl node
  | SingleProtectedDecl :
      single_protected_decl node
  | SingleTaskDecl :
      single_task_decl node
  | CaseStmtAlternative :
      case_stmt_alternative node
  | CompilationUnit :
      compilation_unit node
  | ComponentClause :
      component_clause node
  | ComponentDef :
      component_def node
  | ConstantNode :
      constant_node node
  | ConstantAbsent :
      constant_absent node
  | ConstantPresent :
      constant_present node
  | Constraint :
      constraint_node node
  | CompositeConstraint :
      composite_constraint node
  | DeltaConstraint :
      delta_constraint node
  | DigitsConstraint :
      digits_constraint node
  | RangeConstraint :
      range_constraint node
  | DeclarativePart :
      declarative_part node
  | PrivatePart :
      private_part node
  | PublicPart :
      public_part node
  | ElsifExprPart :
      elsif_expr_part node
  | ElsifStmtPart :
      elsif_stmt_part node
  | Expr :
      expr node
  | AbstractStateDeclExpr :
      abstract_state_decl_expr node
  | Allocator :
      allocator node
  | BaseAggregate :
      base_aggregate node
  | Aggregate :
      aggregate node
  | BracketAggregate :
      bracket_aggregate node
  | DeltaAggregate :
      delta_aggregate node
  | BracketDeltaAggregate :
      bracket_delta_aggregate node
  | NullRecordAggregate :
      null_record_aggregate node
  | BinOp :
      bin_op node
  | RelationOp :
      relation_op node
  | BoxExpr :
      box_expr node
  | CaseExprAlternative :
      case_expr_alternative node
  | ConcatOp :
      concat_op node
  | ConcatOperand :
      concat_operand node
  | CondExpr :
      cond_expr node
  | CaseExpr :
      case_expr node
  | IfExpr :
      if_expr node
  | ContractCases :
      contract_cases node
  | DeclExpr :
      decl_expr node
  | MembershipExpr :
      membership_expr node
  | Name :
      name node
  | AttributeRef :
      attribute_ref node
  | CallExpr :
      call_expr node
  | DefiningName :
      defining_name node
  | SyntheticDefiningName :
      synthetic_defining_name node
  | DiscreteSubtypeName :
      discrete_subtype_name node
  | DottedName :
      dotted_name node
  | EndName :
      end_name node
  | ExplicitDeref :
      explicit_deref node
  | QualExpr :
      qual_expr node
  | ReduceAttributeRef :
      reduce_attribute_ref node
  | SingleTokNode :
      single_tok_node node
  | BaseId :
      base_id node
  | CharLiteral :
      char_literal node
  | Identifier :
      identifier node
  | Op :
      op node
  | OpAbs :
      op_abs node
  | OpAnd :
      op_and node
  | OpAndThen :
      op_and_then node
  | OpConcat :
      op_concat node
  | OpDiv :
      op_div node
  | OpDoubleDot :
      op_double_dot node
  | OpEq :
      op_eq node
  | OpGt :
      op_gt node
  | OpGte :
      op_gte node
  | OpIn :
      op_in node
  | OpLt :
      op_lt node
  | OpLte :
      op_lte node
  | OpMinus :
      op_minus node
  | OpMod :
      op_mod node
  | OpMult :
      op_mult node
  | OpNeq :
      op_neq node
  | OpNot :
      op_not node
  | OpNotIn :
      op_not_in node
  | OpOr :
      op_or node
  | OpOrElse :
      op_or_else node
  | OpPlus :
      op_plus node
  | OpPow :
      op_pow node
  | OpRem :
      op_rem node
  | OpXor :
      op_xor node
  | StringLiteral :
      string_literal node
  | NullLiteral :
      null_literal node
  | NumLiteral :
      num_literal node
  | IntLiteral :
      int_literal node
  | RealLiteral :
      real_literal node
  | SyntheticIdentifier :
      synthetic_identifier node
  | TargetName :
      target_name node
  | UpdateAttributeRef :
      update_attribute_ref node
  | ParenExpr :
      paren_expr node
  | QuantifiedExpr :
      quantified_expr node
  | RaiseExpr :
      raise_expr node
  | UnOp :
      un_op node
  | HandledStmts :
      handled_stmts node
  | InterfaceKind :
      interface_kind node
  | InterfaceKindLimited :
      interface_kind_limited node
  | InterfaceKindProtected :
      interface_kind_protected node
  | InterfaceKindSynchronized :
      interface_kind_synchronized node
  | InterfaceKindTask :
      interface_kind_task node
  | IterType :
      iter_type node
  | IterTypeIn :
      iter_type_in node
  | IterTypeOf :
      iter_type_of node
  | LibraryItem :
      library_item node
  | LimitedNode :
      limited_node node
  | LimitedAbsent :
      limited_absent node
  | LimitedPresent :
      limited_present node
  | LoopSpec :
      loop_spec node
  | ForLoopSpec :
      for_loop_spec node
  | WhileLoopSpec :
      while_loop_spec node
  | Mode :
      mode node
  | ModeDefault :
      mode_default node
  | ModeIn :
      mode_in node
  | ModeInOut :
      mode_in_out node
  | ModeOut :
      mode_out node
  | MultiAbstractStateDecl :
      multi_abstract_state_decl node
  | NotNull :
      not_null node
  | NotNullAbsent :
      not_null_absent node
  | NotNullPresent :
      not_null_present node
  | NullComponentDecl :
      null_component_decl node
  | OthersDesignator :
      others_designator node
  | OverridingNode :
      overriding_node node
  | OverridingNotOverriding :
      overriding_not_overriding node
  | OverridingOverriding :
      overriding_overriding node
  | OverridingUnspecified :
      overriding_unspecified node
  | Params :
      params node
  | ParenAbstractStateDecl :
      paren_abstract_state_decl node
  | PpDirective :
      pp_directive node
  | PpElseDirective :
      pp_else_directive node
  | PpElsifDirective :
      pp_elsif_directive node
  | PpEndIfDirective :
      pp_end_if_directive node
  | PpIfDirective :
      pp_if_directive node
  | PpThenKw :
      pp_then_kw node
  | PragmaNode :
      pragma_node node
  | PrivateNode :
      private_node node
  | PrivateAbsent :
      private_absent node
  | PrivatePresent :
      private_present node
  | ProtectedDef :
      protected_def node
  | ProtectedNode :
      protected_node node
  | ProtectedAbsent :
      protected_absent node
  | ProtectedPresent :
      protected_present node
  | Quantifier :
      quantifier node
  | QuantifierAll :
      quantifier_all node
  | QuantifierSome :
      quantifier_some node
  | RangeSpec :
      range_spec node
  | RenamingClause :
      renaming_clause node
  | SyntheticRenamingClause :
      synthetic_renaming_clause node
  | ReverseNode :
      reverse_node node
  | ReverseAbsent :
      reverse_absent node
  | ReversePresent :
      reverse_present node
  | SelectWhenPart :
      select_when_part node
  | Stmt :
      stmt node
  | CompositeStmt :
      composite_stmt node
  | AcceptStmt :
      accept_stmt node
  | AcceptStmtWithStmts :
      accept_stmt_with_stmts node
  | BaseLoopStmt :
      base_loop_stmt node
  | ForLoopStmt :
      for_loop_stmt node
  | LoopStmt :
      loop_stmt node
  | WhileLoopStmt :
      while_loop_stmt node
  | BlockStmt :
      block_stmt node
  | BeginBlock :
      begin_block node
  | DeclBlock :
      decl_block node
  | CaseStmt :
      case_stmt node
  | ExtendedReturnStmt :
      extended_return_stmt node
  | IfStmt :
      if_stmt node
  | NamedStmt :
      named_stmt node
  | SelectStmt :
      select_stmt node
  | ErrorStmt :
      error_stmt node
  | SimpleStmt :
      simple_stmt node
  | AbortStmt :
      abort_stmt node
  | AssignStmt :
      assign_stmt node
  | CallStmt :
      call_stmt node
  | DelayStmt :
      delay_stmt node
  | ExitStmt :
      exit_stmt node
  | GotoStmt :
      goto_stmt node
  | Label :
      label node
  | NullStmt :
      null_stmt node
  | RaiseStmt :
      raise_stmt node
  | RequeueStmt :
      requeue_stmt node
  | ReturnStmt :
      return_stmt node
  | TerminateAlternative :
      terminate_alternative node
  | SubpKind :
      subp_kind node
  | SubpKindFunction :
      subp_kind_function node
  | SubpKindProcedure :
      subp_kind_procedure node
  | Subunit :
      subunit node
  | SynchronizedNode :
      synchronized_node node
  | SynchronizedAbsent :
      synchronized_absent node
  | SynchronizedPresent :
      synchronized_present node
  | TaggedNode :
      tagged_node node
  | TaggedAbsent :
      tagged_absent node
  | TaggedPresent :
      tagged_present node
  | TaskDef :
      task_def node
  | TypeAttributesRepository :
      type_attributes_repository node
  | TypeDef :
      type_def node
  | AccessDef :
      access_def node
  | AccessToSubpDef :
      access_to_subp_def node
  | BaseTypeAccessDef :
      base_type_access_def node
  | AnonymousTypeAccessDef :
      anonymous_type_access_def node
  | TypeAccessDef :
      type_access_def node
  | ArrayTypeDef :
      array_type_def node
  | DerivedTypeDef :
      derived_type_def node
  | EnumTypeDef :
      enum_type_def node
  | FormalDiscreteTypeDef :
      formal_discrete_type_def node
  | InterfaceTypeDef :
      interface_type_def node
  | ModIntTypeDef :
      mod_int_type_def node
  | PrivateTypeDef :
      private_type_def node
  | RealTypeDef :
      real_type_def node
  | DecimalFixedPointDef :
      decimal_fixed_point_def node
  | FloatingPointDef :
      floating_point_def node
  | OrdinaryFixedPointDef :
      ordinary_fixed_point_def node
  | RecordTypeDef :
      record_type_def node
  | SignedIntTypeDef :
      signed_int_type_def node
  | TypeExpr :
      type_expr node
  | AnonymousType :
      anonymous_type node
  | EnumLitSynthTypeExpr :
      enum_lit_synth_type_expr node
  | SubtypeIndication :
      subtype_indication node
  | ConstrainedSubtypeIndication :
      constrained_subtype_indication node
  | DiscreteSubtypeIndication :
      discrete_subtype_indication node
  | SyntheticTypeExpr :
      synthetic_type_expr node
  | UnconstrainedArrayIndex :
      unconstrained_array_index node
  | UntilNode :
      until_node node
  | UntilAbsent :
      until_absent node
  | UntilPresent :
      until_present node
  | UseClause :
      use_clause node
  | UsePackageClause :
      use_package_clause node
  | UseTypeClause :
      use_type_clause node
  | ValueSequence :
      value_sequence node
  | Variant :
      variant node
  | VariantPart :
      variant_part node
  | WithClause :
      with_clause node
  | WithPrivate :
      with_private node
  | WithPrivateAbsent :
      with_private_absent node
  | WithPrivatePresent :
      with_private_present node

module WithPrivatePresent = struct
  type t =
    [
      | `WithPrivatePresent of
          with_private_present_fields
    ]

  type fields = with_private_present_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))





end

module WithPrivateAbsent = struct
  type t =
    [
      | `WithPrivateAbsent of
          with_private_absent_fields
    ]

  type fields = with_private_absent_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))





end

module WithPrivate = struct
  type t =
    [
      | WithPrivateAbsent.t
      | WithPrivatePresent.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))

let p_as_bool
    (node)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.with_private_p_as_bool
          (addr (unwrap_ada_node (node)))
          (result_ptr)
      in
      !@ result_ptr





end

module WithClause = struct
  type t =
    [
      | `WithClause of
          with_clause_fields
    ]

  type fields = with_clause_fields =
    
  {
         
    f_has_limited: limited_node
    Lazy.t;
         
    f_has_private: private_node
    Lazy.t;
         
    f_packages: name_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_has_limited node =
    match (node :> with_clause) with
    | `WithClause fields ->
        Lazy.force fields.f_has_limited
  let f_has_private node =
    match (node :> with_clause) with
    | `WithClause fields ->
        Lazy.force fields.f_has_private
  let f_packages node =
    match (node :> with_clause) with
    | `WithClause fields ->
        Lazy.force fields.f_packages



end

module VariantPart = struct
  type t =
    [
      | `VariantPart of
          variant_part_fields
    ]

  type fields = variant_part_fields =
    
  {
         
    f_discr_name: identifier
    Lazy.t;
         
    f_variant: variant_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_discr_name node =
    match (node :> variant_part) with
    | `VariantPart fields ->
        Lazy.force fields.f_discr_name
  let f_variant node =
    match (node :> variant_part) with
    | `VariantPart fields ->
        Lazy.force fields.f_variant



end

module Variant = struct
  type t =
    [
      | `Variant of
          variant_fields
    ]

  type fields = variant_fields =
    
  {
         
    f_choices: alternatives_list
    Lazy.t;
         
    f_components: component_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_choices node =
    match (node :> variant) with
    | `Variant fields ->
        Lazy.force fields.f_choices
  let f_components node =
    match (node :> variant) with
    | `Variant fields ->
        Lazy.force fields.f_components



end

module ValueSequence = struct
  type t =
    [
      | `ValueSequence of
          value_sequence_fields
    ]

  type fields = value_sequence_fields =
    
  {
         
    f_iter_assoc: iterated_assoc
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_iter_assoc node =
    match (node :> value_sequence) with
    | `ValueSequence fields ->
        Lazy.force fields.f_iter_assoc



end

module UseTypeClause = struct
  type t =
    [
      | `UseTypeClause of
          use_type_clause_fields
    ]

  type fields = use_type_clause_fields =
    
  {
         
    f_has_all: all_node
    Lazy.t;
         
    f_types: name_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_has_all node =
    match (node :> use_type_clause) with
    | `UseTypeClause fields ->
        Lazy.force fields.f_has_all
  let f_types node =
    match (node :> use_type_clause) with
    | `UseTypeClause fields ->
        Lazy.force fields.f_types



end

module UsePackageClause = struct
  type t =
    [
      | `UsePackageClause of
          use_package_clause_fields
    ]

  type fields = use_package_clause_fields =
    
  {
         
    f_packages: name_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_packages node =
    match (node :> use_package_clause) with
    | `UsePackageClause fields ->
        Lazy.force fields.f_packages



end

module UseClause = struct
  type t =
    [
      | UsePackageClause.t
      | UseTypeClause.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))





end

module UntilPresent = struct
  type t =
    [
      | `UntilPresent of
          until_present_fields
    ]

  type fields = until_present_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))





end

module UntilAbsent = struct
  type t =
    [
      | `UntilAbsent of
          until_absent_fields
    ]

  type fields = until_absent_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))





end

module UntilNode = struct
  type t =
    [
      | UntilAbsent.t
      | UntilPresent.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))

let p_as_bool
    (node)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.until_node_p_as_bool
          (addr (unwrap_ada_node (node)))
          (result_ptr)
      in
      !@ result_ptr





end

module UnconstrainedArrayIndex = struct
  type t =
    [
      | `UnconstrainedArrayIndex of
          unconstrained_array_index_fields
    ]

  type fields = unconstrained_array_index_fields =
    
  {
         
    f_subtype_indication: subtype_indication
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_subtype_indication node =
    match (node :> unconstrained_array_index) with
    | `UnconstrainedArrayIndex fields ->
        Lazy.force fields.f_subtype_indication



end

module SyntheticTypeExpr = struct
  type t =
    [
      | `SyntheticTypeExpr of
          synthetic_type_expr_fields
    ]

  type fields = synthetic_type_expr_fields =
    
  {
         
    f_target_type: base_type_decl
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_target_type node =
    match (node :> synthetic_type_expr) with
    | `SyntheticTypeExpr fields ->
        Lazy.force fields.f_target_type



end

module DiscreteSubtypeIndication = struct
  type t =
    [
      | `DiscreteSubtypeIndication of
          discrete_subtype_indication_fields
    ]

  type fields = discrete_subtype_indication_fields =
    
  {
         
    f_has_not_null: not_null
    Lazy.t;
         
    f_name: [
      | `AttributeRef
          of attribute_ref_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
         
    f_constraint: constraint_node
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_has_not_null node =
    match (node :> discrete_subtype_indication) with
    | `DiscreteSubtypeIndication fields ->
        Lazy.force fields.f_has_not_null
  let f_name node =
    match (node :> discrete_subtype_indication) with
    | `DiscreteSubtypeIndication fields ->
        Lazy.force fields.f_name
  let f_constraint node =
    match (node :> discrete_subtype_indication) with
    | `DiscreteSubtypeIndication fields ->
        Lazy.force fields.f_constraint



end

module ConstrainedSubtypeIndication = struct
  type t =
    [
      | `ConstrainedSubtypeIndication of
          constrained_subtype_indication_fields
    ]

  type fields = constrained_subtype_indication_fields =
    
  {
         
    f_has_not_null: not_null
    Lazy.t;
         
    f_name: [
      | `AttributeRef
          of attribute_ref_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
         
    f_constraint: constraint_node
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_has_not_null node =
    match (node :> constrained_subtype_indication) with
    | `ConstrainedSubtypeIndication fields ->
        Lazy.force fields.f_has_not_null
  let f_name node =
    match (node :> constrained_subtype_indication) with
    | `ConstrainedSubtypeIndication fields ->
        Lazy.force fields.f_name
  let f_constraint node =
    match (node :> constrained_subtype_indication) with
    | `ConstrainedSubtypeIndication fields ->
        Lazy.force fields.f_constraint



end

module SubtypeIndication = struct
  type t =
    [
      | `SubtypeIndication of
          subtype_indication_fields
      | `ConstrainedSubtypeIndication of
          constrained_subtype_indication_fields
      | `DiscreteSubtypeIndication of
          discrete_subtype_indication_fields
    ]

  type fields = subtype_indication_fields =
    
  {
         
    f_has_not_null: not_null
    Lazy.t;
         
    f_name: [
      | `AttributeRef
          of attribute_ref_fields
      | `CharLiteral
          of char_literal_fields
      | `DottedName
          of dotted_name_fields
      | `Identifier
          of identifier_fields
      | `StringLiteral
          of string_literal_fields
    ]
    Lazy.t;
         
    f_constraint: constraint_node
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))

let p_subtype_constraints
    (node)
    =
      let result_ptr =
        allocate_n ParamActualArrayStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.subtype_indication_p_subtype_constraints
          (addr (unwrap_ada_node (node)))
          (result_ptr)
      in
      ParamActualArray.wrap (context node) (!@ result_ptr)

let p_is_static_subtype
    ?(imprecise_fallback=false)
    (node)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let c_imprecise_fallback =
            
        imprecise_fallback
      in
      let _ : int =
        CFunctions.subtype_indication_p_is_static_subtype
          (addr (unwrap_ada_node (node)))
          c_imprecise_fallback
          (result_ptr)
      in
         
      !@ result_ptr


  let f_has_not_null node =
    match (node :> subtype_indication) with
    | `SubtypeIndication fields ->
        Lazy.force fields.f_has_not_null
    | `ConstrainedSubtypeIndication fields ->
        Lazy.force fields.f_has_not_null
    | `DiscreteSubtypeIndication fields ->
        Lazy.force fields.f_has_not_null
  let f_name node =
    match (node :> subtype_indication) with
    | `SubtypeIndication fields ->
        Lazy.force fields.f_name
    | `ConstrainedSubtypeIndication fields ->
        Lazy.force fields.f_name
    | `DiscreteSubtypeIndication fields ->
        Lazy.force fields.f_name
  let f_constraint node =
    match (node :> subtype_indication) with
    | `SubtypeIndication fields ->
        Lazy.force fields.f_constraint
    | `ConstrainedSubtypeIndication fields ->
        Lazy.force fields.f_constraint
    | `DiscreteSubtypeIndication fields ->
        Lazy.force fields.f_constraint



end

module EnumLitSynthTypeExpr = struct
  type t =
    [
      | `EnumLitSynthTypeExpr of
          enum_lit_synth_type_expr_fields
    ]

  type fields = enum_lit_synth_type_expr_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))





end

module AnonymousType = struct
  type t =
    [
      | `AnonymousType of
          anonymous_type_fields
    ]

  type fields = anonymous_type_fields =
    
  {
         
    f_type_decl: anonymous_type_decl
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_type_decl node =
    match (node :> anonymous_type) with
    | `AnonymousType fields ->
        Lazy.force fields.f_type_decl



end

module TypeExpr = struct
  type t =
    [
      | AnonymousType.t
      | EnumLitSynthTypeExpr.t
      | SubtypeIndication.t
      | SyntheticTypeExpr.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))

let p_type_name
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.type_expr_p_type_name
          (addr (unwrap_ada_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_name (context node) (!@ result_ptr))

let p_designated_type_decl
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.type_expr_p_designated_type_decl
          (addr (unwrap_ada_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_base_type_decl (context node) (!@ result_ptr))

let p_designated_type_decl_from
    (node)
    (origin_node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let c_origin_node =
            
        addr (unwrap_ada_node (origin_node))
      in
      let _ : int =
        CFunctions.type_expr_p_designated_type_decl_from
          (addr (unwrap_ada_node (node)))
          c_origin_node
          (result_ptr)
      in
         
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_base_type_decl (context node) (!@ result_ptr))





end

module SignedIntTypeDef = struct
  type t =
    [
      | `SignedIntTypeDef of
          signed_int_type_def_fields
    ]

  type fields = signed_int_type_def_fields =
    
  {
         
    f_range: range_spec
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_range node =
    match (node :> signed_int_type_def) with
    | `SignedIntTypeDef fields ->
        Lazy.force fields.f_range



end

module RecordTypeDef = struct
  type t =
    [
      | `RecordTypeDef of
          record_type_def_fields
    ]

  type fields = record_type_def_fields =
    
  {
         
    f_has_abstract: abstract_node
    Lazy.t;
         
    f_has_tagged: tagged_node
    Lazy.t;
         
    f_has_limited: limited_node
    Lazy.t;
         
    f_record_def: base_record_def
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_has_abstract node =
    match (node :> record_type_def) with
    | `RecordTypeDef fields ->
        Lazy.force fields.f_has_abstract
  let f_has_tagged node =
    match (node :> record_type_def) with
    | `RecordTypeDef fields ->
        Lazy.force fields.f_has_tagged
  let f_has_limited node =
    match (node :> record_type_def) with
    | `RecordTypeDef fields ->
        Lazy.force fields.f_has_limited
  let f_record_def node =
    match (node :> record_type_def) with
    | `RecordTypeDef fields ->
        Lazy.force fields.f_record_def



end

module OrdinaryFixedPointDef = struct
  type t =
    [
      | `OrdinaryFixedPointDef of
          ordinary_fixed_point_def_fields
    ]

  type fields = ordinary_fixed_point_def_fields =
    
  {
         
    f_delta: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BoxExpr
          of box_expr_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_range: range_spec
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_delta node =
    match (node :> ordinary_fixed_point_def) with
    | `OrdinaryFixedPointDef fields ->
        Lazy.force fields.f_delta
  let f_range node =
    match (node :> ordinary_fixed_point_def) with
    | `OrdinaryFixedPointDef fields ->
        Lazy.force fields.f_range



end

module FloatingPointDef = struct
  type t =
    [
      | `FloatingPointDef of
          floating_point_def_fields
    ]

  type fields = floating_point_def_fields =
    
  {
         
    f_num_digits: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BoxExpr
          of box_expr_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_range: range_spec
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_ada_node ((node1 :> ada_node)))
      (unwrap_ada_node ((node2 :> ada_node)))

  let hash node =
    Entity.hash
      (unwrap_ada_node ((node :> ada_node)))


  let f_num_digits node =
    match (node :> floating_point_def) with
    | `FloatingPointDef fields ->
        Lazy.force fields.f_num_digits
  let f_range node =
    match (node :> floating_point_def) with
    | `FloatingPointDef fields ->
        Lazy.force fields.f_range



end

module DecimalFixedPointDef = struct
  type t =
    [
      | `DecimalFixedPointDef of
          decimal_fixed_point_def_fields
    ]

  type fields = decimal_fixed_point_def_fields =
    
  {
         
    f_delta: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BoxExpr
          of box_expr_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fields
      | `Identifier
          of identifier_fields
      | `IfExpr
          of if_expr_fields
      | `IntLiteral
          of int_literal_fields
      | `NullLiteral
          of null_literal_fields
      | `NullRecordAggregate
          of null_record_aggregate_fields
      | `ParenExpr
          of paren_expr_fields
      | `QualExpr
          of qual_expr_fields
      | `QuantifiedExpr
          of quantified_expr_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RealLiteral
          of real_literal_fields
      | `ReduceAttributeRef
          of reduce_attribute_ref_fields
      | `RelationOp
          of relation_op_fields
      | `StringLiteral
          of string_literal_fields
      | `TargetName
          of target_name_fields
      | `UnOp
          of un_op_fields
      | `UpdateAttributeRef
          of update_attribute_ref_fields
    ]
    Lazy.t;
         
    f_digits: [
      | `Aggregate
          of aggregate_fields
      | `Allocator
          of allocator_fields
      | `AttributeRef
          of attribute_ref_fields
      | `BinOp
          of bin_op_fields
      | `BoxExpr
          of box_expr_fields
      | `BracketAggregate
          of bracket_aggregate_fields
      | `BracketDeltaAggregate
          of bracket_delta_aggregate_fields
      | `CallExpr
          of call_expr_fields
      | `CaseExpr
          of case_expr_fields
      | `CharLiteral
          of char_literal_fields
      | `ConcatOp
          of concat_op_fields
      | `DeclExpr
          of decl_expr_fields
      | `DeltaAggregate
          of delta_aggregate_fields
      | `DottedName
          of dotted_name_fields
      | `ExplicitDeref
          of explicit_deref_fie