%feature("docstring") OT::ImportanceSamplingExperiment
"Importance Sampling experiment.

Available constructors:
    ImportanceSamplingExperiment(*importanceDistribution*)

    ImportanceSamplingExperiment(*importanceDistribution, size*)

    ImportanceSamplingExperiment(*initialDistribution, importanceDistribution, size*)

Parameters
----------
initialDistribution : :class:`~openturns.Distribution`
    Distribution :math:`\\mu` which is the initial distribution used to generate the
    set of input data.
size : positive int
    Number of points that will be generated in the experiment.
importanceDistribution : :class:`~openturns.Distribution`
    Distribution :math:`p` according to which the points of the 
    experiments will be generated with the Importance Sampling technique.

Notes
-----
ImportanceSamplingExperiment is a random weighted design of experiments.
We can use it to generate a sample :math:`(\\inputReal_i)_{1 \\leq i \\leq \\sampleSize}`
based on independent observations from the distribution :math:`\\inputMeasure`
(see [hammersley1961]_ page 57, [lemieux2009]_ page 11). 
Importance sampling is a variance reduction method i.e. it aims at 
reducing the variance of the estimator of the weighted integral.
The sample is generated from the importance distribution :math:`p` and each realization is weighted by   
:math:`w_i = \\frac{\\inputProbabilityDensityFunction(\\inputReal_i)}{p(\\inputReal_i)}` where
:math:`\\inputProbabilityDensityFunction` is the probability density function of the 
input random vector.

There is no general method in the library to provide the importance distribution, 
which must be specified by the user.
In the specific case of rare event estimation, [morio2015]_ page 53 provides 
different methods to compute the instrumental distribution, 
including simple changes of measure and exponential twisting.
The :class:`~openturns.PostAnalyticalImportanceSampling` class combines the :class:`~openturns.FORM` class
and importance sampling to estimate a probability.

The `ImportanceSamplingExperiment` class is nonadaptive, i.e. the parameters of the instrumental distribution are 
set once for all.
See :class:`~openturns.NAIS` for an adaptive importance sampling algorithm 
and :class:`~openturns.experimental.CrossEntropyImportanceSampling` for another algorithm.

See also
--------
WeightedExperiment

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> distribution = ot.JointDistribution([ot.Uniform(0, 1)] * 2)
>>> importanceDistribution = ot.JointDistribution([ot.Uniform(0, 1)] * 2)
>>> experiment = ot.ImportanceSamplingExperiment(distribution, importanceDistribution, 5)
>>> print(experiment.generate())
    [ X0        X1        ]
0 : [ 0.629877  0.882805  ]
1 : [ 0.135276  0.0325028 ]
2 : [ 0.347057  0.969423  ]
3 : [ 0.92068   0.50304   ]
4 : [ 0.0632061 0.292757  ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::ImportanceSamplingExperiment::getImportanceDistribution
"Accessor to the importance distribution.

Returns
-------
importanceDistribution : :class:`~openturns.Distribution`
    Distribution :math:`p` according to which the points of the design of
    experiments will be generated with the Importance Sampling technique."
