%feature("docstring") OT::GaussianProcess
"Gaussian processes.

Available constructor:
    GaussianProcess(*trend, covarianceModel, mesh*)

    GaussianProcess(*covarianceModel, mesh*)

Parameters
----------
trend : :class:`~openturns.TrendTransform`
    Trend function of the process. By default the trend is null.
covarianceModel : :class:`~openturns.CovarianceModel`
    Temporal covariance model :math:`C`.
mesh : :class:`~openturns.Mesh`
    Mesh :math:`\\cM` over which the domain :math:`\\cD` is discretized.

Notes
-----
GaussianProcess creates the processes,
:math:`X: \\Omega \\times\\cD \\mapsto \\Rset^d` where :math:`\\cD \\in \\Rset^n`,
from their temporal covariance function
:math:`\\cC: \\cD \\times \\cD \\mapsto \\cM_{d \\times d}(\\Rset)`, which writes, in
the stationary case: :math:`\\cC^{stat}: \\cD \\mapsto \\cM_{d \\times d}(\\Rset)`. A
process is *normal*, if all its finite dimensional joint distributions are
normal (See the method :meth:`~openturns.Process.isNormal` for a detailed definition).

The gaussian processes may have a trend: in that case, the Gaussian 
process is the sum of the trend function
:math:`f_{trend}: \\Rset^n \\mapsto \\Rset^d` and a zero-mean Gaussian process.


Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> # Amplitude values
>>> amplitude = [1.0]
>>> # Scale values
>>> scale = [1.0]
>>> # Second order model with parameters
>>> covarianceModel = ot.AbsoluteExponential(scale, amplitude)
>>> # Time grid
>>> tmin = 0.0
>>> step = 0.1
>>> n = 11
>>> meshGrid = ot.RegularGrid(tmin, step, n)
>>> size = 100
>>> myProcess = ot.GaussianProcess(covarianceModel, meshGrid)
>>> myProcess.setSamplingMethod(myProcess.CHOLESKY)"

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcess::getCovarianceModel
"Get the covariance model.

Returns
-------
covarianceModel : :class:`~openturns.CovarianceModel`
    Temporal covariance model :math:`C`."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcess::getTrend
"Get the trend function.

Returns
-------
trend : :class:`~openturns.TrendTransform`
    Trend function."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcess::isTrendStationary
"Tell if the process is trend stationary or not.

Returns
-------
isTrendStationary : bool
    *True* if the process is trend stationary."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcess::getSamplingMethod
"Get the used method for getRealization.

Returns
-------
samplingMethod : int
    Used method for sampling.

Notes
-----

Available parameters are :

  * 0 (GaussianProcess.CHOLESKY) : Cholesky factor sampling (default method)

  * 1 (GaussianProcess.HMAT) : H-Matrix method (if H-Mat available)

  * 2 (GaussianProcess.GALLIGAOGIBBS) : Gibbs method (in dimension 1 only)

"

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcess::setSamplingMethod
"Set the used method for getRealization.

Available parameters are :

  * 0 (GaussianProcess.CHOLESKY) : Cholesky factor sampling (default method)

  * 1 (GaussianProcess.HMAT) : H-Matrix method (if H-Mat available)

  * 2 (GaussianProcess.GALLIGAOGIBBS) : Gibbs method (in dimension 1 only)

Parameters
----------
samplingMethod : int
    Fix a method for sampling.

"
