/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */
#include <aws/common/string.h>
#include <aws/io/pem.h>
#include <aws/io/private/pem_utils.h>

#include <aws/testing/aws_test_harness.h>

static int s_check_clean_pem_result(
    struct aws_byte_cursor dirty_pem,
    struct aws_byte_cursor expected_clean_pem,
    struct aws_allocator *allocator) {
    struct aws_byte_buf pem_buf;
    ASSERT_SUCCESS(aws_byte_buf_init_copy_from_cursor(&pem_buf, allocator, dirty_pem));
    ASSERT_SUCCESS(aws_sanitize_pem(&pem_buf, allocator));
    ASSERT_TRUE(aws_byte_cursor_eq_byte_buf(&expected_clean_pem, &pem_buf));
    aws_byte_buf_clean_up(&pem_buf);
    return AWS_OP_SUCCESS;
}

static int s_test_pem_sanitize_comments_around_pem_object_removed(struct aws_allocator *allocator, void *ctx) {
    (void)ctx;
    /* comments around pem object will be removed */
    struct aws_byte_cursor dirty_pem = AWS_BYTE_CUR_INIT_FROM_STRING_LITERAL("# comments\r\n"
                                                                             "-----BEGIN CERTIFICATE-----\n"
                                                                             "CERTIFICATES\n"
                                                                             "-----END CERTIFICATE-----\n"
                                                                             "# another comments\r\n"
                                                                             "-----BEGIN CERTIFICATE-----\n"
                                                                             "CERTIFICATES\n"
                                                                             "-----END CERTIFICATE-----\n"
                                                                             "# final comments\r\n");

    struct aws_byte_cursor expected_clean_pem = AWS_BYTE_CUR_INIT_FROM_STRING_LITERAL("-----BEGIN CERTIFICATE-----\n"
                                                                                      "CERTIFICATES\n"
                                                                                      "-----END CERTIFICATE-----\n"
                                                                                      "-----BEGIN CERTIFICATE-----\n"
                                                                                      "CERTIFICATES\n"
                                                                                      "-----END CERTIFICATE-----\n");

    return s_check_clean_pem_result(dirty_pem, expected_clean_pem, allocator);
}

AWS_TEST_CASE(pem_sanitize_comments_around_pem_object_removed, s_test_pem_sanitize_comments_around_pem_object_removed);

static int s_test_pem_sanitize_empty_file_rejected(struct aws_allocator *allocator, void *ctx) {
    (void)ctx;
    /* We don't allow empty files. */
    struct aws_byte_buf pem;
    ASSERT_SUCCESS(aws_byte_buf_init(&pem, allocator, 512));

    ASSERT_ERROR(AWS_ERROR_INVALID_ARGUMENT, aws_sanitize_pem(&pem, allocator));

    aws_byte_buf_clean_up(&pem);
    return AWS_OP_SUCCESS;
}

AWS_TEST_CASE(pem_sanitize_empty_file_rejected, s_test_pem_sanitize_empty_file_rejected)

static int s_test_pem_sanitize_bad_format_rejected(struct aws_allocator *allocator, void *ctx) {
    (void)ctx;
    struct aws_byte_buf pem = aws_byte_buf_from_c_str("aaaaa-");

    ASSERT_ERROR(AWS_ERROR_INVALID_ARGUMENT, aws_sanitize_pem(&pem, allocator));

    aws_byte_buf_clean_up(&pem);
    return AWS_OP_SUCCESS;
}

AWS_TEST_CASE(pem_sanitize_bad_format_rejected, s_test_pem_sanitize_bad_format_rejected)

AWS_TEST_CASE(pem_sanitize_wrong_format_rejected, s_test_pem_sanitize_wrong_format_rejected)
static int s_test_pem_sanitize_wrong_format_rejected(struct aws_allocator *allocator, void *ctx) {
    (void)ctx;
    /* A file with the wrong format will "sanitize" to an empty PEM file, which we do not accept */

    /* This is not a PEM file, it's a DER encoded binary x.509 certificate */
    const uint8_t not_a_pem_src[] = {
        0x30, 0x82, 0x04, 0xD3, 0x30, 0x82, 0x03, 0xBB, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x18, 0xDA, 0xD1,
        0x9E, 0x26, 0x7D, 0xE8, 0xBB, 0x4A, 0x21, 0x58, 0xCD, 0xCC, 0x6B, 0x3B, 0x4A, 0x30, 0x0D, 0x06, 0x09, 0x2A,
        0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x81, 0xCA, 0x31, 0x0B, 0x30, 0x09, 0x06,
        0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13,
        0x0E, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6E, 0x2C, 0x20, 0x49, 0x6E, 0x63, 0x2E, 0x31, 0x1F, 0x30,
        0x1D, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13, 0x16, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6E, 0x20, 0x54,
        0x72, 0x75, 0x73, 0x74, 0x20, 0x4E, 0x65, 0x74, 0x77, 0x6F, 0x72, 0x6B, 0x31, 0x3A, 0x30, 0x38, 0x06, 0x03,
        0x55, 0x04, 0x0B, 0x13, 0x31, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x30, 0x36, 0x20, 0x56, 0x65, 0x72, 0x69,
        0x53, 0x69, 0x67, 0x6E, 0x2C, 0x20, 0x49, 0x6E, 0x63, 0x2E, 0x20, 0x2D, 0x20, 0x46, 0x6F, 0x72, 0x20, 0x61,
        0x75, 0x74, 0x68, 0x6F, 0x72, 0x69, 0x7A, 0x65, 0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x6F, 0x6E, 0x6C, 0x79,
        0x31, 0x45, 0x30, 0x43, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x3C, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67,
        0x6E, 0x20, 0x43, 0x6C, 0x61, 0x73, 0x73, 0x20, 0x33, 0x20, 0x50, 0x75, 0x62, 0x6C, 0x69, 0x63, 0x20, 0x50,
        0x72, 0x69, 0x6D, 0x61, 0x72, 0x79, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69,
        0x6F, 0x6E, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6F, 0x72, 0x69, 0x74, 0x79, 0x20, 0x2D, 0x20, 0x47, 0x35, 0x30,
        0x1E, 0x17, 0x0D, 0x30, 0x36, 0x31, 0x31, 0x30, 0x38, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5A, 0x17, 0x0D,
        0x33, 0x36, 0x30, 0x37, 0x31, 0x36, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5A, 0x30, 0x81, 0xCA, 0x31, 0x0B,
        0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55,
        0x04, 0x0A, 0x13, 0x0E, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6E, 0x2C, 0x20, 0x49, 0x6E, 0x63, 0x2E,
        0x31, 0x1F, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13, 0x16, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67,
        0x6E, 0x20, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x4E, 0x65, 0x74, 0x77, 0x6F, 0x72, 0x6B, 0x31, 0x3A, 0x30,
        0x38, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13, 0x31, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x30, 0x36, 0x20, 0x56,
        0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6E, 0x2C, 0x20, 0x49, 0x6E, 0x63, 0x2E, 0x20, 0x2D, 0x20, 0x46, 0x6F,
        0x72, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6F, 0x72, 0x69, 0x7A, 0x65, 0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x6F,
        0x6E, 0x6C, 0x79, 0x31, 0x45, 0x30, 0x43, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x3C, 0x56, 0x65, 0x72, 0x69,
        0x53, 0x69, 0x67, 0x6E, 0x20, 0x43, 0x6C, 0x61, 0x73, 0x73, 0x20, 0x33, 0x20, 0x50, 0x75, 0x62, 0x6C, 0x69,
        0x63, 0x20, 0x50, 0x72, 0x69, 0x6D, 0x61, 0x72, 0x79, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63,
        0x61, 0x74, 0x69, 0x6F, 0x6E, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6F, 0x72, 0x69, 0x74, 0x79, 0x20, 0x2D, 0x20,
        0x47, 0x35, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01,
        0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xAF,
        0x24, 0x08, 0x08, 0x29, 0x7A, 0x35, 0x9E, 0x60, 0x0C, 0xAA, 0xE7, 0x4B, 0x3B, 0x4E, 0xDC, 0x7C, 0xBC, 0x3C,
        0x45, 0x1C, 0xBB, 0x2B, 0xE0, 0xFE, 0x29, 0x02, 0xF9, 0x57, 0x08, 0xA3, 0x64, 0x85, 0x15, 0x27, 0xF5, 0xF1,
        0xAD, 0xC8, 0x31, 0x89, 0x5D, 0x22, 0xE8, 0x2A, 0xAA, 0xA6, 0x42, 0xB3, 0x8F, 0xF8, 0xB9, 0x55, 0xB7, 0xB1,
        0xB7, 0x4B, 0xB3, 0xFE, 0x8F, 0x7E, 0x07, 0x57, 0xEC, 0xEF, 0x43, 0xDB, 0x66, 0x62, 0x15, 0x61, 0xCF, 0x60,
        0x0D, 0xA4, 0xD8, 0xDE, 0xF8, 0xE0, 0xC3, 0x62, 0x08, 0x3D, 0x54, 0x13, 0xEB, 0x49, 0xCA, 0x59, 0x54, 0x85,
        0x26, 0xE5, 0x2B, 0x8F, 0x1B, 0x9F, 0xEB, 0xF5, 0xA1, 0x91, 0xC2, 0x33, 0x49, 0xD8, 0x43, 0x63, 0x6A, 0x52,
        0x4B, 0xD2, 0x8F, 0xE8, 0x70, 0x51, 0x4D, 0xD1, 0x89, 0x69, 0x7B, 0xC7, 0x70, 0xF6, 0xB3, 0xDC, 0x12, 0x74,
        0xDB, 0x7B, 0x5D, 0x4B, 0x56, 0xD3, 0x96, 0xBF, 0x15, 0x77, 0xA1, 0xB0, 0xF4, 0xA2, 0x25, 0xF2, 0xAF, 0x1C,
        0x92, 0x67, 0x18, 0xE5, 0xF4, 0x06, 0x04, 0xEF, 0x90, 0xB9, 0xE4, 0x00, 0xE4, 0xDD, 0x3A, 0xB5, 0x19, 0xFF,
        0x02, 0xBA, 0xF4, 0x3C, 0xEE, 0xE0, 0x8B, 0xEB, 0x37, 0x8B, 0xEC, 0xF4, 0xD7, 0xAC, 0xF2, 0xF6, 0xF0, 0x3D,
        0xAF, 0xDD, 0x75, 0x91, 0x33, 0x19, 0x1D, 0x1C, 0x40, 0xCB, 0x74, 0x24, 0x19, 0x21, 0x93, 0xD9, 0x14, 0xFE,
        0xAC, 0x2A, 0x52, 0xC7, 0x8F, 0xD5, 0x04, 0x49, 0xE4, 0x8D, 0x63, 0x47, 0x88, 0x3C, 0x69, 0x83, 0xCB, 0xFE,
        0x47, 0xBD, 0x2B, 0x7E, 0x4F, 0xC5, 0x95, 0xAE, 0x0E, 0x9D, 0xD4, 0xD1, 0x43, 0xC0, 0x67, 0x73, 0xE3, 0x14,
        0x08, 0x7E, 0xE5, 0x3F, 0x9F, 0x73, 0xB8, 0x33, 0x0A, 0xCF, 0x5D, 0x3F, 0x34, 0x87, 0x96, 0x8A, 0xEE, 0x53,
        0xE8, 0x25, 0x15, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x81, 0xB2, 0x30, 0x81, 0xAF, 0x30, 0x0F, 0x06, 0x03,
        0x55, 0x1D, 0x13, 0x01, 0x01, 0xFF, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xFF, 0x30, 0x0E, 0x06, 0x03, 0x55,
        0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x6D, 0x06, 0x08, 0x2B, 0x06, 0x01,
        0x05, 0x05, 0x07, 0x01, 0x0C, 0x04, 0x61, 0x30, 0x5F, 0xA1, 0x5D, 0xA0, 0x5B, 0x30, 0x59, 0x30, 0x57, 0x30,
        0x55, 0x16, 0x09, 0x69, 0x6D, 0x61, 0x67, 0x65, 0x2F, 0x67, 0x69, 0x66, 0x30, 0x21, 0x30, 0x1F, 0x30, 0x07,
        0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x04, 0x14, 0x8F, 0xE5, 0xD3, 0x1A, 0x86, 0xAC, 0x8D, 0x8E, 0x6B,
        0xC3, 0xCF, 0x80, 0x6A, 0xD4, 0x48, 0x18, 0x2C, 0x7B, 0x19, 0x2E, 0x30, 0x25, 0x16, 0x23, 0x68, 0x74, 0x74,
        0x70, 0x3A, 0x2F, 0x2F, 0x6C, 0x6F, 0x67, 0x6F, 0x2E, 0x76, 0x65, 0x72, 0x69, 0x73, 0x69, 0x67, 0x6E, 0x2E,
        0x63, 0x6F, 0x6D, 0x2F, 0x76, 0x73, 0x6C, 0x6F, 0x67, 0x6F, 0x2E, 0x67, 0x69, 0x66, 0x30, 0x1D, 0x06, 0x03,
        0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0x7F, 0xD3, 0x65, 0xA7, 0xC2, 0xDD, 0xEC, 0xBB, 0xF0, 0x30, 0x09,
        0xF3, 0x43, 0x39, 0xFA, 0x02, 0xAF, 0x33, 0x31, 0x33, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7,
        0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x93, 0x24, 0x4A, 0x30, 0x5F, 0x62, 0xCF,
        0xD8, 0x1A, 0x98, 0x2F, 0x3D, 0xEA, 0xDC, 0x99, 0x2D, 0xBD, 0x77, 0xF6, 0xA5, 0x79, 0x22, 0x38, 0xEC, 0xC4,
        0xA7, 0xA0, 0x78, 0x12, 0xAD, 0x62, 0x0E, 0x45, 0x70, 0x64, 0xC5, 0xE7, 0x97, 0x66, 0x2D, 0x98, 0x09, 0x7E,
        0x5F, 0xAF, 0xD6, 0xCC, 0x28, 0x65, 0xF2, 0x01, 0xAA, 0x08, 0x1A, 0x47, 0xDE, 0xF9, 0xF9, 0x7C, 0x92, 0x5A,
        0x08, 0x69, 0x20, 0x0D, 0xD9, 0x3E, 0x6D, 0x6E, 0x3C, 0x0D, 0x6E, 0xD8, 0xE6, 0x06, 0x91, 0x40, 0x18, 0xB9,
        0xF8, 0xC1, 0xED, 0xDF, 0xDB, 0x41, 0xAA, 0xE0, 0x96, 0x20, 0xC9, 0xCD, 0x64, 0x15, 0x38, 0x81, 0xC9, 0x94,
        0xEE, 0xA2, 0x84, 0x29, 0x0B, 0x13, 0x6F, 0x8E, 0xDB, 0x0C, 0xDD, 0x25, 0x02, 0xDB, 0xA4, 0x8B, 0x19, 0x44,
        0xD2, 0x41, 0x7A, 0x05, 0x69, 0x4A, 0x58, 0x4F, 0x60, 0xCA, 0x7E, 0x82, 0x6A, 0x0B, 0x02, 0xAA, 0x25, 0x17,
        0x39, 0xB5, 0xDB, 0x7F, 0xE7, 0x84, 0x65, 0x2A, 0x95, 0x8A, 0xBD, 0x86, 0xDE, 0x5E, 0x81, 0x16, 0x83, 0x2D,
        0x10, 0xCC, 0xDE, 0xFD, 0xA8, 0x82, 0x2A, 0x6D, 0x28, 0x1F, 0x0D, 0x0B, 0xC4, 0xE5, 0xE7, 0x1A, 0x26, 0x19,
        0xE1, 0xF4, 0x11, 0x6F, 0x10, 0xB5, 0x95, 0xFC, 0xE7, 0x42, 0x05, 0x32, 0xDB, 0xCE, 0x9D, 0x51, 0x5E, 0x28,
        0xB6, 0x9E, 0x85, 0xD3, 0x5B, 0xEF, 0xA5, 0x7D, 0x45, 0x40, 0x72, 0x8E, 0xB7, 0x0E, 0x6B, 0x0E, 0x06, 0xFB,
        0x33, 0x35, 0x48, 0x71, 0xB8, 0x9D, 0x27, 0x8B, 0xC4, 0x65, 0x5F, 0x0D, 0x86, 0x76, 0x9C, 0x44, 0x7A, 0xF6,
        0x95, 0x5C, 0xF6, 0x5D, 0x32, 0x08, 0x33, 0xA4, 0x54, 0xB6, 0x18, 0x3F, 0x68, 0x5C, 0xF2, 0x42, 0x4A, 0x85,
        0x38, 0x54, 0x83, 0x5F, 0xD1, 0xE8, 0x2C, 0xF2, 0xAC, 0x11, 0xD6, 0xA8, 0xED, 0x63, 0x6A};
    struct aws_byte_cursor not_a_pem_cursor = aws_byte_cursor_from_array(not_a_pem_src, sizeof(not_a_pem_src));
    struct aws_byte_buf not_a_pem;
    ASSERT_SUCCESS(aws_byte_buf_init_copy_from_cursor(&not_a_pem, allocator, not_a_pem_cursor));

    ASSERT_ERROR(AWS_ERROR_INVALID_ARGUMENT, aws_sanitize_pem(&not_a_pem, allocator));

    aws_byte_buf_clean_up(&not_a_pem);
    return AWS_OP_SUCCESS;
}

static int s_test_pem_cert_parse_from_file(struct aws_allocator *allocator, void *ctx) {
    (void)ctx;

    static const uint8_t s_expected[] = {
        0x30, 0x82, 0x03, 0xec, 0x30, 0x82, 0x02, 0xd4, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x09, 0x00, 0x84, 0x7d,
        0x2e, 0xed, 0x4d, 0xfc, 0x26, 0x87, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
        0x0b, 0x05, 0x00, 0x30, 0x81, 0x9a, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55,
        0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6e,
        0x67, 0x74, 0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65, 0x61,
        0x74, 0x74, 0x6c, 0x65, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x06, 0x41, 0x6d, 0x61,
        0x7a, 0x6f, 0x6e, 0x31, 0x0d, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x04, 0x53, 0x44, 0x4b, 0x73,
        0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x09, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f,
        0x73, 0x74, 0x31, 0x30, 0x30, 0x2e, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16,
        0x21, 0x61, 0x77, 0x73, 0x2d, 0x73, 0x64, 0x6b, 0x2d, 0x63, 0x6f, 0x6d, 0x6d, 0x6f, 0x6e, 0x2d, 0x72, 0x75,
        0x6e, 0x74, 0x69, 0x6d, 0x65, 0x40, 0x61, 0x6d, 0x61, 0x7a, 0x6f, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1e,
        0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x31, 0x36, 0x30, 0x36, 0x31, 0x37, 0x30, 0x30, 0x5a, 0x17, 0x0d, 0x32,
        0x33, 0x30, 0x39, 0x31, 0x38, 0x30, 0x36, 0x31, 0x37, 0x30, 0x30, 0x5a, 0x30, 0x81, 0x9a, 0x31, 0x0b, 0x30,
        0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04,
        0x08, 0x0c, 0x0a, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x74, 0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06,
        0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65, 0x61, 0x74, 0x74, 0x6c, 0x65, 0x31, 0x0f, 0x30, 0x0d, 0x06,
        0x03, 0x55, 0x04, 0x0a, 0x0c, 0x06, 0x41, 0x6d, 0x61, 0x7a, 0x6f, 0x6e, 0x31, 0x0d, 0x30, 0x0b, 0x06, 0x03,
        0x55, 0x04, 0x0b, 0x0c, 0x04, 0x53, 0x44, 0x4b, 0x73, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x03,
        0x0c, 0x09, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74, 0x31, 0x30, 0x30, 0x2e, 0x06, 0x09, 0x2a,
        0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x21, 0x61, 0x77, 0x73, 0x2d, 0x73, 0x64, 0x6b, 0x2d,
        0x63, 0x6f, 0x6d, 0x6d, 0x6f, 0x6e, 0x2d, 0x72, 0x75, 0x6e, 0x74, 0x69, 0x6d, 0x65, 0x40, 0x61, 0x6d, 0x61,
        0x7a, 0x6f, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48,
        0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02,
        0x82, 0x01, 0x01, 0x00, 0xd7, 0x6a, 0x57, 0x48, 0xf8, 0x0e, 0x44, 0x03, 0x25, 0x42, 0xd6, 0x11, 0x6f, 0x1b,
        0xb3, 0xfc, 0xe7, 0x1a, 0xa2, 0xb6, 0xa7, 0xdc, 0x2d, 0x85, 0x8f, 0x28, 0xe1, 0xbb, 0x4b, 0xee, 0x71, 0x21,
        0x19, 0x4b, 0x0c, 0x43, 0x26, 0x9e, 0xf9, 0x4c, 0x14, 0x04, 0x31, 0xa7, 0xd2, 0xa5, 0x21, 0x0a, 0x01, 0x02,
        0xde, 0x0e, 0xde, 0xf1, 0xb8, 0x34, 0x43, 0x62, 0x7e, 0x76, 0x57, 0x85, 0x04, 0xe9, 0xc1, 0x7e, 0xc5, 0x35,
        0xa1, 0xb7, 0x3b, 0x1f, 0xee, 0x68, 0x4d, 0xfe, 0x51, 0xda, 0x0c, 0xf7, 0x2f, 0x47, 0x60, 0x12, 0x3c, 0x01,
        0x24, 0xce, 0x48, 0xa5, 0xf0, 0xa0, 0x8b, 0x63, 0x87, 0xba, 0xb5, 0x3c, 0x52, 0xc1, 0x0f, 0x7b, 0xb2, 0x99,
        0x4d, 0xb8, 0x46, 0x74, 0xf7, 0xd1, 0xe8, 0x25, 0x84, 0xd3, 0x2c, 0x56, 0x91, 0x78, 0x87, 0xdd, 0xd4, 0x3d,
        0xf3, 0x67, 0x51, 0x18, 0x71, 0x2c, 0x3c, 0xc3, 0xe1, 0x99, 0xd9, 0x2c, 0x44, 0x51, 0xf6, 0x14, 0x48, 0xbd,
        0x82, 0x16, 0x62, 0x18, 0x4a, 0x44, 0x23, 0x9e, 0x5b, 0x09, 0x08, 0x8a, 0x42, 0xa0, 0x68, 0x03, 0x88, 0x10,
        0x0f, 0x6c, 0x85, 0x09, 0x3b, 0x72, 0x96, 0x04, 0x35, 0xf4, 0x26, 0x01, 0x83, 0x6f, 0x1d, 0xd6, 0x7f, 0x78,
        0xd7, 0x1b, 0xf6, 0x3a, 0x4f, 0xad, 0xcb, 0x3e, 0xc3, 0xbe, 0x01, 0x2d, 0xb4, 0x44, 0x2b, 0xdc, 0x10, 0x5d,
        0x05, 0xfe, 0xb9, 0x43, 0x20, 0xdc, 0xc8, 0xe4, 0x40, 0x07, 0x3b, 0x54, 0xce, 0x11, 0xdf, 0x5f, 0x28, 0xeb,
        0xbe, 0x24, 0x02, 0xb4, 0xe8, 0xfc, 0x35, 0x9b, 0xbe, 0xc1, 0x80, 0xea, 0xc4, 0xec, 0x5b, 0x6f, 0x20, 0x6e,
        0xe4, 0x60, 0xd5, 0x6e, 0x38, 0x43, 0xde, 0x22, 0x73, 0x87, 0x90, 0xeb, 0xaa, 0xaf, 0x20, 0xe2, 0xb0, 0x1d,
        0x4f, 0xc2, 0x2c, 0x8f, 0x34, 0x86, 0xea, 0x75, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x33, 0x30, 0x31, 0x30,
        0x13, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x04, 0x0c, 0x30, 0x0a, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07,
        0x03, 0x01, 0x30, 0x1a, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x13, 0x30, 0x11, 0x82, 0x09, 0x6c, 0x6f, 0x63,
        0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74, 0x87, 0x04, 0x7f, 0x00, 0x00, 0x01, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
        0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x07, 0xb4, 0x9a, 0x48,
        0x4e, 0x6d, 0x71, 0x32, 0xf4, 0x35, 0x89, 0xf5, 0xe1, 0xe8, 0x27, 0x5e, 0xe3, 0x51, 0x0d, 0x54, 0xf2, 0xde,
        0x1e, 0x2f, 0x9a, 0x0d, 0xdd, 0x02, 0xd4, 0xce, 0x15, 0x93, 0x8b, 0xe6, 0x75, 0x77, 0xc2, 0x41, 0xf6, 0xbf,
        0xfc, 0xac, 0x25, 0x96, 0xea, 0x80, 0x38, 0x68, 0xe2, 0xa5, 0x72, 0x9a, 0x31, 0xa2, 0x95, 0x43, 0xa9, 0x90,
        0x39, 0x64, 0xe3, 0x6c, 0x29, 0x37, 0x0c, 0x7a, 0xb7, 0x18, 0x97, 0x47, 0x0e, 0x16, 0x79, 0x2f, 0x9a, 0x92,
        0x7b, 0x51, 0xac, 0xe4, 0x4c, 0x70, 0xc2, 0xe4, 0xf3, 0x7f, 0x2b, 0x63, 0x53, 0x2c, 0x3b, 0xdb, 0xf1, 0xef,
        0x84, 0xda, 0xf3, 0x71, 0x6c, 0x6e, 0xb8, 0x41, 0x48, 0xae, 0xb5, 0x12, 0x1b, 0x20, 0xec, 0xdf, 0xff, 0x9f,
        0x2b, 0x2d, 0x66, 0x52, 0x0a, 0x72, 0x17, 0x99, 0xa5, 0x4d, 0x28, 0x29, 0x8a, 0x9c, 0xc8, 0x51, 0xd0, 0xe8,
        0x5c, 0x42, 0x66, 0x3e, 0xef, 0x06, 0xda, 0x72, 0xea, 0xa8, 0x5a, 0x5a, 0x02, 0x0f, 0xa2, 0x68, 0x80, 0xa9,
        0x9a, 0xa4, 0x30, 0x8c, 0x9e, 0x69, 0x75, 0xa5, 0x5c, 0x34, 0x8b, 0x71, 0x49, 0xe5, 0x3a, 0x1a, 0x74, 0xa9,
        0x51, 0x86, 0xee, 0x06, 0xf9, 0x54, 0x37, 0x0c, 0xf6, 0x17, 0x8a, 0x1e, 0xc3, 0x54, 0x6d, 0xa9, 0x52, 0x4a,
        0x2f, 0xf8, 0xd0, 0xe3, 0x56, 0xc2, 0x61, 0x14, 0xfd, 0x7c, 0x77, 0x2b, 0x5b, 0x8b, 0x93, 0x2c, 0x6e, 0x76,
        0x46, 0xa9, 0x00, 0x34, 0x8d, 0x55, 0x42, 0x6a, 0xe2, 0x6b, 0xa3, 0xd8, 0xe6, 0x5a, 0x5b, 0x65, 0x98, 0xa8,
        0xb1, 0x85, 0x01, 0x92, 0x42, 0xf4, 0xd6, 0x73, 0x4d, 0xc6, 0xf6, 0xf1, 0x34, 0x36, 0x16, 0x44, 0xc6, 0x09,
        0xc7, 0x94, 0x46, 0x1c, 0x06, 0x94, 0x84, 0xa9, 0x4f, 0x41, 0x0b, 0x46, 0xa6, 0xb4, 0x48, 0x1a, 0x14, 0x45,
    };

    struct aws_array_list output_list;
    ASSERT_SUCCESS(aws_pem_objects_init_from_file_path(&output_list, allocator, "testparse.crt"));
    ASSERT_UINT_EQUALS(1, aws_array_list_length(&output_list));

    struct aws_pem_object *pem_object = NULL;
    aws_array_list_get_at_ptr(&output_list, (void **)&pem_object, 0);
    ASSERT_BIN_ARRAYS_EQUALS(s_expected, sizeof(s_expected), pem_object->data.buffer, pem_object->data.len);
    ASSERT_CURSOR_VALUE_CSTRING_EQUALS(aws_byte_cursor_from_string(pem_object->type_string), "CERTIFICATE");
    ASSERT_INT_EQUALS(AWS_PEM_TYPE_X509, pem_object->type);

    aws_pem_objects_clean_up(&output_list);

    return AWS_OP_SUCCESS;
}
AWS_TEST_CASE(test_pem_cert_parse_from_file, s_test_pem_cert_parse_from_file)

static int s_test_pem_cert_parse_from_file_crlf(struct aws_allocator *allocator, void *ctx) {
    (void)ctx;

    static const uint8_t s_expected[] = {
        0x30, 0x82, 0x03, 0xec, 0x30, 0x82, 0x02, 0xd4, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x09, 0x00, 0x84, 0x7d,
        0x2e, 0xed, 0x4d, 0xfc, 0x26, 0x87, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
        0x0b, 0x05, 0x00, 0x30, 0x81, 0x9a, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55,
        0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6e,
        0x67, 0x74, 0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65, 0x61,
        0x74, 0x74, 0x6c, 0x65, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x06, 0x41, 0x6d, 0x61,
        0x7a, 0x6f, 0x6e, 0x31, 0x0d, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x04, 0x53, 0x44, 0x4b, 0x73,
        0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x09, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f,
        0x73, 0x74, 0x31, 0x30, 0x30, 0x2e, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16,
        0x21, 0x61, 0x77, 0x73, 0x2d, 0x73, 0x64, 0x6b, 0x2d, 0x63, 0x6f, 0x6d, 0x6d, 0x6f, 0x6e, 0x2d, 0x72, 0x75,
        0x6e, 0x74, 0x69, 0x6d, 0x65, 0x40, 0x61, 0x6d, 0x61, 0x7a, 0x6f, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1e,
        0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x31, 0x36, 0x30, 0x36, 0x31, 0x37, 0x30, 0x30, 0x5a, 0x17, 0x0d, 0x32,
        0x33, 0x30, 0x39, 0x31, 0x38, 0x30, 0x36, 0x31, 0x37, 0x30, 0x30, 0x5a, 0x30, 0x81, 0x9a, 0x31, 0x0b, 0x30,
        0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04,
        0x08, 0x0c, 0x0a, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x74, 0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06,
        0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65, 0x61, 0x74, 0x74, 0x6c, 0x65, 0x31, 0x0f, 0x30, 0x0d, 0x06,
        0x03, 0x55, 0x04, 0x0a, 0x0c, 0x06, 0x41, 0x6d, 0x61, 0x7a, 0x6f, 0x6e, 0x31, 0x0d, 0x30, 0x0b, 0x06, 0x03,
        0x55, 0x04, 0x0b, 0x0c, 0x04, 0x53, 0x44, 0x4b, 0x73, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x03,
        0x0c, 0x09, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74, 0x31, 0x30, 0x30, 0x2e, 0x06, 0x09, 0x2a,
        0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x21, 0x61, 0x77, 0x73, 0x2d, 0x73, 0x64, 0x6b, 0x2d,
        0x63, 0x6f, 0x6d, 0x6d, 0x6f, 0x6e, 0x2d, 0x72, 0x75, 0x6e, 0x74, 0x69, 0x6d, 0x65, 0x40, 0x61, 0x6d, 0x61,
        0x7a, 0x6f, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48,
        0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02,
        0x82, 0x01, 0x01, 0x00, 0xd7, 0x6a, 0x57, 0x48, 0xf8, 0x0e, 0x44, 0x03, 0x25, 0x42, 0xd6, 0x11, 0x6f, 0x1b,
        0xb3, 0xfc, 0xe7, 0x1a, 0xa2, 0xb6, 0xa7, 0xdc, 0x2d, 0x85, 0x8f, 0x28, 0xe1, 0xbb, 0x4b, 0xee, 0x71, 0x21,
        0x19, 0x4b, 0x0c, 0x43, 0x26, 0x9e, 0xf9, 0x4c, 0x14, 0x04, 0x31, 0xa7, 0xd2, 0xa5, 0x21, 0x0a, 0x01, 0x02,
        0xde, 0x0e, 0xde, 0xf1, 0xb8, 0x34, 0x43, 0x62, 0x7e, 0x76, 0x57, 0x85, 0x04, 0xe9, 0xc1, 0x7e, 0xc5, 0x35,
        0xa1, 0xb7, 0x3b, 0x1f, 0xee, 0x68, 0x4d, 0xfe, 0x51, 0xda, 0x0c, 0xf7, 0x2f, 0x47, 0x60, 0x12, 0x3c, 0x01,
        0x24, 0xce, 0x48, 0xa5, 0xf0, 0xa0, 0x8b, 0x63, 0x87, 0xba, 0xb5, 0x3c, 0x52, 0xc1, 0x0f, 0x7b, 0xb2, 0x99,
        0x4d, 0xb8, 0x46, 0x74, 0xf7, 0xd1, 0xe8, 0x25, 0x84, 0xd3, 0x2c, 0x56, 0x91, 0x78, 0x87, 0xdd, 0xd4, 0x3d,
        0xf3, 0x67, 0x51, 0x18, 0x71, 0x2c, 0x3c, 0xc3, 0xe1, 0x99, 0xd9, 0x2c, 0x44, 0x51, 0xf6, 0x14, 0x48, 0xbd,
        0x82, 0x16, 0x62, 0x18, 0x4a, 0x44, 0x23, 0x9e, 0x5b, 0x09, 0x08, 0x8a, 0x42, 0xa0, 0x68, 0x03, 0x88, 0x10,
        0x0f, 0x6c, 0x85, 0x09, 0x3b, 0x72, 0x96, 0x04, 0x35, 0xf4, 0x26, 0x01, 0x83, 0x6f, 0x1d, 0xd6, 0x7f, 0x78,
        0xd7, 0x1b, 0xf6, 0x3a, 0x4f, 0xad, 0xcb, 0x3e, 0xc3, 0xbe, 0x01, 0x2d, 0xb4, 0x44, 0x2b, 0xdc, 0x10, 0x5d,
        0x05, 0xfe, 0xb9, 0x43, 0x20, 0xdc, 0xc8, 0xe4, 0x40, 0x07, 0x3b, 0x54, 0xce, 0x11, 0xdf, 0x5f, 0x28, 0xeb,
        0xbe, 0x24, 0x02, 0xb4, 0xe8, 0xfc, 0x35, 0x9b, 0xbe, 0xc1, 0x80, 0xea, 0xc4, 0xec, 0x5b, 0x6f, 0x20, 0x6e,
        0xe4, 0x60, 0xd5, 0x6e, 0x38, 0x43, 0xde, 0x22, 0x73, 0x87, 0x90, 0xeb, 0xaa, 0xaf, 0x20, 0xe2, 0xb0, 0x1d,
        0x4f, 0xc2, 0x2c, 0x8f, 0x34, 0x86, 0xea, 0x75, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x33, 0x30, 0x31, 0x30,
        0x13, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x04, 0x0c, 0x30, 0x0a, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07,
        0x03, 0x01, 0x30, 0x1a, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x13, 0x30, 0x11, 0x82, 0x09, 0x6c, 0x6f, 0x63,
        0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74, 0x87, 0x04, 0x7f, 0x00, 0x00, 0x01, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
        0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x07, 0xb4, 0x9a, 0x48,
        0x4e, 0x6d, 0x71, 0x32, 0xf4, 0x35, 0x89, 0xf5, 0xe1, 0xe8, 0x27, 0x5e, 0xe3, 0x51, 0x0d, 0x54, 0xf2, 0xde,
        0x1e, 0x2f, 0x9a, 0x0d, 0xdd, 0x02, 0xd4, 0xce, 0x15, 0x93, 0x8b, 0xe6, 0x75, 0x77, 0xc2, 0x41, 0xf6, 0xbf,
        0xfc, 0xac, 0x25, 0x96, 0xea, 0x80, 0x38, 0x68, 0xe2, 0xa5, 0x72, 0x9a, 0x31, 0xa2, 0x95, 0x43, 0xa9, 0x90,
        0x39, 0x64, 0xe3, 0x6c, 0x29, 0x37, 0x0c, 0x7a, 0xb7, 0x18, 0x97, 0x47, 0x0e, 0x16, 0x79, 0x2f, 0x9a, 0x92,
        0x7b, 0x51, 0xac, 0xe4, 0x4c, 0x70, 0xc2, 0xe4, 0xf3, 0x7f, 0x2b, 0x63, 0x53, 0x2c, 0x3b, 0xdb, 0xf1, 0xef,
        0x84, 0xda, 0xf3, 0x71, 0x6c, 0x6e, 0xb8, 0x41, 0x48, 0xae, 0xb5, 0x12, 0x1b, 0x20, 0xec, 0xdf, 0xff, 0x9f,
        0x2b, 0x2d, 0x66, 0x52, 0x0a, 0x72, 0x17, 0x99, 0xa5, 0x4d, 0x28, 0x29, 0x8a, 0x9c, 0xc8, 0x51, 0xd0, 0xe8,
        0x5c, 0x42, 0x66, 0x3e, 0xef, 0x06, 0xda, 0x72, 0xea, 0xa8, 0x5a, 0x5a, 0x02, 0x0f, 0xa2, 0x68, 0x80, 0xa9,
        0x9a, 0xa4, 0x30, 0x8c, 0x9e, 0x69, 0x75, 0xa5, 0x5c, 0x34, 0x8b, 0x71, 0x49, 0xe5, 0x3a, 0x1a, 0x74, 0xa9,
        0x51, 0x86, 0xee, 0x06, 0xf9, 0x54, 0x37, 0x0c, 0xf6, 0x17, 0x8a, 0x1e, 0xc3, 0x54, 0x6d, 0xa9, 0x52, 0x4a,
        0x2f, 0xf8, 0xd0, 0xe3, 0x56, 0xc2, 0x61, 0x14, 0xfd, 0x7c, 0x77, 0x2b, 0x5b, 0x8b, 0x93, 0x2c, 0x6e, 0x76,
        0x46, 0xa9, 0x00, 0x34, 0x8d, 0x55, 0x42, 0x6a, 0xe2, 0x6b, 0xa3, 0xd8, 0xe6, 0x5a, 0x5b, 0x65, 0x98, 0xa8,
        0xb1, 0x85, 0x01, 0x92, 0x42, 0xf4, 0xd6, 0x73, 0x4d, 0xc6, 0xf6, 0xf1, 0x34, 0x36, 0x16, 0x44, 0xc6, 0x09,
        0xc7, 0x94, 0x46, 0x1c, 0x06, 0x94, 0x84, 0xa9, 0x4f, 0x41, 0x0b, 0x46, 0xa6, 0xb4, 0x48, 0x1a, 0x14, 0x45,
    };

    struct aws_array_list output_list;
    ASSERT_SUCCESS(aws_pem_objects_init_from_file_path(&output_list, allocator, "testparse_crlf.crt"));
    ASSERT_UINT_EQUALS(1, aws_array_list_length(&output_list));

    struct aws_pem_object *pem_object = NULL;
    aws_array_list_get_at_ptr(&output_list, (void **)&pem_object, 0);
    ASSERT_BIN_ARRAYS_EQUALS(s_expected, sizeof(s_expected), pem_object->data.buffer, pem_object->data.len);
    ASSERT_CURSOR_VALUE_CSTRING_EQUALS(aws_byte_cursor_from_string(pem_object->type_string), "CERTIFICATE");
    ASSERT_INT_EQUALS(AWS_PEM_TYPE_X509, pem_object->type);

    aws_pem_objects_clean_up(&output_list);

    return AWS_OP_SUCCESS;
}
AWS_TEST_CASE(test_pem_cert_parse_from_file_crlf, s_test_pem_cert_parse_from_file_crlf)

static int s_test_pem_private_key_parse_from_file(struct aws_allocator *allocator, void *ctx) {
    (void)ctx;

    static const uint8_t s_expected[] = {
        0x30, 0x82, 0x04, 0xa4, 0x02, 0x01, 0x00, 0x02, 0x82, 0x01, 0x01, 0x00, 0xd7, 0x6a, 0x57, 0x48, 0xf8, 0x0e,
        0x44, 0x03, 0x25, 0x42, 0xd6, 0x11, 0x6f, 0x1b, 0xb3, 0xfc, 0xe7, 0x1a, 0xa2, 0xb6, 0xa7, 0xdc, 0x2d, 0x85,
        0x8f, 0x28, 0xe1, 0xbb, 0x4b, 0xee, 0x71, 0x21, 0x19, 0x4b, 0x0c, 0x43, 0x26, 0x9e, 0xf9, 0x4c, 0x14, 0x04,
        0x31, 0xa7, 0xd2, 0xa5, 0x21, 0x0a, 0x01, 0x02, 0xde, 0x0e, 0xde, 0xf1, 0xb8, 0x34, 0x43, 0x62, 0x7e, 0x76,
        0x57, 0x85, 0x04, 0xe9, 0xc1, 0x7e, 0xc5, 0x35, 0xa1, 0xb7, 0x3b, 0x1f, 0xee, 0x68, 0x4d, 0xfe, 0x51, 0xda,
        0x0c, 0xf7, 0x2f, 0x47, 0x60, 0x12, 0x3c, 0x01, 0x24, 0xce, 0x48, 0xa5, 0xf0, 0xa0, 0x8b, 0x63, 0x87, 0xba,
        0xb5, 0x3c, 0x52, 0xc1, 0x0f, 0x7b, 0xb2, 0x99, 0x4d, 0xb8, 0x46, 0x74, 0xf7, 0xd1, 0xe8, 0x25, 0x84, 0xd3,
        0x2c, 0x56, 0x91, 0x78, 0x87, 0xdd, 0xd4, 0x3d, 0xf3, 0x67, 0x51, 0x18, 0x71, 0x2c, 0x3c, 0xc3, 0xe1, 0x99,
        0xd9, 0x2c, 0x44, 0x51, 0xf6, 0x14, 0x48, 0xbd, 0x82, 0x16, 0x62, 0x18, 0x4a, 0x44, 0x23, 0x9e, 0x5b, 0x09,
        0x08, 0x8a, 0x42, 0xa0, 0x68, 0x03, 0x88, 0x10, 0x0f, 0x6c, 0x85, 0x09, 0x3b, 0x72, 0x96, 0x04, 0x35, 0xf4,
        0x26, 0x01, 0x83, 0x6f, 0x1d, 0xd6, 0x7f, 0x78, 0xd7, 0x1b, 0xf6, 0x3a, 0x4f, 0xad, 0xcb, 0x3e, 0xc3, 0xbe,
        0x01, 0x2d, 0xb4, 0x44, 0x2b, 0xdc, 0x10, 0x5d, 0x05, 0xfe, 0xb9, 0x43, 0x20, 0xdc, 0xc8, 0xe4, 0x40, 0x07,
        0x3b, 0x54, 0xce, 0x11, 0xdf, 0x5f, 0x28, 0xeb, 0xbe, 0x24, 0x02, 0xb4, 0xe8, 0xfc, 0x35, 0x9b, 0xbe, 0xc1,
        0x80, 0xea, 0xc4, 0xec, 0x5b, 0x6f, 0x20, 0x6e, 0xe4, 0x60, 0xd5, 0x6e, 0x38, 0x43, 0xde, 0x22, 0x73, 0x87,
        0x90, 0xeb, 0xaa, 0xaf, 0x20, 0xe2, 0xb0, 0x1d, 0x4f, 0xc2, 0x2c, 0x8f, 0x34, 0x86, 0xea, 0x75, 0x02, 0x03,
        0x01, 0x00, 0x01, 0x02, 0x82, 0x01, 0x00, 0x44, 0x90, 0xb8, 0x8f, 0xa5, 0x45, 0x05, 0x28, 0xeb, 0x27, 0x46,
        0xf3, 0xed, 0xa5, 0xa7, 0xb8, 0x8b, 0xe6, 0xd2, 0x7b, 0xc9, 0x1a, 0x2f, 0xf3, 0x1f, 0x0a, 0x28, 0x2f, 0x71,
        0x8f, 0xc7, 0xba, 0x7d, 0x4e, 0x81, 0xec, 0xad, 0xd2, 0x54, 0x0f, 0x7f, 0x1f, 0x86, 0x9e, 0xa0, 0x51, 0xa7,
        0x1e, 0x84, 0x0b, 0xe1, 0x9a, 0x62, 0x24, 0x16, 0x39, 0xac, 0x69, 0x21, 0x4f, 0x91, 0xb3, 0xe9, 0x48, 0x6e,
        0x2a, 0x67, 0xa3, 0x16, 0x82, 0x37, 0xf3, 0x85, 0xf2, 0xf5, 0x40, 0x49, 0xd5, 0x59, 0xe3, 0x23, 0xcd, 0x58,
        0x2a, 0xf5, 0xa6, 0x77, 0x8c, 0xa1, 0x5b, 0x10, 0x28, 0x49, 0xb5, 0xb8, 0x72, 0x19, 0x55, 0xc6, 0x11, 0x65,
        0x58, 0x3e, 0x14, 0xc5, 0xc4, 0x2d, 0xc8, 0xf5, 0x48, 0x7e, 0xd7, 0xd2, 0x5b, 0x54, 0xf5, 0x89, 0x00, 0x10,
        0x5e, 0xef, 0x3b, 0x78, 0xca, 0x1d, 0xe9, 0xe5, 0xbb, 0x55, 0x69, 0x72, 0x30, 0xa8, 0x9c, 0x62, 0x40, 0x46,
        0x07, 0x6a, 0x21, 0x23, 0x48, 0x56, 0xf1, 0xc8, 0x71, 0xdf, 0xad, 0x73, 0xf7, 0xa4, 0x1c, 0xa7, 0x18, 0x40,
        0xc8, 0x10, 0x1f, 0x9e, 0x1c, 0x6e, 0x4e, 0x02, 0x85, 0x61, 0x24, 0x55, 0x7f, 0x06, 0x12, 0x3a, 0x31, 0xd8,
        0x3c, 0xeb, 0xe8, 0xce, 0x65, 0x3b, 0x5a, 0x3d, 0x22, 0x51, 0x14, 0xfe, 0xd4, 0xc3, 0x38, 0x88, 0xef, 0x18,
        0x94, 0x10, 0xee, 0x64, 0x42, 0x40, 0xae, 0xcd, 0xd6, 0x01, 0xd2, 0x1e, 0xa6, 0x60, 0xaa, 0xea, 0xc9, 0xf3,
        0x38, 0x02, 0x7a, 0x63, 0xd8, 0x84, 0xd0, 0x41, 0xad, 0x8b, 0xd4, 0x06, 0x88, 0x0a, 0x3a, 0x9d, 0xaf, 0xe7,
        0x58, 0x07, 0xd5, 0x95, 0x14, 0x8c, 0xc9, 0x2f, 0xc2, 0xd4, 0x60, 0xb4, 0xa0, 0xcd, 0x0c, 0x9e, 0x94, 0x4a,
        0x48, 0xb5, 0xb4, 0xb6, 0xf2, 0xd5, 0xbe, 0xd2, 0x46, 0xf3, 0x51, 0x02, 0x81, 0x81, 0x00, 0xee, 0x5c, 0xc6,
        0xa5, 0xd9, 0x40, 0x2b, 0x05, 0x8d, 0x28, 0xf7, 0x36, 0x60, 0x86, 0xed, 0x50, 0xda, 0x26, 0x0f, 0xf7, 0x8e,
        0xaf, 0xb4, 0xf3, 0x61, 0xe7, 0x58, 0xc4, 0x9f, 0x3c, 0x48, 0x6e, 0x76, 0x4c, 0x78, 0xe0, 0x13, 0x73, 0xee,
        0xa6, 0x81, 0x77, 0xc1, 0x91, 0x63, 0x76, 0xd9, 0x70, 0xc7, 0x5b, 0xb8, 0x9e, 0xcc, 0x65, 0x55, 0xee, 0x74,
        0x14, 0x14, 0xc2, 0x37, 0x9b, 0x36, 0x15, 0x5e, 0x3f, 0xf1, 0x83, 0xfd, 0xf3, 0x4c, 0xe2, 0xb3, 0xe1, 0xed,
        0x50, 0x2e, 0x69, 0x58, 0x23, 0xb7, 0x3b, 0x2e, 0xbe, 0x0e, 0x34, 0xa3, 0x2b, 0xdb, 0x2d, 0xfa, 0x61, 0xb2,
        0xcd, 0x88, 0xe5, 0xde, 0x8a, 0x55, 0xa9, 0xc4, 0x19, 0x90, 0x78, 0xf5, 0x2c, 0xfa, 0x8d, 0xc4, 0x19, 0xaf,
        0x16, 0x90, 0xe0, 0x02, 0xd5, 0x59, 0x7d, 0xd2, 0x92, 0x77, 0x2b, 0xb3, 0x66, 0x98, 0xfc, 0xb3, 0x9b, 0x02,
        0x81, 0x81, 0x00, 0xe7, 0x5a, 0xe3, 0x10, 0x56, 0xbf, 0x8a, 0x32, 0x0b, 0xa7, 0x53, 0xf9, 0xbc, 0xa9, 0xfc,
        0x6f, 0x7a, 0x48, 0x7d, 0x01, 0x52, 0xb1, 0x4b, 0x17, 0xe4, 0xd5, 0xd3, 0xcb, 0x7d, 0x5f, 0xff, 0x65, 0x30,
        0x55, 0x5e, 0x3d, 0xd5, 0xd8, 0xcc, 0xc8, 0xdc, 0xa1, 0xb5, 0xa4, 0x5c, 0xad, 0x73, 0xfd, 0x09, 0x8a, 0x6a,
        0xdf, 0xca, 0x35, 0xc6, 0xf5, 0x1a, 0xc5, 0xed, 0xa1, 0x94, 0xd0, 0xff, 0x8e, 0x20, 0x63, 0x04, 0x77, 0xec,
        0x0b, 0x5d, 0xe8, 0x50, 0xe5, 0x73, 0xf1, 0x3a, 0xc0, 0xcf, 0x10, 0xca, 0x03, 0x36, 0xc6, 0x2d, 0xc3, 0x93,
        0xda, 0xda, 0xe0, 0xc4, 0xc1, 0x5b, 0x47, 0xc1, 0x33, 0xfa, 0x3b, 0xab, 0xd7, 0x24, 0x1b, 0x3e, 0x7a, 0x0a,
        0x66, 0xb0, 0x7b, 0x4a, 0x8a, 0x40, 0x91, 0xc5, 0x6a, 0x66, 0xfe, 0x24, 0xb3, 0x42, 0xcb, 0xbb, 0xe0, 0x4b,
        0x7c, 0x41, 0x57, 0x63, 0x2f, 0x02, 0x81, 0x81, 0x00, 0xa4, 0xdf, 0x31, 0x5c, 0x38, 0x28, 0x45, 0x59, 0xc2,
        0xa9, 0x0a, 0x4d, 0xe7, 0x78, 0x8c, 0x9f, 0xf7, 0x34, 0x8a, 0xa8, 0xce, 0x5e, 0x44, 0xc8, 0x6f, 0xf8, 0xc8,
        0x92, 0xc0, 0x1d, 0xbf, 0x70, 0x00, 0x8d, 0xa6, 0xb2, 0x3f, 0x62, 0x5a, 0x39, 0x7b, 0xa5, 0xed, 0x12, 0xf6,
        0x7c, 0x97, 0xac, 0x85, 0x88, 0xb0, 0xeb, 0xce, 0x2f, 0x6d, 0xbf, 0xd1, 0x34, 0xae, 0xa3, 0x24, 0x39, 0x4c,
        0xb0, 0x7d, 0x0f, 0xb7, 0xab, 0x77, 0xb5, 0x99, 0x81, 0xd9, 0xb0, 0xb5, 0x28, 0x57, 0xe1, 0xef, 0xe0, 0x4c,
        0x76, 0x38, 0x3f, 0xa7, 0xad, 0xcb, 0x0b, 0xa3, 0xc0, 0x6a, 0xc6, 0xb7, 0x19, 0xa9, 0xce, 0x6e, 0x1e, 0xbb,
        0x60, 0x00, 0xcf, 0x39, 0xfa, 0x20, 0x84, 0x2b, 0x0e, 0x72, 0x0c, 0xdd, 0xe9, 0xba, 0xed, 0xe7, 0xa7, 0xd1,
        0x0d, 0xd1, 0xe0, 0x13, 0x63, 0xfb, 0xe4, 0x44, 0x7f, 0xce, 0x6f, 0x02, 0x81, 0x81, 0x00, 0xac, 0x0c, 0x71,
        0xe9, 0xb7, 0xa9, 0x4f, 0x7b, 0x32, 0x21, 0x68, 0x98, 0xc3, 0x0d, 0xe2, 0xb5, 0x80, 0x49, 0xa1, 0xf4, 0xb6,
        0xeb, 0x33, 0xfd, 0xfb, 0xe6, 0x6c, 0x4f, 0xda, 0xd7, 0xe6, 0x14, 0xf9, 0x21, 0xb3, 0x28, 0xe6, 0xfc, 0x08,
        0x26, 0xa3, 0xb4, 0xfa, 0x60, 0xd5, 0xaf, 0x04, 0x1f, 0xbb, 0xd5, 0x9c, 0xee, 0xf9, 0xf0, 0x8e, 0x19, 0xbe,
        0xa4, 0x4c, 0xb8, 0xa9, 0xf3, 0xd6, 0xe8, 0x79, 0xfb, 0x48, 0xda, 0x69, 0xc6, 0x76, 0x3a, 0x8a, 0xd6, 0x68,
        0x27, 0x8f, 0xda, 0xcc, 0xe2, 0x1e, 0x68, 0xcf, 0x76, 0x07, 0x98, 0x77, 0x3e, 0xfd, 0x20, 0xc4, 0x11, 0x4a,
        0xf1, 0x8c, 0xa3, 0x3b, 0xc6, 0xde, 0x5e, 0xea, 0xf1, 0xfb, 0xbf, 0x44, 0x36, 0xe3, 0xad, 0x7c, 0x5c, 0x5d,
        0xf2, 0x49, 0xce, 0x7b, 0xf3, 0x29, 0x95, 0xc9, 0xe9, 0xba, 0xb8, 0xed, 0x49, 0xe5, 0x49, 0xb8, 0x6f, 0x02,
        0x81, 0x80, 0x71, 0x11, 0x8a, 0x2e, 0x38, 0xcf, 0x54, 0xb9, 0x99, 0x5b, 0x95, 0x74, 0x17, 0x7e, 0xe7, 0x53,
        0x59, 0x67, 0xfe, 0xc7, 0x90, 0x84, 0x5b, 0x1c, 0x89, 0x80, 0xa6, 0xa4, 0xb4, 0x71, 0x21, 0xde, 0x27, 0x9e,
        0xb3, 0x58, 0x01, 0xed, 0x93, 0xdb, 0x39, 0xec, 0x0b, 0x6b, 0xc0, 0x18, 0x56, 0x3a, 0x9b, 0x36, 0x04, 0xbf,
        0xaf, 0xf6, 0x94, 0x16, 0x3a, 0x41, 0x6c, 0x2a, 0x2f, 0xf0, 0x80, 0xb1, 0x73, 0x2f, 0x3a, 0x4a, 0xe1, 0x9d,
        0x6b, 0x5d, 0x0b, 0x0c, 0x55, 0xfc, 0xde, 0xc6, 0xf2, 0x32, 0x6f, 0x17, 0x86, 0x4b, 0x5f, 0xc8, 0x2d, 0xcb,
        0xe7, 0x88, 0xab, 0x55, 0x6e, 0x66, 0x35, 0x40, 0xdc, 0x03, 0xcb, 0x3d, 0xf8, 0x39, 0x68, 0x79, 0x39, 0x54,
        0x94, 0x92, 0x2b, 0xf0, 0x9f, 0xd1, 0x00, 0x30, 0xbd, 0xae, 0x9a, 0x87, 0x2d, 0xa6, 0x73, 0x71, 0xdb, 0xe9,
        0x20, 0xc8, 0x55, 0xb3,
    };

    struct aws_array_list output_list;

    ASSERT_SUCCESS(aws_pem_objects_init_from_file_path(&output_list, allocator, "unittests.key"));
    ASSERT_UINT_EQUALS(1, aws_array_list_length(&output_list));

    struct aws_pem_object *pem_object = NULL;
    aws_array_list_get_at_ptr(&output_list, (void **)&pem_object, 0);
    ASSERT_BIN_ARRAYS_EQUALS(s_expected, sizeof(s_expected), pem_object->data.buffer, pem_object->data.len);
    ASSERT_CURSOR_VALUE_CSTRING_EQUALS(aws_byte_cursor_from_string(pem_object->type_string), "RSA PRIVATE KEY");
    ASSERT_INT_EQUALS(AWS_PEM_TYPE_PRIVATE_RSA_PKCS1, pem_object->type);

    aws_pem_objects_clean_up(&output_list);

    return AWS_OP_SUCCESS;
}
AWS_TEST_CASE(test_pem_private_key_parse_from_file, s_test_pem_private_key_parse_from_file)

static int s_test_pem_single_cert_parse(struct aws_allocator *allocator, void *ctx) {
    (void)ctx;
    static const char *s_rsa_1024_sha224_client_crt_pem =
        "-----BEGIN CERTIFICATE-----\n"
        "MIICeDCCAeGgAwIBAgIJAObttnPKQhVlMA0GCSqGSIb3DQEBDgUAMF8xCzAJBgNV\n"
        "BAYTAlVTMQswCQYDVQQIDAJXQTEQMA4GA1UEBwwHU2VhdHRsZTEPMA0GA1UECgwG\n"
        "QW1hem9uMQwwCgYDVQQLDANzMm4xEjAQBgNVBAMMCWxvY2FsaG9zdDAgFw0xNzA4\n"
        "MDEyMjQzMzJaGA8yMTE3MDcwODIyNDMzMlowXzELMAkGA1UEBhMCVVMxCzAJBgNV\n"
        "BAgMAldBMRAwDgYDVQQHDAdTZWF0dGxlMQ8wDQYDVQQKDAZBbWF6b24xDDAKBgNV\n"
        "BAsMA3MybjESMBAGA1UEAwwJbG9jYWxob3N0MIGfMA0GCSqGSIb3DQEBAQUAA4GN\n"
        "ADCBiQKBgQCisRoXXcTh4ejn/sUjGosLlE7GlpLGtvWFEEX6Vl3klVoQdkyabLIH\n"
        "7bHB2P7uyt9bPzeqvWYjuepDBSQUUeb6Mkqfx237bTy8JhXIfpIhbgksTk7IPzgo\n"
        "XLPl1oNl7uB9HQaDQ7UPlaKbfp1gNvs6uGOH4vvyhhJGiblNJKnVwwIDAQABozow\n"
        "ODALBgNVHQ8EBAMCBDAwEwYDVR0lBAwwCgYIKwYBBQUHAwEwFAYDVR0RBA0wC4IJ\n"
        "MTI3LjAuMC4xMA0GCSqGSIb3DQEBDgUAA4GBACleH44LSYhzHHaV70VbnLbtbv8T\n"
        "eaUvzstFW6YvdP1XnZKssZNdvMhoiMuMD5n40/iPbv+grtjxacRQCinLk1SEjpsu\n"
        "3lw90Ds0Ksd/Pdsv7d0cCiJkjadON+ZQEEJ2FP/G19KZFxC3GLk9sxIUXyUW0TXn\n"
        "YxwtPz26+xvPRWCS\n"
        "-----END CERTIFICATE-----";

    static const uint8_t s_expected[] = {
        0x30, 0x82, 0x02, 0x78, 0x30, 0x82, 0x01, 0xe1, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x09, 0x00, 0xe6, 0xed,
        0xb6, 0x73, 0xca, 0x42, 0x15, 0x65, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
        0x0e, 0x05, 0x00, 0x30, 0x5f, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53,
        0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x02, 0x57, 0x41, 0x31, 0x10, 0x30, 0x0e, 0x06,
        0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65, 0x61, 0x74, 0x74, 0x6c, 0x65, 0x31, 0x0f, 0x30, 0x0d, 0x06,
        0x03, 0x55, 0x04, 0x0a, 0x0c, 0x06, 0x41, 0x6d, 0x61, 0x7a, 0x6f, 0x6e, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03,
        0x55, 0x04, 0x0b, 0x0c, 0x03, 0x73, 0x32, 0x6e, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c,
        0x09, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x37, 0x30, 0x38,
        0x30, 0x31, 0x32, 0x32, 0x34, 0x33, 0x33, 0x32, 0x5a, 0x18, 0x0f, 0x32, 0x31, 0x31, 0x37, 0x30, 0x37, 0x30,
        0x38, 0x32, 0x32, 0x34, 0x33, 0x33, 0x32, 0x5a, 0x30, 0x5f, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04,
        0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x02, 0x57, 0x41,
        0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65, 0x61, 0x74, 0x74, 0x6c, 0x65,
        0x31, 0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x06, 0x41, 0x6d, 0x61, 0x7a, 0x6f, 0x6e, 0x31,
        0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x03, 0x73, 0x32, 0x6e, 0x31, 0x12, 0x30, 0x10, 0x06,
        0x03, 0x55, 0x04, 0x03, 0x0c, 0x09, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74, 0x30, 0x81, 0x9f,
        0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d,
        0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xa2, 0xb1, 0x1a, 0x17, 0x5d, 0xc4, 0xe1, 0xe1, 0xe8, 0xe7,
        0xfe, 0xc5, 0x23, 0x1a, 0x8b, 0x0b, 0x94, 0x4e, 0xc6, 0x96, 0x92, 0xc6, 0xb6, 0xf5, 0x85, 0x10, 0x45, 0xfa,
        0x56, 0x5d, 0xe4, 0x95, 0x5a, 0x10, 0x76, 0x4c, 0x9a, 0x6c, 0xb2, 0x07, 0xed, 0xb1, 0xc1, 0xd8, 0xfe, 0xee,
        0xca, 0xdf, 0x5b, 0x3f, 0x37, 0xaa, 0xbd, 0x66, 0x23, 0xb9, 0xea, 0x43, 0x05, 0x24, 0x14, 0x51, 0xe6, 0xfa,
        0x32, 0x4a, 0x9f, 0xc7, 0x6d, 0xfb, 0x6d, 0x3c, 0xbc, 0x26, 0x15, 0xc8, 0x7e, 0x92, 0x21, 0x6e, 0x09, 0x2c,
        0x4e, 0x4e, 0xc8, 0x3f, 0x38, 0x28, 0x5c, 0xb3, 0xe5, 0xd6, 0x83, 0x65, 0xee, 0xe0, 0x7d, 0x1d, 0x06, 0x83,
        0x43, 0xb5, 0x0f, 0x95, 0xa2, 0x9b, 0x7e, 0x9d, 0x60, 0x36, 0xfb, 0x3a, 0xb8, 0x63, 0x87, 0xe2, 0xfb, 0xf2,
        0x86, 0x12, 0x46, 0x89, 0xb9, 0x4d, 0x24, 0xa9, 0xd5, 0xc3, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x3a, 0x30,
        0x38, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04, 0x03, 0x02, 0x04, 0x30, 0x30, 0x13, 0x06, 0x03,
        0x55, 0x1d, 0x25, 0x04, 0x0c, 0x30, 0x0a, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30,
        0x14, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x0d, 0x30, 0x0b, 0x82, 0x09, 0x31, 0x32, 0x37, 0x2e, 0x30, 0x2e,
        0x30, 0x2e, 0x31, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0e, 0x05, 0x00,
        0x03, 0x81, 0x81, 0x00, 0x29, 0x5e, 0x1f, 0x8e, 0x0b, 0x49, 0x88, 0x73, 0x1c, 0x76, 0x95, 0xef, 0x45, 0x5b,
        0x9c, 0xb6, 0xed, 0x6e, 0xff, 0x13, 0x79, 0xa5, 0x2f, 0xce, 0xcb, 0x45, 0x5b, 0xa6, 0x2f, 0x74, 0xfd, 0x57,
        0x9d, 0x92, 0xac, 0xb1, 0x93, 0x5d, 0xbc, 0xc8, 0x68, 0x88, 0xcb, 0x8c, 0x0f, 0x99, 0xf8, 0xd3, 0xf8, 0x8f,
        0x6e, 0xff, 0xa0, 0xae, 0xd8, 0xf1, 0x69, 0xc4, 0x50, 0x0a, 0x29, 0xcb, 0x93, 0x54, 0x84, 0x8e, 0x9b, 0x2e,
        0xde, 0x5c, 0x3d, 0xd0, 0x3b, 0x34, 0x2a, 0xc7, 0x7f, 0x3d, 0xdb, 0x2f, 0xed, 0xdd, 0x1c, 0x0a, 0x22, 0x64,
        0x8d, 0xa7, 0x4e, 0x37, 0xe6, 0x50, 0x10, 0x42, 0x76, 0x14, 0xff, 0xc6, 0xd7, 0xd2, 0x99, 0x17, 0x10, 0xb7,
        0x18, 0xb9, 0x3d, 0xb3, 0x12, 0x14, 0x5f, 0x25, 0x16, 0xd1, 0x35, 0xe7, 0x63, 0x1c, 0x2d, 0x3f, 0x3d, 0xba,
        0xfb, 0x1b, 0xcf, 0x45, 0x60, 0x92};

    struct aws_byte_cursor pem_data = aws_byte_cursor_from_c_str(s_rsa_1024_sha224_client_crt_pem);
    struct aws_array_list output_list;

    ASSERT_SUCCESS(aws_pem_objects_init_from_file_contents(&output_list, allocator, pem_data));
    ASSERT_UINT_EQUALS(1, aws_array_list_length(&output_list));

    struct aws_pem_object *pem_object = NULL;
    aws_array_list_get_at_ptr(&output_list, (void **)&pem_object, 0);
    ASSERT_BIN_ARRAYS_EQUALS(s_expected, sizeof(s_expected), pem_object->data.buffer, pem_object->data.len);
    ASSERT_CURSOR_VALUE_CSTRING_EQUALS(aws_byte_cursor_from_string(pem_object->type_string), "CERTIFICATE");
    ASSERT_INT_EQUALS(AWS_PEM_TYPE_X509, pem_object->type);

    aws_pem_objects_clean_up(&output_list);

    return AWS_OP_SUCCESS;
}
AWS_TEST_CASE(test_pem_single_cert_parse, s_test_pem_single_cert_parse)

static int s_test_pem_cert_chain_parse(struct aws_allocator *allocator, void *ctx) {
    (void)ctx;

    static const char *s_rsa_2048_pkcs1_crt_pem = "-----BEGIN CERTIFICATE-----\n"
                                                  "MIICrTCCAZUCAn3VMA0GCSqGSIb3DQEBBQUAMB4xHDAaBgNVBAMME3MyblRlc3RJ\n"
                                                  "bnRlcm1lZGlhdGUwIBcNMTYwMzMwMTg1NzQzWhgPMjExNjAzMDYxODU3NDNaMBgx\n"
                                                  "FjAUBgNVBAMMDXMyblRlc3RTZXJ2ZXIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw\n"
                                                  "ggEKAoIBAQDRw6AuYXAeRT0YuptCfJjRB/EDJyyGXnv+8TV2H1WJWhMLk8qND27r\n"
                                                  "79A6EjbVmJaOV9qrokVqpDmXS712Z3BDprJ+1LFMymm3A+AFuK/skeGy0skik+Tg\n"
                                                  "MmFT5XBVvmsw4uB1S9uUqktHauXgjhFPPsfvk4ewL4LulVEN2TEeI1Odj4CaMxAO\n"
                                                  "Iuowm8wI2OHVzRHlrRmyJ9hYGuHHQ2TaTGIjr3WpAFuXi9pHGGMYa0uXAVPmgjdE\n"
                                                  "XZ8t46u/ZKQ9W1uJkZEVKhcijT7G2VBrsBUq0CDiL+TDaGfthnBzUc9zt4fx/S/3\n"
                                                  "qulC2WbKI3xrasQyjrsHTAJ75Md3rK09AgMBAAEwDQYJKoZIhvcNAQEFBQADggEB\n"
                                                  "AHHkXNA9BtgAebZC2zriW4hRfeIkJMOwvfKBXHTuY5iCLD1otis6AZljcCKXM6O9\n"
                                                  "489eHBC4T6mJwVsXhH+/ccEKqNRD2bUfQgOij32PsteV1eOHfHIFqdJmnBVb8tYa\n"
                                                  "jxUvy7UQvXrPqaHbODrHe+7f7r1YCzerujiP5SSHphY3GQq88KemfFczp/4GnYas\n"
                                                  "sE50OYe7DQcB4zvnxmAXp51JIN4ooktUU9oKIM5y2cgEWdmJzeqPANYxf0ZIPlTg\n"
                                                  "ETknKw1Dzf8wlK5mFbbG4LPQh1mkDVcwQV3ogG6kGMRa7neH+6SFkNpAKuPCoje4\n"
                                                  "NAE+WQ5ve1wk7nIRTQwDAF4=\n"
                                                  "-----END CERTIFICATE-----\n"
                                                  "-----BEGIN CERTIFICATE-----\n"
                                                  "MIIDKTCCAhGgAwIBAgICVxYwDQYJKoZIhvcNAQEFBQAwFjEUMBIGA1UEAwwLczJu\n"
                                                  "VGVzdFJvb3QwIBcNMTYwMzMwMTg1NzA5WhgPMjExNjAzMDYxODU3MDlaMB4xHDAa\n"
                                                  "BgNVBAMME3MyblRlc3RJbnRlcm1lZGlhdGUwggEiMA0GCSqGSIb3DQEBAQUAA4IB\n"
                                                  "DwAwggEKAoIBAQDM/i3eclxYcvedPCEnVe6A/HYsYPeP1qKBZQhbpuuX061jFZKw\n"
                                                  "lecb0eau1PORLbcsYK40u3xUzoA5u6Q0ebDuqPbqSJkCazsh66cu9STl8ubbk7oI\n"
                                                  "8LJjUJFhhy2Jmm9krXhPyRscU+CXOCZ2G1GhBqTI8cgMYhEVHwb3qy1EHg6G3n4W\n"
                                                  "AjV+cKQcbUytq8DRmVe0bNJxDOX8ivzfAp3lUIwub+JfpxrWIUhb3iVGj5CauI98\n"
                                                  "bNFHTWwYp7tviIIi21Q+L3nExCyE4yTUP/mebBZ62JnbvsWSs3r3//Am5d8G3WdY\n"
                                                  "BXsERoDoLBvHnqlO/oo4ppGCRI7GkDroACi/AgMBAAGjdzB1MAwGA1UdEwQFMAMB\n"
                                                  "Af8wHQYDVR0OBBYEFGqUKVWVlL03sHuOggFACdlHckPBMEYGA1UdIwQ/MD2AFE2X\n"
                                                  "AbNDryMlBpMNI6Ce927uUFwToRqkGDAWMRQwEgYDVQQDDAtzMm5UZXN0Um9vdIIJ\n"
                                                  "ANDUkH+UYdz1MA0GCSqGSIb3DQEBBQUAA4IBAQA3O3S9VT0EC1yG4xyNNUZ7+CzF\n"
                                                  "uFA6uiO38ygcN5Nz1oNPy2eQer7vYmrHtqN6gS/o1Ag5F8bLRCqeuZTsOG80O29H\n"
                                                  "kNhs5xYprdU82AqcaWwEd0kDrhC5rEvs6fj1J0NKmmhbovYxuDboj0a7If7HEqX0\n"
                                                  "NizyU3M3JONPZgadchZ+F5DosatF1Bpt/gsQRy383IogQ0/FS+juHCCc4VIUemuk\n"
                                                  "YY1J8o5XdrGWrPBBiudTWqCobe+N541b+YLWbajT5UKzvSqJmcqpPTniJGc9eZxc\n"
                                                  "z3cCNd3cKa9bK51stEnQSlA7PQXYs3K+TD3EmSn/G2x6Hmfr7lrpbIhEaD+y\n"
                                                  "-----END CERTIFICATE-----\n"
                                                  "-----BEGIN CERTIFICATE-----\n"
                                                  "MIIDATCCAemgAwIBAgIJANDUkH+UYdz1MA0GCSqGSIb3DQEBCwUAMBYxFDASBgNV\n"
                                                  "BAMMC3MyblRlc3RSb290MCAXDTE2MDMzMDE4NTYzOVoYDzIxMTYwMzA2MTg1NjM5\n"
                                                  "WjAWMRQwEgYDVQQDDAtzMm5UZXN0Um9vdDCCASIwDQYJKoZIhvcNAQEBBQADggEP\n"
                                                  "ADCCAQoCggEBAMY5532000oaeed7Jmo3ssx1723ZDLpn3WGz6FxpWM0zsKA/YvdD\n"
                                                  "7J6qXDvfxU6dZlmsCS+bSNAqpARKmKsBEDPTsdLmrN1V1clOxvKm6GvU1eloRTw6\n"
                                                  "xukEUXJ+uxrQMLYvSJBiCBVGI+UYNCK5c6guNMRYBCGdk5/iayjmK0Nxz1918Cx9\n"
                                                  "z4va8HPAgYIz0ogOdYB21O9FQGPdH1mYqRzljcSsZ7EFo1P8HJr8oKK76ZeYi2or\n"
                                                  "pjzMHGnlufHaul508wQPeFAMa1Tku3HyGZRaieRAck6+QcO2NujXxKNyCBlWON23\n"
                                                  "FQTuBjN/CAl74MZtcAM2hVSmpm9t4cWVN5MCAwEAAaNQME4wHQYDVR0OBBYEFE2X\n"
                                                  "AbNDryMlBpMNI6Ce927uUFwTMB8GA1UdIwQYMBaAFE2XAbNDryMlBpMNI6Ce927u\n"
                                                  "UFwTMAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQELBQADggEBAAXkVvQdXDmozPix\n"
                                                  "uZi1o9cw4Si0syqfJ4sSunrzPbbmw/Qxhth5V7XGrnsQVNxamgnbzpjGhiBF6isM\n"
                                                  "ldj33zQYtke+ojOjFlhEvrPo6eW29RkLBEtJadGs2bkMLztJbf+cbH2u6irzr6S4\n"
                                                  "3OgVOSuB+zG56ksTnEVmum+C/8tSIAyi3eaoStPcgEU8+3/KMrH7uuenmTOCKdD1\n"
                                                  "FvSDHXT9qPgTttVQGXbXzJEr5tGE+Py6yib5uoJ0dJZNtjs7HOQEDk5J0wZaX0DC\n"
                                                  "MShYLiN5qLJAk0qwl+js488BJ18M9dg4TxdBYFkwHSzKXSj9TJN77Bb0RZr8LL9T\n"
                                                  "r9IyvfU=\n"
                                                  "-----END CERTIFICATE-----";

    static const uint8_t s_expected_intermediate_1[] = {
        0x30, 0x82, 0x02, 0xad, 0x30, 0x82, 0x01, 0x95, 0x02, 0x02, 0x7d, 0xd5, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
        0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x1e, 0x31, 0x1c, 0x30, 0x1a, 0x06, 0x03, 0x55,
        0x04, 0x03, 0x0c, 0x13, 0x73, 0x32, 0x6e, 0x54, 0x65, 0x73, 0x74, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65,
        0x64, 0x69, 0x61, 0x74, 0x65, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x36, 0x30, 0x33, 0x33, 0x30, 0x31, 0x38, 0x35,
        0x37, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x32, 0x31, 0x31, 0x36, 0x30, 0x33, 0x30, 0x36, 0x31, 0x38, 0x35, 0x37,
        0x34, 0x33, 0x5a, 0x30, 0x18, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0d, 0x73, 0x32,
        0x6e, 0x54, 0x65, 0x73, 0x74, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06,
        0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30,
        0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xd1, 0xc3, 0xa0, 0x2e, 0x61, 0x70, 0x1e, 0x45, 0x3d, 0x18,
        0xba, 0x9b, 0x42, 0x7c, 0x98, 0xd1, 0x07, 0xf1, 0x03, 0x27, 0x2c, 0x86, 0x5e, 0x7b, 0xfe, 0xf1, 0x35, 0x76,
        0x1f, 0x55, 0x89, 0x5a, 0x13, 0x0b, 0x93, 0xca, 0x8d, 0x0f, 0x6e, 0xeb, 0xef, 0xd0, 0x3a, 0x12, 0x36, 0xd5,
        0x98, 0x96, 0x8e, 0x57, 0xda, 0xab, 0xa2, 0x45, 0x6a, 0xa4, 0x39, 0x97, 0x4b, 0xbd, 0x76, 0x67, 0x70, 0x43,
        0xa6, 0xb2, 0x7e, 0xd4, 0xb1, 0x4c, 0xca, 0x69, 0xb7, 0x03, 0xe0, 0x05, 0xb8, 0xaf, 0xec, 0x91, 0xe1, 0xb2,
        0xd2, 0xc9, 0x22, 0x93, 0xe4, 0xe0, 0x32, 0x61, 0x53, 0xe5, 0x70, 0x55, 0xbe, 0x6b, 0x30, 0xe2, 0xe0, 0x75,
        0x4b, 0xdb, 0x94, 0xaa, 0x4b, 0x47, 0x6a, 0xe5, 0xe0, 0x8e, 0x11, 0x4f, 0x3e, 0xc7, 0xef, 0x93, 0x87, 0xb0,
        0x2f, 0x82, 0xee, 0x95, 0x51, 0x0d, 0xd9, 0x31, 0x1e, 0x23, 0x53, 0x9d, 0x8f, 0x80, 0x9a, 0x33, 0x10, 0x0e,
        0x22, 0xea, 0x30, 0x9b, 0xcc, 0x08, 0xd8, 0xe1, 0xd5, 0xcd, 0x11, 0xe5, 0xad, 0x19, 0xb2, 0x27, 0xd8, 0x58,
        0x1a, 0xe1, 0xc7, 0x43, 0x64, 0xda, 0x4c, 0x62, 0x23, 0xaf, 0x75, 0xa9, 0x00, 0x5b, 0x97, 0x8b, 0xda, 0x47,
        0x18, 0x63, 0x18, 0x6b, 0x4b, 0x97, 0x01, 0x53, 0xe6, 0x82, 0x37, 0x44, 0x5d, 0x9f, 0x2d, 0xe3, 0xab, 0xbf,
        0x64, 0xa4, 0x3d, 0x5b, 0x5b, 0x89, 0x91, 0x91, 0x15, 0x2a, 0x17, 0x22, 0x8d, 0x3e, 0xc6, 0xd9, 0x50, 0x6b,
        0xb0, 0x15, 0x2a, 0xd0, 0x20, 0xe2, 0x2f, 0xe4, 0xc3, 0x68, 0x67, 0xed, 0x86, 0x70, 0x73, 0x51, 0xcf, 0x73,
        0xb7, 0x87, 0xf1, 0xfd, 0x2f, 0xf7, 0xaa, 0xe9, 0x42, 0xd9, 0x66, 0xca, 0x23, 0x7c, 0x6b, 0x6a, 0xc4, 0x32,
        0x8e, 0xbb, 0x07, 0x4c, 0x02, 0x7b, 0xe4, 0xc7, 0x77, 0xac, 0xad, 0x3d, 0x02, 0x03, 0x01, 0x00, 0x01, 0x30,
        0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01,
        0x00, 0x71, 0xe4, 0x5c, 0xd0, 0x3d, 0x06, 0xd8, 0x00, 0x79, 0xb6, 0x42, 0xdb, 0x3a, 0xe2, 0x5b, 0x88, 0x51,
        0x7d, 0xe2, 0x24, 0x24, 0xc3, 0xb0, 0xbd, 0xf2, 0x81, 0x5c, 0x74, 0xee, 0x63, 0x98, 0x82, 0x2c, 0x3d, 0x68,
        0xb6, 0x2b, 0x3a, 0x01, 0x99, 0x63, 0x70, 0x22, 0x97, 0x33, 0xa3, 0xbd, 0xe3, 0xcf, 0x5e, 0x1c, 0x10, 0xb8,
        0x4f, 0xa9, 0x89, 0xc1, 0x5b, 0x17, 0x84, 0x7f, 0xbf, 0x71, 0xc1, 0x0a, 0xa8, 0xd4, 0x43, 0xd9, 0xb5, 0x1f,
        0x42, 0x03, 0xa2, 0x8f, 0x7d, 0x8f, 0xb2, 0xd7, 0x95, 0xd5, 0xe3, 0x87, 0x7c, 0x72, 0x05, 0xa9, 0xd2, 0x66,
        0x9c, 0x15, 0x5b, 0xf2, 0xd6, 0x1a, 0x8f, 0x15, 0x2f, 0xcb, 0xb5, 0x10, 0xbd, 0x7a, 0xcf, 0xa9, 0xa1, 0xdb,
        0x38, 0x3a, 0xc7, 0x7b, 0xee, 0xdf, 0xee, 0xbd, 0x58, 0x0b, 0x37, 0xab, 0xba, 0x38, 0x8f, 0xe5, 0x24, 0x87,
        0xa6, 0x16, 0x37, 0x19, 0x0a, 0xbc, 0xf0, 0xa7, 0xa6, 0x7c, 0x57, 0x33, 0xa7, 0xfe, 0x06, 0x9d, 0x86, 0xac,
        0xb0, 0x4e, 0x74, 0x39, 0x87, 0xbb, 0x0d, 0x07, 0x01, 0xe3, 0x3b, 0xe7, 0xc6, 0x60, 0x17, 0xa7, 0x9d, 0x49,
        0x20, 0xde, 0x28, 0xa2, 0x4b, 0x54, 0x53, 0xda, 0x0a, 0x20, 0xce, 0x72, 0xd9, 0xc8, 0x04, 0x59, 0xd9, 0x89,
        0xcd, 0xea, 0x8f, 0x00, 0xd6, 0x31, 0x7f, 0x46, 0x48, 0x3e, 0x54, 0xe0, 0x11, 0x39, 0x27, 0x2b, 0x0d, 0x43,
        0xcd, 0xff, 0x30, 0x94, 0xae, 0x66, 0x15, 0xb6, 0xc6, 0xe0, 0xb3, 0xd0, 0x87, 0x59, 0xa4, 0x0d, 0x57, 0x30,
        0x41, 0x5d, 0xe8, 0x80, 0x6e, 0xa4, 0x18, 0xc4, 0x5a, 0xee, 0x77, 0x87, 0xfb, 0xa4, 0x85, 0x90, 0xda, 0x40,
        0x2a, 0xe3, 0xc2, 0xa2, 0x37, 0xb8, 0x34, 0x01, 0x3e, 0x59, 0x0e, 0x6f, 0x7b, 0x5c, 0x24, 0xee, 0x72, 0x11,
        0x4d, 0x0c, 0x03, 0x00, 0x5e};

    static const uint8_t s_expected_intermediate_2[] = {
        0x30, 0x82, 0x03, 0x29, 0x30, 0x82, 0x02, 0x11, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x02, 0x57, 0x16, 0x30,
        0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x16, 0x31, 0x14,
        0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0b, 0x73, 0x32, 0x6e, 0x54, 0x65, 0x73, 0x74, 0x52, 0x6f,
        0x6f, 0x74, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x36, 0x30, 0x33, 0x33, 0x30, 0x31, 0x38, 0x35, 0x37, 0x30, 0x39,
        0x5a, 0x18, 0x0f, 0x32, 0x31, 0x31, 0x36, 0x30, 0x33, 0x30, 0x36, 0x31, 0x38, 0x35, 0x37, 0x30, 0x39, 0x5a,
        0x30, 0x1e, 0x31, 0x1c, 0x30, 0x1a, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x13, 0x73, 0x32, 0x6e, 0x54, 0x65,
        0x73, 0x74, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x74, 0x65, 0x30, 0x82, 0x01, 0x22,
        0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01,
        0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xcc, 0xfe, 0x2d, 0xde, 0x72, 0x5c, 0x58,
        0x72, 0xf7, 0x9d, 0x3c, 0x21, 0x27, 0x55, 0xee, 0x80, 0xfc, 0x76, 0x2c, 0x60, 0xf7, 0x8f, 0xd6, 0xa2, 0x81,
        0x65, 0x08, 0x5b, 0xa6, 0xeb, 0x97, 0xd3, 0xad, 0x63, 0x15, 0x92, 0xb0, 0x95, 0xe7, 0x1b, 0xd1, 0xe6, 0xae,
        0xd4, 0xf3, 0x91, 0x2d, 0xb7, 0x2c, 0x60, 0xae, 0x34, 0xbb, 0x7c, 0x54, 0xce, 0x80, 0x39, 0xbb, 0xa4, 0x34,
        0x79, 0xb0, 0xee, 0xa8, 0xf6, 0xea, 0x48, 0x99, 0x02, 0x6b, 0x3b, 0x21, 0xeb, 0xa7, 0x2e, 0xf5, 0x24, 0xe5,
        0xf2, 0xe6, 0xdb, 0x93, 0xba, 0x08, 0xf0, 0xb2, 0x63, 0x50, 0x91, 0x61, 0x87, 0x2d, 0x89, 0x9a, 0x6f, 0x64,
        0xad, 0x78, 0x4f, 0xc9, 0x1b, 0x1c, 0x53, 0xe0, 0x97, 0x38, 0x26, 0x76, 0x1b, 0x51, 0xa1, 0x06, 0xa4, 0xc8,
        0xf1, 0xc8, 0x0c, 0x62, 0x11, 0x15, 0x1f, 0x06, 0xf7, 0xab, 0x2d, 0x44, 0x1e, 0x0e, 0x86, 0xde, 0x7e, 0x16,
        0x02, 0x35, 0x7e, 0x70, 0xa4, 0x1c, 0x6d, 0x4c, 0xad, 0xab, 0xc0, 0xd1, 0x99, 0x57, 0xb4, 0x6c, 0xd2, 0x71,
        0x0c, 0xe5, 0xfc, 0x8a, 0xfc, 0xdf, 0x02, 0x9d, 0xe5, 0x50, 0x8c, 0x2e, 0x6f, 0xe2, 0x5f, 0xa7, 0x1a, 0xd6,
        0x21, 0x48, 0x5b, 0xde, 0x25, 0x46, 0x8f, 0x90, 0x9a, 0xb8, 0x8f, 0x7c, 0x6c, 0xd1, 0x47, 0x4d, 0x6c, 0x18,
        0xa7, 0xbb, 0x6f, 0x88, 0x82, 0x22, 0xdb, 0x54, 0x3e, 0x2f, 0x79, 0xc4, 0xc4, 0x2c, 0x84, 0xe3, 0x24, 0xd4,
        0x3f, 0xf9, 0x9e, 0x6c, 0x16, 0x7a, 0xd8, 0x99, 0xdb, 0xbe, 0xc5, 0x92, 0xb3, 0x7a, 0xf7, 0xff, 0xf0, 0x26,
        0xe5, 0xdf, 0x06, 0xdd, 0x67, 0x58, 0x05, 0x7b, 0x04, 0x46, 0x80, 0xe8, 0x2c, 0x1b, 0xc7, 0x9e, 0xa9, 0x4e,
        0xfe, 0x8a, 0x38, 0xa6, 0x91, 0x82, 0x44, 0x8e, 0xc6, 0x90, 0x3a, 0xe8, 0x00, 0x28, 0xbf, 0x02, 0x03, 0x01,
        0x00, 0x01, 0xa3, 0x77, 0x30, 0x75, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01,
        0x01, 0xff, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x6a, 0x94, 0x29, 0x55, 0x95,
        0x94, 0xbd, 0x37, 0xb0, 0x7b, 0x8e, 0x82, 0x01, 0x40, 0x09, 0xd9, 0x47, 0x72, 0x43, 0xc1, 0x30, 0x46, 0x06,
        0x03, 0x55, 0x1d, 0x23, 0x04, 0x3f, 0x30, 0x3d, 0x80, 0x14, 0x4d, 0x97, 0x01, 0xb3, 0x43, 0xaf, 0x23, 0x25,
        0x06, 0x93, 0x0d, 0x23, 0xa0, 0x9e, 0xf7, 0x6e, 0xee, 0x50, 0x5c, 0x13, 0xa1, 0x1a, 0xa4, 0x18, 0x30, 0x16,
        0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0b, 0x73, 0x32, 0x6e, 0x54, 0x65, 0x73, 0x74,
        0x52, 0x6f, 0x6f, 0x74, 0x82, 0x09, 0x00, 0xd0, 0xd4, 0x90, 0x7f, 0x94, 0x61, 0xdc, 0xf5, 0x30, 0x0d, 0x06,
        0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x37,
        0x3b, 0x74, 0xbd, 0x55, 0x3d, 0x04, 0x0b, 0x5c, 0x86, 0xe3, 0x1c, 0x8d, 0x35, 0x46, 0x7b, 0xf8, 0x2c, 0xc5,
        0xb8, 0x50, 0x3a, 0xba, 0x23, 0xb7, 0xf3, 0x28, 0x1c, 0x37, 0x93, 0x73, 0xd6, 0x83, 0x4f, 0xcb, 0x67, 0x90,
        0x7a, 0xbe, 0xef, 0x62, 0x6a, 0xc7, 0xb6, 0xa3, 0x7a, 0x81, 0x2f, 0xe8, 0xd4, 0x08, 0x39, 0x17, 0xc6, 0xcb,
        0x44, 0x2a, 0x9e, 0xb9, 0x94, 0xec, 0x38, 0x6f, 0x34, 0x3b, 0x6f, 0x47, 0x90, 0xd8, 0x6c, 0xe7, 0x16, 0x29,
        0xad, 0xd5, 0x3c, 0xd8, 0x0a, 0x9c, 0x69, 0x6c, 0x04, 0x77, 0x49, 0x03, 0xae, 0x10, 0xb9, 0xac, 0x4b, 0xec,
        0xe9, 0xf8, 0xf5, 0x27, 0x43, 0x4a, 0x9a, 0x68, 0x5b, 0xa2, 0xf6, 0x31, 0xb8, 0x36, 0xe8, 0x8f, 0x46, 0xbb,
        0x21, 0xfe, 0xc7, 0x12, 0xa5, 0xf4, 0x36, 0x2c, 0xf2, 0x53, 0x73, 0x37, 0x24, 0xe3, 0x4f, 0x66, 0x06, 0x9d,
        0x72, 0x16, 0x7e, 0x17, 0x90, 0xe8, 0xb1, 0xab, 0x45, 0xd4, 0x1a, 0x6d, 0xfe, 0x0b, 0x10, 0x47, 0x2d, 0xfc,
        0xdc, 0x8a, 0x20, 0x43, 0x4f, 0xc5, 0x4b, 0xe8, 0xee, 0x1c, 0x20, 0x9c, 0xe1, 0x52, 0x14, 0x7a, 0x6b, 0xa4,
        0x61, 0x8d, 0x49, 0xf2, 0x8e, 0x57, 0x76, 0xb1, 0x96, 0xac, 0xf0, 0x41, 0x8a, 0xe7, 0x53, 0x5a, 0xa0, 0xa8,
        0x6d, 0xef, 0x8d, 0xe7, 0x8d, 0x5b, 0xf9, 0x82, 0xd6, 0x6d, 0xa8, 0xd3, 0xe5, 0x42, 0xb3, 0xbd, 0x2a, 0x89,
        0x99, 0xca, 0xa9, 0x3d, 0x39, 0xe2, 0x24, 0x67, 0x3d, 0x79, 0x9c, 0x5c, 0xcf, 0x77, 0x02, 0x35, 0xdd, 0xdc,
        0x29, 0xaf, 0x5b, 0x2b, 0x9d, 0x6c, 0xb4, 0x49, 0xd0, 0x4a, 0x50, 0x3b, 0x3d, 0x05, 0xd8, 0xb3, 0x72, 0xbe,
        0x4c, 0x3d, 0xc4, 0x99, 0x29, 0xff, 0x1b, 0x6c, 0x7a, 0x1e, 0x67, 0xeb, 0xee, 0x5a, 0xe9, 0x6c, 0x88, 0x44,
        0x68, 0x3f, 0xb2};

    static const uint8_t s_expected_leaf[] = {
        0x30, 0x82, 0x03, 0x01, 0x30, 0x82, 0x01, 0xe9, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x09, 0x00, 0xd0, 0xd4,
        0x90, 0x7f, 0x94, 0x61, 0xdc, 0xf5, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
        0x0b, 0x05, 0x00, 0x30, 0x16, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0b, 0x73, 0x32,
        0x6e, 0x54, 0x65, 0x73, 0x74, 0x52, 0x6f, 0x6f, 0x74, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x36, 0x30, 0x33, 0x33,
        0x30, 0x31, 0x38, 0x35, 0x36, 0x33, 0x39, 0x5a, 0x18, 0x0f, 0x32, 0x31, 0x31, 0x36, 0x30, 0x33, 0x30, 0x36,
        0x31, 0x38, 0x35, 0x36, 0x33, 0x39, 0x5a, 0x30, 0x16, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x03,
        0x0c, 0x0b, 0x73, 0x32, 0x6e, 0x54, 0x65, 0x73, 0x74, 0x52, 0x6f, 0x6f, 0x74, 0x30, 0x82, 0x01, 0x22, 0x30,
        0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f,
        0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xc6, 0x39, 0xe7, 0x7d, 0xb4, 0xd3, 0x4a, 0x1a,
        0x79, 0xe7, 0x7b, 0x26, 0x6a, 0x37, 0xb2, 0xcc, 0x75, 0xef, 0x6d, 0xd9, 0x0c, 0xba, 0x67, 0xdd, 0x61, 0xb3,
        0xe8, 0x5c, 0x69, 0x58, 0xcd, 0x33, 0xb0, 0xa0, 0x3f, 0x62, 0xf7, 0x43, 0xec, 0x9e, 0xaa, 0x5c, 0x3b, 0xdf,
        0xc5, 0x4e, 0x9d, 0x66, 0x59, 0xac, 0x09, 0x2f, 0x9b, 0x48, 0xd0, 0x2a, 0xa4, 0x04, 0x4a, 0x98, 0xab, 0x01,
        0x10, 0x33, 0xd3, 0xb1, 0xd2, 0xe6, 0xac, 0xdd, 0x55, 0xd5, 0xc9, 0x4e, 0xc6, 0xf2, 0xa6, 0xe8, 0x6b, 0xd4,
        0xd5, 0xe9, 0x68, 0x45, 0x3c, 0x3a, 0xc6, 0xe9, 0x04, 0x51, 0x72, 0x7e, 0xbb, 0x1a, 0xd0, 0x30, 0xb6, 0x2f,
        0x48, 0x90, 0x62, 0x08, 0x15, 0x46, 0x23, 0xe5, 0x18, 0x34, 0x22, 0xb9, 0x73, 0xa8, 0x2e, 0x34, 0xc4, 0x58,
        0x04, 0x21, 0x9d, 0x93, 0x9f, 0xe2, 0x6b, 0x28, 0xe6, 0x2b, 0x43, 0x71, 0xcf, 0x5f, 0x75, 0xf0, 0x2c, 0x7d,
        0xcf, 0x8b, 0xda, 0xf0, 0x73, 0xc0, 0x81, 0x82, 0x33, 0xd2, 0x88, 0x0e, 0x75, 0x80, 0x76, 0xd4, 0xef, 0x45,
        0x40, 0x63, 0xdd, 0x1f, 0x59, 0x98, 0xa9, 0x1c, 0xe5, 0x8d, 0xc4, 0xac, 0x67, 0xb1, 0x05, 0xa3, 0x53, 0xfc,
        0x1c, 0x9a, 0xfc, 0xa0, 0xa2, 0xbb, 0xe9, 0x97, 0x98, 0x8b, 0x6a, 0x2b, 0xa6, 0x3c, 0xcc, 0x1c, 0x69, 0xe5,
        0xb9, 0xf1, 0xda, 0xba, 0x5e, 0x74, 0xf3, 0x04, 0x0f, 0x78, 0x50, 0x0c, 0x6b, 0x54, 0xe4, 0xbb, 0x71, 0xf2,
        0x19, 0x94, 0x5a, 0x89, 0xe4, 0x40, 0x72, 0x4e, 0xbe, 0x41, 0xc3, 0xb6, 0x36, 0xe8, 0xd7, 0xc4, 0xa3, 0x72,
        0x08, 0x19, 0x56, 0x38, 0xdd, 0xb7, 0x15, 0x04, 0xee, 0x06, 0x33, 0x7f, 0x08, 0x09, 0x7b, 0xe0, 0xc6, 0x6d,
        0x70, 0x03, 0x36, 0x85, 0x54, 0xa6, 0xa6, 0x6f, 0x6d, 0xe1, 0xc5, 0x95, 0x37, 0x93, 0x02, 0x03, 0x01, 0x00,
        0x01, 0xa3, 0x50, 0x30, 0x4e, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x4d, 0x97,
        0x01, 0xb3, 0x43, 0xaf, 0x23, 0x25, 0x06, 0x93, 0x0d, 0x23, 0xa0, 0x9e, 0xf7, 0x6e, 0xee, 0x50, 0x5c, 0x13,
        0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x4d, 0x97, 0x01, 0xb3, 0x43,
        0xaf, 0x23, 0x25, 0x06, 0x93, 0x0d, 0x23, 0xa0, 0x9e, 0xf7, 0x6e, 0xee, 0x50, 0x5c, 0x13, 0x30, 0x0c, 0x06,
        0x03, 0x55, 0x1d, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48,
        0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x05, 0xe4, 0x56, 0xf4, 0x1d,
        0x5c, 0x39, 0xa8, 0xcc, 0xf8, 0xb1, 0xb9, 0x98, 0xb5, 0xa3, 0xd7, 0x30, 0xe1, 0x28, 0xb4, 0xb3, 0x2a, 0x9f,
        0x27, 0x8b, 0x12, 0xba, 0x7a, 0xf3, 0x3d, 0xb6, 0xe6, 0xc3, 0xf4, 0x31, 0x86, 0xd8, 0x79, 0x57, 0xb5, 0xc6,
        0xae, 0x7b, 0x10, 0x54, 0xdc, 0x5a, 0x9a, 0x09, 0xdb, 0xce, 0x98, 0xc6, 0x86, 0x20, 0x45, 0xea, 0x2b, 0x0c,
        0x95, 0xd8, 0xf7, 0xdf, 0x34, 0x18, 0xb6, 0x47, 0xbe, 0xa2, 0x33, 0xa3, 0x16, 0x58, 0x44, 0xbe, 0xb3, 0xe8,
        0xe9, 0xe5, 0xb6, 0xf5, 0x19, 0x0b, 0x04, 0x4b, 0x49, 0x69, 0xd1, 0xac, 0xd9, 0xb9, 0x0c, 0x2f, 0x3b, 0x49,
        0x6d, 0xff, 0x9c, 0x6c, 0x7d, 0xae, 0xea, 0x2a, 0xf3, 0xaf, 0xa4, 0xb8, 0xdc, 0xe8, 0x15, 0x39, 0x2b, 0x81,
        0xfb, 0x31, 0xb9, 0xea, 0x4b, 0x13, 0x9c, 0x45, 0x66, 0xba, 0x6f, 0x82, 0xff, 0xcb, 0x52, 0x20, 0x0c, 0xa2,
        0xdd, 0xe6, 0xa8, 0x4a, 0xd3, 0xdc, 0x80, 0x45, 0x3c, 0xfb, 0x7f, 0xca, 0x32, 0xb1, 0xfb, 0xba, 0xe7, 0xa7,
        0x99, 0x33, 0x82, 0x29, 0xd0, 0xf5, 0x16, 0xf4, 0x83, 0x1d, 0x74, 0xfd, 0xa8, 0xf8, 0x13, 0xb6, 0xd5, 0x50,
        0x19, 0x76, 0xd7, 0xcc, 0x91, 0x2b, 0xe6, 0xd1, 0x84, 0xf8, 0xfc, 0xba, 0xca, 0x26, 0xf9, 0xba, 0x82, 0x74,
        0x74, 0x96, 0x4d, 0xb6, 0x3b, 0x3b, 0x1c, 0xe4, 0x04, 0x0e, 0x4e, 0x49, 0xd3, 0x06, 0x5a, 0x5f, 0x40, 0xc2,
        0x31, 0x28, 0x58, 0x2e, 0x23, 0x79, 0xa8, 0xb2, 0x40, 0x93, 0x4a, 0xb0, 0x97, 0xe8, 0xec, 0xe3, 0xcf, 0x01,
        0x27, 0x5f, 0x0c, 0xf5, 0xd8, 0x38, 0x4f, 0x17, 0x41, 0x60, 0x59, 0x30, 0x1d, 0x2c, 0xca, 0x5d, 0x28, 0xfd,
        0x4c, 0x93, 0x7b, 0xec, 0x16, 0xf4, 0x45, 0x9a, 0xfc, 0x2c, 0xbf, 0x53, 0xaf, 0xd2, 0x32, 0xbd, 0xf5,
    };

    struct aws_byte_cursor pem_data = aws_byte_cursor_from_c_str(s_rsa_2048_pkcs1_crt_pem);
    struct aws_array_list output_list;

    ASSERT_SUCCESS(aws_pem_objects_init_from_file_contents(&output_list, allocator, pem_data));
    ASSERT_UINT_EQUALS(3, aws_array_list_length(&output_list));

    struct aws_pem_object *pem_object = NULL;
    aws_array_list_get_at_ptr(&output_list, (void **)&pem_object, 0);
    ASSERT_BIN_ARRAYS_EQUALS(
        s_expected_intermediate_1, sizeof(s_expected_intermediate_1), pem_object->data.buffer, pem_object->data.len);
    ASSERT_CURSOR_VALUE_CSTRING_EQUALS(aws_byte_cursor_from_string(pem_object->type_string), "CERTIFICATE");
    ASSERT_INT_EQUALS(AWS_PEM_TYPE_X509, pem_object->type);
    aws_array_list_get_at_ptr(&output_list, (void **)&pem_object, 1);
    ASSERT_BIN_ARRAYS_EQUALS(
        s_expected_intermediate_2, sizeof(s_expected_intermediate_2), pem_object->data.buffer, pem_object->data.len);
    ASSERT_CURSOR_VALUE_CSTRING_EQUALS(aws_byte_cursor_from_string(pem_object->type_string), "CERTIFICATE");
    ASSERT_INT_EQUALS(AWS_PEM_TYPE_X509, pem_object->type);
    aws_array_list_get_at_ptr(&output_list, (void **)&pem_object, 2);
    ASSERT_BIN_ARRAYS_EQUALS(s_expected_leaf, sizeof(s_expected_leaf), pem_object->data.buffer, pem_object->data.len);
    ASSERT_CURSOR_VALUE_CSTRING_EQUALS(aws_byte_cursor_from_string(pem_object->type_string), "CERTIFICATE");
    ASSERT_INT_EQUALS(AWS_PEM_TYPE_X509, pem_object->type);

    aws_pem_objects_clean_up(&output_list);

    return AWS_OP_SUCCESS;
}
AWS_TEST_CASE(test_pem_cert_chain_parse, s_test_pem_cert_chain_parse)

static int s_test_pem_private_key_parse(struct aws_allocator *allocator, void *ctx) {
    (void)ctx;
    static const char *s_private_key_pem = "-----BEGIN RSA PRIVATE KEY-----\n"
                                           "MIIEpAIBAAKCAQEA12pXSPgORAMlQtYRbxuz/Ocaoran3C2Fjyjhu0vucSEZSwxD\n"
                                           "Jp75TBQEMafSpSEKAQLeDt7xuDRDYn52V4UE6cF+xTWhtzsf7mhN/lHaDPcvR2AS\n"
                                           "PAEkzkil8KCLY4e6tTxSwQ97splNuEZ099HoJYTTLFaReIfd1D3zZ1EYcSw8w+GZ\n"
                                           "2SxEUfYUSL2CFmIYSkQjnlsJCIpCoGgDiBAPbIUJO3KWBDX0JgGDbx3Wf3jXG/Y6\n"
                                           "T63LPsO+AS20RCvcEF0F/rlDINzI5EAHO1TOEd9fKOu+JAK06Pw1m77BgOrE7Ftv\n"
                                           "IG7kYNVuOEPeInOHkOuqryDisB1PwiyPNIbqdQIDAQABAoIBAESQuI+lRQUo6ydG\n"
                                           "8+2lp7iL5tJ7yRov8x8KKC9xj8e6fU6B7K3SVA9/H4aeoFGnHoQL4ZpiJBY5rGkh\n"
                                           "T5Gz6UhuKmejFoI384Xy9UBJ1VnjI81YKvWmd4yhWxAoSbW4chlVxhFlWD4UxcQt\n"
                                           "yPVIftfSW1T1iQAQXu87eMod6eW7VWlyMKicYkBGB2ohI0hW8chx361z96QcpxhA\n"
                                           "yBAfnhxuTgKFYSRVfwYSOjHYPOvozmU7Wj0iURT+1MM4iO8YlBDuZEJArs3WAdIe\n"
                                           "pmCq6snzOAJ6Y9iE0EGti9QGiAo6na/nWAfVlRSMyS/C1GC0oM0MnpRKSLW0tvLV\n"
                                           "vtJG81ECgYEA7lzGpdlAKwWNKPc2YIbtUNomD/eOr7TzYedYxJ88SG52THjgE3Pu\n"
                                           "poF3wZFjdtlwx1u4nsxlVe50FBTCN5s2FV4/8YP980zis+HtUC5pWCO3Oy6+DjSj\n"
                                           "K9st+mGyzYjl3opVqcQZkHj1LPqNxBmvFpDgAtVZfdKSdyuzZpj8s5sCgYEA51rj\n"
                                           "EFa/ijILp1P5vKn8b3pIfQFSsUsX5NXTy31f/2UwVV491djMyNyhtaRcrXP9CYpq\n"
                                           "38o1xvUaxe2hlND/jiBjBHfsC13oUOVz8TrAzxDKAzbGLcOT2trgxMFbR8Ez+jur\n"
                                           "1yQbPnoKZrB7SopAkcVqZv4ks0LLu+BLfEFXYy8CgYEApN8xXDgoRVnCqQpN53iM\n"
                                           "n/c0iqjOXkTIb/jIksAdv3AAjaayP2JaOXul7RL2fJeshYiw684vbb/RNK6jJDlM\n"
                                           "sH0Pt6t3tZmB2bC1KFfh7+BMdjg/p63LC6PAasa3GanObh67YADPOfoghCsOcgzd\n"
                                           "6brt56fRDdHgE2P75ER/zm8CgYEArAxx6bepT3syIWiYww3itYBJofS26zP9++Zs\n"
                                           "T9rX5hT5IbMo5vwIJqO0+mDVrwQfu9Wc7vnwjhm+pEy4qfPW6Hn7SNppxnY6itZo\n"
                                           "J4/azOIeaM92B5h3Pv0gxBFK8YyjO8beXurx+79ENuOtfFxd8knOe/Mplcnpurjt\n"
                                           "SeVJuG8CgYBxEYouOM9UuZlblXQXfudTWWf+x5CEWxyJgKaktHEh3iees1gB7ZPb\n"
                                           "OewLa8AYVjqbNgS/r/aUFjpBbCov8ICxcy86SuGda10LDFX83sbyMm8XhktfyC3L\n"
                                           "54irVW5mNUDcA8s9+DloeTlUlJIr8J/RADC9rpqHLaZzcdvpIMhVsw==\n"
                                           "-----END RSA PRIVATE KEY-----";

    static const uint8_t s_expected[] = {
        0x30, 0x82, 0x04, 0xa4, 0x02, 0x01, 0x00, 0x02, 0x82, 0x01, 0x01, 0x00, 0xd7, 0x6a, 0x57, 0x48, 0xf8, 0x0e,
        0x44, 0x03, 0x25, 0x42, 0xd6, 0x11, 0x6f, 0x1b, 0xb3, 0xfc, 0xe7, 0x1a, 0xa2, 0xb6, 0xa7, 0xdc, 0x2d, 0x85,
        0x8f, 0x28, 0xe1, 0xbb, 0x4b, 0xee, 0x71, 0x21, 0x19, 0x4b, 0x0c, 0x43, 0x26, 0x9e, 0xf9, 0x4c, 0x14, 0x04,
        0x31, 0xa7, 0xd2, 0xa5, 0x21, 0x0a, 0x01, 0x02, 0xde, 0x0e, 0xde, 0xf1, 0xb8, 0x34, 0x43, 0x62, 0x7e, 0x76,
        0x57, 0x85, 0x04, 0xe9, 0xc1, 0x7e, 0xc5, 0x35, 0xa1, 0xb7, 0x3b, 0x1f, 0xee, 0x68, 0x4d, 0xfe, 0x51, 0xda,
        0x0c, 0xf7, 0x2f, 0x47, 0x60, 0x12, 0x3c, 0x01, 0x24, 0xce, 0x48, 0xa5, 0xf0, 0xa0, 0x8b, 0x63, 0x87, 0xba,
        0xb5, 0x3c, 0x52, 0xc1, 0x0f, 0x7b, 0xb2, 0x99, 0x4d, 0xb8, 0x46, 0x74, 0xf7, 0xd1, 0xe8, 0x25, 0x84, 0xd3,
        0x2c, 0x56, 0x91, 0x78, 0x87, 0xdd, 0xd4, 0x3d, 0xf3, 0x67, 0x51, 0x18, 0x71, 0x2c, 0x3c, 0xc3, 0xe1, 0x99,
        0xd9, 0x2c, 0x44, 0x51, 0xf6, 0x14, 0x48, 0xbd, 0x82, 0x16, 0x62, 0x18, 0x4a, 0x44, 0x23, 0x9e, 0x5b, 0x09,
        0x08, 0x8a, 0x42, 0xa0, 0x68, 0x03, 0x88, 0x10, 0x0f, 0x6c, 0x85, 0x09, 0x3b, 0x72, 0x96, 0x04, 0x35, 0xf4,
        0x26, 0x01, 0x83, 0x6f, 0x1d, 0xd6, 0x7f, 0x78, 0xd7, 0x1b, 0xf6, 0x3a, 0x4f, 0xad, 0xcb, 0x3e, 0xc3, 0xbe,
        0x01, 0x2d, 0xb4, 0x44, 0x2b, 0xdc, 0x10, 0x5d, 0x05, 0xfe, 0xb9, 0x43, 0x20, 0xdc, 0xc8, 0xe4, 0x40, 0x07,
        0x3b, 0x54, 0xce, 0x11, 0xdf, 0x5f, 0x28, 0xeb, 0xbe, 0x24, 0x02, 0xb4, 0xe8, 0xfc, 0x35, 0x9b, 0xbe, 0xc1,
        0x80, 0xea, 0xc4, 0xec, 0x5b, 0x6f, 0x20, 0x6e, 0xe4, 0x60, 0xd5, 0x6e, 0x38, 0x43, 0xde, 0x22, 0x73, 0x87,
        0x90, 0xeb, 0xaa, 0xaf, 0x20, 0xe2, 0xb0, 0x1d, 0x4f, 0xc2, 0x2c, 0x8f, 0x34, 0x86, 0xea, 0x75, 0x02, 0x03,
        0x01, 0x00, 0x01, 0x02, 0x82, 0x01, 0x00, 0x44, 0x90, 0xb8, 0x8f, 0xa5, 0x45, 0x05, 0x28, 0xeb, 0x27, 0x46,
        0xf3, 0xed, 0xa5, 0xa7, 0xb8, 0x8b, 0xe6, 0xd2, 0x7b, 0xc9, 0x1a, 0x2f, 0xf3, 0x1f, 0x0a, 0x28, 0x2f, 0x71,
        0x8f, 0xc7, 0xba, 0x7d, 0x4e, 0x81, 0xec, 0xad, 0xd2, 0x54, 0x0f, 0x7f, 0x1f, 0x86, 0x9e, 0xa0, 0x51, 0xa7,
        0x1e, 0x84, 0x0b, 0xe1, 0x9a, 0x62, 0x24, 0x16, 0x39, 0xac, 0x69, 0x21, 0x4f, 0x91, 0xb3, 0xe9, 0x48, 0x6e,
        0x2a, 0x67, 0xa3, 0x16, 0x82, 0x37, 0xf3, 0x85, 0xf2, 0xf5, 0x40, 0x49, 0xd5, 0x59, 0xe3, 0x23, 0xcd, 0x58,
        0x2a, 0xf5, 0xa6, 0x77, 0x8c, 0xa1, 0x5b, 0x10, 0x28, 0x49, 0xb5, 0xb8, 0x72, 0x19, 0x55, 0xc6, 0x11, 0x65,
        0x58, 0x3e, 0x14, 0xc5, 0xc4, 0x2d, 0xc8, 0xf5, 0x48, 0x7e, 0xd7, 0xd2, 0x5b, 0x54, 0xf5, 0x89, 0x00, 0x10,
        0x5e, 0xef, 0x3b, 0x78, 0xca, 0x1d, 0xe9, 0xe5, 0xbb, 0x55, 0x69, 0x72, 0x30, 0xa8, 0x9c, 0x62, 0x40, 0x46,
        0x07, 0x6a, 0x21, 0x23, 0x48, 0x56, 0xf1, 0xc8, 0x71, 0xdf, 0xad, 0x73, 0xf7, 0xa4, 0x1c, 0xa7, 0x18, 0x40,
        0xc8, 0x10, 0x1f, 0x9e, 0x1c, 0x6e, 0x4e, 0x02, 0x85, 0x61, 0x24, 0x55, 0x7f, 0x06, 0x12, 0x3a, 0x31, 0xd8,
        0x3c, 0xeb, 0xe8, 0xce, 0x65, 0x3b, 0x5a, 0x3d, 0x22, 0x51, 0x14, 0xfe, 0xd4, 0xc3, 0x38, 0x88, 0xef, 0x18,
        0x94, 0x10, 0xee, 0x64, 0x42, 0x40, 0xae, 0xcd, 0xd6, 0x01, 0xd2, 0x1e, 0xa6, 0x60, 0xaa, 0xea, 0xc9, 0xf3,
        0x38, 0x02, 0x7a, 0x63, 0xd8, 0x84, 0xd0, 0x41, 0xad, 0x8b, 0xd4, 0x06, 0x88, 0x0a, 0x3a, 0x9d, 0xaf, 0xe7,
        0x58, 0x07, 0xd5, 0x95, 0x14, 0x8c, 0xc9, 0x2f, 0xc2, 0xd4, 0x60, 0xb4, 0xa0, 0xcd, 0x0c, 0x9e, 0x94, 0x4a,
        0x48, 0xb5, 0xb4, 0xb6, 0xf2, 0xd5, 0xbe, 0xd2, 0x46, 0xf3, 0x51, 0x02, 0x81, 0x81, 0x00, 0xee, 0x5c, 0xc6,
        0xa5, 0xd9, 0x40, 0x2b, 0x05, 0x8d, 0x28, 0xf7, 0x36, 0x60, 0x86, 0xed, 0x50, 0xda, 0x26, 0x0f, 0xf7, 0x8e,
        0xaf, 0xb4, 0xf3, 0x61, 0xe7, 0x58, 0xc4, 0x9f, 0x3c, 0x48, 0x6e, 0x76, 0x4c, 0x78, 0xe0, 0x13, 0x73, 0xee,
        0xa6, 0x81, 0x77, 0xc1, 0x91, 0x63, 0x76, 0xd9, 0x70, 0xc7, 0x5b, 0xb8, 0x9e, 0xcc, 0x65, 0x55, 0xee, 0x74,
        0x14, 0x14, 0xc2, 0x37, 0x9b, 0x36, 0x15, 0x5e, 0x3f, 0xf1, 0x83, 0xfd, 0xf3, 0x4c, 0xe2, 0xb3, 0xe1, 0xed,
        0x50, 0x2e, 0x69, 0x58, 0x23, 0xb7, 0x3b, 0x2e, 0xbe, 0x0e, 0x34, 0xa3, 0x2b, 0xdb, 0x2d, 0xfa, 0x61, 0xb2,
        0xcd, 0x88, 0xe5, 0xde, 0x8a, 0x55, 0xa9, 0xc4, 0x19, 0x90, 0x78, 0xf5, 0x2c, 0xfa, 0x8d, 0xc4, 0x19, 0xaf,
        0x16, 0x90, 0xe0, 0x02, 0xd5, 0x59, 0x7d, 0xd2, 0x92, 0x77, 0x2b, 0xb3, 0x66, 0x98, 0xfc, 0xb3, 0x9b, 0x02,
        0x81, 0x81, 0x00, 0xe7, 0x5a, 0xe3, 0x10, 0x56, 0xbf, 0x8a, 0x32, 0x0b, 0xa7, 0x53, 0xf9, 0xbc, 0xa9, 0xfc,
        0x6f, 0x7a, 0x48, 0x7d, 0x01, 0x52, 0xb1, 0x4b, 0x17, 0xe4, 0xd5, 0xd3, 0xcb, 0x7d, 0x5f, 0xff, 0x65, 0x30,
        0x55, 0x5e, 0x3d, 0xd5, 0xd8, 0xcc, 0xc8, 0xdc, 0xa1, 0xb5, 0xa4, 0x5c, 0xad, 0x73, 0xfd, 0x09, 0x8a, 0x6a,
        0xdf, 0xca, 0x35, 0xc6, 0xf5, 0x1a, 0xc5, 0xed, 0xa1, 0x94, 0xd0, 0xff, 0x8e, 0x20, 0x63, 0x04, 0x77, 0xec,
        0x0b, 0x5d, 0xe8, 0x50, 0xe5, 0x73, 0xf1, 0x3a, 0xc0, 0xcf, 0x10, 0xca, 0x03, 0x36, 0xc6, 0x2d, 0xc3, 0x93,
        0xda, 0xda, 0xe0, 0xc4, 0xc1, 0x5b, 0x47, 0xc1, 0x33, 0xfa, 0x3b, 0xab, 0xd7, 0x24, 0x1b, 0x3e, 0x7a, 0x0a,
        0x66, 0xb0, 0x7b, 0x4a, 0x8a, 0x40, 0x91, 0xc5, 0x6a, 0x66, 0xfe, 0x24, 0xb3, 0x42, 0xcb, 0xbb, 0xe0, 0x4b,
        0x7c, 0x41, 0x57, 0x63, 0x2f, 0x02, 0x81, 0x81, 0x00, 0xa4, 0xdf, 0x31, 0x5c, 0x38, 0x28, 0x45, 0x59, 0xc2,
        0xa9, 0x0a, 0x4d, 0xe7, 0x78, 0x8c, 0x9f, 0xf7, 0x34, 0x8a, 0xa8, 0xce, 0x5e, 0x44, 0xc8, 0x6f, 0xf8, 0xc8,
        0x92, 0xc0, 0x1d, 0xbf, 0x70, 0x00, 0x8d, 0xa6, 0xb2, 0x3f, 0x62, 0x5a, 0x39, 0x7b, 0xa5, 0xed, 0x12, 0xf6,
        0x7c, 0x97, 0xac, 0x85, 0x88, 0xb0, 0xeb, 0xce, 0x2f, 0x6d, 0xbf, 0xd1, 0x34, 0xae, 0xa3, 0x24, 0x39, 0x4c,
        0xb0, 0x7d, 0x0f, 0xb7, 0xab, 0x77, 0xb5, 0x99, 0x81, 0xd9, 0xb0, 0xb5, 0x28, 0x57, 0xe1, 0xef, 0xe0, 0x4c,
        0x76, 0x38, 0x3f, 0xa7, 0xad, 0xcb, 0x0b, 0xa3, 0xc0, 0x6a, 0xc6, 0xb7, 0x19, 0xa9, 0xce, 0x6e, 0x1e, 0xbb,
        0x60, 0x00, 0xcf, 0x39, 0xfa, 0x20, 0x84, 0x2b, 0x0e, 0x72, 0x0c, 0xdd, 0xe9, 0xba, 0xed, 0xe7, 0xa7, 0xd1,
        0x0d, 0xd1, 0xe0, 0x13, 0x63, 0xfb, 0xe4, 0x44, 0x7f, 0xce, 0x6f, 0x02, 0x81, 0x81, 0x00, 0xac, 0x0c, 0x71,
        0xe9, 0xb7, 0xa9, 0x4f, 0x7b, 0x32, 0x21, 0x68, 0x98, 0xc3, 0x0d, 0xe2, 0xb5, 0x80, 0x49, 0xa1, 0xf4, 0xb6,
        0xeb, 0x33, 0xfd, 0xfb, 0xe6, 0x6c, 0x4f, 0xda, 0xd7, 0xe6, 0x14, 0xf9, 0x21, 0xb3, 0x28, 0xe6, 0xfc, 0x08,
        0x26, 0xa3, 0xb4, 0xfa, 0x60, 0xd5, 0xaf, 0x04, 0x1f, 0xbb, 0xd5, 0x9c, 0xee, 0xf9, 0xf0, 0x8e, 0x19, 0xbe,
        0xa4, 0x4c, 0xb8, 0xa9, 0xf3, 0xd6, 0xe8, 0x79, 0xfb, 0x48, 0xda, 0x69, 0xc6, 0x76, 0x3a, 0x8a, 0xd6, 0x68,
        0x27, 0x8f, 0xda, 0xcc, 0xe2, 0x1e, 0x68, 0xcf, 0x76, 0x07, 0x98, 0x77, 0x3e, 0xfd, 0x20, 0xc4, 0x11, 0x4a,
        0xf1, 0x8c, 0xa3, 0x3b, 0xc6, 0xde, 0x5e, 0xea, 0xf1, 0xfb, 0xbf, 0x44, 0x36, 0xe3, 0xad, 0x7c, 0x5c, 0x5d,
        0xf2, 0x49, 0xce, 0x7b, 0xf3, 0x29, 0x95, 0xc9, 0xe9, 0xba, 0xb8, 0xed, 0x49, 0xe5, 0x49, 0xb8, 0x6f, 0x02,
        0x81, 0x80, 0x71, 0x11, 0x8a, 0x2e, 0x38, 0xcf, 0x54, 0xb9, 0x99, 0x5b, 0x95, 0x74, 0x17, 0x7e, 0xe7, 0x53,
        0x59, 0x67, 0xfe, 0xc7, 0x90, 0x84, 0x5b, 0x1c, 0x89, 0x80, 0xa6, 0xa4, 0xb4, 0x71, 0x21, 0xde, 0x27, 0x9e,
        0xb3, 0x58, 0x01, 0xed, 0x93, 0xdb, 0x39, 0xec, 0x0b, 0x6b, 0xc0, 0x18, 0x56, 0x3a, 0x9b, 0x36, 0x04, 0xbf,
        0xaf, 0xf6, 0x94, 0x16, 0x3a, 0x41, 0x6c, 0x2a, 0x2f, 0xf0, 0x80, 0xb1, 0x73, 0x2f, 0x3a, 0x4a, 0xe1, 0x9d,
        0x6b, 0x5d, 0x0b, 0x0c, 0x55, 0xfc, 0xde, 0xc6, 0xf2, 0x32, 0x6f, 0x17, 0x86, 0x4b, 0x5f, 0xc8, 0x2d, 0xcb,
        0xe7, 0x88, 0xab, 0x55, 0x6e, 0x66, 0x35, 0x40, 0xdc, 0x03, 0xcb, 0x3d, 0xf8, 0x39, 0x68, 0x79, 0x39, 0x54,
        0x94, 0x92, 0x2b, 0xf0, 0x9f, 0xd1, 0x00, 0x30, 0xbd, 0xae, 0x9a, 0x87, 0x2d, 0xa6, 0x73, 0x71, 0xdb, 0xe9,
        0x20, 0xc8, 0x55, 0xb3,
    };

    struct aws_byte_cursor pem_data = aws_byte_cursor_from_c_str(s_private_key_pem);
    struct aws_array_list output_list;

    ASSERT_SUCCESS(aws_pem_objects_init_from_file_contents(&output_list, allocator, pem_data));
    ASSERT_UINT_EQUALS(1, aws_array_list_length(&output_list));

    struct aws_pem_object *pem_object = NULL;
    aws_array_list_get_at_ptr(&output_list, (void **)&pem_object, 0);
    ASSERT_BIN_ARRAYS_EQUALS(s_expected, sizeof(s_expected), pem_object->data.buffer, pem_object->data.len);
    ASSERT_CURSOR_VALUE_CSTRING_EQUALS(aws_byte_cursor_from_string(pem_object->type_string), "RSA PRIVATE KEY");
    ASSERT_INT_EQUALS(AWS_PEM_TYPE_PRIVATE_RSA_PKCS1, pem_object->type);

    aws_pem_objects_clean_up(&output_list);

    return AWS_OP_SUCCESS;
}
AWS_TEST_CASE(test_pem_private_key_parse, s_test_pem_private_key_parse)

static int s_test_pem_cert_chain_comments_and_whitespace(struct aws_allocator *allocator, void *ctx) {
    (void)ctx;

    static const char *s_pem_data_str = "# -----Comment\n"
                                        "// Style\n"
                                        "/* from */\n"
                                        "''' multiple\n"
                                        "!* languages\n"
                                        "\n"
                                        "-----BEGIN CERTIFICATE-----\n"
                                        "MIICrTCCAZUCAn3VMA0GCSqGSIb3DQEBBQUAMB4xHDAaBgNVBAMME3MyblRlc3RJ\n"
                                        "bnRlcm1lZGlhdGUwIBcNMTYwMzMwMTg1NzQzWhgPMjExNjAzMDYxODU3NDNaMBgx\n"
                                        "FjAUBgNVBAMMDXMyblRlc3RTZXJ2ZXIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw\n"
                                        "ggEKAoIBAQDRw6AuYXAeRT0YuptCfJjRB/EDJyyGXnv+8TV2H1WJWhMLk8qND27r\n"
                                        "79A6EjbVmJaOV9qrokVqpDmXS712Z3BDprJ+1LFMymm3A+AFuK/skeGy0skik+Tg\n"
                                        "MmFT5XBVvmsw4uB1S9uUqktHauXgjhFPPsfvk4ewL4LulVEN2TEeI1Odj4CaMxAO\n"
                                        "Iuowm8wI2OHVzRHlrRmyJ9hYGuHHQ2TaTGIjr3WpAFuXi9pHGGMYa0uXAVPmgjdE\n"
                                        "XZ8t46u/ZKQ9W1uJkZEVKhcijT7G2VBrsBUq0CDiL+TDaGfthnBzUc9zt4fx/S/3\n"
                                        "qulC2WbKI3xrasQyjrsHTAJ75Md3rK09AgMBAAEwDQYJKoZIhvcNAQEFBQADggEB\n"
                                        "AHHkXNA9BtgAebZC2zriW4hRfeIkJMOwvfKBXHTuY5iCLD1otis6AZljcCKXM6O9\n"
                                        "489eHBC4T6mJwVsXhH+/ccEKqNRD2bUfQgOij32PsteV1eOHfHIFqdJmnBVb8tYa\n"
                                        "jxUvy7UQvXrPqaHbODrHe+7f7r1YCzerujiP5SSHphY3GQq88KemfFczp/4GnYas\n"
                                        "sE50OYe7DQcB4zvnxmAXp51JIN4ooktUU9oKIM5y2cgEWdmJzeqPANYxf0ZIPlTg\n"
                                        "ETknKw1Dzf8wlK5mFbbG4LPQh1mkDVcwQV3ogG6kGMRa7neH+6SFkNpAKuPCoje4\n"
                                        "NAE+WQ5ve1wk7nIRTQwDAF4=\n"
                                        "-----END CERTIFICATE-----               \n"
                                        "\n"
                                        "\n"
                                        "            \n"
                                        "\n"
                                        "\n"
                                        "              \n"
                                        "\n"
                                        "                 \n"
                                        "\n"
                                        "                 \n"
                                        "    -----BEGIN CERTIFICATE-----\n"
                                        "MIIDKTCCAhGgAwIBAgICVxYwDQYJKoZIhvcNAQEFBQAwFjEUMBIGA1UEAwwLczJu\n"
                                        "VGVzdFJvb3QwIBcNMTYwMzMwMTg1NzA5WhgPMjExNjAzMDYxODU3MDlaMB4xHDAa\n"
                                        "BgNVBAMME3MyblRlc3RJbnRlcm1lZGlhdGUwggEiMA0GCSqGSIb3DQEBAQUAA4IB\n"
                                        "DwAwggEKAoIBAQDM/i3eclxYcvedPCEnVe6A/HYsYPeP1qKBZQhbpuuX061jFZKw\n"
                                        "lecb0eau1PORLbcsYK40u3xUzoA5u6Q0ebDuqPbqSJkCazsh66cu9STl8ubbk7oI\n"
                                        "8LJjUJFhhy2Jmm9krXhPyRscU+CXOCZ2G1GhBqTI8cgMYhEVHwb3qy1EHg6G3n4W\n"
                                        "AjV+cKQcbUytq8DRmVe0bNJxDOX8ivzfAp3lUIwub+JfpxrWIUhb3iVGj5CauI98\n"
                                        "bNFHTWwYp7tviIIi21Q+L3nExCyE4yTUP/mebBZ62JnbvsWSs3r3//Am5d8G3WdY\n"
                                        "BXsERoDoLBvHnqlO/oo4ppGCRI7GkDroACi/AgMBAAGjdzB1MAwGA1UdEwQFMAMB\n"
                                        "Af8wHQYDVR0OBBYEFGqUKVWVlL03sHuOggFACdlHckPBMEYGA1UdIwQ/MD2AFE2X\n"
                                        "AbNDryMlBpMNI6Ce927uUFwToRqkGDAWMRQwEgYDVQQDDAtzMm5UZXN0Um9vdIIJ\n"
                                        "ANDUkH+UYdz1MA0GCSqGSIb3DQEBBQUAA4IBAQA3O3S9VT0EC1yG4xyNNUZ7+CzF\n"
                                        "uFA6uiO38ygcN5Nz1oNPy2eQer7vYmrHtqN6gS/o1Ag5F8bLRCqeuZTsOG80O29H\n"
                                        "kNhs5xYprdU82AqcaWwEd0kDrhC5rEvs6fj1J0NKmmhbovYxuDboj0a7If7HEqX0\n"
                                        "NizyU3M3JONPZgadchZ+F5DosatF1Bpt/gsQRy383IogQ0/FS+juHCCc4VIUemuk\n"
                                        "YY1J8o5XdrGWrPBBiudTWqCobe+N541b+YLWbajT5UKzvSqJmcqpPTniJGc9eZxc\n"
                                        "z3cCNd3cKa9bK51stEnQSlA7PQXYs3K+TD3EmSn/G2x6Hmfr7lrpbIhEaD+y\n"
                                        "-----END CERTIFICATE-----\n"
                                        "-----BEGIN CERTIFICATE-----\n"
                                        "MIIDATCCAemgAwIBAgIJANDUkH+UYdz1MA0GCSqGSIb3DQEBCwUAMBYxFDASBgNV\n"
                                        "BAMMC3MyblRlc3RSb290MCAXDTE2MDMzMDE4NTYzOVoYDzIxMTYwMzA2MTg1NjM5\n"
                                        "WjAWMRQwEgYDVQQDDAtzMm5UZXN0Um9vdDCCASIwDQYJKoZIhvcNAQEBBQADggEP\n"
                                        "ADCCAQoCggEBAMY5532000oaeed7Jmo3ssx1723ZDLpn3WGz6FxpWM0zsKA/YvdD\n"
                                        "7J6qXDvfxU6dZlmsCS+bSNAqpARKmKsBEDPTsdLmrN1V1clOxvKm6GvU1eloRTw6\n"
                                        "xukEUXJ+uxrQMLYvSJBiCBVGI+UYNCK5c6guNMRYBCGdk5/iayjmK0Nxz1918Cx9\n"
                                        "z4va8HPAgYIz0ogOdYB21O9FQGPdH1mYqRzljcSsZ7EFo1P8HJr8oKK76ZeYi2or\n"
                                        "pjzMHGnlufHaul508wQPeFAMa1Tku3HyGZRaieRAck6+QcO2NujXxKNyCBlWON23\n"
                                        "FQTuBjN/CAl74MZtcAM2hVSmpm9t4cWVN5MCAwEAAaNQME4wHQYDVR0OBBYEFE2X\n"
                                        "AbNDryMlBpMNI6Ce927uUFwTMB8GA1UdIwQYMBaAFE2XAbNDryMlBpMNI6Ce927u\n"
                                        "UFwTMAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQELBQADggEBAAXkVvQdXDmozPix\n"
                                        "uZi1o9cw4Si0syqfJ4sSunrzPbbmw/Qxhth5V7XGrnsQVNxamgnbzpjGhiBF6isM\n"
                                        "ldj33zQYtke+ojOjFlhEvrPo6eW29RkLBEtJadGs2bkMLztJbf+cbH2u6irzr6S4\n"
                                        "3OgVOSuB+zG56ksTnEVmum+C/8tSIAyi3eaoStPcgEU8+3/KMrH7uuenmTOCKdD1\n"
                                        "FvSDHXT9qPgTttVQGXbXzJEr5tGE+Py6yib5uoJ0dJZNtjs7HOQEDk5J0wZaX0DC\n"
                                        "MShYLiN5qLJAk0qwl+js488BJ18M9dg4TxdBYFkwHSzKXSj9TJN77Bb0RZr8LL9T\n"
                                        "r9IyvfU=\n"
                                        "-----END CERTIFICATE-----";

    static const uint8_t s_expected_intermediate_1[] = {
        0x30, 0x82, 0x02, 0xad, 0x30, 0x82, 0x01, 0x95, 0x02, 0x02, 0x7d, 0xd5, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
        0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x1e, 0x31, 0x1c, 0x30, 0x1a, 0x06, 0x03, 0x55,
        0x04, 0x03, 0x0c, 0x13, 0x73, 0x32, 0x6e, 0x54, 0x65, 0x73, 0x74, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65,
        0x64, 0x69, 0x61, 0x74, 0x65, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x36, 0x30, 0x33, 0x33, 0x30, 0x31, 0x38, 0x35,
        0x37, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x32, 0x31, 0x31, 0x36, 0x30, 0x33, 0x30, 0x36, 0x31, 0x38, 0x35, 0x37,
        0x34, 0x33, 0x5a, 0x30, 0x18, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0d, 0x73, 0x32,
        0x6e, 0x54, 0x65, 0x73, 0x74, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06,
        0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30,
        0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xd1, 0xc3, 0xa0, 0x2e, 0x61, 0x70, 0x1e, 0x45, 0x3d, 0x18,
        0xba, 0x9b, 0x42, 0x7c, 0x98, 0xd1, 0x07, 0xf1, 0x03, 0x27, 0x2c, 0x86, 0x5e, 0x7b, 0xfe, 0xf1, 0x35, 0x76,
        0x1f, 0x55, 0x89, 0x5a, 0x13, 0x0b, 0x93, 0xca, 0x8d, 0x0f, 0x6e, 0xeb, 0xef, 0xd0, 0x3a, 0x12, 0x36, 0xd5,
        0x98, 0x96, 0x8e, 0x57, 0xda, 0xab, 0xa2, 0x45, 0x6a, 0xa4, 0x39, 0x97, 0x4b, 0xbd, 0x76, 0x67, 0x70, 0x43,
        0xa6, 0xb2, 0x7e, 0xd4, 0xb1, 0x4c, 0xca, 0x69, 0xb7, 0x03, 0xe0, 0x05, 0xb8, 0xaf, 0xec, 0x91, 0xe1, 0xb2,
        0xd2, 0xc9, 0x22, 0x93, 0xe4, 0xe0, 0x32, 0x61, 0x53, 0xe5, 0x70, 0x55, 0xbe, 0x6b, 0x30, 0xe2, 0xe0, 0x75,
        0x4b, 0xdb, 0x94, 0xaa, 0x4b, 0x47, 0x6a, 0xe5, 0xe0, 0x8e, 0x11, 0x4f, 0x3e, 0xc7, 0xef, 0x93, 0x87, 0xb0,
        0x2f, 0x82, 0xee, 0x95, 0x51, 0x0d, 0xd9, 0x31, 0x1e, 0x23, 0x53, 0x9d, 0x8f, 0x80, 0x9a, 0x33, 0x10, 0x0e,
        0x22, 0xea, 0x30, 0x9b, 0xcc, 0x08, 0xd8, 0xe1, 0xd5, 0xcd, 0x11, 0xe5, 0xad, 0x19, 0xb2, 0x27, 0xd8, 0x58,
        0x1a, 0xe1, 0xc7, 0x43, 0x64, 0xda, 0x4c, 0x62, 0x23, 0xaf, 0x75, 0xa9, 0x00, 0x5b, 0x97, 0x8b, 0xda, 0x47,
        0x18, 0x63, 0x18, 0x6b, 0x4b, 0x97, 0x01, 0x53, 0xe6, 0x82, 0x37, 0x44, 0x5d, 0x9f, 0x2d, 0xe3, 0xab, 0xbf,
        0x64, 0xa4, 0x3d, 0x5b, 0x5b, 0x89, 0x91, 0x91, 0x15, 0x2a, 0x17, 0x22, 0x8d, 0x3e, 0xc6, 0xd9, 0x50, 0x6b,
        0xb0, 0x15, 0x2a, 0xd0, 0x20, 0xe2, 0x2f, 0xe4, 0xc3, 0x68, 0x67, 0xed, 0x86, 0x70, 0x73, 0x51, 0xcf, 0x73,
        0xb7, 0x87, 0xf1, 0xfd, 0x2f, 0xf7, 0xaa, 0xe9, 0x42, 0xd9, 0x66, 0xca, 0x23, 0x7c, 0x6b, 0x6a, 0xc4, 0x32,
        0x8e, 0xbb, 0x07, 0x4c, 0x02, 0x7b, 0xe4, 0xc7, 0x77, 0xac, 0xad, 0x3d, 0x02, 0x03, 0x01, 0x00, 0x01, 0x30,
        0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01,
        0x00, 0x71, 0xe4, 0x5c, 0xd0, 0x3d, 0x06, 0xd8, 0x00, 0x79, 0xb6, 0x42, 0xdb, 0x3a, 0xe2, 0x5b, 0x88, 0x51,
        0x7d, 0xe2, 0x24, 0x24, 0xc3, 0xb0, 0xbd, 0xf2, 0x81, 0x5c, 0x74, 0xee, 0x63, 0x98, 0x82, 0x2c, 0x3d, 0x68,
        0xb6, 0x2b, 0x3a, 0x01, 0x99, 0x63, 0x70, 0x22, 0x97, 0x33, 0xa3, 0xbd, 0xe3, 0xcf, 0x5e, 0x1c, 0x10, 0xb8,
        0x4f, 0xa9, 0x89, 0xc1, 0x5b, 0x17, 0x84, 0x7f, 0xbf, 0x71, 0xc1, 0x0a, 0xa8, 0xd4, 0x43, 0xd9, 0xb5, 0x1f,
        0x42, 0x03, 0xa2, 0x8f, 0x7d, 0x8f, 0xb2, 0xd7, 0x95, 0xd5, 0xe3, 0x87, 0x7c, 0x72, 0x05, 0xa9, 0xd2, 0x66,
        0x9c, 0x15, 0x5b, 0xf2, 0xd6, 0x1a, 0x8f, 0x15, 0x2f, 0xcb, 0xb5, 0x10, 0xbd, 0x7a, 0xcf, 0xa9, 0xa1, 0xdb,
        0x38, 0x3a, 0xc7, 0x7b, 0xee, 0xdf, 0xee, 0xbd, 0x58, 0x0b, 0x37, 0xab, 0xba, 0x38, 0x8f, 0xe5, 0x24, 0x87,
        0xa6, 0x16, 0x37, 0x19, 0x0a, 0xbc, 0xf0, 0xa7, 0xa6, 0x7c, 0x57, 0x33, 0xa7, 0xfe, 0x06, 0x9d, 0x86, 0xac,
        0xb0, 0x4e, 0x74, 0x39, 0x87, 0xbb, 0x0d, 0x07, 0x01, 0xe3, 0x3b, 0xe7, 0xc6, 0x60, 0x17, 0xa7, 0x9d, 0x49,
        0x20, 0xde, 0x28, 0xa2, 0x4b, 0x54, 0x53, 0xda, 0x0a, 0x20, 0xce, 0x72, 0xd9, 0xc8, 0x04, 0x59, 0xd9, 0x89,
        0xcd, 0xea, 0x8f, 0x00, 0xd6, 0x31, 0x7f, 0x46, 0x48, 0x3e, 0x54, 0xe0, 0x11, 0x39, 0x27, 0x2b, 0x0d, 0x43,
        0xcd, 0xff, 0x30, 0x94, 0xae, 0x66, 0x15, 0xb6, 0xc6, 0xe0, 0xb3, 0xd0, 0x87, 0x59, 0xa4, 0x0d, 0x57, 0x30,
        0x41, 0x5d, 0xe8, 0x80, 0x6e, 0xa4, 0x18, 0xc4, 0x5a, 0xee, 0x77, 0x87, 0xfb, 0xa4, 0x85, 0x90, 0xda, 0x40,
        0x2a, 0xe3, 0xc2, 0xa2, 0x37, 0xb8, 0x34, 0x01, 0x3e, 0x59, 0x0e, 0x6f, 0x7b, 0x5c, 0x24, 0xee, 0x72, 0x11,
        0x4d, 0x0c, 0x03, 0x00, 0x5e};

    static const uint8_t s_expected_intermediate_2[] = {
        0x30, 0x82, 0x03, 0x29, 0x30, 0x82, 0x02, 0x11, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x02, 0x57, 0x16, 0x30,
        0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x16, 0x31, 0x14,
        0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0b, 0x73, 0x32, 0x6e, 0x54, 0x65, 0x73, 0x74, 0x52, 0x6f,
        0x6f, 0x74, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x36, 0x30, 0x33, 0x33, 0x30, 0x31, 0x38, 0x35, 0x37, 0x30, 0x39,
        0x5a, 0x18, 0x0f, 0x32, 0x31, 0x31, 0x36, 0x30, 0x33, 0x30, 0x36, 0x31, 0x38, 0x35, 0x37, 0x30, 0x39, 0x5a,
        0x30, 0x1e, 0x31, 0x1c, 0x30, 0x1a, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x13, 0x73, 0x32, 0x6e, 0x54, 0x65,
        0x73, 0x74, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x74, 0x65, 0x30, 0x82, 0x01, 0x22,
        0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01,
        0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xcc, 0xfe, 0x2d, 0xde, 0x72, 0x5c, 0x58,
        0x72, 0xf7, 0x9d, 0x3c, 0x21, 0x27, 0x55, 0xee, 0x80, 0xfc, 0x76, 0x2c, 0x60, 0xf7, 0x8f, 0xd6, 0xa2, 0x81,
        0x65, 0x08, 0x5b, 0xa6, 0xeb, 0x97, 0xd3, 0xad, 0x63, 0x15, 0x92, 0xb0, 0x95, 0xe7, 0x1b, 0xd1, 0xe6, 0xae,
        0xd4, 0xf3, 0x91, 0x2d, 0xb7, 0x2c, 0x60, 0xae, 0x34, 0xbb, 0x7c, 0x54, 0xce, 0x80, 0x39, 0xbb, 0xa4, 0x34,
        0x79, 0xb0, 0xee, 0xa8, 0xf6, 0xea, 0x48, 0x99, 0x02, 0x6b, 0x3b, 0x21, 0xeb, 0xa7, 0x2e, 0xf5, 0x24, 0xe5,
        0xf2, 0xe6, 0xdb, 0x93, 0xba, 0x08, 0xf0, 0xb2, 0x63, 0x50, 0x91, 0x61, 0x87, 0x2d, 0x89, 0x9a, 0x6f, 0x64,
        0xad, 0x78, 0x4f, 0xc9, 0x1b, 0x1c, 0x53, 0xe0, 0x97, 0x38, 0x26, 0x76, 0x1b, 0x51, 0xa1, 0x06, 0xa4, 0xc8,
        0xf1, 0xc8, 0x0c, 0x62, 0x11, 0x15, 0x1f, 0x06, 0xf7, 0xab, 0x2d, 0x44, 0x1e, 0x0e, 0x86, 0xde, 0x7e, 0x16,
        0x02, 0x35, 0x7e, 0x70, 0xa4, 0x1c, 0x6d, 0x4c, 0xad, 0xab, 0xc0, 0xd1, 0x99, 0x57, 0xb4, 0x6c, 0xd2, 0x71,
        0x0c, 0xe5, 0xfc, 0x8a, 0xfc, 0xdf, 0x02, 0x9d, 0xe5, 0x50, 0x8c, 0x2e, 0x6f, 0xe2, 0x5f, 0xa7, 0x1a, 0xd6,
        0x21, 0x48, 0x5b, 0xde, 0x25, 0x46, 0x8f, 0x90, 0x9a, 0xb8, 0x8f, 0x7c, 0x6c, 0xd1, 0x47, 0x4d, 0x6c, 0x18,
        0xa7, 0xbb, 0x6f, 0x88, 0x82, 0x22, 0xdb, 0x54, 0x3e, 0x2f, 0x79, 0xc4, 0xc4, 0x2c, 0x84, 0xe3, 0x24, 0xd4,
        0x3f, 0xf9, 0x9e, 0x6c, 0x16, 0x7a, 0xd8, 0x99, 0xdb, 0xbe, 0xc5, 0x92, 0xb3, 0x7a, 0xf7, 0xff, 0xf0, 0x26,
        0xe5, 0xdf, 0x06, 0xdd, 0x67, 0x58, 0x05, 0x7b, 0x04, 0x46, 0x80, 0xe8, 0x2c, 0x1b, 0xc7, 0x9e, 0xa9, 0x4e,
        0xfe, 0x8a, 0x38, 0xa6, 0x91, 0x82, 0x44, 0x8e, 0xc6, 0x90, 0x3a, 0xe8, 0x00, 0x28, 0xbf, 0x02, 0x03, 0x01,
        0x00, 0x01, 0xa3, 0x77, 0x30, 0x75, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01,
        0x01, 0xff, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x6a, 0x94, 0x29, 0x55, 0x95,
        0x94, 0xbd, 0x37, 0xb0, 0x7b, 0x8e, 0x82, 0x01, 0x40, 0x09, 0xd9, 0x47, 0x72, 0x43, 0xc1, 0x30, 0x46, 0x06,
        0x03, 0x55, 0x1d, 0x23, 0x04, 0x3f, 0x30, 0x3d, 0x80, 0x14, 0x4d, 0x97, 0x01, 0xb3, 0x43, 0xaf, 0x23, 0x25,
        0x06, 0x93, 0x0d, 0x23, 0xa0, 0x9e, 0xf7, 0x6e, 0xee, 0x50, 0x5c, 0x13, 0xa1, 0x1a, 0xa4, 0x18, 0x30, 0x16,
        0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0b, 0x73, 0x32, 0x6e, 0x54, 0x65, 0x73, 0x74,
        0x52, 0x6f, 0x6f, 0x74, 0x82, 0x09, 0x00, 0xd0, 0xd4, 0x90, 0x7f, 0x94, 0x61, 0xdc, 0xf5, 0x30, 0x0d, 0x06,
        0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x37,
        0x3b, 0x74, 0xbd, 0x55, 0x3d, 0x04, 0x0b, 0x5c, 0x86, 0xe3, 0x1c, 0x8d, 0x35, 0x46, 0x7b, 0xf8, 0x2c, 0xc5,
        0xb8, 0x50, 0x3a, 0xba, 0x23, 0xb7, 0xf3, 0x28, 0x1c, 0x37, 0x93, 0x73, 0xd6, 0x83, 0x4f, 0xcb, 0x67, 0x90,
        0x7a, 0xbe, 0xef, 0x62, 0x6a, 0xc7, 0xb6, 0xa3, 0x7a, 0x81, 0x2f, 0xe8, 0xd4, 0x08, 0x39, 0x17, 0xc6, 0xcb,
        0x44, 0x2a, 0x9e, 0xb9, 0x94, 0xec, 0x38, 0x6f, 0x34, 0x3b, 0x6f, 0x47, 0x90, 0xd8, 0x6c, 0xe7, 0x16, 0x29,
        0xad, 0xd5, 0x3c, 0xd8, 0x0a, 0x9c, 0x69, 0x6c, 0x04, 0x77, 0x49, 0x03, 0xae, 0x10, 0xb9, 0xac, 0x4b, 0xec,
        0xe9, 0xf8, 0xf5, 0x27, 0x43, 0x4a, 0x9a, 0x68, 0x5b, 0xa2, 0xf6, 0x31, 0xb8, 0x36, 0xe8, 0x8f, 0x46, 0xbb,
        0x21, 0xfe, 0xc7, 0x12, 0xa5, 0xf4, 0x36, 0x2c, 0xf2, 0x53, 0x73, 0x37, 0x24, 0xe3, 0x4f, 0x66, 0x06, 0x9d,
        0x72, 0x16, 0x7e, 0x17, 0x90, 0xe8, 0xb1, 0xab, 0x45, 0xd4, 0x1a, 0x6d, 0xfe, 0x0b, 0x10, 0x47, 0x2d, 0xfc,
        0xdc, 0x8a, 0x20, 0x43, 0x4f, 0xc5, 0x4b, 0xe8, 0xee, 0x1c, 0x20, 0x9c, 0xe1, 0x52, 0x14, 0x7a, 0x6b, 0xa4,
        0x61, 0x8d, 0x49, 0xf2, 0x8e, 0x57, 0x76, 0xb1, 0x96, 0xac, 0xf0, 0x41, 0x8a, 0xe7, 0x53, 0x5a, 0xa0, 0xa8,
        0x6d, 0xef, 0x8d, 0xe7, 0x8d, 0x5b, 0xf9, 0x82, 0xd6, 0x6d, 0xa8, 0xd3, 0xe5, 0x42, 0xb3, 0xbd, 0x2a, 0x89,
        0x99, 0xca, 0xa9, 0x3d, 0x39, 0xe2, 0x24, 0x67, 0x3d, 0x79, 0x9c, 0x5c, 0xcf, 0x77, 0x02, 0x35, 0xdd, 0xdc,
        0x29, 0xaf, 0x5b, 0x2b, 0x9d, 0x6c, 0xb4, 0x49, 0xd0, 0x4a, 0x50, 0x3b, 0x3d, 0x05, 0xd8, 0xb3, 0x72, 0xbe,
        0x4c, 0x3d, 0xc4, 0x99, 0x29, 0xff, 0x1b, 0x6c, 0x7a, 0x1e, 0x67, 0xeb, 0xee, 0x5a, 0xe9, 0x6c, 0x88, 0x44,
        0x68, 0x3f, 0xb2};

    static const uint8_t s_expected_leaf[] = {
        0x30, 0x82, 0x03, 0x01, 0x30, 0x82, 0x01, 0xe9, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x09, 0x00, 0xd0, 0xd4,
        0x90, 0x7f, 0x94, 0x61, 0xdc, 0xf5, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
        0x0b, 0x05, 0x00, 0x30, 0x16, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0b, 0x73, 0x32,
        0x6e, 0x54, 0x65, 0x73, 0x74, 0x52, 0x6f, 0x6f, 0x74, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x36, 0x30, 0x33, 0x33,
        0x30, 0x31, 0x38, 0x35, 0x36, 0x33, 0x39, 0x5a, 0x18, 0x0f, 0x32, 0x31, 0x31, 0x36, 0x30, 0x33, 0x30, 0x36,
        0x31, 0x38, 0x35, 0x36, 0x33, 0x39, 0x5a, 0x30, 0x16, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x03,
        0x0c, 0x0b, 0x73, 0x32, 0x6e, 0x54, 0x65, 0x73, 0x74, 0x52, 0x6f, 0x6f, 0x74, 0x30, 0x82, 0x01, 0x22, 0x30,
        0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f,
        0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xc6, 0x39, 0xe7, 0x7d, 0xb4, 0xd3, 0x4a, 0x1a,
        0x79, 0xe7, 0x7b, 0x26, 0x6a, 0x37, 0xb2, 0xcc, 0x75, 0xef, 0x6d, 0xd9, 0x0c, 0xba, 0x67, 0xdd, 0x61, 0xb3,
        0xe8, 0x5c, 0x69, 0x58, 0xcd, 0x33, 0xb0, 0xa0, 0x3f, 0x62, 0xf7, 0x43, 0xec, 0x9e, 0xaa, 0x5c, 0x3b, 0xdf,
        0xc5, 0x4e, 0x9d, 0x66, 0x59, 0xac, 0x09, 0x2f, 0x9b, 0x48, 0xd0, 0x2a, 0xa4, 0x04, 0x4a, 0x98, 0xab, 0x01,
        0x10, 0x33, 0xd3, 0xb1, 0xd2, 0xe6, 0xac, 0xdd, 0x55, 0xd5, 0xc9, 0x4e, 0xc6, 0xf2, 0xa6, 0xe8, 0x6b, 0xd4,
        0xd5, 0xe9, 0x68, 0x45, 0x3c, 0x3a, 0xc6, 0xe9, 0x04, 0x51, 0x72, 0x7e, 0xbb, 0x1a, 0xd0, 0x30, 0xb6, 0x2f,
        0x48, 0x90, 0x62, 0x08, 0x15, 0x46, 0x23, 0xe5, 0x18, 0x34, 0x22, 0xb9, 0x73, 0xa8, 0x2e, 0x34, 0xc4, 0x58,
        0x04, 0x21, 0x9d, 0x93, 0x9f, 0xe2, 0x6b, 0x28, 0xe6, 0x2b, 0x43, 0x71, 0xcf, 0x5f, 0x75, 0xf0, 0x2c, 0x7d,
        0xcf, 0x8b, 0xda, 0xf0, 0x73, 0xc0, 0x81, 0x82, 0x33, 0xd2, 0x88, 0x0e, 0x75, 0x80, 0x76, 0xd4, 0xef, 0x45,
        0x40, 0x63, 0xdd, 0x1f, 0x59, 0x98, 0xa9, 0x1c, 0xe5, 0x8d, 0xc4, 0xac, 0x67, 0xb1, 0x05, 0xa3, 0x53, 0xfc,
        0x1c, 0x9a, 0xfc, 0xa0, 0xa2, 0xbb, 0xe9, 0x97, 0x98, 0x8b, 0x6a, 0x2b, 0xa6, 0x3c, 0xcc, 0x1c, 0x69, 0xe5,
        0xb9, 0xf1, 0xda, 0xba, 0x5e, 0x74, 0xf3, 0x04, 0x0f, 0x78, 0x50, 0x0c, 0x6b, 0x54, 0xe4, 0xbb, 0x71, 0xf2,
        0x19, 0x94, 0x5a, 0x89, 0xe4, 0x40, 0x72, 0x4e, 0xbe, 0x41, 0xc3, 0xb6, 0x36, 0xe8, 0xd7, 0xc4, 0xa3, 0x72,
        0x08, 0x19, 0x56, 0x38, 0xdd, 0xb7, 0x15, 0x04, 0xee, 0x06, 0x33, 0x7f, 0x08, 0x09, 0x7b, 0xe0, 0xc6, 0x6d,
        0x70, 0x03, 0x36, 0x85, 0x54, 0xa6, 0xa6, 0x6f, 0x6d, 0xe1, 0xc5, 0x95, 0x37, 0x93, 0x02, 0x03, 0x01, 0x00,
        0x01, 0xa3, 0x50, 0x30, 0x4e, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x4d, 0x97,
        0x01, 0xb3, 0x43, 0xaf, 0x23, 0x25, 0x06, 0x93, 0x0d, 0x23, 0xa0, 0x9e, 0xf7, 0x6e, 0xee, 0x50, 0x5c, 0x13,
        0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x4d, 0x97, 0x01, 0xb3, 0x43,
        0xaf, 0x23, 0x25, 0x06, 0x93, 0x0d, 0x23, 0xa0, 0x9e, 0xf7, 0x6e, 0xee, 0x50, 0x5c, 0x13, 0x30, 0x0c, 0x06,
        0x03, 0x55, 0x1d, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48,
        0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x05, 0xe4, 0x56, 0xf4, 0x1d,
        0x5c, 0x39, 0xa8, 0xcc, 0xf8, 0xb1, 0xb9, 0x98, 0xb5, 0xa3, 0xd7, 0x30, 0xe1, 0x28, 0xb4, 0xb3, 0x2a, 0x9f,
        0x27, 0x8b, 0x12, 0xba, 0x7a, 0xf3, 0x3d, 0xb6, 0xe6, 0xc3, 0xf4, 0x31, 0x86, 0xd8, 0x79, 0x57, 0xb5, 0xc6,
        0xae, 0x7b, 0x10, 0x54, 0xdc, 0x5a, 0x9a, 0x09, 0xdb, 0xce, 0x98, 0xc6, 0x86, 0x20, 0x45, 0xea, 0x2b, 0x0c,
        0x95, 0xd8, 0xf7, 0xdf, 0x34, 0x18, 0xb6, 0x47, 0xbe, 0xa2, 0x33, 0xa3, 0x16, 0x58, 0x44, 0xbe, 0xb3, 0xe8,
        0xe9, 0xe5, 0xb6, 0xf5, 0x19, 0x0b, 0x04, 0x4b, 0x49, 0x69, 0xd1, 0xac, 0xd9, 0xb9, 0x0c, 0x2f, 0x3b, 0x49,
        0x6d, 0xff, 0x9c, 0x6c, 0x7d, 0xae, 0xea, 0x2a, 0xf3, 0xaf, 0xa4, 0xb8, 0xdc, 0xe8, 0x15, 0x39, 0x2b, 0x81,
        0xfb, 0x31, 0xb9, 0xea, 0x4b, 0x13, 0x9c, 0x45, 0x66, 0xba, 0x6f, 0x82, 0xff, 0xcb, 0x52, 0x20, 0x0c, 0xa2,
        0xdd, 0xe6, 0xa8, 0x4a, 0xd3, 0xdc, 0x80, 0x45, 0x3c, 0xfb, 0x7f, 0xca, 0x32, 0xb1, 0xfb, 0xba, 0xe7, 0xa7,
        0x99, 0x33, 0x82, 0x29, 0xd0, 0xf5, 0x16, 0xf4, 0x83, 0x1d, 0x74, 0xfd, 0xa8, 0xf8, 0x13, 0xb6, 0xd5, 0x50,
        0x19, 0x76, 0xd7, 0xcc, 0x91, 0x2b, 0xe6, 0xd1, 0x84, 0xf8, 0xfc, 0xba, 0xca, 0x26, 0xf9, 0xba, 0x82, 0x74,
        0x74, 0x96, 0x4d, 0xb6, 0x3b, 0x3b, 0x1c, 0xe4, 0x04, 0x0e, 0x4e, 0x49, 0xd3, 0x06, 0x5a, 0x5f, 0x40, 0xc2,
        0x31, 0x28, 0x58, 0x2e, 0x23, 0x79, 0xa8, 0xb2, 0x40, 0x93, 0x4a, 0xb0, 0x97, 0xe8, 0xec, 0xe3, 0xcf, 0x01,
        0x27, 0x5f, 0x0c, 0xf5, 0xd8, 0x38, 0x4f, 0x17, 0x41, 0x60, 0x59, 0x30, 0x1d, 0x2c, 0xca, 0x5d, 0x28, 0xfd,
        0x4c, 0x93, 0x7b, 0xec, 0x16, 0xf4, 0x45, 0x9a, 0xfc, 0x2c, 0xbf, 0x53, 0xaf, 0xd2, 0x32, 0xbd, 0xf5,
    };

    struct aws_byte_cursor pem_data = aws_byte_cursor_from_c_str(s_pem_data_str);
    struct aws_array_list output_list;

    ASSERT_SUCCESS(aws_pem_objects_init_from_file_contents(&output_list, allocator, pem_data));
    ASSERT_UINT_EQUALS(3, aws_array_list_length(&output_list));

    struct aws_pem_object *pem_object = NULL;
    aws_array_list_get_at_ptr(&output_list, (void **)&pem_object, 0);
    ASSERT_BIN_ARRAYS_EQUALS(
        s_expected_intermediate_1, sizeof(s_expected_intermediate_1), pem_object->data.buffer, pem_object->data.len);
    ASSERT_CURSOR_VALUE_CSTRING_EQUALS(aws_byte_cursor_from_string(pem_object->type_string), "CERTIFICATE");
    ASSERT_INT_EQUALS(AWS_PEM_TYPE_X509, pem_object->type);
    aws_array_list_get_at_ptr(&output_list, (void **)&pem_object, 1);
    ASSERT_BIN_ARRAYS_EQUALS(
        s_expected_intermediate_2, sizeof(s_expected_intermediate_2), pem_object->data.buffer, pem_object->data.len);
    ASSERT_CURSOR_VALUE_CSTRING_EQUALS(aws_byte_cursor_from_string(pem_object->type_string), "CERTIFICATE");
    ASSERT_INT_EQUALS(AWS_PEM_TYPE_X509, pem_object->type);
    aws_array_list_get_at_ptr(&output_list, (void **)&pem_object, 2);
    ASSERT_BIN_ARRAYS_EQUALS(s_expected_leaf, sizeof(s_expected_leaf), pem_object->data.buffer, pem_object->data.len);
    ASSERT_CURSOR_VALUE_CSTRING_EQUALS(aws_byte_cursor_from_string(pem_object->type_string), "CERTIFICATE");
    ASSERT_INT_EQUALS(AWS_PEM_TYPE_X509, pem_object->type);

    aws_pem_objects_clean_up(&output_list);

    return AWS_OP_SUCCESS;
}
AWS_TEST_CASE(test_pem_cert_chain_comments_and_whitespace, s_test_pem_cert_chain_comments_and_whitespace)

static int s_test_pem_invalid_parse(struct aws_allocator *allocator, void *ctx) {
    (void)ctx;
    /* Maintainers note: I removed a '-' from the beginning */
    static const char *s_invalid_pem = "----BEGIN CERTIFICATE-----\n"
                                       "MIICeDCCAeGgAwIBAgIJAObttnPKQhVlMA0GCSqGSIb3DQEBDgUAMF8xCzAJBgNV\n"
                                       "BAYTAlVTMQswCQYDVQQIDAJXQTEQMA4GA1UEBwwHU2VhdHRsZTEPMA0GA1UECgwG\n"
                                       "QW1hem9uMQwwCgYDVQQLDANzMm4xEjAQBgNVBAMMCWxvY2FsaG9zdDAgFw0xNzA4\n"
                                       "MDEyMjQzMzJaGA8yMTE3MDcwODIyNDMzMlowXzELMAkGA1UEBhMCVVMxCzAJBgNV\n"
                                       "BAgMAldBMRAwDgYDVQQHDAdTZWF0dGxlMQ8wDQYDVQQKDAZBbWF6b24xDDAKBgNV\n"
                                       "BAsMA3MybjESMBAGA1UEAwwJbG9jYWxob3N0MIGfMA0GCSqGSIb3DQEBAQUAA4GN\n"
                                       "ADCBiQKBgQCisRoXXcTh4ejn/sUjGosLlE7GlpLGtvWFEEX6Vl3klVoQdkyabLIH\n"
                                       "7bHB2P7uyt9bPzeqvWYjuepDBSQUUeb6Mkqfx237bTy8JhXIfpIhbgksTk7IPzgo\n"
                                       "XLPl1oNl7uB9HQaDQ7UPlaKbfp1gNvs6uGOH4vvyhhJGiblNJKnVwwIDAQABozow\n"
                                       "ODALBgNVHQ8EBAMCBDAwEwYDVR0lBAwwCgYIKwYBBQUHAwEwFAYDVR0RBA0wC4IJ\n"
                                       "MTI3LjAuMC4xMA0GCSqGSIb3DQEBDgUAA4GBACleH44LSYhzHHaV70VbnLbtbv8T\n"
                                       "eaUvzstFW6YvdP1XnZKssZNdvMhoiMuMD5n40/iPbv+grtjxacRQCinLk1SEjpsu\n"
                                       "3lw90Ds0Ksd/Pdsv7d0cCiJkjadON+ZQEEJ2FP/G19KZFxC3GLk9sxIUXyUW0TXn\n"
                                       "YxwtPz26+xvPRWCS\n"
                                       "-----END CERTIFICATE-----";

    struct aws_byte_cursor pem_data = aws_byte_cursor_from_c_str(s_invalid_pem);
    struct aws_array_list output_list;

    ASSERT_ERROR(AWS_ERROR_PEM_MALFORMED, aws_pem_objects_init_from_file_contents(&output_list, allocator, pem_data));
    ASSERT_UINT_EQUALS(0, aws_array_list_length(&output_list));

    aws_array_list_clean_up(&output_list);

    return AWS_OP_SUCCESS;
}
AWS_TEST_CASE(test_pem_invalid_parse, s_test_pem_invalid_parse)

static int s_test_pem_valid_data_invalid_parse(struct aws_allocator *allocator, void *ctx) {
    (void)ctx;
    /* Maintainers note: I added a character (the 'q') to the end to make the base64 invalid */
    static const char *s_invalid_data = "-----BEGIN CERTIFICATE-----\n"
                                        "MIICeDCCAeGgAwIBAgIJAObttnPKQhVlMA0GCSqGSIb3DQEBDgUAMF8xCzAJBgNV\n"
                                        "BAYTAlVTMQswCQYDVQQIDAJXQTEQMA4GA1UEBwwHU2VhdHRsZTEPMA0GA1UECgwG\n"
                                        "QW1hem9uMQwwCgYDVQQLDANzMm4xEjAQBgNVBAMMCWxvY2FsaG9zdDAgFw0xNzA4\n"
                                        "MDEyMjQzMzJaGA8yMTE3MDcwODIyNDMzMlowXzELMAkGA1UEBhMCVVMxCzAJBgNV\n"
                                        "BAgMAldBMRAwDgYDVQQHDAdTZWF0dGxlMQ8wDQYDVQQKDAZBbWF6b24xDDAKBgNV\n"
                                        "BAsMA3MybjESMBAGA1UEAwwJbG9jYWxob3N0MIGfMA0GCSqGSIb3DQEBAQUAA4GN\n"
                                        "ADCBiQKBgQCisRoXXcTh4ejn/sUjGosLlE7GlpLGtvWFEEX6Vl3klVoQdkyabLIH\n"
                                        "7bHB2P7uyt9bPzeqvWYjuepDBSQUUeb6Mkqfx237bTy8JhXIfpIhbgksTk7IPzgo\n"
                                        "XLPl1oNl7uB9HQaDQ7UPlaKbfp1gNvs6uGOH4vvyhhJGiblNJKnVwwIDAQABozow\n"
                                        "ODALBgNVHQ8EBAMCBDAwEwYDVR0lBAwwCgYIKwYBBQUHAwEwFAYDVR0RBA0wC4IJ\n"
                                        "MTI3LjAuMC4xMA0GCSqGSIb3DQEBDgUAA4GBACleH44LSYhzHHaV70VbnLbtbv8T\n"
                                        "eaUvzstFW6YvdP1XnZKssZNdvMhoiMuMD5n40/iPbv+grtjxacRQCinLk1SEjpsu\n"
                                        "3lw90Ds0Ksd/Pdsv7d0cCiJkjadON+ZQEEJ2FP/G19KZFxC3GLk9sxIUXyUW0TXn\n"
                                        "YxwtPz26+xvPRWCSq\n"
                                        "-----END CERTIFICATE-----";

    struct aws_byte_cursor pem_data = aws_byte_cursor_from_c_str(s_invalid_data);
    struct aws_array_list output_list;

    ASSERT_ERROR(AWS_ERROR_PEM_MALFORMED, aws_pem_objects_init_from_file_contents(&output_list, allocator, pem_data));
    ASSERT_UINT_EQUALS(0, aws_array_list_length(&output_list));

    aws_array_list_clean_up(&output_list);

    return AWS_OP_SUCCESS;
}
AWS_TEST_CASE(test_pem_valid_data_invalid_parse, s_test_pem_valid_data_invalid_parse)

static int s_test_pem_invalid_in_chain_parse(struct aws_allocator *allocator, void *ctx) {
    (void)ctx;
    /* Maintainers note: I added a character (the 'f') to the end of the 3rd cert to make the base64 invalid */
    static const char *s_invalid_data = "-----BEGIN CERTIFICATE-----\n"
                                        "MIICrTCCAZUCAn3VMA0GCSqGSIb3DQEBBQUAMB4xHDAaBgNVBAMME3MyblRlc3RJ\n"
                                        "bnRlcm1lZGlhdGUwIBcNMTYwMzMwMTg1NzQzWhgPMjExNjAzMDYxODU3NDNaMBgx\n"
                                        "FjAUBgNVBAMMDXMyblRlc3RTZXJ2ZXIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw\n"
                                        "ggEKAoIBAQDRw6AuYXAeRT0YuptCfJjRB/EDJyyGXnv+8TV2H1WJWhMLk8qND27r\n"
                                        "79A6EjbVmJaOV9qrokVqpDmXS712Z3BDprJ+1LFMymm3A+AFuK/skeGy0skik+Tg\n"
                                        "MmFT5XBVvmsw4uB1S9uUqktHauXgjhFPPsfvk4ewL4LulVEN2TEeI1Odj4CaMxAO\n"
                                        "Iuowm8wI2OHVzRHlrRmyJ9hYGuHHQ2TaTGIjr3WpAFuXi9pHGGMYa0uXAVPmgjdE\n"
                                        "XZ8t46u/ZKQ9W1uJkZEVKhcijT7G2VBrsBUq0CDiL+TDaGfthnBzUc9zt4fx/S/3\n"
                                        "qulC2WbKI3xrasQyjrsHTAJ75Md3rK09AgMBAAEwDQYJKoZIhvcNAQEFBQADggEB\n"
                                        "AHHkXNA9BtgAebZC2zriW4hRfeIkJMOwvfKBXHTuY5iCLD1otis6AZljcCKXM6O9\n"
                                        "489eHBC4T6mJwVsXhH+/ccEKqNRD2bUfQgOij32PsteV1eOHfHIFqdJmnBVb8tYa\n"
                                        "jxUvy7UQvXrPqaHbODrHe+7f7r1YCzerujiP5SSHphY3GQq88KemfFczp/4GnYas\n"
                                        "sE50OYe7DQcB4zvnxmAXp51JIN4ooktUU9oKIM5y2cgEWdmJzeqPANYxf0ZIPlTg\n"
                                        "ETknKw1Dzf8wlK5mFbbG4LPQh1mkDVcwQV3ogG6kGMRa7neH+6SFkNpAKuPCoje4\n"
                                        "NAE+WQ5ve1wk7nIRTQwDAF4=\n"
                                        "-----END CERTIFICATE-----\n"
                                        "-----BEGIN CERTIFICATE-----\n"
                                        "MIIDKTCCAhGgAwIBAgICVxYwDQYJKoZIhvcNAQEFBQAwFjEUMBIGA1UEAwwLczJu\n"
                                        "VGVzdFJvb3QwIBcNMTYwMzMwMTg1NzA5WhgPMjExNjAzMDYxODU3MDlaMB4xHDAa\n"
                                        "BgNVBAMME3MyblRlc3RJbnRlcm1lZGlhdGUwggEiMA0GCSqGSIb3DQEBAQUAA4IB\n"
                                        "DwAwggEKAoIBAQDM/i3eclxYcvedPCEnVe6A/HYsYPeP1qKBZQhbpuuX061jFZKw\n"
                                        "lecb0eau1PORLbcsYK40u3xUzoA5u6Q0ebDuqPbqSJkCazsh66cu9STl8ubbk7oI\n"
                                        "8LJjUJFhhy2Jmm9krXhPyRscU+CXOCZ2G1GhBqTI8cgMYhEVHwb3qy1EHg6G3n4W\n"
                                        "AjV+cKQcbUytq8DRmVe0bNJxDOX8ivzfAp3lUIwub+JfpxrWIUhb3iVGj5CauI98\n"
                                        "bNFHTWwYp7tviIIi21Q+L3nExCyE4yTUP/mebBZ62JnbvsWSs3r3//Am5d8G3WdY\n"
                                        "BXsERoDoLBvHnqlO/oo4ppGCRI7GkDroACi/AgMBAAGjdzB1MAwGA1UdEwQFMAMB\n"
                                        "Af8wHQYDVR0OBBYEFGqUKVWVlL03sHuOggFACdlHckPBMEYGA1UdIwQ/MD2AFE2X\n"
                                        "AbNDryMlBpMNI6Ce927uUFwToRqkGDAWMRQwEgYDVQQDDAtzMm5UZXN0Um9vdIIJ\n"
                                        "ANDUkH+UYdz1MA0GCSqGSIb3DQEBBQUAA4IBAQA3O3S9VT0EC1yG4xyNNUZ7+CzF\n"
                                        "uFA6uiO38ygcN5Nz1oNPy2eQer7vYmrHtqN6gS/o1Ag5F8bLRCqeuZTsOG80O29H\n"
                                        "kNhs5xYprdU82AqcaWwEd0kDrhC5rEvs6fj1J0NKmmhbovYxuDboj0a7If7HEqX0\n"
                                        "NizyU3M3JONPZgadchZ+F5DosatF1Bpt/gsQRy383IogQ0/FS+juHCCc4VIUemuk\n"
                                        "YY1J8o5XdrGWrPBBiudTWqCobe+N541b+YLWbajT5UKzvSqJmcqpPTniJGc9eZxc\n"
                                        "z3cCNd3cKa9bK51stEnQSlA7PQXYs3K+TD3EmSn/G2x6Hmfr7lrpbIhEaD+y\n"
                                        "-----END CERTIFICATE-----\n"
                                        "-----BEGIN CERTIFICATE-----\n"
                                        "MIIDATCCAemgAwIBAgIJANDUkH+UYdz1MA0GCSqGSIb3DQEBCwUAMBYxFDASBgNV\n"
                                        "BAMMC3MyblRlc3RSb290MCAXDTE2MDMzMDE4NTYzOVoYDzIxMTYwMzA2MTg1NjM5\n"
                                        "WjAWMRQwEgYDVQQDDAtzMm5UZXN0Um9vdDCCASIwDQYJKoZIhvcNAQEBBQADggEP\n"
                                        "ADCCAQoCggEBAMY5532000oaeed7Jmo3ssx1723ZDLpn3WGz6FxpWM0zsKA/YvdD\n"
                                        "7J6qXDvfxU6dZlmsCS+bSNAqpARKmKsBEDPTsdLmrN1V1clOxvKm6GvU1eloRTw6\n"
                                        "xukEUXJ+uxrQMLYvSJBiCBVGI+UYNCK5c6guNMRYBCGdk5/iayjmK0Nxz1918Cx9\n"
                                        "z4va8HPAgYIz0ogOdYB21O9FQGPdH1mYqRzljcSsZ7EFo1P8HJr8oKK76ZeYi2or\n"
                                        "pjzMHGnlufHaul508wQPeFAMa1Tku3HyGZRaieRAck6+QcO2NujXxKNyCBlWON23\n"
                                        "FQTuBjN/CAl74MZtcAM2hVSmpm9t4cWVN5MCAwEAAaNQME4wHQYDVR0OBBYEFE2X\n"
                                        "AbNDryMlBpMNI6Ce927uUFwTMB8GA1UdIwQYMBaAFE2XAbNDryMlBpMNI6Ce927u\n"
                                        "UFwTMAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQELBQADggEBAAXkVvQdXDmozPix\n"
                                        "uZi1o9cw4Si0syqfJ4sSunrzPbbmw/Qxhth5V7XGrnsQVNxamgnbzpjGhiBF6isM\n"
                                        "ldj33zQYtke+ojOjFlhEvrPo6eW29RkLBEtJadGs2bkMLztJbf+cbH2u6irzr6S4\n"
                                        "3OgVOSuB+zG56ksTnEVmum+C/8tSIAyi3eaoStPcgEU8+3/KMrH7uuenmTOCKdD1\n"
                                        "FvSDHXT9qPgTttVQGXbXzJEr5tGE+Py6yib5uoJ0dJZNtjs7HOQEDk5J0wZaX0DC\n"
                                        "MShYLiN5qLJAk0qwl+js488BJ18M9dg4TxdBYFkwHSzKXSj9TJN77Bb0RZr8LL9T\n"
                                        "r9IyvfUf=\n"
                                        "-----END CERTIFICATE-----";

    struct aws_byte_cursor pem_data = aws_byte_cursor_from_c_str(s_invalid_data);
    struct aws_array_list output_list;

    ASSERT_ERROR(AWS_ERROR_PEM_MALFORMED, aws_pem_objects_init_from_file_contents(&output_list, allocator, pem_data));
    ASSERT_UINT_EQUALS(0, aws_array_list_length(&output_list));

    aws_array_list_clean_up(&output_list);

    return AWS_OP_SUCCESS;
}

AWS_TEST_CASE(test_pem_invalid_in_chain_parse, s_test_pem_invalid_in_chain_parse)
