///////////////////////////////////////////////////////////////////////////////
//                                                         
// Main.cc
// -------
// Dragon main module
//                                               
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2007-2010 by Bjoern Lemke
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// IMPLEMENTATION MODULE
//
// Class: main
// 
// Description: main module for dragon 
//
///////////////////////////////////////////////////////////////////////////////


#include <stdlib.h>

// BASE INCLUDES
#include <lfcbase/Exception.h>
#include <lfcbase/Chain.h>
#include <lfcbase/GetOpt.h>

// DRAGON INCLUDES
#include "Dragon.h"
#include "CPlusGenerator.h"
#include "JavaGenerator.h"
#include "GoLangGenerator.h"

#include <config.h>
#define DRAGONVERSION PACKAGE_VERSION

int main(int argc, char** argv)
{

    Chain parserName;
    bool dumpIt=false;
    Chain targetSource("c++");
    bool dynamicTable=false;
    Chain parserMode("lalr");

    try
    {
	GetOpt opt(argc, argv, (char*)"vhdt:yp:m:");
    
	int i = opt.firstOpt();
    
	while ( i > 0 )
	{
	    switch (i) 
	    {
	    case 'v':
		cout << "Dragon Version " << DRAGONVERSION << endl;
		exit (0);
	    case 'h':
		cout << argv[0] << " [ -v ] [ -h ] [ -d ] [ -t <c++|java|golang> ] [ -y ] [ -m <lr1|lalr> ] -p < grammar name >  " << endl;
		exit(0);
	    case 'p':
		parserName = Chain(opt.getArg());
		break;
	    case 'm':
		parserMode = Chain(opt.getArg());
		break;
	    case 't':
		targetSource = Chain(opt.getArg());
		break;
	    case 'd':
		dumpIt=true;
		break;
	    case 'y':
		dynamicTable= true;
		break;
	    }
	    
	    i = opt.nextOpt();
	    
	}
    
	if ( parserName.length() == 0 )
	{
	    throw Exception(EXLOC, "No grammar specified");
	}
	
	if ( targetSource != Chain("c++")
	     && targetSource != Chain("java")
	     && targetSource != Chain("golang")
	    )
	{
	    Chain msg = Chain("Target source ") + targetSource + Chain(" not supported"); 
	    throw Exception(EXLOC, msg);
	}

	Dragon::ParserMode mode=Dragon::LR1;
	
	if ( parserMode == Chain("lalr"))
	{
	    mode=Dragon::LALR;
	}
	else if ( parserMode == Chain("lr1"))
	{
	    mode=Dragon::LR1;
	}
	else
	{
	    Chain msg = Chain("Invalid parser analysis mode, must be <lr1> or <lalr>");
	    throw Exception(EXLOC, msg);
	}

	if ( targetSource == Chain("c++") )
	{
	    CPlusGenerator parserBuilder(parserName, mode, dynamicTable);
	    parserBuilder.generate(dumpIt);
	}
	else if ( targetSource == Chain("java") )
	{
	    JavaGenerator parserBuilder(parserName, mode, dynamicTable);
	    parserBuilder.generate(dumpIt);
	}
	else if ( targetSource == Chain("golang") )
	{
	    GoLangGenerator parserBuilder(parserName, mode, dynamicTable);
	    parserBuilder.generate(dumpIt);
	}

    }
    catch (Exception e)
    {
	Chain msg;
	e.pop(msg);
	cerr << msg << endl;
	exit(1);
    }

    exit(0);

}

